///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdmAction.cc  
// ----------------                                                        
// Cego semantic parser adm actions
//
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2010 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdmAction
// 
// Description: 
//
// Status: QS-2.6
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/Tokenizer.h>
#include <lfcbase/Host.h>
#include <lfcbase/Datetime.h>

#include "CegoXMLdef.h"
#include "CegoAdmAction.h"
#include "CegoOutput.h"

#define DEFTMPSIZE 100
#define DEFSYSSIZE 100
#define DEFLOGNUM 3
#define DEFLOGSIZE 1000000
#define DEFSORTAREASIZE 10000000
#define DEFTSROOT "./"
#define DEFPRIMARY "localhost"
#define DEFSECONDARY "localhost"
#define DEFTSTICKET "tsticket.xml"
#define DEFTIMEOUT 10

#define MAXHOSTLEN 25

CegoAdmAction::CegoAdmAction(CegoAdminHandler *pAH, CegoModule *pModule, CegoAdmAction::ResultMode resultMode, bool rawMode) 
{
    _i=0;
    _pC = 0;
    _pModule = pModule;
    _pAH = pAH;
    _pit = 0;
    _rawMode = rawMode;
    _resultMode = resultMode;
    _keepTicket = false;
    _doLogging = true;
    _doCleanup = false;
    _doForceload = false;
    _xportMode = XML_XML_VALUE;
    _timeout = DEFTIMEOUT;
    setDefault();

    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("ROLE"), VARCHAR_TYPE, 10)); 
    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("HOSTNAME"), VARCHAR_TYPE, MAXHOSTLEN)); 
    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("MSG"), VARCHAR_TYPE, 50));
        
    _modId = _pModule->getModId("CegoAdmAction");
}

CegoAdmAction::~CegoAdmAction()  
{
}

void CegoAdmAction::statementAction()
{
}

void CegoAdmAction::rawModeOn()
{
    _rawMode = true;
}

void CegoAdmAction::rawModeOff()
{
    _rawMode = false;
}

bool CegoAdmAction::getRawMode()
{
    return _rawMode;
}

void CegoAdmAction::setCommandChain(char *pC)
{
    _i = 0;
    _pC = pC;
    
    // init some vars
    _buMsg = Chain("NONE");
    _syncCmd = Chain("");
    _timeout = DEFTIMEOUT;
 }

char CegoAdmAction::nextChar() 
{

    if (_pC[_i])
    {
	if ( _pC[_i] == '\'')
	{
	    setReserved(STRINGVAL);
	    
	    _i++;
	    int j=0;
	    while ( _pC[_i] != '\'')
	    {
		// cout << "Storing: " << _pC[_i] << endl;
		_stringBuf[j] = _pC[_i];
		j++;
		_i++;
		if (_i == MAXSTRINGLEN)
		{
		    throw Exception(EXLOC, "Stringbuf exceeded");
		}

	    }
	    _stringBufLen = j+1;
	    _stringBuf[j] = 0;
	    _i++;
	    return 0;
	    
	}
	_i++;
	return _pC[_i-1];
	// return (toupper(_pC[_i-1]));
    }
    return 0;
}

void CegoAdmAction::backChar() 
{
    _i--;
}

    
void CegoAdmAction::printTokenList() 
{
    
    cout << "TokenList is " << endl;
	
    ListT<Chain> tlist =  getTokenList();
    Chain* pS = tlist.First();
    while (pS)
    {
	cout << *pS << endl;
	pS = tlist.Next();
    }
    
}

void CegoAdmAction::cleanUp() 
{
    
}

void CegoAdmAction::setDefault()
{
    _primary = DEFPRIMARY;
    _secondary = DEFSECONDARY;
    _tsRoot = DEFTSROOT;
    _tsTicket = DEFTSTICKET;
    _logSize = DEFLOGSIZE;
    _logNum = DEFLOGNUM;
    _sortAreaSize = DEFSORTAREASIZE;
    _sysSize = DEFSYSSIZE;
    _tmpSize = DEFTMPSIZE;
}

void CegoAdmAction::setPitFormat(const Chain& pitFormat)
{
    _pitFormat = pitFormat;
}

/////////////////////////////////
// cego semantic action methods //
/////////////////////////////////

void CegoAdmAction::lockInfoAction() 
{

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    if ( pS )
    {

	Chain lockCat;
	if ( *pS  == Chain("rec") )
	{
	    lockCat  = Chain(XML_LOCK_REC_VALUE);
	}
	else if ( *pS  == Chain("rbrec") )
	{
	    lockCat = Chain(XML_LOCK_RBREC_VALUE);
	}
	else if ( *pS  == Chain("sysrec") )
	{
	    lockCat = Chain(XML_LOCK_SYSREC_VALUE);
	}
	else if ( *pS  == Chain("tp") )
	{
	    lockCat = Chain(XML_LOCK_THREADPOOL_VALUE);
	}
	else if ( *pS == Chain("sp") )
	{
	    lockCat = Chain(XML_LOCK_SYSPAGE_VALUE);
	}
	else if ( *pS == Chain("dp") )
	{
	    lockCat = Chain(XML_LOCK_DATAPAGE_VALUE);
	}
	else if ( *pS == Chain("ix") )
	{
	    lockCat = Chain(XML_LOCK_IDXPAGE_VALUE);
	}
	else if ( *pS == Chain("rb") )
	{
	    lockCat = Chain(XML_LOCK_RBPAGE_VALUE);
	}
	else if ( *pS == Chain("df") )
	{
	    lockCat = Chain(XML_LOCK_DATAFILE_VALUE);
	}
	else if ( *pS == Chain("bp") )
	{
	    lockCat = Chain(XML_LOCK_BUFFERPOOL_VALUE);
	}
	else if ( *pS == Chain("ts") )
	{
	    lockCat = Chain(XML_LOCK_TABLESET_VALUE);
	}
	else
	{
	    throw Exception(EXLOC, "Invalid lock category");
	}

	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqLockInfo(lockCat);

	handleMedResult(res);
	
	Chain msg;
	_pAH->getMsg(msg);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getLockInfo(oe, info);
	
	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);
	if ( _rawMode == false )
	    cout << msg << endl;

    }
}


void CegoAdmAction::lockStatAction() 
{

    
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqLockStat();    
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getLockStat(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::poolInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqPoolInfo();
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info1;
    ListT<ListT<CegoFieldValue> > info2;
    
    _pAH->getPoolInfo(oe, info1, info2);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();
    ListT<CegoFieldValue> *pFVL = info1.First();
    while ( pFVL )
    {
	output.rowOut(*pFVL);	
	pFVL = info1.Next();
    }

    pFVL = info2.First();
    while ( pFVL )
    {
	output.rowOut(*pFVL);	
	pFVL = info2.Next();
    }
    output.tailOut();

    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::poolEntryListAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqPoolEntryList();
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getPoolEntryList(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::dbSessionInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbSessionInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getDbSessionInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    
    if ( _rawMode == false )
	cout << msg << endl;
	
}


void CegoAdmAction::listTableSetAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetTableSetList();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getTableSetList(oe, info);
    
    CegoOutput output(oe.getSchema(), Chain("llllll"));
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::threadInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqThreadInfo();
    handleMedResult(res);	

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;

    _pAH->getThreadInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);

    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::dbThreadInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbThreadInfo();	
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
    
}


void CegoAdmAction::dbThreadLastQueryAction() 
{

    int threadId;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	threadId = pS->asInteger();
    }

        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbThreadInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadLastQuery(threadId, oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
    
}

void CegoAdmAction::admThreadInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqAdmThreadInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getAdmThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
    
}

void CegoAdmAction::logThreadInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqLogThreadInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getLogThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
    
}

void CegoAdmAction::copyInfoAction() 
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqCopyInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getCopyInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
	
}

void CegoAdmAction::buFileInfoAction()
{

    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getBuFileInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::buStatInfoAction()
{

    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetBUStatInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getBUStatInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::buStatResetAction()
{

    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medResetBUStat(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::abortDbThreadAction() 
{
    int threadId;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	threadId = pS->asInteger();
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->reqDbThreadAbort(threadId);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::retrieveTableSetAction()
{
    Chain masterHost(_stringBuf);
    CegoAdminHandler::ResultType res;
    res = _pAH->reqRetrieveTableSet(masterHost);	
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
} 

void CegoAdmAction::defineTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medDefineTableSet(tableSet,
				  _tsRoot,
				  _tsTicket,
				  _primary,
				  _secondary,
				  _sysSize,
				  _tmpSize,
				  _logSize,
				  _logNum, 
				  _sortAreaSize);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;


    setDefault();

}

void CegoAdmAction::startTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medStartTableSet(tableSet, _doCleanup, _doForceload);
    _doCleanup=false;
    _doForceload=false;

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::stopTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;    
    res = _pAH->medStopTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::copyTableSetAction() 
{
    CegoAdminHandler::ResultType res;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medCopyTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::switchTableSetAction() 
{
    CegoAdminHandler::ResultType res;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medSwitchTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::dropTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medDropTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::removeTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medRemoveTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::resetTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medResetTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::createTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medCreateTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::syncTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSyncTableSet(tableSet, _buMsg, _syncCmd, _timeout);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::beginBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medBeginBackup(tableSet, _buMsg);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::endBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medEndBackup(tableSet, _buMsg, _keepTicket);
    _keepTicket = false;
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::keepTicketAction()
{
    _keepTicket = true;
}


void CegoAdmAction::syncCmdAction()
{
    _syncCmd = Chain(_stringBuf);
}

void CegoAdmAction::timeoutAction()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_timeout = pS->asInteger();
    }
}

void CegoAdmAction::nologAction()
{
    _doLogging = false;
}

void CegoAdmAction::msgAction()
{
    _buMsg = Chain(_stringBuf);
}

void CegoAdmAction::cleanupAction()
{
    _doCleanup = true;
}

void CegoAdmAction::forceloadAction()
{
    _doForceload = true;
}

void CegoAdmAction::xmlModeAction()
{
    _xportMode = XML_XML_VALUE;
}

void CegoAdmAction::binModeAction()
{
    _xportMode = XML_BIN_VALUE;
}

void CegoAdmAction::plainModeAction()
{
    _xportMode = XML_PLAIN_VALUE;
}

void CegoAdmAction::recoverTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medRecover(tableSet, _pit);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::setLogMngAction()
{

    Chain progName(_stringBuf);

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetLogMng(progName, _timeout);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

    _timeout = DEFTIMEOUT;

}


void CegoAdmAction::showLogMngAction()
{

    CegoAdminHandler::ResultType res;
    res = _pAH->reqShowLogMng();

    if ( res == CegoAdminHandler::ADM_OK )
    {

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getLogMngInfo(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);

	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}


void CegoAdmAction::setTSInitFileAction()
{

    Chain initFileName(_stringBuf);

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSInitFile(tableSet, initFileName);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::setTSRootPathAction()
{

    Chain tsRoot(_stringBuf);

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSRootPath(tableSet, tsRoot);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::setTSTicketAction()
{

    Chain tsTicket(_stringBuf);

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSTicket(tableSet, tsTicket);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSLogNumAction()
{

    int logNum;

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	logNum = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSLogNum(tableSet, logNum);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSLogSizeAction()
{

    int logSize;

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	logSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSLogSize(tableSet, logSize);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSSortAreaSizeAction()
{

    int sortAreaSize;

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	sortAreaSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSSortAreaSize(tableSet, sortAreaSize);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSSysSizeAction()
{

    Chain tableSet;
    int sysSize;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	sysSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSSysSize(tableSet, sysSize);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::setTSTmpSizeAction()
{

    Chain tableSet;
    int tmpSize;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tmpSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSTmpSize(tableSet, tmpSize);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::setCheckpointAction()
{

    Chain tableSet;
    int interval;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	interval = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetCheckpoint(tableSet, interval);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::switchSecondaryAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medSecSwitch(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::switchMediatorAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->secMedSwitch(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::relocateSecondaryAction()
{
    Chain secondary(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medSecRelocate(tableSet, secondary);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::relocateMediatorAction()
{
    Chain mediator(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->secMedRelocate(tableSet, mediator);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::checkTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medCheckTableSet(tableSet);
    handleMedResult(res);
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetCheck(oe, info);
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
}

void CegoAdmAction::verifyTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medVerifyTableSet(tableSet);
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetVerificationInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;
	_pAH->getTableSetVerificationInfo(oe, info);
	
	ListT<CegoFieldValue>* pFVL = info.First();
	while ( pFVL )
	{
	    output.rowOut(*pFVL);
	    pFVL = info.Next();
	}	
	res = _pAH->nextInfo();
    }
    output.tailOut();

    Chain msg;
    _pAH->getMsg(msg);

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	    cout << msg << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }
}


void CegoAdmAction::correctTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medCorrectTableSet(tableSet);
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }


    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetCorrectionInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;
	_pAH->getTableSetCorrectionInfo(oe, info);
	
	ListT<CegoFieldValue>* pFVL = info.First();
	while ( pFVL )
	{
	    output.rowOut(*pFVL);
	    pFVL = info.Next();
	}	
	res = _pAH->nextInfo();
    }
    output.tailOut();

    Chain msg;
    _pAH->getMsg(msg);

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	    cout << msg << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }

}


void CegoAdmAction::getDbSpecAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain dbSpec;
    res = _pAH->reqGetDbSpec(dbSpec);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	_pAH->getMsg(msg);
	cout << msg << endl;
	cout << "DBSPEC=" << dbSpec << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}


void CegoAdmAction::listNodeAction() 
{
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqGetNodeList();

    if ( res == CegoAdminHandler::ADM_OK )
    {

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getNodeInfo(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);

	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::getDbInfoAction() 
{

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqGetDbInfo();

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::tableSetInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getDetailedTableSetInfo(oe, info);

    CegoOutput output(oe.getSchema(), Chain("rl"));
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}
 

void CegoAdmAction::dataFileInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;   
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain msg;
    Chain format;
    _pAH->getMsg(msg);
    _pAH->getDataFileInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::archLogInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getArchLogInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::logInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getLogInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::objInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetObjectInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getObjectInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::transactionInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetTransactionInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getTransactionInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}


void CegoAdmAction::setTableSetNodeAction() 
{
    Chain tableSet;
    Chain primary;
    Chain secondary;
    Chain mediator;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	mediator = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 secondary = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 primary = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTableSetNode(tableSet, primary, secondary, mediator);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::copyFileAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain fileName(_stringBuf);
    res = _pAH->reqCopyFile(fileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::exportTableAction()
{

    Chain exportFile(_stringBuf);

    Chain tableSet;
    Chain tableName;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableName = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqExportTable(tableSet, tableName, exportFile, _xportMode);
    _xportMode = XML_XML_VALUE;

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::exportTableSetAction()
{
    Chain exportFile(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    res = _pAH->reqExportTableSet(tableSet, false, exportFile, _xportMode);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
    
    _xportMode = XML_XML_VALUE;

}

void CegoAdmAction::exportTableSetStructureAction()
{
    Chain exportFile(_stringBuf);
    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqExportTableSet(tableSet, true, exportFile, _xportMode);
    handleMedResult(res);

    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::importTableAction()
{
    Chain importFile(_stringBuf);

    Chain tableSet;
    Chain tableName;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableName = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTable(tableSet, tableName, importFile, _doLogging, _xportMode);	
    handleMedResult(res);

    _doLogging = true;
    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;


}

void CegoAdmAction::importTableSetAction()
{
    Chain importFile(_stringBuf);
    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTableSet(tableSet, false, importFile, _doLogging, _xportMode);
    handleMedResult(res);

    _doLogging = true;
    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::importTableSetStructureAction()
{
    Chain importFile(_stringBuf);

    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTableSet(tableSet, true, importFile, _doLogging, _xportMode);
    handleMedResult(res);

    _doLogging = true;
    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::addDataFileAction()
{

    int numPages;
    Chain tableSet;
    Chain dataFile(_stringBuf);
    Chain fileType;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	numPages = pS->asInteger();
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	if ( *pS == Chain("app" ) )
	{
	    fileType = Chain(XML_APPFILE_VALUE);
	}
	else if ( *pS == Chain("temp" ) )
	{
	    fileType = Chain(XML_TEMPFILE_VALUE);
	}
	else if ( *pS == Chain("sys" ) )
	{
	    fileType = Chain(XML_SYSFILE_VALUE);
	}
	else
	{
	    throw Exception(EXLOC, Chain("Invalid file type"));
	}
    }
   
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medAddDataFile(tableSet, fileType, dataFile, numPages);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::listUserAction()
{
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqShowUser();
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getUserInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    
    /*
    if ( _rawMode == false )
    {
	Chain msg;
	_pAH->getMsg(msg);
	cout << msg << endl;
    }
    
    */
}

void CegoAdmAction::addUserAction()
{

    Chain user;
    Chain password(_stringBuf);

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqAddUser(user, password);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::removeUserAction()
{

    Chain tableSet;
    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqRemoveUser(user);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::changePwdAction()
{

    Chain user;
    Chain password(_stringBuf);

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqChangePwd(user, password);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::assignRoleAction()
{

    Chain user;
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqAssignRole(user, role);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;


}

void CegoAdmAction::removeRoleAction()
{

    Chain user;
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqRemoveRole(user, role);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::listRoleAction()
{

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqListRole();
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getRoleList(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);

    /*
    if ( _rawMode == false )
    {
	Chain msg;
	_pAH->getMsg(msg);
	cout << msg << endl;
    }
    */

}

void CegoAdmAction::showRoleAction()
{

    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqShowRole(role);
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getRoleInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
}


void CegoAdmAction::createRoleAction()
{

    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqCreateRole(role);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::dropRoleAction()
{

    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDropRole(role);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::setPermAction()
{
    Chain role;
    Chain permid;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	permid = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqSetPermission(role, permid, _tsPerm, _filterPerm, _rightPerm);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::setTableSetPerm()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_tsPerm = *pS;
    }
}

void CegoAdmAction::setFilterPerm()
{
    _filterPerm = Chain(_stringBuf);    
}

void CegoAdmAction::setRightPerm()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_rightPerm = *pS;
    }
}

void CegoAdmAction::removePermAction()
{

    Chain role;
    Chain permid;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	permid = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqRemovePermission(role, permid);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::traceOnAction()
{

    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqUserTrace(user, true);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::traceOffAction()
{

    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqUserTrace(user, false);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;


}

void CegoAdmAction::addArchLogAction()
{

    Chain tableSet;
    Chain archId;
    Chain archPath(_stringBuf);

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	archId = *pS;
    }

        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medAddArchLog(tableSet, archId, archPath);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}


void CegoAdmAction::removeArchLogAction()
{

    Chain tableSet;
    Chain archId;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	archId = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medRemoveArchLog(tableSet, archId);    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::enableArchLogAction()
{

    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medEnableArchLog(tableSet);    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::disableArchLogAction()
{

    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medDisableArchLog(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

}

void CegoAdmAction::storeSysSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_sysSize = pS->asInteger();
    }
}

void CegoAdmAction::storeTmpSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_tmpSize = pS->asInteger();
    }
}

void CegoAdmAction::storeLogSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_logSize = pS->asInteger();
    }
}

void CegoAdmAction::storeLogNumAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_logNum = pS->asInteger();
    }
}


void CegoAdmAction::storeSortAreaAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_sortAreaSize = pS->asLong();
    }
}

void CegoAdmAction::storePrimaryAttr()
{
    _primary = Chain(_stringBuf);   
}

void CegoAdmAction::storeSecondaryAttr()
{
    _secondary = Chain(_stringBuf);   
}

void CegoAdmAction::storeTSRootAttr()
{
    _tsRoot = Chain(_stringBuf);
}

void CegoAdmAction::storeTSTicketAttr()
{
    _tsTicket = Chain(_stringBuf);
}

void CegoAdmAction::storePit()
{
    Chain* pS = getTokenList().First();
    Chain dateVal = Chain(_stringBuf);
    Datetime dt(dateVal, _pitFormat);   
    _pit = dt.asInt(); 
}

void CegoAdmAction::dumpTable()
{

    Chain tableSet;
    Chain table;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	table = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDumpObject(tableSet, CegoObject::TABLE, table);
	
    if ( res == CegoAdminHandler::ADM_INFO )
    {
	
	Chain tableDump;

	_pAH->nextInfo();
	_pAH->getMsg(tableDump);

	cout << tableDump;

    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }    
}

void CegoAdmAction::dumpUIndex()
{

    Chain tableSet;
    Chain table;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	table = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDumpObject(tableSet, CegoObject::UINDEX, table);
	
    if ( res == CegoAdminHandler::ADM_INFO )
    {
	
	Chain tableDump;

	_pAH->nextInfo();
	_pAH->getMsg(tableDump);

	cout << tableDump;

    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }    
}


void CegoAdmAction::handleMedResult(CegoAdminHandler::ResultType res)
{

    Chain msg;
    _pAH->getMsg(msg);
    
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }
    
    CegoOutput output(_admSchema, Chain("llm"));
    output.setRawMode(_rawMode);    

    if ( _resultMode == VERBOSE )
	output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	
	if ( _resultMode == VERBOSE )
	{
	    Chain hostRole;
	    Chain hostName;
	    Chain msg;
	    
	    _pAH->getHostRole(hostRole);
	    _pAH->getHostName(hostName);
	    _pAH->getMsg(msg);

	    Chain normHost;
	    if ( hostName.length() > MAXHOSTLEN-3 )
		normHost = hostName.subChain(1, MAXHOSTLEN-3) + Chain(".. ");
	    else
		normHost = hostName;

	    ListT<CegoFieldValue> fvl;
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, hostRole));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, normHost));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, msg));
	    
	    output.rowOut(fvl);
	}
	else if ( _resultMode == NORMAL )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}

	res = _pAH->nextInfo();
	
    }
    if ( _resultMode == VERBOSE )
	output.tailOut();
    
    _pAH->getMsg(msg);
    
    /* if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	    cout << msg << endl;
    }
    else */
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }
}
