///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdminThread.cc  
// ------------------                                                     
// Cego admin thread class implementation
//                                                         
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2012 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdminThread
// 
// Description: 
//
// Status: QG-2.6
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/Exception.h>
#include <lfcbase/Host.h>
#include <lfcbase/Datetime.h>
#include <lfcbase/AESCrypt.h>
#include <lfcbase/Sleeper.h>
#include <lfcbase/Net.h>

// cego includes
#include "CegoXMLdef.h"
#include "CegoAdminThread.h"
#include "CegoAdminThreadPool.h"
#include "CegoXPorter.h"
#include "CegoTypeConverter.h"
#include "CegoRecoveryManager.h"
#include "CegoBufferPoolEntry.h"

#include <stdlib.h>
#include <string.h>

CegoAdminThread::CegoAdminThread() : Thread()
{
}

CegoAdminThread::CegoAdminThread(CegoAdminThreadPool *pPool, CegoDatabaseManager *pDBMng, CegoDbThreadPool *pDbPool, CegoLogThreadPool *pLogPool) : Thread()
{
    _pDBMng = pDBMng;
    _pPool = pPool;
    _pTabMng = new CegoDistManager(pDBMng);

    _pDbPool = pDbPool;
    _pLogPool = pLogPool;

    _modId = _pDBMng->getModId("CegoAdminThread");
}

CegoAdminThread::~CegoAdminThread()  
{
}

const Chain& CegoAdminThread::lastAction() const
{
    return _lastAction;
}

void* CegoAdminThread::job(void* arg)
{
	
    _idx = *(long*)arg;

    _pTim = new NanoTimer();
    
    _pPool->setTid(_idx, getTid());
    _pTabMng->setThreadId(getTid());

    while ( ! _pPool->isTerminated() )
    {
	
	try
	{

	    _pTim->reset();
	    _pTim->start();

	    _pRequest = _pPool->nextRequest();
	    
	    if ( _pRequest ) 
	    {

#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": serving service request"));
#endif

		_pPool->setState(_idx, CegoAdminThreadPool::CONNECTED);
		_pPool->incNumRequest(_idx);
	
		_pDBMng->increaseActiveAdmThread();			

		CegoAdminHandler* pAH = 0;
		try {
		    pAH = new CegoAdminHandler(_pDBMng, _pRequest);
		    serveSession(pAH);
		    delete pAH;
		}
		catch ( Exception e)
		{	
		    if ( pAH )
			delete pAH;

		    Chain msg;
		    Chain module;
		    int line;
		    
		    Chain exep;
		    while ( e.pop(module, line, msg) )
		    {
			exep += Chain("\n\t") + module + Chain("(") +  Chain(line) + Chain(") : ") + msg;
		    }

		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": Admin session aborted : ") + exep);

		}
		_pDBMng->decreaseActiveAdmThread();
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": service request finished"));
#endif

		_pPool->setState(_idx, CegoAdminThreadPool::READY);

		delete _pRequest;
		
	    }
	    else
	    {
		Sleeper s;
		s.nanoSleep(NETMNG_QUEUE_DELAY);
	    }

	    Chain tableSet;
	    Chain secondary;
	    Chain mediator;
	    Chain user;
	    Chain passwd;

	    // check for asynchronous recovery requests
	    bool doRecovery = _pDBMng->nextRecovery(tableSet);
	    if ( doRecovery )
	    {
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);
		
		try 
		{
		    _pDBMng->increaseActiveAdmThread();
		    
		    CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::REMOTE);
		    long lsn = recoveryMng.recoverTableSet(tableSet, 0);
		    
		    _pDBMng->decreaseActiveAdmThread();
		    _pPool->setState(_idx, CegoAdminThreadPool::READY);
		}
		catch ( Exception e)
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg);
		}
		_pPool->setState(_idx, CegoAdminThreadPool::READY);

	    }

	    // check for asynchronous copy requests
	    int copyId;
	    bool doCopy = _pDBMng->nextCopy(copyId, tableSet, secondary, mediator, user, passwd);
	    if ( doCopy )
	    {
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);
		_pDBMng->increaseActiveAdmThread();

		_pDBMng->setCopyStatus(copyId, Chain("Starting copy"));

		try 
		{

#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) 
				 + Chain(": Copying tableset ") + tableSet + Chain(" to secondary"));
#endif
    

		    if ( _pDBMng->getTableSetRunState(tableSet) == Chain(XML_OFFLINE_VALUE) )
		    {
			copyTableSet(copyId, tableSet, secondary, mediator, user, passwd, false);
		    }
		    else if ( _pDBMng->getTableSetRunState(tableSet) == Chain(XML_ONLINE_VALUE) )
		    {
			_pTabMng->beginBackup(tableSet, Chain("Tableset copy"));		    
			copyTableSet(copyId, tableSet, secondary, mediator, user, passwd, true);			
			_pTabMng->endBackup(tableSet, Chain("Tableset copy"), false);
		    }
		    else
		    {
			Chain msg = Chain("Invalid runstate ") + _pDBMng->getTableSetRunState(tableSet) + Chain(" for tableset copy");
			throw Exception(EXLOC, msg);
		    }

		    _pDBMng->setCopyStatus(copyId, "Copy finished");
		}
		catch ( Exception e)
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->setCopyStatus(copyId, msg);		
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg );
		}
		_pDBMng->decreaseActiveAdmThread();
		_pPool->setState(_idx, CegoAdminThreadPool::READY);
		    
	    }

	    _pTim->stop();
	    _pPool->addThreadIdle(_idx, _pTim->getSum());

	}
	catch ( Exception e)
	{
	    Chain msg;
	    e.pop(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg );
	}
    }
    return 0;
}

void CegoAdminThread::serveSession(CegoAdminHandler *pAH)
{   
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": serving session"));
#endif
    

    if ( pAH->acceptSession() )
    {

	if ( _pDBMng->checkAdminUser(pAH->getUser(), pAH->getPassword()) == false )
	{
	    Chain msg = Chain("Invalid user or password");
	    pAH->sendError(msg);
	    return;
	}
	else
	{
	    Chain msg = Chain("Access granted");
	    pAH->sendResponse(msg);
	}

	_user = pAH->getUser();
	_password = pAH->getPassword();

	bool isTerminated = false;
	
	while ( isTerminated == false && _pPool->isTerminated() == false)
	{
	    
	    CegoAdminHandler::RequestType reqType;
	    reqType = pAH->acceptRequest();

	    _pTim->stop();
	    _pPool->addThreadIdle(_idx, _pTim->getSum());
	    _pTim->reset();
	    _pTim->start();

	    
	    if ( reqType != CegoAdminHandler::REQTIMEOUT )
	    {
		
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);

		_pTim->stop();
		_pPool->addThreadIdle(_idx, _pTim->getSum());
		
		isTerminated = serveRequest(pAH, reqType);

		_pTim->reset();
		_pTim->start();
		
		_pPool->setState(_idx, CegoAdminThreadPool::CONNECTED);
	    }
	    else
	    {

#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": admin client request timeout occured, waitung ..."));
#endif

	    }
	}
    }
}

bool CegoAdminThread::serveRequest(CegoAdminHandler *pAH, CegoAdminHandler::RequestType reqType)
{

    bool isTerminated = false;

    try {
	
	switch ( reqType ) 
	{		    
	case CegoAdminHandler::RETRIEVETABLESET:
	{
	    srvRetrieveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::LOCKINFO:
	{
	    srvLockInfo(pAH);
	    break;
	}
	case CegoAdminHandler::LOCKSTAT:
	{
	    srvLockStat(pAH);
	    break;
	}
	case CegoAdminHandler::POOLINFO:
	{
	    srvPoolInfo(pAH);
	    break;
	}
	case CegoAdminHandler::POOLENTRYLIST:
	{
	    srvPoolEntryList(pAH);
	    break;
	}
	case CegoAdminHandler::THREADINFO:
	{
	    srvThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DBTHREADINFO:
	{
	    srvDbThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::ADMTHREADINFO:
	{
	    srvAdmThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::LOGTHREADINFO:
	{
	    srvLogThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DBTHREADABORT:
	{
	    srvDbThreadAbort(pAH);
	    break;
	}
	case CegoAdminHandler::DBSESSIONINFO:
	{
	    srvDbSessionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::COPYINFO:
	{
	    srvCopyInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DEFINE_TABLESET:
	{
	    srvDefineTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_TABLESET:
	{
	    srvRemoveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::RESET_TABLESET:
	{
	    srvResetTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::CREATE_TABLESET:
	{
	    srvCreateTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::DROP_TABLESET:
	{
	    srvDropTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::START_TABLESET:
	{
	    srvStartTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::STOP_TABLESET:
	{
	    srvStopTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::START_RECOVERY:
	{
	    srvStartRecovery(pAH);	      	
	    break;
	}
	case CegoAdminHandler::STOP_RECOVERY:
	{
	    srvStopRecovery(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DBSPEC:
	{		
	    srvGetDbSpec(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DBINFO:
	{		
	    srvGetDbInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_TSINFO:
	{		
	    srvGetTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DETAILED_TSINFO:
	{		
	    srvGetDetailedTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_TRANSACTION_INFO:
	{		
	    srvGetTransactionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_BUSTAT:
	{		
	    srvGetBUStatInfo(pAH);
	    break;
	}
	case CegoAdminHandler::PROP_TSINFO:
	{		
	    srvPropTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::COPY_TABLESET:
	{
	    srvCopyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::COPY_FILE:
	{
	    srvCopyFile(pAH);
	    break;
	}
	case CegoAdminHandler::GET_NODELIST:
	{		
	    srvGetNodeList(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TABLESETNODE:
	{
	    srvSetTableSetNode(pAH);
	    break;
	}
	case CegoAdminHandler::SET_LSN:
	{
	    srvSetLSN(pAH);
	    break;
	}
	case CegoAdminHandler::SET_RUNSTATE:
	{
	    srvSetRunState(pAH);
	    break;
	}
	case CegoAdminHandler::SET_SYNCSTATE:
	{
	    srvSetSyncState(pAH);
	    break;
	}
	case CegoAdminHandler::EXPORT_TABLESET:
	{
	    srvExportTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::IMPORT_TABLESET:
	{
	    srvImportTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::EXPORT_TABLE:
	{
	    srvExportTable(pAH);
	    break;
	}
	case CegoAdminHandler::IMPORT_TABLE:
	{
	    srvImportTable(pAH);
	    break;
	}
	case CegoAdminHandler::ADD_DATAFILE:
	{
	    srvAddDatafile(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_USER:
	{
	    srvUserInfo(pAH);
	    break;
	}	    
	case CegoAdminHandler::ADD_USER:
	{
	    srvAddUser(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_USER:
	{
	    srvRemoveUser(pAH);
	    break;
	}
	case CegoAdminHandler::CHANGE_PWD:
	{
	    srvChangePwd(pAH);
	    break;
	}
	case CegoAdminHandler::LIST_ROLE:
	{
	    srvListRole(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_ROLE:
	{
	    srvShowRole(pAH);
	    break;
	}
	case CegoAdminHandler::CREATE_ROLE:
	{
	    srvCreateRole(pAH);
	    break;
	}
	case CegoAdminHandler::DROP_ROLE:
	{
	    srvDropRole(pAH);
	    break;
	}
	case CegoAdminHandler::ASSIGN_ROLE:
	{
	    srvAssignRole(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_ROLE:
	{
	    srvRemoveRole(pAH);
	    break;
	}
	case CegoAdminHandler::SET_PERM:
	{
	    srvSetPerm(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_PERM:
	{
	    srvRemovePerm(pAH);
	    break;
	}
	case CegoAdminHandler::USER_TRACE:
	{
	    srvUserTrace(pAH);
	    break;
	}
	case CegoAdminHandler::SYNC_TABLESET:
	{
	    srvSyncTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::INIT_LOGS:
	{
	    srvInitLogs(pAH);
	    break;
	}
	case CegoAdminHandler::ADD_ARCHLOG:
	{
	    srvAddArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_ARCHLOG:
	{
	    srvRemoveArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::BEGIN_BACKUP:
	{
	    srvBeginBackup(pAH);
	    break;
	}
	case CegoAdminHandler::END_BACKUP:
	{
	    srvEndBackup(pAH);
	    break;
	}
	case CegoAdminHandler::RECOVER:
	{
	    srvRecover(pAH);
	    break;
	}
	case CegoAdminHandler::SECSWITCH:
	{
	    srvSecSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::VERIFY_TABLESET:
	{
	    srvVerifyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::CORRECT_TABLESET:
	{
	    srvCorrectTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::DUMP_OBJECT:
	{
	    srvDumpObject(pAH);
	    break;
	}
	case CegoAdminHandler::SET_LOGMNG:
	{
	    srvSetLogMng(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_LOGMNG:
	{
	    srvShowLogMng(pAH);
	    break;
	}
	case CegoAdminHandler::SET_CHECKPOINT:
	{
	    srvSetCheckpoint(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSINITFILE:
	{
	    srvSetTSInitFile(pAH);
	    break;
	}

	case CegoAdminHandler::SET_TSSYSSIZE:
	{
	    srvSetTSSysSize(pAH);
	    break;
	}

	case CegoAdminHandler::SET_TSTMPSIZE:
	{
	    srvSetTSTmpSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSROOTPATH:
	{
	    srvSetTSRootPath(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSTICKET:
	{
	    srvSetTSTicket(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSLOGNUM:
	{
	    srvSetTSLogNum(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSLOGSIZE:
	{
	    srvSetTSLogSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSSORTAREASIZE:
	{
	    srvSetTSSortAreaSize(pAH);
	    break;
	}

	// mediator requests
	case CegoAdminHandler::MED_NOTIFY:
	{
	    medNotify(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DEFINE_TABLESET:
	{
	    medDefineTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_REMOVE_TABLESET:
	{
	    medRemoveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RESET_TABLESET:
	{
	    medResetTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CREATE_TABLESET:
	{
	    medCreateTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DROP_TABLESET:
	{
	    medDropTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_START_TABLESET:
	{
	    medStartTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_STOP_TABLESET:
	{
	    medStopTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_COPY_TABLESET:
	{
	    medCopyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SWITCH_TABLESET:
	{
	    medSwitchTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ADD_ARCHLOG:
	{
	    medAddArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_REMOVE_ARCHLOG:
	{
	    medRemoveArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ADD_DATAFILE:
	{
	    medAddDatafile(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_TABLESETLIST:
	{
	    medGetTableSetList(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_DETAILEDTABLESETINFO:
	{
	    medGetDetailedTableSetInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_OBJECTINFO:
	{
	    medGetObjectInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_TRANSACTIONINFO:
	{
	    medGetTransactionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_BUSTAT:
	{
	    medGetBUStatInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RESET_BUSTAT:
	{
	    medResetBUStat(pAH);
	    break;
	}

	case CegoAdminHandler::MED_BEGIN_BACKUP:
	{
	    medBeginBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_END_BACKUP:
	{
	    medEndBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RECOVER:
	{
	    medRecover(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SECSWITCH:
	{
	    medSecSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SECRELOCATE:
	{
	    medSecRelocate(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CHECK_TABLESET:
	{
	    medCheckTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_VERIFY_TABLESET:
	{
	    medVerifyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CORRECT_TABLESET:
	{
	    medCorrectTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ENABLE_ARCHLOG:
	{
	    medEnableArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DISABLE_ARCHLOG:
	{
	    medDisableArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::SEC_MEDSWITCH:
	{
	    secMedSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::SEC_MEDRELOCATE:
	{
	    secMedRelocate(pAH);
	    break;
	}
	case CegoAdminHandler::SESSION_CLOSE:
	{
	    isTerminated=true;
	    
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain(" Thread ") + Chain(_idx) + Chain(": terminating session"));
#endif
	    pAH->sendResponse(Chain("Session Closed"));
	    
	    break;
	}
	default:
	{
	    pAH->sendError(Chain("Unknown Request"));
	}    
	}
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	pAH->sendError(msg);
    }
    
    return isTerminated;

}
void CegoAdminThread::copyTableSet(int copyId, const Chain& tableSet, const Chain& secondary, const Chain& mediator, const Chain& user, const Chain& passwd, bool copyOnline)
{

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Copying tableset ") +  tableSet + Chain("..."));
#endif


    int adminPort;
    _pDBMng->getAdminPort(adminPort);

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Connecting to secondary ") +  secondary + Chain("..."));
#endif

    Net n (  NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN );
    
    NetHandler *pNS = n.connect(secondary, adminPort);
    CegoAdminHandler *pAHS = new CegoAdminHandler(_pDBMng, pNS);

    pAHS->requestSession(user, passwd, false);

    NetHandler *pNM = n.connect(mediator, adminPort);

    CegoAdminHandler *pAHM = new CegoAdminHandler(_pDBMng, pNM);
    pAHM->requestSession(user, passwd, false);

    CegoAdminHandler::ResultType res;	
    
    res = pAHS->reqSetSyncState(tableSet, XML_ON_COPY_VALUE);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);      
    }    
    
    res = pAHM->reqSetSyncState(tableSet, XML_ON_COPY_VALUE);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);      
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);      
	
    }    

    // request system files

    Chain dbSysFileName = _pDBMng->getSysFileName(tableSet);
    Chain dbTempFileName = _pDBMng->getTmpFileName(tableSet);
    Chain tsTicket = _pDBMng->getTSTicket(tableSet);
    
    res = pAHS->reqCopyFile(dbSysFileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    
    
    res = pAHS->reqCopyFile(dbTempFileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    

    if ( copyOnline )
    {
	res = pAHS->reqCopyFile(tsTicket);
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}
    }
    else
    {
	res = pAHS->reqSetLSN(tableSet, _pDBMng->getCommittedLSN(tableSet));
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}
    }

    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> sizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, sizeList);
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, sizeList);
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, sizeList);
    
    Chain *pFileName = dfList.First();
    
    while ( pFileName  ) 
    {
	
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Copying datafile ") + *pFileName + Chain(" ..."));

	if ( _pDBMng )
	{
	    _pDBMng->setCopyStatus(copyId, Chain("Copying ") + *pFileName );		
	}
	
	CegoAdminHandler::ResultType res;
	res = pAHS->reqCopyFile( *pFileName, copyId);
	
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}    
	   
	pFileName = dfList.Next();
	    
    }

    res = pAHS->reqInitLogs(tableSet);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    


    res = pAHS->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    


    res = pAHM->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    

    _pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);

    pAHS->closeSession();
    pAHM->closeSession();
    
    delete pAHS;
    delete pAHM;

    delete pNS;
    delete pNM;
    		
}


void CegoAdminThread::srvRetrieveTableSet(CegoAdminHandler *pAH)
{
    
    Chain masterHost;
    pAH->getHostName( masterHost );
    

    CegoAdminHandler* pMasterAH = 0;

    pMasterAH = getSession(Chain("master"), masterHost, pAH->getUser(), pAH->getPassword());

    Chain dbSpec;
    CegoAdminHandler::ResultType res = pMasterAH->reqGetDbSpec(dbSpec);
    
    Chain msg;
    pMasterAH->getMsg(msg);

    closeSession(pMasterAH);

    if ( pAH->syncWithInfo(Chain("primary"), masterHost, msg, 0) == false )
	return;
        
    if ( res != CegoAdminHandler::ADM_OK )
	throw Exception(EXLOC, Chain("Admin action failed"));
    
    XMLSuite xml(dbSpec);
    Document *pDoc = new Document;
    xml.setDocument(pDoc);
    
    try {

	xml.parse();
    }
    catch ( Exception e )
    {
	Chain xmlmsg;
	e.pop(xmlmsg);
	Chain msg = Chain("XML parse error at line ") + Chain( xml.getLineNo()) + Chain(" : ") + xmlmsg ;
	_pDBMng->log(_modId, Logger::LOGERR, msg);
	throw e;
    }

    Element *pDBElement = pDoc->getRootElement();

    ListT<Element*> tsList = pDBElement->getChildren(XML_TABLESET_ELEMENT);

    ListT<Chain> localList;

    _pDBMng->getTableSetList(localList);

    Element **pTS = tsList.First();
    while ( pTS ) 
    {

	Chain tableSet = (*pTS)->getAttributeValue(XML_NAME_ATTR);

	if ( localList.Find( tableSet ) == 0 )
	{
	    
	    _pDBMng->setTableSetInfo(tableSet, (*pTS)->createClone() );
	        	 
	}   
	pTS = tsList.Next();
	
    }
   
    pAH->sendResponse(Chain("Tableset info retrieved"));

}


void CegoAdminThread::srvLockInfo(CegoAdminHandler *pAH)
{

    try
    {
	Chain lockCat;
	pAH->getLockCat( lockCat );
	
	Element *pLockInfo = new Element(XML_LOCKINFO_ELEMENT);
	Element *pLockEntry;
	
	Chain lockName;
	long lockCount;
	long numRdLock;
	long numWrLock;
	long sumRdDelay;
	long sumWrDelay;
	
	if ( lockCat == Chain(XML_LOCK_THREADPOOL_VALUE) )
	{
	    _pPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));    
	    pLockInfo->addContent(pLockEntry);
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDbPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pLogPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDBMng->getDBMLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDBMng->getXMLLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);
	}  
	else if ( lockCat == Chain(XML_LOCK_TABLESET_VALUE) )
	{
	    Host h;
	    ListT<Chain> tsList;    
	    _pDBMng->getActiveTableSet(h.getName(), tsList);
	    
	    Chain *pTabSet = tsList.First();
	    while ( pTabSet )
	    {
		int tabSetId = _pDBMng->getTabSetId(*pTabSet);
		
		pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
		_pTabMng->getTSLockStat(tabSetId, lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
		pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
		pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		pLockInfo->addContent(pLockEntry);    
		
		
		pTabSet = tsList.Next();
	    }
	}
	else 
	{
	    
	    int low = 0;
	    int high = 0;
	    
	    if ( lockCat == Chain(XML_LOCK_REC_VALUE) )
	    {
		low = 0;
		high = _pDBMng->getNumRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_RBREC_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema();
		high = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_SYSREC_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_SYSPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_DATAPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema() 
		    + _pDBMng->getNumDataPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_IDXPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_RBPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema();		
	    }
	    else if ( lockCat == Chain(XML_LOCK_DATAFILE_VALUE) )
	    {
		
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema() 
		    + _pDBMng->getNumDataFileSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_BUFFERPOOL_VALUE) )
	    {
		
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema() 
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema()
		    + _pDBMng->getNumDataFileSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema() 
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema() 
		    + _pDBMng->getNumDataFileSema() 
		    + _pDBMng->getNumBufferPoolSema();
	    }
	    else
	    {		
		throw Exception(EXLOC, "Unknown lock cat");
	    }
	    

	    int semId;	    
	    for (  semId = low; semId < high; semId++ )
	    {
		pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
		_pTabMng->getLHLockStat(semId, lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
		pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
		pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		pLockInfo->addContent(pLockEntry);    
	    }
	}
	
	_lastAction = Chain("LockInfo");
	pAH->sendResponse(Chain("LockInfo"), 
		      pLockInfo);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	pAH->sendError(msg);	
    }

}


void CegoAdminThread::srvLockStat(CegoAdminHandler *pAH)
{

    try
    {
	
	Element *pLockStat = new Element(XML_LOCKSTAT_ELEMENT);
	Element *pLockEntry;
	
	Chain lockGroup;
	Chain lockName;
	int numLock;
	long lockCount;
	long numRdLock;
	long numWrLock;
	long sumRdDelay;
	long sumWrDelay;
	

	_pPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));    
	pLockStat->addContent(pLockEntry);
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDbPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pLogPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getDBMLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getXMLLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);
	
	
	lockGroup = Chain(RECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(RBRECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(SYSRECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(SYSPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(DATAPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(IDXPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(RBPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(DATAFILELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(POOLLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	_lastAction = Chain("LockStat");
	pAH->sendResponse(Chain("LockStat"), 
			  pLockStat);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	pAH->sendError(msg);	
    }
}

void CegoAdminThread::srvPoolInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("PoolInfo");
    
    int pageSize;
    long numTotal;
    long numUsed;
    long numFree;
    long numDirty;
    long numFixes;
    long numPersistent;
    long numNoSync;
    long numDiskRead;
    long numDiskWrite;
    long hitRate;
    long readDelay;
    long writeDelay;
    long curFixCount;
    long maxFixCount;
    
    int statStart;
    
    _pDBMng->poolInfo(pageSize, numTotal, numUsed, numFree, numDirty, numFixes, numPersistent, numNoSync, numDiskRead, numDiskWrite, hitRate, readDelay, writeDelay, curFixCount, maxFixCount, statStart);
    
    Element *pPoolInfo = new Element(XML_POOLINFO_ELEMENT);
    
    pPoolInfo->setAttribute(XML_PAGESIZE_ATTR, Chain(pageSize));
    pPoolInfo->setAttribute(XML_NUMTOTAL_ATTR, Chain(numTotal));
    pPoolInfo->setAttribute(XML_NUMUSED_ATTR, Chain(numUsed));
    pPoolInfo->setAttribute(XML_NUMFREE_ATTR, Chain(numFree));
    pPoolInfo->setAttribute(XML_NUMDIRTY_ATTR, Chain(numDirty));
    pPoolInfo->setAttribute(XML_NUMFIXES_ATTR, Chain(numFixes));
    pPoolInfo->setAttribute(XML_NUMPERSISTENT_ATTR, Chain(numPersistent));
    pPoolInfo->setAttribute(XML_NUMNOSYNC_ATTR, Chain(numNoSync));
    pPoolInfo->setAttribute(XML_NUMDISKREAD_ATTR, Chain(numDiskRead));
    pPoolInfo->setAttribute(XML_NUMDISKWRITE_ATTR, Chain(numDiskWrite));
    pPoolInfo->setAttribute(XML_BPHITRATE_ATTR, Chain(hitRate));
    pPoolInfo->setAttribute(XML_READDELAY_ATTR, Chain(readDelay));
    pPoolInfo->setAttribute(XML_WRITEDELAY_ATTR, Chain(writeDelay));
    pPoolInfo->setAttribute(XML_CURFIXCOUNT_ATTR, Chain(curFixCount));
    pPoolInfo->setAttribute(XML_MAXFIXCOUNT_ATTR, Chain(maxFixCount));

    Datetime ts(statStart);
    pPoolInfo->setAttribute(XML_STATSTART_ATTR, ts.asChain(BUPMNG_STATDTFORMAT));
        
    pAH->sendResponse(Chain("PoolInfo"), 
		      pPoolInfo);       	
    
}


void CegoAdminThread::srvPoolEntryList(CegoAdminHandler *pAH)
{

    _lastAction = Chain("PoolEntryList");
     
    ListT<CegoBufferPoolEntry> entryList;

    _pDBMng->getPoolEntryList(entryList);
        
    Element *pPoolEntryList = new Element(XML_POOLENTRYLIST_ELEMENT);
    
    CegoBufferPoolEntry *pEntry = entryList.First();

    while ( pEntry )
    {

	Element *pEntryElement = new Element(XML_POOLENTRY_ELEMENT);

	pEntryElement->setAttribute(XML_POS_ATTR, Chain(pEntry->getPos()));
	pEntryElement->setAttribute(XML_OCCSTATE_ATTR, Chain(pEntry->getOccState()));

	if ( pEntry->isDirty() )
	    pEntryElement->setAttribute(XML_ISDIRTY_ATTR, Chain("y"));
	else
	    pEntryElement->setAttribute(XML_ISDIRTY_ATTR, Chain("n"));

	pEntryElement->setAttribute(XML_NUMFIXES_ATTR, Chain(pEntry->getNumFixes()));



	pEntryElement->setAttribute(XML_TSID_ATTR, Chain(pEntry->getTabSetId()));
	pEntryElement->setAttribute(XML_FILEID_ATTR, Chain(pEntry->getFileId()));
	pEntryElement->setAttribute(XML_PAGEID_ATTR, Chain(pEntry->getPageId()));
	pEntryElement->setAttribute(XML_FIXSTAT_ATTR, Chain(pEntry->getFixStat()));
    
	pPoolEntryList->addContent(pEntryElement);	

	pEntry = entryList.Next();

    }

    pAH->sendResponse(Chain("PoolEntryList"), 
		      pPoolEntryList);
}


void CegoAdminThread::srvThreadInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("ThreadInfo");
		    
    int numDbThread;
    int numAdmThread;
    int numLogThread;
    int actDbThread;
    int actAdmThread;
    int actLogThread;
    
    
    _pDBMng->getThreadInfo(numDbThread, numAdmThread, numLogThread, actDbThread, actAdmThread, actLogThread);
    
    Element *pDbInfo = new Element(XML_THREADINFO_ELEMENT);	
    
    pDbInfo->setAttribute(XML_NUMDBTHREAD_ATTR, Chain(numDbThread));
    pDbInfo->setAttribute(XML_NUMADMTHREAD_ATTR, Chain(numAdmThread));
    pDbInfo->setAttribute(XML_NUMLOGTHREAD_ATTR, Chain(numLogThread));
    
    pDbInfo->setAttribute(XML_ACTIVEDBTHREAD_ATTR, Chain(actDbThread));
    pDbInfo->setAttribute(XML_ACTIVEADMTHREAD_ATTR, Chain(actAdmThread));
    pDbInfo->setAttribute(XML_ACTIVELOGTHREAD_ATTR, Chain(actLogThread));
    
    pAH->sendResponse(Chain("ThreadInfo"), 
		      pDbInfo);    
}

void CegoAdminThread::srvDbThreadInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("DBThreadInfo");
        
    int poolLimit = _pDbPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoDbThreadPool::ThreadState state;
	long numRequest;
	long numQueryRequest;
	long threadLoad;
	long allocatedSortArea;
	Chain lastAction;
	_pDbPool->getThreadInfo(i, numRequest, numQueryRequest, threadLoad, allocatedSortArea, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));	
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_NUMQUERYREQUEST_ATTR, Chain(numQueryRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));
	pN->setAttribute(XML_ALLOCATEDSORT_ATTR, Chain(allocatedSortArea));
	
	if ( state == CegoDbThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoDbThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoDbThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	pThreadInfo->addContent(pN);
    }
    
    pAH->sendResponse(Chain("DBThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvAdmThreadInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("AdminThreadInfo");    
    
    int poolLimit = _pPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoAdminThreadPool::ThreadState state;
	long numRequest;
	long threadLoad;
	Chain lastAction;
	_pPool->getThreadInfo(i, numRequest, threadLoad, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));
	
	if ( state == CegoAdminThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoAdminThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoAdminThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	pThreadInfo->addContent(pN);
    }


    pAH->sendResponse(Chain("AdminThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvLogThreadInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("LogThreadInfo");    

    int poolLimit = _pLogPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoLogThreadPool::ThreadState state;
	long numRequest;
	Chain lastAction;
	long threadLoad;
	_pLogPool->getThreadInfo(i, numRequest, threadLoad, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));

	if ( state == CegoLogThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoLogThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoLogThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	pThreadInfo->addContent(pN);
    }
    

    pAH->sendResponse(Chain("LogThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvDbThreadAbort(CegoAdminHandler *pAH)
{
    int threadId;
    pAH->getThreadId(threadId);

    _lastAction = Chain("ThreadAbort");
    _pDbPool->abortThread(threadId);
    pAH->sendResponse(Chain("Thread aborted"));		
}	


void CegoAdminThread::srvDbSessionInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("DbSessionInfo");    

    Element *pSessionInfo = _pDBMng->getSessionInfo(NETMNG_DBHANDLE_TTL);
    
    pAH->sendResponse(Chain("DbSessionInfo"), 
		      pSessionInfo);
}


void CegoAdminThread::srvCopyInfo(CegoAdminHandler *pAH)
{

    _lastAction = Chain("CopyInfo");    
        
    Element *pCopyInfo = _pDBMng->getCopyInfo();
    
    pAH->sendResponse(Chain("CopyInfo"), 
		      pCopyInfo);
}

void CegoAdminThread::srvDefineTableSet(CegoAdminHandler *pAH)
{   

    Chain tableSet;
    Chain tsRoot;
    Chain tsTicket;

    Chain primary;
    Chain secondary;
    Chain mediator;

    int sysFid;
    int tmpFid;
    
    int sysSize;
    int tmpSize;
    int logFileSize;
    int logFileNum;

    long sortAreaSize;

    pAH->getTableSet(tableSet);
    pAH->getTableSetRoot(tsRoot);
    pAH->getTableSetTSTicket(tsTicket);

    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getMediator(mediator);

    pAH->getSysFileId(sysFid);
    pAH->getTmpFileId(tmpFid);

    pAH->getSysSize(sysSize);
    pAH->getTmpSize(tmpSize);

    pAH->getLogFileSize(logFileSize);
    pAH->getLogFileNum(logFileNum);

    pAH->getSortAreaSize(sortAreaSize);

    _lastAction = Chain("Define TableSet") + tableSet;
    
    _pDBMng->addTableSetDef(tableSet, tsRoot, tsTicket, primary, secondary, mediator, sysFid, tmpFid, sysSize, tmpSize,
			    logFileSize, logFileNum, sortAreaSize); 
    

    pAH->sendResponse(Chain("Table set defined"));
}


void CegoAdminThread::srvRemoveTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _pDBMng->rmTableSetDef(tableSet);
    _lastAction = Chain("Remove TableSet ") + tableSet;    
    pAH->sendResponse(Chain("Table set removed"));
}

void CegoAdminThread::srvResetTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _pTabMng->resetTableSet(tableSet);
    _lastAction = Chain("Reset TableSet ") + tableSet;    
    pAH->sendResponse(Chain("Table set reset"));
}



void CegoAdminThread::srvCreateTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Create TableSet ") + tableSet;
    _pTabMng->createTableSet(tableSet);
    pAH->sendResponse(Chain("Table set created"));
}


void CegoAdminThread::srvDropTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Drop TableSet ") + tableSet;    
    _pTabMng->dropTableSet(tableSet);
    pAH->sendResponse(Chain("Table set dropped"));
}


void CegoAdminThread::srvStartTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;   
    pAH->getTableSet(tableSet);
    bool cleanIt = pAH->getCleanup();
    bool forceload = pAH->getForceload();
    
    Chain secondary = _pDBMng->getSecondary(tableSet);

    _lastAction = Chain("Start TableSet ") + tableSet;    

    _pTabMng->startDistTableSet(tableSet, secondary, cleanIt);
    
    if ( forceload )
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	_pDbPool->loadObjects(tabSetId);
    }

    pAH->sendResponse(Chain("Tableset started"));
}

void CegoAdminThread::srvStopTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Stop TableSet ") + tableSet;    

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDbPool->unloadObjects(tabSetId);

    _pTabMng->stopTableSet(tableSet, true);

    pAH->sendResponse(Chain("Tableset stopped"));
}


void CegoAdminThread::srvVerifyTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Verify TableSet ") + tableSet;

    _pTabMng->setActiveUser(tableSet, _user, _password);

    ListT<Chain> tableList; 
    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);

    Host h;
    Chain *pTable = tableList.First();
    while ( pTable )
    {
	Element* pVerificationInfo = _pTabMng->verifyTable(tableSet, *pTable);
	if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of table ") + *pTable, pVerificationInfo) == false)
	    return;

	pTable = tableList.Next();		
    }

    ListT<Chain> viewList; 
    _pTabMng->getDistObjectList(tableSet, CegoObject::VIEW, viewList);

    Chain *pView = tableList.First();
    while ( pView )
    {
	Element* pVerificationInfo = _pTabMng->verifyView(tableSet, *pView);
	if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of view ") + *pView, pVerificationInfo) == false)
	    return;

	pView = viewList.Next();		
    }

    ListT<Chain> procList; 
    _pTabMng->getDistObjectList(tableSet, CegoObject::PROCEDURE, procList);

    Chain *pProc = procList.First();
    while ( pProc )
    {
	Element* pVerificationInfo = _pTabMng->verifyProcedure(tableSet, *pProc);
	if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of procedure ") + *pProc, pVerificationInfo) == false)
	    return;

	pProc = procList.Next();		
    }

    pAH->sendResponse(Chain("Tableset verified"));

}


void CegoAdminThread::srvCorrectTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Correct TableSet ") + tableSet;

    _pTabMng->setActiveUser(tableSet, _user, _password);

    ListT<Chain> tableList; 
    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);

    Host h;
    Chain *pTable = tableList.First();
    while ( pTable )
    {
	Element* pVerificationInfo = _pTabMng->correctTable(tableSet, *pTable);
	if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Correction of table ") + *pTable, pVerificationInfo) == false)
	    return;

	pTable = tableList.Next();		
    }

    pAH->sendResponse(Chain("Tableset corrected"));

}


void CegoAdminThread::srvDumpObject(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    Chain objName;
    CegoObject::ObjectType objType;
    pAH->getTableSet(tableSet);
    pAH->getObjName(objName);
    pAH->getObjType(objType);
    
    _lastAction = Chain("Dump object ") + objName;    

    int tabSetId = _pDBMng->getTabSetId(tableSet);    
    CegoTypeConverter tc;

    Chain objDump;
    _pTabMng->dumpObject(tabSetId, objName, objType, objDump);

    cout << "Objectdump=" << objDump << endl;

    pAH->sendData(objDump);
}

void CegoAdminThread::srvSetLogMng(CegoAdminHandler *pAH)
{   
    Chain progName;
    int timeout;
    pAH->getLogManager(progName);
    pAH->getLogManagerTimeout(timeout);
    
    _lastAction = Chain("set log manager");    

    _pDBMng->setArchRestoreProg(progName);
    _pDBMng->setArchRestoreTimeout(timeout);

    pAH->sendResponse(Chain("Log Manager set"));
    
}

void CegoAdminThread::srvShowLogMng(CegoAdminHandler *pAH)
{   
    
    _lastAction = Chain("show log manager");    

    Chain progName = _pDBMng->getArchRestoreProg();
    int timeout = _pDBMng->getArchRestoreTimeout();
    
    Element *pLMI = new Element(XML_LOGMNGINFO_ELEMENT);
    pLMI->setAttribute(XML_LOGMNGPROG_ATTR, progName);
    pLMI->setAttribute(XML_LOGMNGTIMEOUT_ATTR, Chain(timeout));
    
    pAH->sendResponse(Chain("Log Manager Info"), pLMI);
    
}


void CegoAdminThread::srvSetCheckpoint(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int tmout;
    pAH->getTimeoutValue(tmout);
    
    _lastAction = Chain("Set checkpoint for ") + tableSet;    

    _pDBMng->setCheckpointInterval(tableSet, tmout);
    
    pAH->sendResponse(Chain("Checkpoint interval set"));
    
}

void CegoAdminThread::srvSetTSInitFile(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain initFile;
    pAH->getTSInitFile(initFile);
    
    _lastAction = Chain("Set initfile for ") + tableSet;    

    _pDBMng->setTSInitFile(tableSet, initFile);
    
    pAH->sendResponse(Chain("Initfile set"));
    
}


void CegoAdminThread::srvSetTSSysSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int sysSize;
    pAH->getTSSysSize(sysSize);
    
    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }

    _lastAction = Chain("Set system size for ") + tableSet;
    
    _pDBMng->setTSSysSize(tableSet, sysSize);
    
    pAH->sendResponse(Chain("System size set"));
    
}

void CegoAdminThread::srvSetTSTmpSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int tmpSize;
    pAH->getTSTmpSize(tmpSize);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set temp size for ") + tableSet;    

    _pDBMng->setTSTmpSize(tableSet, tmpSize);
    
    pAH->sendResponse(Chain("Temp size set"));
    
}

void CegoAdminThread::srvSetTSRootPath(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain tsRoot;
    pAH->getTSRootPath(tsRoot);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
        
    _lastAction = Chain("Set tsroot for ") + tableSet;    

    _pDBMng->setTSRoot(tableSet, tsRoot);
    
    pAH->sendResponse(Chain("TSRoot set"));
    
}

void CegoAdminThread::srvSetTSTicket(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain tsTicket;
    pAH->getTSTicket(tsTicket);
    
    _lastAction = Chain("Set tsticket for ") + tableSet;    

    _pDBMng->setTSTicket(tableSet, tsTicket);
    
    pAH->sendResponse(Chain("TSTicket set"));
    
}

void CegoAdminThread::srvSetTSLogNum(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int logNum;
    pAH->getTSLogNum(logNum);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set logNum for ") + tableSet;    

    _pDBMng->setTSLogNum(tableSet, logNum);
    
    pAH->sendResponse(Chain("Log number set"));
    
}

void CegoAdminThread::srvSetTSLogSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int logSize;
    pAH->getTSLogSize(logSize);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set logSize for ") + tableSet;    

    _pDBMng->setTSLogSize(tableSet, logSize);
    
    pAH->sendResponse(Chain("Log size set"));
    
}

void CegoAdminThread::srvSetTSSortAreaSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int sortAreaSize;
    pAH->getTSSortAreaSize(sortAreaSize);
    
    _lastAction = Chain("Set sort area size for ") + tableSet;    

    _pDBMng->setTSSortAreaSize(tableSet, sortAreaSize);
    
    pAH->sendResponse(Chain("Sort area size set"));
    
}

void CegoAdminThread::srvStartRecovery(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Start Recovery for ") + tableSet;    
    _pDBMng->startRecovery(tableSet);
   
    pAH->sendResponse(Chain("Starting recovery ..."));
}

void CegoAdminThread::srvStopRecovery(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);

    _lastAction = Chain("Stop Recovery for ") + tableSet;    
	
    if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
    {
	_pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
	
	while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
	{
	    Sleeper s;
	    s.secSleep(LOGMNG_RECOVERY_DELAY);
	}
    }

    pAH->sendResponse(Chain("Stopped recovery"));
}      	

void CegoAdminThread::srvGetDbSpec(CegoAdminHandler *pAH)
{
    pAH->sendResponse(Chain("Copy of DbSpec complete"), 
		      _pDBMng->getDbSpec());       	
}

void CegoAdminThread::srvGetDbInfo(CegoAdminHandler *pAH)
{
    pAH->sendResponse(Chain(CEGO_PRODUCT) + Chain(" ") + Chain(CEGO_VERSION));
}

void CegoAdminThread::srvCopyTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain secondary;
    pAH->getSecondary(secondary);
    Chain mediator;
    pAH->getMediator(mediator);

    _pDBMng->startCopy(tableSet, secondary, mediator, pAH->getUser(), pAH->getPassword(), Chain("Copy requested"));

    pAH->sendResponse(Chain("Starting asynchronous copy"));
}
       	
void CegoAdminThread::srvCopyFile(CegoAdminHandler *pAH)
{
    Chain fileName;
    pAH->getFileName(fileName);
    int fileSize;
    pAH->getFileSize(fileSize);

    File copyFile(fileName);
    
    copyFile.open(File::WRITE);
    
    NetHandler *pN = pAH->getNetHandle();
    
    Chain msg = Chain("File <") + fileName + Chain("> (") + Chain(fileSize) + Chain(" bytes) expected to receive"); 
    pAH->sendResponse(msg);
    
    int recvLen = 0;
    while ( recvLen < fileSize )
    {
	
	pN->readMsg();
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Received ") + Chain(pN->getMsgSize()) + Chain(" bytes"));
#endif	

	copyFile.writeByte(pN->getMsg(), pN->getMsgSize());
	recvLen += pN->getMsgSize();
	
	pN->sendAck();
	
    }
    copyFile.close();       	
}

void CegoAdminThread::srvGetNodeList(CegoAdminHandler *pAH)
{
    Element* pNodeInfo = _pDBMng->getNodeList();     
    _lastAction = Chain("GetNodeList");
    pAH->sendResponse(Chain("Node Info"), 
		      pNodeInfo);
}

void CegoAdminThread::srvGetTSInfo(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTSInfo = _pDBMng->getTableSetInfo(tableSet); 
    
    _lastAction = Chain("GetTSInfo");
    pAH->sendResponse(Chain("TS Info"), 
		      pTSInfo);
}

void CegoAdminThread::srvGetDetailedTSInfo(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTSInfo = getDetailedTSInfo(tableSet); 
    
    _lastAction = Chain("GetDetailedTSInfo");
    pAH->sendResponse(Chain("Detailed TS Info"), 
		      pTSInfo);
}

void CegoAdminThread::srvGetTransactionInfo(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTAInfo = getTransactionInfo(tableSet); 
    
    _lastAction = Chain("GetTransactionInfo");
    pAH->sendResponse(Chain("Transaction Info"), 
		      pTAInfo);
}

void CegoAdminThread::srvGetBUStatInfo(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pBUSInfo = getBUStatInfo(tableSet); 
    
    _lastAction = Chain("GetBUStatInfo");
    pAH->sendResponse(Chain("BUStat Info"), 
		      pBUSInfo);
}




void CegoAdminThread::srvPropTSInfo(CegoAdminHandler *pAH)
{
    
    Element *pTSInfo = pAH->getTSInfo();

    if ( pTSInfo )
    {
	Chain tableSet = pTSInfo->getAttributeValue(XML_NAME_ATTR);
	_pDBMng->setTableSetInfo(tableSet, pTSInfo->createClone()); 
    }
    pAH->sendResponse(Chain("Propagate TS Info"));
    
}


void CegoAdminThread::srvSetTableSetNode(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain primary;
    Chain secondary;
    Chain mediator;
    
    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getMediator(mediator);
    
    _pDBMng->setPrimary(tableSet, primary);
    _pDBMng->setSecondary(tableSet, secondary);
    _pDBMng->setMediator(tableSet, mediator);
    
    pAH->sendResponse(Chain("Tableset nodes set"));
}

void CegoAdminThread::srvSetLSN(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    long lsn;
    pAH->getLSN(lsn);
    
    _pDBMng->setTableSetRunState(tableSet, XML_OFFLINE_VALUE);
    _pDBMng->setCommittedLSN(tableSet, lsn);
    
    pAH->sendResponse(Chain("Tableset LSN set"));
}


void CegoAdminThread::srvSetRunState(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain runState;
    pAH->getRunState(runState);
    
    _pDBMng->setTableSetRunState(tableSet, runState);
    
    pAH->sendResponse(Chain("Tableset runstate set"));
}

void CegoAdminThread::srvSetSyncState(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain syncState;
    pAH->getSyncState(syncState);
    
    _pDBMng->setTableSetSyncState(tableSet, syncState);
    
    pAH->sendResponse(Chain("Tableset syncstate set"));
}

void CegoAdminThread::srvExportTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain expFile;
    bool isStructure;
    Chain expMode;

    pAH->getTableSet(tableSet);
    pAH->getFileName(expFile);
    pAH->getIsStructure(isStructure);
    pAH->getMode(expMode);


    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online for export");
    }

    _lastAction = Chain("Export tableset ") + tableSet;    
        	
    _pTabMng->setActiveUser(tableSet, _user, _password);
    CegoXPorter exp(_pTabMng);

    if ( expMode == Chain(XML_XML_VALUE) )
    {
	exp.xmlExportTableSet(tableSet, isStructure, expFile);
    }
    else if ( expMode == Chain(XML_BIN_VALUE) )
    {
	exp.binExportTableSet(tableSet, isStructure, expFile, false);
    }
    else if ( expMode == Chain(XML_PLAIN_VALUE) )
    {
	exp.binExportTableSet(tableSet, isStructure, expFile, true);
    }
    pAH->sendResponse(Chain("Tableset " + tableSet + " exported"));

}

void CegoAdminThread::srvImportTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain impFile;
    bool isStructure;
    bool doLogging;
    Chain impMode;

    pAH->getTableSet(tableSet);
    pAH->getFileName(impFile);
    pAH->getIsStructure(isStructure);
    pAH->getDoLogging(doLogging);
    pAH->getMode(impMode);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online for import");
    }

    _lastAction = Chain("Import tableset ") + tableSet;    

    _pTabMng->setActiveUser(tableSet, _user, _password);    
    CegoXPorter imp(_pTabMng, _pDbPool);

    if ( impMode == Chain(XML_XML_VALUE) )
    {
	imp.xmlImportTableSet(tableSet, isStructure, impFile, doLogging);
    }
    else if ( impMode == Chain(XML_BIN_VALUE) )
    {
	imp.binImportTableSet(tableSet, isStructure, impFile, doLogging, false);
    }
    else if ( impMode == Chain(XML_PLAIN_VALUE) )
    {
	// logging is forced disabled in case of plain mode
	imp.binImportTableSet(tableSet, isStructure, impFile, false, true);
    }


    pAH->sendResponse(Chain("Tableset " + tableSet + " imported"));
}


void CegoAdminThread::srvExportTable(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain tableName;
    Chain expFile;
    Chain expMode;

    pAH->getTableSet(tableSet);
    pAH->getTableName(tableName);
    pAH->getFileName(expFile);
    pAH->getMode(expMode);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online for export");
    }

    _pTabMng->setActiveUser(tableSet, _user, _password);

    CegoXPorter exp(_pTabMng);
    if ( expMode == Chain(XML_XML_VALUE) )
    {
	exp.xmlExportTable(tableSet, tableName, expFile);
    }
    else if ( expMode == Chain(XML_BIN_VALUE) )
    {
	exp.binExportTable(tableSet, tableName, expFile, false);
    }
    else if ( expMode == Chain(XML_PLAIN_VALUE) )
    {
	exp.binExportTable(tableSet, tableName, expFile, true);
    }

    pAH->sendResponse(Chain("Table " + tableName + " exported"));
}


void CegoAdminThread::srvImportTable(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain tableName;
    Chain impFile;
    bool doLogging;
    Chain impMode;

    pAH->getTableSet(tableSet);
    pAH->getTableName(tableName);
    pAH->getFileName(impFile);
    pAH->getDoLogging(doLogging);
    pAH->getMode(impMode);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online for import");
    }

    _pTabMng->setActiveUser(tableSet, _user, _password);
    
    CegoXPorter imp(_pTabMng);

    if ( impMode == Chain(XML_XML_VALUE) )
    {
	imp.xmlImportTable(tableSet, tableName, impFile, doLogging);
    }
    else if ( impMode == Chain(XML_BIN_VALUE) )
    {
	imp.binImportTable(tableSet, tableName, impFile, doLogging, false);
    }
    else if ( impMode == Chain(XML_PLAIN_VALUE) )
    {
	imp.binImportTable(tableSet, tableName, impFile, doLogging, true);
    }

    pAH->sendResponse(Chain("Table " + tableName + " imported"));
}

void CegoAdminThread::srvAddDatafile(CegoAdminHandler *pAH)
{

    Chain tableSet;
    Chain fileType;
    Chain dataFile;
    int fileSize;
    int fileId;
    
    pAH->getTableSet(tableSet);
    pAH->getFileType(fileType);
    pAH->getFileId(fileId);
    pAH->getFileName(dataFile);
    pAH->getFileSize(fileSize);
        
    _pTabMng->addDataFile(tableSet, fileType, fileId,  dataFile, fileSize);
    _pTabMng->writeCheckPoint(tableSet, true, false);

    try
    {
	_pDBMng->addDataFile(tableSet, fileType, fileId, dataFile, fileSize); 	    
    }
    catch ( Exception e )
    {
	Chain msg = e.getBaseMsg();
	throw Exception(EXLOC, msg);		
    }

    pAH->sendResponse(Chain("Data file " + dataFile + " added"));
    
}
    

void CegoAdminThread::srvUserInfo(CegoAdminHandler *pAH)
{
    Element* pUserInfo = _pDBMng->getUserList(); 
    
    _lastAction = Chain("showUser");
    pAH->sendResponse(Chain("User Info"), pUserInfo);
 
}

void CegoAdminThread::srvAddUser(CegoAdminHandler *pAH)
{
    Chain user;
    Chain password;
    pAH->getUser(user);
    pAH->getPassword(password);

    // AESCrypt aescrypt(CEGOAESKEY, CEGOAESKEYLEN);
    //_pDBMng->addUser(user, aescrypt.encrypt(password));

    _pDBMng->addUser(user, password);
    
    pAH->sendResponse(Chain("User " + user + " added"));
}	

void CegoAdminThread::srvRemoveUser(CegoAdminHandler *pAH)
{
    Chain user;
    pAH->getUser(user);
    
    _pDBMng->removeUser(user);
    pAH->sendResponse(Chain("User " + user + " removed"));
}
	
void CegoAdminThread::srvChangePwd(CegoAdminHandler *pAH)
{

    Chain user;
    Chain password;

    pAH->getUser(user);
    pAH->getPassword(password);

    // AESCrypt aescrypt(CEGOAESKEY, CEGOAESKEYLEN);
    // _pDBMng->changePassword(user, aescrypt.encrypt(passwd));
    _pDBMng->changePassword(user, password);
    
    pAH->sendResponse(Chain("Password changed for user " + user));
		    
}

void CegoAdminThread::srvListRole(CegoAdminHandler *pAH)
{
    
    Element* pRoleList = _pDBMng->getRoleList(); 
    
    _lastAction = Chain("listRole");
    pAH->sendResponse(Chain("Role List"), pRoleList);

}

void CegoAdminThread::srvShowRole(CegoAdminHandler *pAH)
{
    
    Chain role;
    pAH->getRole(role);
    
    Element* pPermInfo = _pDBMng->getPermInfo(role);
    
    Host h;    
    Chain mediator = h.getName();
       
    pAH->sendResponse(Chain("Role info"), pPermInfo);

}


void CegoAdminThread::srvCreateRole(CegoAdminHandler *pAH)
{
    
    Chain role;
    pAH->getRole(role);

    _pDBMng->createRole(role);

    Host h;    
    Chain mediator = h.getName();
	    
    pAH->sendResponse(Chain("Role " ) + role + Chain(" created"));

}


void CegoAdminThread::srvDropRole(CegoAdminHandler *pAH)
{
    
    Chain role;
    pAH->getRole(role);

    _pDBMng->dropRole(role);

    Host h;    
    Chain mediator = h.getName();
	    
    pAH->sendResponse(Chain("Role ") + role + Chain(" dropped"));

}


void CegoAdminThread::srvAssignRole(CegoAdminHandler *pAH)
{
    Chain user;
    Chain role;
    pAH->getUser(user);
    pAH->getRole(role);
    
    _pDBMng->assignUserRole(user, role);
    pAH->sendResponse(Chain("Role set for user " + user));
		    
}

void CegoAdminThread::srvRemoveRole(CegoAdminHandler *pAH)
{
    Chain user;
    Chain role;

    pAH->getUser(user);
    pAH->getRole(role);
    
    _pDBMng->removeUserRole(user, role);
    pAH->sendResponse(Chain("Role ") + role + Chain("removed from user ") + user);
		    
}

void CegoAdminThread::srvSetPerm(CegoAdminHandler *pAH)
{
    
    Chain role;
    pAH->getRole(role);

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain filter;
    pAH->getFilter(filter);

    Chain perm;
    pAH->getPerm(perm);

    Chain permid;
    pAH->getPermId(permid);

    _pDBMng->setPerm(role, permid, tableSet, filter, perm);
    
    pAH->sendResponse(Chain("Permission ") + permid + Chain(" set"));

}

void CegoAdminThread::srvRemovePerm(CegoAdminHandler *pAH)
{
    
    Chain role;
    pAH->getRole(role);

    Chain permid;
    pAH->getPermId(permid);

    _pDBMng->removePerm(role, permid);
    
    pAH->sendResponse(Chain("Permission ") + permid + Chain(" removed"));

}


void CegoAdminThread::srvUserTrace(CegoAdminHandler *pAH)
{
    Chain user;
    bool isOn;
    pAH->getUser(user);
    pAH->getTrace(isOn);
    
    _pDBMng->setUserTrace(user, isOn);
    pAH->sendResponse(Chain("User trace set for ") + user);
		    
}

void CegoAdminThread::srvSyncTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    Chain escCmd;
    int timeout;

    pAH->getTableSet(tableSet);
    pAH->getEscapeCmd(escCmd);
    pAH->getTimeoutValue(timeout);
    pAH->getBUMsg(bumsg);    

    _pTabMng->syncDistTableSet(tableSet, bumsg, escCmd, timeout);

    Chain msg;
    if ( escCmd == Chain("") )
    {
	msg = Chain("Tableset ") + tableSet + Chain(" in sync");
    }
    else
    {
	msg = Chain("Tableset ") + tableSet + Chain(" in sync with escape command");
    }
    pAH->sendResponse(msg);
    
}

void CegoAdminThread::srvInitLogs(CegoAdminHandler *pAH)
{
    Chain tableSet;
    
    pAH->getTableSet(tableSet);    
    _pDBMng->initLogFiles(tableSet, true);
    pAH->sendResponse(Chain("Logs for tableset " + tableSet + " initialized"));
}


void CegoAdminThread::srvAddArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain archId;
    Chain archPath;

    pAH->getTableSet(tableSet);
    pAH->getArchId(archId);
    pAH->getArchPath(archPath);
    
    _pDBMng->addArchLog(tableSet, archId, archPath);

    pAH->sendResponse(Chain("Archlog " + archPath + " added to " + tableSet));
    
}

void CegoAdminThread::srvRemoveArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain archId;

    pAH->getTableSet(tableSet);
    pAH->getArchId(archId);
    _pDBMng->removeArchLog(tableSet, archId);
    pAH->sendResponse(Chain("Archlog id " + archId + " removed from " + tableSet));
}

void CegoAdminThread::srvBeginBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain msg;

    pAH->getTableSet(tableSet);
    pAH->getBUMsg(msg);

    _pTabMng->beginBackup(tableSet, msg);
    
    pAH->sendResponse(Chain("Backup started"));
}

void CegoAdminThread::srvEndBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain msg;
    bool keepTicket;

    pAH->getTableSet(tableSet);
    pAH->getBUMsg(msg);
    pAH->getKeepTicket(keepTicket);

    _pTabMng->endBackup(tableSet, msg, keepTicket);    
    pAH->sendResponse(Chain("Backup ended"));
}

void CegoAdminThread::srvRecover(CegoAdminHandler *pAH)
{
    Chain tableSet;
    int pit;

    pAH->getTableSet(tableSet);
    pAH->getPit(pit);
    CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::LOCAL);
    long lsn = recoveryMng.recoverTableSet(tableSet, pit);
    pAH->sendResponse(Chain("Tableset recovered to lsn ") + Chain(lsn));
}

void CegoAdminThread::srvSecSwitch(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain secondary;
    pAH->getSecondary(secondary);
    _pTabMng->logTo(tableSet, secondary);
    _pDBMng->setSecondary(tableSet, secondary);
    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" log to  ") + secondary);
}
	
//////////////////////////////
///// mediator services //////
//////////////////////////////

void CegoAdminThread::medNotify(CegoAdminHandler *pAH)
{
    
    Chain hostName;
    Chain hostStatus;


    pAH->getHostName(hostName);
    pAH->getHostStatus(hostStatus);

    ListT<Chain> tsList;
    ListT<Chain> syncList;
    ListT<Chain> runList;

    pAH->getTableSyncStateList(tsList, runList, syncList);
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Setting host status <") + hostStatus + Chain("> for host <") 
		+ hostName + Chain(">"));
#endif	

    _pDBMng->setHostStatus(hostName, hostStatus);

    Chain *pTS = tsList.First();
    Chain *pSync = syncList.First();
    Chain *pRun = runList.First();
    
    while ( pTS && pSync && pRun)
    {

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting tableset status for <") + *pTS + Chain("> to ") + *pSync );
#endif

	_pDBMng->setTableSetRunState(*pTS, *pRun);
	_pDBMng->setTableSetSyncState(*pTS, *pSync);
	pTS = tsList.Next();
	pSync = syncList.Next();
	pRun = runList.Next();
    }
    pAH->sendResponse(Chain("Notify ok"));
    
}

void CegoAdminThread::medDefineTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    Chain tsRoot;
    Chain tsTicket;

    Chain primary;
    Chain secondary;

    int sysSize;
    int tmpSize;
    int logFileSize;
    int logFileNum;

    long sortAreaSize;

    pAH->getTableSet(tableSet);
    pAH->getTableSetRoot(tsRoot);
    pAH->getTableSetTSTicket(tsTicket);

    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getSysSize(sysSize);
    pAH->getTmpSize(tmpSize);

    pAH->getLogFileSize(logFileSize);
    pAH->getLogFileNum(logFileNum);

    pAH->getSortAreaSize(sortAreaSize);

    
    Host h;
    int sysFid = _pDBMng->nextTSID();
    int tmpFid = _pDBMng->nextFID();

    Chain resultMsg;

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;

    if ( primary  == Chain("localhost") )
	primary = h.getName();
    if ( secondary  == Chain("localhost") )
	secondary = h.getName();

    if ( h.getName() != primary )
    {
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }

    if ( h.getName() != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }
    
    CegoAdminHandler::ResultType res;
    
    if ( pPrimaryAH )
    {
	res = pPrimaryAH->reqDefineTableSet(tableSet, tsRoot, tsTicket, primary, secondary, h.getName(), sysFid, tmpFid, sysSize, tmpSize, logFileSize, logFileNum, sortAreaSize);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK )
	    throw Exception(EXLOC, msg);
	
    }
    
    if ( pSecondaryAH )
    {
	res = pSecondaryAH->reqDefineTableSet(tableSet, tsRoot, tsTicket, primary, secondary, h.getName(), sysFid, tmpFid, sysSize, tmpSize, logFileSize, logFileNum, sortAreaSize);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
		
	if ( res != CegoAdminHandler::ADM_OK )
	    throw Exception(EXLOC, msg);
	
    }
    
    _pDBMng->addTableSetDef(tableSet, tsRoot, tsTicket, primary, secondary, h.getName(), sysFid, tmpFid, sysSize, tmpSize, logFileSize, logFileNum, sortAreaSize); 
    
    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" defined"));
    
}

void CegoAdminThread::medRemoveTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);


    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    Host h;

    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status defined to be removed");	
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);	
    }
    
    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;


    if ( h.getName() != primary )
    {
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }

    
    CegoAdminHandler::ResultType res;
    
    
    if ( pPrimaryAH )
    {
	res = pPrimaryAH->reqRemoveTableSet(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( h.getName() != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }

    if ( pSecondaryAH )
    {
	res = pSecondaryAH->reqRemoveTableSet(tableSet);
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);	

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}

    }
    
    _pDBMng->rmTableSetDef(tableSet); 

    Chain msg = Chain("Tableset ") + tableSet + Chain(" removed");
    pAH->sendResponse(msg);

}

void CegoAdminThread::medResetTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    Host h;
    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);	
    }
    

    if ( h.getName() != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;
	res = pPrimaryAH->reqResetTableSet(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	
	pAH->syncWithInfo(Chain("primary"), primary, msg);

    }
    else
    {	   
	_pTabMng->resetTableSet(tableSet); 	
    }

    Chain msg = Chain("Tableset ") + tableSet + Chain(" reset");
    pAH->sendResponse(msg);

}


void CegoAdminThread::medCreateTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);
    }


    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status defined to be created");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;	
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    
    try 
    {	
	_pDBMng->getStatusByHost(primary, primaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	try 
	{
	    // create local
	    _pTabMng->createTableSet(tableSet);
	}	
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);	
	    throw Exception(EXLOC, msg);
	}	
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
 
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());	
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqCreateTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{
	    res = pPrimaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	    Chain msg;
	    pPrimaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pPrimaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("primary"), primary, msg);
	    }
	}
    }

    if ( primary == secondary )
    {
	_pDBMng->setTableSetSyncState(tableSet,XML_SYNCHED_VALUE);
    }
    else
    {


	CegoAdminHandler* pSecondaryAH = 0;
 
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pSecondaryAH->reqSetRunState(tableSet, XML_OFFLINE_VALUE);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}

	res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);

	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
	
	closeSession(pSecondaryAH);
	
	_pDBMng->setTableSetSyncState(tableSet,XML_NOT_SYNCHED_VALUE);
    }

    _pDBMng->setTableSetRunState(tableSet,XML_OFFLINE_VALUE);    

    Chain msg = Chain("Tableset ") + tableSet + Chain(" created");

    pAH->sendResponse(msg);

}

void CegoAdminThread::medCopyTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primary == secondary )
    {
	Chain msg = Chain("Cannot copy tablset, primary and secondary are identical (") + primary + Chain("=") + secondary + Chain(")");
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;

    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);

    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;
    
    if ( mediator != primary )
    { 
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }
    
    if ( mediator != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }

    if ( pSecondaryAH == 0 )
    {
	// local reset

	int tabSetId = _pDBMng->getTabSetId(tableSet);
	    
	if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
	{

	    _pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
	    
	    while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Waiting for recovery end ..."));
#endif
		Sleeper s;
		s.secSleep(LOGMNG_RECOVERY_DELAY);
	    }
	}
    }
    else
    {

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Stopping recovery on secondary ..."));
#endif

	CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	       
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);    
	}	
    }
    
    if ( pPrimaryAH == 0 )
    {
	_pDBMng->startCopy(tableSet, secondary, mediator, pAH->getUser(), pAH->getPassword(), Chain("Copy requested"));
	pAH->syncWithInfo(Chain("primary"), primary, Chain("Copy started"));
    }
    else
    {

	CegoAdminHandler::ResultType res = pPrimaryAH->reqCopyTableSet(tableSet, secondary, mediator);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	if ( pSecondaryAH == 0 )
	{
	    _pDBMng->startRecovery(tableSet);	    
	}
	else
	{
	    
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStartRecovery(tableSet);    

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    } 
    
    pAH->sendResponse(Chain("Tableset copy started"));

}

void CegoAdminThread::medDropTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);
    }

    if ( status != Chain(XML_OFFLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline to be dropped");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;

    try 
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");	
	throw Exception(EXLOC, msg);	
    }

    if ( mediator == primary )
    {
	// create local

	try {

	    _pTabMng->dropTableSet(tableSet);   

	}
	catch ( Exception e )
	{	    
	    Chain msg;
	    e.pop(msg);
	    throw Exception(EXLOC, msg);
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	CegoAdminHandler::ResultType res = pPrimaryAH->reqDropTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	    
	pAH->syncWithInfo(Chain("primary"), primary, msg);
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, Chain("Admin action failed"));
	}

	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqDropTableSet(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}
    }

    _pDBMng->setTableSetRunState(tableSet,XML_DEFINED_VALUE);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" dropped");
    pAH->sendResponse(msg);

}

void CegoAdminThread::medStartTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);
    bool cleanIt = pAH->getCleanup();
    bool forceload = pAH->getForceload();

    _lastAction = Chain("Start tableset ") + tableSet;

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    Host h;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_OFFLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline to be started");       	
	throw Exception(EXLOC, msg);
    }
        
    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + mediator + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");	
	throw Exception(EXLOC, msg);

    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");	
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	try 
	{
	    _pTabMng->startDistTableSet(tableSet, secondary, cleanIt);
	    
	    if ( forceload )
	    {
		int tabSetId = _pDBMng->getTabSetId(tableSet);
		_pDbPool->loadObjects(tabSetId);
	    }
	    
	}
	catch ( Exception e )
	{	    
	    Chain msg;
	    e.pop(msg);	    
	    throw Exception(EXLOC, msg);	    
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqStartTableSet(tableSet, cleanIt, forceload);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( primary != secondary )
    {
	
	CegoAdminHandler* pSecondaryAH = 0;
	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	CegoAdminHandler::ResultType res = pSecondaryAH->reqStartRecovery(tableSet);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
		
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}	
    }
    
    _pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" started");
    pAH->sendResponse(msg);


}

void CegoAdminThread::medStopTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Stop tableset ") + tableSet;

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);		
	throw Exception(EXLOC, msg);
    }


    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status active to be stopped");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + mediator + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try {
	
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
	
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	try 
	{

	    int tabSetId = _pDBMng->getTabSetId(tableSet);
	    _pDbPool->unloadObjects(tabSetId);

	    _pTabMng->stopDistTableSet(tableSet, true);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);	
	    throw Exception(EXLOC, msg);
	}
    }
    else
    {      	

	CegoAdminHandler* pPrimaryAH = 0;
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqStopTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }

	}
    }

    _pDBMng->setTableSetRunState(tableSet,XML_OFFLINE_VALUE);

    Chain msg = Chain("Tableset ") + tableSet + Chain(" stopped");
    pAH->sendResponse(msg);
    
}


void CegoAdminThread::medSwitchTableSet(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain runStatus;
    Chain syncStatus;

    Chain resultMsg;

    bool isOnline = false;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	runStatus = _pDBMng->getTableSetRunState(tableSet);
	syncStatus = _pDBMng->getTableSetSyncState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);	
    }


    Chain newPrimary = secondary;
    Chain newSecondary = mediator;
    Chain newMediator = mediator;

    if ( syncStatus != Chain(XML_SYNCHED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" is not in sync");	
	throw Exception(EXLOC, msg);	
    }


    if ( runStatus == Chain(XML_ONLINE_VALUE) )
    {
	isOnline = true;
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;	
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try {

	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);

    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);	
    }

    if ( primary == secondary )
    {
	Chain msg = Chain("Primary and secondary identical, switch not useful");
	throw Exception(EXLOC, msg);	
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    // first step : make sure primary host is down
    
    if ( mediator == primary )
    {
	
	if ( isOnline )
	{
	    try 
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Stopping tableset ") + tableSet);
#endif

		_pTabMng->stopTableSet(tableSet, true);

		int tabSetId = _pDBMng->getTabSetId(tableSet);
		_pDbPool->unloadObjects(tabSetId);

	    }
	    catch ( Exception e )
	    {
		Chain msg;
		e.pop(msg);
		throw Exception(EXLOC, msg);	
	    }
	}
    }
    else
    {
	
	try
	{

	    CegoAdminHandler* pPrimaryAH = 0;
	    CegoAdminHandler::ResultType res;
	    
	    pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	    if ( isOnline )
	    {

		res = pPrimaryAH->reqStopTableSet(tableSet);

		Chain msg;
		pPrimaryAH->getMsg(msg);
		
		if ( res != CegoAdminHandler::ADM_OK)
		{
		    closeSession(pPrimaryAH);
		    throw Exception(EXLOC, msg);
		}
		else
		{
		    pAH->syncWithInfo(Chain("primary"), primary, msg);    		    
		}
	    }
	    
	    res = pPrimaryAH->reqSetTableSetNode(tableSet, newPrimary, newSecondary, newMediator);
	    Chain msg;
	    pPrimaryAH->getMsg(msg);
	    closeSession(pPrimaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }

	}
	catch ( Exception e ) 
	{
	    // ignore, primary no more available
	}
	
    }
	
    // second step : activate secondary

    if ( mediator == secondary )
    {
	if ( isOnline )
	{
	    	    
	    int tabSetId = _pDBMng->getTabSetId(tableSet);
	    
	    if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
	    {
		
		_pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
		
		while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
		{
#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Waiting for recovery end ... "));
#endif
		    Sleeper s;
		    s.secSleep(LOGMNG_RECOVERY_DELAY);
		}
	    }

#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Starting tableset ") + tableSet);
#endif

	    _pTabMng->startDistTableSet(tableSet, newSecondary, false);
	    
	    // load objects not really required, for performance reasons we do not
	    // _pDbPool->loadObjects(tabSetId);
	    
	}
    }
    else
    {

	CegoAdminHandler* pSecondaryAH = 0;	
	CegoAdminHandler::ResultType res;

	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	if ( isOnline )
	{
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	 
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}

	res = pSecondaryAH->reqSetTableSetNode(tableSet, newPrimary, newSecondary, newMediator);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);    
	}

	if ( isOnline )
	{
	    res = pSecondaryAH->reqStartTableSet(tableSet, false, false);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }

	}

	if ( newPrimary != newSecondary )
	{    
	    res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}
	else
	{
	    res = pSecondaryAH->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}   
	closeSession(pSecondaryAH);
	
    }

    if ( newPrimary != newSecondary )
	_pDBMng->setTableSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
    else
	_pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);

    _pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);

    _pDBMng->setPrimary(tableSet, newPrimary);
    _pDBMng->setSecondary(tableSet, newSecondary);
    _pDBMng->setMediator(tableSet, newMediator);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" switched");
    pAH->sendResponse(msg);

    
}

void CegoAdminThread::medAddArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain archId;
    pAH->getArchId(archId);
    
    Chain archPath;
    pAH->getArchPath(archPath);


    Chain primary;
    Chain secondary;
    Chain mediator;

    Chain resultMsg;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);		
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;
 
    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);		
    }
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }
    
    if ( mediator != primary )
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqAddArchLog(tableSet, archId, archPath);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), secondary, msg);
	}

	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqAddArchLog(tableSet, archId, archPath);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("primary"), secondary, msg);
	    }
	}	
    }
    
    _pDBMng->addArchLog(tableSet, archId, archPath);
   
    pAH->sendResponse("Archlog added");
    

}

void CegoAdminThread::medRemoveArchLog(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain archId;
    pAH->getArchId(archId);

    Chain primary;
    Chain secondary;
    Chain mediator;

    Chain resultMsg;

    Host h;

    Chain primaryStatus;
    Chain secondaryStatus;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);		
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }
       
    if ( mediator != primary )
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqRemoveArchLog(tableSet, archId);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);	    
	}

	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqRemoveArchLog(tableSet, archId);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    }

    _pDBMng->removeArchLog(tableSet, archId);
    
    pAH->sendResponse(Chain("Archlog removed"));

    
}

void CegoAdminThread::medAddDatafile(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain fileName;
    pAH->getFileName(fileName);
    
    int fileSize;
    pAH->getFileSize(fileSize);

    Chain fileType;
    pAH->getFileType(fileType);

    Chain primary;
    Chain secondary;
    Chain mediator;

    Host h;

    try
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg = e.getBaseMsg();
	throw Exception(EXLOC, msg);			    
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
	
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    int fileId = _pDBMng->nextFID();
    
    if ( mediator == primary )
    {
	try
	{
	    _pTabMng->addDataFile(tableSet, fileType, fileId, fileName, fileSize);
	    _pTabMng->writeCheckPoint(tableSet, true, false);
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);			    
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqAddDataFile(tableSet, fileType, fileId,  fileName, fileSize);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( primary != secondary && secondary != mediator )
    {
	
	CegoAdminHandler* pSecondaryAH = 0;
	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pSecondaryAH->reqAddDataFile(tableSet, fileType, fileId, fileName, fileSize);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }	
    
    try
    {
	_pDBMng->addDataFile(tableSet, fileType, fileId, fileName, fileSize); 	    
    }
    catch ( Exception e )
    {
	Chain msg = e.getBaseMsg();
	throw Exception(EXLOC, msg);		
    }
	
    pAH->sendResponse(Chain("Datafile added"));
    
}

void CegoAdminThread::medGetTableSetList(CegoAdminHandler *pAH)
{    
    Element* pTabSetInfo = _pDBMng->getTableSetList(); 

    bool getUsage;
    pAH->getUsage(getUsage);
    if ( getUsage )
    {	
	ListT<Element*> tabSetList = pTabSetInfo->getChildren(XML_TABLESET_ELEMENT);
	Element **pTabSet = tabSetList.First();
	while ( pTabSet )
	{
	    Chain tableSet = (*pTabSet)->getAttributeValue(XML_NAME_ATTR);
	    addTableSetUsage(*pTabSet);
	    
	    pTabSet = tabSetList.Next();
	}
    }
    pAH->sendResponse(Chain("Tableset list"), pTabSetInfo);		      
}

void CegoAdminThread::medGetDetailedTableSetInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Host h;

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }


    Element *pTSInfo;

    if ( primary == mediator )
    {    
	pTSInfo = getDetailedTSInfo(tableSet);
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetDetailedTSInfo(tableSet);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    Chain msg;
	    pPrimaryAH->getMsg(msg);   
	    throw Exception(EXLOC, msg);
	}

	pTSInfo = pPrimaryAH->getTSInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
    }
    
    pAH->sendResponse(Chain("Tableset information"), pTSInfo);

}

void CegoAdminThread::medGetTransactionInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Host h;

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg); 
	throw Exception(EXLOC, msg);
    }


    Element *pTAInfo;

    if ( primary == mediator )
    {    
	pTAInfo = getTransactionInfo(tableSet);
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetTransactionInfo(tableSet);

	pTAInfo = pPrimaryAH->getTAInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("Transaction info"), pTAInfo);   
}

void CegoAdminThread::medGetBUStatInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Host h;

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg); 
	throw Exception(EXLOC, msg);
    }

    Element *pBUSInfo;

    if ( primary == mediator )
    {    
	pBUSInfo = getBUStatInfo(tableSet);
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetBUStatInfo(tableSet);

	// TODO
	// pBUSInfo = pPrimaryAH->getBUStatInfo(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("BUStat info"), pBUSInfo);   
}

void CegoAdminThread::medResetBUStat(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Host h;

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg); 
	throw Exception(EXLOC, msg);
    }


    Element *pBUSInfo;

    if ( primary == mediator )
    {    
	_pTabMng->resetBUStat(tableSet);
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = CegoAdminHandler::ADM_OK; // pPrimaryAH->reqGetBUStatInfo(tableSet);

	// TODO
	// pBUSInfo = pPrimaryAH->getBUStatInfo(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("BUStat reset done"));   
}

void CegoAdminThread::medGetObjectInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Host h;

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg); 
	throw Exception(EXLOC, msg);
    }

    Element *pObjectInfo;

    if ( primary == mediator )
    {    
	pObjectInfo = getObjectInfo(tableSet);
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetObjectInfo(tableSet);

	pObjectInfo = pPrimaryAH->getObjectInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("Object info"), pObjectInfo);

}

void CegoAdminThread::medBeginBackup(CegoAdminHandler *pAH)
{

    Chain tableSet;
    Chain bumsg;
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(bumsg);


    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Host h;
    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online\nto be backup up online");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain("\nfor tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {    
	_pTabMng->beginBackup(tableSet, bumsg);
    }
    else
    {
	
	CegoAdminHandler* pPrimaryAH = 0;
	
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqBeginBackup(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	_pDBMng->setTableSetRunState(tableSet,XML_BACKUP_VALUE);	
	
    }

    pAH->sendResponse(Chain("Backup mode started"));    

}


void CegoAdminThread::medEndBackup(CegoAdminHandler *pAH)
{


    Chain tableSet;
    Chain bumsg;
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(bumsg);

    bool keepTicket;
    pAH->getKeepTicket(keepTicket);
    
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;


    Host h;
    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( status != Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status backup to end backup");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {	
	_pTabMng->endBackup(tableSet, bumsg, keepTicket);	
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;
	
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqEndBackup(tableSet, keepTicket);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	_pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);
	
    }

    pAH->sendResponse(Chain("Backup mode finished"));        


}


void CegoAdminThread::medRecover(CegoAdminHandler *pAH)
{

    Chain tableSet;
    int pit;

    pAH->getTableSet(tableSet);
    pAH->getPit(pit);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Host h;

    Chain primaryStatus;
    Chain secondaryStatus;
    Chain logMngProg;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( status != Chain(XML_OFFLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline to be started");
	throw Exception(EXLOC, msg);

    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	long lsn;

	CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::LOCAL);
	lsn = recoveryMng.recoverTableSet(tableSet, pit);
	
	pAH->syncWithInfo(Chain("mediator"), mediator, Chain("Tableset recovered to lsn ") + Chain(lsn));
	
    }
    else
    {
	
	CegoAdminHandler* pPrimaryAH = 0;
		
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqRecover(tableSet, pit);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}	
    }

    pAH->sendResponse(Chain("Recover completed"));
        
}

void CegoAdminThread::medSecSwitch(CegoAdminHandler *pAH)
{
    
    Chain tableSet;
    
    pAH->getTableSet(tableSet);
    
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    Host h;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	try
	{
	    _pTabMng->logTo(tableSet, mediator);
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);  
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqSecSwitch(tableSet, mediator);


	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

    }

    pAH->sendResponse(Chain("Secondary switch"));

}


void CegoAdminThread::medSecRelocate(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain newSecondary;
    pAH->getSecondary(newSecondary);
    
    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    Host h;

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    bool isArchiveMode;


    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    isArchiveMode = _pDBMng->isArchiveMode(tableSet);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);

    }

    if ( isArchiveMode == false )
    {
	Chain msg = Chain("No archive mode enabled");
	throw Exception(EXLOC, msg);

    }
    if ( mediator == primary )
    {       
	try
	{
	    _pTabMng->logTo(tableSet, newSecondary);
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);	   
	}
    }
    else // ( mediator != primary )
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res;

	res = pPrimaryAH->reqSecSwitch(tableSet, newSecondary);

	Chain msg;
	pPrimaryAH->getMsg(msg);
		    
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	res = pPrimaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	pPrimaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, newSecondary, mediator);

	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( mediator != secondary && primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    }

    if ( mediator != newSecondary )
    {

	CegoAdminHandler* pSecondaryAH = 0;
       
	pSecondaryAH = getSession(Chain("secondary"), newSecondary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;
	Chain msg;

	res = pSecondaryAH->reqPropTSInfo(_pDBMng->getTableSetInfo(tableSet));
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}

	res = pSecondaryAH->reqSecSwitch(tableSet, newSecondary);
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetRunState(tableSet, _pDBMng->getTableSetRunState(tableSet));
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	pSecondaryAH->getMsg(msg);
	    
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetTableSetNode(tableSet, primary, newSecondary, mediator);

	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
    }

    if ( primary != newSecondary )
    {
	_pDBMng->setTableSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
    }
    else
    {
	_pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);
    }

    _pDBMng->setSecondary(tableSet, newSecondary);
    
    pAH->sendResponse(Chain("Secondary relocated"));    


}

void CegoAdminThread::secMedSwitch(CegoAdminHandler *pAH)
{

    
    Chain tableSet;
    
    pAH->getTableSet(tableSet);
    
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    Host h;

    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( h.getName() != secondary )
    {
	Chain msg = Chain("Invalid secondary host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator != secondary )
    {

	CegoAdminHandler* pSecondaryAH = 0;
	CegoAdminHandler::ResultType res;

	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());

	res = pSecondaryAH->reqSetTableSetNode(tableSet, primary, secondary, secondary);	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }

    if ( mediator != primary )
    {

	CegoAdminHandler* pPrimaryAH = 0;
	CegoAdminHandler::ResultType res;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, secondary, secondary);	
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

    }

    _pDBMng->setMediator(tableSet, secondary);    

    pAH->sendResponse("Secondary switched");


}

void CegoAdminThread::secMedRelocate(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain newMediator;
    pAH->getMediator(newMediator);
    
    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    Host h;

    if ( h.getName() != secondary )
    {
	Chain msg = Chain("Invalid secondary host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);       
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( primary != newMediator )
    {

	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, secondary, newMediator);
	Chain msg;
	pPrimaryAH->getMsg(msg);
     	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( secondary != newMediator )
    {

	CegoAdminHandler* pMediatorAH = 0;
       
	pMediatorAH = getSession(Chain("mediator"), newMediator, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;

	res = pMediatorAH->reqPropTSInfo(_pDBMng->getTableSetInfo(tableSet));

	Chain msg;
	pMediatorAH->getMsg(msg);
		
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pMediatorAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("mediator"), newMediator, msg);
	}

	res = pMediatorAH->reqSetTableSetNode(tableSet, primary, secondary, newMediator);
	pMediatorAH->getMsg(msg);
	closeSession(pMediatorAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("mediator"), newMediator, msg);
	}
    }

    _pDBMng->setMediator(tableSet, newMediator);
    
    pAH->sendResponse("Mediator relocated");

}

void CegoAdminThread::medCheckTableSet(CegoAdminHandler *pAH)
{


    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    Host h;

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);	
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);	
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;

    Element* pTSInfo = _pDBMng->getTableSetInfo(tableSet); 
	
    if ( mediator != primary )
    {

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetTSInfo(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    if ( mediator != secondary )
    {	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pSecondaryAH->reqGetTSInfo(tableSet);
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }

    Element *pCheck = checkTS(Chain("mediator"), mediator, pTSInfo, pPrimaryAH, pSecondaryAH);
    
    if ( pPrimaryAH )
	closeSession(pPrimaryAH);
    if ( pSecondaryAH )
	closeSession(pSecondaryAH);

    pCheck->setAttribute(XML_MEDIATOR_ATTR, mediator);
    pCheck->setAttribute(XML_PRIMARY_ATTR, primary);
    pCheck->setAttribute(XML_SECONDARY_ATTR, secondary);
    
    pAH->sendResponse(Chain("Check completed"), pCheck);
}

void CegoAdminThread::medVerifyTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Verify tableset ") + tableSet;

    Chain resultMsg;

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }


    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online to be verified");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try {
	
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	try 
	{

	    _pTabMng->setActiveUser(tableSet, _user, _password);
	    
	    ListT<Chain> tableList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);
	    
	    Chain *pTable = tableList.First();
	    while ( pTable)
	    {
		
		Element *pVerificationInfo = _pTabMng->verifyTable(tableSet, *pTable);
		if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of ") + *pTable, pVerificationInfo) == false )
		{
		    return;
		}

		pTable = tableList.Next();
	    }

	    ListT<Chain> viewList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::VIEW, viewList);
	    
	    Chain *pView = viewList.First();
	    while ( pView )
	    {
		Element* pVerificationInfo = _pTabMng->verifyView(tableSet, *pView);
		if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of ") + *pView, pVerificationInfo) == false )
		    return;
		pView = viewList.Next();
	    }
	    

	    ListT<Chain> procList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::PROCEDURE, procList);

	    
	    Chain *pProc = procList.First();
	    while ( pProc )
	    {
		
		Element* pVerificationInfo = _pTabMng->verifyProcedure(tableSet, *pProc);
		if ( pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Verification of ") + *pProc, pVerificationInfo) == false )
		    return;

		pProc = procList.Next();
	    }
	    
	    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" verified"));

	}
	catch ( Exception e )
	{	    
	    Chain msg;
	    e.pop(msg);
	    throw Exception(EXLOC,msg);
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqVerifyTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}


	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;

	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    
	    CegoAdminHandler::ResultType res; // TODO  = pSecondaryAH->reqVerifyTS(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }

	    closeSession(pSecondaryAH);
	}	
    }
}


void CegoAdminThread::medCorrectTableSet(CegoAdminHandler *pAH)
{

    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Correct tableset ") + tableSet;

    Chain resultMsg;

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Host h;

    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	secondary = _pDBMng->getSecondary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
	status = _pDBMng->getTableSetRunState(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }


    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online to be corrected");
	throw Exception(EXLOC, msg);		
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try {
	
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	try 
	{

	    _pTabMng->setActiveUser(tableSet, _user, _password);

	    ListT<Chain> tableList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);

	    
	    Chain *pTable = tableList.First();
	    while ( pTable)
	    {		
		Element *pCorrectionInfo = _pTabMng->correctTable(tableSet, *pTable);
		pAH->syncWithInfo(Chain("primary"), h.getName(), Chain("Correction of ") + *pTable, pCorrectionInfo);
		pTable = tableList.Next();	    
	    }

    
	    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" corrected"));

	}
	catch ( Exception e )
	{	    
	    Chain msg;
	    e.pop(msg);
	    throw Exception(EXLOC, msg); 
	}
    }
    else
    {

	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqVerifyTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}


	if ( primary != secondary )
	{

	    CegoAdminHandler* pSecondaryAH = 0;


	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    
	    CegoAdminHandler::ResultType res; // TODO  = pSecondaryAH->reqVerifyTS(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }

	    closeSession(pSecondaryAH);
	}	
    }
}


void CegoAdminThread::medEnableArchLog(CegoAdminHandler *pAH)
{

    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Host h;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	ListT<Chain> archIdList;
	ListT<Chain> archPathList;
	_pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);

	if ( archIdList.Size() == 0 )
	{	    
	    Chain msg = Chain("No valid archive locations available");
	    throw Exception(EXLOC, msg);	    
	}
	_pDBMng->enableArchLog(tableSet);
    }
    else
    {
	Chain msg = Chain("Archive mode just supported in single node mode");
	throw Exception(EXLOC, msg);	
    }
    
    pAH->sendResponse(Chain("Arch mode enabled"));

}


void CegoAdminThread::medDisableArchLog(CegoAdminHandler *pAH)
{

    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    Host h;
    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status == Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset is already in backup mode");
	throw Exception(EXLOC, msg);
    }

    if ( h.getName() != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + h.getName() + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	_pDBMng->disableArchLog(tableSet);
    }
    else
    {
	Chain msg = Chain("Archive mode just supported in single node mode");
	throw Exception(EXLOC, msg);
    }

    pAH->sendResponse(Chain("Arch mode disabled"));    

}


CegoAdminHandler* CegoAdminThread::getSession(const Chain& role, const Chain& hostname, const Chain& user, const Chain& pwd)
{    
    Net n (  NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN );
    
    NetHandler *pN;

    int adminPort;
    _pDBMng->getAdminPort(adminPort);
     
    try 
    {
	pN = n.connect(hostname, adminPort);	
    }
    catch ( Exception e )
    {
	throw Exception(EXLOC, Chain("Cannot get session to ") + hostname, e);	
    }

    CegoAdminHandler* pAH = new CegoAdminHandler( _pDBMng, pN);
    
    CegoAdminHandler::ResultType res = pAH->requestSession(user, pwd, false);
    if ( res != CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAH->getMsg(msg);
	closeSession(pAH);

	throw Exception(EXLOC, msg);	
    }
    
    return pAH;

}

void CegoAdminThread::closeSession(CegoAdminHandler* pAH)
{
    pAH->closeSession();
    NetHandler* pN = pAH->getNetHandle();
    delete pAH;
    delete pN;
}


Element* CegoAdminThread::checkTS(const Chain& hostType, const Chain& hostName, Element *pInfo, CegoAdminHandler *pPAH,  CegoAdminHandler *pSAH)
{
    
    Chain primaryRS;
    Chain secondaryRS;

    Chain primarySS;
    Chain secondarySS;

    Chain primaryMed;
    Chain secondaryMed;

    Chain primaryPrim;
    Chain secondaryPrim;

    Chain primarySec;
    Chain secondarySec;

    
    if ( pPAH )
    {

	Element *pPrimaryCheck = pPAH->getTableSetCheckInfo();
	
	if  ( pPrimaryCheck )
	{
	    primaryRS = pPrimaryCheck->getAttributeValue(XML_RUNSTATE_ATTR);
	    primarySS = pPrimaryCheck->getAttributeValue(XML_SYNCSTATE_ATTR);
	    primaryPrim = pPrimaryCheck->getAttributeValue(XML_PRIMARY_ATTR);
	    primarySec = pPrimaryCheck->getAttributeValue(XML_SECONDARY_ATTR);
	    primaryMed = pPrimaryCheck->getAttributeValue(XML_MEDIATOR_ATTR);
	}
    }
    if ( pSAH )
    {

	Element *pSecondaryCheck = pSAH->getTableSetCheckInfo();
	
	if ( pSecondaryCheck ) 
	{
	    secondaryRS = pSecondaryCheck->getAttributeValue(XML_RUNSTATE_ATTR);
	    secondarySS = pSecondaryCheck->getAttributeValue(XML_SYNCSTATE_ATTR);
	    secondaryPrim = pSecondaryCheck->getAttributeValue(XML_PRIMARY_ATTR);
	    secondarySec = pSecondaryCheck->getAttributeValue(XML_SECONDARY_ATTR);
	    secondaryMed = pSecondaryCheck->getAttributeValue(XML_MEDIATOR_ATTR);
	}
    }
    
    Element *pCheck =  new Element(XML_TABLESETCHECK_ELEMENT);


    
    Element *pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);

    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_RUNSTATE_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_RUNSTATE_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryRS);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryRS);
    pCheck->addContent(pAttrCheck);
    

    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_SYNCSTATE_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_SYNCSTATE_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primarySS);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondarySS);
    pCheck->addContent(pAttrCheck);


    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_PRIMARY_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_PRIMARY_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryPrim);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryPrim);
    pCheck->addContent(pAttrCheck);


    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_SECONDARY_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_SECONDARY_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primarySec);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondarySec);
    pCheck->addContent(pAttrCheck);


    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_MEDIATOR_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_MEDIATOR_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryMed);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryMed);
    pCheck->addContent(pAttrCheck);

    return pCheck;
}

Element* CegoAdminThread::getDetailedTSInfo(const Chain& tableSet)
{

    Chain tsRoot;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain runState;
    Chain syncState;
    Chain tsTicket;
    Chain archMode;
    Chain tsInitFile;
    int checkpoint;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    runState = _pDBMng->getTableSetRunState(tableSet);
    syncState = _pDBMng->getTableSetSyncState(tableSet);
    tsRoot = _pDBMng->getTSRoot(tableSet);
    tsTicket = _pDBMng->getTSTicket(tableSet);
    tsInitFile = _pDBMng->getTSInitFile(tableSet);
    checkpoint = _pDBMng->getCheckpointInterval(tableSet);
    
    if ( _pDBMng->isArchiveMode(tableSet) )
	archMode = Chain(XML_ON_VALUE);
    else
	archMode = Chain(XML_OFF_VALUE);

    Element *pTSInfo = new Element(XML_TABLESET_ELEMENT);
    
    pTSInfo->setAttribute(XML_NAME_ATTR, tableSet);
    pTSInfo->setAttribute(XML_RUNSTATE_ATTR, runState);
    pTSInfo->setAttribute(XML_SYNCSTATE_ATTR, syncState);
    pTSInfo->setAttribute(XML_PRIMARY_ATTR, primary);		
    pTSInfo->setAttribute(XML_SECONDARY_ATTR, secondary);	
    pTSInfo->setAttribute(XML_MEDIATOR_ATTR, mediator);		
    pTSInfo->setAttribute(XML_TSROOT_ATTR, tsRoot);
    pTSInfo->setAttribute(XML_TSTICKET_ATTR, tsTicket);
    pTSInfo->setAttribute(XML_TSINITFILE_ATTR, tsInitFile);
    pTSInfo->setAttribute(XML_CHECKPOINT_ATTR, Chain(checkpoint));
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);
    long clsn = _pDBMng->getCommittedLSN(tableSet);
    long nlsn = _pDBMng->getLSN(tabSetId);
    
    int tid = _pDBMng->getTID(tableSet);
    int sortAreaSize = _pDBMng->getSortAreaSize(tabSetId);
    
    Chain sysFileName = _pDBMng->getSysFileName(tableSet);
    Chain tempFileName = _pDBMng->getTmpFileName(tableSet);
    
    pTSInfo->setAttribute(XML_SYSFILE_ATTR, sysFileName);
    pTSInfo->setAttribute(XML_TEMPFILE_ATTR, tempFileName);
    
    pTSInfo->setAttribute(XML_SORTAREASIZE_ATTR, Chain(sortAreaSize));
    pTSInfo->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
    pTSInfo->setAttribute(XML_TID_ATTR, Chain(tid));
    pTSInfo->setAttribute(XML_LSN_ATTR, Chain(clsn));
    pTSInfo->setAttribute(XML_NEXTLSN_ATTR, Chain(nlsn));
    pTSInfo->setAttribute(XML_ARCHMODE_ATTR, archMode);
    
    int totalSysPages=0;
    int usedSysPages=0;
    int totalTempPages=0;
    int usedTempPages=0;
    
    Chain status =_pDBMng->getTableSetRunState(tableSet);
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	int sysfid = tabSetId;
	int tmpfid = _pDBMng->getTmpFid(tableSet);
	
	totalSysPages = _pDBMng->getNumPages(sysfid);
	usedSysPages = _pDBMng->getNumUsedPages(sysfid, _pTabMng->getLockHandle());
	totalTempPages = _pDBMng->getNumPages(tmpfid);
	usedTempPages = _pDBMng->getNumUsedPages(tmpfid, _pTabMng->getLockHandle());
	    
    }
    else
    {
	totalSysPages = _pDBMng->getSysSize(tableSet);
	totalTempPages = _pDBMng->getTmpSize(tableSet);
    }
    
    pTSInfo->setAttribute(XML_SYSTOTAL_ATTR, Chain(totalSysPages));
    pTSInfo->setAttribute(XML_SYSUSED_ATTR, Chain(usedSysPages));
    
    pTSInfo->setAttribute(XML_TEMPTOTAL_ATTR, Chain(totalTempPages));
    pTSInfo->setAttribute(XML_TEMPUSED_ATTR, Chain(usedTempPages));
    
    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> dfSizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, dfSizeList);
    
    Chain *pFileName = dfList.First();
    int *pFid = fidList.First();
    int *pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {
	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_APPFILE_VALUE));
	
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}
	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();
	pDFSize = dfSizeList.Next();
    }

   
    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {
	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_SYSFILE_VALUE));
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}

	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();
	pDFSize = dfSizeList.Next();
    }

    
    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {
	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_TEMPFILE_VALUE));
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}
	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();	
	pDFSize = dfSizeList.Next();

    }
    
    ListT<Chain> lfList;
    ListT<int> sizeList;
    ListT<Chain> statusList;
    
    _pDBMng->getLogFileInfo(tableSet, lfList, sizeList, statusList);	
    
    Chain *pLogName = lfList.First();
    int *pSize= sizeList.First();
    Chain *pStatus = statusList.First();
    
    while ( pLogName && pSize && pStatus )
    {
	
	Element *pLE = new Element(XML_LOGFILE_ELEMENT);
	pLE->setAttribute(XML_NAME_ATTR, *pLogName);
	pLE->setAttribute(XML_SIZE_ATTR, Chain(*pSize));
	
	pLE->setAttribute(XML_STATUS_ATTR, *pStatus);

	if ( *pStatus == Chain(XML_ACTIVE_VALUE) )
	{
	    pLE->setAttribute(XML_OFFSET_ATTR, Chain(_pDBMng->getLogOffset(tabSetId)));
	}
	else
	{
	    pLE->setAttribute(XML_OFFSET_ATTR, Chain(0));
	}
	
	pTSInfo->addContent(pLE);
	
	pLogName = lfList.Next();
	pSize = sizeList.Next();
	pStatus = statusList.Next();
    }
      
    ListT<Chain> archIdList;
    ListT<Chain> archPathList;
    
    _pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);
    
    Chain *pArchId = archIdList.First();
    Chain *pArchPath = archPathList.First();
    
    while ( pArchId && pArchPath )
    {
	
	Element *pNAL = new Element(XML_ARCHIVELOG_ELEMENT);
	pNAL->setAttribute(XML_ARCHID_ATTR, *pArchId);
	pNAL->setAttribute(XML_ARCHPATH_ATTR, *pArchPath);
	
	pTSInfo->addContent(pNAL);

	pArchId = archIdList.Next();	
	pArchPath = archPathList.Next();
	
    }

    return pTSInfo; 
}


void CegoAdminThread::addTableSetUsage(Element *pTSE)
{

    Chain tableSet = pTSE->getAttributeValue(XML_NAME_ATTR);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);
    
    int totalSysPages=0;
    int usedSysPages=0;
    int totalTempPages=0;
    int usedTempPages=0;
    int totalAppPages=0;
    int usedAppPages=0;

    
    Chain status =_pDBMng->getTableSetRunState(tableSet);
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	int sysfid = tabSetId;
	int tmpfid = _pDBMng->getTmpFid(tableSet);
	
	totalSysPages = _pDBMng->getNumPages(sysfid);
	usedSysPages = _pDBMng->getNumUsedPages(sysfid, _pTabMng->getLockHandle());
	totalTempPages = _pDBMng->getNumPages(tmpfid);
	usedTempPages = _pDBMng->getNumUsedPages(tmpfid, _pTabMng->getLockHandle());
	    
    }
    
    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> dfSizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, dfSizeList);
    
    Chain *pFileName = dfList.First();
    int *pFid = fidList.First();
    
    while ( pFileName && pFid )
    {
	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalAppPages+=totalPages;
	usedAppPages+=usedPages;

	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    
    while ( pFileName && pFid )
    {
		
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalSysPages+=totalPages;
	usedSysPages+=usedPages;
		
	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    
    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    
    while ( pFileName && pFid )
    {
		
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalTempPages+=totalPages;
	usedTempPages+=usedPages;
	       
	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    int percentSysUsage = 0;
    if ( totalSysPages > 0 )
	percentSysUsage = usedSysPages  / ( totalSysPages  / 100 );

    int percentTempUsage = 0;
    if ( totalTempPages > 0 )
	percentTempUsage = usedTempPages  / ( totalTempPages  / 100 );

    int percentAppUsage = 0;
    if ( totalAppPages > 0 )
	percentAppUsage = usedAppPages  / ( totalAppPages  / 100 );

    pTSE->setAttribute(XML_SYSUSAGE_ATTR, Chain(percentSysUsage) + Chain("%"));  
    pTSE->setAttribute(XML_TEMPUSAGE_ATTR, Chain(percentTempUsage) + Chain("%"));  
    pTSE->setAttribute(XML_APPUSAGE_ATTR, Chain(percentAppUsage) + Chain("%"));  
    
}

Element* CegoAdminThread::getTransactionInfo(const Chain& tableSet)
{

    Element *pTA = new Element(XML_TAINFO_ELEMENT);

    CegoTransactionManager *pTM = _pTabMng->getTransactionManager();

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    ListT<Chain> rboList;    

    _pTabMng->getObjectList(tabSetId, CegoObject::RBSEG, rboList);

    Chain *pRBO = rboList.First();
    while ( pRBO )
    {
	Element *pTE = new Element(XML_TAENTRY_ELEMENT);
	
	int numOp;
	pTM->getTransactionInfo(tabSetId, *pRBO, numOp);
	pTE->setAttribute(XML_RBO_ATTR, *pRBO);
	pTE->setAttribute(XML_NUMOP_ATTR, Chain(numOp));
	
	pTA->addContent(pTE);

	pRBO = rboList.Next();
    }
    
    return pTA;
}


Element* CegoAdminThread::getBUStatInfo(const Chain& tableSet)
{

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online to list backup statistics");
    }

    Element *pBUS = new Element(XML_BUSINFO_ELEMENT);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    CegoTableObject boe;    
    _pTabMng->getObject(tabSetId, Chain(SYSTAB_BUSTAT_ID), CegoObject::SYSTEM, boe);
    ListT<CegoField> fvl = boe.getSchema();

    CegoObjectCursor* pC = _pTabMng->getObjectCursor(tabSetId, SYSTAB_BUSTAT_ID, SYSTAB_BUSTAT_ID, CegoObject::SYSTEM);
    
    CegoDataPointer dp;    
    int len;
    char *pc;
    
    pc = (char*)pC->getFirst(len, dp);		
    
    while ( pc && len > 0 )
    {
	int tid;
	int tastep;	    
	CegoTupleState ts;
	
	CegoQueryHelper qh;

	qh.decodeFVL(fvl, pc, len, tid, tastep, ts);			    

	Element *pBUE = new Element(XML_BUENTRY_ELEMENT);
		
	CegoField *pF = fvl.First();
	while ( pF )	    
	{
	    if ( pF->getAttrName() == Chain(SYSTAB_TS_ATTR) )		
		pBUE->setAttribute(XML_TIMESTAMP_ATTR, pF->getValue().valAsChain());
	    else if ( pF->getAttrName() == Chain(SYSTAB_BUINFO_ATTR) )
		pBUE->setAttribute(XML_BUINFO_ATTR, pF->getValue().valAsChain());
	    else if ( pF->getAttrName() == Chain(SYSTAB_BUMSG_ATTR) )
	    {
		if ( pF->getValue().isNull() )
		    pBUE->setAttribute(XML_BUMSG_ATTR, Chain("none"));
		else			 
		    pBUE->setAttribute(XML_BUMSG_ATTR, pF->getValue().valAsChain());   
	    }
	    pF = fvl.Next();
	}
	
	pBUS->addContent(pBUE);

	pc = (char*)pC->getNext(len, dp);

    }

    pC->abort();
    delete pC;


    
    return pBUS;
}



Element* CegoAdminThread::getObjectInfo(const Chain& tableSet)
{

    Element *pOI = new Element(XML_OBJLIST_ELEMENT);

    int tabSetId = _pDBMng->getTabSetId(tableSet);    

    ListT<Chain> tableList;
    _pTabMng->getObjectList(tabSetId, CegoObject::TABLE, tableList);

    Chain *pTableName = tableList.First();

    while ( pTableName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_TABOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pTableName);	
	pOI->addContent(pOE);

	pTableName = tableList.Next();
    }

    ListT<Chain> idxList;
    _pTabMng->getObjectList(tabSetId, CegoObject::INDEX, idxList);

    Chain *pIndexName = idxList.First();

    while ( pIndexName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_IDXOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pIndexName);	
	pOI->addContent(pOE);

	pIndexName = idxList.Next();
    }

    ListT<Chain> keyList;
    _pTabMng->getObjectList(tabSetId, CegoObject::FKEY, keyList);

    Chain *pKeyName = keyList.First();

    while ( pKeyName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_FKEYOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pKeyName);	
	pOI->addContent(pOE);

	pKeyName = keyList.Next();
    }

    ListT<Chain> checkList;
    _pTabMng->getObjectList(tabSetId, CegoObject::CHECK, checkList);

    Chain *pCheckName = checkList.First();

    while ( pCheckName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_CHECKOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pCheckName);	
	pOI->addContent(pOE);

	pCheckName = checkList.Next();
    }

    ListT<Chain> viewList;
    _pTabMng->getObjectList(tabSetId, CegoObject::VIEW, viewList);

    Chain *pViewName = viewList.First();

    while ( pViewName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_VIEWOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pViewName);	
	pOI->addContent(pOE);

	pViewName = viewList.Next();
    }

    ListT<Chain> procList;
    _pTabMng->getObjectList(tabSetId, CegoObject::PROCEDURE, procList);

    Chain *pProcName = procList.First();

    while ( pProcName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_PROCOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pProcName);	
	pOI->addContent(pOE);

	pProcName = procList.Next();
    }
    
    return pOI;
}
    
