#ifndef _CEGODBHANDLER_H_INCLUDED_
#define _CEGODBHANDLER_H_INCLUDED_
///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDbHandler.h
// ---------------
// Cego db handler class definition
//     
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2010 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// INTERFACE MODULE
//
// Class: CegoDbHandler
// 
// Description: The CegoDbHandler implements the XML based protocol for db client network access. 
//
// Status: QG-2.6
//
///////////////////////////////////////////////////////////////////////////////


#include <lfcbase/Logger.h>
#include <lfcbase/NetHandler.h>

#include <lfcxml/XMLSuite.h>

#include "CegoSerial.h"
#include "CegoObject.h"
#include "CegoTableObject.h"
#include "CegoProcVar.h"
#include "CegoBlob.h"
#include "CegoModule.h"

class CegoDbHandler {
    
public:

    enum ProtocolType { XML, SERIAL };

    enum RequestType { QUERY,
		       QUERYABORT,
		       DBPRODINFO,
		       INSERT, 
		       DELETETABLE,
		       UPDATE,
		       CREATETABLE, 
		       CREATEVIEW,
		       CREATEPROCEDURE, 
		       ALTERTABLE, 
		       DROPOBJECT, 
		       CREATEINDEX,
		       CREATECHECK,
		       OBJECTINFO, 
		       GETTABLE, 
		       GETOBJLIST,
		       GETOBJLISTBYTABLE,
		       OBJRENAME,
		       REORGOBJ,
		       SYNC,
		       GETPAGECOUNT,
		       PUTBLOB,
		       GETBLOB,
		       DELBLOB,
		       STARTTRANSACTION,
		       COMMITTRANSACTION,
		       ROLLBACKTRANSACTION,
		       GETTID,
		       SESSION_CLOSE,
		       REQTIMEOUT,
		       UNKNOWN };

    enum ResultType { DB_OK, DB_ERROR, DB_DATA, DB_INFO, DB_FIN };


    CegoDbHandler(NetHandler *pN, ProtocolType pt, CegoModule *pModule);
    ~CegoDbHandler();

    ////////////////////////////////////
    // client and client node methods //
    ////////////////////////////////////

    ResultType requestSession(const Chain& tableSet, const Chain& user, const Chain& password, bool doEncrypt = true);
    ResultType closeSession();
    ResultType reqQueryOp(const Chain& cmd);
    ResultType reqQueryAbort(long idx);

    CegoObject::ObjectType getObjType();
    Element* getObjElement();

    const Chain& getMsg();
    long getTid();
    long getAffected();
    const ListT<CegoField>& getSchema();
    const Chain& getFormat();

    void getProcResult(ListT<CegoProcVar>& outParamList, CegoFieldValue& retValue);
    ResultType receiveTableData(const ListT<CegoField>& schema);
    ResultType receiveTableData(const ListT<CegoField>& schema, ListT<CegoFieldValue>& fvl);
    void abortQuery();
    void resetQuery();
    bool wasReset();

    /////////////////////////
    // server node methods //
    /////////////////////////

    bool acceptSession();
    RequestType acceptRequest();
    
    Chain getQueryArg();

    void collectSchema(const ListT<CegoField>& schema, const Chain& format = Chain(""));
    void collectData(const ListT<CegoField>& schema);
    void collectData(const ListT<CegoField>& schema, const ListT<CegoFieldValue>& fvl);

    void sendCollectedData();
    void sendFinishData();
    void sendErrorData(const Chain& msg);
    void sendProdInfo();
    void sendObjInfo(CegoDecodableObject& oe);

    void sendSessionConfirm(const Chain& msg, 
			    long tid, 
			    const Chain& dbName, 
			    const Chain& dbVersion);
	
    void sendResponse(const Chain& msg, long affected = 0 );
    void sendError(const Chain& msg );
    void sendBlobInfo(int fileId, int pageId);
    void sendBlobSize(long blobSize);


    ResultType putBlob(const Chain& tableSet, CegoBlob& blob);
    ResultType getBlob(const Chain& tableSet, CegoBlob& blob);
    ResultType delBlob(const Chain& tableSet, CegoBlob& blob);

    void getPutBlobArg(Chain& tableSet, long& blobSize);
    void getGetBlobArg(Chain& tableSet, int& fileId, int& pageId);

    void sendProcResult(const Chain& msg, const ListT<CegoProcVar>& outParamList, CegoFieldValue *pRetVal);

    const Chain& getTableSet();
    const Chain& getUser();
    const Chain& getPassword();

    NetHandler* getNetHandler();

protected:

    ProtocolType _protType;

    // xml request
    ResultType sendXMLReq(const Chain& reqType, Element* pRoot);
    
    // native serial request
    ResultType sendSerialReq();

    ResultType getMoreTableData();

    NetHandler *_pN;
    CegoModule *_pModule;

    Chain _tableSet;
    Chain _user;
    Chain _password;


    // for xml protocol
    XMLSuite _xml;
    Element **_pRow;
    ListT<Element*> _rowList;

    // for native protocol
    CegoSerial* _pSer;
    Chain _serQueryCmd;
    long  _serTid;
    long  _serAffected;
    Chain _serMsg;
    Chain _serFormat;

    Chain _serTableSet;
    long _serBlobSize;
    int _serFileId;
    int _serPageId;

    bool _serSync;
    Chain _serDbName;
    Chain _serDbVersion;

    ListT<CegoField> _serSchema;

    ListT<CegoProcVar> _outParamList;
    CegoFieldValue _retValue;    

    bool _wasReset;

    unsigned long _modId;
};



#endif




