#ifndef _CEGOLOCKHANDLER_H_INCLUDED_
#define _CEGOLOCKHANDLER_H_INCLUDED_
///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoLockHandler.h
// -----------------
// Cego lock handler class definition
//     
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2010 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// INTERFACE MODULE
//
// Class: CegoLockHandler
// 
// Description: All semaphore lock operations on database files, pool buffers, pages and records level
//              are managed using the LockHandler class.
//              A number of available semaphores is distributed among all required lock operations.
//              
//             
// Status: QG-2.6
//
///////////////////////////////////////////////////////////////////////////////

// cego includes
#include "CegoDefs.h"
#include "CegoDataPointer.h"
#include "CegoBufferPage.h"
#include "CegoObject.h"

class CegoDatabaseManager;

class CegoLockHandler {
    
public:

    enum LockMode { READ, WRITE };

    CegoLockHandler(CegoDatabaseManager* pDBMng);
    ~CegoLockHandler();

    void initLocks();
    void deleteLocks();

    CegoBufferPage getPage(unsigned long lockId);
    
    unsigned long lockRecord(const CegoBufferPage &bp, const CegoDataPointer& dp, enum LockMode lockMode);
    void unlockRecord(unsigned long id);

    unsigned long lockRBRecord(const CegoBufferPage &bp, const CegoDataPointer& dp, enum LockMode lockMode);
    void unlockRBRecord(unsigned long id);

    unsigned long lockSysRecord(const CegoBufferPage &bp, const CegoDataPointer& dp, enum LockMode lockMode);
    void unlockSysRecord(unsigned long id);

    unsigned long lockSysPage(int fileId, int pageId, enum LockMode lockMode);
    void unlockSysPage(unsigned long id);
    
    unsigned long lockData(CegoObject::ObjectType type, int fileId, int pageId, enum LockMode lockMode);
    void unlockData(CegoObject::ObjectType type, unsigned long id);

    void lockDataFile(int fileId, enum LockMode lockMode);
    void unlockDataFile(int fileId);

    void lockBufferPool(int hashId, enum LockMode lockMode);
    void unlockBufferPool(int hashId);

    void lockBufferPool();
    void unlockBufferPool();

    void getLockStat(int lockId, Chain& lockName, long& lockCount, long &numRdLock, long &numWrLock, long &sumRdDelay, long &sumWrDelay);
    void getAggLockStat(const Chain& lockName, int& numLock, long& lockCount, long &numRdLock, long &numWrLock, long &sumRdDelay, long &sumWrDelay);

private:

    unsigned long lockDataPage(int fileId, int pageId, enum LockMode lockMode);
    void unlockDataPage(unsigned long id);

    unsigned long lockIndexPage(int fileId, int pageId, enum LockMode lockMode);
    void unlockIndexPage(unsigned long id);

    unsigned long lockRBPage(int fileId, int pageId, enum LockMode lockMode);
    void unlockRBPage(unsigned long id);

    unsigned long _nextLockId;

    int *_lockCount; 

    struct RecordLock {
 	unsigned long lockId;
	CegoDataPointer dp;
	int semId;
	CegoBufferPage bp;
    } *_recordLock, *_orderRecordLock, *_rbRecordLock, *_sysRecordLock; 
    
    struct PageLock {
	unsigned long lockId;
	int fileId;
	int pageId;
	int semId;
    } *_sysPageLock, 
	  *_dataPageLock, 
	  *_idxPageLock,
	  *_rbPageLock;
           
    bool _isDisabled;
    
    int _numRecSema;
    int _numRBRecSema;
    int _numSysRecSema;

    int _numSysPageSema;
    int _numDataPageSema;
    int _numIdxPageSema;
    int _numRBPageSema;
    int _numDataFileSema;
    int _numBufferPoolSema;

    int _numSema;

    int _pageLockTimeout;
    int _recLockTimeout;
    int _fileLockTimeout;
    int _poolLockTimeout;
    int _numLockTries;

    CegoDatabaseManager *_pDBMng;

    unsigned long _modId;

};

#endif




