///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoNet.cc
// ----------
// Cego net client interface
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2010 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: CegoNet
// 
// Description: 
//
// Status: QG-2.6
//
///////////////////////////////////////////////////////////////////////////////

// cego includes
#include "CegoNet.h"
#include "CegoDefs.h"

#include <lfcbase/Net.h>

CegoNet::CegoNet(CegoDbHandler::ProtocolType protType, const Chain& logFile, const Chain& logMode)
{
    _protType = protType;
    _logFile = logFile;
    _logMode = logMode;
    _pN = 0;
    _pSH = 0;
    _pModule = 0;
    _isFetchable=false;
}

CegoNet::~CegoNet()
{
    if ( _pSH )
	delete _pSH;
    if ( _pN ) 
	delete _pN;
    if ( _pModule )
	delete _pModule;
}

void CegoNet::connect(const Chain& serverName, int port, const Chain& tableSet, const Chain& user, const Chain& pwd)
{
    Net n (  NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN );

    _pN = n.connect(serverName, port);

    if ( _logFile.length() > 1 )
	_pModule = new CegoModule(_logFile);
    else
	_pModule = new CegoModule;

    int modId = 100;

    if ( _logMode == Chain("notice"))
	_pModule->logModule(modId, "dbdimp", Logger::NOTICE);
    else if ( _logMode == Chain("error"))
	_pModule->logModule(modId, "dbdimp", Logger::LOGERR);
    else if ( _logMode == Chain("debug"))
	_pModule->logModule(modId, "dbdimp", Logger::DEBUG);
    else
	_pModule->logModule(modId, "dbdimp", Logger::NONE);

    
    _pSH = new CegoDbHandler(_pN, _protType, _pModule);

    CegoDbHandler::ResultType res = _pSH->requestSession(tableSet, user, pwd);
    if ( res != CegoDbHandler::DB_OK )
    {
	Chain msg = _pSH->getMsg();
	throw Exception(EXLOC, msg);
    }
}
   
long CegoNet::doQuery(const Chain& query)
{        

    CegoDbHandler::ResultType res;

    _isFetchable=false;

    res = _pSH->reqQueryOp(query);

    if ( res == CegoDbHandler::DB_OK )
    {
	Chain msg = _pSH->getMsg();
	long affCount = _pSH->getAffected();
	return affCount;
    }
    else if ( res == CegoDbHandler::DB_DATA )
    {
	_isFetchable=true;
	return 0;
    }
    else if ( res == CegoDbHandler::DB_FIN )
    {	
	_isFetchable=false;
	return 0;
    }
    else if ( res == CegoDbHandler::DB_ERROR )
    {
	Chain msg = _pSH->getMsg();
	throw Exception(EXLOC, msg);      
    }

    return 0;

}

bool CegoNet::fetchData(const ListT<CegoField>& schema, ListT<CegoFieldValue>& fvl)
{

    if ( _isFetchable == true )
    {
	CegoDbHandler::ResultType res;

	res = _pSH->receiveTableData(schema, fvl);
	
	if ( res == CegoDbHandler::DB_DATA )
	{
	    _isFetchable = true;
	}
	else
	{
	    _isFetchable = false;
	}
    }
    return _isFetchable;
}

bool CegoNet::isFetchable()
{        
    return _isFetchable;
}

void CegoNet::abortQuery()
{        
    _pSH->abortQuery();
}

void CegoNet::resetQuery()
{        
    _pSH->resetQuery();
}

bool CegoNet::wasReset()
{        
    return _pSH->wasReset();
}

const Chain& CegoNet::getMsg()
{
    return _pSH->getMsg();
}

long CegoNet::getAffected()
{
    return _pSH->getAffected();
}

void CegoNet::getSchema(ListT<CegoField>& schema)
{
    schema = _pSH->getSchema();
}

void CegoNet::getProcResult(ListT<CegoProcVar>& outParamList, CegoFieldValue& retValue)
{
    _pSH->getProcResult(outParamList, retValue);
}
   
void CegoNet::disconnect()
{        
    _pSH->closeSession();
}
