///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoTypeConverter.cc
// --------------------
// Cego type converter implementation
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2010 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: CegoTypeConverter
// 
// Description: 
//
// Status: QG-2.6
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/Exception.h>

// cego includes
#include "CegoTypeConverter.h"
#include "CegoXMLdef.h"

const char *CEGO_TYPE_MAP[] =
{
    "int",
    "long",
    "string",
    "bool",
    "datetime",
    "bigint",
    "float",
    "double",
    "decimal",
    "fixed",
    "smallint",
    "tinyint",
    "blob",
    "null",
};

const int CEGO_JAVATYPE_MAP[] =
{
    4,
    2,
    12,
    16,
    93,
    -5,
    6,
    8,
    3,
    3,
    5,
    -6,
    2004,
    0,
};

CegoTypeConverter::CegoTypeConverter()
{
}

CegoTypeConverter::~CegoTypeConverter()
{
}

Chain CegoTypeConverter::getTypeString(CegoDataType type) const
{
    switch  ( type ) 
    {
    case INT_TYPE:
	return Chain(XML_INT_VALUE);
    case LONG_TYPE:
	return Chain(XML_LONG_VALUE);
    case VARCHAR_TYPE:
	return Chain(XML_STRING_VALUE);
    case BOOL_TYPE:
	return Chain(XML_BOOL_VALUE);
    case DATETIME_TYPE:
	return Chain(XML_DATETIME_VALUE);
    case BIGINT_TYPE:
	return Chain(XML_BIGINT_VALUE);
    case FLOAT_TYPE:
	return Chain(XML_FLOAT_VALUE);
    case DOUBLE_TYPE:
	return Chain(XML_DOUBLE_VALUE);
    case DECIMAL_TYPE:
	return Chain(XML_DECIMAL_VALUE);
    case FIXED_TYPE:
	return Chain(XML_FIXED_VALUE);
    case SMALLINT_TYPE:
	return Chain(XML_SMALLINT_VALUE);
    case TINYINT_TYPE:
	return Chain(XML_TINYINT_VALUE);
    case BLOB_TYPE:
	return Chain(XML_BLOB_VALUE);
    case NULL_TYPE:
	return Chain(XML_NULL_VALUE);
    }
}

int CegoTypeConverter::getTypeLen(CegoDataType type, const Chain& val) const
{
    switch  ( type ) 
    {
    case INT_TYPE:
	return sizeof(int);
    case LONG_TYPE:
	return sizeof(long);
    case BOOL_TYPE:
	return sizeof(char);
    case DATETIME_TYPE:
	return sizeof(int);
    case VARCHAR_TYPE:
    case BIGINT_TYPE:
    case FIXED_TYPE:
    case DECIMAL_TYPE:
	return val.length();
    case FLOAT_TYPE:
	return sizeof(float);
    case DOUBLE_TYPE:
	return sizeof(double);
    case SMALLINT_TYPE:
	return sizeof(short);
    case TINYINT_TYPE:
	return sizeof(char);
    case BLOB_TYPE:
	return 2 * sizeof(int);
    case NULL_TYPE:
	return 0;
    }
}

CegoDataType CegoTypeConverter::getTypeId(const Chain& typeString)
{

    CegoDataType type;
    
    if ( typeString == Chain(XML_STRING_VALUE) )
    {
	type = VARCHAR_TYPE;	
    } 
    else if ( typeString == Chain(XML_INT_VALUE) )
    {		
	type = INT_TYPE;
    }
    else if ( typeString == Chain(XML_LONG_VALUE) )
    {		
	type = LONG_TYPE;
    }
    else if ( typeString == Chain(XML_BOOL_VALUE) )
    {
	type = BOOL_TYPE;
    }
    else if ( typeString == Chain(XML_DATETIME_VALUE) )
    {
	type = DATETIME_TYPE;
    }
    else if ( typeString == Chain(XML_DECIMAL_VALUE) )
    {
	type = DECIMAL_TYPE;
    }
    else if ( typeString == Chain(XML_FIXED_VALUE) )
    {
	type = FIXED_TYPE;
    }
    else if ( typeString == Chain(XML_BIGINT_VALUE) )
    {
	type = BIGINT_TYPE;
    }
    else if ( typeString == Chain(XML_SMALLINT_VALUE) )
    {
	type = SMALLINT_TYPE;
    }
    else if ( typeString == Chain(XML_TINYINT_VALUE) )
    {
	type = TINYINT_TYPE;
    }
    else if ( typeString == Chain(XML_FLOAT_VALUE) )
    {
	type = FLOAT_TYPE;
    }
    else if ( typeString == Chain(XML_DOUBLE_VALUE) )
    {
	type = DOUBLE_TYPE;
    }
    else if ( typeString == Chain(XML_BLOB_VALUE) )
    {
	type = BLOB_TYPE;
    }
    return type;
}

Chain CegoTypeConverter::getObjectTypeString(CegoObject::ObjectType type) const
{    
    switch ( type )
    {
    case CegoObject::SYSTEM:
	return Chain(XML_SYSOBJ_VALUE);
    case CegoObject::TABLE:
	return Chain(XML_TABOBJ_VALUE);
    case CegoObject::PINDEX:
	return Chain(XML_PIXOBJ_VALUE);
    case CegoObject::UINDEX:
	return Chain(XML_UIXOBJ_VALUE);
    case CegoObject::INDEX:
	return Chain(XML_IDXOBJ_VALUE);
    case CegoObject::VIEW:
	return Chain(XML_VIEWOBJ_VALUE);
    case CegoObject::RBSEG:
	return Chain(XML_RBSEGOBJ_VALUE);
    case CegoObject::FKEY:
	return Chain(XML_FKEYOBJ_VALUE);
    case CegoObject::PROCEDURE:
	return Chain(XML_PROCOBJ_VALUE);
    default:
	throw Exception(EXLOC, "Unknown type id");
    }
}

CegoObject::ObjectType CegoTypeConverter::getObjectTypeId(const Chain& typeString)
{
 
    if ( typeString == Chain(XML_SYSOBJ_VALUE) )
	return CegoObject::SYSTEM;
    if ( typeString == Chain(XML_TABOBJ_VALUE) )
	return CegoObject::TABLE;
    if ( typeString == Chain(XML_PIXOBJ_VALUE) )
	return CegoObject::PINDEX;
    if ( typeString == Chain(XML_UIXOBJ_VALUE) )
	return CegoObject::UINDEX;
    if ( typeString == Chain(XML_IDXOBJ_VALUE) )
	return CegoObject::INDEX;
    if ( typeString == Chain(XML_VIEWOBJ_VALUE) )
	return CegoObject::VIEW;
    if ( typeString == Chain(XML_RBSEGOBJ_VALUE) )
	return CegoObject::RBSEG;
    if ( typeString == Chain(XML_FKEYOBJ_VALUE) )
	return CegoObject::FKEY;
    if ( typeString == Chain(XML_PROCOBJ_VALUE) )
	return CegoObject::PROCEDURE;
    throw Exception(EXLOC, Chain("Unknown object type"));
}
