///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdmMon.cc  
// -------------                                                        
// Cego admin monitor
//
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdmMon
// 
// Description: Cursor based administration database administration frontend
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////


#ifndef CGNOCURSES

#include <stdio.h>
#include <stdlib.h>

#include "CegoAdmMon.h"
#include "CegoXMLdef.h"


#define COL_HEADER_PASSIVE 1
#define COL_HEADER_ACTIVE 2
#define COL_TITLE 3
#define COL_CONTENT 4

#define LEFTOFFSET 1
#define TOPOFFSET 1

#define KEY_TAB 9
#define KEY_RETURN 10
#define KEY_ESC 27
#define KEY_DEL 127

#define INPUT_TYPE_STRING "S"
#define INPUT_TYPE_ID "I"
#define INPUT_TYPE_NUMBER "N"
#define INPUT_TYPE_PWD "P"

#define BUFFERPOOLMODE 1
#define DBTHREADMODE 2
#define ADMTHREADMODE 3
#define LOGTHREADMODE 4
#define TSMODE 5
#define ROLMODE 6
#define USRMODE 7
#define REFRESHMODE 8
#define QUITMODE 9

#define DEFREFINT 3000

// action defines
#define TS_CREATE_ACTION 1
#define TS_START_ACTION 2
#define TS_STOP_ACTION 3
#define TS_DROP_ACTION 4
#define TS_REMOVE_ACTION 5
#define TS_ADD_FILE 6
#define TS_LIST_FILE 7

// ts status definition
#define TS_DEFINED_STATUS "DEFINED"
#define TS_ONLINE_STATUS "ONLINE"
#define TS_OFFLINE_STATUS "OFFLINE"

// user action defines
#define USER_TRACEON_ACTION 1
#define USER_TRACEOFF_ACTION 2
#define USER_DELETE_ACTION 3

// role action defines
#define ROLE_PERM_LIST_ACTION 1
#define ROLE_PERM_SET_ACTION 2
#define ROLE_ASSIGN_ACTION 3
#define ROLE_REMOVE_ACTION 4
#define ROLE_DROP_ACTION 5

#define TERM_WIDTH 80
#define TERM_HEIGHT 40

#define ERROR_TITLE "ERROR"
#define INFO_TITLE "INFORMATION"

#define INFOMASK_WIDTH 50
#define INFOMASK_HEIGHT 5

#define HELPMASK_WIDTH 50
#define HELPMASK_HEIGHT 30

#define CUR_ROW_OFFSET 3
#define CUR_COL_OFFSET 20
#define VAL_MAX_LEN 30
#define VAL_MAX_NUM 20
#define OK_COL_POS 6
#define ABORT_COL_POS 16
 

CegoAdmMon::CegoAdmMon(CegoAdminHandler *pAH) 
{

    _pAH = pAH;
    
    /*  Initialize ncurses  */
    
    if ( (_mainwin = initscr()) == NULL ) 
    {
	throw Exception(EXLOC, Chain("Error initializing ncurses"));
    }

    curs_set(0);

    start_color();    
    
    init_pair(COL_HEADER_PASSIVE,  COLOR_CYAN,     COLOR_BLACK);    
    init_pair(COL_HEADER_ACTIVE,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_TITLE,  COLOR_WHITE,     COLOR_BLACK);
    init_pair(COL_CONTENT,  COLOR_YELLOW,     COLOR_BLACK);

    noecho();                  /*  Turn off key echoing                 */
    keypad(_mainwin, TRUE);     /*  Enable the keypad for non-char keys  */
    
    timeout(0);
    
}

CegoAdmMon::~CegoAdmMon()  
{    
    delwin(_mainwin);
    endwin();
    refresh();
}

void CegoAdmMon::showMonitor()
{    
    
    _refInterval = DEFREFINT;    

    bool modFlag = false;
    
    bool doQuit = false;
    
    _menuSelected = BUFFERPOOLMODE;

    while ( doQuit == false ) 
    {
	
	clear();

	showHeader();
			
	if ( _menuSelected == REFRESHMODE )
	{
	    int r = showRefreshMenu();
	    if ( r > 0 )
		_refInterval = r; 
	}
	else
	{

	    switch ( _menuSelected )
	    {
	    case BUFFERPOOLMODE:
	    {
		showPool();
		break;
	    }
	    case DBTHREADMODE:
	    {
		showDBThread();
		break;
	    }
	    case ADMTHREADMODE:
	    {
		showAdmThread();
		break;
	    }
	    case LOGTHREADMODE:
	    {
		showLogThread();
		break;
	    }
	    case TSMODE:
	    {
		modFlag = showTableSet();
		break;
	    }
	    case USRMODE:
	    {
		modFlag = showUser();
		break;
	    }
	    case ROLMODE:
	    {
		modFlag = showRole();
		break;
	    }

	    case QUITMODE:
	    {
		doQuit = true;
		break;
	    }
	    }
	}
    }
}

void CegoAdmMon::showHeader()
{

    
    bool isSelected=false;

    while( isSelected == false )
    {	

	int c = getch();
	
	switch(c)
	{	
	case KEY_RIGHT:
	    if ( _menuSelected <  QUITMODE)
		_menuSelected++;
	    else
		_menuSelected = 1;
	    break;
	case KEY_LEFT:
	    if ( _menuSelected > 1  )
		_menuSelected--;
	    else
		_menuSelected = QUITMODE;
	    break;
	case KEY_RETURN:
	{
	    isSelected = true;
	    break;
	}
	case 'h':
	{
	    showHelp();
	    break;
	}
	}

	refreshHeader();
    }
}

void CegoAdmMon::refreshHeader()
{
    int colno=LEFTOFFSET;
    attroff(A_REVERSE);

    if ( _menuSelected == BUFFERPOOLMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "BufferPool");
    colno+=11;
    attroff(A_REVERSE);

    if ( _menuSelected == DBTHREADMODE )
    {	
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "DBThread");
    colno+=9;
    attroff(A_REVERSE);
    
    if ( _menuSelected == ADMTHREADMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "AdmThread");
    colno+=10;
    attroff(A_REVERSE);

    if ( _menuSelected == LOGTHREADMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "LogThread");
    colno+=10;
    attroff(A_REVERSE);
    
    if ( _menuSelected == TSMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "TableSet");
    colno+=9;
    attroff(A_REVERSE);
    
    if ( _menuSelected == ROLMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "Role");
    colno+=5;
    attroff(A_REVERSE);
    
    if ( _menuSelected == USRMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "User");
    colno+=5;
    attroff(A_REVERSE);
    
    if ( _menuSelected == REFRESHMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    mvprintw(0, colno, "Interval");
    colno+=9;
    attroff(A_REVERSE);
    
    if ( _menuSelected == QUITMODE )
    {
	attron(A_REVERSE);
	color_set(COL_HEADER_ACTIVE, NULL);
    }
    else
    {
	color_set(COL_HEADER_PASSIVE, NULL);
    }
    
    mvprintw(0, colno, "Quit");
    attroff(A_REVERSE);

    refresh();
}


void CegoAdmMon::showPool()
{
    
    bool showIt = true;

    while ( showIt )
    {
	ListT<ListT<CegoFieldValue> > poolinfo1;
	ListT<ListT<CegoFieldValue> > poolinfo2;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	
	res = _pAH->reqPoolInfo();
	
	res = _pAH->reqPoolInfo();
	_pAH->getPoolInfo(oe, poolinfo1, poolinfo2);	    	    
	
	WINDOW *poolWin1;
	WINDOW *poolWin2;
	
	poolWin1 = newwin(10, 41, 2, 2);
	keypad(poolWin1, TRUE);
	
	poolWin2 = newwin(10, 41, 12, 2);
	keypad(poolWin2, TRUE);

	box(poolWin1, 0, 0);
	box(poolWin2, 0, 0);
	
	int rowno=1; 
	
	ListT<CegoFieldValue> *pFVL = poolinfo1.First();
	
	while ( pFVL )
	{
	    int colno=2;    
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		wcolor_set(poolWin1, COL_TITLE, NULL);
		mvwprintw(poolWin1, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    colno+=18;
	    
	    pFV = pFVL->Next();
	    if ( pFV )
	    {
		wcolor_set(poolWin1, COL_CONTENT, NULL);
		mvwprintw(poolWin1, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    rowno++;
	    pFVL = poolinfo1.Next();
	}
	
	rowno=1; 
	pFVL = poolinfo2.First();
	
	while ( pFVL )
	{
	    int colno=2;    
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		wcolor_set(poolWin2, COL_TITLE, NULL);
		mvwprintw(poolWin2, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    colno+=18;
	    
	    pFV = pFVL->Next();
	    if ( pFV )
	    {
		wcolor_set(poolWin2, COL_CONTENT, NULL);
		mvwprintw(poolWin2, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    rowno++;
	    pFVL = poolinfo2.Next();
	}
	
	
	wrefresh(poolWin1);
	wrefresh(poolWin2);

	wtimeout(poolWin1, _refInterval);
	int c = wgetch(poolWin1);
	if ( c == KEY_ESC )
	{
	    showIt = false;
	    // _menuSelected = HEADERMODE;
	}

	delwin(poolWin1);
	delwin(poolWin2);
	
	refresh();
	
    }
}

void CegoAdmMon::showDBThread()
{

    bool showIt = true;

    while ( showIt )
    {

	ListT<ListT<CegoFieldValue> > dbthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
	
	res = _pAH->reqDbThreadInfo();
	_pAH->getDbThreadInfo(oe, dbthreadinfo, format);
	
	WINDOW *threadWin;
	
	threadWin = newwin(dbthreadinfo.Size() + 3, 97, 2, 2);
	keypad(threadWin, TRUE);
	
	box(threadWin, 0, 0);
	
	int rowno=TOPOFFSET;
	int colno=LEFTOFFSET;
	
	color_set(COL_TITLE, NULL);
	wattron(threadWin, A_BOLD);
	mvwprintw(threadWin, rowno, colno, "%s", "DB Thread");
	colno = colno + 25;
	mvwprintw(threadWin, rowno, colno, "%s", "NumConReq");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "NumQueryReq");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Load");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "SortAlloc");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Status");
	wattroff(threadWin, A_BOLD);

	rowno++;
	
	color_set(COL_CONTENT, NULL);
	ListT<CegoFieldValue> *pFVL = dbthreadinfo.First();
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 25;
	    }
	    if ( pFV )
	    {		       
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    
	    rowno++;
	    pFVL = dbthreadinfo.Next();
	}
	
	wrefresh(threadWin);

	wtimeout(threadWin, _refInterval);
	int c = wgetch(threadWin);
	if ( c == KEY_ESC )
	{
	    showIt = false;
	}

	delwin(threadWin);
	refresh();
    }
}

void CegoAdmMon::showAdmThread()
{

    bool showIt = true;

    while ( showIt )
    {

	ListT<ListT<CegoFieldValue> > admthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
	
	res = _pAH->reqAdmThreadInfo();
	_pAH->getAdmThreadInfo(oe, admthreadinfo, format);
	
	WINDOW *threadWin;
	
	threadWin = newwin(admthreadinfo.Size() + 3, 97, 2, 2);
	keypad(threadWin, TRUE);
	
	box(threadWin, 0, 0);
	
	int rowno=TOPOFFSET;
	int colno=LEFTOFFSET;
	
	color_set(COL_TITLE, NULL);
	wattron(threadWin, A_BOLD);
	mvwprintw(threadWin, rowno, colno, "%s", "Admin Thread");
	colno = colno + 25;
	mvwprintw(threadWin, rowno, colno, "%s", "NumRequest");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Load");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Status");
	wattroff(threadWin, A_BOLD);

	rowno++;
	
	color_set(COL_CONTENT, NULL);
	ListT<CegoFieldValue> *pFVL = admthreadinfo.First();
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 25;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    
	    rowno++;
	    pFVL = admthreadinfo.Next();
	}
	
	wrefresh(threadWin);
	
	wtimeout(threadWin, _refInterval);
	int c = wgetch(threadWin);
	if ( c == KEY_ESC )
	{
	    showIt = false;
	}
	
	delwin(threadWin);
	refresh();
    }
}

void CegoAdmMon::showLogThread()
{

    bool showIt = true;

    while ( showIt )
    {

	ListT<ListT<CegoFieldValue> > logthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
    
	res = _pAH->reqLogThreadInfo();
	_pAH->getLogThreadInfo(oe, logthreadinfo, format);
	
	WINDOW *threadWin;
	
	threadWin = newwin(logthreadinfo.Size() + 3, 97, 2, 2);
	keypad(threadWin, TRUE);

	box(threadWin, 0, 0);
	
	int rowno=TOPOFFSET;
	int colno=LEFTOFFSET;
	
	color_set(COL_TITLE, NULL);
	wattron(threadWin, A_BOLD);
	mvwprintw(threadWin, rowno, colno, "%s", "Thread Id");
	colno = colno + 25;
	mvwprintw(threadWin, rowno, colno, "%s", "NumRequest");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Load");
	colno = colno + 15;
	mvwprintw(threadWin, rowno, colno, "%s", "Status");
	wattroff(threadWin, A_BOLD);

	rowno++;
	
	color_set(COL_CONTENT, NULL);
	ListT<CegoFieldValue> *pFVL = logthreadinfo.First();
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 25;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(threadWin, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }
	    
	    rowno++;
	    pFVL = logthreadinfo.Next();
	}
	
	wrefresh(threadWin);
	
	wtimeout(threadWin, _refInterval);
	int c = wgetch(threadWin);
	if ( c == KEY_ESC )
	{
	    showIt = false;
	}
	
	delwin(threadWin);
	refresh();
    }
}

bool CegoAdmMon::showTableSet()
{
    
    int c = 0;
    
    bool modFlag = false;
        
    int tsSelected = 1;

    bool showTS = true;
    while ( showTS )
    {

	CegoAdminHandler::ResultType res;
	CegoTableObject oe;	

	WINDOW* tswin = 0;
	ListT<ListT<CegoFieldValue> > tsList;
    
	res = _pAH->medGetTableSetList(true);
	_pAH->getTableSetList(oe, tsList, true);
        
	Chain tableSet;
	if ( tsList.Size() > 0 )	
	    tableSet = tsList[tsSelected-1][0].valAsChain();
	    	   	    
	switch(c)
	{	
	case KEY_UP:
	    if (tsSelected == 1)
		tsSelected = tsList.Size();
	    else
		--tsSelected;
	    break;
	case KEY_DOWN:
	    if( tsSelected == tsList.Size() )
		tsSelected = 1;
	    else 
		++tsSelected;
	    break;
	case 'i':
	{
	    if ( tsList.Size() > 0 )
	    {
		modFlag = true;
		showTableSetInfo(tableSet);
	    }
	    break;
	}
	case 'n':
	{
	    modFlag = true;
	    showTableSetDefineMask();
	    break;
	}
	case 'a':
	{
	    if ( tsList.Size() > 0 )
	    {
		modFlag = true;
		showTableSetAddFile(tableSet);
	    }
	    break;
	}
	case 'd':
	{
	    if ( tsList.Size() > 0 )
	    {
		modFlag = true;
		showTableSetFileInfo(tableSet);
	    }
	    break;
	}
	case KEY_RETURN:
	{
	    if ( tsList.Size() > 0 )
	    {
		Chain tsStatus = tsList[tsSelected-1][1].valAsChain();
		int tsAction = showTableSetActionMenu(tsStatus);
		if ( tsAction > 0 )
		{
		    if ( tsAction == TS_ADD_FILE )
		    {
			showTableSetAddFile(tableSet);
		    }
		    else if ( tsAction == TS_LIST_FILE )
		    {
			showTableSetFileInfo(tableSet);
		    }
		    else
		    {			
			performTableSetAction(tableSet, tsAction);
			if ( tsAction == TS_DROP_ACTION )
			    tsSelected = 1;
		    }
		}
		modFlag = true;
	    }
	    break;
	}
	case KEY_ESC:
	{
	    modFlag=true;
	    showTS = false;
	    break;
	}
	}

	clear();
	refreshHeader();
	
	tswin = newwin(tsList.Size() + 3, 60, 2, 1);
	noecho();		
	keypad(tswin, TRUE);
	
	box(tswin, 0, 0);

	int rowno=1;
	int colno=1;
	
	wcolor_set(tswin, COL_TITLE, NULL);
	
	wattron(tswin, A_BOLD);
	mvwprintw(tswin, rowno, colno, "%s", "Name");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "RunState");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "SyncState");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "SysUsage");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "TempUsage");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "AppUsage");
	rowno++;
	wattroff(tswin, A_BOLD);
	
	int pos=0;
	ListT<CegoFieldValue> *pFVL = tsList.First();
	
	wcolor_set(tswin, COL_CONTENT, NULL);
	int tsRow=1;
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    if ( tsSelected == tsRow )
		wattron(tswin, A_REVERSE); 
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
	    }

	    if ( tsSelected == tsRow )
		wattroff(tswin, A_REVERSE); 		
	    	    
	    tsRow++;
	    rowno++;
	    pFVL = tsList.Next();
	}
	
	wrefresh(tswin);

	if ( modFlag == false )
	{
	    wtimeout(tswin, _refInterval);
	    c = wgetch(tswin);
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}

	delwin(tswin);
	refresh();

    }
    
    return modFlag;
}

void CegoAdmMon::showTableSetInfo(const Chain& tableSet)
{
    
    bool showInfo=true;
    while( showInfo )
    {	

	ListT<ListT<CegoFieldValue> > tsInfo;
	
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;	
	
	res = _pAH->medGetDetailedTableSetInfo(tableSet);	
	
	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}
	
	_pAH->getDetailedTableSetInfo(oe, tsInfo);
	
	
	int attrwidth = 0;
	int valwidth = 0;
	ListT<CegoFieldValue> *pFVL = tsInfo.First();	
	while ( pFVL )
	{
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		if ( pFV->valAsChain().length() > attrwidth )
		    attrwidth = pFV->valAsChain().length();
		pFV = pFVL->Next();
	    }
	    if ( pFV )
	    {
		if ( pFV->valAsChain().length() > valwidth )
		    valwidth = pFV->valAsChain().length();
		pFV = pFVL->Next();			
	    }
	    pFVL = tsInfo.Next();
	}
	
	WINDOW *tswin = newwin(tsInfo.Size() + 3, attrwidth + valwidth + 1, 5, 10);
	
	noecho();
	
	wtimeout(tswin, 0);
	
	keypad(tswin, TRUE);
	
   	
	int rowno=1;
	int colno=1;
	
	wcolor_set(tswin, COL_TITLE, NULL);
	wattron(tswin, A_BOLD);
	mvwprintw(tswin, rowno, colno, "%s", "Name");
	colno = colno + attrwidth;
	mvwprintw(tswin, rowno, colno, "%s", "Value");
	colno = colno + 10;
	wattroff(tswin, A_BOLD);

	rowno++;
	
	int pos=0;
	pFVL = tsInfo.First();

	box(tswin, 0, 0);
	
	// wcolor_set(tswin, COL_CONTENT, NULL);
	int tsRow=1;
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + attrwidth;
	    }
	    if ( pFV )
	    {
		mvwprintw(tswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    
	    tsRow++;
	    rowno++;
	    pFVL = tsInfo.Next();
	}
 
	wrefresh(tswin);	
	int c = wgetch(tswin);
	if ( c == KEY_ESC )
	{
	    showInfo = false;
	    clear();
	}

	delwin(tswin);
    }
    return;
}

void CegoAdmMon::showTableSetFileInfo(const Chain& tableSet)
{
    
    bool showInfo=true;
    while( showInfo )
    {	
	
	CegoAdminHandler::ResultType res;
	
	res = _pAH->medGetDetailedTableSetInfo(tableSet);	
	
	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}

	ListT<ListT<CegoFieldValue> > fileInfo;
	CegoTableObject oe;
	Chain msg;
	Chain format;
	_pAH->getMsg(msg);
	_pAH->getDataFileInfo(oe, fileInfo, format);
	
		
	WINDOW *tswin = newwin(fileInfo.Size() + 3, 80, 5, 10);
	
	noecho();
	
	wtimeout(tswin, 0);
	
	keypad(tswin, TRUE);
	
   	
	int rowno=1;
	int colno=1;
	
	// wcolor_set(tswin, COL_TITLE, NULL);
	
	wattron(tswin, A_BOLD);
	mvwprintw(tswin, rowno, colno, "%s", "Type");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "Size");
	colno = colno + 10;
	mvwprintw(tswin, rowno, colno, "%s", "Name");
	colno = colno + 50;
	wattroff(tswin, A_BOLD);
	
	rowno++;

	int pos=0;
	ListT<CegoFieldValue>* pFVL = fileInfo.First();

	box(tswin, 0, 0);
	
	// wcolor_set(tswin, COL_CONTENT, NULL);
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    Chain fileName;
	    Chain fileType;
	    Chain fileSize;

	    CegoFieldValue *pFV = pFVL->First();
	    if (pFV )
		fileName=(char*)pFV->valAsChain();
	    pFV = pFVL->Next();
	    if (pFV )
		fileType=(char*)pFV->valAsChain();
	    pFV = pFVL->Next();
	    if (pFV )
		fileSize=(char*)pFV->valAsChain();
	    
	    mvwprintw(tswin, rowno, colno, "%s", (char*)fileType);	    
	    colno = colno + 10;
	    mvwprintw(tswin, rowno, colno, "%s", (char*)fileSize);
	    colno = colno + 10;
	    mvwprintw(tswin, rowno, colno, "%s", (char*)fileName);
	    colno = colno + 50;

	   
	    rowno++;
	    pFVL = fileInfo.Next();
	}
 
	wrefresh(tswin);	
	int c = wgetch(tswin);
	if ( c == KEY_ESC )
	{
	    showInfo = false;
	    clear();
	}
	
	delwin(tswin);
    }
    return;
}

void CegoAdmMon::showTableSetDefineMask()
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("Name");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("TSX");

    attrList.Insert("Root Path");
    typeList.Insert(INPUT_TYPE_STRING);
    valList.Insert("<root path>");

    attrList.Insert("Primary");
    typeList.Insert(INPUT_TYPE_STRING);
    valList.Insert("localhost");

    attrList.Insert("Secodary");
    typeList.Insert(INPUT_TYPE_STRING);
    valList.Insert("localhost");
    
    attrList.Insert("System Space");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("100");

    attrList.Insert("Temp Space");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("100");

    attrList.Insert("App Space");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("1000");

    attrList.Insert("Log Size");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("10000000");

    attrList.Insert("Log Num");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("3");

    attrList.Insert("Sort Area Size");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("10000000");

    
    int r = showInputMask("Define tableset", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain tableSet = valList[0];
	Chain tsRoot = valList[1];
	Chain primary = valList[2];
	Chain secondary = valList[3];
	int sysSize = valList[4].asInteger();
	int tmpSize = valList[5].asInteger();
	int appSize = valList[6].asInteger();
	int logSize = valList[7].asInteger();
	int logNum = valList[8].asInteger();
	unsigned long long sortAreaSize = valList[9].asUnsignedLongLong();

	CegoAdminHandler::ResultType res;
	res = _pAH->medDefineTableSet(tableSet,
				      tsRoot,
				      primary,
				      secondary,
				      sysSize,
				      tmpSize,
				      appSize,
				      logSize,
				      logNum, 
				      sortAreaSize);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }

    clear();
}

void CegoAdmMon::showTableSetAddFile(const Chain&  tableSet)
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("Type");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("app");

    attrList.Insert("Path");
    typeList.Insert(INPUT_TYPE_STRING);
    valList.Insert("<filepath>");

    attrList.Insert("NumPages");
    typeList.Insert(INPUT_TYPE_NUMBER);
    valList.Insert("1000");
    
    int r = showInputMask("Add datafile", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain fileType = valList[0];
	Chain dataFile = valList[1];
	Chain numPages = valList[2];
	
	bool typeCheck = true;
	if ( fileType == Chain("app" ) )
	{
	    fileType = Chain(XML_APPFILE_VALUE);
	}
	else if ( fileType == Chain("temp" ) )
	{
	    fileType = Chain(XML_TEMPFILE_VALUE);
	}
	else if ( fileType == Chain("sys" ) )
	{
	    fileType = Chain(XML_SYSFILE_VALUE);
	}
	else
	{
	    typeCheck=false;
	    showInfoMask(ERROR_TITLE, "Invalid file type, must be either app, temp or sys");
	}
	

	if ( typeCheck )
	{
	    CegoAdminHandler::ResultType res;
	
	    res = _pAH->medAddDataFile(tableSet, fileType, dataFile, numPages.asInteger());
	    
	    Chain msg;
	    _pAH->getMsg(msg);
	    
	    if ( res == CegoAdminHandler::ADM_ERROR )
	    {
		showInfoMask(ERROR_TITLE, msg);
	    }
	    else
	    {
		showInfoMask(INFO_TITLE, msg);
	    }
	}
    }

    clear();
}


void CegoAdmMon::performTableSetAction(const Chain& tableSet, int action)
{
    if ( action == TS_START_ACTION )
    {
	bool doCleanup=false;
	bool doForceload=false;

	CegoAdminHandler::ResultType res;
	res = _pAH->medStartTableSet(tableSet, doCleanup, doForceload);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
	
    }
    else if ( action == TS_STOP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medStopTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}

    }
    else if ( action == TS_CREATE_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medCreateTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}

    }
    else if ( action == TS_DROP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medDropTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}

    }
    else if ( action == TS_REMOVE_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medRemoveTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
}


int CegoAdmMon::showRefreshMenu()
{

    Chain title("Select refresh interval");

    ListT<Chain> refreshMenu;
    ListT<int> refreshOpt;

    refreshMenu.Insert(Chain("1 sec delay"));
    refreshOpt.Insert(1000);

    refreshMenu.Insert(Chain("3 sec delay"));
    refreshOpt.Insert(3000);

    refreshMenu.Insert(Chain("10 sec delay"));
    refreshOpt.Insert(10000);

    refreshMenu.Insert(Chain("30 sec delay"));
    refreshOpt.Insert(30000);

    int width = title.length() + 5;
    int height = refreshMenu.Size() + 4;
 
    int highlight=1;
    int choice = 0;
    int c;
    
    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;
    
    WINDOW* menuwin = newwin(height, width, starty, startx);

    noecho();

    keypad(menuwin, TRUE);

    refresh();
    printMenu(menuwin, title, refreshMenu, highlight);

    bool showIt=true;

    while( showIt )
    {	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	{
	    if (highlight == 1)
		highlight = refreshMenu.Size();
	    else
		--highlight;
	    break;
	}
	case KEY_DOWN:
	{
	    if(highlight == refreshMenu.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	}
	case KEY_RETURN:
	{
	    showIt = false;
	    choice = highlight;
	    break;
	}
	case KEY_ESC:
	{
	    showIt = false;
	    break;
	}

	}

	// printf("You chose choice %d", choice);
	printMenu(menuwin, title, refreshMenu, highlight);	    
    }
    delwin(menuwin);

    if ( choice > 0 )
	return refreshOpt[highlight -1];
    return 0;
	
}


int CegoAdmMon::showTableSetActionMenu(const Chain& tsStatus)
{

    Chain title("Select tableset action");

    ListT<Chain> tsMenu;
    ListT<int> tsOpt;

    if ( tsStatus == Chain(TS_DEFINED_STATUS) )
    {
	tsMenu.Insert(Chain("Create tableset"));
	tsOpt.Insert(TS_CREATE_ACTION);
	tsMenu.Insert(Chain("Remove tableset"));
	tsOpt.Insert(TS_REMOVE_ACTION);
    }
    else if ( tsStatus == Chain(TS_OFFLINE_STATUS) )
    {
	tsMenu.Insert(Chain("Start tableset"));
	tsOpt.Insert(TS_START_ACTION);
	tsMenu.Insert(Chain("Drop tableset"));
	tsOpt.Insert(TS_DROP_ACTION);
    }
    else if ( tsStatus == Chain(TS_ONLINE_STATUS) )
    {
	tsMenu.Insert(Chain("Stop tableset"));
	tsOpt.Insert(TS_STOP_ACTION);
    }

    tsMenu.Insert(Chain("List datafile"));
    tsOpt.Insert(TS_LIST_FILE);

    tsMenu.Insert(Chain("Add datafile"));
    tsOpt.Insert(TS_ADD_FILE);
    
    int width = title.length() + 5;
    int height = tsMenu.Size() + 4;
 
    int highlight=1;
    int c;
    
    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;
    
    WINDOW* menuwin = newwin(height, width, starty, startx);

    noecho();

    keypad(menuwin, TRUE);

    refresh();
    printMenu(menuwin, title, tsMenu, highlight);

    bool isSelected=false;
    while( isSelected == false )
    {	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	    if (highlight == 1)
		highlight = tsMenu.Size();
	    else
		--highlight;
	    break;
	case KEY_DOWN:
	    if(highlight == tsMenu.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	case KEY_RETURN:
	    isSelected = true;
	    break;
	case KEY_ESC:
	    isSelected = true;
	    highlight = -1;
	    break;
	}

	// printf("You chose choice %d", choice);
	printMenu(menuwin, title, tsMenu, highlight);	    
    }
    delwin(menuwin);

    if ( highlight >= 0 )
	return tsOpt[highlight -1];
    return -1;
}
    
void CegoAdmMon::printMenu(WINDOW* win, const Chain& title, const ListT<Chain>& menuList, int highlight)
{
    int x, y, i;	
    wattron(win, A_BOLD);
    mvwprintw(win, 1, 1, "%s", (char*)title);
    wattroff(win, A_BOLD);
    x = 2;
    y = 3;
    box(win, 0, 0);
    for(i = 0; i < menuList.Size(); ++i)
    {	
	if (highlight == i + 1) /* High light the present choice */
	{	
	    wattron(win, A_REVERSE); 
	    mvwprintw(win, y, x, "%s", (char*)menuList[i]);
	    wattroff(win, A_REVERSE);
	}
	else
	    mvwprintw(win, y, x, "%s", (char*)menuList[i]);
	++y;
    }

    wrefresh(win);
    
}


bool CegoAdmMon::showUser()
{
    
    int c = 0;
    
    bool modFlag = false;
    
    int usSelected = 1;
   
    bool showIt = true;
    while ( showIt )
    {

	CegoAdminHandler::ResultType res;	
	res = _pAH->reqShowUser();

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > userList;
	
	_pAH->getUserInfo(oe, userList);
            	
	switch(c)
	{	
	case KEY_UP:
	    if (usSelected == 1)
		usSelected = userList.Size();
	    else
		--usSelected;
	    break;
	case KEY_DOWN:
	    if( usSelected == userList.Size() )
		usSelected = 1;
	    else 
		++usSelected;
	    break;
	case 'n':
	    modFlag = true;
	    showUserAddMask();
	    break;
	case KEY_RETURN:
	{
	    Chain usName = userList[usSelected-1][0].valAsChain();
	    Chain roleList = userList[usSelected-1][1].valAsChain();
	    int usAction = showUserActionMenu();
	    if ( usAction > 0 )
		performUserAction(usName, usAction);
	    if ( usAction == USER_DELETE_ACTION )
		usSelected=1;
	    modFlag = true;
	    break;
	}
	case KEY_ESC:
	{
	    modFlag=true;
	    showIt = false;
	    break;
	}
	}
	
	clear();
	refreshHeader();

	WINDOW* uswin = newwin(userList.Size() + 3, 80, 2, 1);
	noecho();		
	keypad(uswin, TRUE);
	box(uswin, 0, 0);

	int rowno=1;
	int colno=1;
	
	wcolor_set(uswin, COL_TITLE, NULL);
	
	wattron(uswin, A_BOLD);
	mvwprintw(uswin, rowno, colno, "%s", "Name");
	colno = colno + 10;
	mvwprintw(uswin, rowno, colno, "%s", "Role");
	colno = colno + 30;
	mvwprintw(uswin, rowno, colno, "%s", "Trace");
	colno = colno + 10;
	mvwprintw(uswin, rowno, colno, "%s", "NumRequest");
	colno = colno + 15;
	mvwprintw(uswin, rowno, colno, "%s", "NumQuery");
	rowno++;
	wattroff(uswin, A_BOLD);
	
	int pos=0;
	ListT<CegoFieldValue> *pFVL = userList.First();
	
	wcolor_set(uswin, COL_CONTENT, NULL);
	int usRow=1;
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    if ( usSelected == usRow )
		wattron(uswin, A_REVERSE); 
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(uswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(uswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 30;
	    }
	    if ( pFV )
	    {
		mvwprintw(uswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    if ( pFV )
	    {
		mvwprintw(uswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 15;
	    }
	    if ( pFV )
	    {
		mvwprintw(uswin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }
	    
	    if ( usSelected == usRow )
		wattroff(uswin, A_REVERSE); 		
	    	    
	    usRow++;
	    rowno++;
	    pFVL = userList.Next();
	}
	
	wrefresh(uswin);

	if ( modFlag == false )
	{
	    wtimeout(uswin, _refInterval);
	    c = wgetch(uswin);
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}

	delwin(uswin);
	refresh();

    }
    
   
    return modFlag;

}


void CegoAdmMon::showUserAddMask()
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("Name");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("lemke");

    attrList.Insert("Password");
    typeList.Insert(INPUT_TYPE_PWD);
    valList.Insert("");
    
    int r = showInputMask("Add User", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain user = valList[0];
	Chain password = valList[1];

	CegoAdminHandler::ResultType res;
    
	res = _pAH->reqAddUser(user, password);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
    
    
    clear();
}

int CegoAdmMon::showUserActionMenu()
{

    Chain title("Select user action");

    ListT<Chain> usMenu;
    ListT<int> usOpt;

    usMenu.Insert(Chain("Trace On"));
    usOpt.Insert(USER_TRACEON_ACTION);
    usMenu.Insert(Chain("Trace Off"));
    usOpt.Insert(USER_TRACEOFF_ACTION);
    usMenu.Insert(Chain("Delete user"));
    usOpt.Insert(USER_DELETE_ACTION);

    int width = title.length() + 5;
    int height = usMenu.Size() + 4;
 
    int highlight=1;
    int c;
    
    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;
    
    WINDOW* menuwin = newwin(height, width, starty, startx);

    noecho();

    keypad(menuwin, TRUE);

    refresh();
    printMenu(menuwin, title, usMenu, highlight);

    bool isSelected=false;
    while( isSelected == false )
    {	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	    if (highlight == 1)
		highlight = usMenu.Size();
	    else
		--highlight;
	    break;
	case KEY_DOWN:
	    if(highlight == usMenu.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	case KEY_RETURN:
	    isSelected = true;
	    break;
	case KEY_ESC:
	    isSelected = true;
	    highlight = -1;
	    break;
	}

	// printf("You chose choice %d", choice);
	printMenu(menuwin, title, usMenu, highlight);	    
    }
    delwin(menuwin);

    if ( highlight >= 0 )
	return usOpt[highlight -1];
    return -1;
}

void CegoAdmMon::performUserAction(const Chain& user, int action)
{
    if ( action == USER_TRACEON_ACTION )
    {
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqUserTrace(user, true);

	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
	
    }
    else if ( action == USER_TRACEOFF_ACTION )
    {
	CegoAdminHandler::ResultType res;

	
	res = _pAH->reqUserTrace(user, false);
	
	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
	
    }
    else if ( action == USER_DELETE_ACTION )
    {
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqRemoveUser(user);
	
	Chain msg;
	_pAH->getMsg(msg);
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
}

bool CegoAdmMon::showRole()
{
    
    int c = 0;
    
    bool modFlag = false;
    
    int rlSelected = 1;
   
    bool showIt = true;
    while ( showIt )
    {
	
	CegoAdminHandler::ResultType res;	
	res = _pAH->reqListRole();

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > roleList;
	
	_pAH->getRoleList(oe, roleList);
        	
	switch(c)
	{	
	case KEY_UP:
	    if (rlSelected == 1)
		rlSelected = roleList.Size();
	    else
		--rlSelected;
	    break;
	case KEY_DOWN:
	    if( rlSelected == roleList.Size() )
		rlSelected = 1;
	    else 
		++rlSelected;
	    break;
	case 'n':
	{
	    modFlag = true;
	    showRoleAddMask();
	    break;
	}
	case 'i':
	{
	    modFlag = true;
	    Chain rlName = roleList[rlSelected-1][0].valAsChain();
	    showRolePermListMask(rlName);
	    break;
	}
	case 's':
	{
	    modFlag = true;
	    Chain rlName = roleList[rlSelected-1][0].valAsChain();
	    showRolePermSetMask(rlName);
	    break;
	}
	case KEY_RETURN:
	{
	    Chain rlName = roleList[rlSelected-1][0].valAsChain();
	    
	    int rlAction = showRoleActionMenu();
	    if ( rlAction > 0 )
	    {
		if ( rlAction == ROLE_PERM_LIST_ACTION )
		{
		    showRolePermListMask(rlName);
		}
		else if ( rlAction == ROLE_PERM_SET_ACTION )
		{
		    showRolePermSetMask(rlName);
		}
		else if ( rlAction == ROLE_ASSIGN_ACTION )
		{
		    showRoleAssignMask(rlName);
		}
		else if ( rlAction == ROLE_REMOVE_ACTION )
		{
		    showRoleRemoveMask(rlName);
		}
		else
		{
		    performRoleAction(rlName, rlAction);
		    if ( rlAction == ROLE_DROP_ACTION )
			rlSelected = 1;
		}
	    }
	    modFlag = true;
	    break;
	}
	case KEY_ESC:
	{
	    modFlag=true;
	    showIt = false;
	}
	}

	clear();
	refreshHeader();
	
	int rowno=1;
	int colno=1;

	WINDOW* rlwin = newwin(roleList.Size() + 3, 30, 2, 1);

	noecho();
		
	keypad(rlwin, TRUE);

	box(rlwin, 0, 0);
	
	wcolor_set(rlwin, COL_TITLE, NULL);
	
	wattron(rlwin, A_BOLD);
	mvwprintw(rlwin, rowno, colno, "%s", "Name");
	colno = colno + 10;
	rowno++;
	wattroff(rlwin, A_BOLD);
	
	int pos=0;
	ListT<CegoFieldValue> *pFVL = roleList.First();
	
	wcolor_set(rlwin, COL_CONTENT, NULL);
	int rlRow=1;
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    if ( rlSelected == rlRow )
		wattron(rlwin, A_REVERSE); 
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		mvwprintw(rlwin, rowno, colno, "%s", (char*)pFV->valAsChain());
		pFV = pFVL->Next();			
		colno = colno + 10;
	    }

	    
	    if ( rlSelected == rlRow )
		wattroff(rlwin, A_REVERSE); 		
	    	    
	    rlRow++;
	    rowno++;
	    pFVL = roleList.Next();
	}
	
	wrefresh(rlwin);

	if ( modFlag == false )
	{
	    wtimeout(rlwin, _refInterval);
	    c = wgetch(rlwin);
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}

	delwin(rlwin);
	refresh();

    }
    
   
    return modFlag;

}

void CegoAdmMon::showRoleAddMask()
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("Name");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("myrole");

    
    int r = showInputMask("Add Role", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain role = valList[0];

	CegoAdminHandler::ResultType res;
    
	res = _pAH->reqCreateRole(role);
  	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
    
    
    clear();
}

void CegoAdmMon::showRoleAssignMask(const Chain& role)
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("User");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("lemke");

    int r = showInputMask("Assign Role", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain user = valList[0];

	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqAssignRole(user, role);

	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
    
    
    clear();
}

void CegoAdmMon::showRoleRemoveMask(const Chain& role)
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("User");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("lemke");

    int r = showInputMask("Remove Role", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain user = valList[0];

	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqRemoveRole(user, role);

	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
    
    
    clear();
}


int CegoAdmMon::showRoleActionMenu()
{

    Chain title("Select role action");

    ListT<Chain> rlMenu;
    ListT<int> rlOpt;

    rlMenu.Insert(Chain("List permission"));
    rlOpt.Insert(ROLE_PERM_LIST_ACTION);

    rlMenu.Insert(Chain("Set permission"));
    rlOpt.Insert(ROLE_PERM_SET_ACTION);

    rlMenu.Insert(Chain("Assign to user"));
    rlOpt.Insert(ROLE_ASSIGN_ACTION);

    rlMenu.Insert(Chain("Remove from user"));
    rlOpt.Insert(ROLE_REMOVE_ACTION);

    rlMenu.Insert(Chain("Drop role"));
    rlOpt.Insert(ROLE_DROP_ACTION);

    int width = title.length() + 5;
    int height = rlMenu.Size() + 4;
 
    int highlight=1;
    int c;
    
    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;
    
    WINDOW* menuwin = newwin(height, width, starty, startx);

    noecho();

    keypad(menuwin, TRUE);

    refresh();
    printMenu(menuwin, title, rlMenu, highlight);

    bool isSelected=false;
    while( isSelected == false )
    {	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	    if (highlight == 1)
		highlight = rlMenu.Size();
	    else
		--highlight;
	    break;
	case KEY_DOWN:
	    if(highlight == rlMenu.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	case KEY_RETURN:
	    isSelected = true;
	    break;
	case KEY_ESC:
	    isSelected = true;
	    highlight = -1;
	    break;
	}

	// printf("You chose choice %d", choice);
	printMenu(menuwin, title, rlMenu, highlight);	    
    }
    delwin(menuwin);

    if ( highlight >= 0 )
	return rlOpt[highlight -1];
    return -1;
}

void CegoAdmMon::performRoleAction(const Chain& role, int action)
{

    if ( action == ROLE_DROP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqDropRole(role);
	
	Chain msg;
	_pAH->getMsg(msg);
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }
}

void CegoAdmMon::showRolePermListMask(const Chain& role)
{
    
    bool showInfo=true;
    while( showInfo )
    {	
	
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqShowRole(role);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > roleInfo;
	
	_pAH->getRoleInfo(oe, roleInfo);

	Chain msg;
	_pAH->getMsg(msg);

			
	WINDOW *piwin = newwin(roleInfo.Size() + 3, 80, 5, 10);
	
	noecho();
	
	wtimeout(piwin, 0);
	
	keypad(piwin, TRUE);
	
   	
	int rowno=1;
	int colno=1;
	
	// wcolor_set(tswin, COL_TITLE, NULL);
	
	wattron(piwin, A_BOLD);
	mvwprintw(piwin, rowno, colno, "%s", "PermId");
	colno = colno + 10;
	mvwprintw(piwin, rowno, colno, "%s", "Tableset");
	colno = colno + 10;
	mvwprintw(piwin, rowno, colno, "%s", "Filter");
	colno = colno + 20;
	mvwprintw(piwin, rowno, colno, "%s", "Permission");
	wattroff(piwin, A_BOLD);
	
	rowno++;

	int pos=0;
	ListT<CegoFieldValue>* pFVL = roleInfo.First();

	box(piwin, 0, 0);
	
	// wcolor_set(tswin, COL_CONTENT, NULL);
	while ( pFVL )
	{
	    
	    colno=LEFTOFFSET;
	    
	    Chain permId;
	    Chain tableSet;
	    Chain filter;
	    Chain permDef;


	    CegoFieldValue *pFV = pFVL->First();
	    if (pFV )
		permId=(char*)pFV->valAsChain();
	    pFV = pFVL->Next();
	    if (pFV )
		tableSet=(char*)pFV->valAsChain();
	    pFV = pFVL->Next();
	    if (pFV )
		filter=(char*)pFV->valAsChain();
	    pFV = pFVL->Next();
	    if (pFV )
		permDef=(char*)pFV->valAsChain();
	    
	    mvwprintw(piwin, rowno, colno, "%s", (char*)permId);
	    colno = colno + 10;
	    mvwprintw(piwin, rowno, colno, "%s", (char*)tableSet);
	    colno = colno + 10;
	    mvwprintw(piwin, rowno, colno, "%s", (char*)filter);
	    colno = colno + 20;
	    mvwprintw(piwin, rowno, colno, "%s", (char*)permDef);

	   
	    rowno++;
	    pFVL = roleInfo.Next();
	}
 
	wrefresh(piwin);	
	int c = wgetch(piwin);
	if ( c == KEY_ESC )
	{
	    showInfo = false;
	    clear();
	}
	
	delwin(piwin);
    }
    return;
}


void CegoAdmMon::showRolePermSetMask(const Chain& role)
{
    ListT<Chain> attrList;
    ListT<Chain> typeList;
    ListT<Chain> valList;

    attrList.Insert("PermId");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("P1");

    attrList.Insert("TableSet");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("TS1");

    attrList.Insert("Filter");
    typeList.Insert(INPUT_TYPE_STRING);
    valList.Insert("*");

    attrList.Insert("RIGHT");
    typeList.Insert(INPUT_TYPE_ID);
    valList.Insert("ALL");

    
    int r = showInputMask("Set permission", attrList, typeList, valList);
    
    if ( r )
    {
	
	Chain permId = valList[0];
	Chain tableSet = valList[1];
	Chain filterPerm = valList[2];
	Chain rightPerm = valList[3];
	
	CegoAdminHandler::ResultType res;
	    
	res = _pAH->reqSetPermission(role, permId, tableSet, filterPerm, rightPerm);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoMask(ERROR_TITLE, msg);
	}
	else
	{
	    showInfoMask(INFO_TITLE, msg);
	}
    }

    clear();

}


//////////////////////////
// mask utility methods //
//////////////////////////

int CegoAdmMon::showInputMask(const Chain& title, const ListT<Chain>& attrList, const ListT<Chain>& typeList, ListT<Chain>& valList)
{

    int returnVal = 0;
    int width = 50;
    int height = attrList.Size() + CUR_ROW_OFFSET + 3;
 
    int c;
    char inputArray[VAL_MAX_NUM][VAL_MAX_LEN];

    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;

    curs_set(1);
        
    // printf("\e]12;%s\a", "white");

    int curRow = CUR_ROW_OFFSET;
    int curCol = CUR_COL_OFFSET;

    WINDOW *maskwin = newwin(height, width, starty, startx);

    noecho();

    keypad(maskwin, TRUE);

    wattron(maskwin, A_BOLD);
    mvwprintw(maskwin, 1, 1, "%s", (char*)title);
    wattroff(maskwin, A_BOLD);
    int x = 2;
    int y = CUR_ROW_OFFSET;
    box(maskwin, 0, 0);
    for(int i = 0; i < attrList.Size(); ++i)
    {	
	mvwprintw(maskwin, y, x, "%s", (char*)attrList[i]);
	mvwprintw(maskwin, y, CUR_COL_OFFSET - 2, ":");
	mvwprintw(maskwin, y, CUR_COL_OFFSET, (char*)valList[i]);
	for(int j = 0; j < valList[i].length(); j++)
	    inputArray[i][j]=valList[i][j];
	for(int j = valList[i].length(); j < VAL_MAX_LEN; j++)
	    inputArray[i][j]=0;
	++y;
    }
    y++;
    wattron(maskwin, A_BOLD);
    mvwprintw(maskwin, y, x, "Ok [ ] Abort [ ]");
    wattroff(maskwin, A_BOLD);

    wmove(maskwin, curRow, curCol);
    
    bool showMask=true;
    while( showMask )
    {	
	c = wgetch(maskwin);
	switch(c)
	{	
	case KEY_UP:
	    int prevCurRow, prevCurCol;
	    prevCursorPos(attrList.Size(), curRow, curCol, prevCurRow, prevCurCol);
	    curRow = prevCurRow;
	    curCol = prevCurCol;
	    wmove(maskwin, curRow, curCol);
	    break;
	case KEY_DOWN:
	case KEY_TAB:
	    int nextCurRow, nextCurCol;
	    nextCursorPos(attrList.Size(), curRow, curCol, nextCurRow, nextCurCol);
	    curRow = nextCurRow;
	    curCol = nextCurCol;
	    wmove(maskwin, curRow, curCol);
	    break;
	case KEY_LEFT:
	    if ( curRow < attrList.Size() + CUR_ROW_OFFSET
		 && curCol > CUR_COL_OFFSET )
		curCol--;
	    wmove(maskwin, curRow, curCol);
	    break;
	case KEY_RIGHT:
	    if ( curRow < attrList.Size() + CUR_ROW_OFFSET
		 && curCol < CUR_COL_OFFSET + VAL_MAX_LEN )
		 curCol++;
	    wmove(maskwin, curRow, curCol);
	    break;
	case KEY_RETURN:
	{
	    if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 )
	    {
		if ( curCol == OK_COL_POS )
		    returnVal = 1;
		else if ( curCol == ABORT_COL_POS )
		    returnVal = 0;
		showMask=false;
	    }
	    break;
	}
	case KEY_DEL:
	{
	    if ( curCol > CUR_COL_OFFSET )
		curCol--;
	    mvwprintw(maskwin, curRow, curCol, "%c", ' ');
	    inputArray[curRow-CUR_ROW_OFFSET][curCol-CUR_COL_OFFSET] = ' ';
	    wmove(maskwin, curRow, curCol);
	    break;
	}
	case KEY_ESC:
	{
	    returnVal = 0;
	    showMask=false;
	    break;
	}
	default:
	{
	    bool inputAccepted=false;
	    
	    if ( typeList[curRow-CUR_ROW_OFFSET] == Chain(INPUT_TYPE_STRING) 
		 && ( isalnum (c) 
		      || c == '_' 
		      || c == '-' 
		      || c == '/' 
		      || c == '.' 
		      || c == '*' 
		      || c == '['
		      || c == ']' ))
		inputAccepted=true;
	    else if ( typeList[curRow-CUR_ROW_OFFSET] == Chain(INPUT_TYPE_ID) 
		      && ( isalnum (c)
			   || c == '_' ))
		inputAccepted=true;	  
	    else if ( typeList[curRow-CUR_ROW_OFFSET] == Chain(INPUT_TYPE_NUMBER) && isdigit (c) )
		inputAccepted=true;
	    else if ( typeList[curRow-CUR_ROW_OFFSET] == Chain(INPUT_TYPE_PWD) )
		inputAccepted=true;
	    
	    if ( inputAccepted )
	    {
		if ( typeList[curRow-CUR_ROW_OFFSET] == Chain(INPUT_TYPE_PWD) )
		    mvwprintw(maskwin, curRow, curCol, "%c", '*');
		else
		    mvwprintw(maskwin, curRow, curCol, "%c", c);
		inputArray[curRow-CUR_ROW_OFFSET][curCol-CUR_COL_OFFSET] = c;
		curCol++;
		wmove(maskwin, curRow, curCol);
	    }
	}
	}
	wrefresh(maskwin);

	// printf("You chose choice %d", choice);
	// printMask(maskwin, title, attrList, valList);	    
    }

    // store values in valList

    valList.Empty();
    for(int i = 0; i < attrList.Size(); ++i)
    {	
	valList.Insert(Chain(inputArray[i]).cutTrailing(" "));
    }

    delwin(maskwin);
    curs_set(0);
    // printf("\e]12;%s\a", "black");

    refresh();

    return returnVal;
}


void CegoAdmMon::showInfoMask(const Chain& title, const Chain& msg)
{

    int width = INFOMASK_WIDTH;
    int height = INFOMASK_HEIGHT;
 
    int c;

    int startx = (TERM_WIDTH - width) / 2;
    int starty = (TERM_HEIGHT - height) / 2;
    
    WINDOW *infowin = newwin(height, width, starty, startx);

    noecho();

    keypad(infowin, TRUE);

    wattron(infowin, A_BOLD);
    mvwprintw(infowin, 1, 1, "%s", (char*)title);
    wattroff(infowin, A_BOLD);
    box(infowin, 0, 0);

    mvwprintw(infowin, 3, 1, "%s", (char*)msg);

    // wrefresh(infowin);

    // any input to continue

    refresh();

    c = wgetch(infowin);
    switch(c)
    {	
    case 'Y':
	break;
    case 'N':
	break;
    }
    
    delwin(infowin);

}

    
int CegoAdmMon::showConfirmMask(const Chain& title, const ListT<Chain>& attrList, const ListT<Chain>& valList)
{
    int x, y, i;	

    int width = 50;
    int height = attrList.Size() + 3;
 
    int c;

    int startx = (80 - width) / 2;
    int starty = (24 - height) / 2;
    
    int currow = 2;
    int curcol = 15;

    WINDOW *maskwin = newwin(height, width, starty, startx);

    noecho();

    keypad(maskwin, TRUE);

    mvwprintw(maskwin, 1, 1, "%s", (char*)title);
    x = 2;
    y = 2;
    box(maskwin, 0, 0);
    for(int i = 0; i < attrList.Size(); ++i)
    {	
	mvwprintw(maskwin, y, x, "%s", (char*)attrList[i]);
	mvwprintw(maskwin, y, 12, ":");
	mvwprintw(maskwin, y, 15, "%s", (char*)valList[i]);
	++y;
    }

    c = wgetch(maskwin);
    switch(c)
    {	
    case 'Y':
	break;
    case 'N':
	break;
    }
    
    delwin(maskwin);

    return 0;
}


void CegoAdmMon::prevCursorPos(int attrListSize, int curRow, int curCol, int& prevCurRow, int& prevCurCol)
{
    int okRow = attrListSize + CUR_ROW_OFFSET + 1;
    int okCol = OK_COL_POS;

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1;
    int abortCol = ABORT_COL_POS;

    if ( curRow == CUR_ROW_OFFSET )
    {
	prevCurRow = abortRow;
	prevCurCol = abortCol;	
    }
    else if ( curRow == okRow && curCol == okCol )
    {
	prevCurRow = attrListSize + CUR_ROW_OFFSET - 1;
	prevCurCol = CUR_COL_OFFSET;
    }
    else if ( curRow == abortRow && curCol == abortCol )
    {
	prevCurRow = okRow; 
	prevCurCol = okCol;
    }
    else
    {
	prevCurRow = curRow-1;
	prevCurCol = CUR_COL_OFFSET;
    }
}


void CegoAdmMon::nextCursorPos(int attrListSize, int curRow, int curCol, int& nextCurRow, int& nextCurCol)
{
    int okRow = attrListSize + CUR_ROW_OFFSET + 1;
    int okCol = OK_COL_POS;

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1;
    int abortCol = ABORT_COL_POS;

    if ( curRow == attrListSize + CUR_ROW_OFFSET - 1)
    {
	nextCurRow = okRow;
	nextCurCol = okCol;	
    }
    else if ( curRow == okRow && curCol == okCol )
    {
	nextCurRow = abortRow;
	nextCurCol = abortCol;
    }
    else if ( curRow == abortRow && curCol == abortCol )
    {
	nextCurRow = CUR_ROW_OFFSET;
	nextCurCol = CUR_COL_OFFSET;
    }
    else
    {
	nextCurRow = curRow+1;
	nextCurCol=CUR_COL_OFFSET;
    }
}

void CegoAdmMon::showHelp()
{

    int width = HELPMASK_WIDTH;
    int height = HELPMASK_HEIGHT;
 
    int c;

    int startx = (TERM_WIDTH - width) / 2;
    int starty = (TERM_HEIGHT - height) / 2;
    
    WINDOW *helpwin = newwin(height, width, starty, startx);

    noecho();

    keypad(helpwin, TRUE);


    wattron(helpwin, A_UNDERLINE);
    mvwprintw(helpwin, 1, 1, "Available commands");
    wattroff(helpwin, A_UNDERLINE);

    box(helpwin, 0, 0);

    int row=3;
    int sepCol=18;
    int descCol=20;

    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Header Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Left");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select left header menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Right");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select right header menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Open selected menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Sub Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Open option menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"n");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"New entry");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"i");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Get information about entry");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Selection Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select item action");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Down");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select next item");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Up");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select previous item");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Input Mask");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Tab,Cursor Down");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select next field");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Up");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select previous field");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Left");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Move input cursor left");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Right");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Move input cursor right");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Confirm or Abort");

    refresh();

    c = wgetch(helpwin);
   
    delwin(helpwin);

    clear();

}


#endif
