///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBTreeCursor.cc
// ------------------
// Cego index cursor class implementation
//         
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBTreeCursor
// 
// Description: Cursor class for traversing btree objects
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

// CEGO INCLUDES
#include "CegoBTreeCursor.h"
#include "CegoTupleState.h"
#include "CegoBTreeNode.h"
#include "CegoBTreeManager.h"

CegoBTreeCursor::CegoBTreeCursor()
{
}

CegoBTreeCursor::CegoBTreeCursor(CegoTableManager *pTabMng, int tabSetId, const Chain& btreeName, CegoObject::ObjectType type, CegoAttrCond* pAttrCond, bool ignoreTouched, bool readUncommitted)
{
    _pTabMng = pTabMng;
    _pDBMng = pTabMng->getDBMng();
    _btreeName = btreeName;
    _type = type;
    _pAttrCond = pAttrCond;

    _tabSetId = tabSetId;
    _ignoreTouched = ignoreTouched;

    _readUncommitted = readUncommitted;
    _cursorCached = false;
    _eoc=true;
    _lockId = 0;
}

CegoBTreeCursor::~CegoBTreeCursor()
{
    abort();
}

bool CegoBTreeCursor::getFirst(ListT<CegoField>& fl, CegoDataPointer& dp)
{
    _eoc=false;

    if ( fl.isEmpty() )
    {
	throw Exception(EXLOC, "Empty field list");
    }

    if ( _cursorCached == false )
    {
	CegoBTreeObject btoe;

	_pTabMng->getObject(_tabSetId, _btreeName, _type, btoe);

	_btreeSchema = btoe.getSchema();

	_keyLen = CegoBTreeValue::getKeyLen(_btreeSchema);

	/*
	CegoField *pF = _btreeSchema.First();
	_keyLen = 0;
	while ( pF )
	{
	    _keyLen += 1; // null value indicator
	    _keyLen += pF->getLength();
	    
	    if ( pF->getType() == VARCHAR_TYPE
		 || pF->getType() == BIGINT_TYPE
		 || pF->getType() == DECIMAL_TYPE
		 || pF->getType() == FIXED_TYPE )
		_keyLen += 1; // termination character
	    
	    pF = _btreeSchema.Next();
	}
	*/
	
	_rootFileId = btoe.getDataFileId();
	_rootPageId = btoe.getDataPageId();
	
	Chain tabName = btoe.getTabName();
	
	_cursorCached = true;

	// CegoBTreeManager btm(_pTabMng);
	// btm.dumpBTree(btoe);
    }

    // cout << "Read Locking " << _rootFileId << " " << _rootPageId << endl;
    _lockId = _pTabMng->getLockHandler()->lockData(CegoObject::BTREE, _rootFileId, _rootPageId, CegoLockHandler::READ);
    
    bool found = false;

    if (_pAttrCond)
    { 
	if ( _pAttrCond->getPrimaryCompMode() == CegoAttrComp::BTWN )
	    _pAttrCond->setPrimaryComparison( MORE_EQUAL_THAN );

	switch ( _pAttrCond->getPrimaryComparison() )
	{
	case NOT_EQUAL:
	case LESS_THAN:
	case LESS_EQUAL_THAN:
	{
	    // go to the beginning

	    bool leafFound = false;
	    _curFileId = _rootFileId;
	    _curPageId = _rootPageId;

	    CegoBTreeValue iv;

	    while ( leafFound == false )
	    {
		CegoBufferPage bp;
		_pDBMng->bufferFix(bp, _tabSetId, _curFileId, _curPageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
				
		if ( bp.getType() == CegoBufferPage::BTREE_NODE )
		{		    
		    CegoBTreeNode traceNode;
		    traceNode.setType(CegoBTreeNode::NODE);
		    traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
		    traceNode.setSchema(&_btreeSchema, _keyLen);
		    traceNode.setFileId(bp.getFileId());
		    traceNode.setPageId(bp.getPageId());
		    
		    traceNode.reset();
		    traceNode.nextChildPointer(_curFileId, _curPageId);
		    
		    _parentPageStack.Push(bp);
		}
		else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
		{
		    _curPage = bp;
		    _curLeaf.setType(CegoBTreeNode::LEAF);
		    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
		    _curLeaf.setFileId(_curPage.getFileId());
		    _curLeaf.setPageId(_curPage.getPageId());
		    _curLeaf.setSchema(&_btreeSchema, _keyLen);

		    leafFound=true;

		    _curLeaf.reset();

		    bool goOn = true;
		    while ( goOn )
		    {
			CegoBTreeValue iv;
			bool hasValue = _curLeaf.nextValue(iv, dp);
					
			if ( hasValue )
			{	
			    CegoComparison comp = compValue(iv);
			    
			    if ( comp == LESS_THAN
				 || ( comp == EQUAL && _pAttrCond->getPrimaryComparison() == LESS_EQUAL_THAN )
				 || ( comp != EQUAL && _pAttrCond->getPrimaryComparison() == NOT_EQUAL ) )
			    {	       
				found = true;
				goOn = false;
			    }
			    else if ( _pAttrCond->getPrimaryComparison() == NOT_EQUAL )
			    {
				return getNext(fl, dp);
			    }
			}
			else
			{   
			    while ( hasValue == false && goOn )
			    {				
				int fileId = _curPage.getNextFileId();
				int pageId = _curPage.getNextPageId();
								
				_pDBMng->bufferUnfix(_curPage, false, _pTabMng->getLockHandler());
				
				if ( fileId || pageId )
				{
				    _pDBMng->bufferFix(_curPage, _tabSetId, fileId, pageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
				    
				    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
				    _curLeaf.setFileId(_curPage.getFileId());
				    _curLeaf.setPageId(_curPage.getPageId());
				    _curLeaf.setSchema(&_btreeSchema, _keyLen);
				
				    _curLeaf.reset();
				    hasValue = _curLeaf.numEntries() > 0;				    
				}
				else
				{
				    goOn = false;
				}
			    }
			}
		    }	    
		}   
	    }
	    break;
	}
	case EQUAL:
	case MORE_EQUAL_THAN:
	{	    
	    bool leafFound = false;

	    _curFileId = _rootFileId;
	    _curPageId = _rootPageId;

	    while ( leafFound == false )
	    {
		CegoBufferPage bp;

		_pDBMng->bufferFix(bp, _tabSetId, _curFileId, _curPageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());

		if ( bp.getType() == CegoBufferPage::BTREE_NODE )
		{	    
		    CegoBTreeNode traceNode;
		    traceNode.setType(CegoBTreeNode::NODE);
		    traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
		    traceNode.setSchema(&_btreeSchema, _keyLen);
		    traceNode.setFileId(bp.getFileId());
		    traceNode.setPageId(bp.getPageId());

		    traceNode.reset();
		   
		    bool goOn = traceNode.nextChildPointer(_curFileId, _curPageId);
		    while ( goOn )
		    {
			CegoBTreeValue iv;
			if ( traceNode.rightChild(iv) )
			{			    
			    CegoComparison comp = compValue(iv);
			    
			    if ( comp == EQUAL
				 || comp == MORE_THAN )
			    {	       
				goOn = false;
			    }
			    else
			    {
				goOn = traceNode.nextChildPointer(_curFileId, _curPageId);
			    }
			}
			else
			{
			    goOn = false;
			}			
		    }				
		    _parentPageStack.Push(bp); 
		}
		else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
		{		
		    _curPage = bp;
		    _curLeaf.setType(CegoBTreeNode::LEAF);
		    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
		    _curLeaf.setFileId(_curPage.getFileId());
		    _curLeaf.setPageId(_curPage.getPageId());
		    _curLeaf.setSchema(&_btreeSchema, _keyLen);

		    leafFound=true;
		    
		    _curLeaf.reset();

		    bool goOn = true;
		    while ( goOn )
		    {
			CegoBTreeValue iv;
			bool hasValue = _curLeaf.nextValue(iv, dp);
	       	
			if ( hasValue )
			{
			    CegoComparison comp = compValue(iv);			
			    if ( comp == EQUAL || comp == MORE_THAN )
			    {	       
				goOn = false;			    
				if ( comp == MORE_THAN && _pAttrCond->getPrimaryComparison() == EQUAL )
				    found = false;
				else
				    found = true;
			    }			    
			}
			else
			{			    
			    while ( hasValue == false && goOn )
			    {								
				int fileId = _curPage.getNextFileId();
				int pageId = _curPage.getNextPageId();
				
				// cout << "Skipping leave, going to next " << pageId << endl;
			
				_pDBMng->bufferUnfix(_curPage, false, _pTabMng->getLockHandler());
			
				if ( fileId || pageId )
				{
				    _pDBMng->bufferFix(_curPage, _tabSetId, fileId, pageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
				    
				    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
				    _curLeaf.setFileId(_curPage.getFileId());
				    _curLeaf.setPageId(_curPage.getPageId());
				    _curLeaf.setSchema(&_btreeSchema, _keyLen);
				    
				    _curLeaf.reset();
				    hasValue = _curLeaf.numEntries() > 0;
				}
				else
				{
				    goOn = false;			      
				}
			    }
			}
		    }
		}
	    }	    	    
	    break;	    
	}
	case MORE_THAN:
	{
	    bool leafFound = false;

	    _curFileId = _rootFileId;
	    _curPageId = _rootPageId;

	    while ( leafFound == false )
	    {
		CegoBufferPage bp;
		_pDBMng->bufferFix(bp, _tabSetId, _curFileId, _curPageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
				
		if ( bp.getType() == CegoBufferPage::BTREE_NODE )
		{	    
		    CegoBTreeNode traceNode;
		    traceNode.setType(CegoBTreeNode::NODE);
		    traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
		    traceNode.setSchema(&_btreeSchema, _keyLen);
		    traceNode.setFileId(bp.getFileId());
		    traceNode.setPageId(bp.getPageId());

		    traceNode.reset();
		   
		    bool goOn = traceNode.nextChildPointer(_curFileId, _curPageId);
		    while ( goOn )
		    {
			CegoBTreeValue iv;
			if ( traceNode.rightChild(iv) )
			{
			    CegoComparison comp = compValue(iv);
			    
			    if ( comp == MORE_THAN )
			    {	       
				goOn = false;
			    }
			    else
			    {
				goOn = traceNode.nextChildPointer(_curFileId, _curPageId);
			    }
			}
			else
			{
			    goOn = false;
			}			
		    }				
		    _parentPageStack.Push(bp); 
		}
		else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
		{		
		    _curPage = bp;
		    _curLeaf.setType(CegoBTreeNode::LEAF);
		    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
		    _curLeaf.setFileId(_curPage.getFileId());
		    _curLeaf.setPageId(_curPage.getPageId());
		    _curLeaf.setSchema(&_btreeSchema, _keyLen);

		    leafFound=true;
		    
		    _curLeaf.reset();

		    bool goOn = true;
		    while ( goOn )
		    {
			CegoBTreeValue iv;
			bool hasValue = _curLeaf.nextValue(iv, dp);
			if  ( hasValue )
			{
			    CegoComparison comp = compValue(iv);			
			    if ( comp == MORE_THAN )
			    {	       
				goOn = false;			    
				if ( comp == MORE_THAN &&  _pAttrCond->getPrimaryComparison() == EQUAL )
				    found = false;
				else
				    found = true;			    
			    }			    
			}
			else
			{			    
			    while ( hasValue == false && goOn )
			    {								
				int fileId = _curPage.getNextFileId();
				int pageId = _curPage.getNextPageId();
												
				_pDBMng->bufferUnfix(_curPage, false, _pTabMng->getLockHandler());
				
				if ( fileId || pageId )
				{
				    _pDBMng->bufferFix(_curPage, _tabSetId, fileId, pageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
				    
				    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
				    _curLeaf.setFileId(_curPage.getFileId());
				    _curLeaf.setPageId(_curPage.getPageId());
				    _curLeaf.setSchema(&_btreeSchema, _keyLen);
				    
				    _curLeaf.reset();
				    hasValue = _curLeaf.numEntries() > 0;				    
				}
				else
				{
				    goOn = false;;
				}
			    }
			}
		    }
		}
	    }	    	    
	    break;	    
	}
	}
    }
    else
    {
	// go to the beginning
	
	bool leafFound = false;
	_curFileId = _rootFileId;
	_curPageId = _rootPageId;
	
	CegoBTreeValue minVal;
	
	while ( leafFound == false )
	{	    
	    CegoBufferPage bp;
	    
	    _pDBMng->bufferFix(bp, _tabSetId, _curFileId, _curPageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
	    
	    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
	    {
		CegoBTreeNode traceNode;
		traceNode.setType(CegoBTreeNode::NODE);
		traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
		traceNode.setSchema(&_btreeSchema, _keyLen);
		traceNode.setFileId(bp.getFileId());
		traceNode.setPageId(bp.getPageId());
				    	
		traceNode.reset();
		traceNode.nextChildPointer(_curFileId, _curPageId);

		_parentPageStack.Push(bp);

	    }
	    else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
	    {	
		_curPage = bp;
		_curLeaf.setType(CegoBTreeNode::LEAF);
		_curLeaf.setPtr(bp.getChunkEntry(), bp.getChunkLen());
		_curLeaf.setFileId(bp.getFileId());
		_curLeaf.setPageId(bp.getPageId());
		_curLeaf.setSchema(&_btreeSchema, _keyLen);

		CegoBTreeValue iv;
		
		_curLeaf.reset();
		found = _curLeaf.nextValue(iv, dp);
		
		leafFound=true;
	    }	   	    
	}
    }

    if ( found )
    {
	char* p;
	int len;
	
	_pTabMng->releaseDataPtrUnlocked(_dataPtr, false);
	_dataPtr = _pTabMng->claimDataPtrUnlocked(_tabSetId, CegoBufferPool::SYNC, dp, p, len);
	
	// skipping tid
	
	unsigned long long tid;
	unsigned long long tastep;
	CegoTupleState ts;

	int toff = _qh.decodeTupleHeader(tid, tastep, ts, p);

	char* tp = p + toff;
	int tlen = len - toff;
	
	if (tid != 0)
	{	    
	    if ( _ignoreTouched )
	    {			
		if ( ts == INSERTED 
		     && tid == _pTabMng->getTID(_tabSetId) 
		     && tastep < _pTabMng->getTAStep(_tabSetId) )
		{
		    _qh.decodeFVL(fl, tp, tlen);
		    return true;   
		}
		else
		{
		    return getNext(fl,dp);
		}
	    }
	    else
	    {
		if ( ( _readUncommitted == true 
		       && ts == INSERTED )
		     || ( _readUncommitted == false 
			  && (( ts == INSERTED && tid == _pTabMng->getTID(_tabSetId)) 
			      || ( ts == DELETED && tid != _pTabMng->getTID(_tabSetId)))))
		{
		    _qh.decodeFVL(fl, tp, tlen);
		    return true;   
		}
		else
		{
		    return getNext(fl,dp);		    
		}
	    }
	}
	else
	{
	    _qh.decodeFVL(fl, tp, tlen);
	    return true;
	}
    }
    else
    {
	_eoc=true;
    }   
    return false;
}

bool CegoBTreeCursor::getNext(ListT<CegoField>& fl, CegoDataPointer& dp)
{
    if ( _eoc )
	return false;

    if ( fl.isEmpty() )
    {
	throw Exception(EXLOC, "Empty field list");
    }

    CegoBTreeValue iv;
    
    bool moreValue = _curLeaf.nextValue(iv, dp);

    while ( moreValue == false )
    {
	int fileId = _curPage.getNextFileId();
	int pageId = _curPage.getNextPageId();

	_pDBMng->bufferUnfix(_curPage, false, _pTabMng->getLockHandler());

	if ( fileId || pageId )
	{
	    _pDBMng->bufferFix(_curPage, _tabSetId, fileId, pageId, CegoBufferPool::SYNC, _pTabMng->getLockHandler());
	    
	    _curLeaf.setPtr(_curPage.getChunkEntry(), _curPage.getChunkLen());
	    _curLeaf.setFileId(_curPage.getFileId());
	    _curLeaf.setPageId(_curPage.getPageId());
	    _curLeaf.setSchema(&_btreeSchema, _keyLen);
	        
	    _curLeaf.reset();
	    moreValue = _curLeaf.nextValue(iv, dp);
	}
	else
	{
	    _eoc = true;
	    return false;	    
	}
    }

    do {

	bool attrMatch = false;

	// get next available entry from leaf page
        
	if (_pAttrCond)
	{
	    CegoComparison comp = compValue(iv);
	    
	    if ( _pAttrCond->getPrimaryCompMode() == CegoAttrComp::BTWN )
	    {
		_pAttrCond->setPrimaryComparison(LESS_EQUAL_THAN);
	    }
	  	  
	    switch (_pAttrCond->getPrimaryComparison())
	    {
		
	    case EQUAL:
	    {	  
		switch (comp)
		{
		case EQUAL:
		    attrMatch=true;
		    break;
		default:
		    break;
		}
		break;
	    }
	    case NOT_EQUAL:
	    {
		switch (comp)
		{
		case LESS_THAN:
		    attrMatch=true;
		    break;
		case MORE_THAN:
		    attrMatch=true;
		    break;
		case EQUAL:
		    return getNext(fl, dp);
		default:
		    break;
		}
		break;
	    }
	    case LESS_THAN:
	    {		
		switch (comp)
		{
		case LESS_THAN:
		    attrMatch=true;
		    break;
		default:
		    break;
		}
		break;
	    }
	    case MORE_THAN:	     
	    {
		switch (comp)
		{
		case MORE_THAN:
		    attrMatch=true;
		    break;
		default:
		    break;		
		}	
		break;       
	    }
	    case LESS_EQUAL_THAN:
	    {
		switch (comp)
		{
		case EQUAL:
		    attrMatch=true;
		    break;
		case LESS_THAN:
		    attrMatch=true;
		    break;
		default:
		    break;
		}
		break;
	    }   
	    case MORE_EQUAL_THAN:
	    {
		switch (comp)
		{
		case EQUAL:
		    attrMatch=true;
		    break;
		case MORE_THAN:
		    attrMatch=true;
		    break;
		default:
		    break;
		}
		break;
	    }
	    }	    
	}
	else
	{
	    attrMatch = true;
	}
	
	if ( attrMatch )
	{
	    
	    char* p;
	    int len;
	    	    
	    // dp = _ie.getData();
	    
	    _pTabMng->releaseDataPtrUnlocked(_dataPtr, false);
	    _dataPtr = _pTabMng->claimDataPtrUnlocked(_tabSetId, CegoBufferPool::SYNC, dp, p, len);
	    
	    // skipping tid
	    
	    unsigned long long tid;
	    unsigned long long tastep;
	    CegoTupleState ts;

	    int toff = _qh.decodeTupleHeader(tid, tastep, ts, p);

	    char* tp = p + toff;
	    int tlen = len - toff;

	    if (tid != 0)
	    {		    
		if ( _ignoreTouched )
		{
		    if ( ts == INSERTED 
			 && tid == _pTabMng->getTID(_tabSetId) 
			 && tastep < _pTabMng->getTAStep(_tabSetId) )
		    {
			_qh.decodeFVL(fl, tp, tlen);
			return true;   
		    }
		    else
		    {
			return getNext(fl, dp);		
		    }
		}
		else
		{
		    if ( ( _readUncommitted == true 
			   && ts == INSERTED )
			 || ( _readUncommitted == false 
			      && (( ts == INSERTED && tid == _pTabMng->getTID(_tabSetId)) 
				  || ( ts == DELETED && tid != _pTabMng->getTID(_tabSetId)))))
		    {
			_qh.decodeFVL(fl, tp, tlen);
			return true;
		    }
		    else
		    {
			return getNext(fl, dp);		
		    }
		}
	    }
	    else
	    {	       
		_qh.decodeFVL(fl, tp, tlen);
		return true;
	    }
	}
	else
	{
	    abort();
	    _eoc = true;
	    return false;
	}
    }  while (1);
}

void CegoBTreeCursor::abort()
{
    _pTabMng->releaseDataPtrUnlocked(_dataPtr, false);
    _dataPtr = CegoBufferPage();
	
    if ( _curPage.isFixed() )
	_pDBMng->bufferUnfix(_curPage, false, _pTabMng->getLockHandler());	

    // unfix node pages on stack
    CegoBufferPage bp;
    while ( _parentPageStack.Pop(bp) )
    {
	// cout << "Unfixing parent page " << bp.getFileId() << " " << bp.getPageId() << endl;
	_pDBMng->bufferUnfix(bp, false, _pTabMng->getLockHandler());
    }

    if ( _lockId )
    {
	_pTabMng->getLockHandler()->unlockData(CegoObject::BTREE, _lockId);
	_lockId = 0;
    }
}

void CegoBTreeCursor::reset()
{
    abort();
}

CegoComparison CegoBTreeCursor::compValue(const CegoBTreeValue& val)
{

    ListT<CegoFieldValue> fvl = val.valueToFVL(_btreeSchema);

    CegoField* pF = _btreeSchema.First();
    CegoFieldValue *pFV = fvl.First();

    while ( pF ) 
    {
	CegoAttrComp* pAC = _pAttrCond->getAttrCompSet().First();

	while ( pAC )
	{
	    if  ( (Chain)pAC->getAttrName() == (Chain)pF->getAttrName()  )
	    {	       
		if ( *pFV < pAC->getFieldValue() )
		{
		    return LESS_THAN;
		}		
		else if ( pAC->getCompMode() == CegoAttrComp::VAL
		     || pAC->getCompMode() == CegoAttrComp::ATTR )
		{
		    if ( *pFV > pAC->getFieldValue() )
		    {
			return MORE_THAN;
		    }
		}
		else if ( pAC->getCompMode() == CegoAttrComp::BTWN )
		{
		    if ( *pFV > pAC->getFieldValue2() )
		    {
			return MORE_THAN;
		    }
		}		
	    }
	    pAC = _pAttrCond->getAttrCompSet().Next();

	    // check for partial modus here
	    if ( pAC )
	    {
		if ( pAC->getComparison() != EQUAL )
		    return EQUAL;
	    }
	}
	pF = _btreeSchema.Next();
	pFV = fvl.Next();
    }
    return EQUAL;
}
