///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBufferPool.cc
// -----------------
// Cego buffer pool implementation module
//
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBufferPool
// 
// Description: The buffer pool management class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/CommandExecuter.h>
#include <lfcbase/Datetime.h>
#include <lfcbase/Sleeper.h>

// CEGO INCLUDES
#include "CegoBufferPool.h"
#include "CegoDefs.h"
#include "CegoXMLdef.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

#define BUFFERPOOLHEAD (((sizeof(BufferPoolHead)-1)/BUPMNG_ALIGNMENT)+1)*BUPMNG_ALIGNMENT
#define BUFFERHEAD (((sizeof(BufferHead)-1)/BUPMNG_ALIGNMENT)+1)*BUPMNG_ALIGNMENT

CegoBufferPool::CegoBufferPool(const Chain& xmlDef, const Chain& logFile, const Chain& progName) : CegoLogManager(xmlDef, logFile, progName)
{    
    _pBufPool = 0;
    _numDiskRead=0;
    _numDiskWrite=0;
    _fixCount=0;
    _avgReadDelay=0;
    _avgWriteDelay=0;

    Datetime ts;
    _statStart = ts.asInt();
    _poolStart = ts.asInt();
    _modId = getModId("CegoBufferPool");
}

CegoBufferPool::~CegoBufferPool()
{
}

void CegoBufferPool::initPool(unsigned long long numSegment, unsigned long long numPages)
{

#ifdef DEBUG
    log(_modId, Logger::DEBUG, Chain("Reading xml def ..."));
#endif

    Chain dbName = getDbName();
    int pageSize = getPageSize();
    _maxFixTries = getMaxFixTries();
    // _hashkey = getBufferPoolHashKey();
    
    _dbName = dbName;
    _numSegment = numSegment;
    _numPages = numPages;
    _pageSize = pageSize;

    if (_pBufPool == 0)
    {
	
	log(_modId, Logger::NOTICE, Chain("Allocating ") + Chain(_numSegment) + Chain(" buffer pool segments ( each ") +  Chain(_numPages) + Chain(" pages ) ..."));

	
	_pBufPool = (void**)malloc(_numSegment * sizeof(void*));
	if ( _pBufPool == NULL )
	{
	    throw Exception(EXLOC, "Cannot initialize pool");
	}

	
	for ( int i=0; i<_numSegment; i++)
	{
	    _pBufPool[i] = malloc(_numPages * (BUFFERHEAD + _pageSize) + BUFFERPOOLHEAD);

	    if ( _pBufPool[i] == NULL )
	    {
		throw Exception(EXLOC, "Cannot initialize pool");
	    }
	    
	    log(_modId, Logger::NOTICE, Chain("Initializing buffer pool pages for segment ") + Chain(i) + Chain(" ..."));
	
	    void *base = (void*)_pBufPool[i];
	    void *ptr = (void*)_pBufPool[i];
	    
	    BufferPoolHead bph;
	    bph.numPages=_numPages;
	    
	    memcpy(base, &bph, sizeof(BufferPoolHead));
	    
	    ptr = (void*)(BUFFERPOOLHEAD + (long long)base);	
	    
	    BufferHead ibh;
	    ibh.isOccupied = NOT_OCCUPIED;
	    ibh.isDirty = 0;
	    ibh.numFixes = 0;
	    ibh.tabSetId = 0;
	    ibh.fileId = 0;
	    ibh.pageId = 0;
	    ibh.fixStat = 0;
	    ibh.numUsage = 0;
	    
	    for (int i = 0; i<_numPages; i++)
	    {
		memcpy(ptr, &ibh,  sizeof(BufferHead) );
		ptr = (void*)((long long)ptr + BUFFERHEAD + _pageSize);
	    }
	    
	    log(_modId, Logger::NOTICE, Chain("Buffer pool initialized"));
	}
    }
    else
    {
	throw Exception(EXLOC, "Buffer pool already created");
    }    
}

void CegoBufferPool::removePool()
{
    if (_pBufPool)
    {
	log(_modId, Logger::NOTICE, Chain("Removing pool ..."));
	
	for ( int i=0; i<_numSegment; i++)
	{
	    free ( _pBufPool[i] );	   
	}
	free ( _pBufPool );
	
	_pBufPool = 0;
	log(_modId, Logger::NOTICE, Chain("Pool removed"));
    }
}

Chain CegoBufferPool::getDBName()
{
    return _dbName;
}

int CegoBufferPool::calcHash(int fileId, int pageId)
{
    // we have to cast to long, since for large fileid values, we might exceed integer range
    // unsigned long long s = (unsigned long long)(fileId+1) * (unsigned long long)_hashkey + (unsigned long long)(pageId+1);    

    unsigned long long s = getRegPageOffset(fileId) + (unsigned long long)(pageId+1);    
    unsigned long long d = (unsigned long long)calcSegment(fileId, pageId) * (unsigned long long)_numPages;
    
    if ( s > d )
	s = s - d;
    
    return (int)( s % (unsigned long long)_numPages);
}

int CegoBufferPool::calcSegment(int fileId, int pageId)
{
    // we have to cast to long, since for large fileid values, we might exceed integer range
    // unsigned long long s = (unsigned long long)(fileId+1) * (unsigned long long)_hashkey + (unsigned long long)(pageId+1);

    unsigned long long s = getRegPageOffset(fileId) + (unsigned long long)(pageId+1);    
    // for a better usage of the whole buffer pool, we use segments in a linear way
    s = s / _numPages;

    return (int)( s % (unsigned long long)_numSegment);
}

void CegoBufferPool::bufferFix(CegoBufferPage &bp, int tabSetId, int fileId, int pageId, FixMode m, CegoLockHandler *pLockHandle, int numTry)
{
    _fixCount++;

    if ( _fixCount % BUPMNG_STATSPERIOD == 0 )
    {
	resetStats();
    }

    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }

    int segid = calcSegment(fileId, pageId);

    void *base = (void*)_pBufPool[segid];

    int hashId = calcHash(fileId, pageId);

    void* bufPtr = (void*)((long long)base + (long long)BUFFERPOOLHEAD + (long long )hashId * (long long)( BUFFERHEAD + _pageSize ));
    
    bool isFixed = false;
    
    int numTries = 0;
    
    // Step I : searching for possible slot

    while ( isFixed == false && numTries < _maxFixTries )
    {
	pLockHandle->lockBufferPool(hashId, CegoLockHandler::WRITE);

	BufferHead bh;
	memcpy(&bh, bufPtr, sizeof(BufferHead));
	
	int minFixStat = -1;

	if ( bh.isOccupied != NOT_OCCUPIED && bh.pageId == pageId && bh.fileId == fileId)
	{	    
	    isFixed = true;

	    if ( m == CegoBufferPool::PERSISTENT )
		bh.isOccupied = PERSISTENT_OCCUPIED;
	    else if ( m == CegoBufferPool::NOSYNC && bh.isOccupied == WRITE_ON_SYNC)
		bh.isOccupied = WRITE_ON_DIRTY;
	    else
	    {
		bh.isOccupied = WRITE_ON_SYNC;
	    }
	    
	    bh.numFixes = bh.numFixes + 1;
	    bh.fixStat++;
	    bh.numUsage++;

	    memcpy(bufPtr, &bh, sizeof(BufferHead));
	    
	    pLockHandle->unlockBufferPool(hashId);

	}
	else 
	{
	    pLockHandle->unlockBufferPool(hashId);

	    hashId =  ( hashId + 1 ) % _numPages;
	    bufPtr = (void*)( (long long)hashId * (long long)( BUFFERHEAD + _pageSize ) + (long long)base + (long long)BUFFERPOOLHEAD );
	    numTries++;
	}
	
    }
    
    // Step II : page is not in buffercache yet, searching free and not occupied slot 

    void* minFixStatBufPtr = 0;
    int minHashId = -1;

    if ( ! isFixed )
    {
	numTries = 0;
	hashId = calcHash(fileId, pageId);
	bufPtr = (void*)((long long)base + (long long)BUFFERPOOLHEAD + (long long)hashId * (long long)( BUFFERHEAD + _pageSize ));
    }
    
    while ( ! isFixed && numTries < _maxFixTries )
    {
	try
	{
	    pLockHandle->lockBufferPool(hashId, CegoLockHandler::WRITE);
	}
	catch ( Exception e )
	{
	    if ( minHashId != -1 )
		pLockHandle->unlockBufferPool(minHashId);		
	    throw e;
	}
	
	// cout << "Trying to fix with try " << numTries << endl;
	BufferHead bh;
	memcpy(&bh, bufPtr, sizeof(BufferHead));
	
	int minFixStat = -1;
	
	if ( bh.isOccupied == NOT_OCCUPIED )
	{
	    if ( m == CegoBufferPool::PERSISTENT )
		bh.isOccupied = PERSISTENT_OCCUPIED;
	    else if ( m == CegoBufferPool::SYNC )
		bh.isOccupied = WRITE_ON_SYNC;
	    else 
		bh.isOccupied = WRITE_ON_DIRTY;

	    bh.pageId = pageId;
	    bh.fileId = fileId;
	    bh.isDirty = 0;
	    bh.numFixes = 1;
	    
	    try
	    {		
		_numDiskRead++;
		_diskReadTimer.start();
		readPage(fileId, pageId, bh.tabSetId, bh.fixStat, (char*)((long long)bufPtr+BUFFERHEAD), pLockHandle);
		_diskReadTimer.stop();
		_avgReadDelay = ( _diskReadTimer.getSum() /  (_numDiskRead+(unsigned long long)1)  ) / 1000;
	    }
	    catch ( Exception e )
	    {
		pLockHandle->unlockBufferPool(hashId);
		if ( minHashId != -1 )
		    pLockHandle->unlockBufferPool(minHashId);		
		throw e;
	    }

	    bh.fixStat++;

	    bh.numUsage++;
	    
	    memcpy(bufPtr, &bh, sizeof(BufferHead));	

	    isFixed = true;

	    pLockHandle->unlockBufferPool(hashId);
	    if ( minHashId != -1 )
		pLockHandle->unlockBufferPool(minHashId);
	    
	}
	else 
	{

	    // for now, we take any tabSetId
	    // tabSetId = bh.tabSetId;
	    
	    if ( ( ( bh.isOccupied != PERSISTENT_OCCUPIED && bh.isDirty == 0 )
		   || bh.isOccupied == WRITE_ON_DIRTY )
		 && bh.numFixes == 0 && ( minFixStat > bh.fixStat || minFixStat == -1))
	    {
		minFixStatBufPtr = bufPtr;
		// cout << "Found page with minFixStat = " << bh.fixStat << endl;
		minFixStat = bh.fixStat;
		
		if ( minHashId != -1 )
		{
		    pLockHandle->unlockBufferPool(minHashId);	   
		}
		
		minHashId = hashId;

		try
		{
		    pLockHandle->lockBufferPool(minHashId, CegoLockHandler::WRITE);
		}
		catch ( Exception e )
		{
		    pLockHandle->unlockBufferPool(hashId);
		    throw e;
		}
	    }
	    
	    pLockHandle->unlockBufferPool(hashId);
	    
	    hashId =  ( hashId + 1 ) % _numPages;

	    bufPtr = (void*)( (long long)hashId * (long long)( BUFFERHEAD + _pageSize ) + (long long)base + (long long)BUFFERPOOLHEAD );
	    numTries++;
	}	
    }

    // Step III : page is not in buffercache yet and all slots are occupied, using the occupied slot
    //            with lowest fix stat 
    
    if ( ! isFixed && minFixStatBufPtr != 0)
    {
	
	bufPtr = minFixStatBufPtr;

	// minHashId is already locked
	// pLockHandle->lockBufferPool(minHashId);
		
	BufferHead bh;
	memcpy(&bh, bufPtr, sizeof(BufferHead));
	
	if ( bh.isOccupied == WRITE_ON_DIRTY && bh.isDirty == 1 )
	{
	    _numDiskWrite++;
	    
#ifdef DEBUG
	    log(_modId, Logger::DEBUG, Chain("Async write of page [") + Chain(bh.fileId) + Chain(",") + Chain(bh.pageId) + Chain("]"));
#endif

	    // cout << "---- >>>> Async write of page [" << bh.fileId << "," << bh.pageId << "]" << endl;

	    _diskWriteTimer.start();
	    writePage(bh.fileId, bh.pageId, bh.fixStat, (char*)((long long)bufPtr + BUFFERHEAD), pLockHandle);
	    _diskWriteTimer.stop();

	    _avgWriteDelay = ( _diskWriteTimer.getSum() /  (_numDiskWrite+(unsigned long long)1)  ) / 1000;

	}
	
	// cout << "Using occupied page ( type " << bh.isOccupied << " ) with fileId " << bh.fileId << " pageId " << bh.pageId << " numFixes " << bh.numFixes << endl;

	if ( m == CegoBufferPool::PERSISTENT )
	    bh.isOccupied = PERSISTENT_OCCUPIED;
	else if ( m == CegoBufferPool::SYNC )
	    bh.isOccupied = WRITE_ON_SYNC;
	else if ( m == CegoBufferPool::NOSYNC )
	    bh.isOccupied = WRITE_ON_DIRTY;

	bh.pageId = pageId;
	bh.fileId = fileId;
	bh.isDirty = 0;
	bh.numFixes = 1;
	
	try
	{
	    _numDiskRead++;
	    _diskReadTimer.start();
	    readPage(fileId, pageId, bh.tabSetId, bh.fixStat, (char*)((long long)bufPtr+BUFFERHEAD), pLockHandle);
	    _diskReadTimer.stop();

	    _avgReadDelay = ( _diskReadTimer.getSum() /  (_numDiskRead+(unsigned long long)1) ) / 1000;

	}
	catch ( Exception e )
	{
	    pLockHandle->unlockBufferPool(minHashId);
	    throw e;
	}
	bh.fixStat++;
	bh.numUsage++;
	
	memcpy(bufPtr, &bh, sizeof(BufferHead));
	
	pLockHandle->unlockBufferPool(minHashId);

	isFixed = true;

    }

    if ( ! isFixed )
    {

	log(_modId, Logger::NOTICE, Chain("Forced checkpoint for tabSetId ") + Chain(tabSetId) + Chain(" at lsn=") + Chain(getLSN(tabSetId)));

	writeCheckPoint(tabSetId, true, Chain(""), 0,  pLockHandle);

	if ( numTry > BUPMNG_FIXLOOP  )
	    throw Exception(EXLOC, "No more buffers available");

	numTry++;
	return bufferFix(bp, tabSetId, fileId, pageId, m, pLockHandle, numTry);
	
    }

    bp.setPageSize(_pageSize);
    bp.setPagePtr((char*)((long long)bufPtr+BUFFERHEAD));
    bp.setPageHead((CegoBufferPage::PageHead*)((long long)bufPtr+BUFFERHEAD));
    bp.setFileId(fileId);
    bp.setPageId(pageId);
    bp.setFixed(true);
}

void CegoBufferPool::emptyFix(CegoBufferPage &bp, int tabSetId, FixMode m, CegoFileHandler::FileType ft, CegoLockHandler *pLockHandle, bool doAppend)
{
    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }
    
    int fileId, pageId;
    
    unsigned* fbm;
    int fbmSize=0;

    allocatePage(tabSetId, ft, fileId, pageId, pLockHandle, fbm, fbmSize, doAppend);
    
    // cout << "Allocated page " << fileId << "/" << pageId << endl;

    if ( fbmSize > 0 )
    {
	logBM(tabSetId, fileId, fbm, fbmSize);
	delete fbm;
    }

    bufferFix(bp, tabSetId, fileId, pageId, m, pLockHandle);
    bp.initPage(CegoBufferPage::TABLE);
    bp.setFixed(true);
}

void CegoBufferPool::bufferUnfix(CegoBufferPage &bp, bool isDirty, CegoLockHandler *pLockHandle)
{
 
    // cout << "BufferUnfix of page " << bp.getFileId() << "," << bp.getPageId() << endl;
    
    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }

    int segid = calcSegment(bp.getFileId(), bp.getPageId());
    
    void *base = (void*)_pBufPool[segid];
    char* bufPtr = (char*)bp.getPagePtr();
    int hashId = ( (long long)bufPtr - (long long)base - BUFFERPOOLHEAD ) / (  BUFFERHEAD + _pageSize );
    
    bufPtr = bufPtr - BUFFERHEAD;
    
    BufferHead bh;
    
    pLockHandle->lockBufferPool(hashId, CegoLockHandler::WRITE);
    
    memcpy(&bh, bufPtr, sizeof(BufferHead));
    
    // cout << "NumFixes = " << bh.numFixes << endl;
    
    if ( bh.numFixes > 0 )
    {
	if ( isDirty )
	{
	    bh.isDirty = 1;
	}
	bh.numFixes--;
    }
    else
    {	
	pLockHandle->unlockBufferPool(hashId);
	throw Exception(EXLOC, "Number of fixes is already zero");
    }
    memcpy(bufPtr, &bh, sizeof(BufferHead));
    
    bp.setFixed(false);
    
    pLockHandle->unlockBufferPool(hashId);
}    

void CegoBufferPool::bufferRelease(CegoBufferPage &bp, CegoLockHandler *pLockHandle)
{

    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }
    
    char* bufPtr = (char*)bp.getPagePtr();
    bufPtr = bufPtr - BUFFERHEAD;
    
    BufferHead bh;
    
    int hashId = calcHash(bp.getFileId(), bp.getPageId());
    
    pLockHandle->lockBufferPool(hashId, CegoLockHandler::WRITE);
    
    try
    {
	memcpy(&bh, bufPtr, sizeof(BufferHead));
	
	bh.isOccupied = NOT_OCCUPIED;
	bh.numFixes=0;
	bh.isDirty = 0;
	bh.tabSetId = 0;
	bh.fileId = 0;
	bh.pageId = 0;
	bh.fixStat = 0;
	bh.numUsage = 0;
	       
	memcpy(bufPtr, &bh, sizeof(BufferHead));
	
	int fbmSize=0;
	unsigned* fbm;

	releasePage(bp.getFileId(), bp.getPageId(), pLockHandle, fbm, fbmSize);
	
	if ( fbmSize > 0 )
	{
	    logBM(bh.tabSetId, bp.getFileId(), fbm, fbmSize);
	    delete fbm;
	}
	
	pLockHandle->unlockBufferPool(hashId);
	
    }
    catch ( Exception e )
    {
	pLockHandle->unlockBufferPool(hashId);
	throw Exception(EXLOC, "Cannot release file page", e);
    }
    bp.setFixed(false);    
}

int CegoBufferPool::writeCheckPoint(int tabSetId, bool switchLog, const Chain& escCmd, int timeout, CegoLockHandler *pLockHandle)
{
    
    pLockHandle->lockBufferPool();
    
    unsigned long long lsn;
    try
    {
	
	if (_pBufPool == 0)
	{
	    throw Exception(EXLOC, "No valid bufferpool" );
	}
	
	if ( escCmd != Chain("") )
	{
	    log(_modId, Logger::NOTICE, Chain("Executing escape command <") + escCmd + Chain(">"));
	    
	    char *pShell = getenv(CGEXESHELLVARNAME);

	    Chain shellCmd;
	    if ( pShell == NULL )
	    {
		shellCmd = Chain(CGSTDEXESHELL);
	    }
	    else
	    {
		shellCmd = Chain(pShell);
	    }

	    CommandExecuter cmdExe(shellCmd);
	    
	    try 
	    {
		int retCode = cmdExe.execute(escCmd, timeout);
		log(_modId, Logger::NOTICE, Chain("Escape command finished with return code : <") + Chain(retCode) + Chain(">"));
	    }
	    catch ( Exception e )
	    {
		Chain msg;
		e.pop(msg);
		log(_modId, Logger::LOGERR, msg);
		throw Exception(EXLOC, msg);
	    }
	}
	
	if ( switchLog )
	{
	    while ( switchLogFile(tabSetId) == false )
	    {
		
		log(_modId, Logger::NOTICE, Chain("Logfile for tabSetId ") + Chain(tabSetId) + Chain(" still active, switch failed"));
		
		Sleeper s;
		s.secSleep(LOGMNG_LOGSWITCH_WAIT_DELAY);
		
	    }
	}
	
	lsn = getLSN(tabSetId);    
	
	if ( lsn > 0 )
	    lsn--;

	setCommittedLSN(tabSetId, lsn);

	Chain currentState = getTableSetRunState(tabSetId);
	setTableSetRunState(tabSetId, XML_CHECKPOINT_VALUE);

	// sync control information to xml database file
	doc2Xml();
	
	for ( int i=0; i<_numSegment; i++)
	{
	    void *bufPtr = (void*)( (long long)_pBufPool[i] + BUFFERPOOLHEAD );
	    
	    int pageCount = 0;
	    
	    while ( pageCount < _numPages )
	    {
		BufferHead bh;
		memcpy(&bh, bufPtr, sizeof(BufferHead));
		       
		if ( bh.isOccupied != NOT_OCCUPIED && bh.isDirty != 0 && bh.tabSetId == tabSetId )
		{		    
		    if ( isBackup(bh.fileId) )
		    {
			if (  isMarked(bh.fileId, bh.pageId) == false )
			{	    
#ifdef DEBUG
			    log(_modId, Logger::DEBUG, Chain("Reading page (") + Chain(bh.fileId) + Chain(",") + Chain(bh.pageId) + Chain(") to log ..."));
#endif	
			    int ts;
			    int fixStat;
			    char *pageData = new char[_pageSize];
			    readPage(bh.fileId, bh.pageId, ts, fixStat, pageData, pLockHandle);
			    
			    CegoLogRecord lr;
			    lr.setAction(CegoLogRecord::LOGREC_BUPAGE);
			    lr.setData(pageData);
			    lr.setDataLen(_pageSize);
			    lr.setFileId(bh.fileId);
			    lr.setPageId(bh.pageId);
			    
			    if (  logAction(tabSetId, lr) == CegoLogManager::LOG_FULL )
			    {
				
				int gid;
				Chain logFile;
				int logSize = 0 ;
				
#ifdef DEBUG
				log(_modId, Logger::DEBUG, Chain("Switching logFiles ..."));
#endif	    
				
				while ( switchLogFile(tabSetId) == false )
				{
				    log(_modId, Logger::NOTICE, Chain("Logfile for tabSetId ") + Chain(tabSetId) + Chain(" still active, switch failed"));
				    
				    Sleeper s;
				    s.secSleep(LOGMNG_LOGSWITCH_WAIT_DELAY);
				    
				}
				
				if ( logAction(tabSetId, lr) != CegoLogManager::LOG_SUCCESS )
				{
				    throw Exception(EXLOC, "Cannot write to log");
				}	    		
			    }
			}
		    }
		    
		    _numDiskWrite++;
		    _diskWriteTimer.start();
		    writePage(bh.fileId, bh.pageId, bh.fixStat, (char*)((long long)bufPtr + BUFFERHEAD), pLockHandle);
		    _diskWriteTimer.stop();
		    
		    _avgWriteDelay = ( _diskWriteTimer.getSum() /  (_numDiskWrite+(unsigned long long)1) ) / 1000;
		        
		    bh.isDirty = 0;
		    memcpy(bufPtr, &bh, sizeof(BufferHead));	    
		}
		bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));
		pageCount++;
	    }
	}
	// set back tableset runstate to saved value
	setTableSetRunState(tabSetId, currentState);
	// sync control information to xml database file
	doc2Xml();
	
    }
    catch ( Exception e )
    {
	pLockHandle->unlockBufferPool();
	throw e;
    }

    pLockHandle->unlockBufferPool();    
    return lsn;
}

void CegoBufferPool::writeAndRemoveTabSet(int tabSetId, CegoLockHandler *pLockHandle)
{

    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }

    for ( int i=0; i<_numSegment; i++)
    {
	void *bufPtr = (void*)( (long long)_pBufPool[i] + BUFFERPOOLHEAD );
	
	int pageCount = 0;
	
	while ( pageCount < _numPages )
	{
	    BufferHead bh;
	    memcpy(&bh, bufPtr, sizeof(BufferHead));
	    
	    if ( bh.tabSetId == tabSetId )
	    {
		
		
		if ( bh.isOccupied != NOT_OCCUPIED && bh.isDirty != 0 )
		{
		    _numDiskWrite++;
		    
		    _diskWriteTimer.start();
		    writePage(bh.fileId, bh.pageId, bh.fixStat, (char*)((long long)bufPtr + BUFFERHEAD), pLockHandle);
		    _diskWriteTimer.stop();
		    
		    _avgWriteDelay = ( _diskWriteTimer.getSum() /  (_numDiskWrite+(unsigned long long)1) ) / 1000;
		}
		
		bh.isOccupied = NOT_OCCUPIED;
		bh.numFixes=0;
		
		memcpy(bufPtr, &bh, sizeof(BufferHead));
		
	    }
	    bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));
	    pageCount++;
	}
    }    
    releaseFiles(tabSetId);
}

int CegoBufferPool::uptime() const
{
    Datetime ts;
    return ts.asInt() - _poolStart;
}

void CegoBufferPool::poolInfo(int& pageSize, 
			      unsigned long long& numTotal, 
			      unsigned long long& numUsed, 
			      unsigned long long& numFree, 
			      unsigned long long& numDirty, 
			      unsigned long long& numFixes, 
			      unsigned long long& numPersistent, 
			      unsigned long long& numNoSync, 
			      unsigned long long& numDiskRead, 
			      unsigned long long& numDiskWrite,
			      double& hitRate,
			      double& spreadRate,
			      unsigned long long& readDelay,
			      unsigned long long& writeDelay,
			      unsigned long long& curFixCount,
			      unsigned long long& maxFixCount,
			      int& statStart,
                              int& uptime) const
{

    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }
    
    pageSize = _pageSize;
    numTotal = _numPages * _numSegment;
    numUsed=0;
    numFree=0;
    numFixes=0;
    numDirty=0;
    numPersistent=0;
    numNoSync=0;

    unsigned long long numUsage = 0;
    
    for ( int i=0; i<_numSegment; i++)
    {
	void *bufPtr = (void*)( (long long)_pBufPool[i] + BUFFERPOOLHEAD );
	
	unsigned long long pageCount = 0;

	unsigned long long numSegPageUsed = 0;
	
	while ( pageCount < _numPages )
	{    
	    BufferHead bh;
	    memcpy(&bh, bufPtr, sizeof(BufferHead));
	    
	    numFixes += (long)bh.numFixes;
	    numDirty += (long)bh.isDirty;

	    // we don't count persistent hot spot pages
	    if ( bh.isOccupied != PERSISTENT_OCCUPIED )
		numUsage += bh.numUsage;
	    
	    if (bh.isOccupied == NOT_OCCUPIED)
	    {
		numFree++;
	    }
	    else if (bh.isOccupied == WRITE_ON_SYNC)
	    {
		numSegPageUsed++;
		numUsed++;
	    }
	    else if (bh.isOccupied == WRITE_ON_DIRTY)
	    {
		numSegPageUsed++;
		numUsed++;
		numNoSync++;
	    }
	    else if (bh.isOccupied == PERSISTENT_OCCUPIED)
	    {
		numSegPageUsed++;
		numUsed++;
		numPersistent++;
	    }   
	    
	    bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));     
	    pageCount++;
	}
	// cout << "Segment " << i << " : " << numSegPageUsed << " pages used" << endl; 
    }	

    numDiskWrite = _numDiskWrite;
    numDiskRead = _numDiskRead;
	
    // hitRate = BUPMNG_STATSPERIOD - _numDiskRead / ( _fixCount / BUPMNG_STATSPERIOD + 1) ;
    hitRate = 100.0 * ( (double)_fixCount - _numDiskRead + 1 ) / ( (double)_fixCount + 1 ); 

    double usageMedian = (double)numUsage / (double)( _numSegment * _numPages );
    double usageDist = 0.0;
    
    for ( int i=0; i<_numSegment; i++)
    {
	void *bufPtr = (void*)( (long long)_pBufPool[i] + BUFFERPOOLHEAD );
	
	unsigned long long pageCount = 0;
    
	while ( pageCount < _numPages )
	{    
	    BufferHead bh;
	    memcpy(&bh, bufPtr, sizeof(BufferHead));

	    if ( bh.isOccupied != PERSISTENT_OCCUPIED )
	    {
		usageMedian > bh.numUsage ? usageDist += usageMedian - (double)bh.numUsage : usageDist += (double)bh.numUsage - usageMedian;
	    }

	    bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));
	    pageCount++;
	}
    }

    double avgDist = usageDist / (double)( _numSegment * _numPages );
    // cout << "DistMed = " << usageMedian << endl;
    // cout << "AvgDist = " << avgDist << endl;
    // cout << "NumUsage  = " << numUsage << endl;
    spreadRate =  avgDist / ( usageMedian + 1 ); 

    // cout << "Fix Dist Rate = " << fixDistRate << endl;
    
    readDelay = _avgReadDelay;
    writeDelay = _avgWriteDelay;
    statStart = _statStart;
    curFixCount = _fixCount;
    maxFixCount = BUPMNG_STATSPERIOD;
    
    Datetime ts;
    uptime = ts.asInt() - _poolStart;    
}
    
void CegoBufferPool::getPoolEntryList(ListT<CegoBufferPoolEntry>& entryList)
{
    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }

    entryList.Empty();
    
    for ( int seg=0; seg<_numSegment; seg++)
    {
	void *bufPtr = (void*)( (long long)_pBufPool[seg] + BUFFERPOOLHEAD );
	    
	unsigned long long pageCount = 0;
    	
	while ( pageCount < _numPages )
	{    
	    BufferHead bh;
	    memcpy(&bh, bufPtr, sizeof(BufferHead));
	    
	    bool isDirty = false;
	    if ( bh.isDirty )
		isDirty=true;
	    
	    Chain occState;
	    if (bh.isOccupied == NOT_OCCUPIED)
		occState=Chain("NOTOCCUPIED");
	    else if (bh.isOccupied == WRITE_ON_SYNC)
		occState=Chain("WRITEONSYNC");
	    else if (bh.isOccupied == WRITE_ON_DIRTY)
		occState=Chain("WRITEONDIRTY");
	    else if (bh.isOccupied == PERSISTENT_OCCUPIED)
		occState=Chain("PERSISTENT");
	    
	    CegoBufferPoolEntry bpe(seg, pageCount, occState, isDirty, bh.numFixes, bh.tabSetId, bh.fileId, bh.pageId, bh.fixStat, bh.numUsage);
	    entryList.Insert(bpe);
	    
	    bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));     
	    pageCount++;
	}	
    }
}

void CegoBufferPool::resetStats()
{
    _numDiskRead=0;
    _numDiskWrite=0;
    _fixCount=0;
    _avgReadDelay=0;
    _avgWriteDelay=0;
    _diskReadTimer.reset();
    _diskWriteTimer.reset();
    Datetime ts;
    _statStart = ts.asInt(); 
}


void CegoBufferPool::logBM(int tabSetId, int fileId, unsigned* fbm, int fbmSize)
{    
    CegoLogRecord lr;
    
    lr.setAction(CegoLogRecord::LOGREC_BUFBM);
    lr.setData((char*)fbm);
    lr.setDataLen(fbmSize * sizeof(int));
    lr.setFileId(fileId);
    
    if ( logAction(tabSetId, lr) == CegoLogManager::LOG_FULL )
    {
	
	int gid;
	Chain logFile;
	int logSize = 0 ;
	
	while ( switchLogFile(tabSetId) == false )
	{
	    log(_modId, Logger::NOTICE, Chain("Logfile for tabSetId ") + Chain(tabSetId) + Chain(" still active, switch failed"));
	    Sleeper s;
	    s.secSleep(LOGMNG_LOGSWITCH_WAIT_DELAY);
	    
	}
	
	if ( logAction(tabSetId, lr) != CegoLogManager::LOG_SUCCESS )
	{
	    throw Exception(EXLOC, "Cannot write to log");
	}	    		
    }
}

void CegoBufferPool::printPool()
{
    if (_pBufPool == 0)
    {
	throw Exception(EXLOC, "No valid bufferpool" );
    }
    
    cout << "--- BufferPool ---" << endl;
    cout << "BasePtr: " << (long long)_pBufPool << endl;
    cout << "PageSize: " << _pageSize << endl;
    cout << "NumPages: " << _numPages << endl;
    
    unsigned long long numUsed=0;
    unsigned long long numFree=0;
    unsigned long long numFixes=0;
    unsigned long long numDirty=0;

    for ( int i=0; i<_numSegment; i++)
    {
	void *bufPtr = (void*)( (long long)_pBufPool[i] + BUFFERPOOLHEAD );
	
	unsigned long long pageCount = 0;
    
	while ( pageCount < _numPages )
	{    
	    BufferHead bh;
	    memcpy(&bh, bufPtr, sizeof(BufferHead));
	    
	    numFixes += bh.numFixes;
	    numDirty += (unsigned long long)bh.isDirty;
	    
	    if (bh.isOccupied == 0)
		numFree++;
	    else
		numUsed++;
	    
	    bufPtr = (void*)((long long)bufPtr + (long long)(BUFFERHEAD + _pageSize));     
	    pageCount++;
	}
    }
	
    cout << "NumUsed: " << numUsed << endl;
    cout << "NumFixes: " << numFixes << endl;
    cout << "NumDirty: " << numDirty << endl;
    cout << "NumFree: " << numFree << endl;

    cout << "NumFree: " << numFree << endl;
}
