///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoCaseCond.cc
// ---------------
// Cego case condition
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoCaseCond
// 
// Description: Query case condition container class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoCaseCond.h"
#include "CegoQueryHelper.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseFormater.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoCaseCond::CegoCaseCond()
{
    _elseExpr = 0;
}

CegoCaseCond::CegoCaseCond(char* buf, CegoDistManager *pGTM, int tabSetId)
{
    _elseExpr = 0;
    decode(buf, pGTM, tabSetId);
}

CegoCaseCond::CegoCaseCond(Element* pCaseElement, CegoDistManager *pGTM)
{
    _elseExpr = 0;
    fromElement(pCaseElement, pGTM);
}

CegoCaseCond::~CegoCaseCond()
{

    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	delete *pPredDesc;
	pPredDesc = _predList.Next();
    }
    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	delete *pExpr;
	pExpr = _exprList.Next();
    }
    
    if ( _elseExpr )
	delete _elseExpr;
}

void CegoCaseCond::addPred(CegoPredDesc *pPredDesc, CegoExpr* pExpr)
{
    _predList.Insert(pPredDesc);
    _exprList.Insert(pExpr);
}

void CegoCaseCond::setElseExpr(CegoExpr *pExpr)
{
    _elseExpr = pExpr;
}

ListT<CegoPredDesc*>& CegoCaseCond::getPredList()
{
    return _predList;
}

ListT<CegoExpr*>& CegoCaseCond::getExprList()
{
    return _exprList;
}

CegoExpr* CegoCaseCond::getElseExpr()
{
    return _elseExpr;
}

void CegoCaseCond::setBlock(CegoProcBlock *pBlock)
{    
    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	(*pPredDesc)->setBlock(pBlock);
	pPredDesc = _predList.Next();
    }
    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	(*pExpr)->setBlock(pBlock);
	pExpr = _exprList.Next();
    }
    if ( _elseExpr )
	_elseExpr->setBlock(pBlock);

    _pBlock = pBlock;
}

void CegoCaseCond::setFieldListArray(ListT<CegoField> **pFLA)
{
    _pFLA = pFLA;
}

ListT<CegoAttrDesc*> CegoCaseCond::getAttrRefList() const
{
    ListT<CegoAttrDesc*> attrList;

    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	attrList += (*pPredDesc)->getAttrRefList();
	pPredDesc = _predList.Next();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	attrList += (*pExpr)->getAttrRefList();
	pExpr = _exprList.Next();
    }
    if ( _elseExpr )
	attrList += _elseExpr->getAttrRefList();

    return attrList;
}

int CegoCaseCond::evalReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;

    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	refCount += (*pPredDesc)->evalReferences(pCO, fl);
	pPredDesc = _predList.Next();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	refCount += (*pExpr)->evalReferences(pCO, fl);
	pExpr = _exprList.Next();
    }
    if ( _elseExpr )
	refCount+= _elseExpr->evalReferences(pCO, fl);
    
    return refCount;
}

void CegoCaseCond::clearAttrCache()
{
    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	(*pPredDesc)->clearAttrCache();
	pPredDesc = _predList.Next();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	(*pExpr)->clearAttrCache();
	pExpr = _exprList.Next();
    }
    if ( _elseExpr )
	_elseExpr->clearAttrCache();
}

ListT<CegoField> CegoCaseCond::getFieldList() const
{
    ListT<CegoField> fl;

    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	fl += (*pPredDesc)->getFieldList();
	pPredDesc = _predList.Next();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	fl += (*pExpr)->getFieldList();
	pExpr = _exprList.Next();
    }
    if ( _elseExpr )
	fl += _elseExpr->getFieldList();

    return fl;
}

CegoField CegoCaseCond::evalField() const
{
    CegoField cf = _elseExpr->evalField();
    CegoField f = CegoField( Chain("CASE"), Chain("CASE"), Chain("case when ..."), cf.getType(), cf.getLength());
    return f;
}

CegoFieldValue CegoCaseCond::evalFieldValue() const
{

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();

    while ( pPredDesc && pExpr) 
    {
	if ( CegoQueryHelper::evalPredicate(0, 0, _pFLA, 0, *pPredDesc, _pBlock) )
	{
	    (*pExpr)->setFieldListArray(_pFLA);
	    return (*pExpr)->evalFieldValue();
	}
	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    CegoFieldValue fv;
    
    _elseExpr->setFieldListArray(_pFLA);
    return _elseExpr->evalFieldValue();   
}

CegoCaseCond* CegoCaseCond::clone(bool isAttrRef)
{
    CegoCaseCond *pCaseCond = new CegoCaseCond();

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {
	pCaseCond->addPred((*pPredDesc)->clone(isAttrRef), (*pExpr)->clone(isAttrRef));
	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    pCaseCond->setElseExpr(_elseExpr->clone(isAttrRef));
    
    return pCaseCond;
}

CegoCaseCond& CegoCaseCond::operator = ( const CegoCaseCond& cc)
{
    _predList = cc._predList;
    _exprList = cc._exprList;
    _elseExpr = cc._elseExpr;
    _pFLA = cc._pFLA;
    _pBlock = cc._pBlock;
    return (*this);
}

Chain CegoCaseCond::getId() const
{
    Chain s;
    s = Chain("case");

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {
	s += Chain("w") + (*pPredDesc)->getId() + Chain("t") + (*pExpr)->getId();
	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    s += Chain("e") + _elseExpr->getId();

    return s;
}

Chain CegoCaseCond::toChain() const
{
    Chain s;
    s = Chain("case");

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {
	s += Chain(" when ") + (*pPredDesc)->toChain() + Chain(" then ") + (*pExpr)->toChain();
	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    s += Chain(" else ") + _elseExpr->toChain();
    s += Chain(" end ");

    return s;
}

Chain CegoCaseCond::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatCaseCond(_predList, _exprList, _elseExpr);
}

Element* CegoCaseCond::toElement() const
{
    Element* pCaseClauseElement = new Element(XML_CASECLAUSE_ELEMENT);

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {

	Element* pCaseElement = new Element(XML_CASE_ELEMENT);
	
	pCaseElement->addContent((*pPredDesc)->toElement());
	pCaseElement->addContent((*pExpr)->toElement());
	
	pCaseClauseElement->addContent(pCaseElement);

	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }

    pCaseClauseElement->addContent(_elseExpr->toElement());
    
    return pCaseClauseElement;
}

void CegoCaseCond::fromElement(Element *pCaseClauseElement, CegoDistManager *pGTM)
{

    CegoPredDesc **pPredDesc = _predList.First();
    while ( pPredDesc )
    {
	delete *pPredDesc;
	pPredDesc = _predList.Next();
    }
    _predList.Empty();

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr) 
    {
	delete *pExpr;
	pExpr = _exprList.Next();
    }
    _exprList.Empty();
    
    if ( _elseExpr )
	delete _elseExpr;
        
    ListT<Element*> cl = pCaseClauseElement->getChildren(XML_CASE_ELEMENT);
    Element **pCE = cl.First();
    while ( pCE )
    {	
	ListT<Element*> pl = (*pCE)->getChildren(XML_PRED_ELEMENT);
	ListT<Element*> el = (*pCE)->getChildren(XML_EXPR_ELEMENT);

	Element **pPE = pl.First();
	Element **pEE = el.First();
	if ( pPE && pEE )
	{
	    addPred( new CegoPredDesc(*pPE, pGTM), new CegoExpr(*pEE, pGTM));
	}

	pCE = cl.Next();
    }

    ListT<Element*> el = pCaseClauseElement->getChildren(XML_EXPR_ELEMENT);
    
    Element **pEE = el.First();
    if ( pEE )
    {
	_elseExpr = new CegoExpr(*pEE, pGTM);
    }
}

void CegoCaseCond::encode(char *buf)
{
    char* pE = (char*)buf;

    int numPred = _predList.Size();
    memcpy( pE, &numPred, sizeof(int));
    pE = pE + sizeof(int);

    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {
	(*pPredDesc)->encode(pE);
	pE = pE + (*pPredDesc)->getEncodingLength();
	
	(*pExpr)->encode(pE);
	pE = pE + (*pExpr)->getEncodingLength();

	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    _elseExpr->encode(pE);    
}

void CegoCaseCond::decode(char *buf, CegoDistManager* pGTM, int tabSetId)
{
    char* pE = (char*)buf;

    int numPred;
    memcpy( &numPred, pE, sizeof(int));
    pE = pE + sizeof(int);

    int i=0;
    while ( i < numPred )
    {
	CegoPredDesc *pPred = new CegoPredDesc(pE, pGTM, tabSetId);
	pE = pE + pPred->getEncodingLength();
	_predList.Insert(pPred);
	
	CegoExpr *pExpr = new CegoExpr(pE, pGTM, tabSetId);
	pE = pE + pExpr->getEncodingLength();
	_exprList.Insert(pExpr);
	i++;
    }

    // else
    _elseExpr = new CegoExpr(pE, pGTM, tabSetId);

}

int CegoCaseCond::getEncodingLength() const
{
    // numPred
    int len = sizeof(int);
    
    CegoPredDesc **pPredDesc = _predList.First();
    CegoExpr **pExpr = _exprList.First();
    while ( pPredDesc && pExpr)
    {
	len += (*pPredDesc)->getEncodingLength();	
	len += (*pExpr)->getEncodingLength();

	pPredDesc = _predList.Next();
	pExpr = _exprList.Next();
    }
    len += _elseExpr->getEncodingLength();
    return len;
}

ostream& operator << (ostream& s, const CegoCaseCond& cc)
{
    s << cc.toChain();
    return s;
}
