///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoObjectCursor.cc
// -------------------
// Cego object cursor class implementation
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoObjectCursor
// 
// Description: Basic cursor to trace on content objects
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoDefs.h"
#include "CegoObjectCursor.h"

CegoObjectCursor::CegoObjectCursor()
{
}

CegoObjectCursor::CegoObjectCursor(CegoBufferPool *pBufPool, CegoLockHandler *pLockHandle, 
				   int tabSetId, 
				   CegoObject::ObjectType type, 
				   int fileId, int pageId)
{
    _pBufPool = pBufPool;
    _pLockHandle = pLockHandle;

    _tabSetId = tabSetId;

    _fileId = fileId;
    _pageId = pageId;

    _startFileId = fileId;
    _startPageId = pageId;
    
    _type = type;
    
    _lockId = 0;
    _recLock = 0;
    
    _isEOC = false;

    _modId = _pBufPool->getModId("CegoObjectCursor");
}

CegoObjectCursor::~CegoObjectCursor()
{
    abort();
}

int CegoObjectCursor::getTabSetId()
{
    return _tabSetId;
}

void* CegoObjectCursor::getFirst(int& len, CegoDataPointer& dp)
{

    if ( _isEOC )
	return 0;
    
    try {
	
	_pBufPool->bufferFix(_bp,_tabSetId, _fileId, _pageId, CegoBufferPool::SYNC, _pLockHandle);
	
	try
	{
	    _lockId = _pLockHandle->lockData(_type, _fileId, _pageId, CegoLockHandler::READ);
	}
	catch ( Exception e)
	{
	    Chain msg;
	    e.pop(msg);
	    
	    _lockId = 0;
	    
	    _pBufPool->log(_modId, Logger::LOGERR, Chain("Data page lock error for fileId=") 
			   + Chain(_fileId) + Chain(" pageId=") + Chain(_pageId) + Chain(" : ") +  msg);		
	    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
	    throw Exception(EXLOC, msg);
	}
	
	if ( void* p = _bp.getFirstEntry() )
	{
	    dp.setFileId(_fileId);
	    dp.setPageId(_pageId);
	    
	    len = _bp.getEntryLen();
	    dp.setOffset(_bp.getEntryPos());

	    return p;
	}
	else
	{	
	    _pLockHandle->unlockData(_type, _lockId);    
	    _lockId = 0;
	    
	    _fileId = _bp.getNextFileId();
	    _pageId = _bp.getNextPageId();
	    
	    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
	    
	    while ( ! ( _fileId == 0 && _pageId == 0 ) )
	    {
		
		CegoBufferPage nbp;						
		
		_pBufPool->bufferFix(nbp, _tabSetId, _fileId, _pageId, CegoBufferPool::SYNC, _pLockHandle);
		
		_bp = nbp;
		
		try
		{
		    _lockId = _pLockHandle->lockData(_type, _fileId, _pageId, CegoLockHandler::READ);
		}
		catch ( Exception e)
		{
		    _lockId = 0;
		    
		    Chain msg;
		    e.pop(msg);			    
		    _pBufPool->log(_modId, Logger::LOGERR, Chain("Data page lock error for fileId=") 
				   + Chain(_fileId) + Chain(" pageId=") + Chain(_pageId) + Chain(" : ") +  msg);		
		    
		    
		    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
		    
		    throw e;
		}
		
		if ( void* p = _bp.getFirstEntry())
		{
		    
		    dp.setFileId(_fileId);
		    dp.setPageId(_pageId);
		    
		    len = _bp.getEntryLen();
		    dp.setOffset(_bp.getEntryPos());

		    return p;
		}
		else
		{
		    _pLockHandle->unlockData(_type, _lockId);    
		    _lockId = 0;
		    
		    _fileId = _bp.getNextFileId();
		    _pageId = _bp.getNextPageId();
		    
		    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
		}
	    }
	    _isEOC = true;
	    return 0;
	} 
    }
    catch ( Exception e)
    {
	abort();
	throw e;			    
    }
}

void* CegoObjectCursor::getNext(int& len, CegoDataPointer& dp)
{
    
    if ( _isEOC )
	return 0;
    
    try
    {
	
	void* p = _bp.getNextEntry();    
	if ( p )
	{
	    
	    dp.setFileId(_fileId);
	    dp.setPageId(_pageId);
	    
	    len = _bp.getEntryLen();
	    dp.setOffset(_bp.getEntryPos());
	    
	    return p;
	}
	else
	{
	    
	    _pLockHandle->unlockData(_type, _lockId);    
	    _lockId = 0;
	    
	    _fileId = _bp.getNextFileId();
	    _pageId = _bp.getNextPageId();
	    
	    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
	    
	    while ( ! ( _fileId == 0 && _pageId == 0 ) )
	    {
		
		CegoBufferPage nbp;
		_pBufPool->bufferFix(nbp, _tabSetId, _fileId, _pageId, CegoBufferPool::SYNC, _pLockHandle);
		
		_bp = nbp;
		
		try
		{
		    _lockId = _pLockHandle->lockData(_type, _fileId, _pageId, CegoLockHandler::READ);
		}
		catch ( Exception e )
		{
		    _lockId = 0;
		    
		    Chain msg;
		    e.pop(msg);
		    _pBufPool->log(_modId, Logger::LOGERR, Chain("Data page lock error for fileId=") 
				   + Chain(_fileId) + Chain(" pageId=") + Chain(_pageId) + Chain(" : ") +  msg);			    
		    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
		    
		    throw e;
		}
		
		if ( void* p = _bp.getFirstEntry())
		{
		    
		    dp.setFileId(_fileId);
		    dp.setPageId(_pageId);
		    
		    len = _bp.getEntryLen();
		    dp.setOffset(_bp.getEntryPos());

		    return p;
		}
		else
		{
		    _pLockHandle->unlockData(_type, _lockId);    
		    _lockId = 0;
		    
		    _fileId = _bp.getNextFileId();
		    _pageId = _bp.getNextPageId();
			
		    _pBufPool->bufferUnfix(_bp, false, _pLockHandle);
		}
	    }
	    _isEOC = true;	
	    return 0;
	}
    }	
    catch ( Exception e)
    {	    	    
	abort();
	throw e;
    }
}

void CegoObjectCursor::abort()
{
    if (_bp.isFixed())
    {
	_pBufPool->bufferUnfix(_bp, false, _pLockHandle);
    }
    if (_lockId > 0 )
    {
	_pLockHandle->unlockData(_type, _lockId);
	_lockId = 0;
    }
    
    _isEOC = true;
}

void CegoObjectCursor::reset()
{
    abort();
    _isEOC = false;
    _fileId = _startFileId;
    _pageId = _startPageId;
}
