///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoTableCache.cc
// -----------------
// Cego table cache implementation
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoTableCache
// 
// Description: Table Cache Management
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/ThreadLock.h>
#include "CegoTableCache.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseManager.h"

#include <stdlib.h>
#include <string.h>

#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

static ThreadLock cacheLock("TABLECACHE");
extern bool __lockStatOn;

CegoTableCache::TableCacheEntry::TableCacheEntry()
{
    _pCacheArray = 0;
    _numHit=0;
    _numUsed=0;
}

CegoTableCache::TableCacheEntry::TableCacheEntry(int tabSetId, const Chain& tableName)
{
    _tabSetId = tabSetId;
    _tableName = tableName;
    _pCacheArray = 0;
    _numHit=0;
    _numRow=0;
    _numCol=0;
    _numUsed=0;
}

CegoTableCache::TableCacheEntry::TableCacheEntry(int tabSetId, const Chain& tableName, ListT< ListT<CegoFieldValue> >* pCacheList)
{
    _tabSetId = tabSetId;
    _tableName = tableName;
    
    _pCacheArray = new CegoFieldValue**[ pCacheList->Size() ];
    ListT<CegoFieldValue>* pFVL = pCacheList->First();
    _numRow = 0;
    _numCol = 0;
    _numUsed=0;
    while ( pFVL )
    {
	CegoFieldValue** pCFVL = new CegoFieldValue*[pFVL->Size()];
	CegoFieldValue* pFV = pFVL->First();
	int i = 0;
	while ( pFV )
	{
	    CegoFieldValue* pCFV = new CegoFieldValue(pFV->getLocalCopy());
	    pCFVL[i] = pCFV;
	    pFV = pFVL->Next();
	    i++;
	}
	_numCol = i;
       
	_pCacheArray[_numRow] = pCFVL;
	_numRow++;
	pFVL = pCacheList->Next();	
    }
    _numHit=1;
}

CegoTableCache::TableCacheEntry::~TableCacheEntry()
{
}

const int CegoTableCache::TableCacheEntry::getSize() const
{
    int s = sizeof(int); // tabsSetId
    s += _tableName.length();

    for ( int i=0; i<_numRow; i++ )
    {
	for ( int j=0; j<_numCol; j++ )
	{
	    CegoFieldValue *pF = _pCacheArray[i][j];
	    s += pF->size();
	}
    }
    return s;	
}

int CegoTableCache::TableCacheEntry::getNumRows() const
{
    return _numRow;
}

int CegoTableCache::TableCacheEntry::getNumCols() const
{
    return _numCol;
}

const Chain& CegoTableCache::TableCacheEntry::getTableName() const
{
    return _tableName;
}

int CegoTableCache::TableCacheEntry::getTabSetId() const
{
    return _tabSetId;
}

CegoFieldValue*** CegoTableCache::TableCacheEntry::claimCache()
{
    _numUsed++;
    return _pCacheArray;
}

void CegoTableCache::TableCacheEntry::releaseCache()
{
    _numUsed--;
}

unsigned long CegoTableCache::TableCacheEntry::getHit() const
{
    return _numHit;
}

void CegoTableCache::TableCacheEntry::incHit()
{
    _numHit++;
}

bool CegoTableCache::TableCacheEntry::cleanCache()
{
    if ( _numUsed > 0 )
	return false;
    
    for ( int i=0; i<_numRow; i++ )
    {
	for ( int j=0; j<_numCol; j++ )
	    delete _pCacheArray[i][j];
	delete _pCacheArray[i];
    }
    delete _pCacheArray;
    _pCacheArray = 0;
    return true;
}

CegoTableCache::TableCacheEntry& CegoTableCache::TableCacheEntry::operator = ( const CegoTableCache::TableCacheEntry& tce)
{
    _tabSetId = tce._tabSetId;
    _tableName = tce._tableName;
    _pCacheArray = tce._pCacheArray;
    _numHit = tce._numHit;
    _numRow = tce._numRow;
    _numCol = tce._numCol;
    return (*this);
}

bool CegoTableCache::TableCacheEntry::operator == ( const CegoTableCache::TableCacheEntry& tce)
{
    if ( _tableName == tce._tableName && _tabSetId == tce._tabSetId )
	return true;
    return false;
}

CegoTableCache::CegoTableCache(int maxEntry, int maxSize, CegoDatabaseManager* pDBMng)
{
    cacheLock.init(LCKMNG_LOCKWAITDELAY, __lockStatOn);  
    _maxEntry = maxEntry;
    _maxSize = maxSize;
    _usedSize = 0;
    _pDBMng = pDBMng;
}

CegoTableCache::~CegoTableCache()
{
    clean();    
}

void CegoTableCache::PR()
{
    cacheLock.readLock(DBM_LOCKTIMEOUT);
}

void CegoTableCache::PW()
{
    cacheLock.writeLock(DBM_LOCKTIMEOUT);
}

void CegoTableCache::V()
{
    cacheLock.unlock();
}

Element* CegoTableCache::getCacheInfo()
{
    Element* pCacheInfo = new Element(XML_CACHEINFO_ELEMENT);

    Element *pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("MaxEntry"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_maxEntry));
    pCacheInfo->addContent(pN);

    pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("MaxSize"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_maxSize));
    pCacheInfo->addContent(pN);

    pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("UsedSize"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_usedSize));
    pCacheInfo->addContent(pN);
 
    
    return pCacheInfo;
}

Element* CegoTableCache::getCacheList()
{
    Element* pCacheInfo = new Element(XML_CACHEINFO_ELEMENT);
    PR();

    TableCacheEntry *pCE = _tableCache.First();	
    while ( pCE )
    {
	Element *pN = new Element(XML_CACHE_ELEMENT);
	Chain tableSet;
	if ( _pDBMng )
	    tableSet = _pDBMng->getTabSetName(pCE->getTabSetId());
	pN->setAttribute(XML_ID_ATTR, pCE->getTableName() + Chain("@") + tableSet);
	pN->setAttribute(XML_NUMROWS_ATTR, Chain(pCE->getNumRows()));
	pN->setAttribute(XML_NUMHITS_ATTR, Chain(pCE->getHit()));
	pN->setAttribute(XML_SIZE_ATTR, Chain(pCE->getSize()));
	pCacheInfo->addContent(pN);
	pCE = _tableCache.Next();
    }
    V();
    return pCacheInfo;
}

int CegoTableCache::getMaxEntry() const
{
    return _maxEntry;
}

void CegoTableCache::setMaxEntry(int maxEntry)
{
    _maxEntry = maxEntry;
}

int CegoTableCache::getMaxSize() const
{
    return _maxSize;
}

void CegoTableCache::setMaxSize(int maxSize)
{
    _maxSize = maxSize;
}

int CegoTableCache::getUsedSize() const
{
    return _usedSize;
}

const int CegoTableCache::getNumTableCache() const
{
    return _tableCache.Size();
}

void CegoTableCache::invalidate(int tabSetId, const Chain& tableName)
{

    bool isClean=false;
    
    while ( isClean == false )
    {
	isClean = true;
	
	PW();

	TableCacheEntry *pTCE = _tableCache.First();
	while ( pTCE )
	{
	    if ( pTCE->getTabSetId() == tabSetId && pTCE->getTableName() == tableName )
	    {
		int s = pTCE->getSize();
		if  ( pTCE->cleanCache() )
		{
		    _usedSize = _usedSize - s;
		    _tableCache.Remove(*pTCE);
		    pTCE = _tableCache.First();
		}
		else
		{
		    // was not able to cleanup, cache still in use
		    // cout << "Cache still in use .." << endl;
		    isClean=false;
		    // we skip this entry and go to next
		    pTCE = _tableCache.Next();
		}
	    }
	    else
	    {
		pTCE = _tableCache.Next();
	    }
	}
	V();
    }
}

CegoFieldValue*** CegoTableCache::claimEntry(int tabSetId, const Chain& tableName, int& numRow, int& numCol)
{
    CegoFieldValue*** pCE = 0;
    PR();
    TableCacheEntry *pTCE = _tableCache.Find( TableCacheEntry(tabSetId, tableName));

    if ( pTCE )
    {
	pTCE->incHit();
	pCE = pTCE->claimCache();
	numRow = pTCE->getNumRows();
	numCol = pTCE->getNumCols();
    }
    V();
    
    return pCE;
}

void CegoTableCache::releaseEntry(int tabSetId, const Chain& tableName)
{
    PR();
    TableCacheEntry *pTCE = _tableCache.Find( TableCacheEntry(tabSetId, tableName));

    if ( pTCE )
    {
	pTCE->releaseCache();
    }
    V();
}

void CegoTableCache::addEntry(int tabSetId, const Chain& tableName, ListT< ListT<CegoFieldValue> >* pCacheList )
{

    bool isFreed = false;
    
    while ( isFreed == false )
    {
	
	PW();

	// at default, we assume, we don't have to free an entry
	isFreed=true;
       	
	if ( _tableCache.Find(TableCacheEntry(tabSetId, tableName)))
	{
	    // cache entry already exists, we can return 
	    V();
	    return;
	}
	
	if ( _tableCache.Size() > _maxEntry )
	{
	    TableCacheEntry *pRE = 0;
	    unsigned long minHit = 0;
	    TableCacheEntry *pTCE = _tableCache.First();
	    while ( pTCE )
	    {
		if ( minHit == 0 || pTCE->getHit() < minHit )
		{
		    pRE = pTCE;
		    minHit = pTCE->getHit();
		}	    
		pTCE = _tableCache.Next();
	    }


	    if ( pRE )
	    {
		int s = pRE->getSize();
		if ( pRE->cleanCache() )
		{
		    _usedSize = _usedSize - s;
		    _tableCache.Remove(*pRE);
		}
		else
		{
		    // cache entry is still occupied, we have to try again
		    isFreed=false;
		}
	    }
	}

	// if we could free an entry, we can insert new one now
	if ( isFreed )
	{

	    TableCacheEntry tce(tabSetId, tableName, pCacheList);
	    
	    _tableCache.Insert( tce );
	    _usedSize = _usedSize + tce.getSize(); 	    
	}	
	V();
    }    
}

void CegoTableCache::clean()
{

    bool isClean=false;
    
    while ( isClean == false )
    {

	isClean=true;
	
	PW();
       
	TableCacheEntry *pTCE = _tableCache.First();

	while ( pTCE )
	{

	    if  ( pTCE->cleanCache() )
	    {
		_tableCache.Remove(*pTCE);
		pTCE = _tableCache.First();
	    }
	    else
	    {
		// was not able to cleanup, cache still in use
		// cout << "Cache still in use .." << endl;
		isClean=false;
		// we skip this entry and go to next
		pTCE = _tableCache.Next();
	    }
	}
	
	V();
    }

    _usedSize = 0;
}
