///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoXPorter.cc
// -----------------
// Cego export and import 
//                                                         
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoXPorter
// 
// Description: Tableset export methods ( binary and xml )
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// cego includes
#include "CegoXPorter.h"
#include "CegoXMLdef.h"
#include "CegoExpOutStream.h"
#include "CegoImpInStream.h"
#include "CegoTypeConverter.h"
#include "CegoDataType.h"

#include <string.h>
#include <stdlib.h>

#define MAXROWBUF 1024

CegoXPorter::CegoXPorter(CegoDistManager* pGTM, CegoAdminHandler* pAH)
{
    _pGTM = pGTM;
    _pDBMng = pGTM->getDBMng();
    _modId = _pDBMng->getModId("CegoXPorter");
    _pAH = pAH;
}

CegoXPorter::~CegoXPorter()
{  
}

void CegoXPorter::xmlExportTableSet(const Chain& tableSet, bool isStructure, const Chain& expFile)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Exporting tableset ") +  tableSet);
    int tabSetId = _pDBMng->getTabSetId(tableSet);

    XMLSuite xml;

    Element* pRoot = new Element(XML_TABLESET_ELEMENT);    
    Document *pDoc = new Document(XML_EXPORT_DOC);

    pDoc->setDocType(XML_EXPORT_DOC);
    pDoc->setAttribute(XML_VERSION_ATTR, XML_VERSION_VALUE);
    pRoot->setAttribute(XML_NAME_ATTR, tableSet);

    pDoc->setRootElement(pRoot);

    xml.setDocument(pDoc);

    // counter export
    
    ListT<Chain> counterNameList;
    _pDBMng->getCounterList(tabSetId, counterNameList);
    Chain *pCounterName = counterNameList.First();
    while ( pCounterName )
    {
	Chain info = Chain("Exporting counter ") +  *pCounterName + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	unsigned long long v = _pDBMng->getCounterValue(tabSetId, *pCounterName);
	
	Element *pCounterElement = new Element(XML_COUNTER_ELEMENT);
	pCounterElement->setAttribute(XML_NAME_ATTR, *pCounterName);
	pCounterElement->setAttribute(XML_VALUE_ATTR, Chain(v));
	pRoot->addContent(pCounterElement);	
	pCounterName = counterNameList.Next();
    }
    
    // table export
    ListT<Chain> tabList;
    _pGTM->getObjectList(tabSetId, CegoObject::TABLE, tabList);

    Chain *pTab = tabList.First();
    while ( pTab )
    {

	Chain info = Chain("Exporting table ") +  *pTab + Chain(" ...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	Element* pTabElement = new Element(XML_TABLE_ELEMENT);

	pTabElement->setAttribute(XML_NAME_ATTR, *pTab);		
	
	CegoTableObject toe;
	_pGTM->getObject(tabSetId, *pTab, CegoObject::TABLE, toe);

	ListT<CegoField> schema = toe.getSchema();

	Element* pSchemaElement = new Element(XML_SCHEMA_ELEMENT);

	CegoField *pF = schema.First();
	while ( pF ) 
	{
	    
	    Chain tname;
	    if (pF->getTableAlias().length() > 0)
	    {
		tname = pF->getTableAlias();
	    }
	    else
	    {
		tname = pF->getTableName();
	    }
	    
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	    if ( pF->isNullable() )
		pColElement->setAttribute(XML_COLNULLABLE_ATTR, XML_TRUE_VALUE);
	    else
		pColElement->setAttribute(XML_COLNULLABLE_ATTR, XML_FALSE_VALUE);
	
	    if ( ! pF->getValue().isNull() )
		pColElement->setAttribute(XML_COLDEFVALUE_ATTR, pF->getValue().valAsChain(false));
	    
	    CegoTypeConverter tc;
	    pColElement->setAttribute( XML_COLTYPE_ATTR, tc.getTypeString( pF->getType() ));
	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));
	    
	    pSchemaElement->addContent(pColElement);
	    pF = schema.Next();
	}
	
	pTabElement->addContent(pSchemaElement);

	if ( isStructure == false )
	{
	    CegoExpOutStream *pOutStream = new CegoExpOutStream(tabSetId, *pTab, schema, _pGTM, _pAH);
	    pTabElement->setOutStream(pOutStream);
	}
	pRoot->addContent(pTabElement);

	pTab = tabList.Next();
    }

    // index export

    ListT<Chain> idxList;
    _pGTM->getObjectList(tabSetId, CegoObject::AVLTREE, idxList);
    
    Chain *pIdx = idxList.First();
    while ( pIdx )
    {

	Chain info = Chain("Exporting index ") +  *pIdx + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	CegoTableObject ioe;
	_pGTM->getObject(tabSetId, *pIdx, CegoObject::AVLTREE, ioe);

	Element *pIdxElement = new Element(XML_INDEX_ELEMENT);
	pIdxElement->setAttribute(XML_NAME_ATTR, *pIdx);
	pIdxElement->setAttribute(XML_TABLENAME_ATTR, ioe.getTabName());

	if ( ioe.getType() == CegoObject::AVLTREE)	
	    pIdxElement->setAttribute(XML_INDEXTYPE_ATTR, XML_INDEX_VALUE);
	if ( ioe.getType() == CegoObject::PAVLTREE)
	    pIdxElement->setAttribute(XML_INDEXTYPE_ATTR, XML_PINDEX_VALUE);
	if ( ioe.getType() == CegoObject::UAVLTREE)
	    pIdxElement->setAttribute(XML_INDEXTYPE_ATTR, XML_UINDEX_VALUE);


	Element* pSchemaElement = new Element(XML_SCHEMA_ELEMENT);

	CegoField *pF = ioe.getSchema().First();
	while ( pF ) 
	{
	    
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	
	    CegoTypeConverter tc;
	    pColElement->setAttribute( XML_COLTYPE_ATTR, tc.getTypeString( pF->getType() ));
	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));
	    
	    pSchemaElement->addContent(pColElement);

	    pF = ioe.getSchema().Next();
	}
	
	pIdxElement->addContent( pSchemaElement );       

	pRoot->addContent(pIdxElement);

	pIdx = idxList.Next();
    }

    // btree export

    ListT<Chain> btreeList;
    _pGTM->getObjectList(tabSetId, CegoObject::BTREE, btreeList);
    
    Chain *pBT = btreeList.First();
    while ( pBT )
    {

	Chain info = Chain("Exporting btree ") +  *pBT + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	CegoBTreeObject btoe;
	_pGTM->getObject(tabSetId, *pBT, CegoObject::BTREE, btoe);

	Element *pBTElement = new Element(XML_INDEX_ELEMENT);
	pBTElement->setAttribute(XML_NAME_ATTR, *pBT);
	pBTElement->setAttribute(XML_TABLENAME_ATTR, btoe.getTabName());

	if ( btoe.getType() == CegoObject::BTREE)	
	    pBTElement->setAttribute(XML_INDEXTYPE_ATTR, XML_BTREE_VALUE);
	if ( btoe.getType() == CegoObject::PBTREE)
	    pBTElement->setAttribute(XML_INDEXTYPE_ATTR, XML_PBTREE_VALUE);
	if ( btoe.getType() == CegoObject::UBTREE)
	    pBTElement->setAttribute(XML_INDEXTYPE_ATTR, XML_UBTREE_VALUE);


	Element* pSchemaElement = new Element(XML_SCHEMA_ELEMENT);

	CegoField *pF = btoe.getSchema().First();
	while ( pF ) 
	{
	    
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	
	    CegoTypeConverter tc;
	    pColElement->setAttribute( XML_COLTYPE_ATTR, tc.getTypeString( pF->getType() ));
	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));
	    
	    pSchemaElement->addContent(pColElement);

	    pF = btoe.getSchema().Next();
	}
	
	pBTElement->addContent( pSchemaElement );       

	pRoot->addContent(pBTElement);

	pBT = btreeList.Next();
    }

    // key export

    ListT<Chain> fkeyList;
    _pGTM->getObjectList(tabSetId, CegoObject::FKEY, fkeyList);
    
    Chain *pFKey = fkeyList.First();
    while ( pFKey )
    {

	Chain info = Chain("Exporting foreign key ") +  *pFKey + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info+ Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	CegoKeyObject koe;
	_pGTM->getObject(tabSetId, *pFKey, CegoObject::FKEY, koe);

	Element *pFKeyElement = new Element(XML_FKEY_ELEMENT);
	pFKeyElement->setAttribute(XML_NAME_ATTR, *pFKey);
	pFKeyElement->setAttribute(XML_TABLENAME_ATTR, koe.getTabName());
	pFKeyElement->setAttribute(XML_REFTABLENAME_ATTR, koe.getRefTable());


	Element* pKeySchemaElement = new Element(XML_KEYSCHEMA_ELEMENT);

	CegoField *pF = koe.getKeySchema().First();
	while ( pF )
	{

	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());	    
	    pKeySchemaElement->addContent(pColElement);
	    pF = koe.getKeySchema().Next();
	}

	Element* pRefSchemaElement = new Element(XML_REFSCHEMA_ELEMENT);

	pF = koe.getRefSchema().First();
	while ( pF )
	{

	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());	    
	    pRefSchemaElement->addContent(pColElement);
	    pF = koe.getRefSchema().Next();
	}
	
	pFKeyElement->addContent(pKeySchemaElement);
	pFKeyElement->addContent(pRefSchemaElement);
	

	pRoot->addContent(pFKeyElement);	
	pFKey = fkeyList.Next();
    }
    
    // check export

    ListT<Chain> checkList;
    _pGTM->getObjectList(tabSetId, CegoObject::CHECK, checkList);
    
    Chain *pCheck = checkList.First();
    while ( pCheck )
    {

	Chain info = Chain("Exporting check ") +  *pCheck + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	CegoCheckObject coe;
	_pGTM->getObject(tabSetId, *pCheck, CegoObject::CHECK, coe);

	Element *pCheckElement = new Element(XML_CHECK_ELEMENT);
	pCheckElement->setAttribute(XML_NAME_ATTR, *pCheck);
	pCheckElement->setAttribute(XML_TABLENAME_ATTR, coe.getTabName());

	Element* pCheckPredElement = coe.getPredDesc()->toElement();

	pCheckElement->addContent(pCheckPredElement);
	
	pRoot->addContent(pCheckElement);	
	pCheck = checkList.Next();
    }
    
    // view export

    ListT<Chain> viewList;
    _pGTM->getObjectList(tabSetId, CegoObject::VIEW, viewList);
    
    Chain *pView = viewList.First();
    while ( pView )
    {

	Chain info = Chain("Exporting view ") +  *pView + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info+ Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	// to ensure, the view is compiled, we load it in any case

	_pDBMng->useObject(tabSetId, *pView, CegoObject::VIEW, CegoDatabaseManager::SHARED, _pGTM->getThreadId());
	try
	{
	    _pGTM->getView(tabSetId, *pView);
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(tabSetId, *pView, CegoObject::VIEW);
	    throw e;
	}

	_pDBMng->unuseObject(tabSetId, *pView, CegoObject::VIEW);
	
	CegoViewObject vo;
	_pGTM->getObject(tabSetId, *pView, CegoObject::VIEW,  vo);

	Element *pViewElement = new Element(XML_VIEW_ELEMENT);
	pViewElement->setAttribute(XML_NAME_ATTR, *pView);
	pViewElement->setAttribute(XML_VIEWSTMT_ATTR, vo.getViewStmt());

	Element* pViewSchemaElement = new Element(XML_SCHEMA_ELEMENT);

	CegoField *pF = vo.getSchema().First();
	while ( pF )
	{

	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());	    
	
	    CegoTypeConverter tc;
	    pColElement->setAttribute( XML_COLTYPE_ATTR, tc.getTypeString( pF->getType() ));

	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));

	    pViewSchemaElement->addContent(pColElement);
	    pF = vo.getSchema().Next();
	}
	
	pViewElement->addContent(pViewSchemaElement);

	pRoot->addContent(pViewElement);	
	pView = viewList.Next();
    }

    // proc export

    ListT<Chain> procList;
    _pGTM->getObjectList(tabSetId, CegoObject::PROCEDURE, procList);
    
    Chain *pProc = procList.First();
    while ( pProc )
    {

	Chain info = Chain("Exporting procedure ") +  *pProc + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	// to ensure, the view is compiled, we load it in any case
	_pDBMng->useObject(tabSetId, *pProc, CegoObject::PROCEDURE, CegoDatabaseManager::SHARED, _pGTM->getThreadId());
	try
	{
	    _pGTM->getProcedure(tabSetId, *pProc);
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(tabSetId, *pProc, CegoObject::PROCEDURE);
	    throw e;
	}       
	_pDBMng->unuseObject(tabSetId, *pProc, CegoObject::PROCEDURE);

	CegoProcObject po;
	_pGTM->getObject(tabSetId, *pProc, CegoObject::PROCEDURE, po);

	Element *pProcElement = new Element(XML_PROCEDURE_ELEMENT);
	pProcElement->setAttribute(XML_NAME_ATTR, *pProc);
	pProcElement->setAttribute(XML_PROCTEXT_ATTR, po.getProcText());
	
	pRoot->addContent(pProcElement);	
	pProc = procList.Next();

    }

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Writing export to ") +  expFile + Chain("..."));

    File *pOutFile = new File(expFile);
    
    try
    {	
	pOutFile->open(File::WRITE);
	xml.getXMLChain( pOutFile );    
    }
    catch ( Exception e )
    {
	pOutFile->close();
	delete pOutFile;
	delete pDoc;
	throw e;
    }

    pOutFile->close();
    delete pOutFile;
    delete pDoc;
    
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Export finished succesful"));    

}

void CegoXPorter::xmlImportTableSet(const Chain& tableSet, bool isStructure, const Chain& impFile, bool doLogging)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Importing tableset ") + tableSet + Chain("..."));

    _pGTM->setAppend(true);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    XMLSuite xml;
    
    File *pInFile = new File(impFile);
    CegoImpInStream* pInStream = 0;
    Document *pDoc = new Document(XML_IMPORT_DOC);
	
    try
    {
	
	pInFile->open(File::READ);
	
	if ( isStructure == false )
	{
	    pInStream = new CegoImpInStream(tableSet, _pGTM, doLogging, _pAH);
	}

	pDoc->setAttribute(XML_VERSION_ATTR, XML_VERSION_VALUE);
	
	xml.setDocument(pDoc);    
	xml.setFile(pInFile);
	xml.setInStream(pInStream);
	
	xml.parse();
	
	if ( pInStream )
	{
	    delete pInStream;
	}
	pInFile->close();
	delete pInFile;
	
    }
    catch ( Exception e )
    {	
	if ( pInFile )
	{
	    pInFile->close();
	    delete pInFile;
	}
	if ( pInStream )
	    delete pInStream;
	if ( pDoc )
	    delete pDoc;

	throw e;
    }
    
	
    Element *pRoot = pDoc->getRootElement();

    // index import

    ListT<Element*> idxList = pRoot->getChildren(XML_INDEX_ELEMENT);
    Element **pIdxElement = idxList.First();
    while ( pIdxElement )
    {
	Chain idxName = (*pIdxElement)->getAttributeValue(XML_NAME_ATTR);
	Chain tabName = (*pIdxElement)->getAttributeValue(XML_TABLENAME_ATTR);	
	
	CegoObject::ObjectType idxType;

	if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_INDEX_VALUE) )
	    idxType = CegoObject::AVLTREE;
	else if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_PINDEX_VALUE) )
	    idxType = CegoObject::PAVLTREE;
	else if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_UINDEX_VALUE) )
	    idxType = CegoObject::UAVLTREE;
	else if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_BTREE_VALUE) )
	    idxType = CegoObject::BTREE;
	else if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_PBTREE_VALUE) )
	    idxType = CegoObject::PBTREE;
	else if ( (*pIdxElement)->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_UBTREE_VALUE) )
	    idxType = CegoObject::UBTREE;
	else
	    throw Exception(EXLOC, Chain("Unknown index type"));

	Chain info;
	if ( idxType == CegoObject::AVLTREE 
	     || idxType == CegoObject::PAVLTREE 
	     || idxType == CegoObject::UAVLTREE ) 
	    info = Chain("Importing index ") + idxName + Chain("...");
	else
	    info = Chain("Importing btree ") + idxName + Chain("...");
	
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);
	
	ListT<Element*> schemaList = (*pIdxElement)->getChildren(XML_SCHEMA_ELEMENT);

	Element **pSchemaElement = schemaList.First();

	if ( pSchemaElement )
	{
	    ListT<Element*> colList = (*pSchemaElement)->getChildren(XML_COL_ELEMENT);
	    ListT<CegoField> fl;
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
		Chain colSize = (*pCol)->getAttributeValue(XML_COLSIZE_ATTR);
		Chain colNullable = (*pCol)->getAttributeValue(XML_COLNULLABLE_ATTR);
		bool isNullable;
		if ( colNullable == Chain(XML_TRUE_VALUE) ) 
		    isNullable = true;
		else
		    isNullable = false;

		CegoTypeConverter tc;
		CegoDataType type = tc.getTypeId( colType );

		Chain colDefValue = (*pCol)->getAttributeValue(XML_COLDEFVALUE_ATTR);
		CegoFieldValue defValue;

		if ( colDefValue != Chain("") )		    
		    defValue = CegoFieldValue(type, colDefValue);
		
		CegoField f(CegoField(tabName, tabName, colName, type, colSize.asInteger(), defValue, isNullable));		
		fl.Insert(f);

		pCol = colList.Next();
	    }
	    
	    _pGTM->createDistIndexTable(tableSet, idxName, tabName, fl, idxType);
	}
	
	pIdxElement = idxList.Next();
    }

    // key import

    ListT<Element*> fkeyList = pRoot->getChildren(XML_FKEY_ELEMENT);
    Element **pFKeyElement = fkeyList.First();
    while ( pFKeyElement )
    {
	Chain fkey = (*pFKeyElement)->getAttributeValue(XML_NAME_ATTR);
	Chain tableName = (*pFKeyElement)->getAttributeValue(XML_TABLENAME_ATTR);	
	Chain refTable = (*pFKeyElement)->getAttributeValue(XML_REFTABLENAME_ATTR);	

	Chain info = Chain("Importing foreign key ") + fkey + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	ListT<Element*> keySchemaList = (*pFKeyElement)->getChildren(XML_KEYSCHEMA_ELEMENT);

	Element **pKeySchemaElement = keySchemaList.First();

	ListT<CegoField> keyList;
	if ( pKeySchemaElement )
	{
	    ListT<Element*> colList = (*pKeySchemaElement)->getChildren(XML_COL_ELEMENT);
	    ListT<CegoField> fl;
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);		
		keyList.Insert(CegoField(Chain(), colName));		
		pCol = colList.Next();
	    }
	}

	ListT<Element*> refSchemaList = (*pFKeyElement)->getChildren(XML_REFSCHEMA_ELEMENT);

	Element **pRefSchemaElement = refSchemaList.First();

	ListT<CegoField> refList;
	if ( pRefSchemaElement )
	{
	    ListT<Element*> colList = (*pKeySchemaElement)->getChildren(XML_COL_ELEMENT);
	    ListT<CegoField> fl;
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);		
		refList.Insert(CegoField(Chain(), colName));		
		pCol = colList.Next();
	    }
	}
	
	_pGTM->createDistForeignKey(tableSet, fkey, tableName, keyList, refTable, refList);
	
	pFKeyElement = fkeyList.Next();
    }

    // check import

    ListT<Element*> checkList = pRoot->getChildren(XML_CHECK_ELEMENT);
    Element **pCheckElement = checkList.First();
    while ( pCheckElement )
    {
	Chain checkName = (*pCheckElement)->getAttributeValue(XML_NAME_ATTR);
	Chain tableName = (*pCheckElement)->getAttributeValue(XML_TABLENAME_ATTR);	


	Chain info = Chain("Importing check ") + checkName + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	ListT<Element*> checkPredList = (*pCheckElement)->getChildren(XML_PRED_ELEMENT);

	Element **pCheckPredElement = checkPredList.First();

	CegoPredDesc *pPredDesc = new CegoPredDesc(*pCheckPredElement, _pGTM);

	_pGTM->createDistCheck(tableSet, checkName, tableName, pPredDesc);

	pCheckElement = checkList.Next();
    }

    // view import

    ListT<Element*> viewList = pRoot->getChildren(XML_VIEW_ELEMENT);
    Element **pViewElement = viewList.First();
    while ( pViewElement )
    {
	Chain viewName = (*pViewElement)->getAttributeValue(XML_NAME_ATTR);
	Chain viewStmt = (*pViewElement)->getAttributeValue(XML_VIEWSTMT_ATTR);	


	Chain info = Chain("Importing view ") + viewName + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	ListT<Element*> viewSchemaList = (*pViewElement)->getChildren(XML_SCHEMA_ELEMENT);

	Element **pViewSchemaElement = viewSchemaList.First();

	ListT<CegoField> fl;
	if ( pViewSchemaElement )
	{
	    ListT<Element*> colList = (*pViewSchemaElement)->getChildren(XML_COL_ELEMENT);
	    ListT<CegoField> fl;
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
		Chain colSize = (*pCol)->getAttributeValue(XML_COLSIZE_ATTR);

		CegoTypeConverter tc;
		CegoDataType type = tc.getTypeId( colType );
		
		CegoField f(CegoField(viewName, viewName, colName, type, colSize.asInteger()));		
		fl.Insert(f);

		pCol = colList.Next();
	    }

	    _pGTM->createDistView(tableSet, viewName, fl, viewStmt);


	}

	pViewElement = viewList.Next();
    }

    ListT<Element*> procList = pRoot->getChildren(XML_PROCEDURE_ELEMENT);
    Element **pProcElement = procList.First();
    while ( pProcElement )
    {
	Chain procName = (*pProcElement)->getAttributeValue(XML_NAME_ATTR);
	Chain procText = (*pProcElement)->getAttributeValue(XML_PROCTEXT_ATTR);	

	_pDBMng->log(_modId, Logger::NOTICE, Chain("Importing procedure ") + procName + Chain("..."));

	_pGTM->createDistProc(tableSet, procName, procText);


	pProcElement = procList.Next();
    }

    // counter import

    ListT<Element*> counterList = pRoot->getChildren(XML_COUNTER_ELEMENT);
    Element **pCounterElement = counterList.First();
    while ( pCounterElement )
    {
	Chain counterName = (*pCounterElement)->getAttributeValue(XML_NAME_ATTR);
	Chain counterValue = (*pCounterElement)->getAttributeValue(XML_VALUE_ATTR);

	Chain info = Chain("Importing counter ") + counterName + Chain("...");
	if ( _pAH )
	    _pAH->sendInfo(info + Chain("\n"));
	_pDBMng->log(_modId, Logger::NOTICE, info);

	int tabSetId = _pDBMng->getTabSetId(tableSet);	
	_pDBMng->addCounter(tabSetId, counterName, counterValue.asUnsignedLongLong(), true);
	
	pCounterElement = counterList.Next();
    }
    
    delete pDoc;
}

void CegoXPorter::xmlExportTable(const Chain& tableSet, const Chain& tableName, const Chain& expFile)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Exporting table ") + tableName + Chain("..."));

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    XMLSuite xml;

    Element* pRoot = new Element(XML_TABLESET_ELEMENT);    
    Document *pDoc = new Document(XML_EXPORT_DOC);

    pDoc->setDocType(XML_EXPORT_DOC);
    pDoc->setAttribute(XML_VERSION_ATTR, XML_VERSION_VALUE);

    pRoot->setAttribute(XML_NAME_ATTR, tableSet);

    pDoc->setRootElement(pRoot);

    xml.setDocument(pDoc);
    
    Element* pTabElement = new Element(XML_TABLE_ELEMENT);
    
    pTabElement->setAttribute(XML_NAME_ATTR, tableName);		
    
    CegoTableObject toe;
    _pGTM->getObject(tabSetId, tableName, CegoObject::TABLE, toe);
    
    ListT<CegoField> schema = toe.getSchema();
    
    Element* pSchemaElement = new Element(XML_SCHEMA_ELEMENT);

    CegoField *pF = schema.First();
    while ( pF ) 
    {
	
	Chain tname;
	if (pF->getTableAlias().length() > 0)
	{
	    tname = pF->getTableAlias();
	}
	else
	{
	    tname = pF->getTableName();
	}
	
	Element *pColElement = new Element(XML_COL_ELEMENT);
	pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	if ( pF->isNullable() )
	    pColElement->setAttribute(XML_COLNULLABLE_ATTR, XML_TRUE_VALUE);
	else
	    pColElement->setAttribute(XML_COLNULLABLE_ATTR, XML_FALSE_VALUE);

	if ( ! pF->getValue().isNull() )
	    pColElement->setAttribute(XML_COLDEFVALUE_ATTR, pF->getValue().valAsChain(false));
	
	CegoTypeConverter tc;
	pColElement->setAttribute( XML_COLTYPE_ATTR, tc.getTypeString( pF->getType() ));
	pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));

	pSchemaElement->addContent(pColElement);
	pF = schema.Next();
    }
    
    pTabElement->addContent(pSchemaElement);
    
    CegoExpOutStream *pOutStream = new CegoExpOutStream(tabSetId, tableName, schema, _pGTM, _pAH);
    pTabElement->setOutStream(pOutStream);
    
    pRoot->addContent(pTabElement);
    
    File *pOutFile = new File(expFile);

    try
    {
	pOutFile->open(File::WRITE);
	xml.getXMLChain( pOutFile );
    }
    catch ( Exception e )
    {
	pOutFile->close();
	delete pOutFile;
	delete pOutStream;
	delete pDoc;
	throw e;
    }

    pOutFile->close();
    delete pOutFile;
    delete pOutStream;
    delete pDoc;
}

void CegoXPorter::xmlImportTable(const Chain& tableSet, const Chain& tableName, const Chain& impFile, bool doLogging)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Importing table ") + tableName + Chain("..."));

    _pGTM->setAppend(true);


    File *pInFile = 0;
    CegoImpInStream* pInStream = 0;
    Document *pDoc = 0;

    try
    {
	XMLSuite xml;

	pInFile = new File(impFile);
	pInFile->open(File::READ);
    
	pInStream = new CegoImpInStream(tableSet, tableName, _pGTM, doLogging, _pAH);
	
	pDoc = new Document(XML_IMPORT_DOC);
	pDoc->setAttribute(XML_VERSION_ATTR, XML_VERSION_VALUE);
	
	xml.setDocument(pDoc);    
	xml.setFile(pInFile);
	xml.setInStream(pInStream);
	
	xml.parse();
	
    }
    catch ( Exception e )
    {	
	if ( pInFile )
	{
	    pInFile->close();
	    delete pInFile;
	}
	if ( pInStream )
	    delete pInStream;
	if ( pDoc )
	    delete pDoc;

	throw e;
    }
    
    pInFile->close();
    delete pInFile;
    delete pInStream;
    delete pDoc;
    
}

void CegoXPorter::binExportTableSet(const Chain& tableSet, bool isStructure, const Chain& expFile, bool doPlain)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Exporting tableset ") + tableSet + Chain(" ( binary ) ..."));

    File *pOutFile = 0;

    try
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	
	pOutFile = new File(expFile);
	pOutFile->open(File::WRITE);

	writeHeader(pOutFile, tableSet);
         
	// table export
	ListT<Chain> tabList;
	_pGTM->getObjectList(tabSetId, CegoObject::TABLE, tabList);
	
	Chain *pTab = tabList.First();
	while ( pTab )
	{

	    Chain info = Chain("Exporting table ") + *pTab + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    	    
	    CegoTableObject toe;
	    _pGTM->getObject(tabSetId, *pTab, CegoObject::TABLE, toe);
	    
	    writeTableObject(pOutFile, tabSetId, *pTab, toe.getSchema(), doPlain);
	    
	    pTab = tabList.Next();
	}
	
	// index export
	
	ListT<Chain> idxList;
	_pGTM->getObjectList(tabSetId, CegoObject::AVLTREE, idxList);
	
	Chain *pIdx = idxList.First();
	while ( pIdx )
	{

	    Chain info = Chain("Exporting index ") + *pIdx + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    	    
	    CegoTableObject ioe;
	    _pGTM->getObject(tabSetId, *pIdx, CegoObject::AVLTREE, ioe);
	    
	    writeIndexObject(pOutFile, tabSetId, *pIdx, ioe.getTabName(), ioe.getType(), ioe.getSchema());
	    
	    pIdx = idxList.Next();
	}

	// btree export
	
	ListT<Chain> btreeList;
	_pGTM->getObjectList(tabSetId, CegoObject::BTREE, btreeList);
	
	Chain *pBT = btreeList.First();
	while ( pBT )
	{

	    Chain info = Chain("Exporting btree ") + *pBT + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    	    
	    CegoBTreeObject btoe;
	    _pGTM->getObject(tabSetId, *pBT, CegoObject::BTREE, btoe);
	    
	    writeIndexObject(pOutFile, tabSetId, *pBT, btoe.getTabName(), btoe.getType(), btoe.getSchema());
	    
	    pBT = btreeList.Next();
	}
	
	// key export
	
	ListT<Chain> fkeyList;
	_pGTM->getObjectList(tabSetId, CegoObject::FKEY, fkeyList);
	
	Chain *pFKey = fkeyList.First();
	while ( pFKey )
	{

	    Chain info = Chain("Exporting foreign key ") + *pFKey + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    
	    CegoKeyObject koe;
	    _pGTM->getObject(tabSetId, *pFKey, CegoObject::FKEY, koe);
	    
	    writeKeyObject(pOutFile, tabSetId, *pFKey, koe.getTabName(), koe.getRefTable(), koe.getKeySchema(), koe.getRefSchema());
	    
	    pFKey = fkeyList.Next();
	}
	
	// check export
	
	ListT<Chain> checkList;
	_pGTM->getObjectList(tabSetId, CegoObject::CHECK, checkList);
	
	Chain *pCheck = checkList.First();
	while ( pCheck )
	{
	    
	    _pDBMng->log(_modId, Logger::NOTICE, Chain("Exporting check ") + *pCheck + Chain("..."));
	    
	    CegoCheckObject coe;
	    _pGTM->getObject(tabSetId, *pCheck, CegoObject::CHECK, coe);
	    
	    writeCheckObject(pOutFile, tabSetId, *pCheck, coe.getTabName(), coe.getPredDesc());
	    
	    pCheck = checkList.Next();
	}
	
	
	// view export
	
	ListT<Chain> viewList;
	_pGTM->getObjectList(tabSetId, CegoObject::VIEW, viewList);
	
	Chain *pView = viewList.First();
	while ( pView )
	{


	    Chain info = Chain("Exporting view ") + *pView + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    
	    // to ensure, the view is compiled, we load it in any case
	    _pGTM->getView(tabSetId, *pView);
	    
	    CegoViewObject vo;
	    _pGTM->getObject(tabSetId, *pView, CegoObject::VIEW,  vo);
	    
	    writeViewObject(pOutFile, tabSetId, *pView, vo.getSchema(), vo.getViewStmt());
	    
	    pView = viewList.Next();
	}
	
	// proc export
	
	ListT<Chain> procList;
	_pGTM->getObjectList(tabSetId, CegoObject::PROCEDURE, procList);
	
	Chain *pProc = procList.First();
	while ( pProc )
	{
	    Chain info = Chain("Exporting procedure ") + *pProc + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);

	    // to ensure, the view is compiled, we load it in any case
	    _pGTM->getProcedure(tabSetId, *pProc);

	    CegoProcObject po;
	    _pGTM->getObject(tabSetId, *pProc, CegoObject::PROCEDURE, po);
	    
	    writeProcObject(pOutFile, tabSetId, *pProc, po.getProcText());
	    
	    pProc = procList.Next();
	    
	}
	
	// counter export
	
	ListT<Chain> counterNameList;
	_pDBMng->getCounterList(tabSetId, counterNameList);
	Chain *pCounterName = counterNameList.First();
	while ( pCounterName )
	{

	    Chain info = Chain("Exporting counter ") + *pCounterName + Chain("...");
	    if ( _pAH )
		_pAH->sendInfo(info + Chain("\n"));
	    _pDBMng->log(_modId, Logger::NOTICE, info);
	    
	    
	    unsigned long long v = _pDBMng->getCounterValue(tabSetId, *pCounterName);	
	    writeCounterObject(pOutFile, tabSetId, *pCounterName, v);
	    pCounterName = counterNameList.Next();
	}
	
	_tag = EOFTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
	
    }
    catch ( Exception e)
    {
	if ( pOutFile )
	{
	    pOutFile->close();
	    delete pOutFile;
	}
	throw e;
    }
    
    pOutFile->close();
    delete pOutFile;
    
}

void CegoXPorter::writeHeader(File *pOutFile, const Chain& tableSet)
{
    int tslen = tableSet.length()-1;
    _tag = TSTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    pOutFile->writeByte((char*)&tslen, sizeof(int));
    pOutFile->writeByte((char*)tableSet, tslen);    
}

void CegoXPorter::writeTableObject(File *pOutFile, int tabSetId, const Chain& tableName, const ListT<CegoField>& schema, bool doPlain)
{

    CegoField *pF;
    if ( doPlain )
    {
	// checking for blobs 
	pF = schema.First();
	while ( pF ) 
	{
	    CegoDataType dt = pF->getType();
	    if ( dt == BLOB_TYPE )
		throw Exception(EXLOC, "Binary export not supported for blob tables");
	    pF = schema.Next();
	}
    }
    
    _tag = TABTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
        
    int tlen = tableName.length()-1;
    pOutFile->writeByte((char*)&tlen, sizeof(int));
    pOutFile->writeByte((char*)tableName, tlen);
    
    pF = schema.First();
    while ( pF ) 
    {

	_tag = FLDTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));

	int id = pF->getId();
	int alen = pF->getAttrName().length()-1;

	pOutFile->writeByte((char*)&id, sizeof(int));
	pOutFile->writeByte((char*)&alen, sizeof(int));
	pOutFile->writeByte((char*)pF->getAttrName(), alen);

	char isNullable;
	if ( pF->isNullable() )     
	    isNullable=1;
	else
	    isNullable=0;
	
	pOutFile->writeByte((char*)&isNullable, sizeof(char));		

	CegoDataType dt = pF->getType();
	pOutFile->writeByte((char*)&dt, sizeof(CegoDataType));		

	int len = pF->getLength();
	pOutFile->writeByte((char*)&len, sizeof(int));

	int defValLen = pF->getValue().getEncodingLength();
	    
	if ( defValLen > XP_MAXCOLBUF )
	    throw Exception(EXLOC, "Col buffer exceeded");

	
	pF->getValue().encode(_colBuffer);
	    
	pOutFile->writeByte((char*)&defValLen, sizeof(int));
	pOutFile->writeByte((char*)_colBuffer, defValLen);

	pF = schema.Next();
    }
    
    writeTableData(pOutFile, tabSetId, tableName, schema, doPlain);
}

void CegoXPorter::writeIndexObject(File *pOutFile, int tabSetId, const Chain& idxName, const Chain& tableName, CegoObject::ObjectType type, const ListT<CegoField>& schema)
{
    
    _tag = IDXTAG;

    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
        
    int ilen = idxName.length()-1;
    pOutFile->writeByte((char*)&ilen, sizeof(int));
    pOutFile->writeByte((char*)idxName, ilen);

    int tlen = tableName.length()-1;
    pOutFile->writeByte((char*)&tlen, sizeof(int));
    pOutFile->writeByte((char*)tableName, tlen);

    pOutFile->writeByte((char*)&type, sizeof(CegoObject::ObjectType));

    CegoField *pF = schema.First();
    while ( pF ) 
    {

	_tag = FLDTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));

	int alen = pF->getAttrName().length()-1;
	pOutFile->writeByte((char*)&alen, sizeof(int));
	pOutFile->writeByte((char*)pF->getAttrName(), alen);

	CegoDataType dt = pF->getType();
	pOutFile->writeByte((char*)&dt, sizeof(CegoDataType));		

	int len = pF->getLength();
	pOutFile->writeByte((char*)&len, sizeof(int));		
	
	pF = schema.Next();
    }
}

void CegoXPorter::writeKeyObject(File *pOutFile, int tabSetId, const Chain& keyName, const Chain& tableName, const Chain& refTable, 
				 const ListT<CegoField>& keySchema, const ListT<CegoField>& refSchema)
{
    _tag = KEYTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    
    int klen = keyName.length()-1;
    pOutFile->writeByte((char*)&klen, sizeof(int));
    pOutFile->writeByte((char*)keyName, klen);

    int tlen = tableName.length()-1;
    pOutFile->writeByte((char*)&tlen, sizeof(int));
    pOutFile->writeByte((char*)tableName, tlen);

    int rtlen = refTable.length()-1;
    pOutFile->writeByte((char*)&rtlen, sizeof(int));
    pOutFile->writeByte((char*)refTable, rtlen);
    

    CegoField *pF = keySchema.First();
    while ( pF ) 
    {
	_tag = FLDTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
	
	int alen = pF->getAttrName().length()-1;
	pOutFile->writeByte((char*)&alen, sizeof(int));
	pOutFile->writeByte((char*)pF->getAttrName(), alen);
	
	pF = keySchema.Next();
    }

    _tag = SEPTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    
    pF = refSchema.First();
    while ( pF )
    {

	_tag = FLDTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
	
	int alen = pF->getAttrName().length()-1;
	pOutFile->writeByte((char*)&alen, sizeof(int));
	pOutFile->writeByte((char*)pF->getAttrName(), alen);
	
	pF = refSchema.Next();	
    }
}

void CegoXPorter::writeCheckObject(File *pOutFile, int tabSetId, const Chain& checkName, const Chain& tableName, 
				   CegoPredDesc *pPredDesc)
{
    _tag = CHECKTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    
    int cklen = checkName.length()-1;
    pOutFile->writeByte((char*)&cklen, sizeof(int));
    pOutFile->writeByte((char*)checkName, cklen);

    int tlen = tableName.length()-1;
    pOutFile->writeByte((char*)&tlen, sizeof(int));
    pOutFile->writeByte((char*)tableName, tlen);

    int pdlen = pPredDesc->getEncodingLength();

    pOutFile->writeByte((char*)&pdlen, sizeof(int));
    
    if ( pdlen < XP_MAXPDBUF )
    {
	pPredDesc->encode(_pdBuffer);
	pOutFile->writeByte(_pdBuffer, pdlen);
    }
    else
    {
	throw Exception(EXLOC, "PredDesc buffer exceeded");
    }
}

void CegoXPorter::writeViewObject(File *pOutFile, int tabSetId, const Chain& viewName, const ListT<CegoField>& viewSchema, const Chain& viewStmt)
{

    _tag = VIEWTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
        
    int vlen = viewName.length()-1;
    pOutFile->writeByte((char*)&vlen, sizeof(int));
    pOutFile->writeByte((char*)viewName, vlen);

    int slen = viewStmt.length()-1;
    pOutFile->writeByte((char*)&slen, sizeof(int));
    pOutFile->writeByte((char*)viewStmt, slen);

    CegoField *pF = viewSchema.First();
    while ( pF )
    {

	_tag = FLDTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
	
	int alen = pF->getAttrName().length()-1;
	pOutFile->writeByte((char*)&alen, sizeof(int));
	pOutFile->writeByte((char*)pF->getAttrName(), alen);


	CegoDataType dt = pF->getType();
	pOutFile->writeByte((char*)&dt, sizeof(CegoDataType));		

	int len = pF->getLength();
	pOutFile->writeByte((char*)&len, sizeof(int));		
	
	pF = viewSchema.Next();
    }        
}

void CegoXPorter::writeProcObject(File *pOutFile, int tabSetId, const Chain& procName, const Chain& procText)
{
    _tag = PROCTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    
    int plen = procName.length()-1;
    pOutFile->writeByte((char*)&plen, sizeof(int));
    pOutFile->writeByte((char*)procName, plen);

    int txtlen = procText.length()-1;
    pOutFile->writeByte((char*)&txtlen, sizeof(int));
    pOutFile->writeByte((char*)procText, txtlen);
}

void CegoXPorter::writeCounterObject(File *pOutFile, int tabSetId, const Chain& counterName, unsigned long long counterValue)
{
    _tag = CNTTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
    
    int clen = counterName.length()-1;
    pOutFile->writeByte((char*)&clen, sizeof(int));
    pOutFile->writeByte((char*)counterName, clen);
    pOutFile->writeByte((char*)&counterValue, sizeof(unsigned long long));

}

void CegoXPorter::writeTableData(File *pOutFile, int tabSetId, const Chain& tableName, const ListT<CegoField>& schema, bool doPlain)
{

    if ( _pAH )
    {
	Chain info = Chain("Exporting table data for ") +  tableName + Chain(" ...");
	_pAH->sendInfo(info + Chain("\n"));
    }

    CegoObjectCursor *pOC = 0;
    ListT<CegoField> rowSchema = schema;

    try
    {
	
	pOC = _pGTM->getObjectCursor(tabSetId, tableName, tableName, CegoObject::TABLE);
	
	CegoDataPointer dp;
	
	unsigned long long rowCount = 0;
	
	if  ( doPlain )
	{
	    char *buf;
	    int len;
	    
	    buf = (char*)pOC->getFirst(len, dp);
	    
	    while ( buf )
	    {
		_tag = ROWTAG;
		pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
		pOutFile->writeByte((char*)&len, sizeof(int));
		pOutFile->writeByte(buf, len);
		
		rowCount++;
		if ( _pAH && ( rowCount % XP_ROWINTERVAL == 0 ) )
		    _pAH->sendInfo(Chain(rowCount) + Chain(" rows\r"));
		
		buf = (char*)pOC->getNext(len, dp);
	    }
	}
	else
	{
	    CegoDataPointer dp;
	    bool moreRow = _pGTM->getFirstTuple( pOC, rowSchema, dp );
	    
	    while ( moreRow )
	    {
		writeRow(pOutFile, tabSetId, rowSchema);
		
		rowCount++;
		if ( _pAH && ( rowCount % XP_ROWINTERVAL == 0 ) )
		    _pAH->sendInfo(Chain(rowCount) + Chain(" rows\r"));
		
		moreRow = _pGTM->getNextTuple(pOC, rowSchema, dp );
		
	    }
	}
	
	if ( _pAH )
	    _pAH->sendInfo(Chain(rowCount) + Chain(" rows exported\n"));
	
    }
    catch ( Exception e)
    {
	if ( pOC )
	{
	    delete pOC;
	}
	throw e;
    }

    delete pOC;

}


void CegoXPorter::writeRow(File *pOutFile, int tabSetId, const ListT<CegoField>& rowSchema)
{
    _tag = ROWTAG;
    pOutFile->writeByte((char*)&_tag, sizeof(BinTag));

    CegoField *pF = rowSchema.First();
    while ( pF ) 
    {
	if ( ! pF->getValue().isNull() )
	{
	    if ( pF->getType() == BLOB_TYPE )
	    {

		int fileId;
		int pageId;
		memcpy(&fileId, pF->getValue().getValue(), sizeof(int));
		memcpy(&pageId, (void*)((long long)pF->getValue().getValue() + sizeof(int)), sizeof(int));
		
		unsigned long long blobSize;
		unsigned char* blobBuf = _pGTM->getBlobData(tabSetId, fileId, pageId, blobSize);

		pOutFile->writeByte((char*)&blobSize, sizeof(unsigned long long));
		pOutFile->writeByte((char*)blobBuf, blobSize);

		delete blobBuf;
	    }
	    if ( pF->getType() == CLOB_TYPE )
	    {

		int fileId;
		int pageId;
		memcpy(&fileId, pF->getValue().getValue(), sizeof(int));
		memcpy(&pageId, (void*)((long long)pF->getValue().getValue() + sizeof(int)), sizeof(int));
		
		unsigned long long clobSize;
		char* clobBuf = _pGTM->getClobData(tabSetId, fileId, pageId, clobSize);

		pOutFile->writeByte((char*)&clobSize, sizeof(unsigned long long));
		pOutFile->writeByte((char*)clobBuf, clobSize);

		delete clobBuf;
	    }
	    else
	    {
		
		int colLen = pF->getValue().getEncodingLength();
		
		if ( colLen > XP_MAXCOLBUF )
		    throw Exception(EXLOC, "Col buffer exceeded");
		
		pF->getValue().encode(_colBuffer);
		
		pOutFile->writeByte((char*)&colLen, sizeof(int));
		pOutFile->writeByte((char*)_colBuffer, colLen);
	    }
	}		
	else
	{
	    int colLen = 0;
	    pOutFile->writeByte((char*)&colLen, sizeof(int));
	}
	pF = rowSchema.Next();
    }
}

void CegoXPorter::binImportTableSet(const Chain& tableSet, bool isStructure, const Chain& impFile, bool doLogging, bool doPlain)
{
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Binary importing tableset ")  + tableSet);

    _pGTM->setAppend(true);

    File *pInFile = new File(impFile);

    try
    {
	pInFile->open(File::READ);
	
	Chain impTableSet;
	readHeader(pInFile, impTableSet);
	
	if ( impTableSet != tableSet )
	{
	    throw Exception(EXLOC, "Tableset mismatch");	
	}
	
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
	while ( _tag != EOFTAG)
	{
	    if ( _tag == TABTAG )
	    {
		readTableObject(pInFile, tableSet, doLogging, doPlain);	    
	    }
	    else if ( _tag == IDXTAG )
	    {
		readIndexObject(pInFile, tableSet);	    
	    }
	    else if ( _tag == KEYTAG )
	    {
		readKeyObject(pInFile, tableSet);	    
	    }
	    else if ( _tag == CHECKTAG )
	    {
		readCheckObject(pInFile, tableSet);	    
	    }
	    else if ( _tag == VIEWTAG )
	    {
		readViewObject(pInFile, tableSet);	    
	    }
	    else if ( _tag == PROCTAG )
	    {
		readProcObject(pInFile, tableSet);	    
	    }
	    else if ( _tag == CNTTAG )
	    {
		readCounterObject(pInFile, tableSet);	    
	    }
	}
    }
    catch ( Exception e)
    {
	pInFile->close();
	delete pInFile;
	throw e;
    }
    
    pInFile->close();
    delete pInFile;
}

void CegoXPorter::binExportTable(const Chain& tableSet, const Chain& tableName, const Chain& expFile, bool doPlain)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Exporting tableset ") + tableSet + Chain(" (binary) ..."));

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    File *pOutFile = new File(expFile);

    try
    {
	
	pOutFile->open(File::WRITE);
	
	writeHeader(pOutFile, tableSet);
	
	CegoTableObject toe;
	_pGTM->getObject(tabSetId, tableName, CegoObject::TABLE, toe);
	
	writeTableObject(pOutFile, tabSetId, tableName, toe.getSchema(), doPlain);
	
	_tag = EOFTAG;
	pOutFile->writeByte((char*)&_tag, sizeof(BinTag));
	
    }
    catch( Exception e )
    {
	pOutFile->close();
	delete pOutFile;       
	throw e;
    }

    pOutFile->close();
    delete pOutFile;
	    
}

void CegoXPorter::binImportTable(const Chain& tableSet, const Chain& tableName, const Chain& impFile, bool doLogging, bool doPlain)
{

    _pDBMng->log(_modId, Logger::NOTICE, Chain("Importing tablename ") + tableName + Chain("..."));

    _pGTM->setAppend(true);

    File *pInFile = new File(impFile);

    try
    {
	pInFile->open(File::READ);
	
	Chain impTableSet;
	readHeader(pInFile, impTableSet);
	
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
	if ( _tag == TABTAG )
	{
	    readTableObject(pInFile, tableSet, doLogging, doPlain);	    
	}
    }
    catch ( Exception e )
    {
	pInFile->close();
	delete pInFile;
	throw e;
    }
    
    pInFile->close();
    delete pInFile;
    
}

void CegoXPorter::readHeader(File *pInFile, Chain& tableSet)
{

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
    
    if ( _tag != TSTAG )
    {
	throw Exception(EXLOC, "Wrong format in import file");
    }
    
    int tslen;
    pInFile->readByte((char*)&tslen, sizeof(int));
    
    if ( tslen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "InBuf exceeded");
    }
    
    pInFile->readByte((char*)_inBuf, tslen);
    tableSet = Chain(_inBuf, tslen);

}

void CegoXPorter::readTableObject(File *pInFile, const Chain& tableSet, bool doLogging, bool doPlain)
{

    int tlen;
    
    pInFile->readByte((char*)&tlen, sizeof(int));
    
    if ( tlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    
    pInFile->readByte((char*)_inBuf, tlen);
    Chain tableName = Chain(_inBuf, tlen);

    Chain info = Chain("Importing table ") + tableName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);
    
    ListT<CegoField> schema;

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    while ( _tag == FLDTAG)
    {

	int id;
	pInFile->readByte((char*)&id, sizeof(int));

	int alen;
	pInFile->readByte((char*)&alen, sizeof(int));
	if ( alen > XP_MAXINBUF )
	{
	    throw Exception(EXLOC, "InBuf exceeded");
	}

	pInFile->readByte((char*)_inBuf, alen);
	Chain attrName = Chain(_inBuf, alen);

	char n;
	pInFile->readByte((char*)&n, sizeof(char));

	bool isNullable;
	if ( n == 1 )
	    isNullable = true;
	else
	    isNullable = false;

	CegoDataType dt;
	pInFile->readByte((char*)&dt, sizeof(CegoDataType));		

	int len;
	pInFile->readByte((char*)&len, sizeof(int));		

	int defValLen;
	pInFile->readByte((char*)&defValLen, sizeof(int));
	
	CegoFieldValue defValue;
	if ( defValLen > 0 )
	{
	    if ( defValLen > XP_MAXCOLBUF )
		throw Exception(EXLOC, "Column buffer exceeded");
	    
	    pInFile->readByte((char*)_colBuffer, defValLen);
	    defValue.decode(_colBuffer);
	}

	schema.Insert(CegoField(tableName, tableName, attrName, dt, len, defValue, isNullable, id));
    
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    }

    if ( _pAH )
    {
	Chain info = Chain("Importing table data for ") +  tableName + Chain(" ...");
	_pAH->sendInfo(info + Chain("\n"));
    }

    ListT<CegoField> idxList;
    _pGTM->createDistDataTable(tableSet, tableName, CegoObject::TABLE, schema, idxList, true);

    unsigned long long rowCount = 0;

    if ( doPlain )
    {
	
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	
	CegoBufferPage bp;
	_pGTM->getObjectWithFix(tabSetId, tableName, CegoObject::TABLE, _oe, bp);
	_sysEntry = CegoDataPointer(bp.getFileId(), bp.getPageId(), bp.getEntryPos());
	_pDBMng->bufferUnfix(bp, false, _pGTM->getLockHandle());	
	
	char rowBuf[MAXROWBUF];
	while ( _tag == ROWTAG )
	{
	    // readRow(pInFile, tabSetId, tableName, schema, doLogging);
	    
	    int rowLen;
	    pInFile->readByte((char*)&rowLen, sizeof(int));
	    pInFile->readByte(rowBuf, rowLen);
	    
	    CegoDataPointer dp;
	    dp = _pGTM->insertData(_sysEntry, _oe, rowBuf, rowLen, true);

	    rowCount++;

	    if ( _pAH && ( rowCount % XP_ROWINTERVAL == 0 ) )
		_pAH->sendInfo(Chain(rowCount) + Chain(" rows\r"));
	    
	    pInFile->readByte((char*)&_tag, sizeof(BinTag));
	}
    }
    else
    {
	_isFirst = true;
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	while ( _tag == ROWTAG )
	{
	    readRow(pInFile, tabSetId, tableName, schema, doLogging);

	    rowCount++;

	    if ( _pAH && ( rowCount % XP_ROWINTERVAL == 0 ) )
		_pAH->sendInfo(Chain(rowCount) + Chain(" rows\r"));

	    pInFile->readByte((char*)&_tag, sizeof(BinTag));
	}
    }

    if ( _pAH )
	_pAH->sendInfo(Chain(rowCount) + Chain(" rows imported\n"));

    
}

void CegoXPorter::readIndexObject(File *pInFile, const Chain& tableSet)
{

    int ilen;

    
    pInFile->readByte((char*)&ilen, sizeof(int));
    
    if ( ilen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "InBuf exceeded");
    }
    
    pInFile->readByte((char*)_inBuf, ilen);
    Chain idxName = Chain(_inBuf, ilen);
    
    int tlen;    
    pInFile->readByte((char*)&tlen, sizeof(int));
    
    if ( tlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    
    pInFile->readByte((char*)_inBuf, tlen);
    Chain tableName = Chain(_inBuf, tlen);
    
    CegoObject::ObjectType type;
    pInFile->readByte((char*)&type, sizeof(CegoObject::ObjectType));
    
    ListT<CegoField> schema;

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    while ( _tag == FLDTAG)
    {

	int alen;
	pInFile->readByte((char*)&alen, sizeof(int));
	if ( alen > XP_MAXINBUF )
	{
	    throw Exception(EXLOC, "Import buffer exceeded");
	}

	pInFile->readByte((char*)_inBuf, alen);
	Chain attrName = Chain(_inBuf, alen);

	CegoDataType dt;
	pInFile->readByte((char*)&dt, sizeof(CegoDataType));		

	int len;
	pInFile->readByte((char*)&len, sizeof(int));		
	
	schema.Insert(CegoField(idxName, idxName, attrName, dt, len));
    
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    }

    Chain info = Chain("Importing index ") + idxName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);

    _pGTM->createDistIndexTable(tableSet, idxName, tableName, schema, type);
}

void CegoXPorter::readKeyObject(File *pInFile, const Chain& tableSet)
{

    int klen;    
    pInFile->readByte((char*)&klen, sizeof(int));
    if ( klen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, klen);
    Chain keyName = Chain(_inBuf, klen);

    int tlen;    
    pInFile->readByte((char*)&tlen, sizeof(int));
    if ( tlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, tlen);
    Chain tableName = Chain(_inBuf, tlen);

    int rtlen;    
    pInFile->readByte((char*)&rtlen, sizeof(int));
    if ( rtlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, rtlen);
    Chain refTable = Chain(_inBuf, rtlen);

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
    ListT<CegoField> keySchema;	
    while ( _tag == FLDTAG)
    {

	int alen;
	pInFile->readByte((char*)&alen, sizeof(int));
	if ( alen > XP_MAXINBUF )
	{
	    throw Exception(EXLOC, "Import buffer exceeded");
	}

	pInFile->readByte((char*)_inBuf, alen);
	Chain attrName = Chain(_inBuf, alen);

	keySchema.Insert(CegoField(Chain(), attrName));
    
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    }

    if ( _tag != SEPTAG )
	throw Exception(EXLOC, "Import separater tag is missing");
    
    pInFile->readByte((char*)&_tag, sizeof(BinTag));
    ListT<CegoField> refSchema;	
    while ( _tag == FLDTAG)
    {

	int alen;
	pInFile->readByte((char*)&alen, sizeof(int));
	if ( alen > XP_MAXINBUF )
	{
	    throw Exception(EXLOC, "Import buffer exceeded");
	}

	pInFile->readByte((char*)_inBuf, alen);
	Chain attrName = Chain(_inBuf, alen);
	
	refSchema.Insert(CegoField(Chain(), attrName));
    
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    }

    Chain info = Chain("Importing foreign key ") + keyName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);

    _pGTM->createDistForeignKey(tableSet, keyName, tableName, keySchema, refTable, refSchema);    
}

void CegoXPorter::readCheckObject(File *pInFile, const Chain& tableSet)
{
    int cklen;    
    pInFile->readByte((char*)&cklen, sizeof(int));
    if ( cklen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, cklen);
    Chain checkName = Chain(_inBuf, cklen);

    int tlen;    
    pInFile->readByte((char*)&tlen, sizeof(int));
    if ( tlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, tlen);
    Chain tableName = Chain(_inBuf, tlen);

    int pdlen;    
    pInFile->readByte((char*)&pdlen, sizeof(int));
    if ( pdlen > XP_MAXPDBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_pdBuffer, pdlen);

    int tabSetId = _pDBMng->getTabSetId(tableSet);    
    CegoPredDesc *pPredDesc = new CegoPredDesc(_pdBuffer, _pGTM, tabSetId);

    Chain info = Chain("Importing check ") + checkName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);

    _pGTM->createDistCheck(tableSet, checkName, tableName, pPredDesc);    
}

void CegoXPorter::readViewObject(File *pInFile, const Chain& tableSet)
{
    int vlen;    
    pInFile->readByte((char*)&vlen, sizeof(int));
    if ( vlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }

    pInFile->readByte((char*)_inBuf, vlen);
    Chain viewName = Chain(_inBuf, vlen);


    int slen;    
    pInFile->readByte((char*)&slen, sizeof(int));
    if ( slen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "InBuf exceeded");
    }
    pInFile->readByte((char*)_inBuf, slen);
    Chain viewStmt = Chain(_inBuf, slen);
    
    pInFile->readByte((char*)&_tag, sizeof(BinTag));

    ListT<CegoField> viewSchema;
    while ( _tag == FLDTAG)
    {

	int alen;
	pInFile->readByte((char*)&alen, sizeof(int));
	if ( alen > XP_MAXINBUF )
	{
	    throw Exception(EXLOC, "InBuf exceeded");
	}

	pInFile->readByte((char*)_inBuf, alen);
	Chain attrName = Chain(_inBuf, alen);

	CegoDataType dt;
	pInFile->readByte((char*)&dt, sizeof(CegoDataType));		
	
	int len;
	pInFile->readByte((char*)&len, sizeof(int));		
	
	viewSchema.Insert(CegoField(viewName, viewName, attrName, dt, len));
    
	pInFile->readByte((char*)&_tag, sizeof(BinTag));
	
    }

    Chain info = Chain("Importing view ") + viewName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);

    _pGTM->createDistView(tableSet, viewName, viewSchema, viewStmt);   
	
}

void CegoXPorter::readProcObject(File *pInFile, const Chain& tableSet)
{
    int plen;    
    pInFile->readByte((char*)&plen, sizeof(int));
    if ( plen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, plen);
    Chain procName = Chain(_inBuf, plen);

    int txtlen;    
    pInFile->readByte((char*)&txtlen, sizeof(int));
    if ( txtlen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, txtlen);
    Chain procText = Chain(_inBuf, txtlen);


    Chain info = Chain("Importing procedure ") + procName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);


    _pGTM->createDistProc(tableSet, procName, procText);

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
}

void CegoXPorter::readCounterObject(File *pInFile, const Chain& tableSet)
{
    int clen;
    pInFile->readByte((char*)&clen, sizeof(int));
    if ( clen > XP_MAXINBUF )
    {
	throw Exception(EXLOC, "Import buffer exceeded");
    }
    pInFile->readByte((char*)_inBuf, clen);
    Chain counterName = Chain(_inBuf, clen);

    unsigned long long counterValue;    
    pInFile->readByte((char*)&counterValue, sizeof(unsigned long long));

    Chain info = Chain("Importing counter ") + counterName + Chain("...");
    if ( _pAH )
	_pAH->sendInfo(info + Chain("\n"));
    _pDBMng->log(_modId, Logger::NOTICE, info);

    int tabSetId = _pDBMng->getTabSetId(tableSet);	
    _pDBMng->addCounter(tabSetId, counterName, counterValue, true);

    pInFile->readByte((char*)&_tag, sizeof(BinTag));
}

void CegoXPorter::readRow(File *pInFile, int tabSetId, const Chain& tableName, const ListT<CegoField>& rowSchema, bool doLogging)
{
    ListT<CegoField> fvl;

    CegoField *pF = rowSchema.First();
    while ( pF ) 
    {
	    
	if ( pF->getType() == BLOB_TYPE )
	{
	    unsigned long long blobSize;
	    pInFile->readByte((char*)&blobSize, sizeof(unsigned long long));
	    	    	    
	    unsigned char *blobBuf = (unsigned char*)malloc(blobSize);
	    if ( blobBuf == NULL )
	    {
		throw Exception(EXLOC, "Malloc system error");     
	    }

	    pInFile->readByte((char*)blobBuf, blobSize);
	    	    
	    int fileId;
	    int pageId;

	    _pGTM->putBlobData(tabSetId, blobBuf, blobSize, fileId, pageId);
		    
	    free (blobBuf );

	    CegoField f(pF->getTableName(), pF->getAttrName() );

	    Chain blobRef = Chain("[") + Chain(fileId) + Chain(",") + Chain(pageId) + Chain("]");
	    CegoFieldValue fv(pF->getType(), blobRef);
	    f.setValue(fv);
	    fvl.Insert(f);
	}
	else if ( pF->getType() == CLOB_TYPE )
	{
	    unsigned long long clobSize;
	    pInFile->readByte((char*)&clobSize, sizeof(unsigned long long));
	    	    	    
	    char *clobBuf = (char*)malloc(clobSize);
	    if ( clobBuf == NULL )
	    {
		throw Exception(EXLOC, "malloc system error");     
	    }

	    pInFile->readByte((char*)clobBuf, clobSize);
	    	    
	    int fileId;
	    int pageId;

	    _pGTM->putClobData(tabSetId, clobBuf, clobSize, fileId, pageId);
		    
	    free (clobBuf );

	    CegoField f(pF->getTableName(), pF->getAttrName() );

	    Chain clobRef = Chain("[") + Chain(fileId) + Chain(",") + Chain(pageId) + Chain("]");
	    CegoFieldValue fv(pF->getType(), clobRef);
	    f.setValue(fv);
	    fvl.Insert(f);

	}
	else
	{

	    int colLen;
	    pInFile->readByte((char*)&colLen, sizeof(int));
	    
	    CegoFieldValue fv;
	    if ( colLen > 0 )
	    {
		if ( colLen > XP_MAXCOLBUF )
		    throw Exception(EXLOC, "Col buffer exceeded");
		
		pInFile->readByte((char*)_colBuffer, colLen);
		fv.decode(_colBuffer);
	    }
	    
	    CegoField f(pF->getTableName(), pF->getAttrName() );
	    f.setValue(fv);
	    fvl.Insert(f);
	}

	pF = rowSchema.Next();
	
    }
    
    if ( _isFirst )
    {
	_idxList.Empty();
	_keyList.Empty();
	_checkList.Empty();

	CegoBufferPage bp;
	_pGTM->getObjectWithFix(tabSetId, tableName, CegoObject::TABLE, _oe, bp);
	_sysEntry = CegoDataPointer(bp.getFileId(), bp.getPageId(), bp.getEntryPos());
	_pDBMng->bufferUnfix(bp, false, _pGTM->getLockHandle());

	int numInvalid;
	_pGTM->getObjectListByTable(_oe.getTabSetId(), _oe.getName(), _idxList, _btreeList, _keyList, _checkList, numInvalid);
	_isFirst = false;
    }

    CegoField* pCF = rowSchema.First();
    CegoField* pCV =  fvl.First();
    
    while ( pCF && pCV)
    {	    
	CegoFieldValue fv = pCV->getValue();
	
	CegoQueryHelper::prepareFieldValue(pCF, fv, _pGTM, _oe.getTabSetId());
	
	pCV->setValue(fv); 
	
	pCF = rowSchema.Next();
	pCV = fvl.Next();
	
    }
    if ( pCF || pCV )
    {
	throw Exception(EXLOC, "Mismatched argument count for value list");
    }

    CegoDataPointer dp;
    Chain virginIndex;

    _pGTM->insertDataTable(_oe, fvl, _idxList, _btreeList, _keyList, _checkList,
			   _sysEntry, virginIndex, dp, doLogging);        
}
