///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBTreeManager.cc
// -------------------
// Cego index manager class implementation
//                                                         
// Design and Implementation by Bjoern Lemke               
//         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBTreeManager
//
// Description: This class implements the btree algorithm to be used for table index objects.
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>

// CEGO INCLUDES
#include "CegoBTreeManager.h"
#include "CegoLockHandler.h"
#include "CegoTupleState.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoBTreeManager::BTreeCache::BTreeCache()
{
}

CegoBTreeManager::BTreeCache::~BTreeCache()
{

    CacheEntry *pEntry = _cache.First();
    while ( pEntry )
    {
	// cout << "Freeing page " << pEntry->getPage().getFileId() << " / " << pEntry->getPage().getPageId() << " ptr = " << (long)(pEntry->getPage().getPagePtr()) << endl;
	free(pEntry->getPage().getPagePtr());
	pEntry = _cache.Next();
    }
    _cache.Empty();    
}

CegoBufferPage CegoBTreeManager::BTreeCache::newCachePage(const CegoBufferPage& bp, bool copyPage)
{
    char* pagePtr = (char*)malloc(bp.getPageSize());

    // cout << "Allocating ptr " << (long)pagePtr << " of size " << bp.getPageSize() << endl;
    CegoBufferPage cachePage = CegoBufferPage(pagePtr, bp.getPageSize());

    cachePage.initPage(bp.getType());
    cachePage.setFileId(bp.getFileId());
    cachePage.setPageId(bp.getPageId());
    cachePage.setNextFileId(bp.getNextFileId());
    cachePage.setNextPageId(bp.getNextPageId());
    if ( copyPage )
	memcpy( cachePage.getChunkEntry(), bp.getChunkEntry(), bp.getChunkLen());	
    
    _cache.Insert(CacheEntry(cachePage));

    // cout << "New cache page allocated" << endl;
    return cachePage;
}

void CegoBTreeManager::BTreeCache::updateCachePage(CegoBufferPage& bp)
{    
    CacheEntry *pEntry = _cache.Find(CacheEntry(bp));
    if ( pEntry )
    {
	// cout << "Setting ptr to " << (long)bp.getPagePtr() << endl;
	pEntry->setPage(bp);
    }
}

bool CegoBTreeManager::BTreeCache::getCachePage(int fileId, int pageId, CegoBufferPage& bp)
{
    CegoBufferPage sp;
    sp.setFileId(fileId);
    sp.setPageId(pageId);
    
    CacheEntry *pEntry = _cache.Find(sp);
    if ( pEntry )
    {
	bp = pEntry->getPage();
	return true;
    }
    return false;
}
		  
bool CegoBTreeManager::BTreeCache::getFirst(CegoBufferPage& bp)
{
    CacheEntry *pEntry = _cache.First();
    if ( pEntry )
    {
	bp = pEntry->getPage();
	return true;
    }
    return false;
}
	
bool CegoBTreeManager::BTreeCache::getNext(CegoBufferPage& bp)
{
    CacheEntry *pEntry = _cache.Next();
    if ( pEntry )
    {
	bp = pEntry->getPage();
	return true;
    }
    return false;
}

CegoBTreeManager::CegoBTreeManager(CegoObjectManager *pObjMng, CegoBTreeObject *pBTO)
{
    _pDBMng = pObjMng->getDBMng();
    _modId = _pDBMng->getModId("CegoBTreeManager");
    _pObjMng = pObjMng;
    _pBTO = pBTO;
    _pCache = 0;
    
    _tabSetId = _pBTO->getTabSetId();
    _btreeName = _pBTO->getName();    
    _btreeType = _pBTO->getType();
    _btreeSchema = _pBTO->getSchema(); 

    _keyLen = CegoBTreeValue::getKeyLen(_btreeSchema);
    /*
    CegoField *pF = _btreeSchema.First();
    _keyLen = 0;
    while ( pF )
    {
	_keyLen += 1; // null value indicator
	_keyLen += pF->getLength();

	if ( pF->getType() == VARCHAR_TYPE
	     || pF->getType() == BIGINT_TYPE
	     || pF->getType() == DECIMAL_TYPE
	     || pF->getType() == FIXED_TYPE )
	    _keyLen += 1; // termination character
	 	
	pF = _btreeSchema.Next();
    }
    */
    
    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
    
}

CegoBTreeManager::~CegoBTreeManager()
{
    if ( _pCache )
	delete _pCache;
}

void CegoBTreeManager::createCache()
{
    if ( _pCache )
	delete _pCache;

    _pCache = new BTreeCache();
}

void CegoBTreeManager::commit(CegoDataPointer& sysEntry)
{    
    if ( _pCache )
    {
	
	CegoBufferPage cachePage;

	bool goOn = _pCache->getFirst(cachePage);
	while ( goOn )
	{
	    // sync page here

	    // cout << "Syncing page " << cachePage.getFileId() << " / " << cachePage.getPageId() << " bytes = " << cachePage.getChunkLen() << endl;
	    
	    CegoBufferPage bp;
	    _pDBMng->bufferFix(bp, _tabSetId, cachePage.getFileId(), cachePage.getPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());

	    memcpy( bp.getChunkEntry(), cachePage.getChunkEntry(), cachePage.getChunkLen());

	    bp.setType(cachePage.getType());
	    bp.setNextFileId(cachePage.getNextFileId());
	    bp.setNextPageId(cachePage.getNextPageId());
	    
	    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
		
	    goOn = _pCache->getNext(cachePage);
	}
	
	// cout << "Deleting cache " << endl;
	delete _pCache;
	_pCache = 0;
    }

    char *p;
    int len;
    CegoBufferPage rbp = _pObjMng->claimDataPtrUnlocked(_tabSetId, 
							CegoBufferPool::NOSYNC,
							sysEntry, p, len);
    _pBTO->encode(p);
    _pObjMng->releaseDataPtrUnlocked(rbp, true);
}

void CegoBTreeManager::rollback()
{    
    if ( _pCache )
    {	
	CegoBufferPage cachePage;

	bool goOn = _pCache->getFirst(cachePage);
	while ( goOn )
	{	    
	    CegoBufferPage bp;
	    _pDBMng->bufferFix(bp, _tabSetId, cachePage.getFileId(), cachePage.getPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
	    _pDBMng->bufferRelease(bp, _pObjMng->getLockHandler());
		
	    goOn = _pCache->getNext(cachePage);
	}
			    
	// cout << "Deleting cache " << endl;
	delete _pCache;
	_pCache = 0;
    }
    else
    {
	freeBTree();
    }    
}

void CegoBTreeManager::insertBTreeWithCommit(const CegoDataPointer& dp,
				   const CegoBTreeValue &iv,
				   unsigned long long tid)
{    
    // we have to keep the index object in buffer pool 
    CegoBufferPage bp;
    _pObjMng->getObjectWithFix(_tabSetId, _btreeName, _btreeType, *_pBTO, bp);
    
    CegoDataPointer sysEntry(bp.getFileId(), bp.getPageId(), bp.getEntryPos());

    try 
    {
	insertBTree( dp, iv, tid);
    }
    catch ( Exception e )
    {
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
	throw e;
    }
    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());

    commit(sysEntry);
}

void CegoBTreeManager::insertBTree(const CegoDataPointer& dp,
				   const CegoBTreeValue &iv,
				   unsigned long long tid)
{   
    StackT<CegoBufferPage> parentPageStack;
    StackT<CegoBufferPage> fixedPageStack;

    unsigned long long lockId = 0;
    
    try
    {	
	// cout << "BTree Insert" << endl;

	int fileId = _pBTO->getDataFileId();
	int pageId = _pBTO->getDataPageId();	    
	
	if ( fileId == 0 && pageId == 0 )
	{
	    throw Exception(EXLOC, Chain("Btree object is not valid"));
	}
    	
	// cout << "Write Locking " << fileId << " " << pageId << endl;
	if ( _pCache == 0 )
	    lockId = _pObjMng->getLockHandler()->lockData(CegoObject::BTREE, fileId, pageId, CegoLockHandler::WRITE);
	
	bool isInserted = false;
	bool isFirst = true;

	while ( ! isInserted )
	{
	    // cout << "Try to insert .." << endl;
	    CegoBufferPage bp;
	    
	    getPage(bp, fileId, pageId);
	    	    
	    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
	    {	
		CegoBTreeNode traceNode;
		traceNode.setType(CegoBTreeNode::NODE);
		traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());
		traceNode.setSchema(&_btreeSchema, _keyLen);
		traceNode.setFileId(bp.getFileId());
		traceNode.setPageId(bp.getPageId());

		// traceNode.printNode();

		// cout << "Tracing node " << bp.getFileId() << " " << bp.getPageId() << endl; 
		traceNode.getChildPage(iv, fileId, pageId);
		
		// cout << "Tracing to child page .." << fileId << " " << pageId << endl; 
		parentPageStack.Push(bp);
	    }
	    else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
	    {			

		fixedPageStack.Push(bp);
		
		CegoBTreeNode leafLeft;
		leafLeft.setType(CegoBTreeNode::LEAF);
		leafLeft.setPtr(bp.getChunkEntry(), bp.getChunkLen());
		leafLeft.setFileId(bp.getFileId());
		leafLeft.setPageId(bp.getPageId());
		leafLeft.setSchema(&_btreeSchema, _keyLen);
		leafLeft.setNextFileId(bp.getNextFileId());
		leafLeft.setNextPageId(bp.getNextPageId());
		
		bool duplicateAdded;
		if ( leafLeft.addValue(iv, dp, _btreeName, _btreeType == CegoObject::PBTREE || _btreeType == CegoObject::UBTREE, _pObjMng, _tabSetId, tid, duplicateAdded) == false )
		{	    
		    // cout << "Creating new leaf page .." << endl;
		    CegoBTreeNode leafRight;
		    
		    // create new
		    CegoBufferPage newPage;
		    // cout << "NEWFILEPAGE LEAF" << endl;

		    allocPage(CegoBufferPage::BTREE_LEAF, newPage);
		    
		    // _pObjMng->getNewFilePage(newPage, _tabSetId, CegoObject::BTREE, true, doAppend);		    
		    // newPage.initPage(CegoBufferPage::BTREE_LEAF);

		    fixedPageStack.Push(newPage);
		    
		    leafRight.setType(CegoBTreeNode::LEAF);
		    leafRight.setPtr(newPage.getChunkEntry(), newPage.getChunkLen());
		    leafRight.initNode();
		    leafRight.setFileId(newPage.getFileId());
		    leafRight.setPageId(newPage.getPageId());
		    leafRight.setSchema(&_btreeSchema, _keyLen);

		    // cout << "NEW LEAF NODE " << newPage.getPageId() << endl;
		    // cout << "Splitting leaf page .." << endl;
		    leafLeft.split(leafRight);

		    // setting up leaf pointer chain
		    newPage.setNextFileId(leafLeft.getNextFileId());
		    newPage.setNextPageId(leafLeft.getNextPageId());
		    
		    leafLeft.setNextFileId(newPage.getFileId());
		    leafLeft.setNextPageId(newPage.getPageId());

		    CegoBufferPage modPage;

		    getPage(modPage, leafLeft.getFileId(), leafLeft.getPageId());
				
		    // _pDBMng->bufferFix(modPage, _tabSetId, leafLeft.getFileId(), leafLeft.getPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
		    modPage.setNextFileId(newPage.getFileId());
		    modPage.setNextPageId(newPage.getPageId());

		    putPage(modPage);
		    
		    // _pDBMng->bufferUnfix(modPage, true, _pObjMng->getLockHandler());	
		    
		    // cout << "Setting next page to " << newPage.getFileId() << " " << newPage.getPageId() << endl;
		    
		    CegoBTreeValue ri = leafRight.getMin();
		    
		    bool dupAdded;
		    if ( iv.isHigher(ri, &_btreeSchema) )
		    {
			leafRight.addValue(iv, dp, _btreeName, _btreeType == CegoObject::PBTREE || _btreeType == CegoObject::UBTREE, _pObjMng, _tabSetId, tid, dupAdded);
		    }
		    else
		    {
			leafLeft.addValue(iv, dp, _btreeName, _btreeType == CegoObject::PBTREE || _btreeType == CegoObject::UBTREE, _pObjMng, _tabSetId, tid, dupAdded);
		    }
		    
		    CegoBTreeValue propValue = leafLeft.getMax();
		    CegoBTreeNode propRight = leafRight;
		    CegoBTreeNode propLeft = leafLeft;
			    		    
		    bool isPropagated = false;
		    while ( ! isPropagated )
		    {
			// propagate newLeaf to parent node
			CegoBufferPage propPage;
			
			if ( parentPageStack.Pop(propPage) )
			{

			    // cout << "Adding propPage to stack" << propPage.getFileId() << " "  << propPage.getPageId() << endl;
			    fixedPageStack.Push(propPage);
			    
			    CegoBTreeNode nodeLeft;
			    nodeLeft.setType(CegoBTreeNode::NODE);
			    nodeLeft.setPtr(propPage.getChunkEntry(), propPage.getChunkLen());
			    nodeLeft.setFileId(propPage.getFileId());
			    nodeLeft.setPageId(propPage.getPageId());

			    nodeLeft.setNextFileId(propPage.getNextFileId());
			    nodeLeft.setNextPageId(propPage.getNextPageId());

			    // cout << "Propagating to page " << nodeLeft.getPageId() << " ==>> Follow up is " << nodeLeft.getNextPageId() << endl;
			    nodeLeft.setSchema(&_btreeSchema, _keyLen);
			    			  
			    if ( nodeLeft.propagate(propValue, propLeft, propRight) == false )
			    {								
				propLeft = nodeLeft;
				
				// since we still have to add the right node, save it
				CegoBTreeNode addNode = propRight;

				// create new 
				CegoBufferPage newPage;

				// cout << "NEWFILEPAGE NODE" << endl;

				allocPage(CegoBufferPage::BTREE_NODE, newPage);
				
				// _pObjMng->getNewFilePage(newPage, _tabSetId, CegoObject::BTREE, true, doAppend);
				// newPage.initPage(CegoBufferPage::BTREE_NODE);
				fixedPageStack.Push(newPage);
				
				propRight.setType(CegoBTreeNode::NODE);
				propRight.setPtr(newPage.getChunkEntry(), newPage.getChunkLen());
				propRight.initNode();
				propRight.setFileId(newPage.getFileId());
				propRight.setPageId(newPage.getPageId());
				propRight.setSchema(&_btreeSchema, _keyLen);

				// cout << "NEW NODE " << newPage.getPageId() << endl;				
				
				// cout << "Num Entries before split = " << propLeft.numEntries() << endl;
				propLeft.split(propRight);
				
				// Setting up follow up node page

				// cout << "Setting up follow up page for " << propLeft.getPageId() << " to " << propRight.getPageId() << endl;

				CegoBufferPage modPage;
				getPage(modPage, propLeft.getFileId(), propLeft.getPageId());
				// _pDBMng->bufferFix(modPage, _tabSetId, propLeft.getFileId(), propLeft.getPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());				
				modPage.setNextFileId(propRight.getFileId());
				modPage.setNextPageId(propRight.getPageId());

				propLeft.setNextFileId(propRight.getFileId());
				propLeft.setNextPageId(propRight.getPageId());

				putPage(modPage);
				// _pDBMng->bufferUnfix(modPage, true, _pObjMng->getLockHandler());	
					
				// cout << "Num Entries after split = " << propLeft.numEntries() << " + " << propRight.numEntries() << endl;
				// propLeft.printNode(0);
				// propRight.printNode(0);

				propValue = propLeft.getMax();
												
				// add new node
				CegoBTreeValue iv1 = addNode.getMin();
				CegoBTreeValue iv2 = propRight.getMin();
				
				if ( iv1.isHigher(iv2, &_btreeSchema) )
				{
				    // cout << "Adding new node on right !!!!" << endl;
				    propRight.addNode(iv1, addNode);
				}
				else
				{				    				   
				    // cout << "Adding new node on left !!!!" << endl;
				    propLeft.addNode(iv1, addNode);
				}				
			    }
			    else
			    {			   
				isPropagated = true;
			    }
			    
			    // Setting up follow up node page
			    			    
			    if ( propPage.getNextFileId() && propPage.getNextPageId() )
			    {
				// cout << "XXX Setting up first pointer for follow up page " << propPage.getNextFileId() << "/" << propPage.getNextPageId() 
				//      << " of page " << propPage.getFileId() << "/" << propPage.getPageId() << endl;

				CegoBufferPage nodePageA;				
				CegoBufferPage nodePageB;
				CegoBTreeNode nodeA;				
				CegoBTreeNode nodeB;				
				int nodeFileId, nodePageId;

				// _pDBMng->bufferFix(nodePageA, _tabSetId, propPage.getFileId(), propPage.getPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
				getPage(nodePageA, propPage.getFileId(), propPage.getPageId());
				
				nodeA.setPtr(nodePageA.getChunkEntry(), nodePageA.getChunkLen());
				nodeA.setType(CegoBTreeNode::NODE);
				nodeA.setSchema(&_btreeSchema, _keyLen);
								
				// _pDBMng->bufferFix(nodePageB, _tabSetId, propPage.getNextFileId(), propPage.getNextPageId(), CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
				getPage(nodePageB, propPage.getNextFileId(), propPage.getNextPageId());
				
				nodeB.setPtr(nodePageB.getChunkEntry(), nodePageB.getChunkLen());
				nodeB.setType(CegoBTreeNode::NODE);
				nodeB.setSchema(&_btreeSchema, _keyLen);
				
				nodeA.getLastChildPointer(nodeFileId, nodePageId);
				nodeB.setFirstChildPointer(nodeFileId, nodePageId);

				putPage(nodePageA);
				putPage(nodePageB);
				
				// _pDBMng->bufferUnfix(nodePageA, true, _pObjMng->getLockHandler());	
				// _pDBMng->bufferUnfix(nodePageB, true, _pObjMng->getLockHandler());	
			    }			    
			}
			else // root node reached, adding new root
			{			    			    
			    CegoBufferPage rootPage;

			    allocPage(CegoBufferPage::BTREE_NODE, rootPage);
			    // cout << "NEWFILEPAGE NODE" << endl;
			    // _pObjMng->getNewFilePage(rootPage, _tabSetId, CegoObject::BTREE, true, doAppend);
			    // rootPage.initPage(CegoBufferPage::BTREE_NODE);
			   
			    // cout << "Adding rootPage to stack" << endl; 
			    fixedPageStack.Push(rootPage);				
			    
			    CegoBTreeNode rootNode;
			    rootNode.setType(CegoBTreeNode::NODE);
			    rootNode.setPtr(rootPage.getChunkEntry(), rootPage.getChunkLen());
			    rootNode.initNode();			
			    rootNode.setFileId(rootPage.getFileId());
			    rootNode.setPageId(rootPage.getPageId());
			    rootNode.setSchema(&_btreeSchema, _keyLen);

			    rootNode.propagate(propValue, propLeft, propRight);
			    
			    _pBTO->setDataFileId(rootPage.getFileId());
			    _pBTO->setDataPageId(rootPage.getPageId());
			    
			    isPropagated=true;
			}
		    }		
		}
		else
		{
		    fixedPageStack.Pop(bp);    
		    if ( _pCache == 0 )			
			_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
		}
		
		if ( duplicateAdded == false )
		{
		    _pBTO->increaseRelevance();
		}

		isInserted = true;	    
	    }

	    isFirst = false;
	}
    }
    catch ( Exception e )
    {
	if ( _pCache == 0 )
	{
	    CegoBufferPage bp;
	    while ( parentPageStack.Pop(bp) )
	    {
		_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());	    
	    }
	    
	    while ( fixedPageStack.Pop(bp) )
	    {
		_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());	    
	    }
	    
	    _pObjMng->getLockHandler()->unlockData(CegoObject::BTREE, lockId);
	}
	throw e;	
    }

    if ( _pCache == 0 )
    {
	// unfix node pages on stack
	CegoBufferPage bp;
	while ( parentPageStack.Pop(bp) )
	{
	    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());	   
	}
	
	while ( fixedPageStack.Pop(bp) )
	{
	    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());	
	}
	
	_pObjMng->getLockHandler()->unlockData(CegoObject::BTREE, lockId);
    }
}

void CegoBTreeManager::deleteBTree(const CegoBTreeValue &iv,
				   const CegoDataPointer& dp)
{    
    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
    
    if ( fileId == 0 && pageId == 0 )
    {
	throw Exception(EXLOC, Chain("Btree object is not valid"));
    }

    CegoBufferPage rootBP;
    // cout << "Fixing root page " << fileId << " " << pageId << endl;
    _pDBMng->bufferFix(rootBP, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
        
    // we have to keep the index object in buffer pool 
    CegoBufferPage bp;
    _pObjMng->getObjectWithFix(_tabSetId, _btreeName, _btreeType, *_pBTO, bp);
    
    CegoDataPointer sysEntry(bp.getFileId(), bp.getPageId(), bp.getEntryPos());

    try 
    {
	deleteBTree(sysEntry, iv, dp);
    }
    catch ( Exception e )
    {
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
	throw e;
    }
    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());

    _pDBMng->bufferUnfix(rootBP, true, _pObjMng->getLockHandler());    
}

void CegoBTreeManager::deleteBTree(const CegoDataPointer& sysEntry, 
				   const CegoBTreeValue &iv,
				   const CegoDataPointer& dp)
{        
    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
        
    if ( fileId == 0 && pageId == 0 )
    {
	throw Exception(EXLOC, Chain("Btree object is not valid"));
    }
    
    bool isDeleted = false;

    StackT<CegoBufferPage> parentPageStack;

    unsigned long long lockId;
    
    try
    {
	lockId = _pObjMng->getLockHandler()->lockData(CegoObject::BTREE, fileId, pageId, CegoLockHandler::WRITE);
	
	while ( ! isDeleted )
	{	    
	    CegoBufferPage bp;
	    
	    _pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
	    
	    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
	    {	
		CegoBTreeNode traceNode;
		traceNode.setType(CegoBTreeNode::NODE);
		traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
		traceNode.setSchema(&_btreeSchema, _keyLen);
		traceNode.setFileId(bp.getFileId());
		traceNode.setPageId(bp.getPageId());
		
		// traceNode.printNode();
		
		traceNode.getChildPage(iv, fileId, pageId);
		
		// cout << "Tracing to child page .." << fileId << " " << pageId << endl; 
		
		parentPageStack.Push(bp);
	    }
	    else if ( bp.getType() == CegoBufferPage::BTREE_LEAF )
	    {	
		CegoBTreeNode leaf;
		leaf.setType(CegoBTreeNode::LEAF);
		leaf.setPtr(bp.getChunkEntry(), bp.getChunkLen());
		leaf.setFileId(bp.getFileId());
		leaf.setPageId(bp.getPageId());
		leaf.setSchema(&_btreeSchema, _keyLen);
		leaf.setNextFileId(bp.getNextFileId());
		leaf.setNextPageId(bp.getNextPageId());

		int leafPrevFileId = 0;
		int leafPrevPageId = 0;
		
		int leafFileId = bp.getFileId();
		int leafPageId = bp.getPageId();
		int leafNextFileId = bp.getNextFileId();
		int leafNextPageId = bp.getNextPageId();
		
		while ( isDeleted == false )
		{
		    isDeleted = leaf.deleteValue(iv, dp);
		    if ( isDeleted == false )
		    {
			if ( leafNextFileId != 0 && leafNextPageId != 0 )
			{
			    _pDBMng->bufferUnfix(bp, false, _pObjMng->getLockHandler());
			    _pDBMng->bufferFix(bp, _tabSetId, leafNextFileId, leafNextPageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
			    leaf.setPtr(bp.getChunkEntry(), bp.getChunkLen());

			    leafPrevFileId = leafFileId;
			    leafPrevPageId = leafPageId;
			    
			    leafFileId = bp.getFileId();
			    leafPageId = bp.getPageId();
			    leafNextFileId = bp.getNextFileId();
			    leafNextPageId = bp.getNextPageId();
			    
			    // leaf.setFileId(bp.getFileId());
			    // leaf.setPageId(bp.getPageId());			
			}
			else
			{
			    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
			    throw Exception(EXLOC, Chain("Entry not found in btree"));
			}
		    }
		}
		
		_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
		
		// check for duplicates
		
		bool hasDuplicate=false;
		
		CegoBufferPage cbp;
		_pDBMng->bufferFix(cbp, _tabSetId, leafFileId, leafPageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
		leaf.setPtr(cbp.getChunkEntry(), cbp.getChunkLen());
		if ( leaf.valueExists(iv) )
		{
		    hasDuplicate=true;
		}
		
		if ( hasDuplicate == false && leafPrevFileId != 0 && leafPrevPageId != 0 )
		{
		    // now we still have to check the next following page for duplicates
		    CegoBufferPage pbp;
		    _pDBMng->bufferFix(pbp, _tabSetId, leafPrevFileId, leafPrevPageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
		    leaf.setPtr(pbp.getChunkEntry(), pbp.getChunkLen());
		    if ( leaf.valueExists(iv) )
		    {
			hasDuplicate=true;
		    }
		    
		    _pDBMng->bufferUnfix(pbp, false, _pObjMng->getLockHandler());
		}
		
		if ( hasDuplicate == false && leafNextFileId != 0 && leafNextPageId != 0 )
		{
		    // now we still have to check the next following page for duplicates
		    CegoBufferPage nbp;
		    _pDBMng->bufferFix(nbp, _tabSetId, leafNextFileId, leafNextPageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
		    leaf.setPtr(nbp.getChunkEntry(), nbp.getChunkLen());
		    if ( leaf.valueExists(iv) )
		    {
			hasDuplicate=true;
		    }
		    
		    _pDBMng->bufferUnfix(nbp, false, _pObjMng->getLockHandler());
		}
	
		_pDBMng->bufferUnfix(cbp, false, _pObjMng->getLockHandler());
		
		if ( hasDuplicate == false )
		{		    
		    _pBTO->decreaseRelevance();
		    char *p;
		    int len;
		    CegoBufferPage bp = _pObjMng->claimDataPtrUnlocked(_tabSetId, 
								       CegoBufferPool::NOSYNC,
								       sysEntry, p, len);
		    _pBTO->encode(p);
		    _pObjMng->releaseDataPtrUnlocked(bp, true);
		    
		}
	    }
	}
    }
    catch ( Exception e ) 
    {
	CegoBufferPage bp;
	while ( parentPageStack.Pop(bp) )
	{
	    _pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
	}
	
	_pObjMng->getLockHandler()->unlockData(CegoObject::BTREE, lockId);
	
	throw e;
    }

    // unfix node pages on stack
    CegoBufferPage bp;
    while ( parentPageStack.Pop(bp) )
    {
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
    }

    _pObjMng->getLockHandler()->unlockData(CegoObject::BTREE, lockId);
}

bool CegoBTreeManager::verifyBTree()
{
    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
        
    if ( fileId == 0 && pageId == 0 )
    {
	return false;
    }
    return verifyNode(fileId, pageId);
}

void CegoBTreeManager::allocPage(CegoBufferPage::PageType type, CegoBufferPage& bp)
{
    bool doAppend = true;
    _pObjMng->getNewFilePage(bp, _tabSetId, CegoObject::BTREE, true, doAppend);		    
    bp.initPage(type);

    // cout << "Allocated page " << bp.getFileId() << " / " << bp.getPageId() << endl;
    if ( _pCache )
    {
	CegoBufferPage cachePage = _pCache->newCachePage(bp, false);
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
	bp = cachePage;
    }
}

void CegoBTreeManager::getPage(CegoBufferPage& bp, int fileId, int pageId)
{
    if ( _pCache )
    {
	if ( _pCache->getCachePage(fileId, pageId, bp) == 0 )
	{
	    // cout << "NOT FOUND " << fileId << " / " << pageId << endl;
	    CegoBufferPage poolPage;
	    _pDBMng->bufferFix(poolPage, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
	    bp = _pCache->newCachePage(poolPage, true);	    
	    _pDBMng->bufferUnfix(poolPage, true, _pObjMng->getLockHandler());
	}	
    }
    else
    {	
	_pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
    }
}

void CegoBTreeManager::putPage(CegoBufferPage& bp)
{
    if ( _pCache )
    {
	_pCache->updateCachePage(bp);
    }
    else
    {
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
    }
}

bool CegoBTreeManager::verifyNode(int fileId, int pageId)
{
    CegoBufferPage bp;
    
    _pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());

    int numError = 0;
    
    CegoBTreeNode traceNode;
    traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
    traceNode.setSchema(&_btreeSchema, _keyLen);
    traceNode.setFileId(fileId);
    traceNode.setPageId(pageId);
    traceNode.setNextFileId(bp.getNextFileId());
    traceNode.setNextPageId(bp.getNextPageId());

    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
    {
	traceNode.setType(CegoBTreeNode::NODE);	
	if ( traceNode.verify() == false )
	    numError++;
	
	int childFid, childPid;
	
	traceNode.reset();
	bool nodeValid = true;
	while ( traceNode.nextChildPointer(childFid, childPid) && nodeValid)
	{	    
	    if ( (nodeValid = verifyNode(childFid, childPid)) == false )
		numError++;
	}
    }
    else
    {
	traceNode.setType(CegoBTreeNode::LEAF);
	if ( traceNode.verify() == false )
	    numError++;
    }
   
    _pDBMng->bufferUnfix(bp, false, _pObjMng->getLockHandler());
    
    return numError == 0;
}

void CegoBTreeManager::dumpBTree()
{
    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
    
    cout << "Root Page is " << fileId << " " << pageId << endl;
    
    if ( fileId == 0 && pageId == 0 )
    {
	throw Exception(EXLOC, Chain("Btree object is not valid"));
    }

    dumpNode(0, fileId, pageId);
}

void CegoBTreeManager::dumpNode(int level, int fileId, int pageId)
{
    CegoBufferPage bp;
    
    _pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());

    CegoBTreeNode traceNode;
    traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
    traceNode.setSchema(&_btreeSchema, _keyLen);
    traceNode.setFileId(fileId);
    traceNode.setPageId(pageId);
    traceNode.setNextFileId(bp.getNextFileId());
    traceNode.setNextPageId(bp.getNextPageId());

    cout << "PagePtr = " << (long long)bp.getPagePtr() << endl;
    
    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
    {
	traceNode.setType(CegoBTreeNode::NODE);	
	traceNode.printNode(level);
	
	int childFid, childPid;
	
	traceNode.reset();
	while ( traceNode.nextChildPointer(childFid, childPid) )
	{
	    dumpNode(level + 3, childFid, childPid);  
	}
    }
    else
    {
	traceNode.setType(CegoBTreeNode::LEAF);
	traceNode.printNode(level);
    }
   
    _pDBMng->bufferUnfix(bp, false, _pObjMng->getLockHandler());
}

int CegoBTreeManager::freeBTree()
{
    int firstLeafFileId = 0, firstLeafPageId = 0;
    bool isFirst = true;

    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();

    int pageCount;

    if ( fileId == 0 && pageId == 0 )
    {
	pageCount = 0;
    }
    else
    {
       
	pageCount = freeNodePages(fileId, pageId, firstLeafFileId, firstLeafPageId, isFirst);
	pageCount += freeLeafPages(firstLeafFileId, firstLeafPageId);
    }
    // cout << "Freeing leaf pages starting with " << firstLeafFileId << "/" << firstLeafPageId << endl;

    return pageCount;
}

int CegoBTreeManager::freeNodePages(int fileId, int pageId, int& firstLeafFileId, int& firstLeafPageId, bool& isFirst)
{
    int pageCount = 0;

    CegoBufferPage bp;    
    _pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());

    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
    {
	CegoBTreeNode traceNode;
	traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
	traceNode.setSchema(&_btreeSchema, _keyLen);
	traceNode.setFileId(fileId);
	traceNode.setPageId(pageId);	
	traceNode.setType(CegoBTreeNode::NODE);	

	int childFid, childPid;
	
	traceNode.reset();

	// if the node is not on the left edge, we ignore the first entry.
	// the child nodes is already referenced by the adjacent left node on the same level.
       
	if ( isFirst == false )
	    traceNode.nextChildPointer(childFid, childPid);
	
	while ( traceNode.nextChildPointer(childFid, childPid) )
	{	    
	    pageCount += freeNodePages(childFid, childPid, firstLeafFileId, firstLeafPageId, isFirst);
	}

	// cout << "Freeing node page " << fileId << "/" << pageId << endl;
	pageCount++;
	_pDBMng->bufferRelease(bp, _pObjMng->getLockHandler());
    }
    else
    {
	if ( isFirst )
	{
	    firstLeafFileId = fileId;
	    firstLeafPageId = pageId;
	    isFirst=false;
	}
	_pDBMng->bufferUnfix(bp, true, _pObjMng->getLockHandler());
    }
    
    return pageCount;
}

int CegoBTreeManager::freeLeafPages(int fileId, int pageId)
{
    int pageCount = 0;

    while ( fileId || pageId )
    {
	// cout << "Freeing leaf page " << fileId << "/" << pageId << endl;

	CegoBufferPage bp;    
	_pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
	fileId=bp.getNextFileId();
	pageId=bp.getNextPageId();
	pageCount++;
       	
	_pDBMng->bufferRelease(bp, _pObjMng->getLockHandler());
    }
    
    return pageCount;
}

int CegoBTreeManager::getNumPages()
{    
    int firstLeafFileId = 0, firstLeafPageId = 0;
    bool isFirst = true;

    int fileId = _pBTO->getDataFileId();
    int pageId = _pBTO->getDataPageId();
    
    int pageCount;

    if ( fileId == 0 && pageId == 0 )
    {
	pageCount = 0;
    }
    else
    {
	int numNodePage = countNodePages(fileId, pageId, firstLeafFileId, firstLeafPageId, isFirst);
	int numLeafPage = countLeafPages(firstLeafFileId, firstLeafPageId);
	// cout << "Num Node Pages = " << numNodePage << endl;
	// cout << "Num Leaf Pages = " << numLeafPage << endl;
	pageCount = numNodePage + numLeafPage;
    }
    // cout << "Freeing leaf pages starting with " << firstLeafFileId << "/" << firstLeafPageId << endl;
    return pageCount;
}

int CegoBTreeManager::countNodePages(int fileId, int pageId, int& firstLeafFileId, int& firstLeafPageId, bool& isFirst)
{
    int pageCount = 0;

    CegoBufferPage bp;
   
    _pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());

    if ( bp.getType() == CegoBufferPage::BTREE_NODE )
    {
	CegoBTreeNode traceNode;
	traceNode.setPtr(bp.getChunkEntry(), bp.getChunkLen());	    
	traceNode.setSchema(&_btreeSchema, _keyLen);
	traceNode.setFileId(fileId);
	traceNode.setPageId(pageId);	
	traceNode.setType(CegoBTreeNode::NODE);	

	int childFid, childPid;
	
	traceNode.reset();
	if ( isFirst == false )
	    traceNode.nextChildPointer(childFid, childPid);
	
	while ( traceNode.nextChildPointer(childFid, childPid) )
	{
	    pageCount += countNodePages(childFid, childPid, firstLeafFileId, firstLeafPageId, isFirst);
	}
	pageCount++;
    }
    else
    {
	if ( isFirst )
	{
	    firstLeafFileId = fileId;
	    firstLeafPageId = pageId;
	    isFirst=false;
	}	
    }
   
    _pDBMng->bufferUnfix(bp, false, _pObjMng->getLockHandler());

    return pageCount;
}

int CegoBTreeManager::countLeafPages(int fileId, int pageId)
{
    int pageCount = 0;

    while ( fileId || pageId )
    {	
	CegoBufferPage bp;    
	_pDBMng->bufferFix(bp, _tabSetId, fileId, pageId, CegoBufferPool::NOSYNC, _pObjMng->getLockHandler());
	fileId=bp.getNextFileId();
	pageId=bp.getNextPageId();
	pageCount++;
       	
	_pDBMng->bufferUnfix(bp, false, _pObjMng->getLockHandler());
    }
    
    return pageCount;
}
