///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDistCursor.cc
// -----------------
// Cego global table cursor class implementation
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoDistCursor
// 
// Description: Table cursor for distributed table access
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Net.h>
// #include <lfcbase/Sleeper.h>

// CEGO INCLUDES
#include "CegoDistCursor.h"
#include "CegoDistManager.h"
#include "CegoSelect.h"
#include "CegoTableObject.h"
#include "CegoJoinObject.h"
#include "CegoXMLdef.h"

CegoDistCursor::CegoDistCursor()
{
}

CegoDistCursor::CegoDistCursor(CegoDistManager* pGTM, CegoContentObject *pCO)
{
    
    _isLocal = true;
    _moreTuple = false;
    
    _pSelect = 0;
    _pSH = 0;
    _pGTM = pGTM;

    _cursorObjectUsed = false;
    
    _tableName = pCO->getTabName();
    _tableAlias = pCO->getName();
    _tabSetId = pCO->getTabSetId();
    
    _pDBMng = _pGTM->getDBMng(); 

    _pCache = _pDBMng->getTableCache(_tabSetId);
    _isCached = false;
    _pCacheArray = 0;
    _pCacheList = 0;
    _pCO = pCO;
    
    _pTCLeft = 0;
    _pTCRight = 0;
    _pTC = 0;
    _pC = 0;

    _idxMatch = CegoAttrCond::INAPP;

    checkType();
    
    _modId = _pDBMng->getModId("CegoDistCursor");

}

CegoDistCursor::~CegoDistCursor()
{
    
    finishCaching();
    
    if ( _pTC )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + _tableName);
#endif

	delete _pTC;
    }
    if ( _pSelect )
    {

	try
	{
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Cleaning up select ..."));
#endif
	    _pSelect->cleanUp();
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, Chain("Ignoring error : ") + msg);
	}
    }
    if ( _pCO->getType() == CegoObject::JOIN )
    {

	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;

	CegoContentObject *pCOLeft = pJCO->getLeftObject();
	CegoContentObject *pCORight = pJCO->getRightObject();

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + pCOLeft->getTabName());
#endif
	
	delete _pTCLeft;
	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing global table cursor for ") + pCORight->getTabName());
#endif
	
	delete _pTCRight;
		
    }

    if ( _pC )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Releasing object cursor for ") + _tableName);
#endif
	delete _pC;
    }

    unuseCursorObject();
    
    if ( _pSH )
    {
	_pDBMng->releaseSession(_pSH);
    }
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Destructor for DistCursor finished"));
#endif

    if ( _pCacheList )
    {
	delete _pCacheList;
    }

    if ( _pCacheArray && _pCache )
    {
	// FOR DEBUGGING : sleep for a while
	// cout << "Sleeping for 10 sec before destructing cursor  ..." << endl;
	// Sleeper s;
	// s.secSleep(10);

	_pCache->releaseEntry(_tabSetId, _tableName);
	_pCacheArray=0;
	_isCached = true;
    }
}

void CegoDistCursor::useCursorObject()
{
    if ( _cursorObjectUsed == false )
    {
	if ( _pCO->getType() == CegoObject::VIEW )
	{
	    _pDBMng->useObject(_tabSetId, _tableName, CegoObject::VIEW, CegoDatabaseManager::SHARED, _pGTM->getThreadId());	
	}
	else if ( _pCO->getType() == CegoObject::TABLE )
	{
	    _pDBMng->useObject(_tabSetId, _tableName, CegoObject::TABLE, CegoDatabaseManager::SHARED, _pGTM->getThreadId());	
	}
	else if ( _pCO->getType() == CegoObject::JOIN )
	{
	    CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	    CegoContentObject *pCOLeft = pJCO->getLeftObject();
	    CegoContentObject *pCORight = pJCO->getRightObject();

	    if ( pCOLeft->getType() == CegoObject::VIEW || pCOLeft->getType() == CegoObject::TABLE )
		_pDBMng->useObject(pCOLeft->getTabSetId(), pCOLeft->getTabName(), pCOLeft->getType(), CegoDatabaseManager::SHARED, _pGTM->getThreadId());	
	    if ( pCORight->getType() == CegoObject::VIEW || pCORight->getType() == CegoObject::TABLE )
		_pDBMng->useObject(pCORight->getTabSetId(), pCORight->getTabName(), pCORight->getType(), CegoDatabaseManager::SHARED, _pGTM->getThreadId());	
	}

	_cursorObjectUsed = true;
    }
}

void CegoDistCursor::unuseCursorObject()
{
    if ( _cursorObjectUsed == true )
    {
	if ( _pCO->getType() == CegoObject::VIEW )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::VIEW);	
	}
	else if ( _pCO->getType() == CegoObject::TABLE )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::TABLE);	
	}
	else if ( _pCO->getType() == CegoObject::JOIN )
	{	
	    CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	    CegoContentObject *pCOLeft = pJCO->getLeftObject();
	    CegoContentObject *pCORight = pJCO->getRightObject();

	    if ( pCOLeft->getType() == CegoObject::VIEW || pCOLeft->getType() == CegoObject::TABLE )
		_pDBMng->unuseObject(pCOLeft->getTabSetId(), pCOLeft->getTabName(), pCOLeft->getType());	
	    if ( pCORight->getType() == CegoObject::VIEW || pCORight->getType() == CegoObject::TABLE )
		_pDBMng->unuseObject(pCORight->getTabSetId(), pCORight->getTabName(), pCORight->getType());	
	}

	_cursorObjectUsed = false;
    }
}

void CegoDistCursor::checkType()
{
    useCursorObject();
    
    if ( _pCO->getType() == CegoObject::VIEW )
    {

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("View ") + _tableName + Chain(" detected as the cursor source"));
#endif

	CegoView *pView;

	try
	{
	    pView = _pGTM->getView(_tabSetId, _tableName);
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(_tabSetId, _tableName, CegoObject::VIEW);
	    throw e;
	}
	_pSelect = pView->getSelect();
	_pSelect->cleanUp();
	
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Table ") + _tableName + Chain(" detected as the cursor source"));
#endif

	Chain tableSet = _pDBMng->getTabSetName(_tabSetId);

	CegoTableObject oe;
	_pGTM->getDistObject(tableSet, _tableName, CegoObject::TABLE, oe);
	
	if ( oe.isLocal() )
	{

#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Native Table ") + _tableName + Chain(" detected as the cursor source"));
#endif

	    _pTC = new CegoTableCursor(_pGTM, _tabSetId, _tableName, false);	

	    _cacheSchema = oe.getSchema();
	    _isLocal = true;
	}
	else
	{

#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Remote Table ") + _tableName + Chain(" detected as the cursor source"));
#endif
	    _isLocal = false;

	    int tabSetId = oe.getTabSetId();
	    Chain hostName = _pDBMng->getPrimary(tabSetId);
	    int portNo;
	    _pDBMng->getDataPort(portNo);

	    _schema = oe.getSchema();

	    Chain user;
	    Chain password;
	    
	    _pGTM->getActiveUser(tableSet, user, password);
	    
	    _pSH = _pDBMng->allocateSession(hostName, tableSet, user, password);
	    	    
	    _pSH->reqTableDataOp(tabSetId, _tableName,  CegoObject::TABLE);

	}      
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	

	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	CegoContentObject *pCOLeft = pJCO->getLeftObject();
	CegoContentObject *pCORight = pJCO->getRightObject();
	
	_pTCLeft = new CegoDistCursor(_pGTM, pCOLeft);
	_pTCRight = new CegoDistCursor(_pGTM, pCORight);
	
    }

    // we have to release object here to avoid dead locks
    unuseCursorObject();    
}

void CegoDistCursor::distSetup()
{

    _isFirst=true;
    _doEval = false;
    _idxMatch = CegoAttrCond::INAPP;
    _isAttrCondValid = false;

    if ( _pCO->getType() == CegoObject::VIEW )
    {
	_pSelect->setTabSetId(_tabSetId);
	CegoAttrCond noCond;
	_pSelect->setViewCond(noCond);
	_pSelect->prepare();
	_pSelect->setParentJoinBuf();

	_pSelect->checkValidRef();
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {	
	CegoAttrCond attrCond; // empty cond
	_pTC->setup(attrCond);
	
	if ( _pCache )
	{

	    _useCache = true;
	    
	    finishCaching();

	    _pCacheArray = _pCache->claimEntry( _tabSetId, _tableName, _cacheRows, _cacheCols);

	    if ( _pCacheArray )
	    {
		_isCached = true;
	    }
	    else
	    {
		_isCached = false;
		_cacheEntrySize = 0;
		_pCacheList = new ListT< ListT< CegoFieldValue > >;
	    }
	}	
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	sysSetup();
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {
	CegoAttrCond attrCond;	
	joinSetup(attrCond);
    }
}

void CegoDistCursor::distSetup( const CegoAttrCond& attrCond)
{
    if ( attrCond.numComp() == 0 )
	return distSetup();

    _isAttrCondValid = false;
    _idxMatch = CegoAttrCond::INAPP;
    _isFirst=true;
    _doEval = false;
    
    if ( _pCO->getType() == CegoObject::VIEW )
    {	
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting view condition ") + attrCond.toChain());
#endif

	_pSelect->setViewCond(attrCond);
	_pSelect->setTabSetId(_tabSetId);
	_pSelect->prepare();
	_pSelect->setParentJoinBuf();

	_pSelect->checkValidRef();

	_cursorCond = attrCond;

	_doEval=false;
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal == true )
	{

	    if ( _pCache )
	    {
		// check if a previous cursor trace has to be finished to complete cache 
		finishCaching();
	    }
	    
	    _useCache = false;
		
	    _idxMatch = _pTC->setup(attrCond);
	    
	    if ( _idxMatch != CegoAttrCond::FULL )
	    {	       
		_doEval=true;

		// if index match was inappropriate, all tuples are retrieved from table cursor
		// In this case, we can use cache
		if ( _pCache && _idxMatch == CegoAttrCond::INAPP )
		{
		    _useCache = true;
		    _pCacheArray = _pCache->claimEntry( _tabSetId, _tableName, _cacheRows, _cacheCols);
		    
		    if ( _pCacheArray )
		    {
			_isCached = true;
		    }
		    else
		    {
			_isCached = false;
			_cacheEntrySize = 0;
			_pCacheList = new ListT< ListT< CegoFieldValue > >;
		    }
		}
	    }
	    
	    _cursorCond = attrCond;
	   
	}
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	sysSetup();
	_cursorCond = attrCond;
	_doEval = true;	
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting join condition ") + attrCond.toChain());	
#endif

	_cursorCond = attrCond;
	_doEval = true;
	joinSetup(attrCond);
    }
}

bool CegoDistCursor::nextTuple(ListT<CegoField>** pFLA, int pos, int size)
{
    useCursorObject();
    
    while ( getTuple(pFLA, pos, size) )
    {	
	if ( _doEval == true )
	{
	    if ( evalCondition(pFLA, pos, size) )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Returning tuple with eval condition"));
#endif

		return true;
	    }
	}
	else
	{
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Returning tuple without eval condition"));
#endif

	    return true;
	}
    }

    return false;
}

bool CegoDistCursor::evalCondition(ListT<CegoField>** pFLA, int pos, int size)
{
    CegoAttrComp *pAC = _cursorCond.getAttrCompSet().First();
    
    while ( pAC )
    {

	CegoField *pF = 0;
	    
	int i=0;
	while ( pF == 0 && i < size)
	{	   
	    if ( _pCO->getType() == CegoObject::VIEW || _pCO->getType() == CegoObject::TABLE )
		pF = pFLA[pos+i]->Find( CegoField(_tableAlias, pAC->getAttrName()));
	    else
		pF = pFLA[pos+i]->Find( CegoField(Chain(), pAC->getAttrName()) );
	    i++;
	}
	
	if ( pF )
	{
	    if ( pAC->getCompMode() == CegoAttrComp::BTWN )
	    {
		if ( ( (CegoFieldValue)pF->getValue() >= (CegoFieldValue)(pAC->getFieldValue()) 
		       && (CegoFieldValue)pF->getValue() <= (CegoFieldValue)(pAC->getFieldValue2()) ) == false )
		    
		    return false;	    
	    }
	    else if ( pAC->getCompMode() == CegoAttrComp::ISLIKE )
	    {
		if ( pAC->getMatcher()->match(pF->getValue().valAsChain()) == false)		
		    return false;
	       
	    }
	    else if ( pAC->getCompMode() == CegoAttrComp::ISNOTLIKE )
	    {
		if ( pAC->getMatcher()->match(pF->getValue().valAsChain()))
		    return false;		
	    }
	    else
	    {
		switch ( pAC->getComparison() )
		{	    
		case EQUAL:
		{
		    if ( ( (CegoFieldValue)pF->getValue() == (CegoFieldValue)(pAC->getFieldValue()) ) == false )		    
			return false;		    
		    break;
		}
		case NOT_EQUAL:
		{
		    if ( ( (CegoFieldValue)pF->getValue() != (CegoFieldValue)(pAC->getFieldValue()) ) == false )		    
			return false;	    
		    break;
		}
		case LESS_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() < (CegoFieldValue)(pAC->getFieldValue()) ) == false )		     
			return false;
		    break;
		}
		case MORE_THAN: 
		{
		    if ( ( (CegoFieldValue)pF->getValue() > (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;
		    break;
		}
		case LESS_EQUAL_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() <= (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;
		    break;
		}
		case MORE_EQUAL_THAN:
		{
		    if ( ( (CegoFieldValue)pF->getValue() >= (CegoFieldValue)(pAC->getFieldValue()) ) == false )
			return false;	    
		    break;
		}
		}
	    }
	}
	else
	{
	    if ( pAC->isSetup() == false )
	    {
		Chain msg = Chain("Cannot get value for attribute ") + pAC->getAttrName();
		throw Exception(EXLOC, msg);
	    }
	}
	pAC = _cursorCond.getAttrCompSet().Next();
    }
    return true;
}

bool CegoDistCursor::getTuple(ListT<CegoField>** pFLA, int offset, int size)
{
    if ( _pCO->getType() == CegoObject::VIEW )
    {
	try 
	{	    
	    _moreTuple = _pSelect->nextTuple(*pFLA[offset]);
	    
	    if ( _moreTuple )
	    {
		CegoField *pF = pFLA[offset]->First();
		while ( pF ) 
		{
		    pF->setTableName(_tableName);
		    pF->setTableAlias(_tableAlias);
		    pF = pFLA[offset]->Next();
		}
	    }
	    return _moreTuple;
	}
	catch ( Exception e )
	{
	    _pSelect->cleanUp();
	    throw e;
	}
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal )
	{
	    if ( _useCache )
	    {
		if ( _isCached )
		{		    
		    CegoFieldValue** pCacheRow = 0;
		    if ( _isFirst ) 
		    {
			_isFirst = false;
			_cachePos = 0;
			if ( _cacheRows > 0 )
			{
			    pCacheRow = _pCacheArray[_cachePos];
			    _cachePos++;			
			}			  
		    }
		    else
		    {
			if ( _cachePos < _cacheRows )
			{			    
			    pCacheRow = _pCacheArray[_cachePos];
			    _cachePos++;
			}
		    }
		    	
		    if ( pCacheRow  )
		    {
			CegoField *pF = pFLA[offset]->First();
			
			while ( pF )
			{
			    CegoField *pSF = _cacheSchema.First();
			    
			    int pos = 0;
			    while ( pF && pSF && ( *pF != *pSF ) )
			    {				
				pSF = _cacheSchema.Next();
				pos++;
			    }
			    if ( *pF == *pSF )
			    {
				pF->setValue(*pCacheRow[pos]);
			    }
			    pF = pFLA[offset]->Next();
			}

			_moreTuple = true;
		    }
		    else
		    {
			
			if ( _pCacheArray && _pCache )
			{
			    // FOR DEBUGGING : sleep for a while
			    // cout << "Sleeping for 10 sec before end of cursor  ..." << endl;
			    // Sleeper s;
			    // s.secSleep(10);
			    
			    _pCache->releaseEntry( _tabSetId, _tableName);
			    _pCacheArray = 0;
			    _isCached = true;
			}

			_moreTuple = false;
		    }

		    return _moreTuple;
		}
		else // still not cached, so we have to retrieve and cache it
		{

		    CegoDataPointer dp;
		    if ( _isFirst )
		    {
			_moreTuple = _pTC->getFirst(_cacheSchema, dp);			
			_isFirst = false;
		    }
		    else
		    {
			_moreTuple = _pTC->getNext(_cacheSchema, dp);	    
		    }
		    
		    if ( _moreTuple )
		    {
			// put tuple into cache

			if ( _pCacheList )
			{
			    ListT<CegoFieldValue> staticFieldList;
			    CegoField* pF = _cacheSchema.First();
			    while ( pF )
			    {			
				staticFieldList.Insert(pF->getValue().getLocalCopy());			       
				_cacheEntrySize += pF->getValue().size();
				pF = _cacheSchema.Next();
			    }

			    if ( _pCache->getMaxSize() > _cacheEntrySize )
			    {
				_pCacheList->Insert(staticFieldList);
			    }
			    else
			    {
				delete _pCacheList;
				_pCacheList = 0;
			    }
			}

			// propagate  tuple to flArray
			CegoField *pF = pFLA[offset]->First();			
			while ( pF )
			{
			    CegoField *pSF = _cacheSchema.First();
			    while ( pF && pSF && ( *pF != *pSF ) )
			    {
				pSF = _cacheSchema.Next();
			    }
			    // cout << "Comparing " << pF->getTableAlias() << "." << pF->getAttrName()  << " and " << pSF->getTableAlias() << "." << pSF->getAttrName() << endl;
			    if ( *pF == *pSF )
			    {
				// cout << "MATCH" << endl;
				pF->setValue(pSF->getValue());
			    }
			    pF = pFLA[offset]->Next();
			}
		    }
		    else
		    {

			if ( _pCacheList )
			{
			    // put table cache entry into cache
			    // cout << "PUT TCE " << _tableName << " into cache" << endl;
			    _pCache->addEntry( _tabSetId, _tableName, _pCacheList );
			    // cout << "PUT DONE" << endl;
			    _isCached = true;
			    delete _pCacheList;
			    _pCacheList = 0;
			}
		    }

		    return _moreTuple;

		}
	    }
	    else // retrieve tuple via native table cursor
	    {
		// cout << "Retrieve native .." << endl;
		CegoDataPointer dp;
		if ( _isFirst )
		{
		    _moreTuple = _pTC->getFirst(*pFLA[offset], dp);
		    _isFirst = false;
	        }
		else
		{
		    _moreTuple = _pTC->getNext(*pFLA[offset], dp);	    
		}
		
		return _moreTuple;
	    }
	}
	else
	{
	    *pFLA[offset] = _schema;
	    _moreTuple = false;

	    if ( _pSH->receiveTableData(*pFLA[offset]) == CegoDbHandler::DB_DATA )
	    {
		_moreTuple = true;
	    }

	    return _moreTuple;	    
	}
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {

	if ( _pC )
	{

	    CegoDataPointer dp;

	    int len;
	    char *pc;
	    if ( _isFirst )
	    {
		pc = (char*)_pC->getFirst(len, dp);		
		_isFirst=false;
	    }
	    else
	    {
		pc = (char*)_pC->getNext(len, dp);
	    }

	    if ( pc && len > 0 )
	    {

		unsigned long long tid;
		unsigned long long tastep;
		CegoTupleState ts;

		int toff = CegoQueryHelper::decodeTupleHeader(tid, tastep, ts, pc);

		char* tp = pc + toff;
		int tlen = len - toff;

		CegoQueryHelper::decodeFVL(*pFLA[offset], tp, tlen);
		
		return true;   
	    }
	    else
	    {
		return false;
	    }
	}
	else
	{
	
	    Chain *pName = 0;
	    if ( _isFirst )
	    {
		_isFirst = false;
		pName = _sysObjList.First();
	    }	
	    else
	    {
		pName = _sysObjList.Next();	    
	    }
	    
	    if ( pName )
	    {
		CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_NAME_ATTR)));
		if ( pF )		    
		{
		    pF->setValue( CegoFieldValue(VARCHAR_TYPE, *pName));
		}
		
		if ( _tableName == Chain(SYSTAB_TABLE_ID) 
		     || _tableName == Chain(SYSTAB_INDEX_ID)
		     || _tableName == Chain(SYSTAB_BTREE_ID) )
		{
		    
		    int pageCount;
		    
		    if ( _tableName == Chain(SYSTAB_TABLE_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::TABLE);
		    else if ( _tableName == Chain(SYSTAB_INDEX_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::AVLTREE);
		    else if ( _tableName == Chain(SYSTAB_BTREE_ID)  )
			pageCount = _pGTM->getPageCount(_tabSetId, *pName, CegoObject::BTREE);

		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_SIZE_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(INT_TYPE, pageCount));
		    }
		    pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )
		    {
			if ( pageCount > 0 )
			    pF->setValue( CegoFieldValue(VARCHAR_TYPE, Chain("valid")));
			else
			    pF->setValue( CegoFieldValue(VARCHAR_TYPE, Chain("invalid")));
		    }
		    
		}
		else if ( _tableName == Chain(SYSTAB_VIEW_ID) )
		{
		    Chain status("not compiled");
		    if ( _pGTM->checkCompView(_tabSetId, *pName) )
			status = Chain("compiled");
		    
		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(VARCHAR_TYPE, status));
		    }		
		}
		else if ( _tableName == Chain(SYSTAB_PROC_ID) )
		{
		    Chain status("not compiled");
		    if ( _pGTM->checkCompProcedure(_tabSetId, *pName) )
			status = Chain("compiled");
		    
		    CegoField *pF = pFLA[offset]->Find(CegoField(_tableAlias, Chain(SYSTAB_STATUS_ATTR)));
		    if ( pF )		    
		    {
			pF->setValue( CegoFieldValue(VARCHAR_TYPE, status));
		    }				
		}
		_moreTuple = true;	    
	    }
	    else
	    {
		_moreTuple = false;
	    }
	}

	return _moreTuple;
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	
	if ( _isFirst )
	{
	    _moreLeft = true;
	    _moreRight = true;
	}

	CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
	
	if ( pJO->getJoinType() == CegoJoinObject::INNER )
	{
	    
	    bool isEvaluated = false;
	    
	    while ( isEvaluated == false && _moreLeft )
	    {
		if ( _isFirst )
		{
		    _pTCLeft->distSetup(_outerCond);
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {			
			_innerCond.setup(pFLA, offset);

			_pTCRight->reset();
			
			if ( _isAttrCondValid )
			    _pTCRight->distSetup(_innerCond);
			else
			    _pTCRight->distSetup();

			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }

		    _isFirst = false;
		}
		else if ( _moreRight )
		{

		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    if ( _moreRight == false )
		    {
			_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
			if ( _moreLeft )
			{
			    _innerCond.setup(pFLA, offset);

			    _pTCRight->reset();

			    if ( _isAttrCondValid )
				_pTCRight->distSetup(_innerCond);
			    else
				_pTCRight->distSetup();

			    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
			}
		    }
		}
		else if ( _moreLeft )
		{
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {
			_innerCond.setup(pFLA, offset);

			_pTCRight->reset();

			if ( _isAttrCondValid )
			    _pTCRight->distSetup(_innerCond);
			else
			    _pTCRight->distSetup();

			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }
		}

		if ( _moreLeft && _moreRight )
		{
		    if ( _evalPredicate )
		    {		       
			
			pJO->getPredDesc()->clearAttrCache();
			
			isEvaluated = CegoQueryHelper::evalPredicate(0,
								     0,
								     pFLA,
								     offset,
								     pJO->getPredDesc(),
								     0);
		    }
		    else
		    {			
			isEvaluated = true;
		    }
		}
	    }	    
	    
	    return _moreLeft && _moreRight;
	}
	else if ( pJO->getJoinType() == CegoJoinObject::LEFTOUTER )
	{

	    bool isEvaluated = false;
	    
	    while ( isEvaluated == false )
	    {
		
		if ( _isFirst )
		{
		    _pTCLeft->distSetup(_outerCond);
		    
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    
		    if ( _moreLeft )
		    {
			_innerCond.setup(pFLA, offset);
			
			_pTCRight->reset();
			
			if ( _isAttrCondValid )
			{
			    _pTCRight->distSetup(_innerCond);
			}
			else
			{
			    _pTCRight->distSetup();
			}
			
			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }
		    _isFirst = false;
		}
		else if ( _moreRight )
		{
		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    
		    if ( _moreRight == false )
		    {
			_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
			if ( _moreLeft )
			{
			    _innerCond.setup(pFLA, offset);
			    
			    _pTCRight->reset();
			    
			    if ( _isAttrCondValid )
			    {
				_pTCRight->distSetup(_innerCond);
			    }
			    else
			    {
				_pTCRight->distSetup();
			    }
			    
			    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
			}
		    }		
		}
		else if ( _moreLeft )
		{
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    if ( _moreLeft )
		    {
			_innerCond.setup(pFLA, offset);
			
			_pTCRight->reset();
			
			if ( _isAttrCondValid )
			{
			    _pTCRight->distSetup(_innerCond);
			}
			else
			{
			    _pTCRight->distSetup();
			}
			
			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    }
		}
		
		if ( _evalPredicate && _moreLeft && _moreRight )
		{		    
		    pJO->getPredDesc()->clearAttrCache();
		    isEvaluated = CegoQueryHelper::evalPredicate(0,
								 0,
								 pFLA,
								 offset,
								 pJO->getPredDesc(),
								 0);		   
		}
		else
		{
		    isEvaluated = true;
		}
	    }
	    	    
	    if ( _moreLeft )
	    {
		if ( _moreRight == false )
		{
		    // make right fields null
		    CegoField *pF = pFLA[offset+size-1]->First();
		    while ( pF )
		    {
			CegoFieldValue nullValue;
			pF->setValue(nullValue);
			pF = pFLA[offset+size-1]->Next();
		    }
		}
		return true;
	    }
	    return false;
	}	
	else if ( pJO->getJoinType() == CegoJoinObject::RIGHTOUTER )
	{

	    bool isEvaluated = false;
	    
	    while ( isEvaluated == false )
	    {

		if ( _isFirst )
		{		
		    _pTCRight->distSetup(_cursorCond);
		    
		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    
		    if ( _moreRight )
		    {
			_innerCond.setup(pFLA, offset+size-1);

			_pTCLeft->reset();
			
			if ( _isAttrCondValid )
			    _pTCLeft->distSetup(_innerCond);
			else
			    _pTCLeft->distSetup();

			_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);

		    }
		    _isFirst = false;
		}
		else if ( _moreLeft )
		{
		    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    
		    if ( _moreLeft == false )
		    {
			_moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
			if ( _moreRight )
			{
			    _innerCond.setup(pFLA, offset+size-1);
			    
			    _pTCLeft->reset();
			    
			    if ( _isAttrCondValid )
				_pTCLeft->distSetup(_innerCond);
			    else
				_pTCLeft->distSetup();
			    
			    _moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
			}
		    }		
		}
		else if ( _moreRight )
		{
		    _moreRight = _pTCRight->nextTuple(pFLA, offset+size-1, 1);
		    if ( _moreRight )
		    {
			_innerCond.setup(pFLA, offset+size-1);
			
			_pTCLeft->reset();
			
			if ( _isAttrCondValid )
			    _pTCLeft->distSetup(_innerCond);
			else
			    _pTCLeft->distSetup();
			
			_moreLeft = _pTCLeft->nextTuple(pFLA, offset, size-1);
		    }
		}

		if ( _evalPredicate && _moreLeft && _moreRight )
		{		    
		    pJO->getPredDesc()->clearAttrCache();
		    isEvaluated = CegoQueryHelper::evalPredicate(0,
								 0,
								 pFLA,
								 offset,
								 pJO->getPredDesc(),
								 0);		   
		}
		else
		{
		    isEvaluated = true;
		}
	    }

	    if ( _moreRight )
	    {
		if ( _moreLeft == false )
		{
		    for ( int i=0; i<size-1; i++)
		    {
			// make right fields null
			CegoField *pF = pFLA[offset+i]->First();
			while ( pF )
			{
			    CegoFieldValue nullValue;
			    pF->setValue(nullValue);
			    pF = pFLA[offset+i]->Next();
			}
		    }
		}
		return true;
	    }
	    return false;
	}	    
    }

    throw Exception(EXLOC, Chain("Cannot resolve object type"));
}

void CegoDistCursor::finishCaching()
{
    
    if ( _useCache && _pCache && _pCacheList && _isCached == false && _pTC && _isFirst == false )
    {	
	CegoDataPointer dp;
	
	while ( _pTC->getNext(_cacheSchema, dp) && _pCacheList )
	{

	    ListT<CegoFieldValue> staticFieldList;
	    CegoField* pF = _cacheSchema.First();
	    while ( pF )
	    {			
		staticFieldList.Insert(pF->getValue().getLocalCopy());			       
		_cacheEntrySize += pF->getValue().size();
		pF = _cacheSchema.Next();
	    }
	    
	    if ( _pCache->getMaxSize() > _cacheEntrySize )
	    {
		_pCacheList->Insert(staticFieldList);
	    }
	    else
	    {
		delete _pCacheList;
		_pCacheList = 0;
	    }
	}
       	    
	if ( _pCacheList )
	{
	    // put table cache entry into cache
	    _pCache->addEntry( _tabSetId, _tableName, _pCacheList );
	    _isCached = true;
	    delete _pCacheList;
	    _pCacheList = 0;
	}
    }
}

void CegoDistCursor::reset()
{
    _isFirst=true;
    _moreTuple = false;

    if ( _pCO->getType() == CegoObject::VIEW )
    {
	_pSelect->reset(true);
    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {
	if ( _isLocal )
	{
	    if ( _pTC )
		_pTC->abort();

	    // we have to check, if cache was created
	    if ( _isCached == false && _pCacheList )
	    {
		delete _pCacheList;
		_pCacheList = 0;
	    }
	    
	    if ( _pCacheArray && _pCache )
	    {

		// FOR DEBUGGING : sleep for a while
		// cout << "Sleeping for 10 sec before cursor reset  ..." << endl;
		// Sleeper s;
		// s.secSleep(10);

		_pCache->releaseEntry( _tabSetId, _tableName);
		_pCacheArray = 0;
		_isCached = 0;
	    }
	}
	else
	{
	    if ( _pSH )
		_pDBMng->releaseSession(_pSH);
	}
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	
	if ( _pTCLeft )
	    _pTCLeft->reset();
	if ( _pTCRight )
	    _pTCRight->reset();
    }
    else if ( _pCO->getType() == CegoObject::SYSTEM )
    {
	if ( _pC )
	    _pC->reset();
    }

    unuseCursorObject();
}

void CegoDistCursor::sysSetup()
{
    
    Chain tableSet = _pDBMng->getTabSetName(_tabSetId);	
    if ( _tableName == Chain(SYSTAB_TABLE_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::TABLE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_PROC_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::PROCEDURE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_VIEW_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::VIEW, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_INDEX_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::AVLTREE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_BTREE_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::BTREE, _sysObjList);
    }
    else  if ( _tableName == Chain(SYSTAB_KEY_ID) )
    {
	_pGTM->getDistObjectList(tableSet, CegoObject::FKEY, _sysObjList);
    }
    else
    {
	_pC = _pGTM->getObjectCursor(_tabSetId, _tableName, _tableName, CegoObject::SYSTEM);
    }
}

void CegoDistCursor::joinSetup(const CegoAttrCond& attrCond)
{
    
    CegoJoinObject *pJO = (CegoJoinObject*)_pCO;
        
    ListT<CegoField> outerSchema;
    ListT<CegoField> innerSchema;
    
    CegoAttrCond addInnerCond;

    if ( pJO->getJoinType() == CegoJoinObject::INNER || pJO->getJoinType() == CegoJoinObject::LEFTOUTER )
    {
	outerSchema = pJO->getLeftObject()->getSchema();
	innerSchema = pJO->getRightObject()->getSchema(); 

	_outerCond = attrCond.getFilterCond(outerSchema);
	addInnerCond = attrCond.getFilterCond(innerSchema);

    }
    else if (  pJO->getJoinType() == CegoJoinObject::RIGHTOUTER )
    {
	innerSchema = pJO->getLeftObject()->getSchema();
	outerSchema = pJO->getRightObject()->getSchema();

	_outerCond = attrCond.getFilterCond(outerSchema);
	addInnerCond = attrCond.getFilterCond(innerSchema);
    }
    
    CegoAttrCond ac;
    
    CegoQueryHelper::AttrCondMatch m  = CegoQueryHelper::checkAttrCond(ac, pJO->getPredDesc(), innerSchema, &outerSchema, 1, 0);
    
    if ( m == CegoQueryHelper::COMPLETE )
    {
	_evalPredicate = false;
	_innerCond = ac + addInnerCond;
    }
    else
    {
	_evalPredicate = true;
	_innerCond = addInnerCond;
    }

    _isAttrCondValid = true;
    
}

Element* CegoDistCursor::getPlan()
{
    Element *pCursorPlan = new Element(XML_JOIN_ELEMENT);

    pCursorPlan->setAttribute(XML_TABLENAME_ATTR, _tableName);       
    pCursorPlan->setAttribute(XML_NAME_ATTR, _tableAlias);

    if ( _pCO->getType() == CegoObject::VIEW )
    {
	pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_VIEW_VALUE);

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Getting view plan for ") + _tableAlias);
#endif

	pCursorPlan->addContent( _pSelect->getPlan() );


#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("View plan for ") + _tableAlias + Chain(" retrieved"));
#endif

    }
    else if ( _pCO->getType() == CegoObject::TABLE )
    {

	pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_TABLE_VALUE);
	
	if ( _idxMatch == CegoAttrCond::FULL)
	{
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full index trace on ") + _cursorCond.toChain()); 
	}
	else if( _idxMatch == CegoAttrCond::PART )
	{
	    pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("index support on ") + _cursorCond.toChain() + Chain(" using index ") + _pTC->getIndexName()); 
	}
	else
	{
	    if ( _cursorCond.numComp() > 0 )
		pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full table scan using condition ") + _cursorCond.toChain());
	    else
		pCursorPlan->setAttribute(XML_JOINSTRAT_ATTR, Chain("full table scan with no condition "));
	}	
    }
    else if ( _pCO->getType() == CegoObject::JOIN )
    {	

	CegoJoinObject *pJCO = (CegoJoinObject*)_pCO;
	
	if ( pJCO->getJoinType() == CegoJoinObject::INNER )	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_INNERJOIN_VALUE);
	else if ( pJCO->getJoinType() == CegoJoinObject::LEFTOUTER)	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_LEFTOUTERJOIN_VALUE);
	else if ( pJCO->getJoinType() == CegoJoinObject::RIGHTOUTER)	    
	    pCursorPlan->setAttribute(XML_TABLETYPE_ATTR, XML_RIGHTOUTERJOIN_VALUE);
	
	if ( _pTCLeft )
	{
	    _pTCLeft->distSetup(_outerCond);
	}
	if ( _pTCRight )
	{
	    
	    if ( _isAttrCondValid )
	    {
		_pTCRight->distSetup(_innerCond);
	    }
	    else
	    {
		_pTCRight->distSetup();
	    }	    
	}
	
	if ( _pTCLeft )
	    pCursorPlan->addContent( _pTCLeft->getPlan() );
	if ( _pTCRight )
	    pCursorPlan->addContent( _pTCRight->getPlan() );
    }
    else
    {
	delete pCursorPlan;
	throw Exception(EXLOC, Chain("Invalid content type"));		
    }

    return pCursorPlan;
}
