///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoQueryHelper.cc
// ------------------
// Cego query helper implementation
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoQueryHelper
//
// Description: The CegoQueryHelper class provides several utility methods with are used for tuple retrieval and evalution
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/BigDecimal.h>
#include <lfcbase/Datetime.h>

// cego includes
#include "CegoQueryHelper.h"
#include "CegoXMLdef.h"
#include "CegoSelect.h"
#include "CegoTypeConverter.h"

#include <string.h>
#include <stdlib.h>

CegoQueryHelper::CegoQueryHelper()
{
}

CegoQueryHelper::~CegoQueryHelper()
{
}

bool CegoQueryHelper::evalPredicate(ListT<CegoField>** pParentJoinBuf,
				    int parentJoinBufPos,
				    ListT<CegoField>** pJoinBuf,
				    int pos,
				    CegoPredDesc* pP, 
				    CegoProcBlock *pBlock)
{

    if (pP == 0)
	return true;

    CegoCondDesc* pC = pP->getCondition();
    if (pC)
    {
	switch ( pC->getCondType() )
	{
	case CegoCondDesc::AND:
	    return ( evalPredicate(pParentJoinBuf, 
				   parentJoinBufPos,
				   pJoinBuf,
				   pos,
				   pC->Left(), 
				   pBlock)
		     && evalPredicate(pParentJoinBuf,
				      parentJoinBufPos,
				      pJoinBuf,
				      pos,
				      pC->Right(),
				      pBlock));
	case CegoCondDesc::OR:
	    	    
	    return ( evalPredicate(pParentJoinBuf,
				   parentJoinBufPos,
				   pJoinBuf,
				   pos,
				   pC->Left(),
				   pBlock) 
		     || evalPredicate(pParentJoinBuf,
				      parentJoinBufPos,
				      pJoinBuf,
				      pos,
				      pC->Right(),
				      pBlock));
	default:
	    break;	
	}
    } 
    else 
    {
	if ( pP->getMode() == CegoPredDesc::NOTPRED )
	{
	    bool np = evalPredicate(pParentJoinBuf,
				    parentJoinBufPos,
				    pJoinBuf,
				    pos,
				    pP->getNotPred(),
				    pBlock);
	    if ( np )
	    {
		return false;
	    }
	    return true;
	}
	else
	{

	    ListT<CegoField>** flArray;
	    ListT<CegoField>* flStaticArray[TABMNG_MAXJOINLEVEL];

	    if ( pParentJoinBuf )
	    {
		flArray = flStaticArray;
		
		int flpos=0;
		int i=parentJoinBufPos;
		while ( pParentJoinBuf[i] )
		{
		    if ( flpos > TABMNG_MAXJOINLEVEL )
		    {
			Chain msg = Chain("Join buffer exceeded");
			throw Exception(EXLOC, msg);
		    }			
			    
		    flArray[flpos] = pParentJoinBuf[i];
		    i++;
		    flpos++;
		}
		i=pos;
		while ( pJoinBuf[i] )
		{
		    if ( flpos > TABMNG_MAXJOINLEVEL )
		    {
			Chain msg = Chain("Join buffer exceeded");
			throw Exception(EXLOC, msg);
		    }			

		    flArray[flpos] = pJoinBuf[i];
		    i++;
		    flpos++;
		}
		// terminate array
		flArray[flpos]=0;
	    }
	    else
	    {
		flArray = pJoinBuf + pos;
	    }
	    
	    bool retVal = false;
	    
	    switch ( pP->getMode() )
	    {
		
	    case CegoPredDesc::EXPRCOMP:
	    {
		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr2()->setFieldListArray(flArray);
		
		pP->getExpr1()->setBlock(pBlock);
		pP->getExpr2()->setBlock(pBlock);

		// cout << "Eval : " << pP->getExpr1()->evalFieldValue() << " comp " << pP->getExpr2()->evalFieldValue() << endl;
		retVal = evalFields(pP->getComparison(), pP->getExpr1()->evalFieldValue(), pP->getExpr2()->evalFieldValue());

		break;
	    }    
	    case CegoPredDesc::BETWEEN:
	    {
		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr2()->setFieldListArray(flArray);
		pP->getExpr3()->setFieldListArray(flArray);
		    
		pP->getExpr1()->setBlock(pBlock);
		pP->getExpr2()->setBlock(pBlock);
		pP->getExpr3()->setBlock(pBlock);

			       
		// cout << " Eval " << pP->getExpr1()->evalFieldValue() << " is btwn " <<  pP->getExpr2()->evalFieldValue() << " and "  << pP->getExpr3()->evalFieldValue() << endl;

		retVal = evalBetween(pP->getExpr1()->evalFieldValue(), pP->getExpr2()->evalFieldValue(), pP->getExpr3()->evalFieldValue());
		break;
	    }    		
	    case CegoPredDesc::NULLCOMP:
	    {	  
		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr1()->setBlock(pBlock);
		
		if ( pP->getExpr1()->evalFieldValue().isNull() )
		{
		    retVal = true;
		}
		else
		{
		    retVal = false;
		}
		break;
	    }
	    case CegoPredDesc::NOTNULLCOMP:
	    {	
		
		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr1()->setBlock(pBlock);
		
		if ( pP->getExpr1()->evalFieldValue().isNull() )
		{
		    retVal = false;
		}
		else
		{
		    retVal = true;
		}
		break;
	    }
	    case CegoPredDesc::EXISTSCOMP:
	    {
		
		CegoSelect* pSelect = pP->getSelectQuery();

		pSelect->reset(false);		
		pSelect->setParentJoinBuf(flArray);
				
		ListT<CegoField> fl;
		
		if ( pSelect->nextTuple(fl) )
		{
		    retVal= true;
		}
		else
		{
		    retVal = false;
		}
		
		pSelect->reset(false);
		break;
	    }
	    case CegoPredDesc::ISLIKE:
	    case CegoPredDesc::ISNOTLIKE:
	    {
		
		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr1()->setBlock(pBlock);

		
		if ( pP->getMode() == CegoPredDesc::ISLIKE )
		    retVal = pP->match(pP->getExpr1()->evalFieldValue());
		else
		    retVal =  ! pP->match(pP->getExpr1()->evalFieldValue());

		break;
	    }
	    case CegoPredDesc::INSUB:
	    case CegoPredDesc::NOTINSUB:
	    {

		pP->getExpr1()->setFieldListArray(flArray);
		pP->getExpr1()->setBlock(pBlock);
		
		CegoSelect* pSelect = pP->getSelectQuery();

		pSelect->reset(false);
				
		pSelect->setParentJoinBuf(flArray);
		pSelect->setProcBlock(pBlock);
					       
		ListT<CegoField> fl;
				
		if ( pP->getMode() == CegoPredDesc::INSUB )
		{		   		   
		    retVal = false;
		    while ( pSelect->nextTuple(fl) && retVal == false )
		    {
			CegoField *pSF = fl.First();

			if ( (CegoFieldValue)pSF->getValue() == (CegoFieldValue)pP->getExpr1()->evalFieldValue() )
			    retVal = true;
		    }
		}
		else
		{ 
		    retVal = true;
		    while ( pSelect->nextTuple(fl) && retVal == true )
		    {
			CegoField *pSF = fl.First();
			if ( (CegoFieldValue)pSF->getValue() == (CegoFieldValue)pP->getExpr1()->evalFieldValue() )
			    retVal = false;
		    }
		}
		
		// in case of caching, we have to read to the end to fill up cache
		if ( pSelect->isCached() == false )
		{
		    while ( pSelect->nextTuple(fl));
		}
		
		pSelect->reset(false);
		
		break;    
	    }
	    default:
		break;
	    }
	    return retVal;
	}
    }
    return false;
}

CegoQueryHelper::AttrCondMatch CegoQueryHelper::checkAttrCond(CegoAttrCond& ac, 
							    CegoPredDesc* pP, 
							    const ListT<CegoField>& schema,
							    ListT<CegoField>* flArray, 
							    int flSize, 
							    CegoProcBlock *pBlock)
{
    AttrCondMatch m = evalAttrCond(ac, pP, schema, flArray, flSize, pBlock);
    if ( ac.numComp() == 0 )
	m = INAPP;

    return m;
}

CegoQueryHelper::AttrCondMatch CegoQueryHelper::evalAttrCond(CegoAttrCond& ac, 
							     CegoPredDesc* pP, 
							     const ListT<CegoField>& schema,
							     ListT<CegoField>* flArray, 
							     int flSize, 
							     CegoProcBlock *pBlock)
{
    
    if (pP->getCondition())
    {
	return evalAttrCondbyCondition(ac, pP->getCondition(), schema, flArray, flSize, pBlock);
    }
    else if (pP->getMode() == CegoPredDesc::EXPRCOMP)
    {
	CegoAttrDesc* pAttrDesc1;
	CegoAttrDesc* pAttrDesc2;
	
	pAttrDesc1 = pP->getExpr1()->checkAttr();
	pAttrDesc2 = pP->getExpr2()->checkAttr();
	ListT<CegoAttrDesc*> attrRefList1 = pP->getExpr1()->getAttrRefList();
	ListT<CegoAttrDesc*> attrRefList2 = pP->getExpr2()->getAttrRefList();

	if ( pAttrDesc1 && pAttrDesc2 )
	{

	    CegoField* pF = schema.First();
	    
	    while (pF)
	    {
		if ( ((Chain)pAttrDesc1->getTableName() == (Chain)pF->getTableName() ||
		      (Chain)pAttrDesc1->getTableName() == (Chain)pF->getTableAlias() ||
		      pAttrDesc1->getTableName() == Chain())
		     && (Chain)pAttrDesc1->getAttrName() == (Chain)pF->getAttrName() )
		{					   
		    for ( int i=0; i<flSize; i++)
		    {
			CegoField* pFB = flArray[i].First();
			while (pFB)
			{
			    if (((Chain)pAttrDesc2->getTableName() == (Chain)pFB->getTableName() ||
				 (Chain)pAttrDesc2->getTableName() == (Chain)pFB->getTableAlias() ||
				 pAttrDesc2->getTableName() == Chain())
				&& (Chain)pAttrDesc2->getAttrName() == (Chain)pFB->getAttrName())
			    {
				ac.add(CegoAttrComp(pF->getTableAlias(), 
						    pF->getAttrName(), 
						    pP->getComparison(),
						    *pAttrDesc2));
			    }
			    pFB = flArray[i].Next();
			}
		    }
		}

		if ( ((Chain)pAttrDesc2->getTableName() == (Chain)pF->getTableName() ||
		      (Chain)pAttrDesc2->getTableName() == (Chain)pF->getTableAlias() ||
		      pAttrDesc2->getTableName() == Chain())
		     && (Chain)pAttrDesc2->getAttrName() == (Chain)pF->getAttrName() )
		{			
		    for ( int i=0; i<flSize; i++)
		    {
			CegoField* pFB = flArray[i].First();
			while (pFB)			   			    
			{
			    if (((Chain)pAttrDesc1->getTableName() == (Chain)pFB->getTableName() ||
				 (Chain)pAttrDesc1->getTableName() == (Chain)pFB->getTableAlias() ||
				 pAttrDesc1->getTableName() == Chain())
				&& (Chain)pAttrDesc1->getAttrName() == (Chain)pFB->getAttrName())
			    {

				// we have to map comparison 
				CegoComparison comp = pP->getComparison();
				if ( comp == LESS_THAN )
				    comp = MORE_THAN;
				else if ( comp == MORE_THAN )
				    comp = LESS_THAN;
				else if ( comp == LESS_EQUAL_THAN )
				    comp = MORE_EQUAL_THAN;
				else if ( comp == MORE_EQUAL_THAN )
				    comp = LESS_EQUAL_THAN;

				ac.add(CegoAttrComp(pF->getTableAlias(), 
						    pF->getAttrName(), 
						    comp,
						    *pAttrDesc1));
			    }
			    pFB = flArray[i].Next();
			}
		    }
		}
		pF = schema.Next();
	    }
	}
	else if ( pAttrDesc1 && attrRefList2.Size() == 0 )
	{

	    CegoField* pF = schema.First();   
	    while (pF)
	    { 
		if ((pAttrDesc1->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc1->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc1->getTableName() == Chain())
		    && pAttrDesc1->getAttrName() == (Chain)pF->getAttrName())
		{
		    pP->getExpr2()->setBlock(pBlock);
		    ac.add(CegoAttrComp(pF->getTableAlias(), 
					pF->getAttrName(),
					pP->getComparison(),
					pP->getExpr2()->evalFieldValue()));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else if ( pAttrDesc2 && attrRefList1.Size() == 0 )    
	{
	    CegoField* pF = schema.First();	    
	    while (pF)
	    {
		
		if ((pAttrDesc2->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc2->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc2->getTableName() == Chain())
		    && pAttrDesc2->getAttrName() == (Chain)pF->getAttrName())
		{
		    pP->getExpr1()->setBlock(pBlock);
		    ac.add(CegoAttrComp(pF->getTableAlias(), 
					pF->getAttrName(),
					pP->getComparison(),
					pP->getExpr1()->evalFieldValue()));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;
	}
    }
    else if (pP->getMode() == CegoPredDesc::BETWEEN)
    {
	
	CegoAttrDesc* pAttrDesc1 = pP->getExpr1()->checkAttr();
	CegoAttrDesc* pAttrDesc2 = pP->getExpr2()->checkAttr();
	CegoAttrDesc* pAttrDesc3 = pP->getExpr3()->checkAttr();

	ListT<CegoAttrDesc*> attrRefList1 = pP->getExpr1()->getAttrRefList();
	ListT<CegoAttrDesc*> attrRefList2 = pP->getExpr2()->getAttrRefList();
	ListT<CegoAttrDesc*> attrRefList3 = pP->getExpr3()->getAttrRefList();

	if ( pAttrDesc1 && attrRefList2.Size() == 0 && attrRefList3.Size() == 0 )
	{

	    CegoField* pF = schema.First(); 
	    while (pF)
	    {		
		if ((pAttrDesc1->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc1->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc1->getTableName() == Chain())
		    && pAttrDesc1->getAttrName() == (Chain)pF->getAttrName())
		{
		    pP->getExpr2()->setBlock(pBlock);
		    pP->getExpr3()->setBlock(pBlock);

		    ac.add(CegoAttrComp(pF->getTableAlias(), 
					pF->getAttrName(),
					pP->getExpr2()->evalFieldValue(),
					pP->getExpr3()->evalFieldValue()));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;
	}
    }
    else if (pP->getMode() == CegoPredDesc::ISLIKE)
    {
	CegoAttrDesc* pAttrDesc;	
	pAttrDesc = pP->getExpr1()->checkAttr();
	if ( pAttrDesc  )
	{
	    CegoField* pF = schema.First(); 
	    while (pF)
	    {		
		if ((pAttrDesc->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc->getTableName() == Chain())
		    && pAttrDesc->getAttrName() == (Chain)pF->getAttrName())
		{
		    ac.add(CegoAttrComp(pAttrDesc->getTableName(), pAttrDesc->getAttrName(), pP->getPattern(), false));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;
	}

    }
    else if (pP->getMode() == CegoPredDesc::ISNOTLIKE)
    {
	CegoAttrDesc* pAttrDesc;	
	pAttrDesc = pP->getExpr1()->checkAttr();
	if ( pAttrDesc  )
	{
	    CegoField* pF = schema.First(); 
	    while (pF)
	    {		
		if ((pAttrDesc->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc->getTableName() == Chain())
		    && pAttrDesc->getAttrName() == (Chain)pF->getAttrName())
		{
		    ac.add(CegoAttrComp(pAttrDesc->getTableName(), pAttrDesc->getAttrName(), pP->getPattern(), true));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;
	}
    }
    else if (pP->getMode() == CegoPredDesc::NULLCOMP)
    {
	CegoAttrDesc* pAttrDesc;	
	pAttrDesc = pP->getExpr1()->checkAttr();
	if ( pAttrDesc  )
	{
	    CegoField* pF = schema.First(); 
	    while (pF)
	    {		
		if ((pAttrDesc->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc->getTableName() == Chain())
		    && pAttrDesc->getAttrName() == (Chain)pF->getAttrName())
		{		    
		    CegoFieldValue nullValue;
		    ac.add(CegoAttrComp(pAttrDesc->getTableName(), pAttrDesc->getAttrName(), EQUAL, nullValue));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;	    
	}
    }
    else if (pP->getMode() == CegoPredDesc::NOTNULLCOMP)
    {
	CegoAttrDesc* pAttrDesc;	
	pAttrDesc = pP->getExpr1()->checkAttr();
	if ( pAttrDesc  )
	{

	    CegoField* pF = schema.First(); 
	    while (pF)
	    {		
		if ((pAttrDesc->getTableName() == (Chain)pF->getTableName() ||
		     pAttrDesc->getTableName() == (Chain)pF->getTableAlias() ||
		     pAttrDesc->getTableName() == Chain())
		    && pAttrDesc->getAttrName() == (Chain)pF->getAttrName())
		{		    
		    CegoFieldValue nullValue;
		    ac.add(CegoAttrComp(pAttrDesc->getTableName(), pAttrDesc->getAttrName(), NOT_EQUAL, nullValue));
		    return COMPLETE;
		}
		pF = schema.Next();
	    }
	    return PARTIAL;
	}
	else
	{
	    return PARTIAL;	    
	}
    }
    else if (pP->getMode() == CegoPredDesc::EXISTSCOMP)
    {
	return PARTIAL;	    
    }
    else if (pP->getMode() == CegoPredDesc::INSUB)
    {
	return PARTIAL;	    
    }
    else if (pP->getMode() == CegoPredDesc::NOTINSUB)
    {
	return PARTIAL;	    
    }
    else if (pP->getMode() == CegoPredDesc::NOTPRED)
    {
	return PARTIAL;
    }
    else
    {
	return PARTIAL;
    }

    return COMPLETE;
}

CegoQueryHelper::AttrCondMatch CegoQueryHelper::evalAttrCondbyCondition(CegoAttrCond& ac,
						       CegoCondDesc* pC,
						       const ListT<CegoField>& schema,
						       ListT<CegoField>* flArray, 
						       int flSize, 
						       CegoProcBlock* pBlock)
{
    
    if ( pC->getCondType() == CegoCondDesc::OR)
    {
	return INAPP;	
    }

    AttrCondMatch m1, m2;
    m1 = COMPLETE;
    m2 = COMPLETE;
    if ( pC->Left() )
    {
	m1 = evalAttrCond(ac, pC->Left(), schema, flArray, flSize, pBlock);
    }
    if ( pC->Right() )
    {
	m2 = evalAttrCond(ac, pC->Right(), schema, flArray, flSize, pBlock);
    }
    
    if ( m1 == INAPP || m2 == INAPP )
	return INAPP;
    if ( m1 == PARTIAL || m2 == PARTIAL )
	return PARTIAL;
    
    return COMPLETE;
}

void CegoQueryHelper::makeCNF(CegoCondDesc* pC)
{
    if ( pC )
    {
	if (pC->Left()->getCondition())
	{
	    makeCNF(pC->Left()->getCondition());
	}
	if (pC->Right()->getCondition())
	{
	    makeCNF(pC->Right()->getCondition());
	}
	if ( pC->getCondType() == CegoCondDesc::AND )
	{
	    // nothing to do
	} 
	else if ( pC->getCondType() == CegoCondDesc::OR )
	{
	    if ( pC->Left()->getCondition() )
	    {
		CegoCondDesc *pLC = pC->Left()->getCondition();
		if (pLC->getCondType() == CegoCondDesc::AND)
		{
		    pC->setCondType(CegoCondDesc::AND);
		    CegoCondDesc* pNC = new CegoCondDesc(CegoCondDesc::OR);
		    pNC->setLeft(pLC->Right());
		    pNC->setRight(pC->Right());
		    CegoPredDesc* pNP = new CegoPredDesc(pNC);
		    pC->setRight(pNP);
		    pLC->setCondType(CegoCondDesc::OR);
		    
		    pLC->setRight(pNC->Right()->clone());
		    
		    makeCNF(pC->Left()->getCondition());
		    makeCNF(pC->Right()->getCondition());
		}
		else
		{
		    makeCNF(pC->Left()->getCondition());
		}
		
	    } 
	    else if ( pC->Right()->getCondition() )
	    {
		CegoCondDesc *pRC = pC->Right()->getCondition();
		if (pRC->getCondType() == CegoCondDesc::AND)
		{
		    pC->setCondType(CegoCondDesc::AND);
		    CegoCondDesc* pNC = new CegoCondDesc(CegoCondDesc::OR);
		    pNC->setLeft(pC->Left());
		    pNC->setRight(pRC->Left());
		    CegoPredDesc* pNP = new CegoPredDesc(pNC);
		    pC->setLeft(pNP);
		    pRC->setCondType(CegoCondDesc::OR);
		    pRC->setLeft(pNC->Left()->clone());	    
		    
		    makeCNF(pC->Left()->getCondition());
		    makeCNF(pC->Right()->getCondition());
		}
		else
		{
		    makeCNF(pC->Right()->getCondition());
		}
		
	    }
	}
    }
}

Chain CegoQueryHelper::condToChain(CegoCondDesc* pC, int depth)
{
    Chain s;
    if (pC->Left())
    {
	s = predToChain(pC->Left(), depth+1);
	for (int i=0;i<depth;i++)
	    s += Chain(" ");
	switch(pC->getCondType())
	{
	case CegoCondDesc::AND:
	    s += Chain(" AND ");
	    break;
	case CegoCondDesc::OR:
	    s += Chain(" OR ");
	    break;
	default:
	    break;
	}
	s += predToChain(pC->Right(), depth+1);
    }
    return s;
}

Chain CegoQueryHelper::predToChain(CegoPredDesc* pP, int depth)
{
    Chain s;

    s += pP->toChain(); 
        
    if (pP->getCondition())
    {
	s = condToChain(pP->getCondition(), depth+1);
    }
    else
    {
	for (int i=0;i<depth;i++)
	    s+= Chain(" ");
	
    }
    return s;
    
}

bool CegoQueryHelper::evalFields(CegoComparison comp, const CegoFieldValue& f1, const CegoFieldValue& f2)
{

    switch (comp)
    {
    case EQUAL:
	return ( f1 == f2 );
    case NOT_EQUAL:
	return ( f1 != f2 );
    case LESS_THAN:
	return ( f1 < f2 );
    case MORE_THAN:
	return ( f1 > f2 );
    case LESS_EQUAL_THAN:
	return ( f1 <= f2 );
    case MORE_EQUAL_THAN:
	return ( f1 >= f2 );
    }
}

bool CegoQueryHelper::evalBetween(const CegoFieldValue& f1, const CegoFieldValue& f2, const CegoFieldValue& f3)
{

    if ( f2 <= f1 && f1 <= f3 )
	return true;
    return false;
}

void CegoQueryHelper::aggregateTuple(ListT<CegoField>& aggTuple, CegoExpr* pExpr)
{

    CegoAggregation **pAgg = pExpr->getAggregationList().First();
    
    while ( pAgg )
    {
	aggregateTuple(aggTuple, *pAgg);	
	pAgg = pExpr->getAggregationList().Next();
    }
}

void CegoQueryHelper::aggregateTuple(ListT<CegoField>& aggTuple, CegoAggregation* pAgg)
{
    
    CegoField f;
    CegoFieldValue fv;
    
    CegoExpr *pAE = pAgg->getExpr();
    
    if ( pAE )
    {
	pAE->setFieldListArray( &aggTuple );

	fv = pAE->evalFieldValue();
		
	if ( fv.getValue() == 0 && pAgg->getType() != CegoAggregation::MIN )
	{
	    switch ( fv.getType() )
	    {
	    case INT_TYPE:
	    {			
		fv = CegoFieldValue(INT_TYPE, Chain("0"));
		break;
	    }
	    case LONG_TYPE:
	    {			
		fv = CegoFieldValue(LONG_TYPE, Chain("0"));
		break;
	    }
	    case VARCHAR_TYPE:
	    {
		fv = CegoFieldValue(VARCHAR_TYPE, Chain(""));
		break;
	    }
	    case DECIMAL_TYPE:
	    {
		fv = CegoFieldValue(DECIMAL_TYPE, Chain("0.0"));
		break;		   
	    }
	    case FIXED_TYPE:
	    {
		fv = CegoFieldValue(FIXED_TYPE, Chain("0.0"));
		break;		   
	    }
	    case NULL_TYPE:
	    {
		// nothing to to
		break;
	    }
	    default:
	    {
		Chain msg = Chain("Aggregation not supported on datatype <") + CEGO_TYPE_MAP[ (int) fv.getType()] + Chain(">");
		throw Exception(EXLOC, msg);
		break;
	    }
	    }
	}
    }
        
    switch ( pAgg->getType() )
    {
	
    case CegoAggregation::COUNT:
    {
	if ( pAgg->getFieldValue().getValue() == 0 )
	{
	    pAgg->setFieldValue( CegoFieldValue(LONG_TYPE, Chain("1")) );
	}
	else
	{
	    
	    (*(((long*)(pAgg->getFieldValue().getValue()))))++;
	    
	    /*
	    long l;
	    memcpy(&l, pAgg->getFieldValue().getValue(), sizeof(long));
	    l++;
	    memcpy(pAgg->getFieldValue().getValue(), &l, sizeof(long));
	    */
	    
	}
	break;
    }
    case CegoAggregation::SUM:
    case CegoAggregation::AVG:
    {
	if ( fv.getType() == VARCHAR_TYPE )
	{
	    Chain msg = Chain("Aggregation not supported on datatype <") + CEGO_TYPE_MAP[ (int) fv.getType()] + Chain(">");
	    throw Exception(EXLOC, msg);
	}

	if ( pAgg->getFieldValue().getValue() == 0 )
	{
	    pAgg->getFieldValue() = fv;
	}
	else
	{
	    pAgg->getFieldValue() = pAgg->getFieldValue() + fv;
	}    		   
	break;
    }
    case CegoAggregation::MIN:
    {
	if ( pAgg->getFieldValue().getValue() == 0 )
	{
	    pAgg->getFieldValue() = fv;
	}
	else
	{		
	    if ( (CegoFieldValue)pAgg->getFieldValue() > fv )
	    {
		pAgg->getFieldValue() = fv;
	    }
	}    		   
	break;
    }	    
    case CegoAggregation::MAX:
    {
	if ( pAgg->getFieldValue().getValue() == 0 )
	{
	    pAgg->getFieldValue() = fv;
	}
	else
	{		
	    if ( (CegoFieldValue)pAgg->getFieldValue() < fv )
	    {
		pAgg->getFieldValue() = fv;
	    }
	}    		   
	break;
    }		
    }    
}

void CegoQueryHelper::prepareFieldValue(CegoField* pFV, CegoFieldValue& fv, CegoTableManager* pTM, int tabSetId)
{
    if ( pFV->isNullable() == false && fv.isNull() )
    {
	throw Exception(EXLOC, Chain("Invalid null value for attribute <") 
			+ pFV->getAttrName() 
			+ Chain("> in value list"));	
    }

    if ( fv.getType() != NULL_TYPE && pFV->getType() != fv.getType() )	
    {

	if ( string2Clob(pFV, fv, pTM, tabSetId) )
	     return;
	     
	if ( fv.castTo(pFV->getType(), pFV->getLength()) == false )
	{
	    throw Exception(EXLOC, Chain("Mismatched datatype <") 
			    + CEGO_TYPE_MAP[(int)fv.getType()] 
			    + Chain("> for attribute ") + pFV->getAttrName() 
			    + ", expected <" 
			    + CEGO_TYPE_MAP[(int)pFV->getType()] + ">");
	}
    }
    
    if ( fv.getValue() != 0 )
    {	
	if ( pFV->getType() == VARCHAR_TYPE  
	     && pFV->getLength() < fv.getLength() )
	{
	    Chain shrinkVal ( (char*) fv.getValue() );
	    fv = CegoFieldValue(VARCHAR_TYPE, shrinkVal.subChain(1, pFV->getLength()));
	}
	
	if ( pFV->getType() == DATETIME_TYPE )
	{
	    int val;
	    memcpy(&val, fv.getValue(), sizeof(int));
	    if  (val == 0 ) 
	    {
		Datetime n;
		int dtval = n.asInt();
		memcpy(fv.getValue(), &dtval, sizeof(int));	   	    
	    }
	}

	if ( pFV->getType() == BLOB_TYPE || pFV->getType() == CLOB_TYPE )
	{
	    int fileId;
	    int pageId;
	    memcpy(&fileId, fv.getValue(), sizeof(int));
	    memcpy(&pageId, (void*)((long long)fv.getValue() + sizeof(int)), sizeof(int));

	    try
	    {
		if ( pTM->getDBMng()->isClaimed(fileId, pageId, pTM->getLockHandler()))
		{
		    
		    CegoBufferPage bp;
		    pTM->getDBMng()->bufferFix(bp, tabSetId, fileId, pageId, CegoBufferPool::SYNC, pTM->getLockHandler());
		    pTM->getDBMng()->bufferUnfix(bp, true, pTM->getLockHandler());
		    
		    if (  pFV->getType() == BLOB_TYPE && bp.getType() == CegoBufferPage::BLOB )
		    {
			// ok !
		    }
		    else if (  pFV->getType() == CLOB_TYPE && bp.getType() == CegoBufferPage::CLOB )
		    {
			// ok !
		    }
		    else
		    {
			throw Exception(EXLOC, Chain("Invalid lob page"));
		    }		    
		}
	    }
	    catch ( Exception e )
	    {
		throw Exception(EXLOC, Chain("Invalid lob value"));
	    }
	}
    }
}

void CegoQueryHelper::encodeFL(const ListT<CegoField>& fvl, char* &pBufBase, int &buflen)
{

    CegoField* pF = fvl.First();
    while (pF)
    {
	buflen += sizeof(CegoDataType); // datatype
	buflen += sizeof(int); // datatype len

	buflen += sizeof(int); // defaultvalue len

	if ( pF->getValue().getLength() > 0 )
	    buflen += pF->getValue().getLength();

	buflen += sizeof(char); // isNullable
	buflen += sizeof(int); // attr length
	buflen += pF->getAttrName().length(); // attr

	pF = fvl.Next();
    }
    
    pBufBase = (char*) malloc(buflen);
    
    if (pBufBase == 0)
    {
	throw Exception(EXLOC, "malloc system error");
    }

    char *pBuf = pBufBase;

    pF = fvl.First();
    while (pF)
    {
	int attrLength =  pF->getAttrName().length();
	CegoDataType dt = pF->getType();
		
	memcpy( pBuf, &dt, sizeof(CegoDataType));
	pBuf += sizeof(CegoDataType);

        int dtlen = pF->getLength();

	memcpy( pBuf, &dtlen, sizeof(int));
	pBuf += sizeof(int);
	
	int deflen = pF->getValue().getLength();

	memcpy( pBuf, &deflen, sizeof(int));
	pBuf += sizeof(int);

	if ( deflen > 0 )
	{
	    memcpy( pBuf, pF->getValue().getValue(), deflen);
	    pBuf += deflen;
	}
	
	char isNullable = 0;
	if ( pF->isNullable() )
	    isNullable = 1;

	memcpy( pBuf, &isNullable, sizeof(char));
	pBuf += sizeof(char);
	
	memcpy( pBuf, &attrLength, sizeof(int));
	pBuf += sizeof(int);

	memcpy( pBuf, (char*)pF->getAttrName(), attrLength);
	pBuf += attrLength;

	pF = fvl.Next();

    }
}

void CegoQueryHelper::decodeFL(ListT<CegoField>& fvl, char* pc, int len)
{


    char *pBase = pc;

    while (pc - pBase < len)
    {

	CegoDataType dt;
	memcpy(&dt, pc, sizeof(CegoDataType));
	pc += sizeof(CegoDataType);

	int dtlen;
	memcpy( &dtlen, pc, sizeof(int));
	pc += sizeof(int);

	CegoFieldValue defVal;

	int deflen;
	memcpy( &deflen, pc, sizeof(int));
	pc += sizeof(int);
	if ( deflen > 0 )
	{
	    defVal = CegoFieldValue(dt, pc, deflen, false);
	    pc += deflen;
	}

	char isNullable;
	memcpy(&isNullable, pc, sizeof(char));
	pc += sizeof(char);

	int attrLength;
	memcpy( &attrLength, pc, sizeof(int));
	pc += sizeof(int);
	
	Chain attrName(pc, attrLength-1);
	pc += attrLength;

	Chain n;

	fvl.Insert(CegoField(n,n, attrName, dt, dtlen, defVal, isNullable));
		
    }
}

void CegoQueryHelper::encodeFVL(unsigned long long tid, unsigned long long tastep, CegoTupleState ts, const ListT<CegoField>& fvl, char* &pBufBase, int &buflen)
{
    ListT<CegoBlob> blobList;
    ListT<CegoClob> clobList;
    
    encodeFVL(tid, tastep, ts, fvl, blobList, clobList, pBufBase, buflen);
}

void CegoQueryHelper::encodeFVL(unsigned long long tid, unsigned long long tastep, CegoTupleState ts, const ListT<CegoField>& fvl, const ListT<CegoBlob>& blobList, const ListT<CegoClob>& clobList, char* &pBufBase, int &buflen)
{

    // buflen = sizeof(unsigned long long); // for tid
    // buflen += sizeof(unsigned long long); // for tastep
    // buflen += sizeof(CegoTupleState); // for tuplestate

    buflen = skipTupleHeader();

    int blobIdx = 0;
    int clobIdx = 0;
    
    CegoField* pF = fvl.First();
    while (pF)
    {

	int l =  pF->getValue().getLength();
	CegoDataType dt = pF->getValue().getType();
	
	// force null type in case of fv length and value constraints 
	if ( l == 0 || pF->getValue().getValue() == 0 )
	    dt = NULL_TYPE;
		
	if ( dt != NULL_TYPE )
	{
	    buflen += sizeof(int); // id
	    buflen += sizeof(CegoDataType);

	    if ( dt == BLOB_TYPE && blobList.Size() > 0)
	    {	
		buflen += sizeof(long); // blobsize
		buflen += blobList[blobIdx].getSize();
		blobIdx++;
	    }
	    else if ( dt == CLOB_TYPE && clobList.Size() > 0)
	    {		
		buflen += sizeof(long); // clobsize
		buflen += clobList[clobIdx].getSize();
		clobIdx++;
	    }
	    else
	    {
		
		if ( dt == VARCHAR_TYPE 
		     || dt == BIGINT_TYPE
		     || dt == DECIMAL_TYPE 
		     || dt == FIXED_TYPE )
		{
		    buflen += sizeof(int);
		}
		
		buflen += pF->getValue().getLength();
	    }
	}
	pF = fvl.Next();
    }

    buflen += sizeof(int); // terminating id
    
    pBufBase = (char*) malloc(buflen);
    
    if (pBufBase == 0)
    {
	throw Exception(EXLOC, "malloc system error");	
    }

    char *pBuf = pBufBase;

    int toff = encodeTupleHeader(tid, tastep, ts, pBuf);
    pBuf += toff;

    blobIdx = 0;
    clobIdx = 0;
    
    pF = fvl.First();
    while (pF)
    {
	int id =  pF->getId();
	int l =  pF->getValue().getLength();
	CegoDataType dt = pF->getValue().getType();

	// cout << "Encoding id " << id << endl;
	// force null type in case of fv length and value constraints 
	if ( l == 0 || pF->getValue().getValue() == 0 )
	    dt = NULL_TYPE;
	
	if ( dt != NULL_TYPE )
	{
	    // cout << "ID = " << id << " Encoding value " << pF->getValue().valAsChain() << endl;

	    memcpy( pBuf, &id, sizeof(int));
	    pBuf += sizeof(int);
	    
	    memcpy( pBuf, &dt, sizeof(CegoDataType));
	    pBuf += sizeof(CegoDataType);
	    
	    if ( dt == BLOB_TYPE && blobList.Size() > 0)
	    {
		unsigned long long blobSize = blobList[blobIdx].getSize();
		memcpy( pBuf, &blobSize, sizeof(unsigned long long));
		pBuf += sizeof(unsigned long long);
		memcpy( pBuf, blobList[blobIdx].getBufPtr(), blobList[blobIdx].getSize());
		pBuf += blobList[blobIdx].getSize();
	    }
	    else if ( dt == CLOB_TYPE && clobList.Size() > 0)
	    {
		unsigned long long clobSize = clobList[clobIdx].getSize();
		memcpy( pBuf, &clobSize, sizeof(unsigned long long));
		pBuf += sizeof(long);
		memcpy( pBuf, clobList[clobIdx].getBufPtr(), clobList[clobIdx].getSize());
		pBuf += clobList[clobIdx].getSize();
	    }
	    else
	    {
		if ( dt == VARCHAR_TYPE 
		     || dt == BIGINT_TYPE 
		     || dt == DECIMAL_TYPE
		     || dt == FIXED_TYPE)
		{
		    memcpy( pBuf, &l, sizeof(int));
		    pBuf += sizeof(int);
		}
		
		memcpy( pBuf, pF->getValue().getValue(), l);	
		
		pBuf += l;
	    }
	}
	
	pF = fvl.Next();
    }

    int termId=0;
    memcpy( pBuf, &termId, sizeof(int));

}

// for performance reasons, we also provide a method to just return the 
// tuple header len
int CegoQueryHelper::skipTupleHeader()
{
    return sizeof(unsigned long long) + sizeof(unsigned long long) + sizeof(CegoTupleState);
}

int CegoQueryHelper::encodeTupleHeader(unsigned long long tid, unsigned long long tastep, CegoTupleState ts, char* p)
{
    memcpy(p, &tid, sizeof(unsigned long long));
    p += sizeof(unsigned long long);
    memcpy(p, &tastep, sizeof(unsigned long long));
    p += sizeof(unsigned long long);
    memcpy(p, &ts, sizeof(CegoTupleState));
    
    return sizeof(unsigned long long) + sizeof(unsigned long long) + sizeof(CegoTupleState);
}

int CegoQueryHelper::decodeTupleHeader(unsigned long long& tid, unsigned long long& tastep, CegoTupleState& ts, char* p)
{
    memcpy(&tid, p, sizeof(unsigned long long));

    char* ptastep = p + sizeof(unsigned long long);
    memcpy(&tastep, ptastep, sizeof(unsigned long long));
    
    char* pts = p + sizeof(unsigned long long) + sizeof(unsigned long long);
    memcpy(&ts, pts, sizeof(CegoTupleState));
	
    return sizeof(unsigned long long) + sizeof(unsigned long long) + sizeof(CegoTupleState);
}

void CegoQueryHelper::decodeFVL(ListT<CegoField>& fvl, char* pc, int len, bool doReset)
{
    
    if ( doReset )
    {
	// make sure no field values are set    
	CegoField* pF = fvl.First(); 
	while ( pF )
	{
	    CegoFieldValue nullValue;
	    pF->setValue(nullValue);
	    pF = fvl.Next();	
	}
    }

    char* pBase = pc;
    
    bool eot = false;

    int fldCount = fvl.Size();

    while (pc - pBase < len && eot == false && fldCount > 0 )
    {
	int id;     
	memcpy(&id, pc, sizeof(int));
	
	if ( id > 0 )
	{
	    
	    pc += sizeof(int);
	    
	    CegoDataType dt;
	    memcpy(&dt, pc, sizeof(CegoDataType));
	    
	    pc += sizeof(CegoDataType);
	    
	    int flen;
	    
	    if (dt == VARCHAR_TYPE
		|| dt == BIGINT_TYPE
		|| dt == DECIMAL_TYPE
		|| dt == FIXED_TYPE )
	    {
		memcpy(&flen, pc, sizeof(int));
		pc += sizeof(int);
	    }
	    else
	    {
		CegoTypeConverter tc;
		flen = tc.getTypeLen(dt);	    
	    }
	    
	    CegoField* pF = fvl.First(); 
	    while ( pF && id != pF->getId() )
	    {
		pF = fvl.Next();
	    }
	    
	    if ( pF )
	    {

		/*
		
		pF->getValue().setType(dt);
		pF->getValue().setLength(flen);
		
		if ( flen > 0)
		    pF->getValue().setValue(pc);
		else
		    pF->getValue().setValue(0);
		pF->getValue().setLocalCopy(false);
	
		since CegoField::getValue() has changed to const, this must be done vie setValue()	
		
		*/
		
		CegoFieldValue fv;
		fv.setType(dt);
		fv.setLength(flen);

		if ( flen > 0)
		    fv.setValue(pc);
		else
		    fv.setValue(0);
		fv.setLocalCopy(false);

		pF->setValue(fv);

		
		fldCount--;
		
	    }
	    
	    if ( flen > 0 )
	    {	    
		pc += flen;
	    }
	}
	else
	{
	    eot = true;
	}
	    
    }
	
    /* fldCount is the number of not encoded values => null values */

}

/* field list decoding with included blob decoding. This is used by the recovery manager for redo log recovery  */

void CegoQueryHelper::decodeFVL(ListT<CegoField>& fvl, ListT<CegoBlob>& blobList, ListT<CegoClob>& clobList, char* pc, int len)
{
    
    // make sure no field values are set
    CegoField* pF = fvl.First(); 
    while ( pF )
    {
	CegoFieldValue nullValue;
	pF->setValue(nullValue);
	pF = fvl.Next();	
    }

    char* pBase = pc;
    
    bool eot = false;
    while (pc - pBase < len && eot == false )
    {
	int id;
	memcpy(&id, pc, sizeof(int));

	if ( id > 0 )
	{
	    pc += sizeof(int);
	    
	    CegoDataType dt;
	    memcpy(&dt, pc, sizeof(CegoDataType));
	    pc += sizeof(CegoDataType);
	    
	    int flen=0;
	    unsigned long long blobSize=0;
	    unsigned long long clobSize=0;
	    
	    if (dt == VARCHAR_TYPE
		|| dt == BIGINT_TYPE
		|| dt == DECIMAL_TYPE
		|| dt == FIXED_TYPE )	    
	    {
		memcpy(&flen, pc, sizeof(int));
		pc += sizeof(int);
	    }
	    else if (dt == BLOB_TYPE )
	    {
		memcpy(&blobSize, pc, sizeof(unsigned long long));
		pc += sizeof(unsigned long long);
	    }
	    else if (dt == CLOB_TYPE )
	    {
		memcpy(&clobSize, pc, sizeof(unsigned long long));
		pc += sizeof(unsigned long long);
	    }
	    else
	    {
		CegoTypeConverter tc;
		flen = tc.getTypeLen(dt);
	    }
	    
	    bool found = false;
	    CegoField* pF = fvl.First(); 
	    while (pF && ! found )
	    {
		if (pF->getId() == id )
		{


		    
		    if ( dt == BLOB_TYPE )
		    {
			// pF->getValue().setType(dt);
			// since CegoField::getValue() has changed to const, this must be done vie setValue()
			CegoFieldValue fv = pF->getValue();
			fv.setType(dt);
			pF->setValue(fv);
			
			unsigned char *blobBuf = (unsigned char*)malloc(blobSize);
			memcpy(blobBuf, pc, blobSize);
			blobList.Insert( CegoBlob(0,0, blobBuf, blobSize) );
		    }
		    else if ( dt == CLOB_TYPE )
		    {
			// pF->getValue().setType(dt);
			// since CegoField::getValue() has changed to const, this must be done vie setValue()
			CegoFieldValue fv = pF->getValue();
			fv.setType(dt);
			pF->setValue(fv);

			char *clobBuf = (char*)malloc(clobSize);
			memcpy(clobBuf, pc, clobSize);
			clobList.Insert( CegoClob(0,0, clobBuf, clobSize) );
		    }
		    else
		    {
			/*
			pF->getValue().setType(dt);
			pF->getValue().setLength(flen);
			pF->setId(id);
			
			if ( flen > 0)
			    pF->getValue().setValue(pc);
			else
			    pF->getValue().setValue(0);
			pF->getValue().setLocalCopy(false);

			since CegoField::getValue() has changed to const, this must be done vie setValue()

			*/

			CegoFieldValue fv = pF->getValue();
			fv.setType(dt);			
			fv.setLength(flen);			
			if ( flen > 0)
			    fv.setValue(pc);
			else
			    fv.setValue(0);
			fv.setLocalCopy(false);
			
			pF->setValue(fv);
			pF->setId(id);
			
			
		    }
		    found = true;
		}
		pF = fvl.Next();
	    }
	    if ( blobSize > 0 )
	    {
		pc += blobSize;
	    }
	    if ( flen > 0 )
	    {
		pc += flen;
	    }
	}
	else
	{
	    eot = true;
	}
    }
}

void CegoQueryHelper::decodeNativeFVL(ListT<CegoFieldValue>& fvl, ListT<CegoBlob>& blobList, ListT<CegoClob>& clobList, char* pc, int len)
{
		
    char* pBase = pc;

    bool eot = false;    
    while (pc - pBase < len && eot == false)
    {		    
	int id;
	memcpy(&id, pc, sizeof(int));

	if ( id > 0 )
	{       
	    pc += sizeof(int);
	    
	    CegoDataType dt;
	    memcpy(&dt, pc, sizeof(CegoDataType));
	    
	    pc += sizeof(CegoDataType);
	    
	    int flen=0;
	    unsigned long long blobSize=0;
	    unsigned long long clobSize=0;
	    
	    if (dt == VARCHAR_TYPE
		|| dt == BIGINT_TYPE
		|| dt == DECIMAL_TYPE
		|| dt == FIXED_TYPE )	    
	    {
		memcpy(&flen, pc, sizeof(int));
		pc += sizeof(int);
	    }
	    else if (dt == BLOB_TYPE )
	    {
		memcpy(&blobSize, pc, sizeof(unsigned long long));
		pc += sizeof(unsigned long long);
	    }
	    else if (dt == CLOB_TYPE )
	    {
		memcpy(&clobSize, pc, sizeof(unsigned long long));
		pc += sizeof(unsigned long long);
	    }
	    else
	    {
		CegoTypeConverter tc;
		flen = tc.getTypeLen(dt);
	    }
	    
	    if ( dt == BLOB_TYPE )
	    {
		CegoFieldValue fv(dt, Chain("[0,0]"));
		unsigned char *blobBuf = (unsigned char*)malloc(blobSize);
		memcpy(blobBuf, pc, blobSize);
		blobList.Insert( CegoBlob(0,0, blobBuf, blobSize) );
		
		fvl.Insert(fv);
	    }
	    else if ( dt == CLOB_TYPE )
	    {
		CegoFieldValue fv(dt, Chain("[0,0]"));
		char *clobBuf = (char*)malloc(clobSize);
		memcpy(clobBuf, pc, clobSize);
		clobList.Insert( CegoClob(0,0, clobBuf, clobSize) );
		
		fvl.Insert(fv);
	    }
	    
	    else
	    {
		CegoFieldValue fv(dt, pc, flen);
		fvl.Insert(fv);
	    }
	    
	    if ( blobSize > 0 )
	    {
		pc += blobSize;
	    }
	    if ( flen > 0 )
	    {
		pc += flen;
	    }
	}   
	else
	{
	    eot = true;
	}
    }
}

void CegoQueryHelper::encodeUpdRec(const Chain& tableAlias,
				   CegoPredDesc* pPred,
				   const ListT<CegoField>& updList,
				   const ListT<CegoExpr*>& exprList,
				   CegoProcBlock* pBlock,
				   char* &pBuf, int &buflen)
{

    char* updBuf;
    int updBufLen = 0;
    char* predBuf;
    int predBufLen = 0;
    char* expBuf;
    int expBufLen = 0;

    if ( pPred )
    {
	if ( pBlock ) 
	    pPred->setBlock(pBlock);
	predBufLen = pPred->getEncodingLength() + sizeof(char);
	predBuf = (char*)malloc( predBufLen );
	char* pP = predBuf;
	char c = 1;
	memcpy(pP, &c, sizeof(char));
	pP = pP + sizeof(char);
	pPred->encode(pP);
	
    }
    else
    {
	char c = 0;
	predBufLen = sizeof(char);
	predBuf = (char*)malloc( predBufLen);
	memcpy(predBuf, &c, sizeof(char));
    }

    encodeFL(updList, updBuf, updBufLen);

    expBufLen = 0;
    CegoExpr **pExpr = exprList.First();
    while ( pExpr )
    {
	expBufLen += (*pExpr)->getEncodingLength();
	pExpr = exprList.Next();
    }
    expBuf = (char*)malloc( expBufLen );
    char *pExpBuf = expBuf;
    pExpr = exprList.First();
    while ( pExpr )
    {
	int len = (*pExpr)->getEncodingLength();
	(*pExpr)->encode(pExpBuf);
	pExpBuf += len;
	pExpr = exprList.Next();
    }

    int aliasLen = tableAlias.length();

    buflen = predBufLen
	+ sizeof(int) + aliasLen
	+ sizeof(int) + updBufLen +
	+ sizeof(int) + expBufLen;

    pBuf = (char*)malloc(buflen);

    char* pBP = pBuf;

    memcpy(pBP, &aliasLen, sizeof(int));
    pBP = pBP + sizeof(int);

    memcpy( pBP, (char*)tableAlias, aliasLen);
    pBP = pBP + aliasLen;

    memcpy(pBP, predBuf, predBufLen);
    pBP = pBP + predBufLen;
    
    memcpy(pBP, &updBufLen, sizeof(int));
    pBP = pBP + sizeof(int);

    memcpy(pBP, updBuf, updBufLen);
    pBP = pBP + updBufLen;

    memcpy(pBP, &expBufLen, sizeof(int));
    pBP = pBP + sizeof(int);

    memcpy(pBP, expBuf, expBufLen);
    pBP = pBP + expBufLen;

    free ( predBuf );
    free ( updBuf );
    free ( expBuf );

}

void CegoQueryHelper::decodeUpdRec(Chain& tableAlias,
				   CegoPredDesc* &pPred, 
				   ListT<CegoField>& updList, 
				   ListT<CegoExpr*>& exprList,
				   char* pBuf, int buflen,
				   CegoDistManager *pGTM, int tabSetId)
{

    int aliasLen;
    memcpy(&aliasLen, pBuf, sizeof(int));
    pBuf += sizeof(int);

    tableAlias = Chain(pBuf, aliasLen-1);
    pBuf += aliasLen;
    
    char c = 0;
    memcpy(&c, pBuf, sizeof(char));
    pBuf += sizeof(char);

    if ( c == 1 )
    {
	pPred = new CegoPredDesc(pBuf, pGTM, tabSetId);
	pBuf += pPred->getEncodingLength();
    }
    else
    {
	pPred = 0;
    }

    int updBufLen;
    memcpy(&updBufLen, pBuf, sizeof(int));
    pBuf += sizeof(int);

    decodeFL(updList, pBuf, updBufLen);

    pBuf += updBufLen;

    int expBufLen;
    memcpy(&expBufLen, pBuf, sizeof(int));
    pBuf += sizeof(int);

    int pos = 0;
    while ( pos < expBufLen  )
    {
	CegoExpr *pExpr = new CegoExpr(pBuf, pGTM, tabSetId);

	int len = pExpr->getEncodingLength();
	pos += len;
	pBuf += len;

	exprList.Insert(pExpr);

    }
}

void CegoQueryHelper::encodeDelRec(const Chain& tableAlias,
				   CegoPredDesc* pPred,
				   CegoProcBlock* pBlock,
				   char* &pBuf, int &buflen)

{

    char* predBuf;
    int predBufLen = 0;

    if ( pPred )
    {
	if ( pBlock ) 
	    pPred->setBlock(pBlock);
	predBufLen = pPred->getEncodingLength() + sizeof(char);
	predBuf = (char*)malloc( predBufLen );
	char* pP = predBuf;
	char c = 1;
	memcpy(pP, &c, sizeof(char));
	pP = pP + sizeof(char);

	pPred->encode(pP);
	
    }
    else
    {
	char c = 0;
	predBufLen = sizeof(char);
	predBuf = (char*)malloc( predBufLen);
	memcpy(predBuf, &c, sizeof(char));
    }

    int aliasLen = tableAlias.length();

    buflen = predBufLen
	+ sizeof(int) + aliasLen;
    
    pBuf = (char*)malloc(buflen);

    char* pBP = pBuf;

    memcpy(pBP, &aliasLen, sizeof(int));
    pBP = pBP + sizeof(int);

    memcpy( pBP, (char*)tableAlias, aliasLen);
    pBP = pBP + aliasLen;

    memcpy(pBP, predBuf, predBufLen);
    pBP = pBP + predBufLen;
    
    free ( predBuf );

}

void CegoQueryHelper::decodeDelRec(Chain& tableAlias,
				   CegoPredDesc* &pPred,
				   char* pBuf, int buflen,
				   CegoDistManager *pGTM, int tabSetId)
{        
    int aliasLen;
    memcpy(&aliasLen, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    tableAlias = Chain(pBuf, aliasLen-1);
    pBuf += aliasLen;
    
    char c = 0;
    memcpy(&c, pBuf, sizeof(char));
    pBuf += sizeof(char);

    if ( c == 1 )
    {
	pPred = new CegoPredDesc(pBuf, pGTM, tabSetId);
	pBuf += pPred->getEncodingLength();
    }
    else
    {
	pPred = 0;
    }

}

void CegoQueryHelper::createConjunctionList(CegoPredDesc* pPred, ListT<CegoPredDesc*>& conjunctionList)
{
    if (pPred)
    {
	CegoCondDesc* pC = pPred->getCondition();	
	if (pC)
	{
	    if (pC->getCondType() == CegoCondDesc::AND)
	    {
		createConjunctionList(pC->Left(), conjunctionList);
		createConjunctionList(pC->Right(), conjunctionList);
	    }
	    else
	    {
		conjunctionList.Insert(pPred);
	    }
	}
	else
	{
	    conjunctionList.Insert(pPred);
	}
    }
}

bool CegoQueryHelper::string2Clob(CegoField* pFV, CegoFieldValue& fv, CegoTableManager* pTM, int tabSetId)
{
    if ( pFV->getType() == CLOB_TYPE && fv.getType() == VARCHAR_TYPE && pTM )
    {
	char* clobData = (char*)fv.getValue();
	unsigned long long clobSize = fv.getLength();
	int fileId, pageId;
	pTM->putClobData(tabSetId, clobData, clobSize, fileId, pageId);

	// cout << "Clob created with " << fileId << " " << pageId << endl;

	fv = CegoFieldValue(CLOB_TYPE, Chain("[") + Chain(fileId) + Chain(",") + Chain(pageId) + Chain("]"));

	// cout << "Clob is " << fv << endl;
	return true;
    }
    return false;
}
