///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBTreeValue.cc
// -----------------
// Cego index value class implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBTreeValue
// 
// Description: This class handles all operations on btree values. Since the values are sequentially stored
//              in a byte array, conversion methods are required to store, retrieve and for compare these values 
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>

// CEGO INCLUDES
#include "CegoFieldValue.h"
#include "CegoBTreeValue.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoBTreeValue::CegoBTreeValue()
{
    _pI = _idxArray;
}

CegoBTreeValue::CegoBTreeValue(char* p)
{
    _pI = p;
}

CegoBTreeValue::~CegoBTreeValue()
{
}

int CegoBTreeValue::getKeyLen(const ListT<CegoField>& schema)
{
    CegoField *pF = schema.First();
    int keyLen = 0;
    while ( pF )
    {
	keyLen += 1; // null value indicator
	keyLen += pF->getLength();

	if ( pF->getType() == VARCHAR_TYPE
	     || pF->getType() == BIGINT_TYPE
	     || pF->getType() == DECIMAL_TYPE
	     || pF->getType() == FIXED_TYPE )
	    keyLen += 1; // termination character
	 	
	pF = schema.Next();
    }
    return keyLen;
}

bool CegoBTreeValue::hasNull(const ListT<CegoField>& schema) const
{
    CegoField* pF = schema.First();
    char* idxPtr = _pI;
    while (pF)
    {
	char nullIndicator;
	memcpy(&nullIndicator, idxPtr, 1);
	idxPtr += 1;

	if ( nullIndicator == 1 )
	{	    
	    int len = pF->getLength();
	    
	    idxPtr+=len;
	    
	    if ( pF->getType() == VARCHAR_TYPE
		 || pF->getType() == BIGINT_TYPE
		 || pF->getType() == DECIMAL_TYPE
		 || pF->getType() == FIXED_TYPE )
		idxPtr+=1;
	}
	else
	{
	    return true;
	}
	
	pF = schema.Next();
    }
    return false;
}

void CegoBTreeValue::valueFromSchema(ListT<CegoField>* pSchema)
{

    char* idxPtr = _pI;
    CegoField *pF = pSchema->First();
    while (pF)
    {
	int vlen = pF->getValue().getLength();
	
	if ( vlen > TABMNG_MAXINDEXVALUE )
	    throw Exception(EXLOC, "Index value exceeded");

	if ( vlen > 0 )
	{
	    char nullIndicator = 1;
	    memcpy(idxPtr, &nullIndicator, 1);
	    idxPtr += 1;

	    memcpy(idxPtr, pF->getValue().getValue(), vlen);

	    idxPtr += pF->getLength();

	    if ( pF->getType() == VARCHAR_TYPE
		 || pF->getType() == BIGINT_TYPE
		 || pF->getType() == DECIMAL_TYPE
		 || pF->getType() == FIXED_TYPE )
		idxPtr+=1;

	    
	    /*
	    if ( pF->getLength() > vlen )
	    {	       		
		// mark end with zero byte
		*idxPtr = 0;
		idxPtr += pF->getLength() - vlen;
	    }
	    */
	}
	else
	{
	    char nullIndicator = 0;
	    memcpy(idxPtr, &nullIndicator, 1);
	    idxPtr += pF->getLength() + 1;
	    
	    // cout << "LEN IS 0" << endl;
	}
	pF = pSchema->Next();
    }
    // cout << "Value from schema : " << toChain(pSchema) << endl;
}

void CegoBTreeValue::valueFromSchema(const ListT<CegoField> tableSchema, const ListT<CegoField> indexSchema)
{
    // get Index len
    int len = 0;
    CegoField* pIF = indexSchema.First();
    while (pIF)
    {	
	len += pIF->getValue().getLength();       	
	pIF = indexSchema.Next();
    }

    if ( len > TABMNG_MAXINDEXVALUE )
	throw Exception(EXLOC, "Index value exceeded");

    char* idxPtr = _pI;
    pIF = indexSchema.First();
    while (pIF)
    {

	CegoField *pTF = tableSchema.First();

	while ( pTF )
	{
	    if ( (Chain)pTF->getAttrName() == (Chain)pIF->getAttrName() )
	    {

		int len = pTF->getValue().getLength();
		if ( len > 0 )
		{

		    char nullIndicator = 1;
		    memcpy(idxPtr, &nullIndicator, 1);
		    idxPtr += 1;

		    memcpy(idxPtr, pTF->getValue().getValue(), len);
		    
		    if ( pTF->getLength() > len )
		    {		
			idxPtr += len;
			// mark end with zero byte
			*idxPtr = 0;
			idxPtr += pTF->getLength() - ( len + 1 );
		    }
		    else
		    {
			idxPtr += len;
		    }
		}
		else
		{
		    char nullIndicator = 0;
		    memcpy(idxPtr, &nullIndicator, 1);
		    idxPtr += len + 1;
		}
	    }
	    
	    pTF = tableSchema.Next();
	}

	pIF = indexSchema.Next();
    }    
}

ListT<CegoFieldValue> CegoBTreeValue::valueToFVL(const ListT<CegoField>& schema) const
{
    ListT<CegoFieldValue> fvl;

    char* idxPtr = _pI;
    CegoField* pF = schema.First();
    while (pF)
    {
	char nullIndicator;
	memcpy(&nullIndicator, idxPtr, 1);
	idxPtr += 1;

	CegoFieldValue fv;

	if ( nullIndicator == 1 )
	{
	    int len = pF->getLength();

	    if ( pF->getType() == VARCHAR_TYPE
		 || pF->getType() == BIGINT_TYPE
		 || pF->getType() == DECIMAL_TYPE
		 || pF->getType() == FIXED_TYPE )
	    {		
		fv = CegoFieldValue(VARCHAR_TYPE, Chain(idxPtr));
		idxPtr += len + 1;
	    }
	    else
	    {
		fv = CegoFieldValue(pF->getType(), idxPtr, len);
		idxPtr+=len;
	    }	    
	}
	
	fvl.Insert(fv);
	pF = schema.Next();
    }
    return fvl;
}

void CegoBTreeValue::setPtr(char* p)
{
    _pI = p;
}

char* CegoBTreeValue::getPtr() const
{
    return _pI;
}
 
CegoBTreeValue& CegoBTreeValue::operator = (const CegoBTreeValue& iv)
{
    _pI = iv._pI;
    memcpy(_idxArray, iv._idxArray, TABMNG_MAXINDEXVALUE);
    return (*this);
}

CegoBTreeValue::Comparison CegoBTreeValue::comp(const CegoBTreeValue& iv, ListT<CegoField>* pSchema) const
{
    char* idxPtr1 = _pI;
    char* idxPtr2 = iv.getPtr();

    CegoField* pF = pSchema->First();
    while (pF)
    {
	int len = pF->getLength();

	if ( pF->getType() == VARCHAR_TYPE
	     || pF->getType() == BIGINT_TYPE
	     || pF->getType() == DECIMAL_TYPE
	     || pF->getType() == FIXED_TYPE )
	    len++;
	
	CegoFieldValue fv1;
	CegoFieldValue fv2;
	
	/* for performance reasons, we construct fieldvalue with pointer reference
           in case of VARCHAR & friend values len might be not correct, but for 
	   the used methods, it does't matter */

	char nullIndicator1;
	memcpy(&nullIndicator1, idxPtr1, 1);
	idxPtr1 += 1;
	
	char nullIndicator2;
	memcpy(&nullIndicator2, idxPtr2, 1);
	idxPtr2 += 1;

	if ( nullIndicator1 == 1 )
	    fv1 = CegoFieldValue(pF->getType(), idxPtr1, len);
	if ( nullIndicator2 == 1 )
	    fv2 = CegoFieldValue(pF->getType(), idxPtr2, len);	    
	
	CegoFieldValue::Comparison c = fv1.comp(fv2);

	if ( c == CegoFieldValue::MORE )
	{
	    return MORE;
	}
	else if ( c == CegoFieldValue::LESS )
	{
	    return LESS;
	}

	idxPtr1+=len;
	idxPtr2+=len;	    
	pF = pSchema->Next();
		
    }

    return EQUAL;
}

bool CegoBTreeValue::isHigher(const CegoBTreeValue& iv, ListT<CegoField>* pSchema) const
{
    char* idxPtr1 = _pI;
    char* idxPtr2 = iv.getPtr();
    
    CegoField* pF = pSchema->First();
    while (pF)
    {
	int len = pF->getLength();

	if ( pF->getType() == VARCHAR_TYPE
	     || pF->getType() == BIGINT_TYPE
	     || pF->getType() == DECIMAL_TYPE
	     || pF->getType() == FIXED_TYPE )
	    len++;

	CegoFieldValue fv1;
	CegoFieldValue fv2;

	char nullIndicator1;
	memcpy(&nullIndicator1, idxPtr1, 1);
	idxPtr1 += 1;

	char nullIndicator2;
	memcpy(&nullIndicator2, idxPtr2, 1);
	idxPtr2 += 1;

	if ( nullIndicator1 == 1 )
	     fv1 = CegoFieldValue(pF->getType(), idxPtr1, len);
	if ( nullIndicator2 == 1 )
	    fv2 = CegoFieldValue(pF->getType(), idxPtr2, len);	    

	CegoFieldValue::Comparison c = fv1.comp(fv2);

	if ( c == CegoFieldValue::MORE )
	    return true;
	else if ( c == CegoFieldValue::LESS )
	    return false;

	idxPtr1+=len;
	idxPtr2+=len;	    
	pF = pSchema->Next();		
    }
    return false;
}

bool CegoBTreeValue::isEqual(const CegoBTreeValue& iv, ListT<CegoField>* pSchema) const
{    
    char* idxPtr1 = _pI;
    char* idxPtr2 = iv.getPtr();
    
    CegoField* pF = pSchema->First();
    while (pF)
    {
	int len = pF->getLength();

	if ( pF->getType() == VARCHAR_TYPE
	     || pF->getType() == BIGINT_TYPE
	     || pF->getType() == DECIMAL_TYPE
	     || pF->getType() == FIXED_TYPE )
	    len++;
	
	CegoFieldValue fv1;
	CegoFieldValue fv2;
	
	char nullIndicator1;
	memcpy(&nullIndicator1, idxPtr1, 1);
	idxPtr1 += 1;

	char nullIndicator2;
	memcpy(&nullIndicator2, idxPtr2, 1);
	idxPtr2 += 1;

	if ( nullIndicator1 == 1 )
	    fv1 = CegoFieldValue(pF->getType(), idxPtr1, len);
	if ( nullIndicator2 == 1 )
	    fv2 = CegoFieldValue(pF->getType(), idxPtr2, len);	    
	
	CegoFieldValue::Comparison c = fv1.comp(fv2);

	if ( c == CegoFieldValue::MORE )
	    return false;
	else if ( c == CegoFieldValue::LESS )
	    return false;

	idxPtr1+=len;
	idxPtr2+=len;	    
	pF = pSchema->Next();
		
    }
    return true;
}

bool CegoBTreeValue::operator > ( const CegoBTreeValue& iv) const
{
    return false;
}

Chain CegoBTreeValue::toChain(ListT<CegoField>* pSchema) const
{
    Chain s;

    char* idxPtr = _pI;
    CegoField* pF = pSchema->First();
    while (pF)
    {
	char nullIndicator;
	memcpy(&nullIndicator, idxPtr, 1);
	idxPtr += 1;

	CegoFieldValue fv;

	int len = pF->getLength();
	if ( nullIndicator == 1 )
	{

	    if ( pF->getType() == VARCHAR_TYPE
		 || pF->getType() == BIGINT_TYPE
		 || pF->getType() == DECIMAL_TYPE
		 || pF->getType() == FIXED_TYPE )
	    {
		fv = CegoFieldValue(VARCHAR_TYPE, Chain(idxPtr));
		idxPtr += len + 1;
	    }
	    else
	    {
		fv = CegoFieldValue(pF->getType(), idxPtr, len);
		idxPtr+=len;
	    }
	}
	else
	{
	    idxPtr+=len;
	}
	
	s += fv.valAsChain();
	pF = pSchema->Next();
	if ( pF )
	    s += Chain(",");
    }
    
    return s;
}
