///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoFieldValue.cc
// -----------------
// Field value implementation
//
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoFieldValue
//
// Description: Data field value which can contain all supported data types
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/Datetime.h>
#include <lfcbase/BigInteger.h>
#include <lfcbase/BigDecimal.h>
#include <lfcbase/Tokenizer.h>
#include <lfcbase/ThreadLock.h>

// CEGO INCLUDES
#include "CegoFieldValue.h"
#include "CegoTypeConverter.h"
#include "CegoDefs.h"
#include "CegoDatabaseFormater.h"


// POSIX INCLUDES
#include <string.h>
#include <strings.h>
#include <stdlib.h>

char __currencySymbol;
char __decimalPoint;
char __caseSensitiveFlag = 0;
char __quoteEscapeFlag = 0;
Chain __dateTimeFormat;
ListT<Chain> __dateFormatList;
ThreadLock __dateFormatLock("DATEFORMAT");

CegoFieldValue::CegoFieldValue(const CegoFieldValue& fv)
{
    _type = fv._type;
    _len = fv._len;
    _isLocalCopy = fv._isLocalCopy;

    if ( fv._type != NULL_TYPE )
    {    
	if (_isLocalCopy && fv._pV)
	{
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    memcpy(_pV, fv._pV, _len);
	}
	else
	{
	    _pV = fv._pV;
	}
    }
    else
    {
	_pV = 0;
    }
}

CegoFieldValue::CegoFieldValue()
{
    _type = NULL_TYPE;
    _pV = 0;
    _len = 0;
    _isLocalCopy = false;
}

CegoFieldValue::CegoFieldValue(CegoDataType type, void* pValue, int len, bool isLocalCopy)
{ 
    _type = type;
    _pV = pValue;
    _len = len;
    _isLocalCopy = isLocalCopy;
}

CegoFieldValue::CegoFieldValue(CegoDataType type, const Chain& v)
{
    if ( v.length() < 2 )
    {
	_type = NULL_TYPE;
	_pV = 0;
	_len = 0;
    }
    else
    {	
	_type = type;
	_isLocalCopy = true;
	switch ( type )
	{
	case INT_TYPE:
	{
	    _len = sizeof(int);
	    _pV = _staticBuf; 
	    int i = v.asInteger();
	    memcpy(_pV, &i, _len);
	    break;
	}
	case LONG_TYPE:
	{
	    _len = sizeof(long long);
	    _pV = _staticBuf;
	    long long l = v.asLongLong();
	    memcpy(_pV, &l, _len);
	    break;
	}
	case VARCHAR_TYPE:
	case BIGINT_TYPE:
	case DECIMAL_TYPE:
	case FIXED_TYPE:
	{
	    _len = v.length();
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    memcpy(_pV, (char*)v, _len);
	    break;
	}
	case DATETIME_TYPE:
	{
	    _len = sizeof(int);
	    _pV = _staticBuf;
	    
	    if ( v == Chain("sysdate") )
	    {    
		Datetime dt;
		*(int*)_pV = dt.asInt();
	    }
	    else
	    {
		Datetime dt(v, Chain(__dateTimeFormat));
		*(int*)_pV = dt.asInt();	    
	    }	    
	    break;
	}	
	case FLOAT_TYPE:
	{
	    _len = sizeof(float);
	    _pV = _staticBuf;
	   
	    Chain normFloat = v;
	    normFloatValue(normFloat);
	    	    
	    float f = normFloat.asFloat();
	    memcpy(_pV, &f, _len);
	    break;
	}
	case DOUBLE_TYPE:
	{
	    _len = sizeof(double);
	    _pV = _staticBuf;
	    double d = v.asDouble();
	    memcpy(_pV, &d, _len);
	    break;
	}
	case BOOL_TYPE:
	{
	    _len = sizeof(char);
	    _pV = _staticBuf;
	    if ( v.asBool() )
		*(char*)_pV = 1;
	    else
		*(char*)_pV = 0;
	    break;
	}
	case SMALLINT_TYPE:
	{
	    _len = sizeof(short);
	    _pV = _staticBuf;
	    short d = v.asShort();
	    memcpy(_pV, &d, _len);	
	    break;
	}
	case TINYINT_TYPE:
	{
	    _len = sizeof(char);
	    _pV = _staticBuf;
	    char d = (char)v.asInteger();
	    memcpy(_pV, &d, _len);
	    break;
	}
	case BLOB_TYPE:
	case CLOB_TYPE:
	{
	    _len = 2 * sizeof(int);
	    _pV = _staticBuf;
	    Tokenizer tok(v, Chain(LOBSEP));
	    Chain fstr;
	    Chain pstr;
	    int fileId;
	    int pageId;
	    if ( tok.nextToken(fstr) )
	    {
		fileId = fstr.asInteger();
		memcpy(_pV, &fileId, sizeof(int));
	    }
	    if ( tok.nextToken(pstr) )
	    {
		pageId = pstr.asInteger();
		memcpy( (void*)((long long)_pV + sizeof(int)), &pageId, sizeof(int));
	    }	    
	    break;	    
	}
	case NULL_TYPE:
	{
	    _pV = 0;
	    _len = 0;
	    break;
	}
	}
    }
}

CegoFieldValue::~CegoFieldValue()
{  
    if ( _isLocalCopy && _pV && _pV != _staticBuf )
    {
	free(_pV);
    }
}

void CegoFieldValue::setType(const CegoDataType t)
{
    _type = t;
}

const CegoDataType CegoFieldValue::getType() const
{
    return _type;
}

void* CegoFieldValue::getValue() const
{
    return _pV;
}

int CegoFieldValue::getLength() const
{
    return _len;
}

bool CegoFieldValue::castTo(CegoDataType t, int d)
{
    if ( _type == t )
	return true;

    if ( _pV == 0 )
    {	
	return true;
    }
    
    if ( _isLocalCopy == false && _pV != 0)
    {
	void* pV = _pV;

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	memcpy(_pV, pV, _len);
	_isLocalCopy = true;
    }

    ///////////////////
    // cast int to X //
    ///////////////////

    if ( _type == INT_TYPE && t == LONG_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	long long longVal=(long long)intVal;
	
	_len = sizeof(long long);
	_pV = _staticBuf;
	_type = LONG_TYPE;
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == VARCHAR_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	Chain strVal(intVal);
	_len = strVal.length();

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == BOOL_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	_pV = _staticBuf;
	_len = sizeof(char);

	if ( intVal > 0 )
	    *(char*)_pV = 1;
	else
	    *(char*)_pV = 0;

	_type = BOOL_TYPE;
	return true;
    }
    else if ( _type == INT_TYPE && t == DATETIME_TYPE)
    {
	_type = DATETIME_TYPE;
	return true;	
    }
    else if ( _type == INT_TYPE && t == BIGINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	Chain bival(intVal);	
	_len = bival.length();

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = BIGINT_TYPE;
	memcpy(_pV, (char*)bival, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == FLOAT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	Chain floatChain(intVal);
	floatChain += Chain(__decimalPoint) + Chain("0");	
	float floatVal = floatChain.asFloat();	
	if ( _pV != _staticBuf )
	    free(_pV);	
	_pV = _staticBuf;
	_len = sizeof(float);	
	_type = FLOAT_TYPE;
	memcpy(_pV, &floatVal, _len);
	return true;	
    }
    else if ( _type == INT_TYPE && t == DOUBLE_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	Chain doubleChain(intVal);
	doubleChain += Chain(__decimalPoint) + Chain("0");	
	double doubleVal = doubleChain.asDouble();	
	if ( _pV != _staticBuf )
	    free(_pV);	
	_pV = _staticBuf;
	_len = sizeof(double);	
	_type = DOUBLE_TYPE;
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == DECIMAL_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain decval(intVal);
	decval += Chain(".0");
	_len = decval.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decval, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == FIXED_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain fixedVal(intVal);
	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == SMALLINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	short shortVal=(short)intVal;
	_len = sizeof(short);
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == INT_TYPE && t == TINYINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	char tinyVal=(char)intVal;	
	_len = sizeof(char);
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	memcpy(_pV, &tinyVal, _len);
	return true;
    }

    ////////////////////
    // cast long to X //
    ////////////////////

    else if ( _type == LONG_TYPE && t == INT_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal=(int)longVal;
	_len = sizeof(int);
	_pV = _staticBuf;
	_type = INT_TYPE;
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == VARCHAR_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain strVal(longVal);
	_len = strVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == BOOL_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	_len = sizeof(char);
	if ( longVal > 0 )
	    *(char*)_pV = 1;
	else
	    *(char*)_pV = 0;
	_type = BOOL_TYPE;
	return true;
    }
    else if ( _type == LONG_TYPE && t == DATETIME_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	int dateVal = (int)longVal;
	_pV = _staticBuf;
	_len = sizeof(int);
	_type = DATETIME_TYPE;
	memcpy(_pV, &dateVal, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == BIGINT_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain bival(longVal);	
	_len = bival.length();
	
	if ( _len > d )
	{
	    throw Exception(EXLOC, "Bigint range exceeds");
	}
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = BIGINT_TYPE;
	memcpy(_pV, (char*)bival, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == FLOAT_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain floatChain(longVal);
	floatChain += Chain(__decimalPoint) + Chain("0");
	float floatVal = floatChain.asFloat();	
	_pV = _staticBuf;
	_len = sizeof(float);
	_type = FLOAT_TYPE;
	memcpy(_pV, &floatVal, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == DOUBLE_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain doubleChain(longVal);
	doubleChain += Chain(__decimalPoint) + Chain("0");	
	double doubleVal = doubleChain.asDouble();	
	_pV = _staticBuf;
	_len = sizeof(double);	
	_type = DOUBLE_TYPE;
	memcpy(_pV, &doubleVal, _len);
	return true;	
    }
    else if ( _type == LONG_TYPE && t == DECIMAL_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain decval(longVal);
	decval += Chain(".0");	
	_len = decval.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decval, _len);
	return true;	
    }
    else if ( _type == LONG_TYPE && t == FIXED_TYPE)
    {	
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);			
	Chain fixedVal(longVal);
	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;	
    }
    else if ( _type == LONG_TYPE && t == SMALLINT_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);				
	short shortVal=(short)longVal;	
	_len = sizeof(short);
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == LONG_TYPE && t == TINYINT_TYPE)
    {
	long long longVal;
	memcpy(&longVal, _pV, sizeof( long long));
	if ( _pV != _staticBuf )
	    free(_pV);	
	char tinyVal=(char)longVal;	
	_len = sizeof(char);
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	memcpy(_pV, &tinyVal, _len);
	return true;
    }
    
    ///////////////////////
    // cast varchar to X //
    ///////////////////////

    else if ( _type == VARCHAR_TYPE && t == INT_TYPE)
    {
	Chain strVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal=(int)strVal.asInteger();
	_len = sizeof(int);
	_pV = _staticBuf;
	_type = INT_TYPE;
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == VARCHAR_TYPE && t == LONG_TYPE)
    {
	Chain strVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	long long longVal=(int)strVal.asLongLong();
	_len = sizeof(long long);
	_pV = _staticBuf;
	_type = LONG_TYPE;
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == VARCHAR_TYPE && t == BOOL_TYPE)
    {
	Chain strVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	_len = sizeof(char);
	_pV = _staticBuf;
	if ( strVal.asBool() )
	    *(char*)_pV = 1;
	else
	    *(char*)_pV = 0;
	_type = BOOL_TYPE;
	return true;	
    }
    else if ( _type == VARCHAR_TYPE && t == DATETIME_TYPE)
    {
	Datetime dt;

	__dateFormatLock.readLock(DBM_LOCKTIMEOUT);
	try
	{
	    dt = Datetime(Chain((char*)_pV), __dateFormatList);
	}
	catch ( Exception e )
	{
	    __dateFormatLock.unlock();
	    throw e;   
	}
	__dateFormatLock.unlock();
	
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	*(int*)_pV = dt.asInt();	    
	_len = sizeof(int);
	_type = DATETIME_TYPE;
	return true;	
    }
    else if ( _type == VARCHAR_TYPE && t == BIGINT_TYPE)
    {	
	// we construct big integer to check format
	BigInteger b = BigInteger( Chain((char*)_pV) );       
	_type = BIGINT_TYPE;
	return true;	
    }
    else if ( _type == VARCHAR_TYPE && t == FLOAT_TYPE)
    {	
	Chain strVal((char*)_pV);

	normFloatValue(strVal);
	
	float floatVal = strVal.asFloat();
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = FLOAT_TYPE;
	_len = sizeof(float);
	memcpy(_pV, &floatVal, _len);
	return true;
    }
    else if ( _type == VARCHAR_TYPE && t == DOUBLE_TYPE)
    {	
	Chain strVal((char*)_pV);
	normFloatValue(strVal);
	
	double doubleVal = strVal.asDouble();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = DOUBLE_TYPE;
	_len = sizeof(double);
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == VARCHAR_TYPE && t == DECIMAL_TYPE)
    {	
	// we construct big decimal to check format
	BigDecimal d = BigDecimal( Chain((char*)_pV) );	
	_type = DECIMAL_TYPE;
	return true;	
    }
    else if ( _type == VARCHAR_TYPE && t == FIXED_TYPE)
    {	
	// we construct big decimal to check format
	BigDecimal decVal = BigDecimal( Chain((char*)_pV) );
	Chain fixedVal(Chain((char*)_pV));
	       
	int pos;
	if ( fixedVal.posStr(Chain("."), pos, 0, 1) )
	{
	    
	    Chain normVal;
	    
	    // if precision of fixed is higher than the float value, we have to concat zero characters
	    if ( pos + d >= fixedVal.length() )
	    {
		normVal=fixedVal;
		int i=fixedVal.length();		
		while (i <= pos+d )
		{
		    normVal+=Chain("0");
		    i++;
		}
	    }
	    else
	    {		
		normVal = fixedVal.subChain(1, pos + d);
	    }
	   	    
	    _type = FIXED_TYPE;
	    _len = normVal.length();
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    memcpy(_pV, (char*)normVal, _len);
	    return true;	
	}
	throw Exception(EXLOC, "Cannot convert from string to fixed");
    }
    else if ( _type == VARCHAR_TYPE && t == SMALLINT_TYPE)
    {
	Chain strVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	short shortVal=(int)strVal.asShort();
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == VARCHAR_TYPE && t == TINYINT_TYPE)
    {
	Chain strVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	char tinyVal=(char)strVal.asInteger();
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	_len = sizeof(char);
	memcpy(_pV, &tinyVal, _len);
	return true;
    }


    ////////////////////////
    // cast bool to X //
    ////////////////////////

    else if ( _type == BOOL_TYPE && t == INT_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	int intVal = (int)boolVal;
	_type = INT_TYPE;
	_len = sizeof(int);
	_pV = _staticBuf;
	memcpy(_pV, &intVal, _len);
	return true;		
    }
    else if ( _type == BOOL_TYPE && t == LONG_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	long long  longVal = (long long)boolVal;
	_type = LONG_TYPE;
	_len = sizeof(long long);
	_pV = _staticBuf;
	memcpy(_pV, &longVal, _len);
	return true;		
    }
    else if ( _type == BOOL_TYPE && t == VARCHAR_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain strVal;
	if ( boolVal > 0 )
	    strVal = Chain("true");
	else
	    strVal = Chain("false");

	_type = VARCHAR_TYPE;
	_len = strVal.length();
	_pV = _staticBuf;
	memcpy(_pV, (char*)strVal, _len);
	return true;		
    }

    // else if ( _type == BOOL_TYPE && t == DATETIME_TYPE) - not supported
    else if ( _type == BOOL_TYPE && t == BIGINT_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain biVal;
	if ( boolVal > 0 )
	    biVal = Chain("1");
	else
	    biVal = Chain("0");
	
	_type = BIGINT_TYPE;
	_len = biVal.length();
	_pV = _staticBuf;
	memcpy(_pV, (char*)biVal, _len);
	return true;		

    }
    else if ( _type == BOOL_TYPE && t == FLOAT_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	float floatVal;
	if ( boolVal > 0 )
	    floatVal = 1.0;
	else
	    floatVal = 0.0;

	_type = FLOAT_TYPE;
	_len = sizeof(float);
	_pV = _staticBuf;
	memcpy(_pV, &floatVal, _len);
	return true;		
    }
    else if ( _type == BOOL_TYPE && t == DOUBLE_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	double doubleVal;
	if ( boolVal > 0 )
	    doubleVal = 1.0;
	else
	    doubleVal = 0.0;

	_type = DOUBLE_TYPE;
	_len = sizeof(double);
	_pV = _staticBuf;
	memcpy(_pV, &doubleVal, _len);
	return true;		
    }
    else if ( _type == BOOL_TYPE && t == DECIMAL_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain decVal;
	if ( boolVal > 0 )
	    decVal = Chain("1.0");
	else
	    decVal = Chain("0.0");
	
	_type = DECIMAL_TYPE;
	_len = decVal.length();
	_pV = _staticBuf;
	memcpy(_pV, (char*)decVal, _len);
	return true;
    }

    else if ( _type == BOOL_TYPE && t == FIXED_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain fixedVal;
	if ( boolVal > 0 )
	    fixedVal = Chain("1");
	else
	    fixedVal = Chain("0");

	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;	
    }
    else if ( _type == BOOL_TYPE && t == SMALLINT_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	short shortVal = (short)boolVal;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);
	_pV = _staticBuf;
	memcpy(_pV, &shortVal, _len);
	return true;		
    }
    else if ( _type == BOOL_TYPE && t == TINYINT_TYPE)
    {
	char boolVal;
	memcpy(&boolVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);

	char tinyVal = (char)boolVal;
	_type = TINYINT_TYPE;
	_len = sizeof(char);
	_pV = _staticBuf;
	memcpy(_pV, &tinyVal, _len);
	return true;		
    }
    
    ////////////////////////
    // cast datetime to X //
    ////////////////////////

    else if ( _type == DATETIME_TYPE && t == INT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}
	
	_len = sizeof(int);
	_pV = _staticBuf;
	_type = INT_TYPE;
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == LONG_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}

	long long longVal = (long long)intVal;
	_len = sizeof(long long);
	_pV = _staticBuf;
	_type = LONG_TYPE;
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == VARCHAR_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain strVal;
	if ( intVal == 0 )
	{
	    Datetime dt;
	    strVal = dt.asChain(__dateTimeFormat);
	}
	else
	{
	    Datetime dt(intVal);
	    strVal = dt.asChain(__dateTimeFormat);
	}
	
	_len = strVal.length();

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == BOOL_TYPE)
    {
	// not supported
	return false;
    }
    else if ( _type == DATETIME_TYPE && t == BIGINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}
	
	Chain bival(intVal);	
	_len = bival.length();

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = BIGINT_TYPE;
	memcpy(_pV, (char*)bival, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == FLOAT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);	
	
	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}

	Chain floatChain(intVal);
	floatChain += Chain(__decimalPoint) + Chain("0");	
	float floatVal = floatChain.asFloat();	
	_pV = _staticBuf;
	_len = sizeof(float);	
	_type = FLOAT_TYPE;
	memcpy(_pV, &floatVal, _len);
	return true;	
    }
    else if ( _type == DATETIME_TYPE && t == DOUBLE_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);	
	
	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}
	Chain doubleChain(intVal);
	doubleChain += Chain(__decimalPoint) + Chain("0");	
	double doubleVal = doubleChain.asDouble();	

	_pV = _staticBuf;
	_len = sizeof(double);	
	_type = DOUBLE_TYPE;
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == DECIMAL_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}

	Chain decval(intVal);
	decval += Chain(".0");
	_len = decval.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decval, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == FIXED_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}

	Chain fixedVal(intVal);
	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == SMALLINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}
	
	short shortVal=(short)intVal;
	_len = sizeof(short);
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == DATETIME_TYPE && t == TINYINT_TYPE)
    {
	int intVal;
	memcpy(&intVal, _pV, sizeof(int));
	if ( _pV != _staticBuf )
	    free(_pV);

	if ( intVal == 0 )
	{
	    Datetime dt;
	    intVal = dt.asInt();
	}

	char tinyVal=(char)intVal;
	_len = sizeof(char);
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	memcpy(_pV, &tinyVal, _len);
	return true;
    }

    ////////////////////////
    // cast bigint to X //
    ////////////////////////

    else if ( _type == BIGINT_TYPE && t == INT_TYPE)
    {
	Chain biVal((char*)_pV);
	int intVal = biVal.asInteger();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = INT_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;
    }

    else if ( _type == BIGINT_TYPE && t == LONG_TYPE)
    {
	Chain biVal((char*)_pV);
	long long longVal = biVal.asLongLong();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = LONG_TYPE;
	_len = sizeof(long long);
	memcpy(_pV, &longVal, _len);
	return true;
    }
	
    else if ( _type == BIGINT_TYPE && t == VARCHAR_TYPE)
    {
	_type = VARCHAR_TYPE;
	return true;		    	
    }
    else if ( _type == BIGINT_TYPE && t == BOOL_TYPE)
    {
	Chain biVal((char*)_pV);
	int intVal = biVal.asInteger();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	char boolVal;
	if ( intVal > 0 )
	{
	    boolVal=1;
	}
	else
	{
	    boolVal=0;
	}
	
	_type = BOOL_TYPE;
	_len = sizeof(int);

	memcpy(_pV, &boolVal, _len);
	return true;

    }
    else if ( _type == BIGINT_TYPE && t == DATETIME_TYPE)
    {
	Chain biVal((char*)_pV);
	int intVal = biVal.asInteger();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}

	_type = DATETIME_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == BIGINT_TYPE && t == FLOAT_TYPE)
    {	
	Chain biVal((char*)_pV);
	float floatVal = biVal.asFloat();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = FLOAT_TYPE;
	_len = sizeof(float);
	memcpy(_pV, &floatVal, _len);
	return true;
    }

    else if ( _type == BIGINT_TYPE && t == DOUBLE_TYPE)
    {	
	Chain biVal((char*)_pV);
	double doubleVal = biVal.asDouble();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = DOUBLE_TYPE;
	_len = sizeof(double);
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == BIGINT_TYPE && t == DECIMAL_TYPE)
    {
	Chain decVal((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);	
	
	decVal += Chain(".0");	
	_len = decVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decVal, _len);
	return true;	
    }
    else if ( _type == BIGINT_TYPE && t == FIXED_TYPE)
    {
	Chain fixedVal((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);	

	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;	
    }
    else if ( _type == BIGINT_TYPE && t == SMALLINT_TYPE)
    {
	Chain biVal((char*)_pV);
	short shortVal = biVal.asShort();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = SMALLINT_TYPE;
	_len = sizeof(short);	
	memcpy(_pV, &shortVal, _len);
	return true;		
    }
    else if ( _type == BIGINT_TYPE && t == TINYINT_TYPE)
    {
	Chain biVal((char*)_pV);
	char tinyVal = (char)biVal.asInteger();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = TINYINT_TYPE;
	_len = sizeof(char);	
	memcpy(_pV, &tinyVal, _len);
	return true;		
    }

    /////////////////////
    // cast float to X //
    /////////////////////

    else if ( _type == FLOAT_TYPE && t == INT_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal = (int)f;
	_type = INT_TYPE;
	_len = sizeof(int);	
	_pV = _staticBuf;	    
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == LONG_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	long long longVal = (long long)f;
	_type = LONG_TYPE;
	_len = sizeof(long long);	
	_pV = _staticBuf;	    
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == VARCHAR_TYPE)
    {	
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain flVal( f );

	normFloatValue(flVal);
				    
	_len = flVal.length();
	
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	    
	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)flVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == BOOL_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	char boolVal;
	if ( f > 0.0 )
	{
	    boolVal = 1;
	}
	else
	{
	    boolVal = 0;
	}

	_type = BOOL_TYPE;
	_len = sizeof(char);	
	_pV = _staticBuf;	    
	memcpy(_pV, &boolVal, _len);
	return true;
	
    }
    else if ( _type == FLOAT_TYPE && t == DATETIME_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	
	int intVal = (int)f;

	_type = DATETIME_TYPE;
	_len = sizeof(int);	
	_pV = _staticBuf;	    
	memcpy(_pV, &intVal, _len);
	return true;

    }
    else if ( _type == FLOAT_TYPE && t == BIGINT_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal = (int)f;

	Chain biVal(intVal);
	_type = BIGINT_TYPE;
	_len = biVal.length();	
	_pV = _staticBuf;	    
	memcpy(_pV, (char*)biVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == DOUBLE_TYPE)
    {	
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	double doubleVal = (double)f;
	_type = DOUBLE_TYPE;
	_len = sizeof(double);	
	_pV = _staticBuf;	    
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == DECIMAL_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain flVal( f );

	denormFloatValue(flVal);
		
	_len = flVal.length();

	if ( _len > d )
	    throw Exception(EXLOC, "Decimal range exceeds"); 

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)flVal, _len);
	return true;	
    }
    else if ( _type == FLOAT_TYPE && t == FIXED_TYPE)
    {	
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain fixedVal( f );

	denormFloatValue(fixedVal);

	int pos;
	if ( fixedVal.posStr(Chain("."), pos, 0, 1) )
	{
	    Chain normVal;
	    
	    // if precision of fixed is higher than the float value, we have to concat zero characters
	    if ( pos + d >= fixedVal.length() )
	    {
		normVal=fixedVal;
		int i=fixedVal.length();		
		while (i <= pos+d )
		{
		    normVal+=Chain("0");
		    i++;
		}
	    }
	    else
	    {		
		normVal = fixedVal.subChain(1, pos + d);
	    }
	    _len = normVal.length();
	
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    _type = FIXED_TYPE;
	    memcpy(_pV, (char*)normVal, _len);
	    return true;	
	}	    
	
	return false;
    }
    else if ( _type == FLOAT_TYPE && t == SMALLINT_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	short shortVal = (short)f;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);	
	_pV = _staticBuf;	    
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == FLOAT_TYPE && t == TINYINT_TYPE)
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	if ( _pV != _staticBuf )
	    free(_pV);
	char tinyVal = (char)f;
	_type = TINYINT_TYPE;
	_len = sizeof(char);	
	_pV = _staticBuf;	    
	memcpy(_pV, &tinyVal, _len);
	return true;
    }
    
    ////////////////////////
    // cast double to X //
    ////////////////////////

    else if ( _type == DOUBLE_TYPE && t == INT_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal = (int)dVal;
	_type = INT_TYPE;
	_len = sizeof(int);	
	_pV = _staticBuf;	    
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == DOUBLE_TYPE && t == LONG_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	long long longVal = (long long)dVal;
	_type = LONG_TYPE;
	_len = sizeof(long long);	
	_pV = _staticBuf;	    
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == DOUBLE_TYPE && t == VARCHAR_TYPE)
    {	
	double dVal;
	memcpy(&dVal, _pV, sizeof(dVal));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain strVal( dVal );

	denormFloatValue(strVal);
				    
	_len = strVal.length();
	
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	    
	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }

    else if ( _type == DOUBLE_TYPE && t == BOOL_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);

	char boolVal;
	if ( dVal >= 0.0 )
	{
	    boolVal = 1;
	}
	else
	{
	    boolVal = 0;
	}

	_type = BOOL_TYPE;
	_len = sizeof(char);	
	_pV = _staticBuf;	    
	memcpy(_pV, &boolVal, _len);
	return true;

    }
    else if ( _type == DOUBLE_TYPE && t == DATETIME_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	int intVal = (int)dVal;
	_type = DATETIME_TYPE;
	_len = sizeof(int);	
	_pV = _staticBuf;	    
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == DOUBLE_TYPE && t == BIGINT_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	long long longVal = (long long)dVal;
	Chain biVal(longVal);
	
	_type = BIGINT_TYPE;
	_len = biVal.length();
	
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	
	memcpy(_pV, (char*)biVal, _len);
	return true;	
    }
    else if ( _type == DOUBLE_TYPE && t == FLOAT_TYPE)
    {	
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	float fVal = (float)dVal;
	_type = FLOAT_TYPE;
	_len = sizeof(float);	
	_pV = _staticBuf;	    
	memcpy(_pV, &fVal, _len);
	return true;
    }
    else if ( _type == DOUBLE_TYPE && t == DECIMAL_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain decVal( dVal );

	denormFloatValue(decVal);
	       
	_len = decVal.length();

	if ( _len > d )
	    throw Exception(EXLOC, "Decimal range exceeds"); 

	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decVal, _len);
	return true;	
    }

    else if ( _type == DOUBLE_TYPE && t == FIXED_TYPE)
    {	
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);

	Chain fixedVal( dVal );

	denormFloatValue(fixedVal);
	
	int pos;
	if ( fixedVal.posStr(Chain("."), pos, 0, 1) )
	{
	    Chain normVal;

	    // if precision of fixed is higher than the float value, we have to concat zero characters
	    if ( pos + d >= fixedVal.length() )
	    {
		normVal=fixedVal;
		int i=fixedVal.length();		
		while (i <= pos+d )
		{
		    normVal+=Chain("0");
		    i++;
		}
	    }
	    else
	    {
		normVal = fixedVal.subChain(1, pos + d);
	    }
	    
	    _len = normVal.length();
	
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    _type = FIXED_TYPE;
	    memcpy(_pV, (char*)normVal, _len);
	    return true;	
	}	    
	
	return false;
    }
    else if ( _type == DOUBLE_TYPE && t == SMALLINT_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	short shortVal = (short)dVal;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);	
	_pV = _staticBuf;	    
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    else if ( _type == DOUBLE_TYPE && t == TINYINT_TYPE)
    {
	double dVal;
	memcpy(&dVal, _pV, sizeof(double));
	if ( _pV != _staticBuf )
	    free(_pV);
	char tinyVal = (char)dVal;
	_type = TINYINT_TYPE;
	_len = sizeof(char);	
	_pV = _staticBuf;	    
	memcpy(_pV, &tinyVal, _len);
	return true;
    }

    ///////////////////////
    // cast decimal to X //
    ///////////////////////

    else if ( _type == DECIMAL_TYPE && t == INT_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	// we cast from float to int
	int intVal = (int)decVal.asDouble();
	_pV = _staticBuf;
	_type = INT_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == DECIMAL_TYPE && t == LONG_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	// we cast from double to long long
	long long longVal = (long long)decVal.asDouble();	
	_pV = _staticBuf;
	_type = LONG_TYPE;
	_len = sizeof(long long);
	memcpy(_pV, &longVal, _len);
	return true;

    }
    else if ( _type == DECIMAL_TYPE && t == VARCHAR_TYPE)
    {
	_type = VARCHAR_TYPE;
	return true;	
    }
    else if ( _type == DECIMAL_TYPE && t == BOOL_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	// we cast from double to char
	char boolVal = (char)decVal.asDouble();
	if ( boolVal > 0 )
	    boolVal=1;
	else
	    boolVal=0;

	_pV = _staticBuf;
	_type = BOOL_TYPE;
	_len = sizeof(char);
	memcpy(_pV, &boolVal, _len);
	return true;

    }
    else if ( _type == DECIMAL_TYPE && t == DATETIME_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);

	// we cast from double to int
	int intVal = (int)decVal.asDouble();
	_pV = _staticBuf;
	_type = DATETIME_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == DECIMAL_TYPE && t == BIGINT_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	int pos;
	if ( decVal.posStr(Chain("."), pos, 0, 1) )
	{
	    Chain biVal = decVal.subChain(1, pos-1);

	    _len = biVal.length();
	
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    _type = BIGINT_TYPE;
	    memcpy(_pV, (char*)biVal, _len);
	    return true;
	}
	return false;	
    }
    else if ( _type == DECIMAL_TYPE && t == FLOAT_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	float floatVal = decVal.asFloat();
	_pV = _staticBuf;
	_type = FLOAT_TYPE;
	_len = sizeof(float);
	memcpy(_pV, &floatVal, _len);
	return true;       
    }   
    else if ( _type == DECIMAL_TYPE && t == DOUBLE_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	double doubleVal = decVal.asDouble();	
	_pV = _staticBuf;
	_type = DOUBLE_TYPE;
	_len = sizeof(double);
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == DECIMAL_TYPE && t == FIXED_TYPE)
    {
	
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	
	int pos;
	if ( fixedVal.posStr(Chain("."), pos, 0, 1) )
	{
	    Chain normVal;
	    // if precision of fixed is higher than the decimal value, we have to concat zero characters
	    if ( pos + d >= fixedVal.length() )
	    {
		
		normVal=fixedVal;
		int i=fixedVal.length();		
		while (i <= pos+d )
		{
		    normVal+=Chain("0");
		    i++;
		}
	    }
	    else
	    {		
		normVal = fixedVal.subChain(1, pos + d);
	    }
	    _len = normVal.length();
	
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    _type = FIXED_TYPE;
	    memcpy(_pV, (char*)normVal, _len);
	    return true;	
	}
	return false;
	
    }
    else if ( _type == DECIMAL_TYPE && t == SMALLINT_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	// we cast from float to int
	short shortVal = (short)decVal.asDouble();
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);
	memcpy(_pV, &shortVal, _len);
	return true;

    }
    else if ( _type == DECIMAL_TYPE && t == TINYINT_TYPE)
    {
	Chain decVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(decVal);
	
	// we cast from float to char
	char tinyVal = (char)decVal.asDouble();
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	_len = sizeof(char);
	memcpy(_pV, &tinyVal, _len);
	return true;
    }

    ////////////////////////
    // cast fixed to X //
    ////////////////////////

    else if ( _type == FIXED_TYPE && t == INT_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(fixedVal);
	
	// we cast from float to int
	int intVal = (int)fixedVal.asDouble();
	_pV = _staticBuf;
	_type = INT_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == FIXED_TYPE && t == LONG_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(fixedVal);
	
	// we cast from double to long long
	long long longVal = (long long)fixedVal.asDouble();	
	_pV = _staticBuf;
	_type = LONG_TYPE;
	_len = sizeof(long long);
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == FIXED_TYPE && t == VARCHAR_TYPE)
    {	
	// we construct big decimal to check format
	// BigDecimal d = BigDecimal( Chain((char*)_pV) );	
	_type = VARCHAR_TYPE;
	return true;	
    }
    else if ( _type == FIXED_TYPE && t == BOOL_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(fixedVal);
	
	// we cast from float to char
	char boolVal = (char)fixedVal.asDouble();
	if ( boolVal > 0 )
	    boolVal=1;
	else
	    boolVal=0;

	_pV = _staticBuf;
	_type = BOOL_TYPE;
	_len = sizeof(char);
	memcpy(_pV, &boolVal, _len);
	return true;

    }
    else if ( _type == FIXED_TYPE && t == DATETIME_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(fixedVal);
	
	// we cast from float to int
	int intVal = (int)fixedVal.asDouble();
	_pV = _staticBuf;
	_type = DATETIME_TYPE;
	_len = sizeof(int);
	memcpy(_pV, &intVal, _len);
	return true;

    }
    else if ( _type == FIXED_TYPE && t == BIGINT_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	int pos;
	if ( fixedVal.posStr(Chain("."), pos, 0, 1) )
	{
	    Chain biVal = fixedVal.subChain(1, pos-1);

	    _len = biVal.length();
	
	    if ( _len <= STATICFIELDBUF )
		_pV = _staticBuf;
	    else
		_pV = malloc(_len);
	    
	    _type = BIGINT_TYPE;
	    memcpy(_pV, (char*)biVal, _len);
	    return true;
	}
	return false;
    }    
    else if ( _type == FIXED_TYPE && t == FLOAT_TYPE)
    {	
	Chain strVal((char*)_pV);

	normFloatValue(strVal);
	
	float floatVal = strVal.asFloat();
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = FLOAT_TYPE;
	_len = sizeof(float);
	memcpy(_pV, &floatVal, _len);
	return true;
    }
    else if ( _type == FIXED_TYPE && t == DOUBLE_TYPE)
    {	
	Chain strVal((char*)_pV);

	normFloatValue(strVal);
	
	double doubleVal = strVal.asDouble();	
    	if ( _pV != _staticBuf )
	{
	    free(_pV);
	    _pV = _staticBuf;
	}
	_type = DOUBLE_TYPE;
	_len = sizeof(double);
	memcpy(_pV, &doubleVal, _len);
	return true;
    }
    else if ( _type == FIXED_TYPE && t == DECIMAL_TYPE)
    {	
	_type = DECIMAL_TYPE;
	return true;	
    }
    else if ( _type == FIXED_TYPE && t == SMALLINT_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);

	normFloatValue(fixedVal);
	
	// we cast from float to short
	short shortVal = (short)fixedVal.asDouble();
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	_len = sizeof(short);
	memcpy(_pV, &shortVal, _len);
	return true;

    }
    else if ( _type == FIXED_TYPE && t == TINYINT_TYPE)
    {
	Chain fixedVal = Chain((char*)_pV);
	if ( _pV != _staticBuf )
	    free(_pV);
	
	normFloatValue(fixedVal);
	
	// we cast from float to char
	char tinyVal = (char)fixedVal.asDouble();
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	_len = sizeof(char);
	memcpy(_pV, &tinyVal, _len);
	return true;

    }

    ////////////////////////
    // cast smallint to X //
    ////////////////////////
    
    else if ( _type == SMALLINT_TYPE && t == INT_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);	
	int intVal=(int)shortVal;
	_len = sizeof(int);
	_pV = _staticBuf;
	_type = INT_TYPE;
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == LONG_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);	
	long long longVal=(long long)shortVal;
	_len = sizeof(long long);
	_pV = _staticBuf;
	_type = LONG_TYPE;
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == VARCHAR_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain strVal(shortVal);
	_len = strVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == BOOL_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	_len = sizeof(char);
	if ( shortVal > 0 )
	    *(char*)_pV = 1;
	else
	    *(char*)_pV = 0;
	_type = BOOL_TYPE;
	return true;
    }
    // else if ( _type == SMALLINT_TYPE && t == DATETIME_TYPE) not supported
    else if ( _type == SMALLINT_TYPE && t == BIGINT_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain bival(shortVal);	
	_len = bival.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = BIGINT_TYPE;
	memcpy(_pV, (char*)bival, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == FLOAT_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	Chain floatChain(shortVal);
	floatChain += Chain(__decimalPoint) + Chain("0");
	float floatVal = floatChain.asFloat();
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	_len = sizeof(float);
	_type = FLOAT_TYPE;
	memcpy(_pV, &floatVal, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == DOUBLE_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	Chain doubleChain(shortVal);
	doubleChain += Chain(__decimalPoint) + Chain("0");	
	double doubleVal = doubleChain.asDouble();	
	if ( _pV != _staticBuf )
	    free(_pV);	
	_pV = _staticBuf;
	_len = sizeof(double);	
	_type = DOUBLE_TYPE;
	memcpy(_pV, &doubleVal, _len);
	return true;	
    }
    else if ( _type == SMALLINT_TYPE && t == DECIMAL_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain decval(shortVal);
	decval += Chain(".0");
	_len = decval.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decval, _len);
	return true;	
    }
    else if ( _type == SMALLINT_TYPE && t == FIXED_TYPE)
    {     
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain fixedVal(shortVal);
	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;
    }
    else if ( _type == SMALLINT_TYPE && t == TINYINT_TYPE)
    {
	short shortVal;
	memcpy(&shortVal, _pV, sizeof(short));
	if ( _pV != _staticBuf )
	    free(_pV);	
	char tinyVal=(char)shortVal;	
	_len = sizeof(char);
	_pV = _staticBuf;
	_type = TINYINT_TYPE;
	memcpy(_pV, &tinyVal, _len);
	return true;
    }

    ////////////////////////
    // cast tinyint to X //
    ////////////////////////
    
    else if ( _type == TINYINT_TYPE && t == INT_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);	
	int intVal=(int)tinyVal;
	_len = sizeof(int);
	_pV = _staticBuf;
	_type = INT_TYPE;
	memcpy(_pV, &intVal, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == LONG_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);	
	long long longVal=(long long)tinyVal;
	_len = sizeof(long long);
	_pV = _staticBuf;
	_type = LONG_TYPE;
	memcpy(_pV, &longVal, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == VARCHAR_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain strVal(tinyVal);
	_len = strVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = VARCHAR_TYPE;
	memcpy(_pV, (char*)strVal, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == BOOL_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	_len = sizeof(char);
	if ( tinyVal > 0 )
	    *(char*)_pV = 1;
	else
	    *(char*)_pV = 0;
	_type = BOOL_TYPE;
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == DATETIME_TYPE)
    {
	// not supported
	return false;
    }
    else if ( _type == TINYINT_TYPE && t == BIGINT_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain bival(tinyVal);	
	_len = bival.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = BIGINT_TYPE;
	memcpy(_pV, (char*)bival, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == FLOAT_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	Chain floatChain(tinyVal);
	floatChain += Chain(__decimalPoint) + Chain("0");
	float floatVal = floatChain.asFloat();
	if ( _pV != _staticBuf )
	    free(_pV);
	_pV = _staticBuf;
	_len = sizeof(float);
	_type = FLOAT_TYPE;
	memcpy(_pV, &floatVal, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == DOUBLE_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	Chain doubleChain((int)tinyVal);
	doubleChain += Chain(__decimalPoint) + Chain("0");	
	double doubleVal = doubleChain.asDouble();	
	if ( _pV != _staticBuf )
	    free(_pV);	
	_pV = _staticBuf;
	_len = sizeof(double);	
	_type = DOUBLE_TYPE;
	memcpy(_pV, &doubleVal, _len);
	return true;	
    }
    else if ( _type == TINYINT_TYPE && t == DECIMAL_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);	
	Chain decval((int)tinyVal);
	decval += Chain(".0");
	_len = decval.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = DECIMAL_TYPE;
	memcpy(_pV, (char*)decval, _len);
	return true;	
    }
    else if ( _type == TINYINT_TYPE && t == FIXED_TYPE)
    {     
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);
	Chain fixedVal((int)tinyVal);
	fixedVal += Chain(".");
	for ( int i=0; i<d; i++ )
	    fixedVal += Chain("0");
	_len = fixedVal.length();
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);
	_type = FIXED_TYPE;
	memcpy(_pV, (char*)fixedVal, _len);
	return true;
    }
    else if ( _type == TINYINT_TYPE && t == SMALLINT_TYPE)
    {
	char tinyVal;
	memcpy(&tinyVal, _pV, sizeof(char));
	if ( _pV != _staticBuf )
	    free(_pV);	
	short shortVal=(short)tinyVal;	
	_len = sizeof(short);
	_pV = _staticBuf;
	_type = SMALLINT_TYPE;
	memcpy(_pV, &shortVal, _len);
	return true;
    }
    
    return false;    
}

void CegoFieldValue::setValue(void* pV)
{
    _pV = pV;
}

void CegoFieldValue::setLength(int len)
{
    _len = len;
}

CegoFieldValue& CegoFieldValue::operator = ( const CegoFieldValue& fv)
{
    _type = fv._type;
    _len = fv._len;
    
    if (_isLocalCopy && _pV && _pV != _staticBuf )
	free (_pV);
    
    _isLocalCopy = fv._isLocalCopy;

    if (_isLocalCopy && fv._pV)
    {
	if ( _len <= STATICFIELDBUF )
	    _pV = _staticBuf;
	else
	    _pV = malloc(_len);

	memcpy(_pV, fv._pV, _len);
    }
    else
    {
	_pV = fv._pV;
    }
    return (*this);
}

bool CegoFieldValue::operator == ( const CegoFieldValue& fv) const
{

    if ( isNull() && fv.isNull() )
	return true;
    if ( isNull() )
	return false;
    if ( fv.isNull() )
	return false;
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return isEqual(fv2);
	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == true )
	    return fv2.isEqual(fv);
	       
	Chain msg = Chain("Mismatched datatypes ") + CEGO_TYPE_MAP[_type]
	    + Chain(" and ") + CEGO_TYPE_MAP[fv._type];
	throw Exception(EXLOC , msg);
	

    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }

    return isEqual(fv);
}

bool CegoFieldValue::operator != ( const CegoFieldValue& fv) const
{    
    if ( isNull() && fv.isNull() )
	return false;
    if ( isNull() )
	return true;
    if ( fv.isNull() )
	return true;
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return ! isEqual(fv2);

	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == false )
	    return ! fv2.isEqual(fv);
	
	throw Exception(EXLOC , "Incompatible Datatypes");

    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    
    return ! isEqual(fv);
}

bool CegoFieldValue::operator < ( const CegoFieldValue& fv) const
{

    if ( isNull() && fv.isNull() )
	return false;
    if ( isNull() )
	return true;
    if ( fv.isNull() )
	return false;
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return isLess(fv2);

	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == true )
	    return fv2.isLess(fv);
	
	Chain msg = Chain("Mismatched Datatypes ") + CEGO_TYPE_MAP[_type] + Chain(" != ") + CEGO_TYPE_MAP[fv._type];
	throw Exception(EXLOC , msg);

    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    
    return isLess(fv);
}

bool CegoFieldValue::operator > ( const CegoFieldValue& fv) const 
{
    if ( isNull() && fv.isNull() )
	return false;
    if ( isNull() )
	return false;
    if ( fv.isNull() )
	return true;
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return isMore(fv2);
	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == true )
	    return fv2.isMore(fv);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return isMore(fv);
}

bool CegoFieldValue::operator <= ( const CegoFieldValue& fv) const
{

    if ( isNull() && fv.isNull() )
	return true;
    if ( isNull() )
	return true;
    if ( fv.isNull() )
	return false;
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return isLessEqual(fv2);

	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == true )
	    return fv2.isLessEqual(fv);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return isLessEqual(fv);
}

bool CegoFieldValue::operator >= ( const CegoFieldValue& fv) const
{

    if ( isNull() && fv.isNull() )
	return true;
    if ( isNull() )
	return false;
    if ( fv.isNull() )
	return true;
    
    
    if (_type != fv._type )
    {
	CegoFieldValue fv2 = fv;
	if ( fv2.castTo(_type) == true )
	    return isMoreEqual(fv2);

	fv2 = *this;
	if ( fv2.castTo(fv.getType()) == true )
	    return fv2.isMoreEqual(fv);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (_pV == 0 || fv._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return isMoreEqual(fv);
}

CegoFieldValue::Comparison CegoFieldValue::comp( const CegoFieldValue& fv) const
{

    if ( isNull() && fv.isNull() )
	return EQUAL;
    if ( isNull() )
	return LESS;
    if ( fv.isNull() )
	return MORE;
    
    switch (_type)
    {
    case INT_TYPE:
    {	
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if (i1 < i2)
	    return LESS;
	if ( i1 > i2)
	    return MORE;
	return EQUAL;	
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	if (i1 < i2)
	    return LESS;
	if ( i1 > i2)
	    return MORE;
	return EQUAL;
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof( long long));
	memcpy(&l2, fv._pV, sizeof(long long));

	if (l1 < l2)
	    return LESS;
	if ( l1 > l2)
	    return MORE;
	return EQUAL;
    }
    case VARCHAR_TYPE:
    {
	int i;

	if ( __caseSensitiveFlag == 2 )
	    i = strncasecmp((char*)_pV, (char*)fv._pV, fv._len);
	else
	    i = strncmp((char*)_pV, (char*)fv._pV, fv._len);
	
	if ( i < 0 )
	    return LESS;
	if ( i > 0 )
	    return MORE;
	return EQUAL;
    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);

	if ( b1 < b2 )
	    return LESS;
	if ( b1 > b2 )
	    return MORE;
	return EQUAL;
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));

	if ( bi1 < bi2 )
	    return LESS;
	if ( bi1 > bi2 )
	    return MORE;
	return EQUAL;
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));

	if ( f1 < f2 )
	    return LESS;
	if ( f1 > f2 )
	    return MORE;
	return EQUAL;
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));	

	if ( d1 < d2 )
	    return LESS;
	if ( d1 > d2 )
	    return MORE;
	return EQUAL;
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));

	if ( d1 < d2 )
	    return LESS;
	if ( d1 > d2 )
	    return MORE;
	return EQUAL;
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));

	if ( s1 < s2 )
	    return LESS;
	if ( s1 > s2 )
	    return MORE;
	return EQUAL;
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);

	if ( c1 < c2 )
	    return LESS;
	if ( c1 > c2 )
	    return MORE;
	return EQUAL;
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

bool CegoFieldValue::isEqual( const CegoFieldValue& fv) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	return (i1 == i2);	
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	return (i1 == i2);
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv._pV, sizeof(long long));
	return (l1 == l2);
    }
    case VARCHAR_TYPE:
    {	
	if ( __caseSensitiveFlag == 2 )
	    return ( strncasecmp((char*)_pV, (char*)fv._pV, fv._len) == 0 );
	else
	    return ( strncmp((char*)_pV, (char*)fv._pV, fv._len) == 0 );	
    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);
	return (b1 == b2);
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));
	return (bi1 == bi2);
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));
	return (f1 == f2);
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));
	return (d1 == d2);
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));
	return (d1 == d2);
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));
	return (s1 == s2);
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);
	return (c1 == c2);
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

bool CegoFieldValue::isLess( const CegoFieldValue& fv) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	return (i1 < i2);
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	return (i1 < i2);
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv._pV, sizeof(long long));
	return (l1 < l2);
    }
    case VARCHAR_TYPE:
    {
	if ( __caseSensitiveFlag == 2 )
	    return ( strncasecmp((char*)_pV, (char*)fv._pV, fv._len) < 0 );	
	else
	    return ( strncmp((char*)_pV, (char*)fv._pV, fv._len) < 0 );
    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);
	return (b1 < b2);
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));
	return (bi1 < bi2);
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));
	return (f1 < f2);
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));	
	return (d1 < d2);
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));
	return (d1 < d2);
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));
	return (s1 < s2);
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);
	return (c1 < c2);
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

bool CegoFieldValue::isMore( const CegoFieldValue& fv) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	return (i1 > i2);
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	return (i1 > i2);
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv._pV, sizeof(long long));
	return (l1 > l2);
    }
    case VARCHAR_TYPE:
    {	
	if ( __caseSensitiveFlag == 2 )
	    return ( strncasecmp((char*)_pV, (char*)fv._pV, fv._len) > 0 );
	else
	    return ( strncmp((char*)_pV, (char*)fv._pV, fv._len) > 0 );
       
    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);
	return (b1 > b2);
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));
	return (bi1 > bi2);
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));
	return (f1> f2);
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));	
	return (d1 > d2);
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));
	return (d1 > d2);
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));
	return (s1 > s2);
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);
	return (c1 > c2);
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

bool CegoFieldValue::isLessEqual( const CegoFieldValue& fv) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	return (i1 <= i2);
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	return (i1 <= i2);
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv._pV, sizeof(long long));
	return (l1 <= l2);
    }
    case VARCHAR_TYPE:
    {
	if ( __caseSensitiveFlag == 2 )
	    return ( strncasecmp((char*)_pV, (char*)fv._pV, fv._len) <= 0 );
	else
	    return ( strncmp((char*)_pV, (char*)fv._pV, fv._len) <= 0 );
    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);
	return (b1 <= b2);
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));
	return (bi1 <= bi2);
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));
	return (f1 <= f2);
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));	
	return (d1 <= d2);
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));
	return (d1 <= d2);
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));
	return (s1 <= s2);
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);
	return (c1 <= c2);
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

bool CegoFieldValue::isMoreEqual( const CegoFieldValue& fv) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	return (i1 >= i2);
    }
    case DATETIME_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv._pV, sizeof(int));
	if ( i1 == 0 )
	{
	    Datetime dt;
	    i1 = dt.asInt();
	}
	if ( i2 == 0 )
	{
	    Datetime dt;
	    i2 = dt.asInt();
	}
	return (i1 >= i2);
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv._pV, sizeof(long long));
	return (l1 >= l2);
    }
    case VARCHAR_TYPE:
    {
	if ( __caseSensitiveFlag == 2 )
	    return ( strncasecmp((char*)_pV, (char*)fv._pV, fv._len) >= 0 );
	else
	    return ( strncmp((char*)_pV, (char*)fv._pV, fv._len) >= 0 );

    }
    case BOOL_TYPE:
    {
	char b1 = *(char*)(_pV);
	char b2 = *(char*)(fv._pV);
	return (b1 >= b2);
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)_pV) );
	BigInteger bi2 = BigInteger( Chain((char*)(fv._pV) ));
	return (bi1 >= bi2);
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv._pV, sizeof(float));
	return (f1>= f2);
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv._pV, sizeof(double));	
	return (d1 >= d2);
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)_pV) );
	BigDecimal d2 = BigDecimal( Chain((char*)(fv._pV) ));
	return (d1 >= d2);
    }
    case SMALLINT_TYPE:
    {
	short s1,s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv._pV, sizeof(short));
	return (s1 >= s2);
    }
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv._pV);
	return (c1 >= c2);
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

CegoFieldValue operator + ( const CegoFieldValue& fv1, const CegoFieldValue& fv2 )
{
    if ( fv1._type == NULL_TYPE ) 
    {
	return fv2;
	// throw Exception(EXLOC , "Cannot operate on null value");
    }
    if ( fv2._type == NULL_TYPE )
    {
	return fv1;
    }

    if (fv1._type != fv2._type)
    {
	CegoFieldValue fv3 = fv2;
	if ( fv3.castTo(fv1.getType()) == true )
	    return fv1.add(fv3);
	
	fv3 = fv1;
	if ( fv3.castTo(fv2.getType()) == true )
	    return fv2.add(fv3);
	    
	throw Exception(EXLOC , "Incompatible Datatypes");
	
    }
    if (fv1._pV == 0 || fv2._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return fv1.add(fv2);
}

CegoFieldValue CegoFieldValue::add( const CegoFieldValue& fv2 ) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv2._pV, sizeof(int));
	int* pI = new int;
	*pI = i1 + i2;
	CegoFieldValue fv3(_type, pI, sizeof(int), true);
	return fv3;
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv2._pV, sizeof(long long));
	long* pL = new long;
	*pL = l1 + l2;
	CegoFieldValue fv3(_type, pL, sizeof(long), true);
	return fv3;
    }
    case VARCHAR_TYPE:
    {
	Chain s1 = Chain((char*)(_pV), _len - 1);
	Chain s2 = Chain((char*)(fv2._pV), fv2._len - 1);
	Chain m =  s1 + s2;
	CegoFieldValue fv3(_type, m);
	return fv3;
    }
    case BOOL_TYPE:
    case DATETIME_TYPE:
    {
	throw Exception(EXLOC, "Operation not supported");
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger( Chain((char*)(_pV)) );
	BigInteger bi2 =  BigInteger( Chain((char*)(fv2._pV)) );
	BigInteger bi3 = bi1.add(bi2);	
	CegoFieldValue fv3(_type, bi3.toChain());
	return fv3;
    }
    case DECIMAL_TYPE:
    {
	BigDecimal d1 = BigDecimal((char*)(_pV));
	BigDecimal d2 =  BigDecimal((char*)(fv2._pV));	
	BigDecimal d3 = d1.add(d2);
	CegoFieldValue fv(_type, d3.toChain());	
	return fv;
    }
    case FIXED_TYPE:
    {

	Chain f1((char*)(_pV));
	Chain f2((char*)(fv2._pV));
	
	int pos1;
	if ( f1.posStr(Chain("."), pos1) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim1 =  _len - pos1 - 1;

	int pos2;	
	if ( f2.posStr(Chain("."), pos2) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim2 = fv2._len - pos2 - 1;

	int fixDim = dim1 > dim2 ? dim1 : dim2;
	
	BigDecimal d1 = BigDecimal(f1);
	BigDecimal d2 =  BigDecimal(f2);
	
	BigDecimal d3 = d1.add(d2);
	

	Chain f3 = d3.toChain();
	int pos3;
	if ( f3.posStr(Chain("."), pos3) == false )
	    throw Exception(EXLOC, "Invalid fixed value");
	int i = f3.length() - pos3 -1;

	while ( i < fixDim )
	{
	    f3 = f3 + Chain("0");
	    i++;
	}
	
	CegoFieldValue fv3(_type, f3);
	
	return fv3;
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv2._pV, sizeof(float));	
	float* pF = new float;
	*pF = f1 + f2;
	CegoFieldValue fv3(_type, pF, sizeof(float), true);
	return fv3;
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv2._pV, sizeof(double));
	double* pD = new double;
	*pD = d1 + d2;
	CegoFieldValue fv3(_type, pD, sizeof(double), true);
	return fv3;
    }
    case SMALLINT_TYPE:
    {
	short s1, s2;
	memcpy(&s1,_pV, sizeof(short));
	memcpy(&s2, fv2._pV, sizeof(short));	
	short* pS = new short;
	*pS = s1 + s2;
	CegoFieldValue fv3(_type, pS, sizeof(short), true);
	return fv3;
    }	
    case TINYINT_TYPE:
    {	
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv2._pV);	
	char* pC = new char;
	*pC = c1 + c2;
	CegoFieldValue fv3(_type, pC, sizeof(char), true);
	return fv3;
    }
    default:	
	throw Exception(EXLOC, "Unknown Type");
    }
}

CegoFieldValue operator - ( const CegoFieldValue& fv1, const CegoFieldValue& fv2 )
{
    if ( fv2._type == NULL_TYPE )
    {
	return fv1;
    }
    if ( fv1._type == NULL_TYPE )
    {
	return fv2.negate();
    }
    if (fv1._type != fv2._type)
    {
	CegoFieldValue fv3 = fv2;
	if ( fv3.castTo(fv1.getType()) == true )
	    return fv1.sub(fv3);

	fv3 = fv1;
	if ( fv3.castTo(fv2.getType()) == true )
	    return fv3.sub(fv2);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (fv1._pV == 0 || fv2._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return fv1.sub(fv2);
}

CegoFieldValue CegoFieldValue::sub( const CegoFieldValue& fv2 ) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv2._pV, sizeof(int));	
	int* pI = new int;
	*pI = i1 - i2;
	CegoFieldValue fv3(_type, pI, sizeof(int), true);
	return fv3;
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv2._pV, sizeof(long long));
	long long* pL = new long long;
	*pL = l1 - l2;
	CegoFieldValue fv3(_type, pL, sizeof(long long), true);
	return fv3;
    }
    case VARCHAR_TYPE:
    case BOOL_TYPE:
    case DATETIME_TYPE:
    {
	throw Exception(EXLOC, "Operation not supported");	
    }
    case BIGINT_TYPE:
    {	
	BigInteger bi1 = BigInteger ( Chain((char*)(_pV)) );
	BigInteger bi2 = BigInteger ( Chain((char*)(fv2._pV)) );	
	BigInteger bi3 =  bi1.sub(bi2);	
	CegoFieldValue fv3(_type, bi3.toChain());	
	return fv3;
    }
    case DECIMAL_TYPE:
    {
	BigDecimal d1 = BigDecimal((char*)(_pV));
	BigDecimal d2 =  BigDecimal((char*)(fv2._pV));	
	BigDecimal d3 = d1.sub(d2);
	CegoFieldValue fv(_type, d3.toChain());	
	return fv;
    }
    case FIXED_TYPE:
    {	

	Chain f1((char*)(_pV));
	Chain f2((char*)(fv2._pV));

	int pos1;
	if ( f1.posStr(Chain("."), pos1) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim1 =  _len - pos1 - 1;

	int pos2;
	if ( f2.posStr(Chain("."), pos2) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim2 = fv2._len - pos2 - 1;

	int fixDim = dim1 > dim2 ? dim1 : dim2;

	BigDecimal d1 = BigDecimal(f1);
	BigDecimal d2 =  BigDecimal(f2);
	BigDecimal d3 = d1.sub(d2);

	Chain f3 = d3.toChain();
	int pos3;
	if ( f3.posStr(Chain("."), pos3) == false )
	    throw Exception(EXLOC, "Invalid fixed value");
	int i = f3.length() - pos3 -1;

	while ( i < fixDim )
	{
	    f3 = f3 + Chain("0");
	    i++;
	}
	
	CegoFieldValue fv3(_type, f3);

	return fv3;
    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv2._pV, sizeof(float));	
	float* pF = new float;
	*pF = f1 - f2;
	CegoFieldValue fv3(_type, pF, sizeof(float), true);
	return fv3;
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv2._pV, sizeof(double));	
	double* pD = new double;
	*pD = d1 - d2;
	CegoFieldValue fv3(_type, pD, sizeof(double), true);
	return fv3;
    }
    case SMALLINT_TYPE:
    {
	short s1, s2;
	memcpy(&s1, _pV, sizeof(short));
	memcpy(&s2, fv2._pV, sizeof(short));	
	short* pS = new short;
	*pS = s1 - s2;
	CegoFieldValue fv3(_type, pS, sizeof(short), true);
	return fv3;
    }	
    case TINYINT_TYPE:
    {	
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv2._pV);
	
	char* pC = new char;
	*pC = c1 - c2;
	CegoFieldValue fv3(_type, pC, sizeof(char), true);
	return fv3;
    }
    default:
	throw Exception(EXLOC, "Unknown Type");	
    }
}

CegoFieldValue operator * ( const CegoFieldValue& fv1, const CegoFieldValue& fv2 )
{
    if ( fv1._type == NULL_TYPE || fv2._type == NULL_TYPE)
    {
	throw Exception(EXLOC , "Cannot operate on null value");
    }
    if (fv1._type != fv2._type)
    {
	CegoFieldValue fv3 = fv2;
	if ( fv3.castTo(fv1.getType()) == true )
	    return fv1.mul(fv3);

	fv3 = fv1;
	if ( fv3.castTo(fv2.getType()) == true )
	    return fv2.mul(fv3);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (fv1._pV == 0 || fv2._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }

    return fv1.mul(fv2);
}

CegoFieldValue CegoFieldValue::mul( const CegoFieldValue& fv2 ) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv2._pV, sizeof(int));		
	int* pI = new int;
	*pI = i1 * i2;
	CegoFieldValue fv3(_type, pI, sizeof(int), true);
	return fv3;
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv2._pV, sizeof(long long));
	long long* pL = new long long;
	*pL = l1 * l2;
	CegoFieldValue fv3(_type, pL, sizeof(long long), true);
	return fv3;
    }
    case VARCHAR_TYPE:
    case BOOL_TYPE:
    case DATETIME_TYPE:
    {
	throw Exception(EXLOC, "Operation not supported");	
    }
    case BIGINT_TYPE:
    {
	BigInteger bi1 = BigInteger ( Chain((char*)(_pV)) );
	BigInteger bi2 = BigInteger ( Chain((char*)(fv2._pV)) );	
	BigInteger bi3 =  bi1.mul(bi2);
	CegoFieldValue fv3(_type, bi3.toChain());
	return fv3;
    }
    case DECIMAL_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)(_pV)) );
	BigDecimal d2 =  BigDecimal( Chain((char*)(fv2._pV)) );	
	BigDecimal d3 = d1.mul(d2);
	CegoFieldValue fv3(_type, d3.toChain());
	return fv3;
    }
    case FIXED_TYPE:
    {
	Chain f1((char*)(_pV));
	Chain f2((char*)(fv2._pV));

	int pos1;
	if ( f1.posStr(Chain("."), pos1) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim1 =  _len - pos1 - 1;

	int pos2;
	if ( f2.posStr(Chain("."), pos2) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim2 = fv2._len - pos2 - 1;

	int fixDim = dim1 > dim2 ? dim1 : dim2;

	BigDecimal d1 = BigDecimal(f1);
	BigDecimal d2 =  BigDecimal(f2);
	BigDecimal d3 = d1.mul(d2);

	Chain f3 = d3.toChain();
	int pos3;
	if ( f3.posStr(Chain("."), pos3) == false )
	    throw Exception(EXLOC, "Invalid fixed value");
	int i = f3.length() - pos3 -1;

	while ( i < fixDim )
	{
	    f3 = f3 + Chain("0");
	    i++;
	}
	
	CegoFieldValue fv3(_type, f3);

	return fv3;

    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv2._pV, sizeof(float));	
	float* pF = new float;
	*pF = f1 * f2;
	CegoFieldValue fv3(_type, pF, sizeof(float), true);
	return fv3;
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv2._pV, sizeof(double));	
	double* pD = new double;
	*pD = d1 * d2;
	CegoFieldValue fv3(_type, pD, sizeof(double), true);
	return fv3;
    }
    case SMALLINT_TYPE:
    {
	short s1, s2;
	memcpy(&s1, _pV, sizeof(short));
	memcpy(&s2, fv2._pV, sizeof(short));
	short* pS = new short;
	*pS = s1 * s2;
	CegoFieldValue fv3(_type, pS, sizeof(short), true);
	return fv3;
    }	
    case TINYINT_TYPE:
    {	
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv2._pV);	
	char* pC = new char;
	*pC = c1 * c2;
	CegoFieldValue fv3(_type, pC, sizeof(char), true);
	return fv3;
    }
    default:	
	throw Exception(EXLOC, "Unknown Type");
    }
}

CegoFieldValue operator / ( const CegoFieldValue& fv1, const CegoFieldValue& fv2 )
{
    if ( fv1._type == NULL_TYPE || fv2._type == NULL_TYPE)
    {
	throw Exception(EXLOC , "Cannot operate on null value");
    }
    if (fv1._type != fv2._type)
    {
	CegoFieldValue fv3 = fv2;
	if ( fv3.castTo(fv1.getType()) == true )
	    return fv1.div(fv3);
	
	fv3 = fv1;
	if ( fv3.castTo(fv2.getType()) == true )
	    return fv3.div(fv2);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (fv1._pV == 0 || fv2._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return fv1.div(fv2);
}

CegoFieldValue CegoFieldValue::div( const CegoFieldValue& fv2 ) const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	if (_pV && fv2._pV)
	{
	    int i1, i2;
	    memcpy(&i1, _pV, sizeof(int));
	    memcpy(&i2, fv2._pV, sizeof(int));		   
	    int* pI = new int;
	    if ( i2 == 0 )
		throw Exception(EXLOC , "Division by zero");
	    *pI = i1 / i2;
	    CegoFieldValue fv3(_type, pI, sizeof(int), true);
	    return fv3;
	}
	else
	{
	    CegoFieldValue fv3(_type, 0, sizeof(int), true);
	    return fv3;	    
	}
    }
    case LONG_TYPE:
    {
	if (_pV && fv2._pV)
	{
	    long long l1, l2;
	    memcpy(&l1, _pV, sizeof(long long));
	    memcpy(&l2, fv2._pV, sizeof(long long));		   
	    long long* pL = new long long;

	    if ( l2 == 0 )
		throw Exception(EXLOC , "Division by zero");

	    *pL = l1 / l2;
	    CegoFieldValue fv3(_type, pL, sizeof( long long), true);
	    return fv3;
	}
	else
	{
	    CegoFieldValue fv3(_type, 0, sizeof(long), true);
	    return fv3;	    
	}
    }
    case VARCHAR_TYPE:
    case BOOL_TYPE:
    case DATETIME_TYPE:
    {
	throw Exception(EXLOC, "Operation not supported");	
    }
    case BIGINT_TYPE:
    {	
	BigInteger bi1 = BigInteger ( Chain((char*)(_pV)) );
	BigInteger bi2 = BigInteger ( Chain((char*)(fv2._pV)) );	
	BigInteger bi3 =  bi1.div(bi2);	
	CegoFieldValue fv3(_type, bi3.toChain());	
	return fv3;
    }
    case DECIMAL_TYPE:
    {
	BigDecimal d1 = BigDecimal( Chain((char*)(_pV)) );
	BigDecimal d2 =  BigDecimal( Chain((char*)(fv2._pV)) );	
	BigDecimal d3 = d1.div(d2);	
	CegoFieldValue fv3(_type, d3.toChain());	
	return fv3;	
    }
    case FIXED_TYPE:
    {
	Chain f1((char*)(_pV));
	Chain f2((char*)(fv2._pV));

	int pos1;
	if ( f1.posStr(Chain("."), pos1) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim1 =  _len - pos1 - 1;

	int pos2;
	if ( f2.posStr(Chain("."), pos2) == false )
	    throw Exception(EXLOC, "Invalid fixed value");

	int dim2 = fv2._len - pos2 - 1;

	int fixDim = dim1 > dim2 ? dim1 : dim2;

	BigDecimal d1 = BigDecimal(f1);
	BigDecimal d2 =  BigDecimal(f2);
	BigDecimal d3 = d1.div(d2);

	Chain f3 = d3.toChain();
	int pos3;
	if ( f3.posStr(Chain("."), pos3) == false )
	    throw Exception(EXLOC, "Invalid fixed value");
	int i = f3.length() - pos3 -1;

	while ( i < fixDim )
	{
	    f3 = f3 + Chain("0");
	    i++;
	}
	
	CegoFieldValue fv3(_type, f3);

	return fv3;


    }
    case FLOAT_TYPE:
    {
	float f1, f2;
	memcpy(&f1, _pV, sizeof(float));
	memcpy(&f2, fv2._pV, sizeof(float));	
	float* pF = new float;

	if ( f2 == 0 )
	    throw Exception(EXLOC , "Division by zero");

	*pF = f1 / f2;
	CegoFieldValue fv3(_type, pF, sizeof(float), true);
	return fv3;
    }
    case DOUBLE_TYPE:
    {
	double d1, d2;
	memcpy(&d1, _pV, sizeof(double));
	memcpy(&d2, fv2._pV, sizeof(double));	
	double* pD = new double;

	if ( d2 == 0 )
	    throw Exception(EXLOC , "Division by zero");

	*pD = d1 / d2;
	CegoFieldValue fv3(_type, pD, sizeof(double), true);
	return fv3;
    }
    case SMALLINT_TYPE:
    {
	short s1, s2;
	memcpy(&s1, _pV, sizeof(short));
	memcpy(&s2, fv2._pV, sizeof(short));	
	short* pS = new short;

	if ( s2 == 0 )
	    throw Exception(EXLOC , "Division by zero");

	*pS = s1 / s2;
	CegoFieldValue fv3(_type, pS, sizeof(short), true);
	return fv3;
    }	
    case TINYINT_TYPE:
    {
	char c1 = *(char*)(_pV);
	char c2 = *(char*)(fv2._pV);	
	char* pC = new char;

	if ( c2 == 0 )
	    throw Exception(EXLOC , "Division by zero");

	*pC = c1 / c2;
	CegoFieldValue fv3(_type, pC, sizeof(char), true);
	return fv3;
    }
    default:	
	throw Exception(EXLOC, "Unknown Type");
    }
}

CegoFieldValue operator | ( const CegoFieldValue& fv1, const CegoFieldValue& fv2 )
{
    if ( fv1._type == NULL_TYPE || fv2._type == NULL_TYPE)
    {
	throw Exception(EXLOC , "Cannot operate on null value");
    }
    if (fv1._type != fv2._type)
    {
	CegoFieldValue fv3 = fv2;
	if ( fv3.castTo(fv1.getType()) == true )
	    return fv1.concat(fv3);

	fv3 = fv1;
	if ( fv3.castTo(fv2.getType()) == true )
	    return fv3.concat(fv2);

	throw Exception(EXLOC , "Incompatible Datatypes");
    }
    if (fv1._pV == 0 || fv2._pV == 0)
    {
	throw Exception(EXLOC , "Undefined field value");
    }
    return fv1.concat(fv2);
}

CegoFieldValue CegoFieldValue::concat( const CegoFieldValue& fv2 ) const
{
    switch (_type ) 
    {
    case INT_TYPE:
    {
	int i1, i2;
	memcpy(&i1, _pV, sizeof(int));
	memcpy(&i2, fv2._pV, sizeof(int));
	CegoFieldValue fv3(VARCHAR_TYPE, Chain(i1) + Chain(i2) ) ;
	return fv3;
    }
    case LONG_TYPE:
    {
	long long l1, l2;
	memcpy(&l1, _pV, sizeof(long long));
	memcpy(&l2, fv2._pV, sizeof(long long));
	CegoFieldValue fv3(VARCHAR_TYPE, Chain(l1) + Chain(l2) ) ;
	return fv3;
    }
    case VARCHAR_TYPE:
    {
	Chain c1 = Chain((char*)(_pV), _len - 1);
	Chain c2 = Chain((char*)(fv2._pV), fv2._len - 1);
	CegoFieldValue fv3(VARCHAR_TYPE, c1 + c2 ) ;
	return fv3;
    }
    case BOOL_TYPE:
    {
	Chain c1 = Chain((char*)(_pV));
	Chain c2 = Chain((char*)(fv2._pV));
	CegoFieldValue fv3(VARCHAR_TYPE, c1 + c2 ) ;
	return fv3;	
    }
    default:
	throw Exception(EXLOC, "Unknown Type");
    }
}

CegoFieldValue CegoFieldValue::negate() const
{
    switch (_type)
    {
    case INT_TYPE:
    {
	int i;
	memcpy(&i, _pV, sizeof(int));
	int* pI = new int;
	*pI = (-1) * i;
	CegoFieldValue fv3(_type, pI, sizeof(int), true);
	return fv3;
    }
    case LONG_TYPE:
    {
	long long l;
	memcpy(&l, _pV, sizeof(long long));
	long long* pL = new long long;
	*pL = (-1) * l;
	CegoFieldValue fv3(_type, pL, sizeof(long long), true);
	return fv3;
    }
    case BIGINT_TYPE:
    {
	BigInteger bi = BigInteger( Chain((char*)(_pV)) );
	bi.negate();
	CegoFieldValue fv3(_type, bi.toChain());
	return fv3;
    }
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {
	BigDecimal bd = BigDecimal( Chain((char*)(_pV)) );
	bd.negate();
	CegoFieldValue fv3(_type, bd.toChain());
	return fv3;
    }
    case FLOAT_TYPE:
    {
	float f;
	memcpy(&f, _pV, sizeof(float));
	float* pF = new float;
	*pF = (-1.0) * f;
	CegoFieldValue fv3(_type, pF, sizeof(float), true);
	return fv3;
    }
    case DOUBLE_TYPE:
    {
	double d;
	memcpy(&d, _pV, sizeof(double));
	double* pD = new double;
	*pD = (-1.0) * d;
	CegoFieldValue fv3(_type, pD, sizeof(double), true);
	return fv3;
    }
    case SMALLINT_TYPE:
    {
	short s;
	memcpy(&s,_pV, sizeof(short));
	short* pS = new short;
	*pS = (-1) * s;
	CegoFieldValue fv3(_type, pS, sizeof(short), true);
	return fv3;
    }	
    case TINYINT_TYPE:
    {	
	char c = *(char*)(_pV);
	char* pC = new char;
	*pC = (-1) * c;
	CegoFieldValue fv3(_type, pC, sizeof(char), true);
	return fv3;
    }
    case VARCHAR_TYPE:
    case BOOL_TYPE:
    case DATETIME_TYPE:
    {
	throw Exception(EXLOC, "Cannot negate date type");
    }
    default:	
	throw Exception(EXLOC, "Unknown Type");
    }
}

void CegoFieldValue::normFloatValue(Chain& floatVal) const
{
    if ( __decimalPoint == ',' )
    {
	Chain normFloat;
	if ( floatVal.replace(".", ",", normFloat) )
	{
	    floatVal = normFloat;
	}
    }
}

void CegoFieldValue::denormFloatValue(Chain& floatVal) const
{
    if ( __decimalPoint == ',' )
    {
	Chain normFloat;
	if ( floatVal.replace(",", ".", normFloat) )
	{
	    floatVal = normFloat;
	}
    }
}


void CegoFieldValue::setLocalCopy(bool isLocal)
{
    _isLocalCopy = isLocal;
}

bool CegoFieldValue::isLocalCopy() const
{
    return _isLocalCopy;
}

CegoFieldValue CegoFieldValue::getLocalCopy() const
{
    CegoFieldValue lc;
    lc._type = _type;
    lc._len = _len;
    lc._pV = malloc(_len);
    memcpy(lc._pV, _pV, _len);
    lc._isLocalCopy = true;
    return lc;
}

bool CegoFieldValue::isNull() const
{
    if ( _type == NULL_TYPE || _pV == 0 )
	return true;
    return false;
}

int CegoFieldValue::size() const
{
    return sizeof(CegoDataType) + sizeof(void*) + _len + sizeof(bool) + STATICFIELDBUF;
}

Chain CegoFieldValue::toChain() const
{

    Chain s;

    if ( _type == NULL_TYPE || _type == BLOB_TYPE || _type == CLOB_TYPE )
    {
	s = Chain("null");
    }
    else
    {
	if ( _pV == 0 )
	{
	    s = Chain("null");
	    return s;
	}
	
	switch (_type)
	{
	case INT_TYPE:
	{
	    int i;
	    memcpy(&i, _pV, sizeof(int));
	    s = Chain( i );
	    break;
	}   
	case LONG_TYPE:
	{
	    long long l;
	    memcpy(&l, _pV, sizeof(long long));
	    s = Chain("(long)") + Chain( l );
	    break;
	}   
	case VARCHAR_TYPE:	    
	{   
	    Chain val((char*)_pV, _len - 1);

	    // escape backslashes
	    Chain bval;
	    val.replaceAll(Chain("\\"), Chain("\\\\"), bval);

	    // escape newlines
	    Chain nval;
	    bval.replaceAll(Chain("\n"), Chain("\\n"), nval);

	    // escape quotes
	    Chain qval;
	    if ( __quoteEscapeFlag )
	    {
		nval.replaceAll(Chain("'"), Chain("''"), qval);
	    }
	    else
	    {
		nval.replaceAll(Chain("'"), Chain("\\'"), qval);
	    }

	    s = Chain("'") + qval + Chain("'");
	    break;
	}   
	case BOOL_TYPE:
	{   
	    char b = *(char*)(_pV);
	    if ( b > 0 )
		s = Chain("true");
	    else
		s = Chain("false");
	    break;
	}    
	case DATETIME_TYPE:
	{
	    int i;
	    memcpy(&i, _pV, sizeof(int));
	    if ( i == 0 )
	    {
		s = Chain("sysdate");
	    }
	    else
	    {
		Datetime dt ( i );
		s = Chain("date('") + __dateTimeFormat + Chain("','") + dt.asChain(__dateTimeFormat) + Chain("')");
	    }
	    break;
	}
	case BIGINT_TYPE:
	{
	    BigInteger bi( Chain((char*)_pV) );
	    s =  Chain("(bigint)") +  bi.toChain();
	    break;
	}
	case DECIMAL_TYPE:
	{
	    BigDecimal d( Chain((char*)_pV) );
	    s = Chain("(decimal)") + d.toChain();
	    break;	    
	}
	case FIXED_TYPE:
	{	    
	    BigDecimal d( Chain((char*)_pV) );
	    s = Chain("(fixed)") + d.toChain();
	    break;
	}
	case FLOAT_TYPE:
	{
	    float f;
	    memcpy(&f, _pV, sizeof(float));
	    
	    Chain normFloat(f);
	    denormFloatValue(normFloat);
	    s=normFloat;
	    
	    break;
	}
	case DOUBLE_TYPE:
	{
	    double d;
	    memcpy(&d, _pV, sizeof(double));

	    Chain normDouble(d);
	    denormFloatValue(normDouble);	   
	    s = Chain("(double)") + normDouble;
	    break;
	}
	case SMALLINT_TYPE:
	{
	    short st;
	    memcpy(&st, _pV, sizeof(short));
	    s = Chain("(smallint)") + Chain( st );
	    break;
	}
	case TINYINT_TYPE:
	{
	    s = Chain("(tinyint)") + Chain( *(char*)(_pV) );
	    break;
	}
	case BLOB_TYPE:
	case CLOB_TYPE:
	case NULL_TYPE:
	    // already handled
	    break;	    
	}
    }
    return s;
}

Chain CegoFieldValue::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatFieldValue(_type, _pV, _len);
}

Chain CegoFieldValue::typeToChain() const
{
    Chain s;

    switch ( _type )
    {
    case NULL_TYPE:
    {
	s = Chain("null");
	break;
    }
    case VARCHAR_TYPE:
    case FIXED_TYPE:
    case BIGINT_TYPE:
    case DECIMAL_TYPE:
    {
	s = CEGO_TYPE_MAP[_type] + Chain("(") + Chain(_len) + Chain(")");
	break;
    }
    default:
    {
	s = CEGO_TYPE_MAP[_type];
	break;
    }
    }
    return s;
}

Chain CegoFieldValue::valAsChain(bool doEval) const
{
    Chain s;

    if ( _type == NULL_TYPE )
    {
	s = Chain("null");
    }
    else
    {
	if ( _pV == 0 )
	{
	    s = Chain("null");
	    return s;
	}
	switch (_type)
	{
	case INT_TYPE:
	{
	    int i;
	    memcpy(&i, _pV, sizeof(int));
	    s = Chain( i );
	    break;
	}   
	case LONG_TYPE:
	{
	    long long l;
	    memcpy(&l, _pV, sizeof(long long));
	    s = Chain( l );
	    break;
	}   
	case VARCHAR_TYPE:
	{	    
	    s = Chain((char*)_pV, _len - 1);
	    break;
	}
	case BIGINT_TYPE:
	case DECIMAL_TYPE:
	case FIXED_TYPE:
	{   
	    s = Chain((char*)_pV);
	    break;
	}   
	case BOOL_TYPE:
	{   
	    char b = *(char*)(_pV);
	    if ( b > 0 )
		s = Chain("true");
	    else
		s = Chain("false");
	    break;
	}    
	case DATETIME_TYPE:
	{
	    int i;
	    memcpy(&i, _pV, sizeof(int));

	    if ( i == 0 )
	    {
		if ( doEval )
		{
		    Datetime dt;
		    s = dt.asChain(__dateTimeFormat);
		}
		else
		{
		    s = Chain("sysdate");
		}
	    }
	    else
	    {
		Datetime dt(i);
		s = dt.asChain(__dateTimeFormat);
	    }
	    break;
	}
	case FLOAT_TYPE:
	{
	    float f;
	    memcpy(&f, _pV, sizeof(float));

	    Chain normFloat(f);
	    denormFloatValue(normFloat);
	    s=normFloat;
	    break;
	}
	case DOUBLE_TYPE:
	{
	    double doubleVal;
	    memcpy(&doubleVal, _pV, sizeof(double));

	    Chain normDouble(doubleVal);
	    denormFloatValue(normDouble);
	    s=normDouble;
	    break;
	}
	case SMALLINT_TYPE:
	{
	    short st;
	    memcpy(&st, _pV, sizeof(short));
	    s = Chain( st );
	    break;
	}
	case TINYINT_TYPE:
	{
	    char c;
	    memcpy(&c, _pV, sizeof(char));
	    s = Chain( (int)c );
	    break;
	}
	case BLOB_TYPE:
	case CLOB_TYPE:
	{
	    int f, p;
	    memcpy(&f, _pV, sizeof(int));
	    memcpy(&p, (void*)((long long)_pV + sizeof(int)), sizeof(int));

	    s = Chain("[") + Chain( f ) + Chain(",") + Chain(p) + Chain("]");
	    break;

	}
	case NULL_TYPE:
	    // already handled
	    break;
	}
    }
    return s;
}

int CegoFieldValue::asInteger() const
{
    CegoFieldValue fv = *this;

    if ( fv.castTo(INT_TYPE) == false )
	throw Exception(EXLOC, Chain("Cannot get integer value"));

    // in case of null value, we map value to zero
    if ( _pV == 0 )
	return 0;
    
    long i;
    memcpy(&i, fv.getValue(), sizeof(int));
    return i;
}

long long CegoFieldValue::asLong() const
{
    CegoFieldValue fv = *this;

    if ( fv.castTo(LONG_TYPE) == false )
	throw Exception(EXLOC, Chain("Cannot get long value"));
    
    long l;
    memcpy(&l, fv.getValue(), sizeof(long));
    return l;
}

void CegoFieldValue::encode(char *buf) const
{    
    char* pE = (char*)buf;
    
    memcpy( pE, &_type, sizeof(CegoDataType));
    pE = pE + sizeof(CegoDataType);

    switch (_type )
    {
    case INT_TYPE:
    {
	memcpy(pE, _pV, sizeof(int));
	pE = pE + sizeof(int);
	break;
    }   
    case LONG_TYPE:
    {
	memcpy(pE, _pV, sizeof(long long));
	pE = pE + sizeof(long long);
	break;
    }
    case VARCHAR_TYPE:	    
    case BIGINT_TYPE:
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {   
	memcpy(pE, &_len, sizeof(int));
	pE = pE + sizeof(int);
	memcpy(pE, _pV, _len);
	pE = pE + _len;
	break;
    }   
    case BOOL_TYPE:
    {   
	memcpy(pE, _pV, 1);
	pE = pE + 1;
	break;
    }    
    case DATETIME_TYPE:
    {
	memcpy(pE, _pV, sizeof(int));
	pE = pE + sizeof(int);
	break;
    }
    case FLOAT_TYPE:
    {
	memcpy(pE, _pV, sizeof(float));
	pE = pE + sizeof(float);
	break;
    }
    case DOUBLE_TYPE:
    {
	memcpy(pE, _pV, sizeof(double));
	pE = pE + sizeof(double);
	break;
    }
    case SMALLINT_TYPE:
    {
	memcpy(pE, _pV, sizeof(short));
	pE = pE + sizeof(short);
	break;
    }
    case TINYINT_TYPE:
    {
	memcpy(pE, _pV, 1);
	pE = pE + 1;
	break;
    }
    case BLOB_TYPE:
    case CLOB_TYPE:
    {   
	memcpy(pE, _pV, 2 * sizeof(int));
	pE = pE + 2 * sizeof(int);
	break;
    }   
    case NULL_TYPE:
	break;
    }
}

void CegoFieldValue::decode(char *buf)
{

    char* pE = (char*)buf;
    
    memcpy( &_type, pE, sizeof(CegoDataType));
    pE = pE + sizeof(CegoDataType);

    switch (_type )
    {
    case INT_TYPE:
    {
	_len = sizeof(int);
	_pV = malloc(_len);
	memcpy(_pV, pE, sizeof(int));
	pE = pE + sizeof(int);
	break;
    }   
    case LONG_TYPE:
    {
	_len = sizeof(long long);
	_pV = malloc(_len);
	memcpy(_pV, pE, _len);
	pE = pE + _len;
	break;
    }
    case VARCHAR_TYPE:	    
    case BIGINT_TYPE:
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {   
	memcpy(&_len, pE, sizeof(int));
	pE = pE + sizeof(int);
	_pV = malloc(_len);
	memcpy(_pV, pE, _len);
	pE = pE + _len;
	break;
    }   
    case BOOL_TYPE:
    {   
	_len = sizeof(char);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }    
    case DATETIME_TYPE:
    {
	_len = sizeof(int);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }
    case FLOAT_TYPE:
    {
	_len = sizeof(float);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }
    case DOUBLE_TYPE:
    {
	_len = sizeof(double);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }
    case SMALLINT_TYPE:
    {
	_len = sizeof(short);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }
    case TINYINT_TYPE:
    {
	_len = sizeof(char);
	_pV = malloc(_len);
	memcpy(_pV, pE,  _len);
	pE = pE + _len;
	break;
    }
    case BLOB_TYPE:
    case CLOB_TYPE:
    {
	_len = 2 * sizeof(int);
	_pV = malloc(_len);
	memcpy(_pV, pE, 2 * sizeof(int));
	pE = pE + 2 * sizeof(int);
	break;
    }   
    case NULL_TYPE:
	break;
    }
}

int CegoFieldValue::getEncodingLength() const
{
    int len = 0;
    
    len += sizeof(CegoDataType);

    switch (_type )
    {
    case INT_TYPE:
    {
	len += sizeof(int);
	break;
    }   
    case LONG_TYPE:
    {
	len += sizeof(long long);
	break;
    }
    case VARCHAR_TYPE:	    
    case BIGINT_TYPE:
    case DECIMAL_TYPE:
    case FIXED_TYPE:
    {   	
	len += _len + sizeof(int);
	break;
    }   
    case BOOL_TYPE:
    {   
	len += 1;
	break;
    }    
    case DATETIME_TYPE:
    {
	len += sizeof(int);
	break;
    }
    case FLOAT_TYPE:
    {
	len += sizeof(float);
	break;
    }
    case DOUBLE_TYPE:
    {
	len += sizeof(double);
	break;
    }
    case SMALLINT_TYPE:
    {
	len += sizeof(short);
	break;
    }
    case TINYINT_TYPE:
    {
	len += 1;
	break;
    }
    case BLOB_TYPE:
    case CLOB_TYPE:
    {
	len += 2 * sizeof(int);
	break;
    }   
    case NULL_TYPE:
	break;
    }    
    return len;
}

ostream& operator << (ostream& s, const CegoFieldValue& fv)
{
    if ( fv._type == NULL_TYPE )
    {
	s << "null";
	return s;
    }
    if ( fv._pV == 0 )
    {
	s << "null";
	return s;
    }
    switch (fv._type)
    {
    case INT_TYPE:
    {
	int i;
	memcpy(&i, fv._pV, sizeof(int));
	s << i;
	break;
    }
    case LONG_TYPE:
    {
	long long l;
	memcpy(&l, fv._pV, sizeof(long long));
	s << l;
	break;
    }
    case VARCHAR_TYPE:
    {
	s <<  Chain((char*)fv._pV, fv._len - 1);
	break;
    }
    case BOOL_TYPE:
    {
	char b = *(char*)(fv._pV);
	if ( b > 0 )
	    s << "true";
	else
	    s << "false";
	break;
    }
    case DATETIME_TYPE:
    {
	int i;
	memcpy(&i, fv._pV, sizeof(int));
	Datetime dt ( i );
	s << dt.asChain();
	break;
    }
    case BIGINT_TYPE:
    {
	BigInteger bi((char*)(fv._pV));
	s << bi.toChain();
	break;	
    }
    case DECIMAL_TYPE:
    {
	BigDecimal d((char*)(fv._pV));
	s << Chain("(decimal)") << d.toChain();
	break;
    }
    case FIXED_TYPE:
    {
	BigDecimal d((char*)(fv._pV));
	s <<  Chain("(fixed)") << d.toChain();
	break;	
    }
    case DOUBLE_TYPE:
    {
	double d;
	memcpy(&d, fv._pV, sizeof(double));
	s << d;
	break;
    }
    case FLOAT_TYPE:
    {
	float f;
	memcpy(&f, fv._pV, sizeof(float));
	s << f;
	break;
    }
    case SMALLINT_TYPE:
    {
	short sv;
	memcpy(&sv, fv._pV, sizeof(short));
	s << sv;
	break;
    }
    case TINYINT_TYPE:
    {
	double t;
	memcpy(&t, fv._pV, sizeof(char));
	s << t;
	break;
    }
    case BLOB_TYPE:
    case CLOB_TYPE:
    {
	s << fv.valAsChain();
	break;
    }
    default:
	s << "Datatype not supported yet";
    }
    return s;
}


