///////////////////////////////////////////////////////////////////////////////
//                                   
// CegoOutput.cc
// -------------
// Cego formatting data for output
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoOutput
// 
// Description: The CegoOutput class converts query output to an appropriate format.
//              This may either be XML-based style for serving a client request or 
//              ascii-based tableformat for batchmode or client-side output
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#ifndef CGNOCURSES

#ifdef HAVE_MINGW
#include <ncurses/ncurses.h>
#else
#include <ncurses.h>
#endif

#include <sstream>
#include <iostream>
#include <string>
#include <vector>
#endif

// base includes
#include <lfcbase/Datetime.h>
#include <lfcbase/Tokenizer.h>
#include <lfcbase/ListT.h>

// cego includes
#include "CegoOutput.h"
#include "CegoDefs.h"

#define DEFRAWSEP ","

CegoOutput::CegoOutput()
{
    _pDbHandle = 0;
    _rawMode = false;
    _rawSep = Chain(DEFRAWSEP);
}

CegoOutput::CegoOutput(const ListT<CegoField>& schema, const Chain& format)
{
    _schema = schema;
    _pDbHandle = 0;
    _rowsPerMsg = 0;
    _rawMode = false;
    _rawSep = Chain(DEFRAWSEP);
    _format = format;
}

CegoOutput::~CegoOutput()
{
}

void CegoOutput::setDbHandle(CegoDbHandler* pDbHandle, int rowsPerMsg)
{
    _pDbHandle = pDbHandle;
    _rowsPerMsg = rowsPerMsg;
}

void CegoOutput::setRawMode(bool isOn)
{
    _rawMode = isOn;
}

void CegoOutput::setRawSeparator(const Chain& rawSep)
{
    _rawSep = rawSep;
}

void CegoOutput::chainOut(const Chain& msg, long affCount)
{

    if ( _pDbHandle ) 
    {
	_pDbHandle->sendResponse(msg, affCount);
    }
    else
    {
	cout << msg << endl;
    }    
}

void CegoOutput::procResultOut(const Chain& msg, const ListT<CegoProcVar>& outParamList, CegoFieldValue *pRetVal)
{
    if ( _pDbHandle ) 
    {	
	_pDbHandle->sendProcResult(msg, outParamList, pRetVal);
    }
    else
    {
	cout << msg << endl;
    }
}

void CegoOutput::tabOut(const ListT< ListT<CegoFieldValue> >& fa)
{
    headOut();
    ListT<CegoFieldValue>* pFVL = fa.First();
    while ( pFVL )
    {
	rowOut(*pFVL);
	pFVL = fa.Next();
    }
    tailOut();
} 

void CegoOutput::headOut()
{
    
    if ( _rawMode == true ) 
	return;

    int maxLen = 0;
    CegoField* pF;

    if ( _pDbHandle ) 
    {
	_pDbHandle->collectSchema(_schema, _format);	
	_rowCount = 0;
    }
    else if ( _rawMode == false )
    {

	pF = _schema.First();
	while (pF)
	{
	    maxLen = maxFieldSize(pF);
	    cout << "+-" << fill("-", maxLen );	    
	    pF = _schema.Next();
	}
	cout << "+" << endl;
	
	
	pF = _schema.First();
	int i=0;
	while (pF)
	{
	    maxLen = maxFieldSize(pF);
	    
	    Chain tname;
	    if (pF->getTableAlias().length() > 0)
	    {
		tname = pF->getTableAlias();
	    }
	    else
	    {
		tname = pF->getTableName();
	    }

	    cout << formatCell(i, tname, maxLen);

	    i++;
	    pF = _schema.Next();
	}
		
	cout << "|" << endl;
    	
	pF = _schema.First();
	i=0;
	while (pF)
	{
	    maxLen = maxFieldSize(pF);	    

	    cout << formatCell(i, pF->getAttrName(), maxLen);

	    i++;
	    pF = _schema.Next();
	}

	cout << "|" << endl;
		
	pF = _schema.First();
	while (pF)
	{	   
	    maxLen = maxFieldSize(pF);	    
	    cout << "+-" << fill("-", maxLen );	    
	    pF = _schema.Next();
	}
	cout << "+" << endl;
    }    
}

void CegoOutput::rowOut(const ListT<CegoField>& fl)
{
    
    if ( _pDbHandle )
    {

	_pDbHandle->collectData(fl);
	_rowCount++;

	if ( _rowCount == _rowsPerMsg )	
	{
	    _pDbHandle->sendCollectedData();
	    _rowCount=0;
	}
    }
    else if ( _rawMode == false )
    {
	int maxLen = 0;
	CegoField *pF1;
	CegoField *pF2;
	
	pF1 = fl.First();
	pF2 = _schema.First();
	
	int i=0;
	_preFill=0;
	while ( pF1 && pF2 )
	{
	    int maxLen = maxFieldSize(pF2);

	    Chain s = pF1->getValue().valAsChain();
	    
	    cout << formatCell(i, s, maxLen);
	    _preFill+=maxLen + 1;
	    
	    i++;	    
	    pF1 = fl.Next();
	    pF2 = _schema.Next();
	}
	
	cout << "|" << endl;

    }
    else
    {
	
	CegoField *pF;
	
	pF = fl.First();
	
	while ( pF )
	{
	    Chain s = pF->getValue().valAsChain();
	    cout << s;
	    pF = fl.Next();
	    if ( pF )
		cout << " ";
	    else 
		cout << endl;

	}	    
    }
}

void CegoOutput::rowOut(const ListT<CegoFieldValue>& fvl)
{

    if ( _pDbHandle )
    {

	_pDbHandle->collectData(_schema, fvl);
	_rowCount++;

	if ( _rowCount == _rowsPerMsg )	
	{
	    _pDbHandle->sendCollectedData();
	    _rowCount=0;
	}
    }
    else if ( _rawMode == false )
    {
	int maxLen = 0;
	CegoFieldValue *pFV;
	CegoField *pF2;
	
	pFV = fvl.First();
	pF2 = _schema.First();

	int i=0;
	_preFill=0;
	while ( pFV && pF2 )
	{
	    int maxLen = maxFieldSize(pF2);

	    Chain s = pFV->valAsChain();
	    cout << formatCell(i, s, maxLen);
	    _preFill+=maxLen + 1;

	    i++;
	    pFV = fvl.Next();
	    pF2 = _schema.Next();
	}	    

	cout << "|" << endl;

    }
    else
    {
	
	CegoFieldValue *pFV;
	
	pFV = fvl.First();
	
	while ( pFV )
	{
	    Chain s = pFV->valAsChain();
	    cout << s;
	    
	    pFV = fvl.Next();
	    if ( pFV )
		cout << _rawSep;
	    else 
		cout << endl;
	}	    
    }
}

void CegoOutput::tailOut()
{

    if ( _pDbHandle )
    {
	if ( _rowCount > 0 )
	{
	    _pDbHandle->sendCollectedData();
	    _rowCount = 0;
	}
	_pDbHandle->sendFinishData();
    }
    else if ( _rawMode == false )
    {
	int maxLen = 0;
	CegoField *pF;
	
	pF = _schema.First();
	while (pF)
	{
	    maxLen = maxFieldSize(pF);
	    
	    cout << "+-" << fill("-", maxLen );
	    
	    pF = _schema.Next();
	}
	cout << "+" << endl;
    }
}


void CegoOutput::abort(const Chain& msg)
{

    if ( _pDbHandle )
    {
	_pDbHandle->sendErrorData(msg);
    }   
    else
    {
	cout << "Aborting Select : " << msg << endl;
    }
}

int CegoOutput::maxFieldSize(CegoField *pF)
{

    int maxLen = 0;

    switch ( pF->getType() )
    {
    case INT_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_INT_LEN);
	break;

    case LONG_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_LONG_LEN);
	break;
	
    case VARCHAR_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     pF->getLength());
	break;
	
    case BOOL_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_BOOL_LEN);
	break;
	
    case DATETIME_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_DATETIME_LEN);
	break;
    case FLOAT_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_FLOAT_LEN);
	break;

    case DOUBLE_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_DOUBLE_LEN);
	break;

    case BIGINT_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     pF->getLength());
	break;

    case DECIMAL_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_DECIMAL_LEN);
	break;

    case SMALLINT_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_SMALLINT_LEN);
	break;

    case TINYINT_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_TINYINT_LEN);
	break;

    case FIXED_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_FIXED_LEN);
	break;
    case BLOB_TYPE:
    case CLOB_TYPE:
	maxLen = max(pF->getAttrName().length(), 
		     pF->getTableAlias().length(), 
		     pF->getTableName().length(),
		     MAX_BLOB_LEN);
	break;

	
    default:
	maxLen = max(pF->getAttrName().length(), 0,0,0);
	break;
		
    }
    
    return maxLen;    
}

int CegoOutput::max(int i1, int i2, int i3, int i4)
{
    int c1 =  i1 < i2 ? i2 : i1 ;
    int c2 =  i3 < i4 ? i4 : i3 ;
    return c1 < c2 ? c2 : c1 ;
}

Chain CegoOutput::fill(const Chain& s, int num)
{
    Chain fs = Chain("");
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }

    return fs;
}

Chain CegoOutput::formatCell(int i, const Chain& s, int maxLen, bool printSep)
{
    Chain cell;
    
    if ( _format.length() < 2 ) // _format == Chain("") )
    {

	Chain normString;
	if ( s.length() - 1  > maxLen )
	    normString = s.subChain(1, maxLen - 3 ) + Chain("...");
	else
	    normString = s;

	if ( printSep )	    
	    cell = Chain("|") + fill(" ", maxLen - normString.visibleLength() + 1) +  normString + Chain(" ");
	else
	    cell = fill(" ", maxLen - normString.visibleLength() + 1) +  normString + Chain(" ");
    }
    else
    {
	if ( _format[i] == 'l' )
	{
	    
	    Chain normString;
	    if ( s.length() - 1  > maxLen )
		normString = s.subChain(1, maxLen - 3 ) + Chain("...");
	    else
		normString = s;

	    if ( printSep )
		cell = Chain("| ") + normString + fill(" ", maxLen - normString.visibleLength() + 1);
	    else
		cell = normString + fill(" ", maxLen - normString.visibleLength() + 1);
	}
	else if ( _format[i] == 'r' )
	{
	    
	    Chain normString;
	    if ( s.length() - 1  > maxLen )
		normString = s.subChain(1, maxLen - 3 ) + Chain("...");
	    else
		normString = s;

	    if ( printSep )
		cell = "|" + fill(" ", maxLen - normString.visibleLength() + 1) + normString + Chain(" ");
	    else
		cell = fill(" ", maxLen - normString.visibleLength() + 1) + normString + Chain(" ");
	}
	else if ( _format[i] == 'm' )
	{
	    Tokenizer lineTok(s, Chain("\n"));
	    Chain line;
	    bool isFirst = true;
	    while ( lineTok.nextToken(line) )
	    {

		Chain normString;
		if ( line.length() - 1  > maxLen )
		    normString = line.subChain(1, maxLen - 3 ) + Chain("...");
		else
		    normString = line;

		if ( isFirst == false ) 
		{
		    if ( printSep )
			cell += Chain("|\n");
		    
		    if ( _preFill )
		    {
			if ( printSep )
			    cell += Chain("| ") + fill(" ", _preFill);
			else
			    cell += fill(" ", _preFill);
		    }
		}
		
		if ( printSep )
		    cell += Chain("| ") + normString + fill(" ", maxLen - normString.visibleLength() + 1);		
		else
		    cell += normString + fill(" ", maxLen - normString.visibleLength() + 1);
		
		isFirst = false;		  
	    }
	}		
    }

    return cell;
}

void CegoOutput::manageOutput(const Chain& cmd, ListT< ListT<CegoFieldValue> >& outData)
{
#ifndef CGNOCURSES

    int rows, cols;       
    initscr();
    
    getmaxyx(stdscr, rows, cols);    
    
    raw();    
    keypad(stdscr, TRUE);    
    noecho();
    
    // add 4 lines for header
    // add 1 lines for footer
    // add 2 lines for query result message
    // => at all 7 lines
    if ( outData.Size() + 7 < rows)
    {
	// clear();
	refresh();
	endwin();
	
	headOut();
	for(int i = 0; i < outData.Size(); i++)
	{
	    rowOut(outData[i]);
	}
	tailOut();
    }
    else
    {

	int i, ch = 0;
	unsigned int start = 0;
	unsigned int headerSize = 2;
	unsigned int footerSize = 1;
	       
	do
	{
	    switch (ch)
	    {
	    case KEY_UP :			
		if ( start > 0 )
		    start--;
		break;
	    case KEY_DOWN :
		if ( outData.Size() >= (rows - headerSize - footerSize) && start < (outData.Size() - (rows - headerSize - footerSize)))
		    start++;
		break;
	    case 'n' :
	    case ' ' :
		if(start < (outData.Size() - (rows - headerSize - footerSize))) start+=(rows-headerSize-footerSize);
		break;
	    case 'b' :
		if ( start > rows )
		    start-=rows;
		else
		    start=0;		    
		break;	    	    
	    }
	    
	    clear();
	    
	    writeHeader(cols);
	    
	    
	    for(i = 0; (i< rows - headerSize-footerSize) && (i + start < outData.Size()); i++)
	    {
		int col=0;
		for ( int j=0; j<outData[i].Size(); j++ )
		{
		    
		    int maxLen = maxFieldSize(&_schema[j]);
		    
		    mvprintw(i+headerSize, col, "%s", (char*)(outData[i+start][j].valAsChain()));
		    col +=maxLen;
		}
	    }
	    
	    writeFooter(rows, cols, (char*)cmd, start + 1, start + i, outData.Size());		
	    refresh();
	    
	    ch = getch();
	}
	while( ch != 'q' );		// continue until user hits q
		
	clear();
	refresh();	
	endwin();
    }
    
#else
    cout << "Curses not enabled" << endl;
#endif
    return;
}


void CegoOutput::writeHeader(int cols)
{
#ifndef CGNOCURSES
    
    attron(A_REVERSE);
    move(0,0);
    
    int col=0;
    for ( int i=0; i < _schema.Size(); i++ )
    {	
	int maxLen = maxFieldSize(&_schema[i]);
	mvprintw(0, col, "%s", (char*)(_schema[i].getTableAlias()));	
	for(int j = _schema[i].getTableAlias().length(); j < maxLen; j++)
	{
	    addch(' ');
	}			// write a black bar across the top	
	col +=maxLen;
    }
    
    col=0;
    for ( int i=0; i < _schema.Size(); i++ )
    {	
	int maxLen = maxFieldSize(&_schema[i]);
	mvprintw(1, col, "%s", (char*)(_schema[i].getAttrName()));	
	for(int j = _schema[i].getAttrName().length(); j < maxLen; j++)
	{
	    addch(' ');
	}
	col +=maxLen;
    }
    attroff(A_REVERSE);
#endif
}

void CegoOutput::writeFooter(int rows, int cols, char* query, int start, int end, int totalRow)
{
#ifndef CGNOCURSES
    
    attron(A_REVERSE);
    move(rows-1,0);
    for(int i = 0; i < cols; i++)
    {
	addch(' ');
    }
    mvprintw(rows-1,0, "up,down,n,b,q ?");
    mvprintw(rows-1,20, "%s", query);
    mvprintw(rows-1,cols-27,"%04d - %04d ( %04d total)", start, end, totalRow);
    attroff(A_REVERSE);
#endif
}
