///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoSelect.cc
// -------------
// Cego query implementation         
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoSelect
// 
// Description: This class implements the SQL select query algorithm for the Cego database system 
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Datetime.h>
#include <lfcbase/SetT.h>
#include <lfcbase/StackT.h>
// #include <lfcbase/Sleeper.h>

// CEGO INCLUDES
#include "CegoSelect.h"
#include "CegoQueryHelper.h"
#include "CegoCaseCond.h"
#include "CegoJoinObject.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseFormater.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoSelect::CegoSelect(char* buf, CegoDistManager* pGTM, int tabSetId)
{
    _pPred = 0;
    _pHaving = 0;
    _pNativeOrderList = 0;
    _pOrderList = 0;
    _pGroupList = 0;
    _pNativeOrderOptList = 0;
    _pOrderOptList = 0;
    _pOrderSpace = 0;
    _pGroupSpace = 0;
    _pGTM = pGTM;
    if ( _pGTM ) 
	_pDBMng = _pGTM->getDBMng();

    _modId = 0;

    int i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	_joinBuf[i] = 0;
	_pTC[i]=0;
	_attrCondFlag[i] = false;
	_firstTuple[i]=true;
	_attrPred[i]=0;
	i++;
    }

    _cacheEnabled = false;
    _pCache = 0;
    _pCacheArray = 0;
    _pCacheList = 0;

    _pBlock = 0;
    _pUnionSelect = 0;

    _checkUnion = false;
    _selectMode = PLAIN;
    _nextAid=0;
    _aggregationCount=0;

    _pOrderCursor=0;
    _pGroupCursor=0;
	    
    _isDistinct = false;
    _pParentJoinBuf = 0;
    _isPrepared = false;
    _rowLimit = 0;
    _isCached = false;
    _extRefCount = 0;

    _tabSetId = tabSetId;
    
    decode(buf, pGTM, tabSetId);

}

CegoSelect::CegoSelect(Element* pSelectElement, CegoDistManager* pGTM)
{
    _pPred = 0;
    _pHaving = 0;
    _pNativeOrderList = 0;
    _pOrderList = 0;
    _pGroupList = 0;
    _pNativeOrderOptList = 0;
    _pOrderOptList = 0;
    _pOrderSpace = 0;
    _pGroupSpace = 0;

    _pGTM = pGTM;
    _pDBMng = 0;
    _modId = 0;

    int i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	_joinBuf[i] = 0;
	_pTC[i]=0;
	_attrCondFlag[i] = false;
	_firstTuple[i]=true;
	_attrPred[i]=0;
	i++;
    }

    _cacheEnabled = false;
    _pCache = 0;
    _pCacheArray = 0;
    _pCacheList = 0;
    
    _pBlock = 0;
    _pUnionSelect = 0;

    _checkUnion = false;
    _selectMode = PLAIN;
    _nextAid=0;
    _aggregationCount=0;

    _pOrderCursor=0;
    _pGroupCursor=0;

    _isDistinct = false;
    _pParentJoinBuf = 0;
    _tabSetId = 0;
    _isPrepared = false;
    _rowLimit = 0;
    _isCached = false;
    _extRefCount = 0;
    
    fromElement(pSelectElement, pGTM);
}

CegoSelect::CegoSelect(ListT<CegoContentObject*>& coList,
		       ListT<CegoExpr*>& exprList, 
		       CegoPredDesc* pPred,
		       ListT<CegoAttrDesc*> *pGroupList,
		       CegoHavingDesc* pHaving,
		       ListT<CegoExpr*> *pOrderList,
		       ListT<CegoOrderNode::Ordering>* pOrderOptList,
		       bool isDistinct,
		       int rowLimit,
		       CegoDistManager* pGTM)
{
    _coList = coList;
    _exprList = exprList;
    _pPred = pPred;
    _pGroupList = pGroupList;
    _pHaving = pHaving;
    _pNativeOrderList = pOrderList;
    _pOrderList = 0;
    _pNativeOrderOptList = pOrderOptList;
    _pOrderOptList = 0;
    _pOrderSpace = 0;
    _pGroupSpace = 0;
    _pGTM = pGTM;
    _cacheEnabled = false;

    _pCache = 0;
    _pCacheArray = 0;
    _pCacheList = 0;

    if ( _pGTM )
    {
	_pDBMng = _pGTM->getDBMng();
	_modId = _pGTM->getDBMng()->getModId("CegoSelect");
    }
    
    int i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	_joinBuf[i] = 0;
	_pTC[i]=0;
	_attrCondFlag[i] = false;
	_firstTuple[i]=true;
	_attrPred[i]=0;
	i++;
    }
    
    _pBlock = 0;
    _pUnionSelect = 0;

    _checkUnion = false;
    _selectMode = PLAIN;
    _nextAid=0;
    _aggregationCount=0;

    _pOrderCursor=0;
    _pGroupCursor=0;

    _isDistinct = isDistinct;
    _pParentJoinBuf = 0;
    _tabSetId = 0;
    _isPrepared = false;
    _rowLimit = rowLimit;
    _isCached = false;

    _orderingDone=false;
    _groupingDone=false;

    _extRefCount = 0;
}

CegoSelect::~CegoSelect()
{
    if ( _pPred )
	delete _pPred;

    CegoPredDesc **pP = _viewConjunctionList.First();
    while ( pP )
    {      
	delete (*pP);
	pP = _viewConjunctionList.Next();
    }
    
    int i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	if ( _pTC[i] )
	    delete _pTC[i];
	if ( _joinBuf[i] )
	    delete _joinBuf[i];
	i++;
    }    

    CegoContentObject **pCO = _coList.First();
    while ( pCO )
    {
	delete (*pCO);
	pCO = _coList.Next();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr )
    {
	delete (*pExpr);
	pExpr = _exprList.Next();
    }

    if ( _pOrderList )
    {
	if ( _pOrderCursor )
	    delete _pOrderCursor;
	if ( _pOrderSpace )
	{
	    delete _pOrderSpace;	
	    _pOrderSpace = 0;
	}

	CegoExpr **pExpr = _pOrderList->First();
	while ( pExpr )
	{
	    delete (*pExpr);
	    pExpr = _pOrderList->Next();
	}

	delete _pOrderList;
	_pOrderList = 0;	
    }

    if ( _pNativeOrderList )
    {
	CegoExpr **pExpr = _pNativeOrderList->First();
	while ( pExpr )
	{
	    delete (*pExpr);
	    pExpr = _pNativeOrderList->Next();
	}

	delete _pNativeOrderList;
	_pNativeOrderList = 0;	
    }

    if ( _pNativeOrderOptList )
    {
	delete _pNativeOrderOptList;
	_pNativeOrderOptList = 0;	
    }
    
    if ( _pGroupList )
    {
	if ( _pGroupCursor )
	    delete _pGroupCursor;
	if ( _pGroupSpace )
	{
	    delete _pGroupSpace;
	    _pGroupSpace=0;
	}

	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    delete (*pAttrDesc);
	    pAttrDesc = _pGroupList->Next();
	}
	
	delete _pGroupList;
	_pGroupList = 0;
    }
    
    if ( _pGTM )
	_pGTM->setAllocatedSortArea(0);

    if ( _pHaving )
    {
	delete _pHaving;
    }
    
    if ( _pUnionSelect )
    {
	delete _pUnionSelect;
    }

    if ( _pCacheList )
	delete _pCacheList;
}

void CegoSelect::setTableManager(CegoDistManager* pGTM)
{
    _pGTM = pGTM;
    _pDBMng = _pGTM->getDBMng();
    _modId = _pGTM->getDBMng()->getModId("CegoSelect"); 	
}

void CegoSelect::setTabSetId(int tabSetId)
{
    _tabSetId = tabSetId;

    _pCache = _pGTM->getDBMng()->getQueryCache(_tabSetId);
    if ( _pCache )
    {
	_cacheEnabled = true;
    }
    else
    {
	_cacheEnabled = false;
    }
}

// the cleanup method makes a complete reinitialization of the select statement to the state
// after the instance has been created

void CegoSelect::cleanUp()
{
    _conjunctionList.Empty();
    _joinList.Empty();
    _joinPredList.Empty();

    _joinFields.Empty();
    _aggregationCount=0;
    _nextAid=0;
    
    int i = 0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {	
	if ( _pTC[i] )
	    delete _pTC[i];
	_pTC[i]=0;

	if ( _joinBuf[i] )
	    delete _joinBuf[i];
	_joinBuf[i]=0;
	
	_attrCondFlag[i] = false;
	_firstTuple[i]=true;
	_attrPred[i]=0;
	i++;
    }

    // cleanup
    if ( _pPred )
    {
	_pPred->setCheckedRec(false);
	_pPred->cleanUp();
	_pPred->clearAttrCache();
    }
    
    _isPrepared = false;

    _checkUnion = false;

    if ( _pUnionSelect )
    {
	_pUnionSelect->cleanUp();
    }

    if ( _pOrderSpace )
    {
	_pOrderSpace->resetOrderSpace();
    }

    _orderingDone=false;

    if ( _pGroupList )
    {
	_groupingDone = false;
		
	if ( _pGroupSpace )
	    _pGroupSpace->resetGroupSpace();
	
    }

    if ( _pCacheArray )
    {
	_pCache->releaseEntry(getQueryId());
	_pCacheArray=0;
    }

    _isCached = false;
}

// the prepare method organizes the select query parameters in a kind,
// that the join can be performed in the best way. This requires evaluating 
// of all attribute references and evaluation of the join condition to build
// an appropriate join table order list.
// the method detects, if the join was already prepared and if so, it just adjuts 
// changed join parameters ( e.g. contained procedure or parent query variables ) to the appropriate value

void CegoSelect::prepare() 
{
    if ( _isPrepared )
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Building join conditions ..."));
#endif
	buildJoinConditions();
	_rowCount=0;
	return;
    }

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Evaluating references ..."));	
#endif

    evalReferences();
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Analyzing join ..."));	
#endif

    analyzeJoin();

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Making order ..."));	
#endif

    makeOrder();

    if ( _selectMode == AGGREGATION )
	_aggDone=false;
        
    if ( _pGroupList )
    {
	_groupingDone = false;

	if ( _pGroupSpace == 0 )
	    _pGroupSpace = new CegoGroupSpace();
	else
	    _pGroupSpace->resetGroupSpace();

	if ( _selectMode == AGGREGATION )
	    _selectMode = GROUPING;
	else
	    throw Exception(EXLOC, "Missing aggregation for group clause");
    }
    
    if ( _pOrderList )
    {
	_orderingDone=false;
	if ( _pOrderSpace == 0 )
	    _pOrderSpace = new CegoOrderSpace();
	else
	    _pOrderSpace->resetOrderSpace();	    
    }
    
    _rowCount=0;

    // Last but not least, clear the attribute cache for all attributes
    
    CegoPredDesc **pPred = _conjunctionList.First();
    while ( pPred )
    {
        (*pPred)->clearAttrCache();
        pPred = _conjunctionList.Next();
    }
    if (_pPred)
    {
       _pPred->clearAttrCache();
    }

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr )
    {
        (*pExpr)->clearAttrCache();
        pExpr = _exprList.Next();
    }

    if ( _pHaving )
    {
	_pHaving->getExpr()->clearAttrCache();
	_pHaving->getAggExpr()->clearAttrCache();	
    }

    if ( _pOrderList )
    {
	pExpr = _pOrderList->First();
	while ( pExpr )
	{
	    (*pExpr)->clearAttrCache();
	    pExpr = _pOrderList->Next();
	}
    }
    
    // end of cache clear

    _isPrepared = true;

    if ( _pUnionSelect )
    {
	_pUnionSelect->prepare();
    }
}

// the reset method performas a reset on the query without making any join evaluation
// actions. This method can be used, if there was no change on any included join variables
// If the query result was cached, cache is still valid after reset

void CegoSelect::reset(bool doRelease)
{
    _joinLevel=0;
    _joinSize=0;

    if ( _selectMode == AGGREGATION )
	_aggDone=false;

    int i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	_firstTuple[i]=true;
	i++;
    }
    i=0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	if ( _pTC[i] )
	{
	    _pTC[i]->reset();
	    if ( doRelease )
	    {
		delete _pTC[i];
		_pTC[i] = 0;
	    }
	}
	i++;
    }

    if ( _pUnionSelect )
    {
	_checkUnion = false;
	_pUnionSelect->reset(doRelease);
    }
    
    if ( _pOrderSpace )
    {
	_pOrderSpace->resetOrderSpace();
    }

    if ( _pPred )
    {
	_pPred->clearAttrCache();
    }

    _orderingDone=false;

    // we have to check, if cache was claimed
    if ( _pCacheArray )
    {
	_pCache->releaseEntry(getQueryId());
	_pCacheArray=0;
    }
        
    _rowCount=0;
}

void CegoSelect::setUnionSelect(CegoSelect* pUnion)
{
    _pUnionSelect = pUnion;
}

bool CegoSelect::isPrepared() const
{
    return _isPrepared;
}

// the analyzeJoin method builds up a join strategy in terms of  
// creating a join order and filtering all required join attributes
// Depending on the analyzed join where condition, a table access plan 
// is created ( either index optimized or  full table scan )
// This also involves join optimization for underlying subqueries
void CegoSelect::analyzeJoin()
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Preparing join ..."));	
#endif    

    if (_pPred)
    {   
	if (_pPred->getCondition())
	{
	    CegoQueryHelper::makeCNF(_pPred->getCondition());   	    
	}
	createConjunctionList(_pPred);	
    }

    _conjunctionList += _viewConjunctionList;
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Consolidating conjunction list ..."));
#endif    

    consolidateConjunctionList();
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Building join tables ..."));
#endif

    buildJoinTables();

#ifdef CGDEBUG    
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Building join refs ..."));
#endif

    buildJoinRefs();

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Building join cursor conditions ..."));
#endif

    buildJoinConditions();

    _joinLevel = 0;
    _joinSize = 0;    

    // prepare sub queries
    if ( _pPred ) 
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Analyzing subselect ..."));
#endif
	ListT<CegoSelect*> queryList;
	_pPred->getSelectQueryList(queryList);
	
	CegoSelect **pSelect = queryList.First();
	while ( pSelect )
	{
	    // TODO : This has to be verified ( we changed from _joinFields to _joinBuf
	    (*pSelect)->setParentJoinBuf(_joinBuf);
	    pSelect = queryList.Next();
	}
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Preparing subselects ..."));
#endif
	_pPred->analyzeSelect();
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Subselects prepared"));
#endif
    }
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Select prepared"));
#endif
}

Element* CegoSelect::getPlan()
{
    prepare();
    checkValidRef();
    setParentJoinBuf();

    Element *pPlan = new Element(XML_PLAN_ELEMENT);

    while ( _joinLevel < _joinList.Size() )
    {

#ifdef CGDEBUG	
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Plan : Creating cursor for ") + _joinList[_joinLevel]->getTabName() + Chain(" on join level ") + Chain(_joinLevel));
#endif	

	_pTC[_joinLevel] = 
	    new CegoDistCursor(_pGTM, _joinList[_joinLevel]);
	
	if ( _attrCondFlag[_joinLevel] == true )
	{
#ifdef CGDEBUG	    
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Plan : Setting attrcond for ") + _joinList[_joinLevel]->getName() + Chain(" to ") + _attrCond[_joinLevel].toChain() + Chain(" on level") + Chain(_joinLevel));
#endif
	    if ( _attrCond[_joinLevel].setup(_joinBuf, 0 ) == false )
	    {
		// attribute condition can not be resolved on this join level
		// for this reason, we disable for the future
		// since this condition is disabled, we also have to enable predicate check again
		// for the final evaluation
#ifdef CGDEBUG	    
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Plan : Disabling condition flag for ") + _joinList[_joinLevel]->getName());
#endif
		
		_attrCondFlag[_joinLevel]=false;
		if ( _attrPred[_joinLevel] )
		{
		    _attrPred[_joinLevel]->setChecked(false);
		}
	    }	    
	    _pTC[_joinLevel]->distSetup(_attrCond[_joinLevel]);
	}
	else
	{
#ifdef CGDEBUG	    
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Plan : Setting no attrcond for ") + _joinList[_joinLevel]->getName() + Chain(" on level ") + Chain(_joinLevel));
#endif
	    _pTC[_joinLevel]->distSetup();
	}
	
	Element* pCursorPlan = _pTC[_joinLevel]->getPlan();
	
	pPlan->addContent(pCursorPlan);
	
	_joinLevel++;
    }
    
    ListT<Element*> planList;
    
    CegoExpr **pExpr = _exprList.First();
    while ( pExpr )
    {	
	(*pExpr)->setFieldListArray( _joinBuf  );
	
	(*pExpr)->getPlanList(planList);
	pExpr = _exprList.Next();
    }
    
    if ( _pPred ) 
    {	
	_pPred->getPlanList(planList);    
    }

    Element** pSubPlan = planList.First();
    while ( pSubPlan )
    {
	pPlan->addContent(*pSubPlan);	
	pSubPlan = planList.Next();
    }

    if ( _pUnionSelect )
    {
	pPlan->addContent(_pUnionSelect->getPlan());
    }    

    return pPlan;
}

void CegoSelect::setParentJoinBuf(ListT<CegoField>** pParentJoinBuf)
{
    _pParentJoinBuf = pParentJoinBuf;
    
    if ( _pUnionSelect )
    {
	_pUnionSelect->setParentJoinBuf(pParentJoinBuf);
    }
}

void CegoSelect::checkValidRef()
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Validating references ..."));
#endif

    ListT<CegoAttrDesc*> attrRefList = getAttrRefList();
    CegoAttrDesc **pAD = attrRefList.First();
    while ( pAD )
    {	
	if ( (*pAD)->isValid() == false )
	{
	    throw Exception(EXLOC, Chain("Unknown attribute ") + (*pAD)->toChain()); 
	}

#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Validated attribute reference ") + (*pAD)->toChain());
#endif

	pAD = attrRefList.Next();
    }
}

void CegoSelect::setProcBlock(CegoProcBlock *pBlock)
{
    _pBlock = pBlock;

    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	(*pExpr)->setBlock(_pBlock);	    
	pExpr = _exprList.Next();
    }

    if ( _pUnionSelect )
    {
	_pUnionSelect->setProcBlock(pBlock);
    }
}

bool CegoSelect::nextTuple(ListT<CegoField>& jfl)
{    
    if ( _rowLimit > 0 && _rowCount >= _rowLimit )
    {
	if ( _cacheEnabled && _pCacheList )
	{
	    _isCached = true;
	    SetT<CegoObject> tableList;
	    getObjectList(tableList);

	    _pCache->addEntry( getQueryId() , tableList, _pCacheList, _cacheSchema );
	    _pCacheList = 0;
	    _rowCount=0;
	}

	if ( _pCache && _pCacheArray )
	{
	    _pCache->releaseEntry(getQueryId());
	    _pCacheArray=0;
	}
	return false;
    }

    if ( _cacheEnabled && _rowCount == 0 )
    {	
	_pCacheArray = _pCache->claimEntry( getQueryId(), _cacheSchema, _cacheRows );
	
	if ( _pCacheArray )
	{
	    _isCached = true;
	}
	else
	{
	    _isCached = false;
	    _cacheEntrySize = 0;

	    // check for system objects
	    SetT<CegoObject> tableList;
	    getObjectList(tableList);
	    bool sysTableFound=false;
	    CegoObject *pO = tableList.First();
	    while ( pO && sysTableFound == false)
	    {
		if ( pO->getType() == CegoObject::SYSTEM )
		{
		    sysTableFound=true;
		}
		pO = tableList.Next();
	    }
	    if ( sysTableFound == false )
		_pCacheList = new ListT< ListT< CegoFieldValue > >;
	}
    }

    if ( _cacheEnabled && _pCacheArray && _isCached )
    {
	CegoFieldValue** pCacheRow = 0;
	if ( _rowCount == 0 ) 
	{
	    if ( _cacheRows > 0 )
	    {
		pCacheRow = _pCacheArray[_rowCount];
	    }			  
	}
	else
	{
	    if ( _rowCount < _cacheRows )
	    {			    
		pCacheRow = _pCacheArray[_rowCount];
	    }
	}
	_rowCount++;
		    
	if ( pCacheRow )
	{	    
	    jfl = _cacheSchema;
	    CegoField *pF = jfl.First();
	    int pos=0;
	    while ( pF )
	    {
		pF->setValue(*pCacheRow[pos]);
		pF = jfl.Next();
		pos++;
	    }
	    return true;
	}

	// FOR DEBUGGING : sleep for a while
	// cout << "Sleeping for 10 sec ..." << endl;
	// Sleeper s;
	// s.secSleep(10);
	
	_pCache->releaseEntry(getQueryId());
	_pCacheArray=0;
	
	_rowCount = 0;
	return false;
    }
    else
    {
	if ( _checkUnion == false )
	{
	    if ( nextAggTuple(jfl) )
	    {
		if ( _cacheEnabled && _pCacheList )
		{
		    ListT<CegoFieldValue> staticFieldList;
		    CegoField* pF = jfl.First();
		    while ( pF )
		    {
			staticFieldList.Insert(pF->getValue().getLocalCopy());
			_cacheEntrySize += pF->getValue().size();
			pF = jfl.Next();
		    }

		    if ( _pCache->getMaxSize() > _cacheEntrySize )
		    {
			_pCacheList->Insert(staticFieldList);
		    }
		    else
		    {
			delete _pCacheList;
			_pCacheList = 0;
		    }

		    // if the cache schema is still not set up, we have to do this
		    // this just has to be done on time 
		    if ( _cacheSchema.Size() == 0 )
			_cacheSchema = jfl;
		}
		_rowCount++;
		return true;
	    }
	}

	if ( _pUnionSelect )
	{
	    _checkUnion = true;

	    jfl.Empty();
	    if ( _pUnionSelect->nextTuple(jfl) )
	    {
		if ( _cacheEnabled && _pCacheList )
		{
		    ListT<CegoFieldValue> staticFieldList;
		    CegoField* pF = jfl.First();
		    while ( pF )
		    {
			staticFieldList.Insert(pF->getValue().getLocalCopy());
			_cacheEntrySize += pF->getValue().size();
			pF = jfl.Next();
		    }

		    if ( _pCache->getMaxSize() > _cacheEntrySize )
		    {
			_pCacheList->Insert(staticFieldList);
		    }
		    else
		    {
			delete _pCacheList;
			_pCacheList = 0;
		    }
		}
		_rowCount++;
		return true;
	    }
	}

	// no more tuples found, finalize query caching

	if ( _cacheEnabled && _pCacheList )
	{
	    _isCached = true;

	    SetT<CegoObject> tableList;

	    getObjectList(tableList);

	    _pCache->addEntry( getQueryId() , tableList, _pCacheList, _cacheSchema );
	    delete _pCacheList;
	    _pCacheList = 0;
	    _rowCount=0;
	}
	return false;
    }
}

bool CegoSelect::isCached() const
{
    return _isCached;
}
    
Chain CegoSelect::getQueryId() const
{
    Chain id;

    if ( _isDistinct && _selectMode != AGGREGATION )
	id += Chain("#d#");
    
    if ( _exprList.Size() == 0 )
    {
	id += Chain("*");
    }
    else
    {
	CegoExpr** pExpr = _exprList.First();
	while (pExpr)
	{
	    id += (*pExpr)->getId();
	    id += Chain("#");
	    if ( (*pExpr)->getAlias() != Chain() )
	    {
		id += (*pExpr)->getAlias();
		id += Chain("#");
	    }
	    pExpr = _exprList.Next();
	}
    }

    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	id += (*pCO)->getId();
	id += Chain("#");
	pCO = _coList.Next();
    }

    CegoPredDesc** pPred = _conjunctionList.First();
    while ( pPred )
    {
	id += (*pPred)->getId();
	id += Chain("#");
	pPred = _conjunctionList.Next();
    }

    if (_pGroupList)
    {
	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    id += (*pAttrDesc)->getId();
	    id += Chain("#");
	    pAttrDesc = _pGroupList->Next();
	}	
	if ( _pHaving )
	{
	    id += _pHaving->getId();
	    id += Chain("#");
	}
    }

    if ( _pNativeOrderList && _pNativeOrderOptList )
    {
	CegoExpr **pExpr = _pNativeOrderList->First();
	CegoOrderNode::Ordering *pOrdering = _pNativeOrderOptList->First();
	while ( pExpr && pOrdering )
	{
	    id += (*pExpr)->getId();
	    id += Chain("#");
	    if ( *pOrdering == CegoOrderNode::ASC )
		id += Chain("A");
	    else
		id += Chain("D");
	    id += Chain("#");
	    pExpr = _pNativeOrderList->Next();
	}
    }

    if ( _rowLimit > 0 )
    {
	id += Chain(_rowLimit);	
    }

    if ( _pUnionSelect )
    {
	id += Chain("#");
	id += _pUnionSelect->getQueryId();
    }
    return id;   
}

void CegoSelect::getObjectList(SetT<CegoObject>& objList) const
{
    // get tables for select expressions
    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	ListT<CegoSelect*> queryList;
	(*pExpr)->getSelectQueryList( queryList);
	CegoSelect** pSelect = queryList.First();
	while ( pSelect )
	{
	    (*pSelect)->getObjectList(objList);
	    pSelect = queryList.Next();
	}

	(*pExpr)->getFunctionList(objList);

	pExpr = _exprList.Next();

    }

    // get tables for join
    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	addObject4CO(*pCO, objList);	
	pCO = _coList.Next();
    }

    if (_pPred)
    {
	ListT<CegoSelect*> queryList;
	_pPred->getSelectQueryList( queryList);

	CegoSelect** pSelect = queryList.First();
	while ( pSelect )
	{
	    (*pSelect)->getObjectList(objList);
	    pSelect = queryList.Next();
	}

	if ( _pPred->getExpr1() )
	    _pPred->getExpr1()->getFunctionList(objList);
	if ( _pPred->getExpr2() )
	    _pPred->getExpr2()->getFunctionList(objList);
	if ( _pPred->getExpr3() )
	    _pPred->getExpr3()->getFunctionList(objList);

    }    
}

void CegoSelect::addObject4CO(CegoContentObject *pCO, SetT<CegoObject>& objList) const
{
    if ( pCO->getType() == CegoObject::JOIN )
    {
	ListT<CegoPredDesc*> predList;
	
	((CegoJoinObject*)pCO)->getPredList(predList);
	
	CegoPredDesc **pPred = predList.First();
	while ( pPred )
	{
	    ListT<CegoSelect*> queryList;
	    (*pPred)->getSelectQueryList( queryList);

	    CegoSelect** pSelect = queryList.First();
	    while ( pSelect )
	    {
		(*pSelect)->getObjectList(objList);
		pSelect = queryList.Next();
	    }

	    pPred = predList.Next();
	}
	
	ListT<CegoContentObject*> subList = pCO->getSubCOList();
	CegoContentObject** pSubCO = subList.First();
	while ( pSubCO )
	{
	    addObject4CO(*pSubCO, objList);
	    pSubCO = subList.Next();
	}
    }
    else if ( pCO->getType() == CegoObject::VIEW )
    {
	
	_pDBMng->useObject(pCO->getTabSetId(), pCO->getTabName(), CegoObject::VIEW, CegoDatabaseManager::SHARED, _pGTM->getThreadId());
	
	CegoView *pView;
	
	try
	{
	    pView = _pGTM->getView(pCO->getTabSetId(), pCO->getTabName());
	}
	catch ( Exception e )
	{
	    _pDBMng->unuseObject(pCO->getTabSetId(), pCO->getTabName(), CegoObject::VIEW);
	    throw e;
	}
	CegoSelect* pSelect = pView->getSelect();

	pSelect->getObjectList(objList);

	objList.Insert( CegoObject(CegoObject::VIEW, pCO->getTabName(), pCO->getTabSetId()));	
	
	_pDBMng->unuseObject(pCO->getTabSetId(), pCO->getTabName(), CegoObject::VIEW);
	
    }
    else if ( pCO->getType() == CegoObject::TABLE )
    {
	objList.Insert( CegoObject(CegoObject::TABLE, pCO->getTabName(), pCO->getTabSetId()));	
    }   
    else if ( pCO->getType() == CegoObject::SYSTEM )
    {
	objList.Insert( CegoObject(CegoObject::SYSTEM, pCO->getTabName()));
    }
}

bool CegoSelect::nextAggTuple(ListT<CegoField>& jfl)
{
    if ( _selectMode == AGGREGATION )
    {
	if ( _aggDone )
	    return false;
	bool moreTuple=true;

	initAggregation();

	while ( moreTuple )
	{
	    ListT<CegoField> aggTuple;
	    moreTuple = nextOrderedTuple(aggTuple);

	    if ( moreTuple )
	    {
		aggregateTuple(aggTuple);
		_aggregationCount++;
	    }	   
	}
	
	evalAggregation(jfl);
	
	_aggDone=true;
	return true;
    }
    else
    {
	return nextOrderedTuple(jfl);
    }
}

bool CegoSelect::nextOrderedTuple(ListT<CegoField>& jfl)
{
    if ( _pGTM == 0 )
	throw Exception(EXLOC, "No tablemanager set");

    if ( _pOrderList )
    {
	if ( _orderingDone )
	{
	    jfl = _orderedSchema;
	    
	    if ( _isDistinct )
	    {	       
		while ( _pOrderCursor->getNext(jfl) )
		{
		    CegoField* pF1=_dfl.First();
		    CegoField* pF2=jfl.First();
		    int i=0;
		    while ( pF1 && pF2 && i < _dcount )
		    {
			if ( pF1->getValue() != pF2->getValue() )
			{
			    _dfl = jfl;
			    return true;
			}
			i++;
			pF1=_dfl.Next();
			pF2=jfl.Next();
		    }
		}
		return false;
	    }
	    else
	    {
		return _pOrderCursor->getNext(jfl);
	    }
	}
	else
	{
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Collecting tuples in orderspace ..."));
#endif
	    ListT<CegoField> dataTuple;
	    ListT<CegoField> orderTuple;
	    bool isInit=false;

	    while ( nextResultTuple(dataTuple, orderTuple) )
	    {
		
		ListT<CegoField> orderKey;

		CegoExpr **pExpr = _pOrderList->First();
		while ( pExpr )
		{	
		    ListT<CegoAttrDesc*> attrRefList = (*pExpr)->getAttrRefList();

		    CegoAttrDesc** pAttrRef = attrRefList.First();

		    while ( pAttrRef )
		    {
			if ( CegoField* pF = orderTuple.Find(CegoField((*pAttrRef)->getTableName(), (*pAttrRef)->getAttrName())) )
			{
			    orderKey.Insert(*pF);
			}
			pAttrRef = attrRefList.Next();
		    }

		    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Checking aggregation..."));
		    // check for aggregation id
		    ListT<CegoAggregation*> aggList = (*pExpr)->getAggregationList();
		    CegoAggregation **pAgg = aggList.First();
		    
		    while ( pAgg )
		    {
			bool notFound = true;
			CegoField* pF = orderTuple.First();
			while ( pF && notFound )
			{
			    if ( pF->getId() == (*pAgg)->getAggregationId() )
			    {
				orderKey.Insert(*pF);
				notFound = false;
			    }
			    else
			    {
				pF = orderTuple.Next();
			    }
			}
			
			if ( notFound )
			{
			    Chain msg = Chain("Unknown order expression ") +  (*pAgg)->toChain();
			    throw Exception(EXLOC, msg);
			}
			pAgg = aggList.Next();				
		    }			    
		    pExpr = _pOrderList->Next();
		}

		if ( isInit == false )
		{
		    _pOrderSpace->initOrderSpace(_pOrderList, _pOrderOptList,  _pGTM->getDBMng()->getSortAreaSize(_tabSetId));
		}

		_pOrderSpace->insertTuple(orderKey, dataTuple);
		if ( _pGroupSpace )
		    _pGTM->setAllocatedSortArea(_pOrderSpace->numAllocated() + _pGroupSpace->numAllocated());
		else
		    _pGTM->setAllocatedSortArea(_pOrderSpace->numAllocated());

		if ( isInit == false )
		{    
		    _orderedSchema = dataTuple;
		    isInit=true;
		}
	    }

	    // no rows were selected
	    if ( isInit == false )
	    {
		return false;
	    }

	    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Ordering done"));
	    _orderingDone = true;
	    
	    if ( _pOrderCursor )
		delete _pOrderCursor;
	    
	    _pOrderCursor = _pOrderSpace->getCursor();
	    
	    jfl = _orderedSchema;

	    bool moreTuple = _pOrderCursor->getFirst(jfl); 

	    if ( _isDistinct )
		_dfl = jfl;
	    return moreTuple;
	}
    }
    else
    {
	ListT<CegoField> ofl;
	return nextResultTuple(jfl, ofl);
    }
}

// the nextResultTuple method is used as a cursor to retrieve the corresponding result set of 
// the defined select query. the method returns true until no further tuple results are
// returned from the query. The current tuple result is provided in the reference parameter jfl

bool CegoSelect::nextResultTuple(ListT<CegoField>& jfl, ListT<CegoField>& ofl)
{
    switch ( _selectMode )
    {
    case PLAIN:
    case AGGREGATION:
    {
	return nextJoinTuple(jfl, ofl);
    }
    case GROUPING:
    {	
	ListT<CegoField> gfl;

	bool moreTuple = nextGroupedTuple(gfl);
	/*
	cout << "---- Grouped Tuple ----" << endl;
	CegoField *pF = gfl.First();
	while ( pF )
	{
	    cout << "Table=" << pF->getTableName() << " Alias=" << pF->getTableAlias() << " Attr=" << pF->getAttrName();
	    cout << "Value=" << pF->getValue() << endl;
	    pF = gfl.Next();
	}
	*/
		
	ofl=gfl;

	if ( moreTuple == false )
	    return false;

	jfl.Empty();
	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    ListT<CegoAggregation*> aggList;	    
	    aggList = (*pExpr)->getAggregationList();

	    if ( aggList.Size() > 0 ) 
	    {
		CegoField f;
		
		CegoAggregation **pAgg = aggList.First();
		while ( pAgg )
		{
		    bool notFound=true;
		    CegoField *pF = gfl.First();
		    while ( pF && notFound )
		    {
			if ( pF->getId() == (*pAgg)->getAggregationId() )
			{
			    (*pAgg)->setFieldValue(pF->getValue());
			    f.setId((*pAgg)->getAggregationId());
			    notFound=false;
			}			    
			pF = gfl.Next();
		    }		
		    pAgg = aggList.Next();
		}		

		(*pExpr)->setFieldListArray(&gfl);
		f.setValue ( (*pExpr)->evalFieldValue() );
		f.setAttrName((*pExpr)->getAlias());
		jfl.Insert(f);
	    }
	    else
	    {
		CegoField f;
		f.setId(0);
		
		(*pExpr)->setFieldListArray(&gfl );
		
		f.setValue((*pExpr)->evalFieldValue() );
		f.setAttrName((*pExpr)->getAlias());
		jfl.Insert(f);
	    }
	    pExpr = _exprList.Next();
	}
	return moreTuple;
    }
    }
}

bool CegoSelect::nextGroupedTuple(ListT<CegoField>& jfl)
{
    if ( _groupingDone )
    {
	jfl = _pGroupSpace->getSchema();

	if ( _pHaving )
	{
	    while ( _pGroupCursor->getNext(jfl) )
	    {		
		_pHaving->getExpr()->setFieldListArray(&jfl);
		_pHaving->getAggExpr()->setFieldListArray(&jfl);
		
		setAggregationValue(_pHaving->getExpr(), jfl);
		setAggregationValue(_pHaving->getAggExpr(), jfl);
		
		if ( matchFieldValue(_pHaving->getAggExpr()->evalFieldValue(),
				     _pHaving->getExpr()->evalFieldValue(), 
				     _pHaving->getComparison()))
		{
		    return true;
		}
	    }
	    return false;
	}
	else
	{
	    return _pGroupCursor->getNext(jfl);
	}
    }
    else
    {	
	ListT<CegoAggregation*> aggList;

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{	    	    
	    aggList += (*pExpr)->getAggregationList();
	    pExpr = _exprList.Next();
	}
	if ( _pHaving )
	{
	    aggList += _pHaving->getAggExpr()->getAggregationList();
	    aggList += _pHaving->getExpr()->getAggregationList();	    
	}

	if ( _pOrderList )
	{
	    CegoExpr **pExpr = _pOrderList->First();
	    while ( pExpr )
	    {
		aggList += (*pExpr)->getAggregationList();
		pExpr = _pOrderList->Next();
	    }
	}
	
	ListT<CegoField> dataTuple;
	bool isInit=false;

	ListT<CegoField> ofl;
	while ( nextJoinTuple(dataTuple, ofl) )
	{	    
	    if ( isInit == false )
	    {
		ListT<CegoField> groupKey;
		CegoAttrDesc** pAttrRef = _pGroupList->First();
		
		while ( pAttrRef )
		{
		    CegoField* pF;
		    if ( (pF = dataTuple.Find(CegoField((*pAttrRef)->getTableName(), (*pAttrRef)->getAttrName()))) != 0 )
		    {
			groupKey.Insert(*pF);
		    }
		    else
		    {
			Chain msg = Chain("Unknown group attribute ") +  (*pAttrRef)->getTableName() + Chain(".") + (*pAttrRef)->getAttrName();
			throw Exception(EXLOC, msg);
		    }
		    pAttrRef = _pGroupList->Next();
		}

		_pGroupSpace->initGroupSpace(groupKey, aggList, _pGTM->getDBMng()->getSortAreaSize(_tabSetId));
		isInit = true;
	    }

	    _pGroupSpace->insertTuple(dataTuple);
	    _pGTM->setAllocatedSortArea(_pGroupSpace->numAllocated());
	}
	
	_groupingDone = true;

	if ( isInit == false )
	    return false;

	if ( _pGroupCursor )
	    delete _pGroupCursor;
	
	_pGroupCursor = _pGroupSpace->getCursor();
	
	jfl = _pGroupSpace->getSchema();

	bool moreTuple = _pGroupCursor->getFirst(jfl); 

	if ( _pHaving )
	{
	    while ( moreTuple )
	    {	
		_pHaving->getExpr()->setFieldListArray(&jfl);
		_pHaving->getAggExpr()->setFieldListArray(&jfl);

		setAggregationValue(_pHaving->getAggExpr(), jfl);
		setAggregationValue(_pHaving->getExpr(), jfl);
		
		if ( matchFieldValue(_pHaving->getAggExpr()->evalFieldValue(),
				     _pHaving->getExpr()->evalFieldValue(), 
				 _pHaving->getComparison()))
		{
		    return true;
		}
		else
		{
		    moreTuple = _pGroupCursor->getNext(jfl);
		}
	    }
	}
	return moreTuple;
    }
}

bool CegoSelect::nextJoinTuple(ListT<CegoField>& jfl, ListT<CegoField>& ofl)
{    
    if ( _pGTM->isAborted() )
    {
	throw Exception(EXLOC, "Query aborted");
    }

    ofl.Empty();

    while ( true )
    {
	bool joinComplete=false;
	while ( _joinLevel < _joinList.Size()  && ! joinComplete)
	{
	    bool moreTuple=false;
    
	    if ( _firstTuple[_joinLevel] ) 
	    {
		if ( _pTC[_joinLevel] == 0 )
		{

#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Creating cursor for ") 
		    		  + _joinList[_joinLevel]->getTabName() + Chain(" on join level ") + Chain(_joinLevel));
#endif
		    
		    _pTC[_joinLevel] = 
			new CegoDistCursor(_pGTM,_joinList[_joinLevel]);
		}
		else
		{
		    _pTC[_joinLevel]->reset();		   
		}
		if ( _attrCondFlag[_joinLevel] == true )
		{
#ifdef CGDEBUG	    
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Setting attrcond =  ") + _attrCond[_joinLevel].toChain() + Chain(" on level ") + Chain(_joinLevel));
#endif
		    bool doSetup = true;

		    if ( _attrCond[_joinLevel].setup(_joinBuf, 0 ) == false )
		    {
			// _pDBMng->log(_modId, Logger::DEBUG, Chain("Native attr cond set up failed"));
			if ( _pParentJoinBuf )
			{
			    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Setting up with parent join buf"));
			    if ( _attrCond[_joinLevel].setup(_pParentJoinBuf, 0) == false )
			    {				    
				// _pDBMng->log(_modId, Logger::DEBUG, Chain("Parent attr cond set up failed"));
				doSetup = false;
			    }
			}
			else
			{
			    doSetup = false;
			}				
		    }

		    if ( doSetup )
		    {
#ifdef CGDEBUG	    
			_pDBMng->log(_modId, Logger::DEBUG, Chain("Cursor set up for attrcond =  ") + _attrCond[_joinLevel].toChain());
#endif
			_pTC[_joinLevel]->distSetup(_attrCond[_joinLevel]);    
		    }
		    else
		    {			    
			_attrCondFlag[_joinLevel]=false;
			if ( _attrPred[_joinLevel] )
			{
			    _attrPred[_joinLevel]->setChecked(false);
			}
			_pTC[_joinLevel]->distSetup();
		    }
		}
		else
		{
		    _pTC[_joinLevel]->distSetup();
		}

		/*		
		cout << "NextTuple : Size=" << _joinSize << " NumCO= " <<  _joinList[_joinLevel]->getSubCOList().Size() << endl;
		for ( int i=_joinSize; i<_joinList[_joinLevel]->getSubCOList().Size(); i++)
		{
		    cout << "SchemaSize=" << _joinBuf[i].Size() << endl;
		}
		*/

#ifdef CGDEBUG	    
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Retrieving first tuple from cursor ..."));
#endif
		
		moreTuple = _pTC[_joinLevel]->nextTuple(_joinBuf, _joinSize, _joinList[_joinLevel]->getSubCOList().Size());		

#ifdef CGDEBUG
		if ( moreTuple )
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Tuple retrieved"));
		else
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("No more tuple available"));
#endif

		
	    }
	    else
	    {				
		/*
		cout << "NextTuple : Size=" << _joinSize << " NumCO= " <<  _joinList[_joinLevel]->getSubCOList().Size() << endl;
		for ( int i=_joinSize; i<_joinList[_joinLevel]->getSubCOList().Size(); i++)
		{
		    cout << "SchemaSize=" << _joinBuf[i].Size() << endl;
		}
		*/

#ifdef CGDEBUG	    
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Retrieving next tuple from cursor ..."));
#endif

		moreTuple = _pTC[_joinLevel]->nextTuple(_joinBuf, _joinSize, _joinList[_joinLevel]->getSubCOList().Size());

#ifdef CGDEBUG
		if ( moreTuple )
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Tuple retrieved"));
		else
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("No more tuple available"));
#endif

	    }

	    bool setFinished = false;

	    if ( moreTuple)
	    {
		_firstTuple[_joinLevel]=false;
		
		if ( _joinLevel < _joinList.Size() - 1 )
		{
		    _joinSize += _joinList[_joinLevel]->getSubCOList().Size();
		    _joinLevel++;
		}
		else
		    joinComplete=true;
	    }
	    else
	    {		
		setFinished = true;
	    }
	    
	    if ( setFinished )
	    {
		_firstTuple[_joinLevel]=true;

		if ( _joinLevel > 0 )
		{
		    _joinSize -= _joinList[_joinLevel]->getSubCOList().Size();
		    _joinLevel--;
		}
		else
		{
		    int i=0;
		    while ( i < TABMNG_MAXJOINLEVEL )
		    {			
			if ( _pTC[i] )
			    _pTC[i]->reset();
			i++;
		    }
		    return false;
		}
	    }
	}

	bool checkPred = true;
	CegoPredDesc **pPred = _conjunctionList.First();

	while ( pPred && checkPred )
	{
	    if ( ! (*pPred)->isChecked() )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Evaluating predicate ") + (*pPred)->toChain());
#endif
				
		checkPred = CegoQueryHelper::evalPredicate(_pParentJoinBuf,
						      0,
						      _joinBuf,
						      0,
						      *pPred, _pBlock);
#ifdef CGDEBUG
		if ( checkPred )
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Predicate matches"));
		else
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Predicate matches not"));
#endif		
	    }

	    pPred = _conjunctionList.Next();
	}

	if ( checkPred )
	{
	    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Join buf matches"));
	    
	    bool isAgg;
	    if ( _selectMode == AGGREGATION || _selectMode == GROUPING )
		isAgg=true;
	    else
		isAgg=false;

	    jfl.Empty();

	    evalSelection(_exprList, _joinBuf, isAgg, jfl);

	    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Selection evaluated"));

	    if ( _pOrderList )
	    {
		CegoExpr **pExpr = _pOrderList->First();
		while ( pExpr )
		{
		    ListT<CegoAttrDesc*> attrRefList = (*pExpr)->getAttrRefList();
		    
		    CegoAttrDesc** pAttrRef = attrRefList.First();
		    
		    while ( pAttrRef )
		    {
			int i=0;
			bool notFound=true;
			while ( i< _joinSize + _joinList[_joinLevel]->getSubCOList().Size()
				&& notFound )
			{
			    if ( CegoField* pF = _joinBuf[i]->Find(CegoField((*pAttrRef)->getTableName(), (*pAttrRef)->getAttrName())))
			    {
				ofl.Insert(*pF);
				notFound=false;
			    }
			    i++;
			}
			if ( notFound )
			{
			    Chain msg = Chain("Unknown order attribute ") + (*pAttrRef)->getTableName() + Chain(".") + (*pAttrRef)->getAttrName();
			    throw Exception(EXLOC, msg);
			}			
			pAttrRef = attrRefList.Next();
		    }
		    
		    pExpr = _pOrderList->Next();
		}
	    }
	    return true;
	}
    }
}

CegoPredDesc* CegoSelect::getPredicate()
{
    return _pPred;
}

void CegoSelect::getSchema(ListT<CegoField>& schema)
{
    if ( _exprList.isEmpty() )
    {
	schema = _joinFields;
    }
    else
    {
	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    CegoAttrDesc* pAttrDesc = (*pExpr)->checkAttr();
	    if ( pAttrDesc )
	    {
		if ( pAttrDesc->getAttrName() == Chain(SELECTION_WILDCARD))
		{
		    bool refFound=false;
		    CegoField *pF = _joinFields.First();
		    while ( pF )
		    {
			if ( (Chain)pF->getTableName() ==  (Chain)pAttrDesc->getTableName()
			     || (Chain)pF->getTableAlias() ==  (Chain)pAttrDesc->getTableName() )
			{
			    refFound=true;

			    schema.Insert(*pF);
			}
			pF = _joinFields.Next();
		    }
		    if ( refFound == false )
		    {
			Chain msg = Chain("Unknown table reference ") + pAttrDesc->getTableName();
			throw Exception(EXLOC, msg);		    
		    }
		}
		else
		{
		    (*pExpr)->setFieldListArray( &_joinFields );		   
		    CegoField f = (*pExpr)->evalField();
		    schema.Insert(f);
		}
	    }
	    else
	    {
		(*pExpr)->setFieldListArray( &_joinFields );
		CegoField f = (*pExpr)->evalField();
		schema.Insert(f);
	    }
	    pExpr = _exprList.Next();
	}
    }
    return;
}

void CegoSelect::createConjunctionList(CegoPredDesc* pPred)
{
    if (pPred)
    {
	CegoCondDesc* pC = pPred->getCondition();	
	if (pC)
	{
	    if (pC->getCondType() == CegoCondDesc::AND)
	    {
		createConjunctionList(pC->Left());
		createConjunctionList(pC->Right());
	    }
	    else
	    {
		_conjunctionList.Insert(pPred);
	    }
	}
	else
	{
	    _conjunctionList.Insert(pPred);
	}
    }
}

void CegoSelect::consolidateConjunctionList()
{
    // first step :
    // ============
    // we have to check for predicates, where the referenced tables 
    // are the same. These predicates are merged then.
    // This is important to analyse the corresponding  table join order

    bool finished = false;
    
    while ( finished == false )
    {
	SetT<Chain> tableRefSetA;
	SetT<Chain> tableRefSetB;
	
	bool predFound = false;
	CegoPredDesc **pPredA = _conjunctionList.First();
	while ( pPredA && predFound == false)
	{
	    if ( (*pPredA)->isChecked() == false && (*pPredA)->hasOrCond() == false )
		predFound = true;
	    else
		pPredA = _conjunctionList.Next();
	}
	
	if ( pPredA )
	{      
	    tableRefSetA = (*pPredA)->getTableRefSet();
	    (*pPredA)->setChecked(true);
	}
	else
	{
	    finished = true;
	}
	
	if ( finished == false )
	{
	    bool isMerged=false;
	    CegoPredDesc **pPredB = _conjunctionList.Next();
	    while ( pPredB && isMerged == false )
	    {
		if ( (*pPredB)->hasOrCond() )
		{
		    pPredB = _conjunctionList.Next();
		}
		else
		{
		    tableRefSetB = (*pPredB)->getTableRefSet();
		    
		    if ( tableRefSetA == tableRefSetB )
		    {
			CegoPredDesc *pA = *pPredA;
			CegoPredDesc *pB = *pPredB;
			
			// cout << "Pred A =  " << (*pPredA)->toChain()  << endl;	    
			// cout << "Pred B =  " << (*pPredB)->toChain()  << endl;	    
			
			_conjunctionList.Remove(*pPredA);
			_conjunctionList.Remove(*pPredB);
			
			CegoCondDesc* pCond = new CegoCondDesc(CegoCondDesc::AND);
			pCond->setLeft(pA);
			pCond->setRight(pB);
			
			_conjunctionList.Insert(new CegoPredDesc(pCond));
			isMerged = true;
		    }
		    if ( isMerged == false )
			pPredB = _conjunctionList.Next();
		}
	    }
	}
    }

    CegoPredDesc **pPred = _conjunctionList.First();
    while ( pPred )
    {
	(*pPred)->setCheckedRec(false);
	pPred = _conjunctionList.Next();
    }

    // second step :
    // ============
    // the conjunction list is ordered regarding the 
    // amount of referenced tables.
    // predicates with more referenced tables are put to the beginning 
    // of the list.
    // The ordered conjunction list results in an appropriate table join order 
    // ( method buildJoinTables ), so predicates with index conditions for two joined
    // tables ( a = b ) are treated first before constant comparisions ( a = 'alpha' )
    // 
    // Additional logic added at 12.07.2015 :
    // Inappropriate predicates ( e.g. a != b, exists ( .. ), a in ( ..), etc.
    // are first filtered. Otherwise, this can lead to inefficient join ordering
    // Also predicates are filtered out, which don't have any table references.
    // Although this might be stupid, it is allowed. Don't filtering out these predicates,
    // would result in an infinite growsize-loop 
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Conjunction list size =  ") + Chain(_conjunctionList.Size()));
#endif
	        
    StackT<CegoPredDesc*> orderedStack;

    // first filter out inappropriate predicates and push to stack first
    pPred = _conjunctionList.First();
    while ( pPred )
    {
	bool isAppropriate = (*pPred)->getMode() == CegoPredDesc::EXPRCOMP && (*pPred)->getComparison() == EQUAL;
	if ( isAppropriate == false || (*pPred)->getTableRefSet().Size() == 0 )
	{

#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Pushing inapp predicate ") + (*pPred)->toChain());
#endif

	    orderedStack.Push(*pPred);
	    _conjunctionList.Remove(*pPred);
	    pPred = _conjunctionList.First();
	}
	else
	{
	    pPred = _conjunctionList.Next();
	}
    }
        
    int growSize = 1;
    int numAdded = 0;
    while ( numAdded < _conjunctionList.Size() )
    {

	pPred = _conjunctionList.First();
	while ( pPred )
	{

	    if ( (*pPred)->getTableRefSet().Size() == growSize )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Pushing app predicate ") + (*pPred)->toChain());
#endif

		numAdded++;
		orderedStack.Push(*pPred);
	    }
	    pPred = _conjunctionList.Next();
	}
	growSize++;
    }
    
    _conjunctionList.Empty();
    CegoPredDesc *pP;
    while ( orderedStack.Pop(pP))
	_conjunctionList.Insert(pP);
    
}

//
// the buildJoinTables method provides an appropriate order of the required join tables
// for the query execution. In this sense, first the table are joined, where predicates have
// been defined. In this sense, inner and outer joins are treated first 
//

void CegoSelect::buildJoinTables()
{

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Building join table list"));
#endif

    // growStep is the number of tables to join with the next join step
    // we start with growStep 1 
    int growStep=1;
    
    while ( _joinList.Size() != _coList.Size() && growStep <= _coList.Size() )
    {
	bool isGrowing = true;
	while ( isGrowing  )
	{
	    isGrowing = false;

	    CegoPredDesc **pPred = _conjunctionList.First();
	    while ( pPred )
	    {
		ListT<CegoContentObject*> newObjectList;
		getPredObjectList(*pPred, newObjectList);

#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Analysing predicate ") + (*pPred)->toChain());

		Chain tlist;
		CegoContentObject **pDO = newObjectList.First();
		while ( pDO )
		{
		    tlist = tlist + Chain(" ") + (*pDO)->getTabName(); 
		    pDO = newObjectList.Next();
		}

		_pDBMng->log(_modId, Logger::DEBUG, Chain("Found ") + Chain(newObjectList.Size()) + Chain(" tables [") + tlist + Chain("]"));
#endif

		CegoContentObject** pOE = _joinList.First();
		while ( pOE )
		{    		    
		    CegoContentObject **pNCO = newObjectList.First();
		    while ( pNCO )
		    {
			if ( (**pNCO) == (**pOE) )
			{
			    newObjectList.Remove(*pNCO);
			    pNCO = newObjectList.First();
			}
			else
			{
			    pNCO = newObjectList.Next();
			}
		    }
		    pOE = _joinList.Next();
		}

		if ( newObjectList.Size() == growStep )
		{

#ifdef CGDEBUG
		    Chain tlist;
		    CegoContentObject **pDO = newObjectList.First();
		    while ( pDO )
		    {
			tlist = tlist + Chain(" ") + (*pDO)->getTabName(); 
			pDO = newObjectList.Next();
		    }
		    
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("On growstep = " ) + Chain(growStep) + Chain(" adding ") + Chain(newObjectList.Size()) + Chain(" tables to join list  [") + tlist + Chain("]"));
#endif
	
		    _joinList += newObjectList;

		    /* we reset growStep to 1, since newObjectList was modified */ 
		    growStep = 1;
		    isGrowing = true;

		    pPred = _conjunctionList.First();
		}
		else
		{
		    pPred = _conjunctionList.Next();
		}
	    }	    
	}
	growStep++;
    }

    if ( _joinList.Size() != _coList.Size() )
    {
	CegoContentObject** pCO = _coList.First();
	
	while (pCO)
	{		    	    
	    bool notFound=true;
	    CegoContentObject **pJCO = _joinList.First();
	    while ( pJCO )
	    {
		if ( (Chain)(*pJCO)->getName() == (Chain)(*pCO)->getName())
		    notFound=false;
		pJCO = _joinList.Next();
	    }
	    
	    if ( notFound )
	    {
		_joinList.Insert(*pCO);
	    }
	
	    pCO = _coList.Next();
	}
    }
        
    CegoContentObject** pJCO = _joinList.First();
    while ( pJCO )
    {
	if ( (*pJCO)->getType() != CegoObject::JOIN )
	{
	    CegoField *pF =  (*pJCO)->getSchema().First();
	    while (pF)
	    {
		pF->setTableName((*pJCO)->getTabName());
		pF->setTableAlias((*pJCO)->getName());
		pF = (*pJCO)->getSchema().Next();
	    }
	}

	pJCO = _joinList.Next();
    }
}

void CegoSelect::buildJoinConditions()
{

    if ( _pPred )
	_pPred->setCheckedRec(false);

    CegoPredDesc **pP = _conjunctionList.First();
    while ( pP )
    {
	(*pP)->setCheckedRec(false);
	pP = _conjunctionList.Next();
    }

    int i = 0;
    while ( i < TABMNG_MAXJOINLEVEL )
    {
	_attrCondFlag[i]=false;
	i++;
    }

    i = 0;
    ListT<CegoField> availJoinFields;

    if ( _pParentJoinBuf )
    {
	int i=0;
	while ( _pParentJoinBuf[i] )
	{
	    availJoinFields += *_pParentJoinBuf[i];
	    i++;
	}
    }

    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Checking view condition ..."));

    if ( i >= TABMNG_MAXJOINLEVEL )
	throw Exception(EXLOC, "Join levels exceeded");

    if ( ! _conjunctionList.isEmpty() )
    {
	
	// _pDBMng->log(_modId, Logger::DEBUG, Chain("Checking conjunctionlist ..."));
	while ( i < _joinList.Size() )
	{	    
	    _attrCondFlag[i] = false;
	    
	    CegoPredDesc **pPred = _conjunctionList.First();
	
	    // _pDBMng->log(_modId, Logger::DEBUG, Chain("Checking next predicate ..."));

	    bool notFound = true;
	    while ( pPred && notFound )
	    {
		// _pDBMng->log(_modId, Logger::DEBUG, Chain("Checking predicate ") + (*pPred)->toChain());
		if ( (*pPred)->isChecked() == false )
		{	
		    CegoAttrCond ac;
#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Getting attribute cond for ") + (*pPred)->toChain());
#endif

		    // cout << "Schema size on level " << i << " is " << _joinList[i]->getSchema().Size() << endl;
		
		    CegoQueryHelper::AttrCondMatch m = CegoQueryHelper::checkAttrCond(ac, *pPred, _joinList[i]->getSchema(), &availJoinFields, 1, _pBlock);		    		    
		   
		    if ( m != CegoQueryHelper::INAPP )
		    {
			bool predChecked = false;
			
			if ( m == CegoQueryHelper::COMPLETE )
			    predChecked = true;

#ifdef CGDEBUG
			_pDBMng->log(_modId, Logger::DEBUG, Chain("Detecting valid attr cond [ ") + ac.toChain() + Chain(" ] on level ") + Chain(i));
#endif 			
			   
			if ( _attrCondFlag[i] == false )
			{
			    _attrCondFlag[i] = true;
			    _attrCond[i] = ac;
			}
			else
			{
			    _attrCond[i] = _attrCond[i] + ac;
			}
			
			if ( predChecked == true )
			    (*pPred)->setChecked(predChecked);
			
			// TODO :  do we have to set up a pred list ?
			// _attrPred[i] = *pPred;
			    
			pPred = _conjunctionList.Next();
		    			
		    }
		    else
		    {
#ifdef CGDEBUG
			_pDBMng->log(_modId, Logger::DEBUG, Chain("Inappropriate predicate on level ") + Chain(i));
#endif 			
			pPred = _conjunctionList.Next();
		    }		    
		}
		else
		{
		    pPred = _conjunctionList.Next();
		}		
	    }

	    availJoinFields += _joinList[i]->getSchema();
	    i++;

	    if ( i >= TABMNG_MAXJOINLEVEL )
		throw Exception(EXLOC, "Join levels exceeded");
	}
    }
}

void CegoSelect::buildJoinRefs()
{
    // provide all selected fields in _joinFields

    int i=0;
    CegoContentObject** pCO = _joinList.First();
    
    while ( pCO )
    {       	
	// check for all subCO
	CegoContentObject **pSubCO = (*pCO)->getSubCOList().First();
	while ( pSubCO )
	{	    
	    if ( (*pSubCO)->getType() != CegoObject::JOIN )
	    {
		CegoField *pF =  (*pSubCO)->getSchema().First();
		while (pF)
		{
		    pF->setTableName((*pSubCO)->getTabName());
		    pF->setTableAlias((*pSubCO)->getName());
		    pF = (*pSubCO)->getSchema().Next();
		}
	    }
	    	    	    
	    ListT<CegoField> outSchema;

	    // in case of select * get all available fields
	    if ( _exprList.isEmpty() )
	    {
		filterRefs( (*pSubCO)->getSchema(), outSchema );
		outSchema = (*pSubCO)->getSchema();
	    }
	    else
	    {
		filterRefs( (*pSubCO)->getSchema(), outSchema );
		(*pSubCO)->setSchema(outSchema);
	    }

	    // sort outSchema

	    if ( _joinBuf[i] )
		_joinBuf[i]->Empty();
	    else
		_joinBuf[i] = new ListT<CegoField>;
	    
	    CegoField *pF = outSchema.First();
	    int maxId=0;
	    while ( pF )
	    {
		if ( pF->getId() > maxId )
		    maxId = pF->getId();
		pF = outSchema.Next();
	    }
	    for ( int id=0; id<=maxId; id++ )
	    {
		CegoField *pF = outSchema.First();
		while ( pF )
		{
		    if ( pF->getId() == id )
		    {
			_joinBuf[i]->Insert(*pF);
			pF = 0;
		    }
		    else
		    {
			pF = outSchema.Next();
		    }
		}
	    }
	    
	    // _joinBuf[i] = outSchema;
	    i++;
	    
	    _joinFields += (*pSubCO)->getSchema();
	    
	    pSubCO = (*pCO)->getSubCOList().Next();
	}

	pCO = _joinList.Next();
    }
}

void CegoSelect::filterRefs(ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{
    if ( _exprList.isEmpty() )
    {
	outSchema = inSchema;
    }
    else
    {
	CegoExpr** pExpr = _exprList.First();
	while (pExpr)
	{
	    filterExprRef(*pExpr, inSchema, outSchema);
	    pExpr = _exprList.Next();
	}
        
	if ( _pGroupList == 0 && _selectMode == AGGREGATION )
	{
	    pExpr = _exprList.First();
	    while (pExpr)
	    {
		// expressions are just allowed containing an aggregation or constant values
		if (  (*pExpr)->getAggregationList().Size() == 0 && (*pExpr)->getAttrRefList().Size() > 0 )
		{
		    Chain msg = Chain("Invalid expression ") + (*pExpr)->toChain() + Chain(" for aggregation");
		    throw Exception(EXLOC, msg);
		}
		pExpr = _exprList.Next();
	    }
	}
    }
    
    ListT<CegoField> tmpSchema[TABMNG_MAXTMPSCHEMA];

    int tmpidx=0;

    if (_pPred)
    {
	filterPredRef(_pPred, inSchema, tmpSchema[tmpidx]);
	tmpidx++;
    }

    CegoPredDesc** pJoinPred = _joinPredList.First();
    while ( pJoinPred )
    {
	filterPredRef(*pJoinPred, inSchema, tmpSchema[tmpidx]);
	tmpidx++;
	pJoinPred = _joinPredList.Next();
    }
        
    if (_pGroupList)
    {
	CegoAttrDesc** pAD = _pGroupList->First();
	while ( pAD )
	{
	    filterAttrRef(*pAD, inSchema, tmpSchema[tmpidx]);
	    pAD = _pGroupList->Next();
	}
	if ( _pHaving )
	{
	    filterExprRef(_pHaving->getAggExpr(), inSchema, tmpSchema[tmpidx]);
	    filterExprRef(_pHaving->getExpr(), inSchema, tmpSchema[tmpidx]);
	}
	tmpidx++;
    }

    if (_pNativeOrderList)
    {
	CegoExpr** pExpr = _pNativeOrderList->First();
	while ( pExpr )
	{	   
	    ListT<CegoAttrDesc*> attrRefList = (*pExpr)->getAttrRefList();
		    
	    CegoAttrDesc** pAttrRef = attrRefList.First();
	    
	    while ( pAttrRef )
	    {
		filterAttrRef(*pAttrRef, inSchema, tmpSchema[tmpidx]);
		pAttrRef = attrRefList.Next();
	    }
	    pExpr = _pNativeOrderList->Next();
	}
	tmpidx++;
    }

    // merge tmpSchemas to outSchema

    for ( int i=0; i < tmpidx; i++)
    {
	CegoField *pF = tmpSchema[i].First();
	while( pF )
	{
	    if ( outSchema.Find(*pF) == 0 )
		outSchema.Insert(*pF);
	    pF = tmpSchema[i].Next();
	}
    }    
}

void CegoSelect::filterFacRef(CegoFactor *pFac, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{
    if (pFac->getType() == CegoFactor::EXPR)
    {
	filterExprRef(pFac->getExpr(), inSchema, outSchema);
    }
    else if ( pFac->getType() == CegoFactor::ATTR)
    {
	filterAttrRef(pFac->getAttr(), inSchema, outSchema);
    }
    else if ( pFac->getType() == CegoFactor::FUNCTION )
    {
	CegoFunction* pFunc = pFac->getFunction();

	CegoExpr **pFuncArg = pFunc->getExprList().First();
	while ( pFuncArg )
	{
	    filterExprRef(*pFuncArg, inSchema, outSchema);
	    pFuncArg = pFunc->getExprList().Next();
	}
    }
    else if (  pFac->getType() == CegoFactor::AGGREGATION )
    {
	_selectMode = AGGREGATION;

	if ( pFac->getAggregation()->getExpr() )
	    filterExprRef(pFac->getAggregation()->getExpr(), inSchema, outSchema);
	
	pFac->getAggregation()->setAggregationId(_nextAid);
	_nextAid++;
    }
    else if (  pFac->getType() == CegoFactor::QUERY )
    {
	pFac->getSelect()->prepare();
	ListT<CegoAttrDesc*> attrRefList = pFac->getAttrRefList();
	CegoAttrDesc** pAttr = attrRefList.First();
	while ( pAttr )
	{
	    filterAttrRef( *pAttr, inSchema, outSchema );
	    pAttr = attrRefList.Next();
	}
    }
    else if (  pFac->getType() == CegoFactor::CASECOND )
    {
	CegoCaseCond *pCaseCond = pFac->getCaseCond();
 
	ListT<CegoField> outSchema2;
	CegoPredDesc **pPred = pCaseCond->getPredList().First();
	while ( pPred )
	{
	    filterPredRef(*pPred, inSchema, outSchema2);
	    pPred = pCaseCond->getPredList().Next();
	}
	ListT<CegoField> outSchema3;
	CegoExpr **pExpr = pCaseCond->getExprList().First();
	while ( pExpr )
	{
	    filterExprRef(*pExpr, inSchema, outSchema3);
	    pExpr = pCaseCond->getExprList().Next();
	}
	ListT<CegoField> outSchema4;
	if ( pCaseCond->getElseExpr() )
	    filterExprRef(pCaseCond->getElseExpr(), inSchema, outSchema4);

	// merge outSchemas
	CegoField *pF = outSchema2.First();
	while( pF )
	{
	    if ( outSchema.Find(*pF) == 0 )
		outSchema.Insert(*pF);
	    pF = outSchema2.Next();
	}
	pF = outSchema3.First();
	while( pF )
	{
	    if ( outSchema.Find(*pF) == 0 )
		outSchema.Insert(*pF);
	    pF = outSchema3.Next();
	}
	pF = outSchema4.First();
	while( pF )
	{
	    if ( outSchema.Find(*pF) == 0 )
		outSchema.Insert(*pF);
	    pF = outSchema4.Next();
	}
    }
}

void CegoSelect::filterExprRef(CegoExpr *pExpr, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{
    switch ( pExpr->getType() )
    {
    case CegoExpr::ADD:
    case CegoExpr::SUB:
    case CegoExpr::CONCAT:
    {
	filterExprRef(pExpr->getExpr(), inSchema, outSchema);	
	filterTermRef(pExpr->getTerm(), inSchema, outSchema);
	break;
    }
    case CegoExpr::TERM:
    {
	filterTermRef(pExpr->getTerm(), inSchema, outSchema);
	break;
    }
    }
}

void CegoSelect::filterTermRef(CegoTerm *pTerm, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{   

    switch ( pTerm->getType() )
    {
    case CegoTerm::MUL:
    case CegoTerm::DIV:
    {
	filterTermRef(pTerm->getTerm(), inSchema, outSchema);	
	filterFacRef(pTerm->getFactor(), inSchema, outSchema);
	break;
    }
    case CegoTerm::FACTOR:
    {	
	filterFacRef(pTerm->getFactor(), inSchema, outSchema);
	break;
    }      
    }
}

void CegoSelect::filterAttrRef(CegoAttrDesc* pAttrDesc, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{
    CegoField* pF = inSchema.First();
    while ( pF )
    {
	if ( ( (Chain)pAttrDesc->getAttrName() == (Chain)pF->getAttrName() 
	       || (Chain)pAttrDesc->getAttrName() == Chain(SELECTION_WILDCARD))
	     && (Chain)pAttrDesc->getTableName() == (Chain)pF->getTableAlias())
	{
	    if ( outSchema.Find(*pF) == 0 )
		outSchema.Insert(*pF);
	    // cout << "Setting " << pAttrDesc->toChain() << " valid " << endl;
	    pAttrDesc->setValid(true);
	}
	
	pF = inSchema.Next();
    }
}

void CegoSelect::filterPredRef(CegoPredDesc *pP, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{    
    if ( pP->getMode() == CegoPredDesc::CONDITION )
    {
	filterCondRef(pP->getCondition(), inSchema, outSchema);
    }
    else if ( pP->getMode() == CegoPredDesc::NOTPRED )
    {
	filterPredRef(pP->getNotPred(), inSchema, outSchema);
    }
    else
    {
	if ( pP->getMode() == CegoPredDesc::EXPRCOMP 
	     || pP->getMode() == CegoPredDesc::ISLIKE
	     || pP->getMode() == CegoPredDesc::ISNOTLIKE
	     || pP->getMode() == CegoPredDesc::INSUB
	     || pP->getMode() == CegoPredDesc::NOTINSUB
	     || pP->getMode() == CegoPredDesc::NULLCOMP
	     || pP->getMode() == CegoPredDesc::NOTNULLCOMP
	     || pP->getMode() == CegoPredDesc::BETWEEN)
	{
	    filterExprRef(pP->getExpr1(), inSchema, outSchema);

	    if ( pP->getMode() == CegoPredDesc::EXPRCOMP 
		 || pP->getMode() == CegoPredDesc::BETWEEN )
	    {
		filterExprRef(pP->getExpr2(), inSchema, outSchema);
	    }
	    if ( pP->getMode() == CegoPredDesc::BETWEEN )
	    {
		filterExprRef(pP->getExpr3(), inSchema, outSchema);
	    }	   	    
	}
	
	if ( pP->getMode() == CegoPredDesc::EXISTSCOMP
	     || pP->getMode() == CegoPredDesc::INSUB
	     || pP->getMode() == CegoPredDesc::NOTINSUB)
	{
	    ListT<CegoSelect*> queryList;
	    pP->getSelectQueryList(queryList);

	    CegoSelect **pSelect = queryList.First();
	    while ( pSelect )
	    {
		ListT<CegoAttrDesc*> attrRefList = (*pSelect)->getAttrRefList();
		CegoAttrDesc **pAD = attrRefList.First();
		while ( pAD )
		{
		    filterAttrRef((*pAD), inSchema, outSchema);
		    pAD = attrRefList.Next();
		}		
		pSelect = queryList.Next();
	    }
	}
    }
}

void CegoSelect::filterCondRef(CegoCondDesc* pC, ListT<CegoField>& inSchema, ListT<CegoField>& outSchema)
{
    if (pC->Left())
    {
	filterPredRef(pC->Left(), inSchema, outSchema);
    }
    if (pC->Right())
    {
	filterPredRef(pC->Right(), inSchema, outSchema);
    }
}

void CegoSelect::getPredObjectList(CegoPredDesc *pPred, ListT<CegoContentObject*>& objectList)
{
    if ( pPred->getCondition() )
    {
	getCondObjectList(pPred->getCondition(), objectList);
    }
    else
    {	
	if ( pPred->getExpr1() )
	{
	    ListT<CegoAttrDesc*> attrRefList = pPred->getExpr1()->getAttrRefList();

	    if ( pPred->getExpr2() )
	    {
		attrRefList += pPred->getExpr2()->getAttrRefList();
	    }

	    CegoContentObject** pCO = _coList.First();	
	    while (pCO)
	    {	
		// check for all subCO
		CegoContentObject **pSubCO = (*pCO)->getSubCOList().First();
		while ( pSubCO )
		{		
		    CegoAttrDesc **pAttr = attrRefList.First();
		    
		    while ( pAttr )
		    {	
			if ( (Chain)(*pAttr)->getTableName() == (Chain)(*pSubCO)->getTabName() 
			     || (Chain)(*pAttr)->getTableName() == (Chain)(*pSubCO)->getName())	    
			{			    
			    // check if content object already in object list
			    bool notFound=true;
			    CegoContentObject **pCCO = objectList.First();
			    while ( pCCO && notFound )
			    {
				if ( (Chain)(*pCCO)->getName() == (Chain)(*pCO)->getName())
				    notFound=false;
				else
				    pCCO = objectList.Next();
			    }
			    if ( notFound )
			    {
				objectList.Insert(*pCO);
			    }	      
			}
			pAttr = attrRefList.Next();	    
		    }

		    pSubCO = (*pCO)->getSubCOList().Next();
		    
		}
		pCO = _coList.Next();	    
	    }
	}
    }
}

void CegoSelect::getCondObjectList(CegoCondDesc* pC, ListT<CegoContentObject*>& objectList)
{
    if (pC->Left())
    {
	getPredObjectList(pC->Left(), objectList);
    }
    if (pC->Right())
    {
	getPredObjectList(pC->Right(), objectList);
    }
}

// the evalReferences method resolves all occuring attribute references to full   
// table information for later processing. In these terms, the defined selection list
// but also all refrences in the condition clause are resolved.

int CegoSelect::evalExtTableReferences(CegoContentObject* pCO, const ListT<CegoField>& fl)
{    
    int refCount = evalTableReferences(pCO, fl);
    if ( refCount > 0 )
    {
	// disable cache
	_cacheEnabled = false;
    }
    _extRefCount += refCount;
    return refCount;
}

int CegoSelect::evalTableReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;

    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {	
	refCount += (*pExpr)->evalReferences(pCO, fl);
	pExpr = _exprList.Next();
    }
    
    if (_pPred)
    {
	refCount += _pPred->evalReferences(pCO, fl);
    }

    CegoPredDesc **pJoinPred = _joinPredList.First();
    while (pJoinPred)
    {
	refCount += (*pJoinPred)->evalReferences(pCO, fl);
	pJoinPred = _joinPredList.Next();
    }
    
    CegoPredDesc **pP = _viewConjunctionList.First();
    while ( pP )
    {      
	refCount += (*pP)->evalReferences(pCO, fl);
	pP = _viewConjunctionList.Next();
    }
                
    if (_pGroupList)
    {
	CegoAttrDesc **pAD = _pGroupList->First();
	while ( pAD )
	{
	    refCount += (*pAD)->evalReferences(pCO, fl);
	    pAD = _pGroupList->Next();
	}
    }
    
    if (_pNativeOrderList)
    {	
	CegoExpr **pExpr = _pNativeOrderList->First();	
	while ( pExpr )
	{
	    refCount += (*pExpr)->evalReferences(pCO, fl);	
	    pExpr = _pNativeOrderList->Next();
	}	    
    }
    
    if ( _pHaving )
    {
	refCount += _pHaving->getExpr()->evalReferences(pCO, fl);
	refCount += _pHaving->getAggExpr()->evalReferences(pCO, fl);	
    }

    return refCount;
}

void CegoSelect::evalReferences()
{
    _joinPredList.Empty();
    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	if ( (*pCO)->getType() == CegoObject::JOIN )
	{
	    ((CegoJoinObject*)(*pCO))->getPredList(_joinPredList);
	}
	pCO = _coList.Next();
    }

    pCO = _coList.First();
    while (pCO)
    {	      
	// uncompiled objects are detected here, just treated, if tablemanager is set up
	if ( (*pCO)->getType() == CegoObject::UNDEFINED && _pGTM != 0 )
	{
	    Chain msg = Chain("Invalid object ") + (*pCO)->getName();
	    throw Exception(EXLOC, msg);	    
	}

	ListT<CegoField> schema;
	schema = (*pCO)->getSchema();
	
	evalTableReferences(*pCO, schema);
	
	pCO = _coList.Next();
    }
}

void CegoSelect::evalSelection(ListT<CegoExpr*>& exprList, ListT<CegoField>** pJoinBuf, bool isAgg, ListT<CegoField>& ol) 
{
    if ( exprList.isEmpty() )
    {
	int i=0;
	while ( pJoinBuf[i] )
	{
	    CegoField* pF = pJoinBuf[i]->First();
	    while ( pF ) 
	    {				
		ol.Insert(*pF);
		pF = pJoinBuf[i]->Next();
	    }
	    i++;
	}
	return;
    }

    if ( isAgg )
    {
	CegoExpr** pExpr = exprList.First();
	while ( pExpr )
	{	    
	    (*pExpr)->setFieldListArray( pJoinBuf );
	    
	    ListT<CegoField> fl = (*pExpr)->getFieldList();

	    CegoField *pF = fl.First();
	    while ( pF )
	    {
		int i =0;	       
		CegoField *pJF = 0;
		while ( pJoinBuf[i] && pJF == 0 )
		{
		    pJF = pJoinBuf[i]->Find(*pF);
		    if ( pJF )
		    {
			pF->setValue(pJF->getValue());			
		    }	       
		    i++;
		}
		
		pF = fl.Next();
	    }
	    
	    ol = ol + fl;

	    pExpr = exprList.Next();
	}
    }
    else
    {
	CegoExpr** pExpr = exprList.First();
	while ( pExpr )
	{
	    CegoAttrDesc* pAttrDesc = (*pExpr)->checkAttr();

	    if ( pAttrDesc )
	    {
		if ( pAttrDesc->getAttrName() == Chain(SELECTION_WILDCARD))
		{	    
		    int i=0;
		    while ( pJoinBuf[i] )
		    {
			CegoField *pF = pJoinBuf[i]->First();

			while ( pF )
			{
			    if ( (Chain)pF->getTableName() ==  (Chain)pAttrDesc->getTableName()
				 || (Chain)pF->getTableAlias() ==  (Chain)pAttrDesc->getTableName() )
			    {			    
				ol.Insert(*pF);
			    }
			    pF = pJoinBuf[i]->Next();
			}
			i++;
		    }
		}
		else
		{
		    (*pExpr)->setFieldListArray( pJoinBuf );
		    CegoField f = (*pExpr)->evalField();
		    f.setValue ( (*pExpr)->evalFieldValue() ); 
		    ol.Insert ( f );
		}
	    }	
	    else
	    {
		(*pExpr)->setFieldListArray( pJoinBuf );
		CegoField f = (*pExpr)->evalField();
		f.setValue ( (*pExpr)->evalFieldValue() );
		ol.Insert ( f );
	    }
	    pExpr = exprList.Next();
	}
    }     
    return;   
}

void CegoSelect::encode(char *buf)
{
    char* pBuf = (char*)buf;

    int size = 0;
    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	size += (*pExpr)->getEncodingLength();
	pExpr = _exprList.Next();
    }
    
    memcpy( pBuf, &size, sizeof(int));
    pBuf = pBuf + sizeof(int);

    pExpr = _exprList.First();
    while (pExpr)
    {
	(*pExpr)->encode(pBuf);
	pBuf = pBuf + (*pExpr)->getEncodingLength();
	pExpr = _exprList.Next();
    }

    size = 0;
    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	size += (*pCO)->getEntrySize();
	pCO = _coList.Next();
    }

    memcpy( pBuf, &size, sizeof(int));
    pBuf = pBuf + sizeof(int);

    pCO = _coList.First();
    while (pCO)
    {
	(*pCO)->encode(pBuf);
	pBuf = pBuf + (*pCO)->getEntrySize();
	pCO = _coList.Next();
    }

    size = 0;
    if (_pPred)
    {
	size = _pPred->getEncodingLength();
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);
	_pPred->encode(pBuf);
	pBuf = pBuf + size;
    }
    else
    {
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);
    }
    
    size = 0;
    if (_pGroupList)
    {
	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    size += (*pAttrDesc)->getEncodingLength();
	    pAttrDesc = _pGroupList->Next();
	}

	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);

	pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    (*pAttrDesc)->encode(pBuf);
	    pBuf = pBuf + (*pAttrDesc)->getEncodingLength();
	    pAttrDesc = _pGroupList->Next();
	}

	size = 0;
	if ( _pHaving )
	{
	    _pHaving->encode(pBuf);
	    pBuf = pBuf + _pHaving->getEncodingLength();
	}
	else
	{
	    memcpy( pBuf, &size, sizeof(int));
	    pBuf = pBuf + sizeof(int);
	}
    }
    else
    {
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);	
    }
    size = 0;
    if ( _pOrderList )
    {
	CegoExpr **pExpr = _pOrderList->First();
	while ( pExpr )
	{
	    size += (*pExpr)->getEncodingLength();
	    pExpr = _pOrderList->Next();
	}
       
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);
	
	pExpr = _pOrderList->First();
	while (pExpr)
	{
	    (*pExpr)->encode(pBuf);
	    pBuf = pBuf + (*pExpr)->getEncodingLength();
	    pExpr = _pOrderList->Next();
	}
    }
    else
    {
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);	
    }

    size = 0;
    if ( _pUnionSelect )
    {
	size = _pUnionSelect->getEncodingLength();
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);

	_pUnionSelect->encode(pBuf);
	pBuf = pBuf + _pUnionSelect->getEncodingLength();
    }
    else
    {
	memcpy( pBuf, &size, sizeof(int));
	pBuf = pBuf + sizeof(int);
    }
}

void CegoSelect::decode(char *buf, CegoDistManager *pGTM, int tabSetId)
{
    char* pBuf = (char*)buf;
    
    int size;
    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    int i = 0;
    while ( i < size )
    {
	CegoExpr *pExpr = new CegoExpr(pBuf, pGTM, tabSetId);
	int len = pExpr->getEncodingLength();
	
	pBuf += len;
	i += len;
	
	_exprList.Insert( pExpr );	
    }

    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    i = 0;
    while ( i < size )
    {
	CegoContentObject *pCO;

	CegoObject obj;
	int size;
	obj.decodeBase(pBuf, size);   
		
	if ( obj.getType() == CegoObject::TABLE )
	{
	    pCO = new CegoTableObject();
	}
	else if ( obj.getType() == CegoObject::VIEW )
	{
	    pCO = new CegoViewObject();
	}
	else if ( obj.getType() == CegoObject::JOIN )
	{
	    pCO = new CegoJoinObject();
	}
	pCO->decode(pBuf);	    

	int len = pCO->getEntrySize();
	
	pBuf += len;
	i += len;
	
	_coList.Insert( pCO );	
    }

    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    if ( size > 0 )
    {
	_pPred = new CegoPredDesc(pBuf, pGTM, tabSetId);
	pBuf += _pPred->getEncodingLength();
    }
    else
    {
	_pPred = 0;
    }

    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    if ( size > 0 )
    {
	_pGroupList = new ListT<CegoAttrDesc*>;

	i = 0;
	while ( i < size )
	{
	    CegoAttrDesc* pAD = new CegoAttrDesc(pBuf);
	    int len = pAD->getEncodingLength();
	    
	    pBuf += len;
	    i += len;
	    
	    _pGroupList->Insert( pAD );	    
	}
	
	memcpy( &size, pBuf, sizeof(int));
	pBuf += sizeof(int);
    
	if ( size > 0 )
	{
	    _pHaving = new CegoHavingDesc(buf, pGTM, tabSetId);
	    pBuf += _pHaving->getEncodingLength();
	}
	else
	{
	    _pHaving = 0;
	}
    }
    else
    {
	_pGroupList = 0;
	_pHaving = 0;
    }

    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    if ( size > 0 )
    {
	_pOrderList = new ListT<CegoExpr*>;

	int i = 0;
	while ( i < size )
	{
	    CegoExpr *pExpr = new CegoExpr(pBuf, pGTM, tabSetId);
	    int len = pExpr->getEncodingLength();
	    
	    pBuf += len;
	    i += len;
	    
	    _pOrderList->Insert( pExpr );
	}
    }
    else
    {
	_pOrderList = 0;
    }

    memcpy( &size, pBuf, sizeof(int));
    pBuf += sizeof(int);
    
    if ( size > 0 )
    {
	_pUnionSelect = new CegoSelect(pBuf, pGTM, tabSetId);
    }
    else
    {
	_pUnionSelect = 0;
    }
}

int CegoSelect::getEncodingLength() const
{    
    int len=0;

    len += sizeof(int);
    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	len += (*pExpr)->getEncodingLength();
	pExpr = _exprList.Next();
    }
    
    len += sizeof(int);
    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	len += (*pCO)->getEntrySize();
	pCO = _coList.Next();
    }

    len += sizeof(int);

    if (_pPred)
    {
	len += _pPred->getEncodingLength();
    }

    len += sizeof(int);

    if (_pGroupList)
    {

	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    len += (*pAttrDesc)->getEncodingLength();
	    pAttrDesc = _pGroupList->Next();
	}

	len += sizeof(int);

	if ( _pHaving )
	{
	    len += _pHaving->getEncodingLength();
	}
    }

    len += sizeof(int);

    if ( _pOrderList )
    {	
	CegoExpr **pExpr = _pOrderList->First();
	while ( pExpr )
	{
	    len += (*pExpr)->getEncodingLength();
	    pExpr = _pOrderList->Next();
	}
    }

    len += sizeof(int);

    if (_pUnionSelect)
    {
	len += _pUnionSelect->getEncodingLength();
    }

    return len;
}

void CegoSelect::setViewCond(const CegoAttrCond& attrCond)
{
    CegoPredDesc **pP = _viewConjunctionList.First();
    while ( pP )
    {      
	delete (*pP);
	pP = _viewConjunctionList.Next();
    }
    _viewConjunctionList.Empty();
    
    if ( attrCond.numComp() > 0 )
	attrCond.asConjunctionList(_exprList, _viewConjunctionList);

    cleanUp();
}

Chain CegoSelect::toChain(const Chain& indent) const
{
    Chain s;

    s = indent + Chain("select");

    if ( _isDistinct && _selectMode != AGGREGATION )
	s += Chain(" distinct");
    
    if ( _exprList.Size() == 0 )
    {
	s += Chain(" *\n");
    }
    else
    {
	s += Chain("\n");
	CegoExpr** pExpr = _exprList.First();
	while (pExpr)
	{
	    s += (*pExpr)->toChain(indent + Chain(" "));
	    if ( (*pExpr)->getAlias() != Chain() )
	    {
		s += Chain(" as ") + (*pExpr)->getAlias();
	    } 
	    pExpr = _exprList.Next();
	    if ( pExpr )
		s += Chain(",\n");
	}
    }

    s += Chain("\n") + indent + Chain("from\n");

    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	s += indent + Chain(" ") + (*pCO)->toChain();
	pCO = _coList.Next();
	if ( pCO )
	    s += Chain(",\n");
    }
    
    if (_pPred)
    {
	s += Chain("\n");
	s += indent + Chain("where\n");
	s += indent + _pPred->toChain(indent + Chain(" "));
	s += Chain("\n");
    }

    if (_pGroupList)
    {
	s += Chain("\n");
	s += indent + Chain("group by\n") + indent;
	
	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    s += indent + Chain(" ") + (*pAttrDesc)->toChain();
	    pAttrDesc = _pGroupList->Next();
	    if ( pAttrDesc )
		s += ",";
	}
	
	if ( _pHaving )
	{
	    s += Chain("\n") + indent + Chain("having\n");
	    s += indent + _pHaving->toChain();
	}
    }
    if ( _pNativeOrderList )
    {
	s += Chain("\n");
	s += indent + Chain("order by\n") + indent;
	
	CegoExpr **pExpr = _pNativeOrderList->First();
	CegoOrderNode::Ordering *pOrdering = _pNativeOrderOptList->First();
	while ( pExpr && pOrdering )
	{
	    s += (*pExpr)->toChain(indent + Chain(" "));

	    if ( *pOrdering == CegoOrderNode::ASC )
		s += Chain(" asc");
	    else
		s += Chain(" desc");
	    
	    pExpr = _pNativeOrderList->Next();
	    if ( pExpr )
		s += ",";
	}
    }

    if ( _rowLimit > 0 )
    {
	s += Chain(" rowlimit ") + Chain(_rowLimit);
    }

    if ( _pUnionSelect )
    {
	s += Chain("\n");
	s += indent + Chain("union all\n") + indent;
	s += _pUnionSelect->toChain(indent);
    }
    return s;
}

Chain CegoSelect::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatSelect(_selectMode, _isDistinct, _exprList, _coList, _pPred, _pGroupList, _pHaving, _pNativeOrderList, _rowLimit, _pUnionSelect);
}

Element* CegoSelect::toElement() const
{
    Element *pSelectElement = new Element( XML_SELECT_ELEMENT );

    Element *pSelectionElement = new Element( XML_SELECTION_ELEMENT );
    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	pSelectionElement->addContent( (*pExpr)->toElement() );	
	pExpr = _exprList.Next();
    }

    pSelectElement->addContent(pSelectionElement);
    
    Element *pCOListElement = new Element( XML_COLIST_ELEMENT );
    CegoContentObject** pCO = _coList.First();
    while (pCO)
    {
	pCOListElement->addContent( (*pCO)->getElement() );
	pCO = _coList.Next();
    }
    
    pSelectElement->addContent(pCOListElement);

    if (_pPred)
    {
	pSelectElement->addContent( _pPred->toElement() );
    }

    if (_pGroupList)
    {
	Element *pGroupingElement = new Element( XML_GROUPING_ELEMENT );

	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    pGroupingElement->addContent( (*pAttrDesc)->toElement() );
	    pAttrDesc = _pGroupList->Next();
	}
	
	if ( _pHaving )
	{
	    pGroupingElement->addContent( _pHaving->toElement() );
	}

	pSelectElement->addContent( pGroupingElement ) ;
    }
    if ( _pNativeOrderList )
    {
	Element *pOrderingElement = new Element( XML_ORDERING_ELEMENT );
	
	CegoExpr **pExpr = _pNativeOrderList->First();
	while ( pExpr )
	{
	    pOrderingElement->addContent( (*pExpr)->toElement() );
	    pExpr = _pNativeOrderList->Next();
	}
	
	pSelectElement->addContent( pOrderingElement ) ;
    }

    if ( _pUnionSelect )
    {
	pSelectElement->addContent( _pUnionSelect->toElement() );
    }

    return pSelectElement;
}

void CegoSelect::fromElement(Element *pSelectElement, CegoDistManager* pGTM)
{
    if ( _pPred )
	delete _pPred;

    _pPred = 0;

    CegoContentObject **pCO = _coList.First();
    while ( pCO )
    {
	delete (*pCO);
	pCO = _coList.Next();
    }
    _coList.Empty();

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr )
    {
	delete (*pExpr);
	pExpr = _exprList.Next();
    }
    _exprList.Empty();
    
    if ( _pNativeOrderList )
    {
	CegoExpr **pExpr = _pNativeOrderList->First();
	while ( pExpr )
	{
	    delete (*pExpr);
	    pExpr = _pNativeOrderList->Next();
	}
	delete _pNativeOrderList;
	_pNativeOrderList = 0;
    }
   
    if ( _pGroupList )
    {
	if ( _pGroupCursor )
	    delete _pGroupCursor;

	if ( _tabSetId != 0 && _pGroupSpace )
	{	    
	    delete _pGroupSpace;
	    _pGroupSpace=0;
	}

	delete _pGroupList;
	_pGroupList = 0;
	
    }
    if ( _pHaving )
    {
	delete _pHaving;
	_pHaving = 0;
    }

    ListT<Element*> sl = pSelectElement->getChildren(XML_SELECTION_ELEMENT);
    Element **pSE = sl.First();
    if ( pSE )
    {
	ListT<Element*> el = pSelectElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	while ( pEE )
	{
	    _exprList.Insert( new CegoExpr(*pEE, pGTM) );
	    pEE = el.Next();
	}
    }

    ListT<Element*> tl = pSelectElement->getChildren(XML_COLIST_ELEMENT);
    Element **pTE = tl.First();
    if ( pTE )
    {
	ListT<Element*> tdl = pSelectElement->getChildren(XML_CODESC_ELEMENT);
	Element **pTDE = tdl.First();
	while ( pTDE )
	{
	    Chain objType = (*pTDE)->getAttributeValue(XML_OBJTYPE_ATTR);

	    CegoContentObject *pCO;

	    if ( objType == Chain(XML_TABOBJ_VALUE) )
	    {
		pCO = new CegoTableObject();
	    }
	    else if ( objType == Chain(XML_VIEWOBJ_VALUE) )
	    {
		pCO = new CegoViewObject();	
	    }
	    else if ( objType == Chain(XML_JOINOBJ_VALUE) )
	    {
		pCO = new CegoJoinObject();
	    }
	    pCO->putElement(*pTDE);
	    
	    _coList.Insert( pCO);
	    pTDE = tdl.Next();
	}
    }

    ListT<Element*> pl = pSelectElement->getChildren(XML_PRED_ELEMENT);
    Element **pPE = pl.First();
    if ( pPE )
    {
	_pPred = new CegoPredDesc(*pPE, pGTM);
    }

    ListT<Element*> gl = pSelectElement->getChildren(XML_GROUPING_ELEMENT);
    Element **pGE = gl.First();
    if ( pGE )
    {
	_pGroupList = new ListT<CegoAttrDesc*>;

	ListT<Element*> adl = (*pGE)->getChildren(XML_ATTRDESC_ELEMENT);
	Element **pADE = adl.First();
	while ( pADE )
	{
	    CegoAttrDesc* pAD = new CegoAttrDesc(*pADE);
	    _pGroupList->Insert( pAD );
	    pADE = adl.Next();
	}

	ListT<Element*> hl = (*pGE)->getChildren(XML_HAVING_ELEMENT);
	Element **pHE = hl.First();
	if ( pHE )
	{
	    _pHaving = new CegoHavingDesc(*pHE, pGTM);
	}
    }

    ListT<Element*> ol = pSelectElement->getChildren(XML_ORDERING_ELEMENT);
    Element **pOE = ol.First();
    if ( pOE )
    {
	_pNativeOrderList = new ListT<CegoExpr*>;

	ListT<Element*> el = (*pOE)->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	while ( pEE )
	{
	    _pNativeOrderList->Insert( new CegoExpr(*pEE, pGTM) );
	    pEE = el.Next();
	}
    }

    ListT<Element*> ul = pSelectElement->getChildren(XML_SELECT_ELEMENT);
    Element **pUE = ul.First();
    if ( pUE )
    {
	_pUnionSelect = new CegoSelect(*pUE, pGTM);
    }
}

ListT<CegoAttrDesc*> CegoSelect::getAttrRefList() const
{
    ListT<CegoAttrDesc*> attrList;

    CegoContentObject** pCO = _coList.First();
    while ( pCO )
    {	
	if ( (*pCO)->getType() == CegoObject::JOIN )
	{	
	    ListT<CegoPredDesc*> predList;
	    ((CegoJoinObject*)(*pCO))->getPredList(predList);
	    
	    CegoPredDesc **pPred = predList.First();
	    while (pPred)
	    {
		attrList += (*pPred)->getAttrRefList();
		pPred = predList.Next();
	    }
	}
	pCO = _coList.Next();
    }

    CegoExpr** pExpr = _exprList.First();
    while (pExpr)
    {
	attrList += (*pExpr)->getAttrRefList();
	pExpr = _exprList.Next();
    }

    if (_pPred)
    {
	attrList += _pPred->getAttrRefList();
    }

    if (_pGroupList)
    {    
	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    attrList.Insert(*pAttrDesc);
	    pAttrDesc = _pGroupList->Next();
	}
	
	if ( _pHaving )
	{
	    attrList += _pHaving->getAttrRefList();
	}
    }
    
    if ( _pNativeOrderList )
    {
	CegoExpr **pExpr = _pNativeOrderList->First();
	while ( pExpr )
	{
	    attrList += (*pExpr)->getAttrRefList();
	    pExpr = _pNativeOrderList->Next();
	}	
    } 
    
    return attrList;
}

ListT<CegoField> CegoSelect::getFieldList() const
{
    ListT<CegoField> fl;

    ListT<CegoAttrDesc*> attrRefList = getAttrRefList();
    CegoAttrDesc **pAD = attrRefList.First();
    while ( pAD )
    {	
	fl.Insert(CegoField((*pAD)->getTableName(), (*pAD)->getAttrName()));
	pAD = attrRefList.Next();
    }
    
    return fl;
}

void CegoSelect::initAggregation()
{
    int nextAid=0;
    
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	ListT<CegoAggregation*> aggList = (*pExpr)->getAggregationList();

	CegoAggregation **pAgg = aggList.First();

	while ( pAgg )
	{	    
	    if ( (*pAgg)->getType() == CegoAggregation::COUNT )
	    {
		(*pAgg)->setFieldValue( CegoFieldValue(INT_TYPE, Chain("0")));
	    }
	    else
	    {
		CegoFieldValue fv;
		(*pAgg)->setFieldValue(fv);
	    }
	    (*pAgg)->setAggregationId(nextAid);
	    nextAid++;
	    
	    pAgg = aggList.Next();
	}
	pExpr = _exprList.Next();
    }   
}

void CegoSelect::aggregateTuple(ListT<CegoField>& aggTuple)
{
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {       
	CegoQueryHelper::aggregateTuple(aggTuple, *pExpr);
	pExpr = _exprList.Next();
    }    
}

void CegoSelect::evalAggregation(ListT<CegoField>& resultList)
{   
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {	
	ListT<CegoAggregation*> aggList = (*pExpr)->getAggregationList();

	CegoAggregation **pAgg = aggList.First();
	
	while ( pAgg )
	{	    
	    if ( (*pAgg)->getType() == CegoAggregation::AVG )
	    {
		CegoFieldValue avg;
		if ( _aggregationCount > 0 && (*pAgg)->getFieldValue().isNull() == false )
		{
		    CegoFieldValue fv((*pAgg)->getFieldValue().getType(), Chain(_aggregationCount));
		    avg = (CegoFieldValue)(*pAgg)->getFieldValue() / fv ; 
		}
		(*pAgg)->setFieldValue(avg);
	    }
	    pAgg = aggList.Next();
	}
	
	bool notFound=true;
	CegoFieldValue fv = (*pExpr)->evalFieldValue();
	
	// is the aggregation is associated with an alias ?
	// this may happen for aggregation in views
	CegoField* pF = resultList.First();
	while ( pF && notFound )
	{
	    if ( pF->getAttrName() != Chain() && pF->getAttrName() == (*pExpr)->getAlias() )
	    {
		pF->setValue(fv);
		notFound = false;
	    }
	    pF = resultList.Next();
	}
	
	// if native aggregation, we add it to list
	if (  notFound )
	    resultList.Insert(CegoField(fv));

	pExpr = _exprList.Next();
    }
}

void CegoSelect::makeOrder()
{
    if ( _isDistinct )
    {
	if ( _pNativeOrderList == 0 )
	{	    
	    _pOrderList = new ListT<CegoExpr*>;
	    _pOrderOptList = new ListT<CegoOrderNode::Ordering>;
	    
	    _dcount=0;
	    CegoExpr** pExpr = _exprList.First();
	    while (pExpr)
	    {
		if ( _selectMode == AGGREGATION )
		{
		    ListT<CegoAttrDesc*> attrList = (*pExpr)->getAttrRefList();
		    CegoAttrDesc **pAD = attrList.First();
		    while ( pAD )
		    {
			CegoExpr* pAggExp = new CegoExpr( 
			    new CegoTerm( 
				new CegoFactor( *pAD, true)));
			_pOrderList->Insert(pAggExp);
			_pOrderOptList->Insert(CegoOrderNode::ASC);
			
			_dcount++;
			pAD = attrList.Next();
		    }
		}
		else
		{
		    _pOrderList->Insert((*pExpr)->clone(true));
		    _pOrderOptList->Insert(CegoOrderNode::ASC);
		    _dcount++;
		}	
		pExpr = _exprList.Next();
	    }
	}
	else
	{
	    _pOrderList = new ListT<CegoExpr*>;
	    _pOrderOptList = new ListT<CegoOrderNode::Ordering>;	    
	    _dcount=0;
	    
	    CegoExpr** pExpr = _pNativeOrderList->First();
	    CegoOrderNode::Ordering* pOrd = _pNativeOrderOptList->First();
	    while ( pExpr && pOrd )
	    {
		_pOrderList->Insert((*pExpr)->clone(true));
		_pOrderOptList->Insert(*pOrd);
		_dcount++;
		pExpr = _pNativeOrderList->Next();
		pOrd = _pNativeOrderOptList->Next();	  
	    }	
	    
	    if ( _selectMode != AGGREGATION )
	    {	
		pExpr = _exprList.First();
		while ( pExpr )
		{
		    CegoAttrDesc* pAttrDesc1 = (*pExpr)->checkAttr();
		    if ( pAttrDesc1 )
		    {
			bool found=false;
			CegoExpr** pOrderExpr = _pNativeOrderList->First();
			while ( pOrderExpr && found == false )
			{
			    CegoAttrDesc* pAttrDesc2 = (*pOrderExpr)->checkAttr();
			    if ( pAttrDesc2 )
			    {
				if ( pAttrDesc1->getAttrName() == pAttrDesc2->getAttrName() )
				{
				    found = true;
				}
			    }
			    pOrderExpr = _pNativeOrderList->Next();
			}
			if ( found == false )
			{
			    _pOrderList->Insert((*pExpr)->clone(true));
			    _pOrderOptList->Insert(CegoOrderNode::ASC);
			    _dcount++;		    
			}		
		    }
		    else
		    {
			_pOrderList->Insert((*pExpr)->clone(true));
			_pOrderOptList->Insert(CegoOrderNode::ASC);
			_dcount++;
		    }
		    pExpr = _exprList.Next();
		}
	    }
	}
    }
    else
    {
	if ( _pNativeOrderList )
	{
	    _pOrderList = new ListT<CegoExpr*>;
	    _pOrderOptList = new ListT<CegoOrderNode::Ordering>;
	    
	    CegoExpr** pExpr = _pNativeOrderList->First();
	    CegoOrderNode::Ordering *pOrdering = _pNativeOrderOptList->First();
	    while (pExpr && pOrdering)
	    {
		_pOrderList->Insert((*pExpr)->clone(true));
		_pOrderOptList->Insert(*pOrdering);
		pExpr = _pNativeOrderList->Next();
		pOrdering = _pNativeOrderOptList->Next();
	    }
	}
    }
    return;
}

bool CegoSelect::matchFieldValue(const CegoFieldValue& fv1, const CegoFieldValue& fv2, CegoComparison comp)
{
    if (fv1 < fv2 && ( comp == LESS_THAN || comp == LESS_EQUAL_THAN || comp == NOT_EQUAL ))
    {       
	return true;
    }
    if (fv1 > fv2 && ( comp == MORE_THAN || comp == MORE_EQUAL_THAN || comp == NOT_EQUAL ))
    {
	return true;
    }    
    if ( fv1 == fv2 && ( comp == MORE_EQUAL_THAN || comp == LESS_EQUAL_THAN || comp == EQUAL) )
    {
	return true;
    }
    return false;
}

void CegoSelect::setAggregationValue(CegoExpr *pExpr, ListT<CegoField>& jfl)
{
    ListT<CegoAggregation*> aggList = pExpr->getAggregationList();
    
    CegoAggregation **pAgg = aggList.First();
    while ( pAgg )
    {
	CegoField *pF = jfl.First();
	while ( pF )
	{
	    if ( pF->getId() == (*pAgg)->getAggregationId() )
	    {
		(*pAgg)->setFieldValue(pF->getValue());
		pF = 0;
	    }
	    else
	    {
		pF = jfl.Next();
	    }
	}
	pAgg = aggList.Next();
    }
}

int CegoSelect::getExtRefCount() const
{
    return _extRefCount;
}

CegoSelect* CegoSelect::clone(bool isAttrRef)
{
    ListT<CegoExpr*> exprList;
    CegoExpr **pExpr = _exprList.First();
    while ( pExpr )
    {
	exprList.Insert((*pExpr)->clone(isAttrRef));
	pExpr = _exprList.Next();
    }
    
    CegoPredDesc *pPred = 0;
    if ( _pPred )
    {
	pPred = _pPred->clone(isAttrRef);
    }

    ListT<CegoAttrDesc*> *pGroupList = 0;

    if ( _pGroupList )
    {
	pGroupList = new ListT<CegoAttrDesc*>;
	CegoAttrDesc **pAttrDesc = _pGroupList->First();
	while ( pAttrDesc )
	{
	    if ( isAttrRef )
		pGroupList->Insert(*pAttrDesc);
	    else
		pGroupList->Insert((*pAttrDesc)->clone());
	    pAttrDesc = _pGroupList->Next();
	}
    }
        
    CegoHavingDesc *pHaving = 0;
    if ( _pHaving )
    {
	pHaving = _pHaving->clone(isAttrRef);
    }

    ListT<CegoExpr*> *pOrderList = 0;
    if ( _pOrderList )
    {
	pOrderList = new ListT<CegoExpr*>;
	CegoExpr **pExpr = _pOrderList->First();
	while ( pExpr )
	{
	    pOrderList->Insert((*pExpr)->clone(isAttrRef));
	    pExpr = _pOrderList->Next();
	}
    }
    
    ListT<CegoOrderNode::Ordering>* pOrderOptList = 0;
    if ( pOrderOptList )
    {
	pOrderOptList = new ListT<CegoOrderNode::Ordering>;
	CegoOrderNode::Ordering *pOrder = _pOrderOptList->First();
	while ( pOrder )
	{
	    pOrderOptList->Insert(*pOrder);
	    pOrder = _pOrderOptList->Next();
	}
    }

    ListT<CegoContentObject*> coList;
    CegoContentObject **pCO = _coList.First();
    while ( pCO )
    {
	CegoContentObject *pCloneCO = (*pCO)->clone(isAttrRef);	
	coList.Insert(pCloneCO);	
	pCO = _coList.Next();
    }

    CegoSelect *pClone = new CegoSelect(coList,
					exprList, 
					pPred,
					pGroupList,
					pHaving,
					pOrderList,
					pOrderOptList,
					_isDistinct,
					_rowLimit,
					_pGTM);        

    if ( _pParentJoinBuf )
	pClone->setParentJoinBuf(_pParentJoinBuf);
    if ( _pBlock )
	pClone->setProcBlock(_pBlock);
    if ( _pUnionSelect )
	pClone->setUnionSelect(_pUnionSelect);
    if ( _isPrepared )
	pClone->prepare();
	    
    return pClone;
}
