///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoField.cc
// ------------
// Cego database table field implementation
//                                                         
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoField
//
// Description: Cego data field container class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoField.h"
#include "CegoXMLdef.h"
#include "CegoTypeConverter.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoField::CegoField()
{
}

CegoField::CegoField(const CegoField& f)
{
    _tableName = f._tableName;
    _tableAlias = f._tableAlias;
    _attrName = f._attrName;
    _type = f._type;
    _len = f._len;
    _dim = f._dim;
    _id = f._id;
    _fv = (CegoFieldValue)f._fv;
}

CegoField::CegoField(const Chain& tableName, const Chain& attrName)
{
    _tableName = tableName;
    _attrName = attrName;
    _type = NULL_TYPE;
    _len = 0;
    _dim = 0;
    _id = 0;
}

CegoField::CegoField(const Chain& tableName, const Chain& tableAlias, const Chain& attrName, CegoDataType type, int len, int dim, const CegoFieldValue& defVal, bool isNullable, int id)
{
    _tableName = tableName;
    _tableAlias = tableAlias;
    _attrName = attrName;
    _type = type;
    _len = len;
    _dim = dim;
    _isNullable = isNullable;
    _id = id;
    _fv = defVal;
}

CegoField::CegoField(const CegoFieldValue& fv)
{ 
    _fv = fv;
}

CegoField::~CegoField()
{  
}

void CegoField::setNullable(bool isNullable)
{
    _isNullable = isNullable;
}

bool CegoField::isNullable() const
{
    return _isNullable;
}

void CegoField::setTableName(const Chain& tableName)
{
    _tableName = tableName;
}

void CegoField::setTableAlias(const Chain& tableAlias)
{
    _tableAlias = tableAlias;
}

const Chain& CegoField::getTableName() const
{
    return _tableName;
}

const Chain& CegoField::getTableAlias() const
{
    return _tableAlias;
}

void CegoField::setId(int id)
{
    _id = id;
}

int CegoField::getId() const
{
    return _id;
}

void CegoField::setAttrName(const Chain& attrName)
{
    _attrName = attrName;
}

const Chain& CegoField::getAttrName() const
{
    return _attrName;
}

void CegoField::setType(const CegoDataType t)
{
    _type = t;
}

const CegoDataType& CegoField::getType() const
{
    return _type;
}

void CegoField::setLength(int len)
{
    _len = len;
}

int CegoField::getLength() const
{
    return _len;
}

void CegoField::setDim(int dim)
{
    _dim = dim;
}

int CegoField::getDim() const
{
    return _dim;
}

const CegoFieldValue& CegoField::getValue() const
{
    return _fv;
}

void CegoField::setValue(const CegoFieldValue& fv)
{
    _fv = (CegoFieldValue)fv;
}

void CegoField::setupNull()
{
    _fv.setNull();
}

void CegoField::setupValue(CegoDataType type, void* pValue, int len)
{
    _fv.setType(type);
    _fv.setValue(pValue);
    _fv.setLength(len);		      
    _fv.setLocalCopy(false);
}

int CegoField::size() const
{
    return _tableName.length() + _tableAlias.length() + _attrName.length() + sizeof(int) + + sizeof(int) + _fv.size() + sizeof(bool) + sizeof(CegoDataType) + sizeof(int); 
}

CegoField& CegoField::operator = ( const CegoField& f)
{
    _tableName = f._tableName;
    _tableAlias = f._tableAlias;
    _attrName = f._attrName;
    _isNullable = f._isNullable;
    _id = f._id;
    _len = f._len;
    _dim = f._dim;
    _type = f._type;
    _fv = (CegoFieldValue)f._fv;

    return (*this);
}

bool CegoField::operator == ( const CegoField& f)
{
    if ( ( ( (Chain)_tableName == (Chain)f._tableName 
	     && (Chain)_tableAlias == (Chain)f._tableAlias )
	   || (Chain)_tableName == (Chain)f._tableAlias
	   || (Chain)_tableAlias == (Chain)f._tableName
	   || (Chain)f._tableName == Chain()
	   || (Chain)_tableName == Chain() )
	 && (Chain)_attrName == (Chain)f._attrName)
	return true;
    return false;
}

bool CegoField::operator != ( const CegoField& f)
{
    return ! ( *this == f );
}

Chain CegoField::typeToChain() const
{
    Chain s;

    switch ( _type )
    {
    case NULL_TYPE:
    {
	s = Chain("null");
	break;
    }
    case VARCHAR_TYPE:
    case BIGINT_TYPE:
    case DECIMAL_TYPE:
    {
	s = CEGO_TYPE_MAP[_type] + Chain("(") + Chain(_len) + Chain(",") + Chain(_dim) + Chain(")");
	break;	
    }
    case INT_TYPE:
    case LONG_TYPE:
    case BOOL_TYPE:
    case DATETIME_TYPE:	
    case FLOAT_TYPE:
    case DOUBLE_TYPE:
    case SMALLINT_TYPE:
    case TINYINT_TYPE:
    case BLOB_TYPE:	
    case CLOB_TYPE:
    case PAGEID_TYPE:
    {
	s = CEGO_TYPE_MAP[_type];
	break;
    }
    }
    return s;
}    
