///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdminHandler.cc
// -------------------
// Cego admin handler class implementation
//                                                         
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdminHandler
// 
// Description: Administration client service request handler class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/AESCrypt.h>
#include <lfcbase/Datetime.h>

// CEGO INCLUDES
#include "CegoDefs.h"
#include "CegoXMLdef.h"
#include "CegoAdminHandler.h"
#include "CegoTypeConverter.h"
#include "CegoDatabaseManager.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

#define MAX(a,b) ((a) > (b) ? (a) : (b))

CegoAdminHandler::CegoAdminHandler(CegoModule *pModule, NetHandler *pN)
{
    _pN = pN;
    _pModule = pModule;
    
    Document *pDoc = new Document(XML_CEGO);
    pDoc->setAttribute(XML_VERSION_ATTR, XML_VERSION_VALUE);
    
    _xml.setDocument(pDoc);

    _modId = _pModule->getModId("CegoAdminHandler");
}

CegoAdminHandler::~CegoAdminHandler()
{
    Document *pDoc = _xml.getDocument();
    pDoc->clear();
    delete pDoc;
}

CegoAdminHandler::ResultType CegoAdminHandler::requestSession(const Chain& user, const Chain& password, bool doEncrypt)
{    
#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Request session for user ") + user + Chain("/") + password);
#endif

    _xml.getDocument()->clear();

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    
    pRoot->setAttribute(XML_NAME_ATTR, user);
    
    if ( doEncrypt )
    {	
	AESCrypt aescrypt(CEGOAESKEY, CEGOAESKEYLEN);
	pRoot->setAttribute(XML_PASSWD_ATTR, aescrypt.encrypt(password));
    }
    else
    {
	pRoot->setAttribute(XML_PASSWD_ATTR, password);
    }

    _xml.getDocument()->setRootElement(pRoot);
    _xml.getDocument()->setDocType(Chain(XML_ADMINSESSION_REQUEST));
    
    Chain request;
    _xml.getXMLChain(request);

    _xml.getDocument()->clear();

    _pN->setMsg(request, request.length());

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Requesting session ..."));
#endif

    _pN->writeMsg();
    
    _pN->readMsg();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, _pN->getMsg());
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _xml.getDocument()->clear();
    _xml.setChain( _pN->getMsg() );
    _xml.parse();
    
    Chain docType = _xml.getDocument()->getDocType();

    if ( docType == Chain(XML_OK_DOC) )
    {
#ifdef CGDEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("Session established"));
#endif
	return ADM_OK;
    }
    else // if ( docType == Chain(XML_ERROR_DOC) )
    {
#ifdef CGDEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("Session rejected"));
#endif

	return ADM_ERROR;
    }    
}

CegoAdminHandler::ResultType CegoAdminHandler::closeSession()
{   
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    
    _xml.getDocument()->setRootElement(pRoot);
    _xml.getDocument()->setDocType(XML_SESSION_CLOSE);
    Chain request;
    _xml.getXMLChain(request);
    _xml.getDocument()->clear();    

    _pN->setMsg(request, request.length());
    
    _pN->writeMsg();

    _pN->readMsg();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, _pN->getMsg());
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _xml.getDocument()->clear();
    _xml.setChain( _pN->getMsg() );
    _xml.parse();
    
    Chain docType = _xml.getDocument()->getDocType();

    if ( docType == Chain(XML_OK_DOC) )
    {
#ifdef CGDEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("Session closed"));
#endif

	return ADM_OK;
    }
    else // if ( docType == Chain(XML_ERROR_DOC) )
    {
#ifdef CGDEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("Session close failed"));
#endif
	return ADM_ERROR;
    }    
}

bool CegoAdminHandler::acceptSession()
{	    
#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Accepting session"));
#endif            

    try
    {
	_xml.getDocument()->clear();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, _pN->getMsg());
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

	_xml.setChain( _pN->getMsg() );	
	_xml.parse();
			
	Chain docType = _xml.getDocument()->getDocType();
	
	if ( docType != Chain(XML_ADMINSESSION_REQUEST) )
	{	    
	    _pModule->log(_modId, Logger::LOGERR, Chain("Invalid request"));

	    _xml.getDocument()->clear();
	    Element* pRoot = new Element(XML_FRAME_ELEMENT);
	    pRoot->setAttribute(XML_MSG_ATTR, Chain("Invalid request"));	    

	    _xml.getDocument()->setRootElement(pRoot);

	    _xml.getDocument()->setDocType(XML_ERROR_DOC);

	    Chain response;
	    _xml.getXMLChain(response);

#ifdef CGDEBUG
	    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
	    _pModule->log(_modId, Logger::DEBUG, response);
	    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif	    
	    _pN->setMsg(response, response.length());
	    _pN->writeMsg();

	    return false;
	}
	else
	{
	    Element *pRoot = _xml.getDocument()->getRootElement();
	    if ( pRoot )
	    {
		_user = pRoot->getAttributeValue(XML_NAME_ATTR);
		_password = pRoot->getAttributeValue(XML_PASSWD_ATTR);
		
		// response is sent after authorization check		
		return true;
	    }	    
	    else
	    {
		throw Exception(EXLOC, "Cannot get root element from message");
	    }	    
	}	
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);	
	_pModule->log(_modId, Logger::LOGERR, Chain("Aborting session. Reason=") +  msg);
	_pN->sendNack();
	return false;	
    }
}

void CegoAdminHandler::setNetHandle(NetHandler *pN)
{
    _pN = pN;
}

NetHandler* CegoAdminHandler::getNetHandle()
{
    return _pN;
}

CegoAdminHandler::RequestType CegoAdminHandler::acceptRequest()
{
#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Accepting request"));
#endif    

    if ( _pN->waitMsg(NETMNG_WAITMSG_TIMEOUT) == false )
	return CegoAdminHandler::REQTIMEOUT;

    _pN->readMsg();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, _pN->getMsg());
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _xml.getDocument()->clear();
    _xml.setChain( _pN->getMsg() );
    _xml.parse();
        
    Chain docType = _xml.getDocument()->getDocType();

    if (  docType == Chain(XML_GET_TABLESETLIST_REQUEST) ) 
    {
	return CegoAdminHandler::GET_TABLESETLIST;
    }
    else if ( docType == Chain(XML_LOCKINFO_REQUEST) ) 
    {
	return CegoAdminHandler::LOCKINFO;
    }
    else if ( docType == Chain(XML_LOCKSTAT_REQUEST) ) 
    {
	return CegoAdminHandler::LOCKSTAT;
    }
    else if ( docType == Chain(XML_POOLINFO_REQUEST) ) 
    {
	return CegoAdminHandler::POOLINFO;
    }
    else if ( docType == Chain(XML_PARAMETERINFO_REQUEST) ) 
    {
	return CegoAdminHandler::PARAMETERINFO;
    }
    else if ( docType == Chain(XML_POOLENTRYLIST_REQUEST) ) 
    {
	return CegoAdminHandler::POOLENTRYLIST;
    }
    else if ( docType == Chain(XML_THREADINFO_REQUEST) ) 
    {
	return CegoAdminHandler::THREADINFO;
    }    
    else if ( docType == Chain(XML_DBTHREADABORT_REQUEST) ) 
    {
	return CegoAdminHandler::DBTHREADABORT;
    }    
    else if ( docType == Chain(XML_DBTHREADINFO_REQUEST) ) 
    {
	return CegoAdminHandler::DBTHREADINFO;
    }
    else if ( docType == Chain(XML_DBTHREADHIST_REQUEST) ) 
    {
	return CegoAdminHandler::DBTHREADHIST;
    }
    else if ( docType == Chain(XML_QUERYHISTLAST_REQUEST) ) 
    {
	return CegoAdminHandler::QUERYHISTLAST;
    }
    else if ( docType == Chain(XML_QUERYHISTCOST_REQUEST) ) 
    {
	return CegoAdminHandler::QUERYHISTCOST;
    }
    else if ( docType == Chain(XML_SETNUMQUERYLAST_REQUEST) ) 
    {
	return CegoAdminHandler::SET_NUMQUERYLAST;
    }
    else if ( docType == Chain(XML_SETNUMQUERYCOST_REQUEST) ) 
    {
	return CegoAdminHandler::SET_NUMQUERYCOST;
    }
    else if ( docType == Chain(XML_ADMTHREADINFO_REQUEST) ) 
    {
	return CegoAdminHandler::ADMTHREADINFO;
    }
    else if ( docType == Chain(XML_SHOWQUERYCACHE_REQUEST) ) 
    {
	return CegoAdminHandler::SHOWQUERYCACHE;
    }
    else if ( docType == Chain(XML_LISTQUERYCACHE_REQUEST) ) 
    {
	return CegoAdminHandler::LISTQUERYCACHE;
    }
    else if ( docType == Chain(XML_CLEANQUERYCACHE_REQUEST) ) 
    {
	return CegoAdminHandler::CLEANQUERYCACHE;
    }
    else if ( docType == Chain(XML_SETQUERYCACHEPARAM_REQUEST) ) 
    {
	return CegoAdminHandler::SETQUERYCACHEPARAM;
    }
    else if ( docType == Chain(XML_SHOWTABLECACHE_REQUEST) ) 
    {
	return CegoAdminHandler::SHOWTABLECACHE;
    }
    else if ( docType == Chain(XML_LISTTABLECACHE_REQUEST) ) 
    {
	return CegoAdminHandler::LISTTABLECACHE;
    }
    else if ( docType == Chain(XML_CLEANTABLECACHE_REQUEST) ) 
    {
	return CegoAdminHandler::CLEANTABLECACHE;
    }
    else if ( docType == Chain(XML_SETTABLECACHEPARAM_REQUEST) ) 
    {
	return CegoAdminHandler::SETTABLECACHEPARAM;
    }
    else if ( docType == Chain(XML_DEFINE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::DEFINE_TABLESET;
    }
    else if ( docType == Chain(XML_REMOVE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::REMOVE_TABLESET;
    }
    else if ( docType == Chain(XML_RESET_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::RESET_TABLESET;
    }
    else if ( docType == Chain(XML_CREATE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::CREATE_TABLESET;
    }
    else if ( docType == Chain(XML_DROP_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::DROP_TABLESET;
    }
    else if ( docType == Chain(XML_START_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::START_TABLESET;
    }
    else if (  docType == Chain(XML_STOP_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::STOP_TABLESET;
    }
    else if ( docType == Chain(XML_SYNC_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::SYNC_TABLESET;
    }
    else if ( docType == Chain(XML_INIT_LOGS_REQUEST) ) 
    {
	return CegoAdminHandler::INIT_LOGS;
    }
    else if (  docType == Chain(XML_START_RECOVER_REQUEST) ) 
    {
	return CegoAdminHandler::START_RECOVERY;
    }
    else if (  docType == Chain(XML_STOP_RECOVER_REQUEST) ) 
    {
	return CegoAdminHandler::STOP_RECOVERY;
    }
    else if (  docType == Chain(XML_GET_TSINFO_REQUEST) ) 
    {
	return CegoAdminHandler::GET_TSINFO;
    }
    else if (  docType == Chain(XML_GET_DETAILED_TSINFO_REQUEST) ) 
    {
	return CegoAdminHandler::GET_DETAILED_TSINFO;
    }
    else if (  docType == Chain(XML_GET_ARCHLOG_INFO_REQUEST) ) 
    {
	return CegoAdminHandler::GET_ARCHLOG_INFO;
    }
    else if (  docType == Chain(XML_GET_OBJECT_INFO_REQUEST) ) 
    {
	return CegoAdminHandler::GET_OBJECT_INFO;
    }
    else if (  docType == Chain(XML_GET_TRANSACTION_INFO_REQUEST) ) 
    {
	return CegoAdminHandler::GET_TRANSACTION_INFO;
    }
    else if (  docType == Chain(XML_GET_BUSTAT_REQUEST) ) 
    {
	return CegoAdminHandler::GET_BUSTAT;
    }
    else if (  docType == Chain(XML_SET_LSN_REQUEST) ) 
    {
	return CegoAdminHandler::SET_LSN;
    }
    else if (  docType == Chain(XML_EXPORT_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::EXPORT_TABLESET;
    }
    else if (  docType == Chain(XML_IMPORT_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::IMPORT_TABLESET;
    }
    else if (  docType == Chain(XML_EXPORT_TABLE_REQUEST) ) 
    {
	return CegoAdminHandler::EXPORT_TABLE;
    }
    else if (  docType == Chain(XML_IMPORT_TABLE_REQUEST) ) 
    {
	return CegoAdminHandler::IMPORT_TABLE;
    }
    else if (  docType == Chain(XML_ADD_DATAFILE_REQUEST) ) 
    {
	return CegoAdminHandler::ADD_DATAFILE;
    }
    else if (  docType == Chain(XML_SHOW_USER_REQUEST) ) 
    {
	return CegoAdminHandler::SHOW_USER;
    }
    else if (  docType == Chain(XML_ADD_USER_REQUEST) ) 
    {
	return CegoAdminHandler::ADD_USER;
    }
    else if (  docType == Chain(XML_REMOVE_USER_REQUEST) ) 
    {
	return CegoAdminHandler::REMOVE_USER;
    }
    else if (  docType == Chain(XML_CHANGE_PWD_REQUEST) ) 
    {
	return CegoAdminHandler::CHANGE_PWD;
    }
    else if (  docType == Chain(XML_ASSIGN_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::ASSIGN_ROLE;
    }
    else if (  docType == Chain(XML_REMOVE_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::REMOVE_ROLE;
    }
    else if (  docType == Chain(XML_LIST_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::LIST_ROLE;
    }
    else if (  docType == Chain(XML_SHOW_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::SHOW_ROLE;
    }
    else if (  docType == Chain(XML_CREATE_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::CREATE_ROLE;
    }
    else if (  docType == Chain(XML_DROP_ROLE_REQUEST) ) 
    {
	return CegoAdminHandler::DROP_ROLE;
    }
    else if (  docType == Chain(XML_SET_PERM_REQUEST) ) 
    {
	return CegoAdminHandler::SET_PERM;
    }
    else if (  docType == Chain(XML_REMOVE_PERM_REQUEST) ) 
    {
	return CegoAdminHandler::REMOVE_PERM;
    }
    else if (  docType == Chain(XML_USER_TRACE_REQUEST) ) 
    {
	return CegoAdminHandler::USER_TRACE;
    }
    else if (  docType == Chain(XML_ADD_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::ADD_ARCHLOG;
    }
    else if (  docType == Chain(XML_REMOVE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::REMOVE_ARCHLOG;
    }
    else if (  docType == Chain(XML_BEGIN_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::BEGIN_BACKUP;
    }
    else if (  docType == Chain(XML_EXECUTE_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::EXECUTE_BACKUP;
    }
    else if (  docType == Chain(XML_END_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::END_BACKUP;
    }
    else if (  docType == Chain(XML_LIST_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::LIST_BACKUP;
    }
    else if (  docType == Chain(XML_RESTORE_REQUEST) ) 
    {
	return CegoAdminHandler::RESTORE;
    }
    else if (  docType == Chain(XML_RECOVER_REQUEST) ) 
    {
	return CegoAdminHandler::RECOVER;
    }
    else if (  docType == Chain(XML_VERIFY_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::VERIFY_TABLESET;
    }
    else if (  docType == Chain(XML_CORRECT_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::CORRECT_TABLESET;
    }
    else if (  docType == Chain(XML_DUMP_OBJECT_REQUEST) ) 
    {
	return CegoAdminHandler::DUMP_OBJECT;
    }
    else if (  docType == Chain(XML_SET_LOGMNG_REQUEST) ) 
    {
	return CegoAdminHandler::SET_LOGMNG;
    }
    else if (  docType == Chain(XML_SET_BACKUPMNG_REQUEST) ) 
    {
	return CegoAdminHandler::SET_BACKUPMNG;
    }
    else if (  docType == Chain(XML_SHOW_LOGMNG_REQUEST) ) 
    {
	return CegoAdminHandler::SHOW_LOGMNG;
    }
    else if (  docType == Chain(XML_SHOW_BACKUPMNG_REQUEST) ) 
    {
	return CegoAdminHandler::SHOW_BACKUPMNG;
    }
    else if (  docType == Chain(XML_SET_CHECKPOINT_REQUEST) ) 
    {
	return CegoAdminHandler::SET_CHECKPOINT;
    }
    else if (  docType == Chain(XML_SET_TSINITFILE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSINITFILE;
    }
    else if (  docType == Chain(XML_SET_TSSYSSIZE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSSYSSIZE;
    }
    else if (  docType == Chain(XML_SET_TSTMPSIZE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSTMPSIZE;
    }
    else if (  docType == Chain(XML_SET_TSAPPSIZE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSAPPSIZE;
    }
    else if (  docType == Chain(XML_SET_TSROOTPATH_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSROOTPATH;
    }
    else if (  docType == Chain(XML_SET_TSLOGNUM_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSLOGNUM;
    }
    else if (  docType == Chain(XML_SET_TSLOGSIZE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSLOGSIZE;
    }
    else if (  docType == Chain(XML_SET_TSSORTAREASIZE_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSSORTAREASIZE;
    }
    else if (  docType == Chain(XML_SET_TSLOGUSER_REQUEST) ) 
    {
	return CegoAdminHandler::SET_TSLOGUSER;
    }
    else if (  docType == Chain(XML_ENABLE_TABLECACHE_REQUEST) ) 
    {
	return CegoAdminHandler::ENABLE_TABLECACHE;
    }
    else if (  docType == Chain(XML_DISABLE_TABLECACHE_REQUEST) ) 
    {
	return CegoAdminHandler::DISABLE_TABLECACHE;
    }
    else if (  docType == Chain(XML_ENABLE_QUERYCACHE_REQUEST) ) 
    {
	return CegoAdminHandler::ENABLE_QUERYCACHE;
    }
    else if (  docType == Chain(XML_DISABLE_QUERYCACHE_REQUEST) ) 
    {
	return CegoAdminHandler::DISABLE_QUERYCACHE;
    }
    else if (  docType == Chain(XML_VERIFY_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::VERIFY_TABLESET;
    }
    else if (  docType == Chain(XML_CORRECT_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::CORRECT_TABLESET;
    }
    else if (  docType == Chain(XML_ENABLE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::ENABLE_ARCHLOG;
    }
    else if (  docType == Chain(XML_DISABLE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::DISABLE_ARCHLOG;
    }
    else if (  docType == Chain(XML_ENABLE_AUTOCORRECT_REQUEST) ) 
    {
	return CegoAdminHandler::ENABLE_AUTOCORRECT;
    }
    else if (  docType == Chain(XML_DISABLE_AUTOCORRECT_REQUEST) ) 
    {
	return CegoAdminHandler::DISABLE_AUTOCORRECT;
    }
    else if (  docType == Chain(XML_ENABLE_APPENDMODE_REQUEST) ) 
    {
	return CegoAdminHandler::ENABLE_APPENDMODE;
    }
    else if (  docType == Chain(XML_DISABLE_APPENDMODE_REQUEST) ) 
    {
	return CegoAdminHandler::DISABLE_APPENDMODE;
    }
    else if (  docType == Chain(XML_RESET_BUSTAT_REQUEST) ) 
    {
	return CegoAdminHandler::RESET_BUSTAT;
    }




// mediator requests

    /*
    else if (  docType == Chain(XML_MED_NOTIFY_REQUEST) ) 
    {
	return CegoAdminHandler::MED_NOTIFY;
    }
    else if ( docType == Chain(XML_MED_DEFINE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_DEFINE_TABLESET;
    }
    else if ( docType == Chain(XML_MED_REMOVE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_REMOVE_TABLESET;
    }
    else if ( docType == Chain(XML_MED_RESET_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_RESET_TABLESET;
    }
    else if ( docType == Chain(XML_MED_CREATE_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_CREATE_TABLESET;
    }
    else if ( docType == Chain(XML_MED_DROP_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_DROP_TABLESET;
    }
    else if (  docType == Chain(XML_MED_START_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_START_TABLESET;
    }
    else if (  docType == Chain(XML_MED_STOP_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_STOP_TABLESET;
    }
    else if (  docType == Chain(XML_MED_COPY_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_COPY_TABLESET;
    }
    else if (  docType == Chain(XML_MED_SWITCH_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_SWITCH_TABLESET;
    }
    else if (  docType == Chain(XML_MED_ADD_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::MED_ADD_ARCHLOG;
    }
    else if (  docType == Chain(XML_MED_REMOVE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::MED_REMOVE_ARCHLOG;
    }
    else if (  docType == Chain(XML_MED_ADD_DATAFILE_REQUEST) ) 
    {
	return CegoAdminHandler::MED_ADD_DATAFILE;
    }
    else if (  docType == Chain(XML_MED_GET_DETAILEDTABLESETINFO_REQUEST) ) 
    {
	return CegoAdminHandler::MED_GET_DETAILEDTABLESETINFO;
    }
    else if (  docType == Chain(XML_MED_GET_ARCHLOGINFO_REQUEST) ) 
    {
	return CegoAdminHandler::MED_GET_ARCHLOGINFO;
    }
    else if (  docType == Chain(XML_MED_GET_OBJECTINFO_REQUEST) ) 
    {
	return CegoAdminHandler::MED_GET_OBJECTINFO;
    }
    else if (  docType == Chain(XML_MED_GET_TRANSACTIONINFO_REQUEST) ) 
    {
	return CegoAdminHandler::MED_GET_TRANSACTIONINFO;
    }
    else if (  docType == Chain(XML_MED_GET_BUSTAT_REQUEST) ) 
    {
	return CegoAdminHandler::MED_GET_BUSTAT;
    }
    else if (  docType == Chain(XML_MED_BEGIN_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::MED_BEGIN_BACKUP;
    }
    else if (  docType == Chain(XML_MED_EXECUTE_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::MED_EXECUTE_BACKUP;
    }
    else if (  docType == Chain(XML_MED_END_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::MED_END_BACKUP;
    }
    else if (  docType == Chain(XML_MED_LIST_BACKUP_REQUEST) ) 
    {
	return CegoAdminHandler::MED_LIST_BACKUP;
    }
    else if (  docType == Chain(XML_MED_RESTORE_REQUEST) ) 
    {
	return CegoAdminHandler::MED_RESTORE;
    }
    else if (  docType == Chain(XML_MED_RECOVER_REQUEST) ) 
    {
	return CegoAdminHandler::MED_RECOVER;
    }
    else if (  docType == Chain(XML_MED_SECSWITCH_REQUEST) ) 
    {
	return CegoAdminHandler::MED_SECSWITCH;
    }
    else if (  docType == Chain(XML_SEC_MEDSWITCH_REQUEST) ) 
    {
	return CegoAdminHandler::SEC_MEDSWITCH;
    }
    else if (  docType == Chain(XML_MED_SECRELOCATE_REQUEST) ) 
    {
	return CegoAdminHandler::MED_SECRELOCATE;
    }
    else if (  docType == Chain(XML_SEC_MEDRELOCATE_REQUEST) ) 
    {
	return CegoAdminHandler::SEC_MEDRELOCATE;
    }
   
    
    else if (  docType == Chain(XML_MED_CHECK_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_CHECK_TABLESET;
    }
    else if (  docType == Chain(XML_MED_VERIFY_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_VERIFY_TABLESET;
    }
    else if (  docType == Chain(XML_MED_CORRECT_TABLESET_REQUEST) ) 
    {
	return CegoAdminHandler::MED_CORRECT_TABLESET;
    }
    else if (  docType == Chain(XML_MED_ENABLE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::MED_ENABLE_ARCHLOG;
    }
    else if (  docType == Chain(XML_MED_DISABLE_ARCHLOG_REQUEST) ) 
    {
	return CegoAdminHandler::MED_DISABLE_ARCHLOG;
    }
    else if (  docType == Chain(XML_MED_ENABLE_AUTOCORRECT_REQUEST) ) 
    {
	return CegoAdminHandler::MED_ENABLE_AUTOCORRECT;
    }
    else if (  docType == Chain(XML_MED_DISABLE_AUTOCORRECT_REQUEST) ) 
    {
	return CegoAdminHandler::MED_DISABLE_AUTOCORRECT;
    }
    else if (  docType == Chain(XML_MED_ENABLE_APPENDMODE_REQUEST) ) 
    {
	return CegoAdminHandler::MED_ENABLE_APPENDMODE;
    }
    else if (  docType == Chain(XML_MED_DISABLE_APPENDMODE_REQUEST) ) 
    {
	return CegoAdminHandler::MED_DISABLE_APPENDMODE;
    }
    */
    
    else if (  docType == Chain(XML_SESSION_CLOSE) ) 
    {
	return CegoAdminHandler::SESSION_CLOSE;
    }
    else
    {
	return CegoAdminHandler::UNKNOWN;
    }     
}

CegoAdminHandler::ResultType CegoAdminHandler::reqLockInfo(const Chain& lockCat)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_LOCKCAT_ATTR, lockCat);
    return sendReq(XML_LOCKINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqLockStat()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_LOCKSTAT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqPoolInfo()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_POOLINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqParameterInfo()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_PARAMETERINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqPoolEntryList()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_POOLENTRYLIST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqThreadInfo()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_THREADINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDbThreadAbort(unsigned threadId)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_THID_ATTR, Chain(threadId));
    return sendReq(XML_DBTHREADABORT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDbThreadInfo()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_DBTHREADINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDbThreadHistory(unsigned threadId)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_THID_ATTR, Chain(threadId));
    return sendReq(XML_DBTHREADHIST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqQueryHistoryLast(unsigned numLast)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NUMQUERYLAST_ATTR, Chain(numLast));
    return sendReq(XML_QUERYHISTLAST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqQueryHistoryCost(unsigned numCost)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NUMQUERYCOST_ATTR, Chain(numCost));
    return sendReq(XML_QUERYHISTCOST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqNumQueryLast(unsigned numLast)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NUMQUERYLAST_ATTR, Chain(numLast));
    return sendReq(XML_SETNUMQUERYLAST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqNumQueryCost(unsigned numCost)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NUMQUERYCOST_ATTR, Chain(numCost));
    return sendReq(XML_SETNUMQUERYCOST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqAdmThreadInfo()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_ADMTHREADINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowQueryCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_SHOWQUERYCACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqListQueryCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_LISTQUERYCACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqCleanQueryCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_CLEANQUERYCACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetQueryCacheParam(const Chain& tableSet, unsigned maxEntry, unsigned maxSize, unsigned threshold, unsigned hashRange)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);  
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_QUERYCACHEMAXENTRY_ATTR, Chain(maxEntry));
    pRoot->setAttribute(XML_QUERYCACHEMAXSIZE_ATTR, Chain(maxSize));
    pRoot->setAttribute(XML_QUERYCACHETHRESHOLD_ATTR, Chain(threshold));
    pRoot->setAttribute(XML_QUERYCACHEHASHRANGE_ATTR, Chain(hashRange));
    
    return sendReq(XML_SETQUERYCACHEPARAM_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowTableCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_SHOWTABLECACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqListTableCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_LISTTABLECACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqCleanTableCache(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_CLEANTABLECACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTableCacheParam(const Chain& tableSet, const Chain& cacheFilter, unsigned maxEntry, unsigned maxSize, unsigned hashRange)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);  
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TABLECACHEFILTER_ATTR, cacheFilter);
    pRoot->setAttribute(XML_TABLECACHEMAXENTRY_ATTR, Chain(maxEntry));
    pRoot->setAttribute(XML_TABLECACHEMAXSIZE_ATTR, Chain(maxSize));
    pRoot->setAttribute(XML_TABLECACHEHASHRANGE_ATTR, Chain(hashRange));
    
    return sendReq(XML_SETTABLECACHEPARAM_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRemoveTableSet(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_REMOVE_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqCreateTableSet(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_CREATE_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDropTableSet(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DROP_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqStartTableSet(const Chain& tableSet, bool cleanIt, bool forceload, bool cpDump, bool noInit)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);

    if ( cleanIt )
	pRoot->setAttribute(XML_CLEANUP_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_CLEANUP_ATTR, XML_FALSE_VALUE);

    if ( forceload )
	pRoot->setAttribute(XML_FORCELOAD_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_FORCELOAD_ATTR, XML_FALSE_VALUE);

    if ( cpDump )
	pRoot->setAttribute(XML_CPDUMP_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_CPDUMP_ATTR, XML_FALSE_VALUE);

    if ( noInit )
	pRoot->setAttribute(XML_NOINIT_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_NOINIT_ATTR, XML_FALSE_VALUE);

    return sendReq(XML_START_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqStopTableSet(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_STOP_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqInitLogs(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_INIT_LOGS_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSyncTableSet(const Chain& tableSet, const Chain& msg, const Chain& escCmd, unsigned timeout)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_BUMSG_ATTR, msg);

    if ( escCmd != Chain() )
    {
	pRoot->setAttribute(XML_ESCCMD_ATTR, escCmd);
	pRoot->setAttribute(XML_TIMEOUT_ATTR, Chain(timeout));
    }
    return sendReq(XML_SYNC_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqStartRecovery(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_START_RECOVER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqStopRecovery(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_STOP_RECOVER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetTSInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_TSINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetLSN(const Chain& tableSet, unsigned long long lsn)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_LSN_ATTR, Chain(lsn));
    return sendReq(XML_SET_LSN_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqExportTableSet(const Chain& tableSet, bool isStructure, const Chain& exportFile, const Chain& expMode)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_FILENAME_ATTR, exportFile);
    if ( isStructure ) 
	pRoot->setAttribute(XML_ISSTRUCT_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_ISSTRUCT_ATTR, XML_FALSE_VALUE);
    pRoot->setAttribute(XML_MODE_ATTR, expMode);

    return sendReq(XML_EXPORT_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqImportTableSet(const Chain& tableSet, bool isStructure, const Chain& importFile, const Chain& impMode)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_FILENAME_ATTR, importFile);    
    if ( isStructure ) 
	pRoot->setAttribute(XML_ISSTRUCT_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_ISSTRUCT_ATTR, XML_FALSE_VALUE);
    
    pRoot->setAttribute(XML_MODE_ATTR, impMode);        
    return sendReq(XML_IMPORT_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqExportTable(const Chain& tableSet, const Chain& tableName, const Chain& exportFile, const Chain& expMode)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
    pRoot->setAttribute(XML_FILENAME_ATTR, exportFile);
    pRoot->setAttribute(XML_MODE_ATTR, expMode);        
    return sendReq(XML_EXPORT_TABLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqImportTable(const Chain& tableSet, const Chain& tableName, const Chain& importFile, const Chain& impMode)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
    pRoot->setAttribute(XML_FILENAME_ATTR, importFile);    
    pRoot->setAttribute(XML_MODE_ATTR, impMode);
    return sendReq(XML_IMPORT_TABLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqAddDataFile(const Chain& tableSet, const Chain& type, const Chain& dataFile, unsigned fileSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TYPE_ATTR, type);
    pRoot->setAttribute(XML_FILENAME_ATTR, dataFile);
    pRoot->setAttribute(XML_FILESIZE_ATTR, Chain(fileSize));
    return sendReq(XML_ADD_DATAFILE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowUser()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_SHOW_USER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqAddUser(const Chain& user, const Chain& password)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    AESCrypt aescrypt(CEGOAESKEY, CEGOAESKEYLEN);
    pRoot->setAttribute(XML_PASSWD_ATTR, aescrypt.encrypt(password));
    return sendReq(XML_ADD_USER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRemoveUser(const Chain& user)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    return sendReq(XML_REMOVE_USER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqChangePwd(const Chain& user, const Chain& password)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    AESCrypt aescrypt(CEGOAESKEY, CEGOAESKEYLEN);
    pRoot->setAttribute(XML_PASSWD_ATTR, aescrypt.encrypt(password));
    return sendReq(XML_CHANGE_PWD_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqListRole()
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_LIST_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowRole(const Chain& role)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    return sendReq(XML_SHOW_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqCreateRole(const Chain& role)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    return sendReq(XML_CREATE_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDropRole(const Chain& role)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    return sendReq(XML_DROP_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqAssignRole(const Chain& user, const Chain& role)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    return sendReq(XML_ASSIGN_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRemoveRole(const Chain& user, const Chain& role)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    return sendReq(XML_REMOVE_ROLE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetPermission(const Chain& role, const Chain& permid, const Chain& tableSet, const Chain& filter, const Chain& perm)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    pRoot->setAttribute(XML_PERMID_ATTR, permid);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_FILTER_ATTR, filter);
    pRoot->setAttribute(XML_PERM_ATTR, perm);
    return sendReq(XML_SET_PERM_REQUEST, pRoot);    
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRemovePermission(const Chain& role, const Chain& permid)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_ROLE_ATTR, role);
    pRoot->setAttribute(XML_PERMID_ATTR, permid);
    return sendReq(XML_REMOVE_PERM_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqUserTrace( const Chain& user, bool isOn)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_NAME_ATTR, user);
    if ( isOn )
	pRoot->setAttribute(XML_TRACE_ATTR, XML_ON_VALUE);
    else
	pRoot->setAttribute(XML_TRACE_ATTR, XML_OFF_VALUE);
	
    return sendReq(XML_USER_TRACE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqAddArchLog(const Chain& tableSet, const Chain& archId, const Chain& archPath)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_ARCHID_ATTR, archId);
    pRoot->setAttribute(XML_ARCHPATH_ATTR, archPath);
    return sendReq(XML_ADD_ARCHLOG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqBeginBackup(const Chain& tableSet, const Chain& msg)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_BUMSG_ATTR, msg);
    return sendReq(XML_BEGIN_BACKUP_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqExecuteBackup(const Chain& tableSet, const Chain& msg)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_BUMSG_ATTR, msg);
    return sendReq(XML_EXECUTE_BACKUP_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqEndBackup(const Chain& tableSet, const Chain& msg,  bool keepTicket)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_BUMSG_ATTR, msg);
    if ( keepTicket )
	pRoot->setAttribute(XML_KEEPTICKET_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_KEEPTICKET_ATTR, XML_FALSE_VALUE);
    return sendReq(XML_END_BACKUP_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqListBackup(const Chain& tableSet, const Chain& pit)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_PIT_ATTR, pit);
    return sendReq(XML_LIST_BACKUP_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRestore(const Chain& tableSet, const Chain& backupId)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_BACKUPID_ATTR, backupId);
    return sendReq(XML_RESTORE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRecover(const Chain& tableSet, const Chain& pit, unsigned long long targetlsn)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_PIT_ATTR, pit);
    pRoot->setAttribute(XML_LSN_ATTR, Chain(targetlsn));    
    return sendReq(XML_RECOVER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqRemoveArchLog(const Chain& tableSet, const Chain& archId)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_ARCHID_ATTR, archId);
    return sendReq(XML_REMOVE_ARCHLOG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetDetailedTSInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_DETAILED_TSINFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetArchLogInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_ARCHLOG_INFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetBUStatInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_BUSTAT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetObjectInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_OBJECT_INFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetTransactionInfo(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_GET_TRANSACTION_INFO_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDumpObject(const Chain& tableSet, CegoObject::ObjectType type, const Chain& objName)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_OBJTYPE_ATTR, CegoTypeConverter::getObjectTypeString(type));
    pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
    return sendReq(XML_DUMP_OBJECT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetLogMng(const Chain& progName, unsigned timeout)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_LOGMNGPROG_ATTR, progName);
    pRoot->setAttribute(XML_LOGMNGTIMEOUT_ATTR, Chain(timeout));
    return sendReq(XML_SET_LOGMNG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetBackupMng(const Chain& progName)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_BACKUPMNGPROG_ATTR, progName);
    return sendReq(XML_SET_BACKUPMNG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowLogMng()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_SHOW_LOGMNG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqShowBackupMng()
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    return sendReq(XML_SHOW_BACKUPMNG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetCheckpoint(const Chain& tableSet, unsigned timeout)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TIMEOUT_ATTR, Chain(timeout));
    return sendReq(XML_SET_CHECKPOINT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSInitFile(const Chain& tableSet, const Chain& initFile)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TSINITFILE_ATTR, initFile);
    return sendReq(XML_SET_TSINITFILE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSSysSize(const Chain& tableSet, unsigned sysSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_SYSSIZE_ATTR, Chain(sysSize));
    return sendReq(XML_SET_TSSYSSIZE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSTmpSize(const Chain& tableSet, unsigned tmpSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TMPSIZE_ATTR, Chain(tmpSize));
    return sendReq(XML_SET_TSTMPSIZE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSAppSize(const Chain& tableSet, unsigned appSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_APPSIZE_ATTR, Chain(appSize));
    return sendReq(XML_SET_TSAPPSIZE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSRootPath(const Chain& tableSet, const Chain& tsRoot)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TSROOT_ATTR, tsRoot);
    return sendReq(XML_SET_TSROOTPATH_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSLogNum(const Chain& tableSet, unsigned logNum)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_LOGFILENUM_ATTR, Chain(logNum));
    return sendReq(XML_SET_TSLOGNUM_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSLogSize(const Chain& tableSet, unsigned logSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_LOGFILESIZE_ATTR, Chain(logSize));
    return sendReq(XML_SET_TSLOGSIZE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSSortAreaSize(const Chain& tableSet, unsigned long long sortAreaSize)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_SORTAREASIZE_ATTR, Chain(sortAreaSize));
    return sendReq(XML_SET_TSSORTAREASIZE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqSetTSLogUser(const Chain& tableSet, const Chain& logUser)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_LOGUSER_ATTR, logUser);
    return sendReq(XML_SET_TSLOGUSER_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqEnableTableCache(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_ENABLE_TABLECACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDisableTableCache(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DISABLE_TABLECACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqEnableQueryCache(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_ENABLE_QUERYCACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDisableQueryCache(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DISABLE_QUERYCACHE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqResetBUStat(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_RESET_BUSTAT_REQUEST, pRoot);
}

////////////////////////////////////////////////
//////////// Mediator Services /////////////////
////////////////////////////////////////////////


/*
CegoAdminHandler::ResultType CegoAdminHandler::medNotifyMediator(const Chain& hostName, const Chain& status, 
								 const ListT<Chain>& tsList, const ListT<Chain>& runList, const ListT<Chain>& syncList)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_HOSTNAME_ATTR, hostName);
    pRoot->setAttribute(XML_STATUS_ATTR, status);

    Chain *pTS = tsList.First();
    Chain *pSync = syncList.First();
    Chain *pRun = runList.First();
    while ( pTS && pSync )
    {
	Element* pTE = new Element(XML_TABLESET_ELEMENT);
	pTE->setAttribute(XML_NAME_ATTR, *pTS);
	pTE->setAttribute(XML_SYNCSTATE_ATTR, *pSync);
	pTE->setAttribute(XML_RUNSTATE_ATTR, *pRun);
	pRoot->addContent(pTE);

	pTS = tsList.Next();
	pSync = syncList.Next();
	pRun = runList.Next();
    }

    return sendReq(XML_MED_NOTIFY_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::medStartTableSet(const Chain& tableSet, bool cleanIt, bool forceload, bool cpDump, bool noInit)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    if ( cleanIt )
	pRoot->setAttribute(XML_CLEANUP_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_CLEANUP_ATTR, XML_FALSE_VALUE);

    if ( forceload )
	pRoot->setAttribute(XML_FORCELOAD_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_FORCELOAD_ATTR, XML_FALSE_VALUE);

    if ( cpDump )
	pRoot->setAttribute(XML_CPDUMP_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_CPDUMP_ATTR, XML_FALSE_VALUE);

    if ( noInit )
	pRoot->setAttribute(XML_NOINIT_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_NOINIT_ATTR, XML_FALSE_VALUE);
    
    return sendReq(XML_MED_START_TABLESET_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::medStopTableSet(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_MED_STOP_TABLESET_REQUEST, pRoot);
}

*/

CegoAdminHandler::ResultType CegoAdminHandler::reqDefineTableSet(const Chain& tableSet,
								 const Chain& tsRoot,
								 unsigned sysFileSize, 
								 unsigned tmpFileSize,
								 unsigned appFileSize,
								 unsigned logFileSize, 
								 unsigned logFileNum,
								 unsigned long long sortAreaSize,
								 unsigned branchId)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    pRoot->setAttribute(XML_TSROOT_ATTR, tsRoot);
    pRoot->setAttribute(XML_SYSSIZE_ATTR, Chain(sysFileSize));
    pRoot->setAttribute(XML_TMPSIZE_ATTR, Chain(tmpFileSize));
    pRoot->setAttribute(XML_APPSIZE_ATTR, Chain(appFileSize));
    pRoot->setAttribute(XML_LOGFILESIZE_ATTR, Chain(logFileSize));
    pRoot->setAttribute(XML_LOGFILENUM_ATTR, Chain(logFileNum));
    pRoot->setAttribute(XML_SORTAREASIZE_ATTR, Chain(sortAreaSize));
    pRoot->setAttribute(XML_BRANCHID_ATTR, Chain(branchId));
    return sendReq(XML_DEFINE_TABLESET_REQUEST, pRoot);
}



CegoAdminHandler::ResultType CegoAdminHandler::reqEnableArchLog(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_ENABLE_ARCHLOG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDisableArchLog(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DISABLE_ARCHLOG_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqEnableAppendMode(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_ENABLE_APPENDMODE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDisableAppendMode(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DISABLE_APPENDMODE_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqEnableAutoCorrect(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_ENABLE_AUTOCORRECT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqDisableAutoCorrect(const Chain& tableSet)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_DISABLE_AUTOCORRECT_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqGetTableSetList(bool getUsage)
{    
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    if ( getUsage )
	pRoot->setAttribute(XML_USAGE_ATTR, XML_TRUE_VALUE);
    else
	pRoot->setAttribute(XML_USAGE_ATTR, XML_FALSE_VALUE);

    return sendReq(XML_GET_TABLESETLIST_REQUEST, pRoot);
}

CegoAdminHandler::ResultType CegoAdminHandler::reqVerifyTableSet(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_VERIFY_TABLESET_REQUEST, pRoot);    
}

CegoAdminHandler::ResultType CegoAdminHandler::reqCorrectTableSet(const Chain& tableSet)
{
    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
    return sendReq(XML_CORRECT_TABLESET_REQUEST, pRoot);    
}

CegoAdminHandler::ResultType CegoAdminHandler::sendReq(const Chain& reqType, Element* pRoot)
{
    _xml.getDocument()->clear();
    _xml.getDocument()->setRootElement(pRoot);
    _xml.getDocument()->setDocType(reqType);
    
    Chain request;
    _xml.getXMLChain(request);
    _xml.getDocument()->clear();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, request);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif    

    _pN->setMsg(request, request.length());

    _pN->writeMsg();
    
    _pN->readMsg();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, _pN->getMsg());
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _xml.getDocument()->clear();
    _xml.setChain( _pN->getMsg() );
    _xml.parse();

    Chain docType = _xml.getDocument()->getDocType();
    
    if ( docType == Chain(XML_OK_DOC) )
    {
	return ADM_OK;
    }
    else if ( docType == Chain(XML_INFO_DOC) )
    {
	return ADM_INFO;
    }
    else // if ( docType == Chain(XML_ERROR_DOC) )
    {
	return ADM_ERROR;
    }
}

void CegoAdminHandler::getMsg(Chain& msg)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	msg = pRoot->getAttributeValue(XML_MSG_ATTR);
    }
}

void CegoAdminHandler::getBUMsg(Chain& msg)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	msg = pRoot->getAttributeValue(XML_BUMSG_ATTR);
    }
}

const Chain& CegoAdminHandler::getUser()
{
    return _user;
}

const Chain& CegoAdminHandler::getPassword()
{
    return _password;
}

void CegoAdminHandler::getBackupId(Chain& backupId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	backupId = pRoot->getAttributeValue(XML_BACKUPID_ATTR);
    }
}

void CegoAdminHandler::getTableSet(Chain& tableSet)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
    }
}

void CegoAdminHandler::getTableSetRoot(Chain& tsRoot)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tsRoot = pRoot->getAttributeValue(XML_TSROOT_ATTR);
    }
}

void CegoAdminHandler::getTableSetTSTicket(Chain& tsTicket)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tsTicket = pRoot->getAttributeValue(XML_TSTICKET_ATTR);
    }
}

void CegoAdminHandler::getTableName(Chain& tableName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
    }
}

void CegoAdminHandler::getThreadId(unsigned& threadId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	threadId = pRoot->getAttributeValue(XML_THID_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getEscapeCmd(Chain& escCmd)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	escCmd = pRoot->getAttributeValue(XML_ESCCMD_ATTR);
    }
}

void CegoAdminHandler::getTSInitFile(Chain& initFile)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	initFile = pRoot->getAttributeValue(XML_TSINITFILE_ATTR);
    }
}

void CegoAdminHandler::getTSRootPath(Chain& tsRoot)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tsRoot = pRoot->getAttributeValue(XML_TSROOT_ATTR);
    }
}

void CegoAdminHandler::getTSLogUser(Chain& logUser)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	logUser = pRoot->getAttributeValue(XML_LOGUSER_ATTR);
    }
}

void CegoAdminHandler::getTSTicket(Chain& tsTicket)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tsTicket = pRoot->getAttributeValue(XML_TSTICKET_ATTR);
    }
}

void CegoAdminHandler::getTSLogNum(unsigned& logNum)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	logNum = pRoot->getAttributeValue(XML_LOGFILENUM_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTSLogSize(unsigned& logSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	logSize = pRoot->getAttributeValue(XML_LOGFILESIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTSSortAreaSize(unsigned long long& sortAreaSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	sortAreaSize = pRoot->getAttributeValue(XML_SORTAREASIZE_ATTR).asUnsignedLongLong();
    }
}

void CegoAdminHandler::getTimeoutValue(unsigned& timeout)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	timeout = pRoot->getAttributeValue(XML_TIMEOUT_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getFileName(Chain& fileName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	fileName = pRoot->getAttributeValue(XML_FILENAME_ATTR);
    }
}

void CegoAdminHandler::getFileType(Chain& fileType)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	fileType = pRoot->getAttributeValue(XML_TYPE_ATTR);
    }
}

void CegoAdminHandler::getFileId(unsigned& fileId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	fileId = pRoot->getAttributeValue(XML_FILEID_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getFileSize(unsigned& fileSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	fileSize = pRoot->getAttributeValue(XML_FILESIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getFilter(Chain& filter)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	filter = pRoot->getAttributeValue(XML_FILTER_ATTR);
    }
}

void CegoAdminHandler::getSysFileId(unsigned& sysFid)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	sysFid = pRoot->getAttributeValue(XML_SYSFID_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTmpFileId(unsigned& tmpFid)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tmpFid = pRoot->getAttributeValue(XML_TMPFID_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getSysSize(unsigned& sysSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	sysSize = pRoot->getAttributeValue(XML_SYSSIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTmpSize(unsigned& tmpSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	tmpSize = pRoot->getAttributeValue(XML_TMPSIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getAppSize(unsigned& appSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	appSize = pRoot->getAttributeValue(XML_APPSIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getLogFileSize(unsigned& logFileSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	logFileSize = pRoot->getAttributeValue(XML_LOGFILESIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getLogFileNum(unsigned& logFileNum)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	logFileNum = pRoot->getAttributeValue(XML_LOGFILENUM_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getSortAreaSize(unsigned long long& sortAreaSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	sortAreaSize = pRoot->getAttributeValue(XML_SORTAREASIZE_ATTR).asUnsignedLongLong();
    }
}

void CegoAdminHandler::getBranchId(unsigned& branchId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
        branchId = pRoot->getAttributeValue(XML_BRANCHID_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTargetHost(Chain& targetHost)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	targetHost = pRoot->getAttributeValue(XML_TARGETHOST_ATTR);
    }
}

void CegoAdminHandler::getUser(Chain& user)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	user = pRoot->getAttributeValue(XML_NAME_ATTR);
    }
}

void CegoAdminHandler::getPassword(Chain& password)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	password = pRoot->getAttributeValue(XML_PASSWD_ATTR);
    }
}

void CegoAdminHandler::getPerm(Chain& perm)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	perm = pRoot->getAttributeValue(XML_PERM_ATTR);
    }
}

void CegoAdminHandler::getPermId(Chain& permid)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	permid = pRoot->getAttributeValue(XML_PERMID_ATTR);
    }
}

void CegoAdminHandler::getRole(Chain& role)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	role = pRoot->getAttributeValue(XML_ROLE_ATTR);
    }
}

void CegoAdminHandler::getTrace(bool& isOn)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if (  pRoot->getAttributeValue(XML_TRACE_ATTR) == Chain(XML_ON_VALUE) )
	    isOn = true;
	else
	    isOn = false;
    }
}

void CegoAdminHandler::getUsage(bool& getUsage)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if (  pRoot->getAttributeValue(XML_USAGE_ATTR) == Chain(XML_TRUE_VALUE) )
	    getUsage = true;
	else
	    getUsage = false;
    }
}

void CegoAdminHandler::getLogManager(Chain& progName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	progName = pRoot->getAttributeValue(XML_LOGMNGPROG_ATTR);
    }
}

void CegoAdminHandler::getLogManagerTimeout(unsigned& timeout)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	timeout = pRoot->getAttributeValue(XML_LOGMNGTIMEOUT_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getBackupManager(Chain& progName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	progName = pRoot->getAttributeValue(XML_BACKUPMNGPROG_ATTR);
    }
}

void CegoAdminHandler::getLockCat(Chain& lockCat)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	lockCat = pRoot->getAttributeValue(XML_LOCKCAT_ATTR);
    }
}

void CegoAdminHandler::getLSN(unsigned long long& lsn)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	lsn = pRoot->getAttributeValue(XML_LSN_ATTR).asUnsignedLongLong();
    }
}

void CegoAdminHandler::getDataId(Chain& dataId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	dataId = pRoot->getAttributeValue(XML_DATAID_ATTR);
    }
}

void CegoAdminHandler::getPit(Chain& pit)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	pit = pRoot->getAttributeValue(XML_PIT_ATTR);
    }
}

void CegoAdminHandler::getIsStructure(bool& isStructure)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( pRoot->getAttributeValue(XML_ISSTRUCT_ATTR) == Chain(XML_TRUE_VALUE) )
	    isStructure = true;
	else
	    isStructure = false;
    }
}

void CegoAdminHandler::getKeepTicket(bool& keepTicket)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( pRoot->getAttributeValue(XML_KEEPTICKET_ATTR) == Chain(XML_TRUE_VALUE) )
	    keepTicket = true;
	else
	    keepTicket = false;
    }
}

void CegoAdminHandler::getHostRole(Chain& hostRole)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	hostRole = pRoot->getAttributeValue(XML_HOSTROLE_ATTR);
    }
}

void CegoAdminHandler::getHostName(Chain& hostName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	hostName = pRoot->getAttributeValue(XML_HOSTNAME_ATTR);
    }
}

void CegoAdminHandler::getHostStatus(Chain& hostStatus)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	hostStatus = pRoot->getAttributeValue(XML_STATUS_ATTR);
    }
}

void CegoAdminHandler::getDataPort(unsigned& dataPort)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	dataPort = pRoot->getAttributeValue(XML_DATAPORT_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getAdminPort(unsigned& adminPort)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	adminPort = pRoot->getAttributeValue(XML_ADMINPORT_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getArchId(Chain& archId)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	archId = pRoot->getAttributeValue(XML_ARCHID_ATTR);
    }
}

void CegoAdminHandler::getArchPath(Chain& archPath)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	archPath = pRoot->getAttributeValue(XML_ARCHPATH_ATTR);
    }
}

void CegoAdminHandler::getMode(Chain& mode)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	mode = pRoot->getAttributeValue(XML_MODE_ATTR);
    }
}

bool CegoAdminHandler::getCleanup()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( (Chain)pRoot->getAttributeValue(XML_CLEANUP_ATTR) == Chain(XML_TRUE_VALUE) )
	    return true;
	else
	    return false;
    }
    else
    {
	throw Exception(EXLOC, "No root element found");
    }
}

bool CegoAdminHandler::getForceload()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( (Chain)pRoot->getAttributeValue(XML_FORCELOAD_ATTR) == Chain(XML_TRUE_VALUE) )
	    return true;
	else
	    return false;
    }
    else
    {
	throw Exception(EXLOC, "No root element found");
    }
}

bool CegoAdminHandler::getCPDump()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( (Chain)pRoot->getAttributeValue(XML_CPDUMP_ATTR) == Chain(XML_TRUE_VALUE) )
	    return true;
	else
	    return false;
    }
    else
    {
	throw Exception(EXLOC, "No root element found");
    }
}

bool CegoAdminHandler::getNoInit()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	if ( (Chain)pRoot->getAttributeValue(XML_NOINIT_ATTR) == Chain(XML_TRUE_VALUE) )
	    return true;
	else
	    return false;
    }
    else
    {
	throw Exception(EXLOC, "No root element found");
    }
}

void CegoAdminHandler::getObjName(Chain& objName)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
    }    
}

void CegoAdminHandler::getObjType(CegoObject::ObjectType& type)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	Chain objType = pRoot->getAttributeValue(XML_OBJTYPE_ATTR);
	type =  CegoTypeConverter::getObjectTypeId(objType);
    }
}

void CegoAdminHandler::getTableCacheFilter(Chain& cacheFilter)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	cacheFilter = pRoot->getAttributeValue(XML_TABLECACHEFILTER_ATTR);
    }
}

void CegoAdminHandler::getTableCacheMaxEntry(unsigned& maxEntry)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	maxEntry = pRoot->getAttributeValue(XML_TABLECACHEMAXENTRY_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTableCacheMaxSize(unsigned& maxSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	maxSize = pRoot->getAttributeValue(XML_TABLECACHEMAXSIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getTableCacheHashRange(unsigned& hashRange)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	hashRange = pRoot->getAttributeValue(XML_TABLECACHEHASHRANGE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getQueryCacheMaxEntry(unsigned& maxEntry)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	maxEntry = pRoot->getAttributeValue(XML_QUERYCACHEMAXENTRY_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getQueryCacheMaxSize(unsigned& maxSize)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	maxSize = pRoot->getAttributeValue(XML_QUERYCACHEMAXSIZE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getQueryCacheThreshold(unsigned& threshold)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	threshold = pRoot->getAttributeValue(XML_QUERYCACHETHRESHOLD_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getQueryCacheHashRange(unsigned& hashRange)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	hashRange = pRoot->getAttributeValue(XML_QUERYCACHEHASHRANGE_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getNumLast(unsigned& numLast)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	numLast = pRoot->getAttributeValue(XML_NUMQUERYLAST_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::getNumCost(unsigned& numCost)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {	
	numCost = pRoot->getAttributeValue(XML_NUMQUERYCOST_ATTR).asUnsigned();
    }
}

void CegoAdminHandler::sendInfo(const Chain& info)
{
    _xml.getDocument()->clear();
    _xml.getDocument()->setDocType(XML_INFO_DOC);

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_MSG_ATTR, info);
    
    _xml.getDocument()->setRootElement(pRoot);
    
    Chain xmlString;
    _xml.getXMLChain(xmlString);

    _xml.getDocument()->clear();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, xmlString);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _pN->setMsg(xmlString, xmlString.length());
    _pN->writeMsg();
    if ( _pN->recvAck() == false )
	throw Exception(EXLOC, Chain("Admin request aborted"));    
}

void CegoAdminHandler::sendResponse(const Chain& msg, Element *pSubElement)
{
    _xml.getDocument()->clear();
    _xml.getDocument()->setDocType(XML_OK_DOC);

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_MSG_ATTR, msg);
    
    if ( pSubElement )
	pRoot->addContent(pSubElement);

    _xml.getDocument()->setRootElement(pRoot);
    
    Chain xmlString;
    _xml.getXMLChain(xmlString);

    _xml.getDocument()->clear();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, xmlString);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif
    
    _pN->setMsg(xmlString, xmlString.length());
    _pN->writeMsg();
}

void CegoAdminHandler::sendData(const Chain& data)
{
    _xml.getDocument()->clear();
    _xml.getDocument()->setDocType(XML_OK_DOC);

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_DATALEN_ATTR, Chain(data.length()));
    
    _xml.getDocument()->setRootElement(pRoot);

    Chain xmlString;
    _xml.getXMLChain(xmlString);

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, xmlString);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _pN->setMsg(xmlString, xmlString.length());

    _pN->writeMsg();

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Waiting for ack ..."));
#endif

    _pN->recvAck();
    
#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Sending data of size ") + Chain(data.length()) + Chain(" ..."));
#endif

    _pN->setMsg(data, data.length());

    _pN->writeMsg();      
}

CegoAdminHandler::ResultType CegoAdminHandler::abort()
{
    _pN->sendNack();
    _pN->readMsg();

    _xml.getDocument()->clear();   
    _xml.setChain( _pN->getMsg() );
    _xml.parse();

    Chain docType = _xml.getDocument()->getDocType();
    
    if ( docType == Chain(XML_OK_DOC) )
    {
	return ADM_OK;
    }
    else if ( docType == Chain(XML_ERROR_DOC) )
    {
	return ADM_ERROR;
    }
    else if ( docType == Chain(XML_INFO_DOC) )
    {
	return ADM_INFO;
    }
    else
    {
	throw Exception(EXLOC, "Invalid document type");
    }
}

CegoAdminHandler::ResultType CegoAdminHandler::nextInfo()
{
    _pN->sendAck();
    _pN->readMsg();

    _xml.getDocument()->clear();
    _xml.setChain( _pN->getMsg() );
    _xml.parse();

    Chain docType = _xml.getDocument()->getDocType();
    
    if ( docType == Chain(XML_OK_DOC) )
    {
	return ADM_OK;
    }
    else if ( docType == Chain(XML_ERROR_DOC) )
    {
	return ADM_ERROR;
    }
    else if ( docType == Chain(XML_INFO_DOC) )
    {
	return ADM_INFO;
    }
    else
    {
	throw Exception(EXLOC, "Invalid document type");
    }
}

bool CegoAdminHandler::syncWithInfo(const Chain& hostRole, const Chain& hostName, const Chain& msg, Element *pInfo)
{
    _xml.getDocument()->clear();

    _xml.getDocument()->setDocType(XML_INFO_DOC);

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_HOSTROLE_ATTR, hostRole);
    pRoot->setAttribute(XML_HOSTNAME_ATTR, hostName);
    pRoot->setAttribute(XML_MSG_ATTR, msg);
    if ( pInfo )
	pRoot->addContent(pInfo);
    
    _xml.getDocument()->setRootElement(pRoot);
    
    Chain xmlString;
    _xml.getXMLChain(xmlString);

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, xmlString);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif    

    _pN->setMsg(xmlString, xmlString.length());
        
    _pN->writeMsg();

    _xml.getDocument()->clear();
    
    return _pN->recvAck();
}

void CegoAdminHandler::sendError(const Chain& msg)
{
    _xml.getDocument()->clear();
    _xml.getDocument()->setDocType(XML_ERROR_DOC);

    Element* pRoot = new Element(XML_FRAME_ELEMENT);
    pRoot->setAttribute(XML_MSG_ATTR, msg);

    _xml.getDocument()->setRootElement(pRoot);

    Chain xmlString;
    _xml.getXMLChain(xmlString);

#ifdef CGDEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
    _pModule->log(_modId, Logger::DEBUG, xmlString);
    _pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif

    _pN->setMsg(xmlString, xmlString.length());
    
    _pN->writeMsg();   
}

void CegoAdminHandler::getThreadInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{    
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> threadInfoList = pRoot->getChildren(XML_THREADINFO_ELEMENT);
	
	Element **pInfo = threadInfoList.First();

	if ( pInfo )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("TYPE"),VARCHAR_TYPE, 20)); 
	    schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("TOTAL"), INT_TYPE, sizeof(int))); 
	    schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("ACTIVE"), INT_TYPE, sizeof(int))); 
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("THREADINFO"), schema, Chain("THREADINFO"));
	       
	    Chain numDbThread = (*pInfo)->getAttributeValue(XML_NUMDBTHREAD_ATTR);
	    Chain numAdmThread = (*pInfo)->getAttributeValue(XML_NUMADMTHREAD_ATTR);
	    Chain actDbThread = (*pInfo)->getAttributeValue(XML_ACTIVEDBTHREAD_ATTR);
	    Chain actAdmThread = (*pInfo)->getAttributeValue(XML_ACTIVEADMTHREAD_ATTR);
	    	    
	    CegoFieldValue f1a(VARCHAR_TYPE, Chain("DatabaseThread"));
	    CegoFieldValue f1b(INT_TYPE,  numDbThread);
	    CegoFieldValue f1c(INT_TYPE,  actDbThread);
	    ListT<CegoFieldValue> fl1;
	    fl1.Insert(f1a);
	    fl1.Insert(f1b);
	    fl1.Insert(f1c);
	    
	    info.Insert(fl1);
	    
	    CegoFieldValue f2a(VARCHAR_TYPE, Chain("AdminThread"));
	    CegoFieldValue f2b(INT_TYPE,  numAdmThread);
	    CegoFieldValue f2c(INT_TYPE,  actAdmThread);
	    ListT<CegoFieldValue> fl2;
	    fl2.Insert(f2a);
	    fl2.Insert(f2b);
	    fl2.Insert(f2c);
	    
	    info.Insert(fl2);
	    
	}
    }
}

void CegoAdminHandler::getDbThreadInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("THRID"), VARCHAR_TYPE, 3)); 
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("NUMREQUEST"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("NUMQUERYREQ"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("LOAD"), VARCHAR_TYPE, 3)); 
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("ALLOCATEDSORT"), VARCHAR_TYPE, 15));
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("LASTACTION"), VARCHAR_TYPE, 50)); 
	schema.Insert(CegoField(Chain("THRINFO"), Chain("THRINFO"), Chain("STATUS"), VARCHAR_TYPE, 10));
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("THRINFO"), schema, Chain("THRINFO"));
	
	format = Chain("rrrrrrl");

	ListT<Element*> threadInfoList = pRoot->getChildren(XML_THREADINFO_ELEMENT);
	
	Element **pInfo = threadInfoList.First();

	if ( pInfo )
	{	    
	    ListT<Element*> threadStateList = (*pInfo)->getChildren(XML_THREAD_ELEMENT);
	    Element **pThreadState = threadStateList.First();
	       
	    while ( pThreadState )
	    {
		Chain threadId = (*pThreadState)->getAttributeValue(XML_THID_ATTR);
		Chain numRequest = (*pThreadState)->getAttributeValue(XML_NUMREQUEST_ATTR);
		Chain numQueryRequest = (*pThreadState)->getAttributeValue(XML_NUMQUERYREQUEST_ATTR);
		Chain threadLoad = (*pThreadState)->getAttributeValue(XML_THREADLOAD_ATTR);
		Chain allocatedSort = (*pThreadState)->getAttributeValue(XML_ALLOCATEDSORT_ATTR);
		Chain lastAction = (*pThreadState)->getAttributeValue(XML_LASTACTION_ATTR);
		Chain threadState = (*pThreadState)->getAttributeValue(XML_STATUS_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, threadId);
		CegoFieldValue f2(VARCHAR_TYPE, numRequest);
		CegoFieldValue f3(VARCHAR_TYPE, numQueryRequest);
		CegoFieldValue f4(VARCHAR_TYPE, threadLoad);
		CegoFieldValue f5(VARCHAR_TYPE, allocatedSort);
		CegoFieldValue f6(VARCHAR_TYPE, lastAction);
		
		CegoFieldValue f7(VARCHAR_TYPE, threadState);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		fl.Insert(f5);
		fl.Insert(f6);
		fl.Insert(f7);
				
		info.Insert(fl);
		
		pThreadState = threadStateList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getDbThreadHistory(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> threadHistList = pRoot->getChildren(XML_QUERYHIST_ELEMENT);
	
	Element **pHist = threadHistList.First();

	if ( pHist )
	{	    
	    ListT<Element*> queryList = (*pHist)->getChildren(XML_QUERY_ELEMENT);
	    Element **pQuery = queryList.First();
	    
	    unsigned int maxQueryLen = 10;
	    
	    while ( pQuery ) 
	    {
		Chain queryString = (*pQuery)->getAttributeValue(XML_QUERY_ATTR);
		
		if ( queryString.length() > maxQueryLen )
		    maxQueryLen = queryString.length();
		
		pQuery = queryList.Next();
	    }
	    
	    pQuery = queryList.First();
	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("THREADHIST"), Chain("THREADHIST"), Chain("THRIDX"), VARCHAR_TYPE, 10));
	    schema.Insert(CegoField(Chain("THREADHIST"), Chain("THREADHIST"), Chain("TIMESTAMP"), VARCHAR_TYPE, 20));
	    schema.Insert(CegoField(Chain("THREADHIST"), Chain("THREADHIST"), Chain("QUERY"), VARCHAR_TYPE, maxQueryLen));
	    schema.Insert(CegoField(Chain("THREADHIST"), Chain("THREADHIST"), Chain("COST"), VARCHAR_TYPE, 10));
	    
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("THREADHIST"), schema, Chain("THREADHIST"));
	    
	    format = Chain("rrlr");
	    
	    while ( pQuery ) 
	    {
		Chain threadIdx = (*pQuery)->getAttributeValue(XML_THRIDX_ATTR);
		Chain queryTS = (*pQuery)->getAttributeValue(XML_TIMESTAMP_ATTR);
		Chain queryString = (*pQuery)->getAttributeValue(XML_QUERY_ATTR);
		Chain queryCost = (*pQuery)->getAttributeValue(XML_COST_ATTR);

		CegoFieldValue f1(VARCHAR_TYPE, threadIdx);
		CegoFieldValue f2(VARCHAR_TYPE, queryTS);
		CegoFieldValue f3(VARCHAR_TYPE, queryString);
		CegoFieldValue f4(VARCHAR_TYPE, queryCost);
				
		ListT<CegoFieldValue> fl;

		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
				     	  
		info.Insert(fl);
	    
		pQuery = queryList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getAdmThreadInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("THREADID"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("NUMREQUEST"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("THREADLOAD"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("STATUS"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("THREADINFO"), Chain("THREADINFO"), Chain("LASTACTION"), VARCHAR_TYPE, 50));
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("THREADINFO"), schema, Chain("THREADINFO"));
	
	format = Chain("rrrrl");

	ListT<Element*> threadInfoList = pRoot->getChildren(XML_THREADINFO_ELEMENT);
	
	Element **pInfo = threadInfoList.First();

	if ( pInfo )
	{	    
	    ListT<Element*> threadStateList = (*pInfo)->getChildren(XML_THREAD_ELEMENT);
	    Element **pThreadState = threadStateList.First();
	    	       
	    while ( pThreadState ) 
	    {
		Chain threadId = (*pThreadState)->getAttributeValue(XML_THID_ATTR);
		Chain numRequest = (*pThreadState)->getAttributeValue(XML_NUMREQUEST_ATTR);
		Chain threadLoad = (*pThreadState)->getAttributeValue(XML_THREADLOAD_ATTR);
		Chain threadState = (*pThreadState)->getAttributeValue(XML_STATUS_ATTR);
		Chain threadAction = (*pThreadState)->getAttributeValue(XML_LASTACTION_ATTR);
		
		CegoFieldValue f1(LONG_TYPE, threadId);
		CegoFieldValue f2(LONG_TYPE, numRequest);
		CegoFieldValue f3(LONG_TYPE, threadLoad);
		CegoFieldValue f4(VARCHAR_TYPE, threadState);
		CegoFieldValue f5(VARCHAR_TYPE, threadAction);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		fl.Insert(f5);
		
		info.Insert(fl);
		
		pThreadState = threadStateList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getDbSessionInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("DBSESSION"), Chain("DBSESSION"), Chain("HOST"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("DBSESSION"), Chain("DBSESSION"), Chain("TABLESET"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("DBSESSION"), Chain("DBSESSION"), Chain("USER"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("DBSESSION"), Chain("DBSESSION"), Chain("ISUSED"), VARCHAR_TYPE, 5)); 
	schema.Insert(CegoField(Chain("DBSESSION"), Chain("DBSESSION"), Chain("TTL"), VARCHAR_TYPE, 5)); 
	
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("DBSESSIONINFO"), schema, Chain("DBSESSIONINFO"));
	    
	ListT<Element*> sessionInfoList = pRoot->getChildren(XML_DBSESSIONINFO_ELEMENT);	
	Element **pInfo = sessionInfoList.First();

	if ( pInfo )
	{
	    ListT<Element*> sessionList = (*pInfo)->getChildren(XML_DBSESSION_ELEMENT);
	    Element **pSession = sessionList.First();
	    
	    while ( pSession ) 
	    {
		Chain hostName = (*pSession)->getAttributeValue(XML_HOSTNAME_ATTR);
		Chain tableSet = (*pSession)->getAttributeValue(XML_TABLESET_ATTR);
		Chain userName = (*pSession)->getAttributeValue(XML_USER_ATTR);
		Chain isUsed = (*pSession)->getAttributeValue(XML_ISUSED_ATTR);
		Chain ttl = (*pSession)->getAttributeValue(XML_TTL_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, hostName);
		CegoFieldValue f2(VARCHAR_TYPE, tableSet);
		CegoFieldValue f3(VARCHAR_TYPE, userName);
		CegoFieldValue f4(VARCHAR_TYPE, isUsed);
		CegoFieldValue f5(VARCHAR_TYPE, ttl);
			
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		fl.Insert(f5);
		
		info.Insert(fl);
		
		pSession = sessionList.Next();
	    }
	}
    }
}


bool CegoAdminHandler::getCacheInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("ATTR"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("VALUE"), VARCHAR_TYPE, 10));
	
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("CACHEINFO"), schema, Chain("CACHEINFO"));
	
	format = Chain("lr");

	ListT<Element*> cacheInfoList = pRoot->getChildren(XML_CACHEINFO_ELEMENT);
	
	Element **pInfo = cacheInfoList.First();

	if ( pInfo )
	{
	    ListT<Element*> cacheList = (*pInfo)->getChildren(XML_CACHE_ELEMENT);
	    Element **pCache = cacheList.First();
		    
	    while ( pCache ) 
	    {		
		Chain attrName = (*pCache)->getAttributeValue(XML_ATTRNAME_ATTR);
		Chain attrValue = (*pCache)->getAttributeValue(XML_VALUE_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, attrName);
		CegoFieldValue f2(VARCHAR_TYPE, attrValue);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		
		info.Insert(fl);
		
		pCache = cacheList.Next();
	    }
	    return true;
	}
    }
    return false;
}

bool CegoAdminHandler::getCacheList(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> cacheInfoList = pRoot->getChildren(XML_CACHEINFO_ELEMENT);
	
	Element **pInfo = cacheInfoList.First();

	if ( pInfo )
	{
	    ListT<Element*> cacheList = (*pInfo)->getChildren(XML_CACHE_ELEMENT);
	    Element **pCache = cacheList.First();

	    unsigned maxLen=0;
	    while ( pCache ) 
	    {		
		Chain id = (*pCache)->getAttributeValue(XML_ID_ATTR);
		if ( (unsigned)id.length() > maxLen )
		    maxLen = id.length();
		
		pCache = cacheList.Next();
	    }
	
	    if ( maxLen > INFO_MAXCACHEIDLEN )
		maxLen = INFO_MAXCACHEIDLEN;
	        
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("POS"), VARCHAR_TYPE, 10)); 
	    schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("ID"), VARCHAR_TYPE, maxLen)); 
	    schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("NUMROWS"), VARCHAR_TYPE, 10));
	    schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("NUMHITS"), VARCHAR_TYPE, 10));
	    schema.Insert(CegoField(Chain("CACHE"), Chain("CACHE"), Chain("SIZE"), VARCHAR_TYPE, 10)); 
	    
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("CACHEINFO"), schema, Chain("CACHEINFO"));
	    
	    format = Chain("lrrrr");

	    pCache = cacheList.First();
		    
	    while ( pCache ) 
	    {
		Chain pos = (*pCache)->getAttributeValue(XML_POS_ATTR);
		Chain id = (*pCache)->getAttributeValue(XML_ID_ATTR);
		if ( (unsigned)id.length() > maxLen )
		    id = id.subChain(1, maxLen - 3) + Chain("...");
		Chain numRows = (*pCache)->getAttributeValue(XML_NUMROWS_ATTR);
		Chain numHits = (*pCache)->getAttributeValue(XML_NUMHITS_ATTR);
		Chain entrySize = (*pCache)->getAttributeValue(XML_SIZE_ATTR);

		CegoFieldValue f1(VARCHAR_TYPE, pos);
		CegoFieldValue f2(VARCHAR_TYPE, id);
		CegoFieldValue f3(VARCHAR_TYPE, numRows);
		CegoFieldValue f4(VARCHAR_TYPE, numHits);
		CegoFieldValue f5(VARCHAR_TYPE, entrySize);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		fl.Insert(f5);
		
		info.Insert(fl);
		
		pCache = cacheList.Next();
	    }
	    return true;
	}
    }
    return false;
}

void CegoAdminHandler::getLockInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{    
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("LOCKID"),VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("LOCKCOUNT"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("RD HITS"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("RD DELAY"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("WR HITS"),  LONG_TYPE, sizeof(long long)));  
	schema.Insert(CegoField(Chain("LOCKINFO"), Chain("LOCKINFO"), Chain("WR DELAY"), LONG_TYPE, sizeof(long long)));  
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("LOCKINFO"), schema, Chain("LOCKINFO"));
	    
	ListT<Element*> lockInfoList = pRoot->getChildren(XML_LOCKINFO_ELEMENT);	
	Element **pInfo = lockInfoList.First();

	if ( pInfo )
	{	    
	    ListT<Element*> lockEntryList = (*pInfo)->getChildren(XML_LOCKENTRY_ELEMENT);
	    Element **pLockEntry = lockEntryList.First();
	    
	    while ( pLockEntry )
	    {	
		CegoFieldValue f1a(VARCHAR_TYPE, (*pLockEntry)->getAttributeValue(XML_LOCKID_ATTR));
		CegoFieldValue f1b(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_LOCKCOUNT_ATTR));
		CegoFieldValue f1c(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_RDLOCKHIT_ATTR));
		CegoFieldValue f1d(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_RDLOCKDELAY_ATTR));
		CegoFieldValue f1e(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_WRLOCKHIT_ATTR));
		CegoFieldValue f1f(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_WRLOCKDELAY_ATTR));
		ListT<CegoFieldValue> fl;
		fl.Insert(f1a);
		fl.Insert(f1b);
		fl.Insert(f1c);
		fl.Insert(f1d);	    
		fl.Insert(f1e);
		fl.Insert(f1f);
		
		info.Insert(fl);
		
		pLockEntry = lockEntryList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getLockStat(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("LOCKID"),VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("NUMLOCK"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("LOCKCOUNT"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("RD HITS"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("RD DELAY"), LONG_TYPE, sizeof(long long))); 
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("WR HITS"),  LONG_TYPE, sizeof(long long)));  
	schema.Insert(CegoField(Chain("LOCKSTAT"), Chain("LOCKSTAT"), Chain("WR DELAY"), LONG_TYPE, sizeof(long long)));  
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("LOCKSTAT"), schema, Chain("LOCKSTAT"));

	ListT<Element*> lockInfoList = pRoot->getChildren(XML_LOCKSTAT_ELEMENT);	
	Element **pInfo = lockInfoList.First();

	if ( pInfo )
	{	    
	    ListT<Element*> lockEntryList = (*pInfo)->getChildren(XML_LOCKENTRY_ELEMENT);
	    Element **pLockEntry = lockEntryList.First();
	    
	    while ( pLockEntry )
	    {	
		CegoFieldValue f1a(VARCHAR_TYPE, (*pLockEntry)->getAttributeValue(XML_LOCKID_ATTR));
		CegoFieldValue f1b(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_NUMLOCK_ATTR));
		CegoFieldValue f1c(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_LOCKCOUNT_ATTR));
		CegoFieldValue f1d(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_RDLOCKHIT_ATTR));
		CegoFieldValue f1e(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_RDLOCKDELAY_ATTR));
		CegoFieldValue f1f(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_WRLOCKHIT_ATTR));
		CegoFieldValue f1g(LONG_TYPE, (*pLockEntry)->getAttributeValue(XML_WRLOCKDELAY_ATTR));
		ListT<CegoFieldValue> fl;
		fl.Insert(f1a);
		fl.Insert(f1b);
		fl.Insert(f1c);
		fl.Insert(f1d);	    
		fl.Insert(f1e);
		fl.Insert(f1f);
		fl.Insert(f1g);
			
		info.Insert(fl);
		
		pLockEntry = lockEntryList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getPoolInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("POOLINFO"), Chain("POOLINFO"), Chain("PARAMETER"),VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("POOLINFO"), Chain("POOLINFO"), Chain("VALUE"), VARCHAR_TYPE, 20)); 
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("POOLINFO"), schema, Chain("POOLINFO"));
	
	ListT<Element*> poolInfoList = pRoot->getChildren(XML_POOLINFO_ELEMENT);
	
	Element **pInfo = poolInfoList.First();

	if ( pInfo )
	{	        
	    Chain pageSize = (*pInfo)->getAttributeValue(XML_PAGESIZE_ATTR);
	    Chain numTotal = (*pInfo)->getAttributeValue(XML_NUMTOTAL_ATTR);
	    Chain numUsed = (*pInfo)->getAttributeValue(XML_NUMUSED_ATTR);
	    Chain numFree = (*pInfo)->getAttributeValue(XML_NUMFREE_ATTR);
	    Chain numDirty = (*pInfo)->getAttributeValue(XML_NUMDIRTY_ATTR);
	    Chain numFixed = (*pInfo)->getAttributeValue(XML_NUMFIXES_ATTR);
	    
	    Chain numPersistent = (*pInfo)->getAttributeValue(XML_NUMPERSISTENT_ATTR);
	    Chain numNoSync = (*pInfo)->getAttributeValue(XML_NUMNOSYNC_ATTR);
	    Chain numDiskRead = (*pInfo)->getAttributeValue(XML_NUMDISKREAD_ATTR);
	    Chain numDiskWrite = (*pInfo)->getAttributeValue(XML_NUMDISKWRITE_ATTR);
	    
	    unsigned long long readDelay = (*pInfo)->getAttributeValue(XML_READDELAY_ATTR).asUnsignedLongLong();
	    unsigned long long writeDelay = (*pInfo)->getAttributeValue(XML_WRITEDELAY_ATTR).asUnsignedLongLong();
	    
	    Chain hitRate = (*pInfo)->getAttributeValue(XML_BPHITRATE_ATTR);
	    Chain spreadRate = (*pInfo)->getAttributeValue(XML_SPREADRATE_ATTR);
	    
	    Chain curFixCount = (*pInfo)->getAttributeValue(XML_CURFIXCOUNT_ATTR);
	    Chain maxFixCount = (*pInfo)->getAttributeValue(XML_MAXFIXCOUNT_ATTR);

	    Chain avgFixTry = (*pInfo)->getAttributeValue(XML_AVGFIXTRY_ATTR);
	    
	    Chain statStart = (*pInfo)->getAttributeValue(XML_STATSTART_ATTR);
	    unsigned long long uptime = (*pInfo)->getAttributeValue(XML_UPTIME_ATTR).asUnsignedLongLong();
	    
	    CegoFieldValue f1a(VARCHAR_TYPE, Chain("Page Size"));
	    CegoFieldValue f1b(VARCHAR_TYPE,  pageSize);
	    ListT<CegoFieldValue> fl1;
	    fl1.Insert(f1a);
	    fl1.Insert(f1b);
	    info.Insert(fl1);
	    
	    CegoFieldValue f2a(VARCHAR_TYPE, Chain("Total Pages"));
	    CegoFieldValue f2b(VARCHAR_TYPE,  numTotal);
	    ListT<CegoFieldValue> fl2;
	    fl2.Insert(f2a);
	    fl2.Insert(f2b);
	    info.Insert(fl2);
	    
	    CegoFieldValue f3a(VARCHAR_TYPE, Chain("Used Pages"));
	    CegoFieldValue f3b(VARCHAR_TYPE, numUsed);
	    ListT<CegoFieldValue> fl3;
	    fl3.Insert(f3a);
	    fl3.Insert(f3b);
	    info.Insert(fl3);
	    
	    CegoFieldValue f4a(VARCHAR_TYPE, Chain("Free Pages"));
	    CegoFieldValue f4b(VARCHAR_TYPE, numFree);
	    ListT<CegoFieldValue> fl4;
	    fl4.Insert(f4a);
	    fl4.Insert(f4b);
	    info.Insert(fl4);
	    
	    CegoFieldValue f5a(VARCHAR_TYPE, Chain("Dirty Pages"));
	    CegoFieldValue f5b(VARCHAR_TYPE, numDirty);
	    ListT<CegoFieldValue> fl5;
	    fl5.Insert(f5a);
	    fl5.Insert(f5b);
	    info.Insert(fl5);
	    
	    CegoFieldValue f6a(VARCHAR_TYPE, Chain("Fixed Pages"));
	    CegoFieldValue f6b(VARCHAR_TYPE, numFixed);
	    ListT<CegoFieldValue> fl6;
	    fl6.Insert(f6a);
	    fl6.Insert(f6b);
	    info.Insert(fl6);
	    
	    CegoFieldValue f7a(VARCHAR_TYPE, Chain("Persistent Pages"));
	    CegoFieldValue f7b(VARCHAR_TYPE, numPersistent);
	    ListT<CegoFieldValue> fl7;
	    fl7.Insert(f7a);
	    fl7.Insert(f7b);
	    info.Insert(fl7);
	    
	    CegoFieldValue f8a(VARCHAR_TYPE, Chain("No Sync Pages"));
	    CegoFieldValue f8b(INT_TYPE, numNoSync);
	    ListT<CegoFieldValue> fl8;
	    fl8.Insert(f8a);
	    fl8.Insert(f8b);
	    info.Insert(fl8);
	    
	    CegoFieldValue f9a(VARCHAR_TYPE, Chain("Spread Rate"));
	    CegoFieldValue f9b(VARCHAR_TYPE, spreadRate);
	    ListT<CegoFieldValue> fl9;
	    fl9.Insert(f9a);
	    fl9.Insert(f9b);
	    info.Insert(fl9);

	    CegoFieldValue f10a(VARCHAR_TYPE, Chain("Stat Start"));
	    CegoFieldValue f10b(VARCHAR_TYPE, statStart);
	    ListT<CegoFieldValue> fl10;
	    fl10.Insert(f10a);
	    fl10.Insert(f10b);
	    info.Insert(fl10);
	    
	    CegoFieldValue f11a(VARCHAR_TYPE, Chain("Hit Rate"));
	    CegoFieldValue f11b(VARCHAR_TYPE, hitRate);
	    ListT<CegoFieldValue> fl11;
	    fl11.Insert(f11a);
	    fl11.Insert(f11b);
	    info.Insert(fl11);
   
	    CegoFieldValue f12a(VARCHAR_TYPE, Chain("Cur Fix Count"));
	    CegoFieldValue f12b(VARCHAR_TYPE, curFixCount);
	    ListT<CegoFieldValue> fl12;
	    fl12.Insert(f12a);
	    fl12.Insert(f12b);
	    info.Insert(fl12);
	    
	    CegoFieldValue f13a(VARCHAR_TYPE, Chain("Max Fix Count"));
	    CegoFieldValue f13b(VARCHAR_TYPE, maxFixCount);
	    ListT<CegoFieldValue> fl13;
	    fl13.Insert(f13a);
	    fl13.Insert(f13b);
	    info.Insert(fl13);

	    CegoFieldValue f14a(VARCHAR_TYPE, Chain("Avg Fix Try"));
	    CegoFieldValue f14b(VARCHAR_TYPE, avgFixTry);
	    ListT<CegoFieldValue> fl14;
	    fl14.Insert(f14a);
	    fl14.Insert(f14b);
	    info.Insert(fl14);

	    
	    CegoFieldValue f15a(VARCHAR_TYPE, Chain("Disk Reads"));
	    CegoFieldValue f15b(VARCHAR_TYPE, Chain(numDiskRead));
	    ListT<CegoFieldValue> fl15;
	    fl15.Insert(f15a);
	    fl15.Insert(f15b);
	    info.Insert(fl15);
	    
	    CegoFieldValue f16a(VARCHAR_TYPE, Chain("Disk Writes"));
	    CegoFieldValue f16b(VARCHAR_TYPE, Chain(numDiskWrite));
	    ListT<CegoFieldValue> fl16;
	    fl16.Insert(f16a);
	    fl16.Insert(f16b);
	    info.Insert(fl16);
	    
	    unsigned long long msecDelay;
	    unsigned long long usecDelay;
	    Chain delayStr;
	    Chain fillStr;
	    
	    msecDelay = readDelay / 1000;
	    usecDelay = readDelay % 1000;
	    fillStr = Chain("000") + Chain(usecDelay);
	    delayStr = Chain(msecDelay) + Chain(".") + fillStr.subChain(fillStr.length()-3, fillStr.length()) + Chain (" msec");
	    
	    CegoFieldValue f17a(VARCHAR_TYPE, Chain("Read Delay"));
	    CegoFieldValue f17b(VARCHAR_TYPE, Chain(delayStr));
	    ListT<CegoFieldValue> fl17;
	    fl17.Insert(f17a);
	    fl17.Insert(f17b);
	    info.Insert(fl17);
	    
	    msecDelay = writeDelay / 1000;
	    usecDelay = writeDelay % 1000;
	    fillStr = Chain("000") + Chain(usecDelay);
	    delayStr = Chain(msecDelay) + Chain(".") + fillStr.subChain(fillStr.length()-3, fillStr.length()) + Chain (" msec");
	    
	    CegoFieldValue f18a(VARCHAR_TYPE, Chain("Write Delay"));
	    CegoFieldValue f18b(VARCHAR_TYPE, Chain(delayStr));
	    ListT<CegoFieldValue> fl18;
	    fl18.Insert(f18a);
	    fl18.Insert(f18b);
	    info.Insert(fl18);
	    
	    // calculate uptime string
	    unsigned long long d = uptime / ( 3600 * 24 );
	    unsigned long long h = ( uptime - ( d * 24 * 3600 )) / 3600;
	    unsigned long long m = ( uptime - ( d * 24 * 3600 ) - ( h * 3600) ) / 60;
	    unsigned long long s = uptime % 60;
	    
	    Chain ss = Chain("0") + Chain(s);
	    Chain sec = ss.subChain(ss.length()-2,ss.length());
	    
	    Chain ms = Chain("0") + Chain(m);
	    Chain min = ms.subChain(ms.length()-2,ms.length());
	    
	    Chain uptimeString = Chain(d) + Chain("d ") + Chain(h) + Chain(":") + min + Chain(":") + sec; 
	    
	    CegoFieldValue f19a(VARCHAR_TYPE, Chain("Pool Uptime"));
	    CegoFieldValue f19b(VARCHAR_TYPE, uptimeString);
	    ListT<CegoFieldValue> fl19;
	    fl19.Insert(f19a);
	    fl19.Insert(f19b);
	    info.Insert(fl19);
	}	    
    }
}

void CegoAdminHandler::getParameterInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("PARAMETERINFO"), Chain("PARAMETERINFO"), Chain("PARAMETER"),VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("PARAMETERINFO"), Chain("PARAMETERINFO"), Chain("VALUE"), VARCHAR_TYPE, 20)); 
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("PARAMETERINFO"), schema, Chain("PARAMETERINFO"));

	ListT<Element*> pList = pRoot->getChildren(XML_PARAMETERLIST_ELEMENT);
	Element **pPL = pList.First();

	if ( pPL )
	{	    	    
	    ListT<Element*> paramInfoList = (*pPL)->getChildren(XML_PARAMETERINFO_ELEMENT);

	    Element **pParam = paramInfoList.First();
	    
	    while ( pParam )
	    {		
		Chain paramName = (*pParam)->getAttributeValue(XML_PARAMNAME_ATTR);
		Chain paramValue = (*pParam)->getAttributeValue(XML_PARAMVALUE_ATTR);
		
		CegoFieldValue f1a(VARCHAR_TYPE, paramName);
		CegoFieldValue f1b(VARCHAR_TYPE, paramValue);
		ListT<CegoFieldValue> fl1;
		fl1.Insert(f1a);
		fl1.Insert(f1b);
		info.Insert(fl1);
		
		pParam = paramInfoList.Next();       
	    }
	}
    }
}

void CegoAdminHandler::getPoolEntryList(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{    
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("SEGMENT"),VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("POS"),VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("OCCSTATE"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("ISDIRTY"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("NUMFIXES"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("TABSETID"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("FILEID"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("PAGEID"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("FIXSTAT"), VARCHAR_TYPE, 15));
	schema.Insert(CegoField(Chain("PEL"), Chain("PEL"), Chain("NUMUSAGE"), VARCHAR_TYPE, 25)); 
	
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("PEL"), schema, Chain("PEL"));

	ListT<Element*> pList = pRoot->getChildren(XML_POOLENTRYLIST_ELEMENT);
	Element **pPL = pList.First();

	if ( pPL )
	{	    
	    ListT<Element*> peList = (*pPL)->getChildren(XML_POOLENTRY_ELEMENT);
	    Element** pPE = peList.First();
	    
	    while ( pPE )
	    {
		CegoFieldValue f1a(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_SEGMENT_ATTR));
		CegoFieldValue f1b(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_POS_ATTR));
		CegoFieldValue f1c(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_OCCSTATE_ATTR));
		CegoFieldValue f1d(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_ISDIRTY_ATTR));
		CegoFieldValue f1e(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_NUMFIXES_ATTR));
		CegoFieldValue f1f(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_TSID_ATTR));
		CegoFieldValue f1g(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_FILEID_ATTR));
		CegoFieldValue f1h(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_PAGEID_ATTR));
		CegoFieldValue f1i(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_FIXSTAT_ATTR));
		CegoFieldValue f1j(VARCHAR_TYPE, (*pPE)->getAttributeValue(XML_NUMUSAGE_ATTR));
		
		ListT<CegoFieldValue> fl;
		fl.Insert(f1a);
		fl.Insert(f1b);
		fl.Insert(f1c);
		fl.Insert(f1d);
		fl.Insert(f1e);
		fl.Insert(f1f);
		fl.Insert(f1g);
		fl.Insert(f1h);
		fl.Insert(f1i);
		fl.Insert(f1j);
		
		info.Insert(fl);
		
		pPE = peList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getTableSetList(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, bool getUsage)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("NAME"),VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("RUNSTATE"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("ARCH"), VARCHAR_TYPE, 5)); 
	if ( getUsage )
	{
	    schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("SYS"), VARCHAR_TYPE, 5)); 
	    schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("TEMP"), VARCHAR_TYPE, 5)); 
	    schema.Insert(CegoField(Chain("TS"), Chain("TS"), Chain("APP"), VARCHAR_TYPE, 5));
	}
	
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("TS"), schema, Chain("TS"));

	ListT<Element*> tList = pRoot->getChildren(XML_TABLESETLIST_ELEMENT);
	Element **pTL = tList.First();

	if ( pTL )
	{	    
	    ListT<Element*> tsList = (*pTL)->getChildren(XML_TABLESET_ELEMENT);
	    Element** pTS = tsList.First();
	    
	    while ( pTS )
	    {	
		Chain tsName = (*pTS)->getAttributeValue(XML_NAME_ATTR);
		Chain tsRunState = (*pTS)->getAttributeValue(XML_RUNSTATE_ATTR);
		Chain tsArchMode = (*pTS)->getAttributeValue(XML_ARCHMODE_ATTR);
				
		CegoFieldValue f1a(VARCHAR_TYPE, tsName);
		CegoFieldValue f1b(VARCHAR_TYPE, tsRunState);
		CegoFieldValue f1d(VARCHAR_TYPE, tsArchMode);
		
		ListT<CegoFieldValue> fl1;
		fl1.Insert(f1a);
		fl1.Insert(f1b);
		fl1.Insert(f1d);
			       
		if ( getUsage )
		{
		    Chain tsSysUsage = (*pTS)->getAttributeValue(XML_SYSUSAGE_ATTR);
		    Chain tsTempUsage = (*pTS)->getAttributeValue(XML_TEMPUSAGE_ATTR);
		    Chain tsAppUsage = (*pTS)->getAttributeValue(XML_APPUSAGE_ATTR);
		    
		    CegoFieldValue f1e(VARCHAR_TYPE, tsSysUsage);
		    CegoFieldValue f1f(VARCHAR_TYPE, tsTempUsage);
		    CegoFieldValue f1g(VARCHAR_TYPE, tsAppUsage);
		    
		    fl1.Insert(f1e);
		    fl1.Insert(f1f);	    
		    fl1.Insert(f1g);		
		}
		
		info.Insert(fl1);
		
		pTS = tsList.Next();
	    }
	}
    }
}



void CegoAdminHandler::getDetailedTableSetInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tsInfoList = pRoot->getChildren(XML_TABLESET_ELEMENT);
	
	Element **pInfo = tsInfoList.First();
	if ( pInfo )
	{	    
	    unsigned maxLen=20;
	    
	    Chain tsRunState = (*pInfo)->getAttributeValue(XML_RUNSTATE_ATTR);
	    	    
	    Chain tsRoot = (*pInfo)->getAttributeValue(XML_TSROOT_ATTR);
	    if ( (unsigned)tsRoot.length() > maxLen )
		maxLen=tsRoot.length();
	    
	    Chain tsTicket = (*pInfo)->getAttributeValue(XML_TSTICKET_ATTR);
	    if ( (unsigned)tsTicket.length() > maxLen )
		maxLen=tsTicket.length();
	    
	    Chain tsInitFile = (*pInfo)->getAttributeValue(XML_TSINITFILE_ATTR);
	    if ( (unsigned)tsInitFile.length() > maxLen )
		maxLen=tsInitFile.length();
	    
	    Chain tsSysFile = (*pInfo)->getAttributeValue(XML_SYSFILE_ATTR);
	    if ( (unsigned)tsSysFile.length() > maxLen )
		maxLen=tsSysFile.length();
	    
	    Chain tsTempFile = (*pInfo)->getAttributeValue(XML_TEMPFILE_ATTR);
	    if ( (unsigned)tsTempFile.length() > maxLen )
		maxLen=tsTempFile.length();
	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("TABLESETINFO"), Chain("TABLESETINFO"), Chain("PARAMETER"),VARCHAR_TYPE, 20)); 
	    schema.Insert(CegoField(Chain("TABLESETINFO"), Chain("TABLESETINFO"), Chain("VALUE"), VARCHAR_TYPE, maxLen)); 
	    
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("TABLESETINFO"), schema, Chain("TABLESETINFO"));
	    
	    Chain tableSet = (*pInfo)->getAttributeValue(XML_NAME_ATTR);
	    
	    Chain checkpoint = (*pInfo)->getAttributeValue(XML_CHECKPOINT_ATTR);
	    
	    Chain tsSysTotal = (*pInfo)->getAttributeValue(XML_SYSTOTAL_ATTR);
	    Chain tsSysUsed = (*pInfo)->getAttributeValue(XML_SYSUSED_ATTR);
	    
	    Chain tsTempTotal = (*pInfo)->getAttributeValue(XML_TEMPTOTAL_ATTR);
	    Chain tsTempUsed = (*pInfo)->getAttributeValue(XML_TEMPUSED_ATTR);
	    
	    unsigned long long tsSysSumTotal = tsSysTotal.asUnsignedLongLong();
	    unsigned long long tsSysSumUsed = tsSysUsed.asUnsignedLongLong();
	    
	    unsigned long long tsTmpSumTotal = tsTempTotal.asUnsignedLongLong();
	    unsigned long long tsTmpSumUsed = tsTempUsed.asUnsignedLongLong();
	    
	    unsigned long long tsAppSumTotal = 0;
	    unsigned long long tsAppSumUsed = 0;
	    
	    ListT<Element*> dataFileList = (*pInfo)->getChildren(XML_DATAFILE_ELEMENT);
	    Element **pDF = dataFileList.First();
	    
	    while ( pDF ) 
	    {
		Chain fileType = (*pDF)->getAttributeValue(XML_TYPE_ATTR);
		
		if ( fileType == Chain(XML_SYSFILE_VALUE) )
		{
		    Chain sfNumTotal = (*pDF)->getAttributeValue(XML_NUMTOTAL_ATTR);		
		    Chain sfNumUsed = (*pDF)->getAttributeValue(XML_NUMUSED_ATTR);
		    
		    tsSysSumTotal += sfNumTotal.asUnsignedLongLong();
		    tsSysSumUsed += sfNumUsed.asUnsignedLongLong();	    
		}
		else if ( fileType == Chain(XML_TEMPFILE_VALUE) )
		{
		    Chain tfNumTotal = (*pDF)->getAttributeValue(XML_NUMTOTAL_ATTR);		
		    Chain tfNumUsed = (*pDF)->getAttributeValue(XML_NUMUSED_ATTR);
		    
		    tsTmpSumTotal += tfNumTotal.asUnsignedLongLong();
		    tsTmpSumUsed += tfNumUsed.asUnsignedLongLong();   
		}
		else if ( fileType == Chain(XML_APPFILE_VALUE) )
		{
		    Chain afNumTotal = (*pDF)->getAttributeValue(XML_NUMTOTAL_ATTR);
		    Chain afNumUsed = (*pDF)->getAttributeValue(XML_NUMUSED_ATTR);
		    
		    tsAppSumTotal += afNumTotal.asUnsignedLongLong();
		    tsAppSumUsed += afNumUsed.asUnsignedLongLong();   
		}
		pDF = dataFileList.Next();
	    }
	    
	    Chain sortAreaSize = (*pInfo)->getAttributeValue(XML_SORTAREASIZE_ATTR);
	    Chain branchId = (*pInfo)->getAttributeValue(XML_BRANCHID_ATTR);
	    Chain clsn = (*pInfo)->getAttributeValue(XML_LSN_ATTR);
	    Chain tid = (*pInfo)->getAttributeValue(XML_TID_ATTR);
	    Chain writtenlsn = (*pInfo)->getAttributeValue(XML_WRITTENLSN_ATTR);
	    
	    Chain archMode = (*pInfo)->getAttributeValue(XML_ARCHMODE_ATTR);
	    Chain autoCorrect = (*pInfo)->getAttributeValue(XML_AUTOCORRECT_ATTR);
	    Chain appendMode = (*pInfo)->getAttributeValue(XML_APPENDMODE_ATTR);
	    
	    Chain cpDump = (*pInfo)->getAttributeValue(XML_CPDUMP_ATTR);

	    Chain logUser = (*pInfo)->getAttributeValue(XML_LOGUSER_ATTR);

	    Chain tableCache = (*pInfo)->getAttributeValue(XML_TABLECACHE_ATTR);
	    Chain queryCache = (*pInfo)->getAttributeValue(XML_QUERYCACHE_ATTR);	    

	    Chain tableCacheFilter = (*pInfo)->getAttributeValue(XML_TABLECACHEFILTER_ATTR);
	    Chain tableCacheMaxSize = (*pInfo)->getAttributeValue(XML_TABLECACHEMAXSIZE_ATTR);

	    Chain tableCacheMaxEntry = (*pInfo)->getAttributeValue(XML_TABLECACHEMAXENTRY_ATTR);
	    Chain tableCacheHashRange = (*pInfo)->getAttributeValue(XML_TABLECACHEHASHRANGE_ATTR);
	
	    // Chain tableCacheUsedSize = (*pInfo)->getAttributeValue(XML_TABLECACHEUSEDSIZE_ATTR);
	    
	    Chain queryCacheMaxEntry = (*pInfo)->getAttributeValue(XML_QUERYCACHEMAXENTRY_ATTR);
	    Chain queryCacheMaxSize = (*pInfo)->getAttributeValue(XML_QUERYCACHEMAXSIZE_ATTR);
	    Chain queryCacheThreshold = (*pInfo)->getAttributeValue(XML_QUERYCACHETHRESHOLD_ATTR);
	    Chain queryCacheHashRange = (*pInfo)->getAttributeValue(XML_QUERYCACHEHASHRANGE_ATTR);
	    // Chain queryCacheUsedNum = (*pInfo)->getAttributeValue(XML_QUERYCACHEUSEDNUM_ATTR);
	    // Chain queryCacheUsedSize = (*pInfo)->getAttributeValue(XML_QUERYCACHEUSEDSIZE_ATTR);
	    // Chain queryCacheNumFail = (*pInfo)->getAttributeValue(XML_QUERYCACHENUMFAIL_ATTR);
	    
	    CegoFieldValue f0a(VARCHAR_TYPE, Chain(INFO_TABLESET));
	    CegoFieldValue f0b(VARCHAR_TYPE, tableSet);
	    ListT<CegoFieldValue> fl0;
	    fl0.Insert(f0a);
	    fl0.Insert(f0b);
	    info.Insert(fl0);
	    
	    CegoFieldValue f1a(VARCHAR_TYPE, Chain(INFO_RUNSTATE));
	    CegoFieldValue f1b(VARCHAR_TYPE, tsRunState);
	    ListT<CegoFieldValue> fl1a;
	    fl1a.Insert(f1a);
	    fl1a.Insert(f1b);
	    info.Insert(fl1a);	    
	    	    	    
	    CegoFieldValue f40a(VARCHAR_TYPE, Chain(INFO_ROOTPATH));
	    CegoFieldValue f40b(VARCHAR_TYPE,  tsRoot);
	    ListT<CegoFieldValue> fl40;
	    fl40.Insert(f40a);
	    fl40.Insert(f40b);
	    info.Insert(fl40);

	    CegoFieldValue f41a(VARCHAR_TYPE, Chain(INFO_LOGUSER));
	    CegoFieldValue f41b(VARCHAR_TYPE,  logUser);
	    ListT<CegoFieldValue> fl41;
	    fl41.Insert(f41a);
	    fl41.Insert(f41b);
	    info.Insert(fl41);
	    
	    CegoFieldValue f42a(VARCHAR_TYPE, Chain(INFO_TICKET));
	    CegoFieldValue f42b(VARCHAR_TYPE,  tsTicket);
	    ListT<CegoFieldValue> fl42;
	    fl42.Insert(f42a);
	    fl42.Insert(f42b);
	    info.Insert(fl42);
	    
	    CegoFieldValue f43a(VARCHAR_TYPE, Chain(INFO_TSINITFILE));
	    CegoFieldValue f43b(VARCHAR_TYPE,  tsInitFile);
	    ListT<CegoFieldValue> fl43;
	    fl43.Insert(f43a);
	    fl43.Insert(f43b);
	    info.Insert(fl43);
	    
	    CegoFieldValue f44a(VARCHAR_TYPE, Chain(INFO_CHECKPOINT));
	    CegoFieldValue f44b(VARCHAR_TYPE,  checkpoint);
	    ListT<CegoFieldValue> fl44;
	    fl44.Insert(f44a);
	    fl44.Insert(f44b);
	    info.Insert(fl44);
	    
	    CegoFieldValue f6a(VARCHAR_TYPE, Chain(INFO_SYSTEMPAGETOTAL));
	    CegoFieldValue f6b(VARCHAR_TYPE, Chain(tsSysSumTotal));
	    ListT<CegoFieldValue> fl6;
	    fl6.Insert(f6a);
	    fl6.Insert(f6b);
	    info.Insert(fl6);
	    
	    CegoFieldValue f7a(VARCHAR_TYPE, Chain(INFO_SYSTEMPAGEUSED));
	    CegoFieldValue f7b(VARCHAR_TYPE, Chain(tsSysSumUsed));
	    ListT<CegoFieldValue> fl7;
	    fl7.Insert(f7a);
	    fl7.Insert(f7b);
	    info.Insert(fl7);
	    
	    // id 8 not used 
	    
	    CegoFieldValue f9a(VARCHAR_TYPE, Chain(INFO_TEMPPAGETOTAL));
	    CegoFieldValue f9b(VARCHAR_TYPE, Chain(tsTmpSumTotal));
	    ListT<CegoFieldValue> fl9;
	    fl9.Insert(f9a);
	    fl9.Insert(f9b);
	    info.Insert(fl9);
	    
	    CegoFieldValue f10a(VARCHAR_TYPE, Chain(INFO_TEMPPAGEUSED));
	    CegoFieldValue f10b(VARCHAR_TYPE, Chain(tsTmpSumUsed));
	    ListT<CegoFieldValue> fl10;
	    fl10.Insert(f10a);
	    fl10.Insert(f10b);
	    info.Insert(fl10);
	    
	    CegoFieldValue f11a(VARCHAR_TYPE, Chain(INFO_APPPAGETOTAL));
	    CegoFieldValue f11b(VARCHAR_TYPE, Chain(tsAppSumTotal));
	    ListT<CegoFieldValue> fl11;
	    fl11.Insert(f11a);
	    fl11.Insert(f11b);
	    info.Insert(fl11);
	    
	    CegoFieldValue f12a(VARCHAR_TYPE, Chain(INFO_APPPAGEUSED));
	    CegoFieldValue f12b(VARCHAR_TYPE, Chain(tsAppSumUsed));
	    ListT<CegoFieldValue> fl12;
	    fl12.Insert(f12a);
	    fl12.Insert(f12b);
	    info.Insert(fl12);
	    
	    CegoFieldValue f13a(VARCHAR_TYPE, Chain(INFO_SORTAREASIZE));
	    CegoFieldValue f13b(VARCHAR_TYPE, sortAreaSize);
	    ListT<CegoFieldValue> fl13;
	    fl13.Insert(f13a);
	    fl13.Insert(f13b);
	    info.Insert(fl13);
	    
	    CegoFieldValue f14a(VARCHAR_TYPE, Chain(INFO_TID));
	    CegoFieldValue f14b(VARCHAR_TYPE, tid);
	    ListT<CegoFieldValue> fl14;
	    fl14.Insert(f14a);
	    fl14.Insert(f14b);
	    info.Insert(fl14);
	    
	    CegoFieldValue f15a(VARCHAR_TYPE, Chain(INFO_CLSN));
	    CegoFieldValue f15b(VARCHAR_TYPE, clsn);
	    ListT<CegoFieldValue> fl15;
	    fl15.Insert(f15a);
	    fl15.Insert(f15b);
	    info.Insert(fl15);
	    
	    CegoFieldValue f16a(VARCHAR_TYPE, Chain(INFO_WLSN));
	    CegoFieldValue f16b(VARCHAR_TYPE, writtenlsn);
	    ListT<CegoFieldValue> fl16;
	    fl16.Insert(f16a);
	    fl16.Insert(f16b);
	    info.Insert(fl16);
	    
	    CegoFieldValue f17a(VARCHAR_TYPE, Chain(INFO_ARCHMODE));
	    CegoFieldValue f17b(VARCHAR_TYPE, archMode);
	    ListT<CegoFieldValue> fl17;
	    fl17.Insert(f17a);
	    fl17.Insert(f17b);
	    info.Insert(fl17);
	    	    
	    CegoFieldValue f18a(VARCHAR_TYPE, Chain(INFO_AUTOCORRECT));
	    CegoFieldValue f18b(VARCHAR_TYPE, autoCorrect);
	    ListT<CegoFieldValue> fl18;
	    fl18.Insert(f18a);
	    fl18.Insert(f18b);
	    info.Insert(fl18);
	    
	    CegoFieldValue f19a(VARCHAR_TYPE, Chain(INFO_APPENDMODE));
	    CegoFieldValue f19b(VARCHAR_TYPE, appendMode);
	    ListT<CegoFieldValue> fl19;
	    fl19.Insert(f19a);
	    fl19.Insert(f19b);
	    info.Insert(fl19);
	    	    
	    CegoFieldValue f20a(VARCHAR_TYPE, Chain(INFO_CPDUMP));
	    CegoFieldValue f20b(VARCHAR_TYPE, cpDump);
	    ListT<CegoFieldValue> fl20;
	    fl20.Insert(f20a);
	    fl20.Insert(f20b);
	    info.Insert(fl20);

	    CegoFieldValue f21a(VARCHAR_TYPE, Chain(INFO_TABLECACHE));
	    CegoFieldValue f21b(VARCHAR_TYPE, tableCache);
	    ListT<CegoFieldValue> fl21;
	    fl21.Insert(f21a);
	    fl21.Insert(f21b);
	    info.Insert(fl21);

	    CegoFieldValue f22a(VARCHAR_TYPE, Chain(INFO_TABLECACHEFILTER));
	    CegoFieldValue f22b(VARCHAR_TYPE, tableCacheFilter);
	    ListT<CegoFieldValue> fl22;
	    fl22.Insert(f22a);
	    fl22.Insert(f22b);
	    info.Insert(fl22);

	    CegoFieldValue f23a(VARCHAR_TYPE, Chain(INFO_TABLECACHEMAXENTRY));
	    CegoFieldValue f23b(VARCHAR_TYPE, tableCacheMaxEntry);
	    ListT<CegoFieldValue> fl23;
	    fl23.Insert(f23a);
	    fl23.Insert(f23b);
	    info.Insert(fl23);

	    CegoFieldValue f24a(VARCHAR_TYPE, Chain(INFO_TABLECACHEMAXSIZE));
	    CegoFieldValue f24b(VARCHAR_TYPE, tableCacheMaxSize);
	    ListT<CegoFieldValue> fl24;
	    fl24.Insert(f24a);
	    fl24.Insert(f24b);
	    info.Insert(fl24);

	    CegoFieldValue f25a(VARCHAR_TYPE, Chain(INFO_TABLECACHEHASHRANGE));
	    CegoFieldValue f25b(VARCHAR_TYPE, tableCacheHashRange);
	    ListT<CegoFieldValue> fl25;
	    fl25.Insert(f25a);
	    fl25.Insert(f25b);
	    info.Insert(fl25);
	    	    
	    CegoFieldValue f26a(VARCHAR_TYPE, Chain(INFO_QUERYCACHE));
	    CegoFieldValue f26b(VARCHAR_TYPE, queryCache);
	    ListT<CegoFieldValue> fl26;
	    fl26.Insert(f26a);
	    fl26.Insert(f26b);
	    info.Insert(fl26);

	    CegoFieldValue f27a(VARCHAR_TYPE, Chain(INFO_QUERYCACHEMAXENTRY));
	    CegoFieldValue f27b(VARCHAR_TYPE, queryCacheMaxEntry);
	    ListT<CegoFieldValue> fl27;
	    fl27.Insert(f27a);
	    fl27.Insert(f27b);
	    info.Insert(fl27);

	    CegoFieldValue f28a(VARCHAR_TYPE, Chain(INFO_QUERYCACHEMAXSIZE));
	    CegoFieldValue f28b(VARCHAR_TYPE, queryCacheMaxSize);
	    ListT<CegoFieldValue> fl28;
	    fl28.Insert(f28a);
	    fl28.Insert(f28b);
	    info.Insert(fl28);

	    CegoFieldValue f29a(VARCHAR_TYPE, Chain(INFO_QUERYCACHETHRESHOLD));
	    CegoFieldValue f29b(VARCHAR_TYPE, queryCacheThreshold);
	    ListT<CegoFieldValue> fl29;
	    fl29.Insert(f29a);
	    fl29.Insert(f29b);
	    info.Insert(fl29);
	    
	    CegoFieldValue f30a(VARCHAR_TYPE, Chain(INFO_QUERYCACHEHASHRANGE));
	    CegoFieldValue f30b(VARCHAR_TYPE, queryCacheHashRange);
	    ListT<CegoFieldValue> fl30;
	    fl30.Insert(f30a);
	    fl30.Insert(f30b);
	    info.Insert(fl30);

	    CegoFieldValue f31a(VARCHAR_TYPE, Chain(INFO_BRANCHID));
	    CegoFieldValue f31b(VARCHAR_TYPE, branchId);
	    ListT<CegoFieldValue> fl31;
	    fl31.Insert(f31a);
	    fl31.Insert(f31b);
	    info.Insert(fl31);

	    ListT<Element*> logFileList = (*pInfo)->getChildren(XML_LOGFILE_ELEMENT);
	    Element **pLF = logFileList.First();
	    
	    if ( pLF )
	    {
		Chain logSize = (*pLF)->getAttributeValue(XML_SIZE_ATTR);
		CegoFieldValue fa(VARCHAR_TYPE, Chain(INFO_LOGSIZE));
		CegoFieldValue fb(VARCHAR_TYPE, logSize);
		ListT<CegoFieldValue> fl;
		fl.Insert(fa);
		fl.Insert(fb);
		info.Insert(fl);		
	    }
	    while ( pLF ) 
	    {
		Chain logName = (*pLF)->getAttributeValue(XML_NAME_ATTR);
		
		CegoFieldValue fa(VARCHAR_TYPE, Chain(INFO_LOGFILE));
		CegoFieldValue fb(VARCHAR_TYPE, logName);
		ListT<CegoFieldValue> fl;
		fl.Insert(fa);
		fl.Insert(fb);
		info.Insert(fl);
		
		pLF = logFileList.Next();
	    }
	}
    }
}
    
Element* CegoAdminHandler::getTSInfo()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tList = pRoot->getChildren(XML_TABLESET_ELEMENT);
	Element **pTL = tList.First();
	if ( pTL ) 
	    return *pTL;
    }
    return 0;
}

Element* CegoAdminHandler::getArchInfo()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> aList = pRoot->getChildren(XML_ARCHINFO_ELEMENT);
	Element **pAI = aList.First();
	if ( pAI ) 
	    return *pAI;
    }
    return 0;
}

Element* CegoAdminHandler::getObjectInfo()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tlist = pRoot->getChildren(XML_OBJLIST_ELEMENT);
	Element **pTL = tlist.First();
	if ( pTL ) 
	    return *pTL;
    }
    return 0;
}

Element* CegoAdminHandler::getTAInfo()
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tlist = pRoot->getChildren(XML_TAINFO_ELEMENT);
	Element **pTL = tlist.First();
	if ( pTL ) 
	    return *pTL;
    }
    return 0;
}

void CegoAdminHandler::getBuFileInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tsInfoList = pRoot->getChildren(XML_TABLESET_ELEMENT);	
	Element **pInfo = tsInfoList.First();

	if (pInfo )
	{	    
	    Chain tsTicket = (*pInfo)->getAttributeValue(XML_TSTICKET_ATTR);
	    unsigned int maxLen=tsTicket.length();
	    
	    CegoFieldValue b1(VARCHAR_TYPE, tsTicket);
	    
	    ListT<CegoFieldValue> bl;
	    
	    bl.Insert(b1);	
	    info.Insert(bl);
	    
	    Chain sysFile = (*pInfo)->getAttributeValue(XML_SYSFILE_ATTR);
	    if ( sysFile.length() > maxLen )
		maxLen=sysFile.length();
	    
	    CegoFieldValue s1(VARCHAR_TYPE, sysFile);
	    
	    ListT<CegoFieldValue> sl;
	    
	    sl.Insert(s1);	
	    info.Insert(sl);
	    
	    Chain tempFile = (*pInfo)->getAttributeValue(XML_TEMPFILE_ATTR);
	    if ( tempFile.length() > maxLen )
		maxLen=tempFile.length();
	    
	    ListT<Element*> dataFileList = (*pInfo)->getChildren(XML_DATAFILE_ELEMENT);
	    Element **pDF = dataFileList.First();
	    
	    while ( pDF ) 
	    {
		Chain fileName = (*pDF)->getAttributeValue(XML_NAME_ATTR);
		if ( fileName.length() > maxLen )
		    maxLen = fileName.length();
		pDF = dataFileList.Next();
	    }
	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("BUFILEINFO"), Chain("BUILEINFO"), Chain("FILENAME"), VARCHAR_TYPE, maxLen)); 	
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("DATAFILEINFO"), schema, Chain("DATAFILEINFO"));
	    
	    CegoFieldValue t1(VARCHAR_TYPE, (*pInfo)->getAttributeValue(XML_TEMPFILE_ATTR));
	    
	    ListT<CegoFieldValue> tl;
	    
	    tl.Insert(t1);
	    
	    info.Insert(tl);
	    
	    pDF = dataFileList.First();
	    while ( pDF ) 
	    {
		Chain fileName = (*pDF)->getAttributeValue(XML_NAME_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, fileName);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		
		info.Insert(fl);
		
		pDF = dataFileList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getDataFileInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info, Chain& format)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tsInfoList = pRoot->getChildren(XML_TABLESET_ELEMENT);	
	Element **pInfo = tsInfoList.First();
	    
	if ( pInfo )
	{	    
	    Chain sysFile = (*pInfo)->getAttributeValue(XML_SYSFILE_ATTR);
	    unsigned int maxLen=sysFile.length();
	    
	    Chain tempFile = (*pInfo)->getAttributeValue(XML_TEMPFILE_ATTR);
	    if ( tempFile.length() > maxLen )
		maxLen=tempFile.length();
	    
	    ListT<Element*> dataFileList = (*pInfo)->getChildren(XML_DATAFILE_ELEMENT);
	    Element **pDF = dataFileList.First();
	    
	    while ( pDF ) 
	    {
		Chain fileName = (*pDF)->getAttributeValue(XML_NAME_ATTR);
		if ( fileName.length() > maxLen )
		    maxLen = fileName.length();
		pDF = dataFileList.Next();
	    }
	    
	    CegoFieldValue s1(VARCHAR_TYPE, sysFile);
	    CegoFieldValue s2(VARCHAR_TYPE, Chain(XML_SYSFILE_ATTR));
	    CegoFieldValue s3(INT_TYPE, (*pInfo)->getAttributeValue(XML_SYSTOTAL_ATTR).asUnsigned());
	    CegoFieldValue s4(INT_TYPE, (*pInfo)->getAttributeValue(XML_SYSUSED_ATTR).asUnsigned());
	    
	    ListT<CegoFieldValue> sl;
	    
	    sl.Insert(s1);
	    sl.Insert(s2);
	    sl.Insert(s3);
	    sl.Insert(s4);
	    
	    info.Insert(sl);
	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("DATAFILEINFO"), Chain("DATAFILEINFO"), Chain("FILENAME"), VARCHAR_TYPE, maxLen)); 
	    schema.Insert(CegoField(Chain("DATAFILEINFO"), Chain("DATAFILEINFO"), Chain("FILETYPE"), VARCHAR_TYPE, 10)); 
	    schema.Insert(CegoField(Chain("DATAFILEINFO"), Chain("DATAFILEINFO"), Chain("SIZE"), INT_TYPE, sizeof(int))); 
	    schema.Insert(CegoField(Chain("DATAFILEINFO"), Chain("DATAFILEINFO"), Chain("USED"), INT_TYPE, sizeof(int)));
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("DATAFILEINFO"), schema, Chain("DATAFILEINFO"));
	    
	    format = Chain("llrr");
	    
	    CegoFieldValue t1(VARCHAR_TYPE, (*pInfo)->getAttributeValue(XML_TEMPFILE_ATTR));
	    CegoFieldValue t2(VARCHAR_TYPE, Chain(XML_TEMPFILE_ATTR));
	    CegoFieldValue t3(INT_TYPE, (*pInfo)->getAttributeValue(XML_TEMPTOTAL_ATTR).asUnsigned());
	    CegoFieldValue t4(INT_TYPE, (*pInfo)->getAttributeValue(XML_TEMPUSED_ATTR).asUnsigned());
	    
	    ListT<CegoFieldValue> tl;
	    
	    tl.Insert(t1);
	    tl.Insert(t2);
	    tl.Insert(t3);
	    tl.Insert(t4);
	    
	    info.Insert(tl);
	    	    
	    pDF = dataFileList.First();
	    while ( pDF ) 
	    {
		Chain fileName = (*pDF)->getAttributeValue(XML_NAME_ATTR);
		Chain fileType = (*pDF)->getAttributeValue(XML_TYPE_ATTR);
		
		Chain numTotal = (*pDF)->getAttributeValue(XML_NUMTOTAL_ATTR);
		Chain numUsed = (*pDF)->getAttributeValue(XML_NUMUSED_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, fileName);
		CegoFieldValue f2(VARCHAR_TYPE, fileType);
		CegoFieldValue f3(INT_TYPE, numTotal);
		CegoFieldValue f4(INT_TYPE, numUsed);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		
		info.Insert(fl);
		
		pDF = dataFileList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getArchLogInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> archInfoList = pRoot->getChildren(XML_ARCHINFO_ELEMENT);
	Element **pArchInfo = archInfoList.First();
	
	if ( pArchInfo )
	{    
	    ListT<Element*> archLogList = (*pArchInfo)->getChildren(XML_ARCHIVELOG_ELEMENT);
	    Element **pAL = archLogList.First();
	    
	    unsigned int maxLen = 0;
	    while ( pAL ) 
	    {
		Chain archPath = (*pAL)->getAttributeValue(XML_ARCHPATH_ATTR);
		
		if ( maxLen < archPath.length() )
		    maxLen = archPath.length();
		
		pAL = archLogList.Next();
	    }
	    
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("ARCHLOGINFO"), Chain("ARCHLOGINFO"), Chain("ARCHID"), VARCHAR_TYPE, 10)); 
	    schema.Insert(CegoField(Chain("ARCHLOGINFO"), Chain("ARCHLOGINFO"), Chain("ARCHPATH"), VARCHAR_TYPE, maxLen));
	    schema.Insert(CegoField(Chain("ARCHLOGINFO"), Chain("ARCHLOGINFO"), Chain("NUMFILE"), VARCHAR_TYPE, 10));
	    schema.Insert(CegoField(Chain("ARCHLOGINFO"), Chain("ARCHLOGINFO"), Chain("NUMARCHIVED"), VARCHAR_TYPE, 10)); 
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("ARCHLOGINFO"), schema, Chain("ARCHLOGINFO"));
	    
	    pAL = archLogList.First();
	    while ( pAL ) 
	    {
		Chain archId = (*pAL)->getAttributeValue(XML_ARCHID_ATTR);
		Chain archPath = (*pAL)->getAttributeValue(XML_ARCHPATH_ATTR);
		Chain numFile = (*pAL)->getAttributeValue(XML_NUMFILE_ATTR);
		Chain numArchived = (*pAL)->getAttributeValue(XML_NUMARCHIVED_ATTR);		
		
		CegoFieldValue f1(VARCHAR_TYPE, archId);
		CegoFieldValue f2(VARCHAR_TYPE, archPath);
		CegoFieldValue f3(VARCHAR_TYPE, numFile);
		CegoFieldValue f4(VARCHAR_TYPE, numArchived);	       
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
				
		info.Insert(fl);
		
		pAL = archLogList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getObjectInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> oList = pRoot->getChildren(XML_OBJLIST_ELEMENT);
	Element **pOL = oList.First();
	    
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("OBJINFO"), Chain("OBJINFO"), Chain("Type"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("OBJINFO"), Chain("OBJINFO"), Chain("Name"), VARCHAR_TYPE, 20)); 
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("OBJINFO"), schema, Chain("OBJINFO"));
	
	if ( pOL ) 
	{
	    ListT<Element*> oeList = (*pOL)->getChildren(XML_OBJ_ELEMENT);	    
	    
	    Element **pOE = oeList.First();
	    
	    while ( pOE )
	    {
		Chain objType = (*pOE)->getAttributeValue(XML_OBJTYPE_ATTR);
		Chain objName = (*pOE)->getAttributeValue(XML_OBJNAME_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, objType);
		CegoFieldValue f2(VARCHAR_TYPE, objName);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		
		info.Insert(fl);

		pOE = oeList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getTransactionInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> taList = pRoot->getChildren(XML_TAINFO_ELEMENT);
	Element **pTA = taList.First();
	    
	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("TAINFO"), Chain("TAINFO"), Chain("RBO"), VARCHAR_TYPE, 15)); 
	schema.Insert(CegoField(Chain("TAINFO"), Chain("TAINFO"), Chain("NUMOP"), INT_TYPE, sizeof(int))); 
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("TAINFO"), schema, Chain("TAINFO"));
	
	if ( pTA ) 
	{
	    ListT<Element*> teList = (*pTA)->getChildren(XML_TAENTRY_ELEMENT);	    

	    Element **pTE = teList.First();
	    
	    while ( pTE )
	    {
		Chain rbo = (*pTE)->getAttributeValue(XML_RBO_ATTR);
		Chain numOp = (*pTE)->getAttributeValue(XML_NUMOP_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, rbo);
		CegoFieldValue f2(VARCHAR_TYPE, numOp);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		
		info.Insert(fl);

		pTE = teList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getBUStatInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> busList = pRoot->getChildren(XML_BUSINFO_ELEMENT);
	Element **pBUS = busList.First();

	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("BUSTAT"), Chain("BUSTAT"), Chain("TS"), VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("BUSTAT"), Chain("BUSTAT"), Chain("BUINFO"), VARCHAR_TYPE, 20)); 
	schema.Insert(CegoField(Chain("BUSTAT"), Chain("BUSTAT"), Chain("BUMSG"), VARCHAR_TYPE, 50)); 
	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("BUSTAT"), schema, Chain("BUSTAT"));
	
	if ( pBUS ) 
	{
	    ListT<Element*> bueList = (*pBUS)->getChildren(XML_BUENTRY_ELEMENT);	    

	    Element **pBUE = bueList.First();
	    
	    while ( pBUE )
	    {
		Chain ts = (*pBUE)->getAttributeValue(XML_TIMESTAMP_ATTR);
		Chain buinfo = (*pBUE)->getAttributeValue(XML_BUINFO_ATTR);
		Chain bumsg = (*pBUE)->getAttributeValue(XML_BUMSG_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, ts);
		CegoFieldValue f2(VARCHAR_TYPE, buinfo);
		CegoFieldValue f3(VARCHAR_TYPE, bumsg);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
	       
		info.Insert(fl);

		pBUE = bueList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getBackupInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> backupInfoList = pRoot->getChildren(XML_BACKUPLIST_ELEMENT);
	
	Element **pInfo = backupInfoList.First();

	if ( pInfo )
	{
	    ListT<Element*> backupList = (*pInfo)->getChildren(XML_BACKUP_ELEMENT);
        
	    ListT<CegoField> schema;
	    schema.Insert(CegoField(Chain("BACKUPLIST"), Chain("BACKUPLIST"), Chain("ID"), VARCHAR_TYPE, 25));
	    schema.Insert(CegoField(Chain("BACKUPLIST"), Chain("BACKUPLIST"), Chain("BUTS"), VARCHAR_TYPE, 25));
	    schema.Insert(CegoField(Chain("BACKUPLIST"), Chain("BACKUPLIST"), Chain("LOGTS"), VARCHAR_TYPE, 25)); 
	    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("BACKUPINFO"), schema, Chain("BACKUPINFO"));
	    
	    Element **pBackup = backupList.First();
		    
	    while ( pBackup ) 
	    {
		ListT<CegoFieldValue> fl;
		
		Chain id = (*pBackup)->getAttributeValue(XML_BACKUPID_ATTR);	
		CegoFieldValue f1(VARCHAR_TYPE, id);
		fl.Insert(f1);
		
		Chain buts = (*pBackup)->getAttributeValue(XML_BACKUPTS_ATTR);		
		CegoFieldValue f2(VARCHAR_TYPE, buts);
		fl.Insert(f2);

		Chain logts = (*pBackup)->getAttributeValue(XML_LOGTS_ATTR);		
		CegoFieldValue f3(VARCHAR_TYPE, logts);
		fl.Insert(f3);	       

		info.Insert(fl);
		
		pBackup = backupList.Next();
	    }
	}
    }
    return;
}

void CegoAdminHandler::getUserInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    unsigned userLen=10;
    unsigned roleLen=10;

    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> userInfoList = pRoot->getChildren(XML_USERINFO_ELEMENT);
	Element **pUserInfoElement = userInfoList.First();
	if ( pUserInfoElement )
	{
	    ListT<Element*> userElementList = (*pUserInfoElement)->getChildren(XML_USER_ELEMENT);
	    Element **pUserElement = userElementList.First();
	    	
	    while ( pUserElement )
	    {
		Chain userName = (*pUserElement)->getAttributeValue(XML_NAME_ATTR);
		if ( (unsigned)userName.length() > userLen )
		    userLen=userName.length();

		Chain role = (*pUserElement)->getAttributeValue(XML_ROLE_ATTR);
		if ( (unsigned)role.length() > roleLen )
		    roleLen=role.length();

		Chain trace = (*pUserElement)->getAttributeValue(XML_TRACE_ATTR);

		Chain numReq = (*pUserElement)->getAttributeValue(XML_NUMREQUEST_ATTR);
		Chain numQuery = (*pUserElement)->getAttributeValue(XML_NUMQUERY_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, userName);
		CegoFieldValue f2(VARCHAR_TYPE, role);
		CegoFieldValue f3(VARCHAR_TYPE, trace);
		CegoFieldValue f4(VARCHAR_TYPE, numReq);
		CegoFieldValue f5(VARCHAR_TYPE, numQuery);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
		fl.Insert(f5);

		info.Insert(fl);
		
		pUserElement = userElementList.Next();
	    }
	}
    }

    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("USERINFO"), Chain("USERINFO"), Chain("NAME"), VARCHAR_TYPE, userLen)); 
    schema.Insert(CegoField(Chain("USERINFO"), Chain("USERINFO"), Chain("ROLE"), VARCHAR_TYPE, roleLen));
    schema.Insert(CegoField(Chain("USERINFO"), Chain("USERINFO"), Chain("TRACE"), VARCHAR_TYPE, 10));
    schema.Insert(CegoField(Chain("USERINFO"), Chain("USERINFO"), Chain("NUMREQ"), VARCHAR_TYPE, 12));
    schema.Insert(CegoField(Chain("USERINFO"), Chain("USERINFO"), Chain("NUMQUERY"), VARCHAR_TYPE, 12));
    
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("USERINFO"), schema, Chain("USERINFO"));
}

void CegoAdminHandler::getTableSetVerificationInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("TSVERIFY"), Chain("TSVERIFY"), Chain("TYPE"), VARCHAR_TYPE, 15)); 
    schema.Insert(CegoField(Chain("TSVERIFY"), Chain("TSVERIFY"), Chain("NAME"), VARCHAR_TYPE, 40)); 
    schema.Insert(CegoField(Chain("TSVERIFY"), Chain("TSVERIFY"), Chain("STATUS"), VARCHAR_TYPE, 50));    
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("TSVERIFY"), schema, Chain("TSVERIFY"));

    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> verifyInfoList = pRoot->getChildren(XML_VERIFICATION_ELEMENT);
	Element **pVerifyInfoElement = verifyInfoList.First();
	if ( pVerifyInfoElement )
	{
	    ListT<Element*> checkElementList = (*pVerifyInfoElement)->getChildren(XML_CHECK_ELEMENT);
	    Element **pCheckElement = checkElementList.First();
	    	
	    while ( pCheckElement )
	    {
		Chain checkType = (*pCheckElement)->getAttributeValue(XML_TYPE_ATTR);
		Chain checkName = (*pCheckElement)->getAttributeValue(XML_NAME_ATTR);
		Chain checkStatus = (*pCheckElement)->getAttributeValue(XML_VALUE_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, checkType);
		CegoFieldValue f2(VARCHAR_TYPE, checkName);
		CegoFieldValue f3(VARCHAR_TYPE, checkStatus);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);

		info.Insert(fl);
		
		pCheckElement = checkElementList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getTableSetCorrectionInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("TSCORRECT"), Chain("TSCORRECT"), Chain("TYPE"), VARCHAR_TYPE, 15)); 
    schema.Insert(CegoField(Chain("TSCORRECT"), Chain("TSCORRECT"), Chain("NAME"), VARCHAR_TYPE, 40)); 
    schema.Insert(CegoField(Chain("TSCORRECT"), Chain("TSCORRECT"), Chain("STATUS"), VARCHAR_TYPE, 50));    
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("TSCORRECT"), schema, Chain("TSCORRECT"));

    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> correctInfoList = pRoot->getChildren(XML_CORRECTION_ELEMENT);
	Element **pCorrectInfoElement = correctInfoList.First();
	if ( pCorrectInfoElement )
	{
	    ListT<Element*> checkElementList = (*pCorrectInfoElement)->getChildren(XML_CHECK_ELEMENT);
	    Element **pCheckElement = checkElementList.First();
	    
	    while ( pCheckElement )
	    {
		Chain checkType = (*pCheckElement)->getAttributeValue(XML_TYPE_ATTR);
		Chain checkName = (*pCheckElement)->getAttributeValue(XML_NAME_ATTR);
		Chain checkStatus = (*pCheckElement)->getAttributeValue(XML_VALUE_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, checkType);
		CegoFieldValue f2(VARCHAR_TYPE, checkName);
		CegoFieldValue f3(VARCHAR_TYPE, checkStatus);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);

		info.Insert(fl);
		
		pCheckElement = checkElementList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getRoleList(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("ROLELIST"), Chain("ROLELIST"), Chain("ROLE"), VARCHAR_TYPE, 15)); 
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("ROLELIST"), schema, Chain("ROLELIST"));

    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> roleListElementList = pRoot->getChildren(XML_ROLELIST_ELEMENT);
	Element **pRoleListElement = roleListElementList.First();
	if ( pRoleListElement )
	{	    
	    ListT<Element*> roleList = (*pRoleListElement)->getChildren(XML_ROLE_ELEMENT);
	    Element **pRoleElement = roleList.First();
	    while ( pRoleElement )
	    {
		Chain role = (*pRoleElement)->getAttributeValue(XML_NAME_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, role);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		
		info.Insert(fl);
		
		pRoleElement = roleList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getRoleInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("ROLEINFO"), Chain("ROLEINFO"), Chain("PERMID"), VARCHAR_TYPE, 15));  
    schema.Insert(CegoField(Chain("ROLEINFO"), Chain("ROLEINFO"), Chain("TABLESET"), VARCHAR_TYPE, 20));
    schema.Insert(CegoField(Chain("ROLEINFO"), Chain("ROLEINFO"), Chain("FILTER"), VARCHAR_TYPE, 15)); 
    schema.Insert(CegoField(Chain("ROLEINFO"), Chain("ROLEINFO"), Chain("PERM"), VARCHAR_TYPE, 15)); 
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("ROLEINFO"), schema, Chain("ROLEINFO"));

    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> permInfoList = pRoot->getChildren(XML_PERMINFO_ELEMENT);

	Element** pPermInfo = permInfoList.First();
	
	if ( pPermInfo )
	{
	    ListT<Element*> permList = (*pPermInfo)->getChildren(XML_PERM_ELEMENT);
	    Element **pPermElement = permList.First();
	    while ( pPermElement )
	    {
		Chain permid = (*pPermElement)->getAttributeValue(XML_PERMID_ATTR);
		Chain tableSet = (*pPermElement)->getAttributeValue(XML_TABLESET_ATTR);
		Chain filter = (*pPermElement)->getAttributeValue(XML_FILTER_ATTR);
		Chain perm = (*pPermElement)->getAttributeValue(XML_PERM_ATTR);
		
		CegoFieldValue f1(VARCHAR_TYPE, permid);
		CegoFieldValue f2(VARCHAR_TYPE, tableSet);
		CegoFieldValue f3(VARCHAR_TYPE, filter);
		CegoFieldValue f4(VARCHAR_TYPE, perm);
		
		ListT<CegoFieldValue> fl;
		
		fl.Insert(f1);
		fl.Insert(f2);
		fl.Insert(f3);
		fl.Insert(f4);
	     
		info.Insert(fl);
		
		pPermElement = permList.Next();
	    }
	}
    }
}

void CegoAdminHandler::getLogInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{
    Element *pRoot = _xml.getDocument()->getRootElement();
    
    if ( pRoot )
    {
	ListT<Element*> tsInfoList = pRoot->getChildren(XML_TABLESET_ELEMENT);
	Element **pInfo = tsInfoList.First();

	ListT<Element*> logList = (*pInfo)->getChildren(XML_LOGFILE_ELEMENT);
	Element **pL = logList.First();

	unsigned int maxLen = 0;
	while ( pL ) 
	{
	    Chain logName = (*pL)->getAttributeValue(XML_NAME_ATTR);
	    
	    if ( maxLen < logName.length() )
		maxLen = logName.length();

	    pL = logList.Next();
	}

	ListT<CegoField> schema;
	schema.Insert(CegoField(Chain("LOGFILE"), Chain("LOGFILE"), Chain("NAME"), VARCHAR_TYPE, maxLen)); 
	schema.Insert(CegoField(Chain("LOGFILE"), Chain("LOGFILE"), Chain("STATUS"), VARCHAR_TYPE, 10)); 
	schema.Insert(CegoField(Chain("LOGFILE"), Chain("LOGFILE"), Chain("SIZE"), VARCHAR_TYPE, 20));
	schema.Insert(CegoField(Chain("LOGFILE"), Chain("LOGFILE"), Chain("OFFSET"), VARCHAR_TYPE, 20));
	schema.Insert(CegoField(Chain("LOGFILE"), Chain("LOGFILE"), Chain("USAGE"), VARCHAR_TYPE, 20));

	oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("LOGFILE"), schema, Chain("LOGFILE"));
	
	pL = logList.First();
	while ( pL ) 
	{
	    Chain logName = (*pL)->getAttributeValue(XML_NAME_ATTR);
	    Chain logStatus = (*pL)->getAttributeValue(XML_STATUS_ATTR);
	    Chain logSize = (*pL)->getAttributeValue(XML_SIZE_ATTR);
	    Chain logOffset = (*pL)->getAttributeValue(XML_OFFSET_ATTR);
	
	    CegoFieldValue f1(VARCHAR_TYPE, logName);
	    CegoFieldValue f2(VARCHAR_TYPE, logStatus);
	    CegoFieldValue f3(VARCHAR_TYPE, logSize);
	    CegoFieldValue f4(VARCHAR_TYPE, logOffset);	    
	    unsigned long long percentUsage = logOffset.asUnsignedLongLong() / ( logSize.asUnsignedLongLong() / 100 );
	    CegoFieldValue f5(VARCHAR_TYPE, Chain(percentUsage) + Chain("%"));

	    ListT<CegoFieldValue> fl;

	    fl.Insert(f1);
	    fl.Insert(f2);
	    fl.Insert(f3);	    
	    fl.Insert(f4);
	    fl.Insert(f5);

	    info.Insert(fl);

	    pL = logList.Next();
	}
    }
}

void CegoAdminHandler::getLogMngInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{    
    Element *pRoot = _xml.getDocument()->getRootElement();

    Chain logMng;
    Chain timeout;
    
    if ( pRoot )
    {
	ListT<Element*> logMngInfo = pRoot->getChildren(XML_LOGMNGINFO_ELEMENT);

	Element** pLogMng = logMngInfo.First();
	
	if ( pLogMng )
	{
	    logMng = (*pLogMng)->getAttributeValue(XML_LOGMNGPROG_ATTR);
	    timeout = (*pLogMng)->getAttributeValue(XML_LOGMNGTIMEOUT_ATTR);
	}
    }

    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("LOGMNGINFO"), Chain("LOGMNGINFO"), Chain("LOGMANAGER"), VARCHAR_TYPE, MAX(logMng.length(),30)));  
    schema.Insert(CegoField(Chain("LOGMNGINFO"), Chain("LOGMNGINFO"), Chain("TIMEOUT"), VARCHAR_TYPE, 10));  
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("LOGMNGINFO"), schema, Chain("LOGMNGINFO"));
    
    CegoFieldValue f1(VARCHAR_TYPE, logMng);
    CegoFieldValue f2(VARCHAR_TYPE, timeout);
    
    ListT<CegoFieldValue> fl;
    
    fl.Insert(f1);
    fl.Insert(f2);
    
    info.Insert(fl);
}

void CegoAdminHandler::getBackupMngInfo(CegoTableObject& oe, ListT<ListT< CegoFieldValue > > & info)
{    
    Element *pRoot = _xml.getDocument()->getRootElement();

    Chain backupMng;
    
    if ( pRoot )
    {
	ListT<Element*> backupMngInfo = pRoot->getChildren(XML_BACKUPMNGINFO_ELEMENT);

	Element** pBackupMng = backupMngInfo.First();
	
	if ( pBackupMng )
	{
	    backupMng = (*pBackupMng)->getAttributeValue(XML_BACKUPMNGPROG_ATTR);
	}
    }

    ListT<CegoField> schema;
    schema.Insert(CegoField(Chain("BACKUPMNGINFO"), Chain("BACKUPMNGINFO"), Chain("BACKUPMANAGER"), VARCHAR_TYPE, MAX(backupMng.length(),30)));  
    oe = CegoTableObject( 0, CegoObject::SYSTEM, Chain("BACKUPMNGINFO"), schema, Chain("BACKUPMNGINFO"));
    
    CegoFieldValue f1(VARCHAR_TYPE, backupMng);
    
    ListT<CegoFieldValue> fl;
    
    fl.Insert(f1);
    
    info.Insert(fl);
}
