///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoObject.cc 
// -------------
// Database object super class class definition
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoObject
//
// Description: Database object super class structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoObject.h"

// POSIX INCLUDES
#include <string.h>

CegoObject::CegoObject()
{
}

CegoObject::CegoObject(const CegoObject& oe)
{
    _type = oe._type;
    _name = oe._name;
    _tabSetId = oe._tabSetId;
}

CegoObject::CegoObject(ObjectType type, const Chain& objName, unsigned tabSetId)
{
    _type = type;
    _tabSetId = tabSetId;
    _name = objName;
}

CegoObject::~CegoObject()
{   
}

void CegoObject::setName(const Chain& name)
{
    _name = name;
}

const Chain& CegoObject::getName() const
{
    return _name;
}

void CegoObject::setType(CegoObject::ObjectType type)
{
    _type = type;
}

CegoObject::ObjectType CegoObject::getType() const
{
    return _type;
}

void CegoObject::setTabSetId(unsigned tabSetId)
{
    _tabSetId = tabSetId;
}

unsigned CegoObject::getTabSetId() const
{
    return _tabSetId;
}

unsigned CegoObject::getBaseSize() const
{
    unsigned baseSize = sizeof(unsigned); // entrySize

    baseSize += 1; // object type
    baseSize += 1; // object name size
    baseSize += _name.length();
    
    baseSize = baseSize + sizeof(unsigned); // tabSetId

    return baseSize;
}

void CegoObject::encodeBase(char *buf, unsigned entrySize) const
{
    char* bufPtr = buf;
    
    memcpy(bufPtr, &entrySize, sizeof(unsigned));
    bufPtr = bufPtr + sizeof(unsigned);

    char c = _type;
    memcpy (bufPtr, &c , 1);
    bufPtr++;
    
    c = (char)_name.length();
    memcpy (bufPtr, &c , 1);
    bufPtr++;

    memcpy(bufPtr, (char*)_name, _name.length());
    bufPtr=bufPtr + _name.length();

    memcpy(bufPtr, &_tabSetId, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

void CegoObject::decodeBase(char *buf, unsigned& entrySize)
{
    char* bufPtr = buf;

    memcpy(&entrySize, bufPtr, sizeof(unsigned));
    bufPtr = bufPtr + sizeof(unsigned);
    
    // encode table information
    char c;
    memcpy (&c, bufPtr, 1);
    
    _type = (ObjectType)c;
    bufPtr++;
    
    bufPtr++;
    _name =  Chain(bufPtr) ;    
    bufPtr=bufPtr + _name.length();

    memcpy(&_tabSetId, bufPtr, sizeof(int));
}

unsigned CegoObject::size() const
{
    return (unsigned)_name.length() + sizeof(ObjectType) + sizeof(unsigned);
}

Chain CegoObject::toChain() const
{
    Chain s;
    s = "ObjectName: " + _name + Chain("\n");
    s +=  Chain("ObjectType: ");
    switch ( _type )
    {
    case SYSTEM:
	s += Chain("system");
	break;
    case TABLE:
	s += Chain("table");
	break;
    case PAVLTREE:
	s += Chain("primary index");
	break;
    case UAVLTREE:
	s += Chain("unique index");
	break;
    case AVLTREE:
	s += Chain("index");
	break;
    case PBTREE:
	s += Chain("primary btree");
	break;
    case UBTREE:
	s += Chain("unique btree");
	break;
    case BTREE:
	s += Chain("btree");
	break;
    case VIEW:
	s += Chain("view");
	break;
    case RBSEG:
	s += Chain("rollback segment");
	break;
    case FKEY:
	s += Chain("foreign key");
	break;
    case PROCEDURE:
	s += Chain("procedure");
	break;
    case CHECK:
	s += Chain("check");
	break;
    case TRIGGER:
	s += Chain("trigger");
	break;
    case ALIAS:
	s += Chain("alias");
	break;
    case JOIN:
	s += Chain("join");
	break;
    case UNDEFINED:
	s += Chain("undefined");
	break;
    }
    s += Chain("\n");
    s += "TableSetId: " + Chain(_tabSetId) + Chain("\n");
    return s;
}

CegoObject& CegoObject::operator = ( const CegoObject& oe)
{
    _type = oe._type;
    _name = oe._name;
    _tabSetId = oe._tabSetId;
    return (*this);
}

bool CegoObject::operator == ( const CegoObject& oe)
{
    if ( _name == oe._name && _type == oe._type && _tabSetId == oe._tabSetId)
	return true;
    return false;
}
