///////////////////////////////////////////////////////////////////////////////
//                                                         
// Datetime.cc
// -----------
// Implementation of a datetime data type
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: Datetime
//
// Description: Date and time handling class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <config.h>

#ifdef HAVE_MINGW
#include "strptime.c"
#endif

#include <time.h>
#include <stdlib.h>

#include "Exception.h"
#include "Datetime.h"

#define BUF 128
#define DTFORMAT "%d.%m.%Y %H:%M:%S"
// old def : "%a %b %d %H:%M:%S %Y"
// on windows, the zoneinfo looks uggly
// #define DTFORMAT "%a %b %d %H:%M:%S %Y"

Datetime::Datetime()
{    
    time ( &_t );
}

Datetime::Datetime(const Chain& v)
{   
    struct tm tmvar = {0};
    if ( strptime((char*)v, DTFORMAT , &tmvar) == 0 )
	throw Exception (EXLOC, Chain("Invalid datetime string <") + v + Chain("> for default format <") + Chain(DTFORMAT) + Chain(">"));
    tmvar.tm_isdst=-1;
    tzset();
    _t = mktime(&tmvar);
}

Datetime::Datetime(const Chain& v, const Chain& format)
{    
    struct tm tmvar = {0};

    if ( strptime((char*)v, (char*)format , &tmvar) == 0 )
	throw Exception (EXLOC, Chain("Invalid datetime string <") + v + Chain("> for format <") + format + Chain(">"));
    tmvar.tm_isdst=-1;
    tzset();
    _t = mktime(&tmvar);
}

Datetime::Datetime(const Chain& v, const ListT<Chain>& formatList)
{

    struct tm tmvar = {0};

    Chain *pFormat = formatList.First();
    bool formatMatch = false;
    while ( pFormat && ! formatMatch )
    {
	if ( strptime((char*)v, (char*)*pFormat , &tmvar) != 0 )
	{
	    tmvar.tm_isdst=-1;
	    tzset();
	    _t = mktime(&tmvar);
	    formatMatch = true;
	}
	else
	{
	    pFormat = formatList.Next();
	}
    }
    if ( formatMatch == false )
    {

	Chain formatListString;

	Chain *pFormat = formatList.First();
	while ( pFormat )
        {
            formatListString += Chain("<") + *pFormat + Chain(">");
            pFormat = formatList.Next();
            if ( pFormat )
                formatListString += Chain(",");
        }
        throw Exception (EXLOC, Chain("Invalid datetime string <") + v + Chain("> for format list ") + formatListString );	
    }
}

Datetime::Datetime(unsigned long long l)
{
    _t = (time_t)l;
}

Datetime::Datetime(int year, int month, int day, int hour, int minute, int second)
{
    struct tm tmvar = {0};
    tmvar.tm_sec = second;
    tmvar.tm_min = minute;
    tmvar.tm_hour = hour;
    tmvar.tm_mon = month - 1;
    tmvar.tm_mday = day;
    tmvar.tm_year = year - 1900;
    tmvar.tm_isdst=-1;
    // tmvar.tm_gmtoff=0;
    // tmvar.tm_zone=zone;
    _t = mktime(&tmvar);
    if ( _t == -1 )
	throw Exception (EXLOC, Chain("Invalid datetime specification"));
}

Datetime::~Datetime()
{
}

Chain Datetime::asChain() const
{
    char buf[BUF];

    tzset();

    struct tm *pTM = localtime( &_t );
    // struct tm *pTM = gmtime( &_t );

    if ( pTM == 0 )
	throw Exception (EXLOC, Chain("Cannot get date value"));

    strftime(buf,BUF,DTFORMAT,pTM);

    Chain s( buf );
    return s;
}

Chain Datetime::asChain(const Chain& format) const
{
    char buf[BUF];

    struct tm *pTM = localtime( &_t );

    if ( pTM == 0 )
	throw Exception (EXLOC, Chain("Cannot get date value"));
    
    strftime(buf,BUF, (char*)format, pTM);

    Chain s( buf );
    return s;
}

unsigned long long Datetime::asLong() const
{
    return (unsigned long long)_t;
}

void Datetime::add(unsigned long long seconds)
{
    _t += seconds;
}

void Datetime::sub(unsigned long long seconds)
{
    _t = _t - seconds;
}

Datetime& Datetime::operator=(const Datetime& d)
{     
    _t = d._t;
    return (*this);
}
    
bool Datetime::operator==(const Datetime& d) const
{
    if ( _t == d._t )
	return true;
    return false;
}
    
bool Datetime::operator!=(const Datetime& d) const
{
    if ( _t != d._t )
	return true;
    return false;
}

bool Datetime::operator < (const Datetime& d) const
{
    if ( _t < d._t )
	return true;
    return false;
}

bool Datetime::operator > (const Datetime& d) const
{
    if ( _t > d._t )
	return true;
    return false;
}

bool Datetime::operator <= (const Datetime& d) const
{
    if ( _t <= d._t )
	return true;
    return false;
}

bool Datetime::operator >= (const Datetime& d) const
{
    if ( _t >= d._t )
	return true;
    return false;
}
    
ostream& operator << (ostream& s, const  Datetime& d)
{
    s << d.asChain();
    return s;
}

