/////////////////////////////////////////////////////////////////////////////// 
// 
// Screen2.cc 
// ---------- 
// Next generation curses based screen implementation 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2017-2022 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Screen 
// 
// Description: Cursor based curses screen implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 


#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Screen2.h"
#include "Tokenizer.h"
#include "Sleeper.h" 

#define COL_HEADER_PASSIVE 1 
#define COL_HEADER_ACTIVE 2 
#define COL_TITLE 3 
#define COL_CONTENT 4
#define COL_SELECTED 5

// advanced color map
#define COL_BLUE 6
#define COL_GREEN 7
#define COL_YELLOW 8
#define COL_RED 9
#define COL_INPUT 10

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_ESC 27 
#define KEY_DEL 127

#define INPUT_TYPE_STRING "S" 
#define INPUT_TYPE_ID "I" 
#define INPUT_TYPE_NUMBER "N" 
#define INPUT_TYPE_PWD "P"
#define INPUT_TYPE_MENU "M" 

#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define OK_COL_POS 2
#define ABORT_COL_POS 9

#define DEFAULT_TIMEOUT 3000
#define FORMSELECT_TITLE "Selection"

#define KEY_BACKSPACE_WIN 8

static Chain _statusLine;

static void refreshStatus();
static void getKeyValue(const Chain& s, Chain& key, Chain& value);
static void calcOffset(WINDOW* parent, int& starty, int& startx, int fraction = 3);


///////////////////////////
// Screen2::Menu methods //
///////////////////////////


void Screen2::Panel::setStatus(const Chain& status)
{
    _statusLine = status;
     refreshStatus();
}

////////////////
// Menu Panel //
////////////////

Screen2::Menu::Menu() : Panel(0) 
{
    _menuSelected = 0;
}

Screen2::Menu::~Menu()
{
}

void Screen2::Menu::regItem(const Chain& name, Panel *pPanel) 
{ 
    _menu.Insert(MenuItem(name, pPanel));
} 

void Screen2::Menu::show(bool doRec)
{

    wclear(stdscr);

    keypad(stdscr, TRUE);
    
    MenuItem *pMI = _menu.First(); 
    int idx=0;
    int colno=0;
    while ( pMI ) 
    {	
        if ( _menuSelected == idx ) 
        { 
            attron(A_REVERSE); 
            color_set(COL_HEADER_ACTIVE, NULL); 
        } 
        else 
        { 
            color_set(COL_HEADER_PASSIVE, NULL); 
        }
	
        mvwprintw(stdscr, 0, colno, "%s", (char*)pMI->getName()); 
        colno += pMI->getName().length() + 1; 
        attroff(A_REVERSE); 
	
        idx++; 
        pMI=_menu.Next(); 
    }

    refreshStatus();
    
    wrefresh(stdscr);
}

WINDOW* Screen2::Menu::getWindow()
{
    return stdscr;
}

void Screen2::Menu::handleKey(int c)
{
    switch(c) 
    { 
    case KEY_RIGHT: 
	if ( _menuSelected < _menu.Size()-1 ) 
	    _menuSelected++; 
	else 
	    _menuSelected = 0;
	break; 
    case KEY_LEFT: 
	if ( _menuSelected > 0  ) 
	    _menuSelected--; 
	else 
	    _menuSelected = _menu.Size()-1; 
	break; 
    case KEY_RETURN: 
    {
	_pNext = _menu[_menuSelected].getPanel();
	return;
    }
    case ERR:
    {
	break;
    }
    default:
    {
	break;
    }
    }

    _pNext = this;    
} 

//////////////////////////////
// Screen2::Message methods //
//////////////////////////////

Screen2::Message::Message(Screen2::Panel *pParent) : Panel(pParent) 
{
    _msgwin = 0;
}

Screen2::Message::~Message()
{
    if ( _msgwin != 0 )
	delwin(_msgwin);
}

void Screen2::Message::setInfo(const Chain& title, const Chain& msg)
{
    _title = title;
    _msg = msg;
    
    _width = _title.length();
    _height=0; 
    
    Tokenizer t(_msg, Chain("\n"));
    Chain l;
    while ( t.nextToken(l))
    {
	_height++;
	if ( l.length() > _width )
	    _width = l.length();
    }
    
    _height+=4;
    _width+=3;

}

void Screen2::Message::show(bool doRec)
{

    _pParent->show(doRec);
    
    int startx, starty;
    calcOffset(_pParent->getWindow(), starty, startx);

    if ( _msgwin == 0 )    
	_msgwin = newwin( _height, _width, starty, startx);

    wclear(_msgwin);
  
    wattron(_msgwin, A_BOLD);
    mvwprintw(_msgwin, 1, 2, "%s", (char*)_title);
    wattroff(_msgwin, A_BOLD);
    box(_msgwin, 0, 0);

    Tokenizer t(_msg, Chain("\n"));
    int lno=3;
    Chain l;
    while ( t.nextToken(l))
    {
	mvwprintw(_msgwin, lno, 2, "%s", (char*)l);
	lno++;
    }

    wrefresh(_msgwin);
}

void Screen2::Message::handleKey(int c)
{
    if ( c == KEY_RESIZE )
    {
	 int yoff, xoff;
	 calcOffset(_pParent->getWindow(), yoff, xoff);
	 mvwin(_msgwin, yoff, xoff);
	_pParent->handleKey(c);
    }
    else
	_pNext = _pParent;    
}

WINDOW* Screen2::Message::getWindow()
{
    return _msgwin;
}

//////////////////////////////
// Screen2::Select methods //
//////////////////////////////

Screen2::Select::Select(Screen2::Panel *pParent) : Panel(pParent) 
{ 
    _selectwin = 0;
    _pError = new Message(pParent);
}

Screen2::Select::~Select()
{
    if ( _selectwin != 0 )
	delwin(_selectwin);
    delete _pError;
}

void Screen2::Select::setItems( const Chain& title, const ListT<Chain>& items)
{
    _title = title;
    _items = items;

    _width = _title.length() + 3;
    
    Chain *pS = _items.First();
    while ( pS )
    {
	Chain key, value;
	getKeyValue(*pS, key, value);
	_width = key.length() + 3 > _width ? key.length() + 3 : _width;   
	pS = _items.Next();
    }
    
    _height = _items.Size() + 4;
    _highlight=1;
}

void Screen2::Select::show(bool doRec)
{
    if ( doRec )
	_pParent->show(doRec);

    int startx, starty;
    calcOffset(_pParent->getWindow(), starty, startx);

    if ( _selectwin == 0 )    
    	_selectwin = newwin( _height, _width, starty, startx);

    wclear(_selectwin);
    
    keypad(_selectwin, TRUE);

    wattron(_selectwin, A_BOLD);
    mvwprintw(_selectwin, 1, 2, "%s", (char*)_title);
    wattroff(_selectwin, A_BOLD);
    int x = 2;
    int y = 3;
    box(_selectwin, 0, 0);
    for(int i = 0; i < _items.Size(); ++i)
    {	
	Chain key, value;

	getKeyValue(_items[i], key, value);
	
	if (_highlight == i + 1) /* High light the present choice */
	{	
	    wattron(_selectwin, A_REVERSE);
	    mvwprintw(_selectwin, y, x, "%s", (char*)key);
	    wattroff(_selectwin, A_REVERSE);
	}
	else
	    mvwprintw(_selectwin, y, x, "%s", (char*)key);
	++y;
    }
    wrefresh(_selectwin);
}

void Screen2::Select::handleKey(int c)
{
    switch(c)
    {	
    case KEY_UP:
    {
	if (_highlight == 1)
	    _highlight = _items.Size();
	else
	    --_highlight;
	break;
    }
    case KEY_DOWN:
    {
	if(_highlight == _items.Size() )
	    _highlight = 1;
	else 
	    ++_highlight;
	break;
    }
    case KEY_RESIZE:
    {
	int startx, starty;
	calcOffset(_pParent->getWindow(), starty, startx);
	mvwin(_selectwin, starty, startx);
	_pParent->handleKey(c);
	break;  
    }    
    case KEY_RETURN:
    {
	delwin(_selectwin);
	_selectwin=0;
	_pNext = onSelect(_highlight);
	return;
    }	
    case KEY_ESC:
    {
	delwin(_selectwin);
	_selectwin=0;
	_pNext = _pParent;
	return;
    }
    }    
    _pNext = this;
}

void Screen2::Select::setSelectedKey(const Chain& key)
{
    for(int i = 0; i < _items.Size(); ++i)
    {	
	Chain k, v;	
	getKeyValue(_items[i], k, v);
	
	if ( k == key )
	{
	    _highlight = i + 1;
	    return;
	}
    }
}

Chain Screen2::Select::getSelectedKey()
{
    Chain key, value;
    getKeyValue(_items[_highlight-1], key, value);
    return key;
}

Chain Screen2::Select::getSelectedValue()
{
    Chain key, value;
    getKeyValue(_items[_highlight-1], key, value);
    return value;
}

WINDOW* Screen2::Select::getWindow()
{
    return _selectwin;
}

Screen2::Panel* Screen2::Select::setError(const Chain& title, const Chain& msg)
{
    _pError->setInfo(title, msg);
    return _pError;
}

//////////////////////////////
// Screen2::Grid methods //
//////////////////////////////

Screen2::Grid::Grid(Screen2::Panel *pParent) : Screen2::Panel(pParent)
{
    _isSelectable=true;
    _rowSelected=1;
    _vStart=1;
    _gridwin = 0;
    _pError = new Message(pParent);
}

Screen2::Grid::~Grid()
{
    if ( _gridwin != 0 )
	delwin(_gridwin);
    delete _pError;
}

void Screen2::Grid::setSchema(const Chain& title, const ListT<Chain>& gridSchema, const ListT<GridColor>& colorMap, int height)
{
    _title = title;
    _schema = gridSchema;
    _colorMap = colorMap;
    
    _numCol=0;
    Chain *pA = _schema.First();
    while ( pA )
    {
	Chain attr, attrLen;
	getKeyValue(*pA, attr, attrLen);
	_numCol = _numCol + attrLen.asInteger();	    
	pA = _schema.Next();
    }
    _initHeight = height;
}
    
void Screen2::Grid::handleKey(int c)
{
    _curStat=0;
    
    switch(c)
    {	
    case KEY_UP:
    {
	if ( _isSelectable )
	{
	    if ( _rowSelected > 1 )
	    {
		if (_vStart >= _rowSelected && _vStart > 0)
		    _vStart--;
		else
		    _curStat=1;
		_rowSelected--;
	    }
	}
	break;
    }
    case KEY_DOWN:
    {
	if ( _isSelectable )
	{
	    if ( _rowSelected < _table.Size() )
	    {
		if ( _rowSelected >= _height + _vStart - 1 )
		    _vStart++;
		else
		    _curStat=2;
		_rowSelected++;
	    }
	}
	break;
    }
    case KEY_RETURN:
    {
	_pNext = onSelect(_rowSelected);
	return;
    }
    case KEY_ESC:
    {
      if ( _gridwin != 0 )
	delwin(_gridwin);
	_gridwin=0;
	_pNext = _pParent;
	return;
    }
    case KEY_RESIZE:
    {	
	int height, width;
	getmaxyx(stdscr, height, width);

	if ( _initHeight == 0 || _initHeight > height )
	{
	    _height=height-6;
	}
	else
	{
	    _height=_initHeight;
	}      
	
	wresize(_gridwin, _height+3, _numCol + 2);

	_pParent->handleKey(c);
	_pNext = this;
	return;
    }
    default:
    {	
	_pNext = onKey(c, _rowSelected);
	return;
    }
    }
    _pNext = this;
}

void Screen2::Grid::show(bool doRec)
{
    // this is a refresh optimization
    // in case of cursor up/down move inside grid, we just have to refresh
    // selected and previous selected grid row
    if ( _curStat == 1 )
    {
	refreshRow(_rowSelected+1, _rowSelected-_vStart+3);
	refreshRow(_rowSelected, _rowSelected-_vStart+2);
	_curStat=0;
	return;
    }
    else if ( _curStat == 2 )
    {
	refreshRow(_rowSelected-1, _rowSelected-_vStart+1);
	refreshRow(_rowSelected, _rowSelected-_vStart+2);
	_curStat=0;
	return;
    }
    
    // since parent grid may overlap for grid panels, we make a recursive refesh in any case
    _pParent->show(true);
    
    int height, width;
    // getmaxyx(_pParent->getWindow(), height, width);
    getmaxyx(stdscr, height, width);

    // if grid columns to wide, we cannot print
    if ( width < _numCol + 2 )
      return;
    
    if ( _initHeight == 0 || _initHeight > height )
    {
	_height=height-6;
    }
    else
    {
	_height=_initHeight;
    }

    if ( _gridwin == 0 )
    {
	_gridwin = newwin( _height + 3, _numCol + 2, 2, 1);
	keypad(_gridwin, TRUE);	
    }

    wclear(_gridwin);
    box(_gridwin, 0, 0);

    _table = getData();
   
    if ( _gridwin == NULL )
      return;

    if ( _isSelectable == false || _rowSelected > _table.Size() )
    {
	_rowSelected=1;
	_vStart=1;
    }

    int colno=2;
    int rowno=1;

    wcolor_set(_gridwin, COL_TITLE, NULL);	
    wattron(_gridwin, A_BOLD);
    
    mvwprintw(_gridwin, 0, 2, "%s", (char*)_title);
    
    Chain *pA = _schema.First();
    while ( pA )
    {
	Chain attr, attrLen;
	getKeyValue(*pA, attr, attrLen);
	
	mvwprintw(_gridwin, rowno, colno, "%s", (char*)attr);
	
	colno = colno + attrLen.asInteger();
	
	pA = _schema.Next();
    }
	
    rowno++;
    
    wattroff(_gridwin, A_BOLD);
    wcolor_set(_gridwin, COL_CONTENT, NULL);
    
    int tabRow=_vStart;
    int numRow = min(_table.Size() - tabRow + 1, _height );
    
    while ( tabRow < numRow + _vStart)
    {
	refreshRow(tabRow, rowno);
	
	tabRow++;
	rowno++;
    }    
    wrefresh(_gridwin);
}

void Screen2::Grid::refreshRow(int tabRow, int rowno)
{
    int colno=2;
    
    int rowColor = COL_CONTENT;
    
    if ( _rowSelected == tabRow && _isSelectable )
    {
	wcolor_set(_gridwin, COL_SELECTED, NULL);
	wattron(_gridwin, A_REVERSE);
	rowColor = COL_SELECTED;
    }
    
    Chain *pA = _schema.First();
    Chain *pVal = _table[tabRow-1].First();
    while ( pA && pVal )
    {		       
	GridColor *pTC = _colorMap.Find(GridColor(*pVal));	       
	if ( pTC )
	{
	    switch ( pTC->getCode() )
	    {
	    case SC_BLUE:
		wcolor_set(_gridwin, COL_BLUE, NULL);
		break;
	    case SC_GREEN:
		wcolor_set(_gridwin, COL_GREEN, NULL);
		break;
	    case SC_RED:
		wcolor_set(_gridwin, COL_RED, NULL);
		break;
	    case SC_YELLOW:
		wcolor_set(_gridwin, COL_YELLOW, NULL);
		break;			
	    }
	}
	
	Chain attr, attrLen;
	getKeyValue(*pA, attr, attrLen);
	
	for ( int i=0; i < attrLen.asInteger() ; i++)
	    mvwprintw(_gridwin, rowno, colno+1, "%s", " ");
	mvwprintw(_gridwin, rowno, colno, "%s", (char*)*pVal);
	
	colno = colno + attrLen.asInteger();
	
	wcolor_set(_gridwin, rowColor, NULL);
	
	pVal = _table[tabRow-1].Next();
	
	pA = _schema.Next();
    }
    
    if ( _rowSelected == tabRow && _isSelectable )
    {		
	wcolor_set(_gridwin, COL_CONTENT, NULL);
	wattroff(_gridwin, A_REVERSE); 		
    }
}
    
WINDOW* Screen2::Grid::getWindow()
{
    return _gridwin;
}

Screen2::Panel* Screen2::Grid::setError(const Chain& title, const Chain& msg)
{
    _pError->setInfo(title, msg);
    return _pError;
}

//////////////////////////////
// Screen2::Confirm methods //
/////////////////////////////

Screen2::Confirm::Confirm(Screen2::Panel *pParent) : Screen2::Panel(pParent)
{
    _confwin = 0;
    _pError = new Message(pParent);
}

Screen2::Confirm::~Confirm()
{
    if ( _confwin != 0 )
	delwin(_confwin);
    delete _pError;
}

void Screen2::Confirm::setInfo(const Chain& title, const Chain& details)
{  
    _title = title;
    _details = details;
    
    _width = max(title.length(), details.length()) + 5; 
    _height = 7;
    
    _confVal = 0;
    
    _curRow = 5; 
    _curCol = OK_COL_POS;
}

void Screen2::Confirm::show(bool doRec)
{
    if ( doRec )
	_pParent->show(doRec);

    int startx, starty;
    calcOffset(_pParent->getWindow(), starty, startx);

    if ( _confwin == 0 )
	_confwin = newwin( _height, _width, starty, startx);

    wclear(_confwin);
    keypad(_confwin, TRUE);

    wattron(_confwin, A_BOLD);
    mvwprintw(_confwin, 1, 2, "%s", (char*)_title);
    wattroff(_confwin, A_BOLD);

    if ( _details != Chain() )
    {
	mvwprintw(_confwin, 3, 2, "%s", (char*)_details);
    }

    box(_confwin, 0, 0); 

    curs_set(0);
    if ( _curCol == OK_COL_POS )
    {
	wattron(_confwin, A_REVERSE);
	mvwprintw(_confwin, _curRow, _curCol, "OK");
	wattroff(_confwin, A_REVERSE);
	mvwprintw(_confwin, _curRow, _curCol+7, "Abort");
    }
    else if ( _curCol == ABORT_COL_POS )
    {
	mvwprintw(_confwin, _curRow, _curCol-7, "OK");
	wattron(_confwin, A_REVERSE);
	mvwprintw(_confwin, _curRow, _curCol, "Abort");
	wattroff(_confwin, A_REVERSE);
    }
    
    wmove(_confwin, _curRow, _curCol);    
    wrefresh(_confwin);
}

void Screen2::Confirm::handleKey(int c)
{
    switch(c)
    {	
    case KEY_RESIZE:
    {
	int startx, starty;
	calcOffset(_pParent->getWindow(), starty, startx);
	mvwin(_confwin, starty, startx);
	_pParent->handleKey(c);
	break;  
    }
    case KEY_TAB:
    case KEY_DOWN:
    case KEY_UP:
    case KEY_LEFT:
    case KEY_RIGHT:
	if ( _curCol == OK_COL_POS )
		_curCol = ABORT_COL_POS;
	else if ( _curCol == ABORT_COL_POS )
	    _curCol = OK_COL_POS;
	break; 
	
    case KEY_RETURN:
    {
	delwin(_confwin);
	_confwin=0;
	if ( _curCol == OK_COL_POS )
	    _pNext = onConfirm();
	else
	    _pNext = onCancel();
	return;
    }
    case KEY_ESC:
    {
	delwin(_confwin);
	_confwin=0;
	_pNext = _pParent;
	return;
    }
    }

    _pNext = this;
}

WINDOW* Screen2::Confirm::getWindow()
{
    return _confwin;
}

Screen2::Panel* Screen2::Confirm::setError(const Chain& title, const Chain& msg)
{
    _pError->setInfo(title, msg);
    return _pError;
}

///////////////////////////
// Screen2::Form methods //
///////////////////////////

Screen2::Form::Form(Screen2::Panel *pParent) : Screen2::Panel(pParent)
{
    _curRow = 5; 
    _curCol = OK_COL_POS;  

    // init array
    for(int i = 0; i < VAL_MAX_NUM; i++)
    {
	for(int j = 0; j < VAL_MAX_LEN; j++)
	    _inputArray[i][j]=0;
	_vposArray[i]=0;
    }

    _formwin = 0;
    _pSelect = new FormSelect((Screen2::Panel*)this);
    _pError = new Message(pParent);
}

Screen2::Form::~Form()
{
    if ( _formwin != 0 )
	delwin(_formwin);
    delete _pSelect;
    delete _pError;
}

void Screen2::Form::setAttrList(const Chain& title, const ListT<Chain>& attrList, int maxVisible)
{
    _title = title;        
    _attrList = attrList;
    _maxVisible = maxVisible;
    
    _height = _attrList.Size() + CUR_ROW_OFFSET + 3;
    
    _colOffset = 0;
    _width = 0;
    
    for(int i = 0; i < _attrList.Size(); ++i)
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(_attrList[i], attr, type, maxLen, value);
	_maxLenArray[i] = maxLen;
	
	if ( (char*)value )
	{	    
	    if ( type == Chain(INPUT_TYPE_MENU) )
	    {
		Tokenizer t1(value, LISTSEPTOKEN);
		Chain menuItem;
		// first token is active value
		if ( t1.nextToken(menuItem))
		{
		    Tokenizer t2(menuItem, VALSEPTOKEN);
		    Chain v;
		    if ( t2.nextToken(v))
			value = v;
		}
	    }
	    
            for(int j = 0; j < value.length(); j++)
                _inputArray[i][j]=value[j];
            for(int j = value.length(); j < VAL_MAX_LEN; j++)
                _inputArray[i][j]=0;
	}
	else
	{
	    _inputArray[i][0]=0;
	}

	if ( _colOffset < attr.length() )
	    _colOffset = attr.length();
	
	if ( _width < _maxVisible + attr.length() )
	    _width = _maxVisible + attr.length();
    }
    
    _width+=8;
    _colOffset+=4;
    
    reset();
}

void Screen2::Form::reset()
{
    _curRow = CUR_ROW_OFFSET;
    _curCol = _colOffset;
}

void Screen2::Form::setValueList(const ListT<Chain>& valList)
{
    for(int i = 0; i < _attrList.Size(); ++i)
    {
	Chain attr, type, defValue;
	int maxLen;
	getAttrTypeValue(_attrList[i], attr, type, maxLen, defValue);

	Chain value = valList[i];
	
	if ( (char*)value )
	{
	    if ( type == Chain(INPUT_TYPE_MENU) )
	    {
		Chain menuKey;
		getListKey(defValue, value, menuKey);
		value = menuKey;
	    }
            for(int j = 0; j < value.length(); j++)
                _inputArray[i][j]=value[j];
            for(int j = value.length(); j < VAL_MAX_LEN; j++)
                _inputArray[i][j]=0;
	}
	else
	{
	    _inputArray[i][0]=0;
	}
    }
}

void Screen2::Form::clear()
{
    for(int i = 0; i < _attrList.Size(); ++i)
    {
	Chain attr, type, defValue;
	int maxLen;
	getAttrTypeValue(_attrList[i], attr, type, maxLen, defValue);

	if ( type == Chain(INPUT_TYPE_MENU) )
	{
	    Chain menuKey;

	    Tokenizer t1(defValue, LISTSEPTOKEN);
	    Chain menuItem;
	    while ( t1.nextToken(menuItem))
	    {
		Tokenizer t2(menuItem, VALSEPTOKEN);	      
		t2.nextToken(menuKey);	
	    }	   
	    for(int j = 0; j < menuKey.length(); j++)
		_inputArray[i][j]=menuKey[j];	    
	}
	else
	{
	    _inputArray[i][0]=0;
	}
    }
}

void Screen2::Form::show(bool doRec)
{
    if ( doRec )
	_pParent->show(doRec);

    curs_set(0);

    int startx, starty;
    calcOffset(_pParent->getWindow(), starty, startx, 4);
            
    if ( _formwin == 0 )
	_formwin = newwin( _height, _width, starty, startx);

    wclear(_formwin);   
	
    wcolor_set(_formwin, COL_CONTENT, NULL);
    keypad(_formwin, TRUE);
	
    wattron(_formwin, A_BOLD);
    mvwprintw(_formwin, 1, 2, "%s", (char*)_title);
    wattroff(_formwin, A_BOLD);
    int x = 2;
    int y = CUR_ROW_OFFSET;
    box(_formwin, 0, 0);
    for(int i = 0; i < _attrList.Size(); ++i)
    {	
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(_attrList[i], attr, type, maxLen, value);
	
	mvwprintw(_formwin, y, x, "%s", (char*)attr);
	mvwprintw(_formwin, y, _colOffset - 2, ":");
	    
	if ( type == Chain(INPUT_TYPE_MENU) )
	{
	    wcolor_set(_formwin, COL_TITLE, NULL);
	    
	    wattron(_formwin, A_BOLD);
	    
	    // cleanup first
	    int j = 0;
	    while ( j < maxLen )
	    {
		mvwprintw(_formwin, y, _colOffset + j, "%c", ' ');
		j++;
	    }
	    
	    mvwprintw(_formwin, y, _colOffset, "%s", (char*)_inputArray[i]);
	    
	    wattroff(_formwin, A_BOLD);
	    wcolor_set(_formwin, COL_CONTENT, NULL);
	}
	else
	{	    
	    wattron(_formwin, A_UNDERLINE);
	    wcolor_set(_formwin, COL_INPUT, NULL);
	    for(int j = 0; j < min(maxLen, _maxVisible); ++j)
		mvwprintw(_formwin, y, _colOffset + j, " ");
	    if ( _inputArray[i] )
	    {
		if ( type == Chain(INPUT_TYPE_PWD) )
		{
		    int j=0;
		    while ( _inputArray[i][j] != 0 )
		    {
			mvwprintw(_formwin, y, _colOffset + j, "%c", '*');
			j++;
		    }
		}
		else
		{
		    int j=_vposArray[y - CUR_ROW_OFFSET];			
		    int pos=0;
		    while ( _inputArray[i][j] != 0 && j < _vposArray[y - CUR_ROW_OFFSET] + _maxVisible )
		    {
			mvwprintw(_formwin, y, _colOffset+pos, "%c", _inputArray[i][j]);
			j++;
			pos++;
		    }			
		}
	    }		
	    wattroff(_formwin, A_UNDERLINE);
	    wcolor_set(_formwin, COL_CONTENT, NULL);
	}	
	++y;
    }
    y++;
    
    wattron(_formwin, A_BOLD);
    mvwprintw(_formwin, y, x, "Ok     Abort");
    wattroff(_formwin, A_BOLD);
	
    if ( _curRow == _attrList.Size() + CUR_ROW_OFFSET + 1 )
    {
	wattron(_formwin, A_BOLD);
	curs_set(0);
	if ( _curCol == OK_COL_POS )
	{
	    wattron(_formwin, A_REVERSE);
	    mvwprintw(_formwin, _curRow, _curCol, "OK");
	    wattroff(_formwin, A_REVERSE);
	    mvwprintw(_formwin, _curRow, _curCol+7, "Abort");
	}
	else if ( _curCol == ABORT_COL_POS )
	{
	    mvwprintw(_formwin, _curRow, _curCol-7, "OK");
	    wattron(_formwin, A_REVERSE);
	    mvwprintw(_formwin, _curRow, _curCol, "Abort");
	    wattroff(_formwin, A_REVERSE);
	}
	wattroff(_formwin, A_BOLD);
    }
    else
    {
	curs_set(1);
	wattron(_formwin, A_BOLD);
	mvwprintw(_formwin, y, x, "Ok     Abort");
	wattroff(_formwin, A_BOLD);	    	    
    }

    wmove(_formwin, _curRow, _curCol);
    wrefresh(_formwin);
}

void Screen2::Form::getAttrTypeValue(const Chain& s, Chain& attr, Chain& type, int& maxLen, Chain& value)
{
    Tokenizer t(s, ATTRSEPTOKEN);
    Chain v;
    t.nextToken(attr);
    t.nextToken(type);
    
    Chain maxLenStr;
    t.nextToken(maxLenStr);
    maxLen = maxLenStr.asInteger();
    
    t.nextToken(value);
}

void Screen2::Form::getListKey(const Chain& s, const Chain& value, Chain& key)
{    
    Tokenizer t1(s, LISTSEPTOKEN);
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, VALSEPTOKEN);
	Chain menuKey;
	t2.nextToken(menuKey);
	Chain menuValue;
	t2.nextToken(menuValue);
	
	if ( menuValue == value )
	{
	    key = menuKey;
	    return;
	}
    }
}

void Screen2::Form::getListValue(const Chain& s, const Chain& key, Chain& value)
{    
    Tokenizer t1(s, LISTSEPTOKEN);	
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, VALSEPTOKEN);
	Chain menuKey;
	while ( t2.nextToken(menuKey))
	{
	    if ( menuKey == key )
	    {
		t2.nextToken(value);
		return;
	    }
	}
    }
}

void Screen2::Form::handleKey(int c)
{
    switch(c)
    {
    case KEY_UP:
	int prevCurRow, prevCurCol;
	prevCursorPos(_attrList.Size(), _colOffset, _curRow, _curCol, prevCurRow, prevCurCol);
	_curRow = prevCurRow;
	_curCol = prevCurCol;
	break;
    case KEY_DOWN:
    case KEY_TAB:	
	int nextCurRow, nextCurCol;
	nextCursorPos(_attrList.Size(), _colOffset, _curRow, _curCol, nextCurRow, nextCurCol, 0);
	_curRow = nextCurRow;
	_curCol = nextCurCol;
	break;
    case KEY_LEFT:
	if ( _curRow < _attrList.Size() + CUR_ROW_OFFSET
	     && _vposArray[_curRow - CUR_ROW_OFFSET] > 0 )
	{
	    _vposArray[_curRow - CUR_ROW_OFFSET]--;
	}
	else if ( _curRow < _attrList.Size() + CUR_ROW_OFFSET
		  && _curCol > _colOffset )
	{
	    _curCol--;
	}
	else if ( _curRow == _attrList.Size() + CUR_ROW_OFFSET + 1 )
	{
	    int prevCurRow, prevCurCol;
	    prevCursorPos(_attrList.Size(), _colOffset, _curRow, _curCol, prevCurRow, prevCurCol);
	    _curRow = prevCurRow;
	    _curCol = prevCurCol;
	}
	break;
    case KEY_RIGHT:
    {	  	    
	if ( _curRow < _attrList.Size() + CUR_ROW_OFFSET
	     && _inputArray[_curRow - CUR_ROW_OFFSET][_curCol - _colOffset + _vposArray[_curRow - CUR_ROW_OFFSET]] != 0)
	{
	    if ( _curCol < _colOffset + _maxVisible )
	    {
		_curCol++;
	    }
	    else if ( _curRow < _attrList.Size() + CUR_ROW_OFFSET
		      && _curCol >= _colOffset + _maxVisible
		      && _vposArray[_curRow - CUR_ROW_OFFSET] + _maxVisible < _maxLenArray[_curRow - CUR_ROW_OFFSET] )
	    {
		_vposArray[_curRow - CUR_ROW_OFFSET]++;
	    }
	}
	else if ( _curRow == _attrList.Size() + CUR_ROW_OFFSET + 1 )
	{
	    int prevCurRow, prevCurCol;
	    nextCursorPos(_attrList.Size(), _colOffset, _curRow, _curCol, prevCurRow, prevCurCol, 0);
	    _curRow = prevCurRow;
	    _curCol = prevCurCol;
	}
	break;
    }
    case KEY_RETURN:
    {
	if ( _curRow == _attrList.Size() + CUR_ROW_OFFSET + 1 )
	{
	    delwin(_formwin);
	    _formwin=0;
	    
	    if ( _curCol == OK_COL_POS )
	    {
		ListT<Chain> valList = getValues();
		_pNext = onConfirm(valList);
	    }
	    else if ( _curCol == ABORT_COL_POS )
		_pNext = onCancel();
	    
	}
	else
	{
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(_attrList[_curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
	    if ( type == Chain(INPUT_TYPE_MENU))
	    {		    
		Tokenizer t(value, LISTSEPTOKEN);
		
		ListT<Chain> menu;
		Chain menuItem;
		
		// first value is just active value
		t.nextToken(menuItem);
		
		while ( t.nextToken(menuItem))
		{
		    menu.Insert(menuItem);
		}
		
		Chain selectKey;
		Chain selectValue;

		_pSelect->setup(Chain(FORMSELECT_TITLE), menu, _inputArray[_curRow-CUR_ROW_OFFSET]);
		
		_pNext = _pSelect;	
            }
	}
	return;
    }
    case KEY_DEL:
    case KEY_BACKSPACE:
    case KEY_BACKSPACE_WIN:
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(_attrList[_curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
	if ( type != Chain(INPUT_TYPE_MENU))
	{
	    if ( _curCol > _colOffset && _vposArray[_curRow - CUR_ROW_OFFSET] == 0 )
		_curCol--;
	    else if ( _vposArray[_curRow - CUR_ROW_OFFSET] > 0 )
		_vposArray[_curRow - CUR_ROW_OFFSET]--;
	    
	    for ( int i=_curCol-_colOffset + _vposArray[_curRow - CUR_ROW_OFFSET];
		  i<VAL_MAX_LEN;
		  i++)
		_inputArray[_curRow-CUR_ROW_OFFSET][i] = _inputArray[_curRow-CUR_ROW_OFFSET][i+1];
	}
	break;
    }
    case KEY_ESC:
    {
	delwin(_formwin);
	_formwin=0;
	_pNext = _pParent;
	return;
    }
    case KEY_RESIZE:
    {
	int startx, starty;
	calcOffset(_pParent->getWindow(), starty, startx, 4);
	mvwin(_formwin, starty, startx);
	_pParent->handleKey(c);
	_pNext = this;
	return;
    }
    case ERR:
    {
	break;
    }
    default:
    {
	if ( _curRow < _attrList.Size() + CUR_ROW_OFFSET )
	{		
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(_attrList[_curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
	    
	    bool inputAccepted=false;
	    
	    if ( _vposArray[_curRow-CUR_ROW_OFFSET] + _curCol - _colOffset < maxLen )
	    {
		if ( type == Chain(INPUT_TYPE_STRING)
		     && ( isalnum (c)
			  || c == '_'
			  || c == '-'
			  || c == '/'
			  || c == '.'
			  || c == ','
			  || c == ';'
			  || c == '*'
			  || c == '['
			  || c == ']'
			  || c == ':'
			  || c == '@'
			  || c == '='
			  || c == '!'
			  || c == '?'
			  || c == '*'
			  || c == '+'
			  || c == '#'
			  || c == '('
			  || c == ')'
			  || c == '{'
			  || c == '}'
			  || c == '['
			  || c == ']'
			  || c == '%'
			  || c == '&'
			  || c == '$'
			  || c == '>'
			  || c == '<'
			  || c == ' ' ))
		    inputAccepted=true;
		else if ( type == Chain(INPUT_TYPE_ID)
			  && ( isalnum (c)
			       || c == '_' ))
		    inputAccepted=true;
		else if ( type == Chain(INPUT_TYPE_NUMBER) && isdigit (c) )
		    inputAccepted=true;
		else if ( type == Chain(INPUT_TYPE_PWD) )
		    inputAccepted=true;
	    }
	    
	    if ( inputAccepted )
	    {
		for ( int i=VAL_MAX_LEN -1;
		      i >= _vposArray[_curRow-CUR_ROW_OFFSET] + _curCol - _colOffset;
		      i--)
		    _inputArray[_curRow-CUR_ROW_OFFSET][i] = _inputArray[_curRow-CUR_ROW_OFFSET][i-1];
		
		
		_inputArray[_curRow-CUR_ROW_OFFSET][_vposArray[_curRow-CUR_ROW_OFFSET] + _curCol - _colOffset] = c;
		
		
		if (  _curCol >= _colOffset + _maxVisible ) // maxLenArray[curRow - CUR_ROW_OFFSET] )
		    _vposArray[_curRow-CUR_ROW_OFFSET]++;
		else
		    _curCol++;
	    }
	}
	break;
    }
    }   
    _pNext = this;
}

ListT<Chain> Screen2::Form::getValues()
{
    // store values in valList

    ListT<Chain> valList;
    
    for(int i = 0; i < _attrList.Size(); ++i)
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(_attrList[i], attr, type, maxLen, value);
	
	if ( type == Chain(INPUT_TYPE_MENU))
	{
	    Chain menuValue;
	    getListValue(value, Chain(_inputArray[i]), menuValue);
	    valList.Insert(menuValue);
	}
	else
	{
	    if ( _inputArray[i][0] )
		valList.Insert(Chain(_inputArray[i]).cutTrailing(" "));
	    else
		valList.Insert(Chain());
	}
    }

    return valList;
}

void Screen2::Form::prevCursorPos(int attrListSize, int colOffset, int curRow, int curCol, int& prevCurRow, int& prevCurCol) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 
    
    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 
    
    if ( curRow == CUR_ROW_OFFSET ) 
    { 
        prevCurRow = abortRow; 
        prevCurCol = abortCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        prevCurRow = attrListSize + CUR_ROW_OFFSET - 1; 
        prevCurCol = colOffset; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        prevCurRow = okRow; 
        prevCurCol = okCol; 
    } 
    else 
    { 
        prevCurRow = curRow-1; 
        prevCurCol = colOffset; 
    } 
} 

void Screen2::Form::nextCursorPos(int attrListSize, int colOffset, int curRow, int curCol, int& nextCurRow, int& nextCurCol, int navMode) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 
    
    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 

    if ( curRow == attrListSize + CUR_ROW_OFFSET - 1) 
    { 
        nextCurRow = okRow; 
        nextCurCol = okCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        nextCurRow = abortRow; 
        nextCurCol = abortCol; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        nextCurRow = CUR_ROW_OFFSET; 
        nextCurCol = colOffset; 
    } 
    else if ( navMode == 0 )
    { 
        nextCurRow = curRow+1; 
        nextCurCol=colOffset; 
    }
    else
    {
        nextCurRow = okRow; 
        nextCurCol = okCol; 	
    }
} 

WINDOW* Screen2::Form::getWindow()
{
    return _formwin;
}

Screen2::Panel* Screen2::Form::setError(const Chain& title, const Chain& msg)
{
    _pError->setInfo(title, msg);
    return _pError;
}

Screen2::Panel* Screen2::Form::FormSelect::onSelect(int selected)
{  
    Chain item = getItem(selected-1);
    
    Chain key, value;
    getKeyValue(item, key, value);
    
    for(int j = 0; j < key.length(); j++)
	_pInput[j]=key[j];
    for(int j = key.length(); j < VAL_MAX_LEN; j++)
	_pInput[j]=0;
    
    return _pParent;
}

////////////////////////////////
// Screen2::Attribute methods //
////////////////////////////////

Screen2::Attribute::Attribute(Screen2::Panel *pParent, bool isVertical) : Panel(pParent) 
{ 
    _isVertical = isVertical;
    for ( int i=0; i < MAX_ATTR_WIN; i++)
	_attrwin[i]=0;
}

Screen2::Attribute::~Attribute()
{
    for ( int i=0; i < MAX_ATTR_WIN; i++)
	if ( _attrwin[i] != 0 )
	    delwin(_attrwin[i]);
}

void Screen2::Attribute::setAttrList(const Chain& title, const ListT< ListT<Chain> >& attrList)
{
    _title = title;    
    _attrList = attrList;
}

void Screen2::Attribute::show(bool doRec)
{
    if ( doRec )
	_pParent->show(doRec);
        
    int widx=0;
    int xpos=1;
    int ypos=2;
    
    ListT<Chain> *pKVL = _attrList.First();
    while ( pKVL )
    {
	int maxKeyLen=0;
	int maxValLen=0;
	Chain *pKV = pKVL->First();
	while ( pKV )
	{
	    Chain key, value;

	    Tokenizer t(*pKV, Chain(VALSEPTOKEN));
	    t.nextToken(key);
	    t.nextToken(value);
	    
	    if ( key.length() > maxKeyLen )
		maxKeyLen = key.length();
	    if ( value.length() > maxValLen )
		maxValLen = value.length();
	    pKV = pKVL->Next();
	}
	
	if ( _attrwin[widx] == 0 )
	    _attrwin[widx] = newwin(pKVL->Size() + 2, maxKeyLen + maxValLen + 3, ypos, xpos);

	wclear(_attrwin[widx]);
	
	noecho();		
	keypad(_attrwin[widx], TRUE);
	
	box(_attrwin[widx], 0, 0);
	
	wcolor_set(_attrwin[widx], COL_TITLE, NULL);
		
	int rowno=1;
	pKV = pKVL->First();
	while ( pKV )
	{
	    Chain key, value;

	    Tokenizer t(*pKV, Chain(VALSEPTOKEN));
	    t.nextToken(key);
	    t.nextToken(value);
	    
	    int colno=2;
	    
	    wcolor_set(_attrwin[widx], COL_TITLE, NULL);
	    mvwprintw(_attrwin[widx], rowno, colno, "%s", (char*)key);

	    colno += maxKeyLen;
	    
	    wcolor_set(_attrwin[widx], COL_CONTENT, NULL);
	    mvwprintw(_attrwin[widx], rowno, colno, "%s", (char*)value);
	    
	    rowno++;
	    pKV = pKVL->Next();
	}

	if ( _isVertical )
	    ypos += pKVL->Size() + 2;
	else
	    xpos += maxKeyLen + maxValLen + 3;

	
	wrefresh(_attrwin[widx]);
	widx++;
	pKVL = _attrList.Next();
    }
}

void Screen2::Attribute::handleKey(int c)
{
    switch(c)
    {	
    case KEY_RETURN:
    case KEY_ESC:
    {	
	for ( int i=0; i < MAX_ATTR_WIN; i++)
	{
	    if ( _attrwin[i] != 0 )
	    {
		delwin(_attrwin[i]);
		_attrwin[i]=0;
	    }
	}
	_pNext = _pParent;
	return;
    }

    case KEY_RESIZE:
    {
	/*
	for ( int i=0; i < MAX_ATTR_WIN; i++)
	{
	    if ( _attrwin[i] != 0 )
	    {
		delwin(_attrwin[i]);
		_attrwin[i]=0;
	    }
	}
	*/
	_pParent->handleKey(c);
	break;
    }
    }
    _pNext = this;    
}

WINDOW* Screen2::Attribute::getWindow()
{
    return _attrwin[0];
}

/////////////////////
// Screen2 methods //
/////////////////////

Screen2::Screen2() : SigHandler() 
{ 
    /*  Initialize ncurses  */ 
    initscr();
    curs_set(0); 
    start_color(); 

    init_pair(COL_HEADER_PASSIVE,  COLOR_WHITE,     COLOR_BLACK); 
    init_pair(COL_HEADER_ACTIVE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_TITLE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_CONTENT,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_SELECTED,  COLOR_GREEN,     COLOR_BLACK);

    init_pair(COL_INPUT,  COLOR_GREEN,  COLOR_BLACK);

    init_pair(COL_BLUE,  COLOR_BLUE,     COLOR_BLACK);
    init_pair(COL_RED,  COLOR_RED,     COLOR_BLACK);
    init_pair(COL_YELLOW,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_GREEN,  COLOR_GREEN,     COLOR_BLACK); 
    
    noecho();                  /*  Turn off key echoing                 */ 
    keypad(stdscr, TRUE);
    
    timeout(0); 

    init();

    _statusLine=Chain();
    _statusOffset=0;
    _timeout=DEFAULT_TIMEOUT;
    
#ifndef HAVE_MINGW 
    install(SIGWINCH); 
#endif
    
    _pPanel = 0;	
} 

Screen2::~Screen2() 
{
    curs_set(1); 
    delwin(stdscr); 
    clear();    
    endwin();	
    initscr();
    refresh();    
} 

void Screen2::setTimeout(int timeout)
{
    _timeout = timeout;
}

void Screen2::sigCatch(int sig) 
{ 
    try 
    {		
        clear();	
	endwin();	
	initscr();
	refresh();

#ifndef HAVE_MINGW 
	install(SIGWINCH);
#endif
	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    }
}

void Screen2::regShortCut(char c, int code) 
{ 
    _scList.Insert(ShortCut(c, code)); 
}

void Screen2::setRoot(Screen2::Panel *pPanel) 
{ 
    _pPanel = pPanel;
}

void Screen2::showScreen() 
{   
    bool doEnter=false;
    bool doReload=false;
    bool doRec=false;
    bool doRefresh=true;
    
    while ( _pPanel )
    {
	if ( doEnter )
	{
	    _pPanel->enter();
	}	
	
	if ( doReload )
	{
	    _pPanel->refresh();
	}

	refreshStatus();

	if ( doRefresh || _pPanel->doRefresh() )
	    _pPanel->show(doRec);

	doReload=false;
	doEnter=false;      
	doRec=false;
	
	int colno=LEFTOFFSET; 
	attroff(A_REVERSE); 
	
	WINDOW* pWin = _pPanel->getWindow();
	if ( pWin == 0 )
	  pWin = stdscr;
	
	wtimeout(pWin, _timeout);
	int c = wgetch(pWin); 

	if ( c == ERR )
	{
	    doReload=true;
	    doRefresh=false;
	}
	else
	{
	    doRefresh=true;
	    
	    if ( c == KEY_RESIZE )
	    {
		doRec=true;
	    }
	    _pPanel->handleKey(c);

	    // if switch to new panel, we make an immediate refresh
	    if ( _pPanel != _pPanel->nextPanel() )
	    {
		doEnter=true;
		doReload=true;
	    }
	    _pPanel = _pPanel->nextPanel();
	}
    }
}

static void refreshStatus()
{    
    if ( (char*)_statusLine )
    {
	int mainy, mainx;   
	getmaxyx(stdscr, mainy, mainx);
	mvwprintw(stdscr, mainy-1, 0, "%s", (char*)_statusLine);
	for ( int i = _statusLine.length(); i < mainx; i++ )
	    mvwprintw(stdscr, mainy-1, i, "%s", (char*)" ");
	wrefresh(stdscr);
    }
}

static void getKeyValue(const Chain& s, Chain& key, Chain& value)
{
    Tokenizer t(s, Chain(VALSEPTOKEN));
    Chain v;
    t.nextToken(key);
    t.nextToken(value);
}

static void calcOffset(WINDOW* parent, int& starty, int& startx, int fraction)
{
    int xoff,yoff;
    getbegyx(parent, yoff, xoff);
    
    int x,y;
    getmaxyx(parent, y, x);

    startx = xoff + (x / fraction);
    starty = yoff + (y / fraction);
}

