/////////////////////////////////////////////////////////////////////////////// 
// 
// Pager.cc 
// --------
// Curses based pager interface definition ( second generation ) 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2025 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Monitor 
// 
// Description: Cursor based pager implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Pager.h" 
#include "Tokenizer.h"
#include "Sleeper.h"
#include "File.h" 

#define COL_HEADER 1 
#define COL_FOOTER 2 
#define COL_CONTENT 4 
#define COL_SELECTED 5
#define COL_OK 6
#define COL_ERROR 7

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_CNTRL_A 1
#define KEY_CNTRL_D 4
#define KEY_CNTRL_E 5
#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_CNTRL_L 12
#define KEY_CNTRL_T 20
#define KEY_CNTRL_X 24
#define KEY_CNTRL_P 16
#define KEY_CNTRL_N 14
#define KEY_ESC 27 
#define KEY_ALT_H 170
#define KEY_ALT_M 181
#define KEY_DEL 127


#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define VAL_MAX_LEN 30 
#define VAL_MAX_NUM 20 
#define OK_COL_POS 6 
#define ABORT_COL_POS 16

#define HEADERSIZE 1
#define PAGESIZE 10

#define MSEC_SLEEPDELAY 70
#define MSEC_RESIZEDELAY 10

#define MODE_EDIT "Edit"
#define MODE_PAGE "Page"

#define STATUS_OK "Ok"
#define STATUS_FAILED "Failed"
#define STATUS_NONE "None"

#define HELP_EDIT 1
#define HELP_PAGE 2

#define MAXSEARCHLEN 30
#define MAXHIST 20
#define HISTSEP "@@@"

Pager::Pager(const Chain& title) : SigHandler() 
{
    
    _footerSize = 5;
    _title = title;
    _helpwin=0;
    _histSep=Chain(HISTSEP);

    initCmd();
        
#ifndef LFCNOCURSES

/*  Initialize ncurses  */ 
    initscr(); 
    curs_set(0); 
    start_color(); 
       
    init_pair(COL_HEADER,  COLOR_WHITE,     COLOR_BLUE); 
    init_pair(COL_FOOTER,  COLOR_WHITE,     COLOR_BLUE); 
    init_pair(COL_CONTENT,  COLOR_WHITE,     COLOR_BLACK); 
    init_pair(COL_SELECTED,  COLOR_GREEN,     COLOR_BLACK);
    init_pair(COL_OK,  COLOR_WHITE,     COLOR_BLUE);
    init_pair(COL_ERROR,  COLOR_WHITE,     COLOR_RED);
    
    noecho();
    keypad(stdscr, TRUE); 
    timeout(0); 

    wbkgd(stdscr, COLOR_PAIR(COL_CONTENT));
    
    init(); 

    // we disable SIGINT since this could be used by calling application
    signal(SIGINT, SIG_IGN);
    install(SIGWINCH);
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif


    _curRefresh=0;
    _rowSelected=0;
    _vStart=0;
    _rowIdx=0;
    // _offset=0;
    
} 

Pager::~Pager() 
{ 
#ifndef LFCNOCURSES 
// we need endwin and erase to reset to original terminal settings
    endwin();
    erase();
#endif 
} 

void Pager::sigCatch(int sig) 
{ 
#ifndef LFCNOCURSES
    try 
    { 
	endwin(); 	
        initscr();

	Sleeper::milliSleep(MSEC_RESIZEDELAY);
	
        install(SIGWINCH);	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    }
#endif
} 

void Pager::setHistSep(const Chain& histSep)
{
    _histSep = histSep;
}

void Pager::readHistory(const Chain& histFileName)
{
    File histFile(histFileName);
    if ( histFile.exists() )
    {
	histFile.open(File::READ);

	Chain line;
	Chain cmd;
	while ( histFile.readLine(line) )
	{
	    if ( line == Chain(_histSep) )
	    {
		_cmdHist.Insert(cmd);
		cmd = Chain();
	    }
	    else
	    {
		if ( cmd == Chain() )
		    cmd = line;
		else
		    cmd = cmd + Chain("\n") + line;
	    }
	}
    }
}

void Pager::writeHistory(const Chain& histFileName)
{
    
    File histFile(histFileName);
	
    histFile.open(File::WRITE);

    Chain *pCmd = _cmdHist.First();
    while ( pCmd )
    {
	histFile.writeChain(*pCmd);
	histFile.writeChain(Chain("\n"));		
	histFile.writeChain(_histSep);
	histFile.writeChain(Chain("\n"));	
	pCmd = _cmdHist.Next();	
    }

    histFile.close();
}


void Pager::manage()
{
#ifndef LFCNOCURSES
    _rows = LINES;
    _cols = COLS;
	
    _cntrl = 0; 

    _mode=Chain(MODE_EDIT);
    curs_set(1);

    _curCol=0;    
    _curRow=_rows-_footerSize+1;
    _curOffset=0;
    
    _terminate = false;
    _activeSearch = false;
    
    _status = Chain(STATUS_NONE);    

    if ( _cmdHist.Size() > 0 )
	_histPos=_cmdHist.Size()-1;
    else
	_histPos=0;
    
    do 
    {	

	_debug=Chain("ESC is not set");
	
        switch (_cntrl) 
        { 
	case KEY_CNTRL_X:
	{
	    // _debug=Chain("Cmd=") + getCommand();

	    ListT<Chain> schema;		
	    ListT< ListT<Chain> > outData;  
	    
	    _histPos = addHist(getCommand());
	    	    
	    if ( execute(getCommand(), _msg) )
	    {
		_status = Chain(STATUS_OK);

		if ( hasData() )
		{
		    _mode=Chain(MODE_PAGE);			    
		    schema = getSchema();		
		    outData = getData();	    
		    setPager(schema, outData);
		    managePager();
		}
	    }
	    else
	    {
		_status = Chain(STATUS_FAILED);
		setPager(schema, outData);		
	    }
	    break;   
	}
	case KEY_CNTRL_L:
	{
	    _mode=Chain(MODE_PAGE);
	    managePager();
	    break;
	}
	case KEY_CNTRL_A:
	{
	    _curCol=0;
	    break;
	}
	case KEY_CNTRL_E:
	{
	    int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset; 
	    while ( _cmdInput[bufRow][_curCol] != 0)		
		_curCol++;	    
	    break;
	}
	
	case KEY_LEFT:
	{
	    if ( _curCol > 0 )
		_curCol--;
	    break;
	}
	case KEY_RIGHT :
	{
	    int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
	    
	    if ( _curCol <  _cols && _cmdInput[bufRow][_curCol] != 0)		
		_curCol++;
	    break;
	}
	case KEY_UP :
	{
	    if ( _curRow > _rows - _footerSize + 1 )
	    {
		_curRow--;
		
		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
		
		while ( _cmdInput[bufRow][_curCol] == 0 && _curCol > 0)
		    _curCol--;
	    }
	    else if ( _curOffset > 0 )
	    {
		_curOffset--;

		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
		
		while ( _cmdInput[bufRow][_curCol] == 0 && _curCol > 0)
		    _curCol--;

	    }
	    break;
	}
	case KEY_DOWN :
	{
	    if ( _curRow <  _rows - 1 )
	    {
		_curRow++;
		
		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
				
		while ( _cmdInput[bufRow][_curCol] == 0 && _curCol > 0)
		    _curCol--;
	    }
	    else if ( _curOffset <= MAXCMDROW -_footerSize )
	    {
		_curOffset++;

		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
		
		while ( _cmdInput[bufRow][_curCol] == 0 && _curCol > 0)
		    _curCol--;

	    }
	    break;
	}
	case KEY_DEL:
	case KEY_BACKSPACE:
	{
	    if ( _curCol > 0 )
	    {
		_curCol--;
		
		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
		
		// _debug=Chain("del rowpos=") + Chain(bufRow) + Chain("cur col=") + Chain(_curCol);
		int i=_curCol;
		while ( _cmdInput[bufRow][i] > 0 )
		{
		    _cmdInput[bufRow][i] = _cmdInput[bufRow][i+1];
		    i++;
		}
		
		// _debug=Chain("del rowpos=") + Chain(bufRow) + Chain("del=") + Chain(i-_curCol);
		// _cmdInput[rowPos][i] = 0;
	    }
	    else  
	    {
		// search for end of line input of previous line

		int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;


		if ( bufRow > 0 )
		{
		    int i=0;
		    while ( _cmdInput[bufRow-1][i] > 0 )
			i++;

		    int j=0;
		    while ( _cmdInput[bufRow][j] > 0 )
			j++;

		    // we just allow backspace if concatenation fits into line buffer
		    if ( i+j < MAXCMDCOL )
		    {
			_curCol=i;
			
			// copy tail of line to end of previous line
			
			int j=0;
			while ( _cmdInput[bufRow][j] > 0 )
			{
			    _cmdInput[bufRow-1][i] = _cmdInput[bufRow][j];
			    i++;
			    j++;
			}
			
			// shift one line up
			for ( int i=bufRow; i<MAXCMDROW; i++)
			    for ( int j=0; j< MAXCMDCOL; j++)
				_cmdInput[i][j] = _cmdInput[i+1][j];
			
			
			_curRow--;
		    }
		}
	    }
	    break;
	}
	case KEY_RETURN:
	{

	    int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset;
	    
	    // shift one line down
	    for ( int i=MAXCMDROW-1; i>bufRow; i--)
		for ( int j=0; j< MAXCMDCOL; j++)
		    _cmdInput[i][j] = _cmdInput[i-1][j];

	    // copy tail to next row
	    
	    for ( int j=_curCol; j< MAXCMDCOL; j++)
	    {
		_cmdInput[bufRow+1][j-_curCol] = _cmdInput[bufRow][j];
	    }
	    // delete tail of next row
	    for ( int j= MAXCMDCOL - _curCol; j< MAXCMDCOL; j++)
	    {
		_cmdInput[bufRow+1][j] = 0;
	    }

	    // delete tail of current row
	    for ( int j=_curCol; j< MAXCMDCOL; j++)
	    {
		_cmdInput[bufRow][j] = 0;
	    }
	    _curCol=0;

	    if ( _curRow <  _rows - 1 )
		_curRow++;
	    else if ( _curOffset <= MAXCMDROW -_footerSize )	    
		_curOffset++;
		

	    break;
	}
	case KEY_ESC:
	{
	    // _debug=Chain("ESC is set");
	    
	    int sub = wgetch(stdscr);
	    switch ( sub )
	    {
	    case '+':
	    {
		// increase footer size here
		if ( _footerSize <  MAXCMDROW + 1 )
		{		
		    _footerSize++;
		    if ( _curOffset > 0 )
			_curOffset--;
		    else
			_curRow--;
		}
		break;
	    }
	    case '-':
	    {
		// shrink footer size here
		if ( _footerSize > 2 )
		{
		    _footerSize--;
		    
		    if ( _curRow < _rows  - 1)
			_curRow++;
		    else
		    {
			_curRow=_rows-_footerSize+ 1;
			_curCol=0;
		    }
		}
		break;
	    }
	    case 'h':
	    {
		showHelp(HELP_EDIT);
		break;
	    }	   
	    }
	    break;
	}
	case KEY_RESIZE:
	{
	    _rows = LINES;
	    _cols = COLS;
	    _curRow=_rows-_footerSize+ 1;
	    _curCol=0;
	    break;  
	}
	case KEY_CNTRL_T:
	{
	    _terminate=true;
	    break;
	}	
	case KEY_CNTRL_D:
	{
	    _curRow = _rows - _footerSize + 1;
	    _curCol=0;
	    _curOffset=0;
	    // init
	    for ( int i=0; i< MAXCMDROW; i++)
		for ( int j=0; j< MAXCMDCOL; j++)
		    _cmdInput[i][j] = 0;
	    break;
	}
	case KEY_CNTRL_N:
	{
	    if ( _cmdHist.Size() > 0 )
	    {
		if ( _histPos < _cmdHist.Size() - 1 )
		    _histPos++;
		else
		    _histPos = 0;
		
		setCommand(_cmdHist[_histPos]);
		
		_curRow=_rows-_footerSize+ 1;
		_curCol=0;
	    }
	    break;
	}
	case KEY_CNTRL_P:
	{
	    if ( _cmdHist.Size() > 0 )
	    {
		if ( _histPos > 0 )
		    _histPos--;
		else
		    _histPos = _cmdHist.Size()-1;
		
		setCommand(_cmdHist[_histPos]);
		
		_curRow=_rows-_footerSize+ 1;
		_curCol=0;
	    }
	    break;
	}
	case KEY_TAB:
	{
	    int c = ' ';
	    for ( int i=0;i<3;i++)
		addInput(c);
	    break;
	}
	default:
	{
	    if ( isCmdInput(_cntrl) )
	    {
		addInput(_cntrl);
	    }
	    break;
	}	
	}
	    
	if ( ! _terminate )
	{
	    writePage();
	    
	    if ( _status == Chain(STATUS_FAILED)
		 || ( _status == Chain(STATUS_OK) && hasData() == false ) )
	    {
		showMessage();		
		_status=Chain(STATUS_NONE);	    
		wrefresh(stdscr);
		_cntrl = 0; 
	    }
	    else
	    {	    
		wtimeout(stdscr, MSEC_SLEEPDELAY); 
		_cntrl = wgetch(stdscr);
	    }
	}
	
    } 
    while( ! _terminate );
        
#endif 
    return; 
}

void Pager::setPager( const ListT<Chain>& schema, const ListT< ListT<Chain> >& outData ) 
{
#ifndef LFCNOCURSES

    _schema = schema;
    
    // init outLen list
    _outLen.Empty();
    Chain *pAttr = _schema.First();
    while ( pAttr )
    {
	Tokenizer t1(*pAttr, ":");
	Chain a;
	t1.nextToken(a);
	_outLen.Insert(a.length() + 1);
	pAttr = _schema.Next();
    }

    formatData(outData);

    _curRefresh=0;
    _rowSelected=0;
    _vStart=0;
    _rowIdx=0;
    //_offset=0;
    _pdi=0;
    _pci=0;
    _ndi=0;
    _nci=0;
}

void Pager::managePager()
{
    curs_set(0);
    
    _rows = LINES;
    _cols = COLS;

    
    int i, ch = 0; 
    
    do 
    { 	
        switch (ch) 
        { 
	case KEY_DOWN :
	{
	    nextRow();
	    break;	    
	}
	case KEY_UP :
	{
	    prevRow();
	    break;
	}
	case 's' :
	{
	    firstRow();
	    break;
	}
	case 'e' :
	{
	    lastRow();
	    break;
	}
	case 'f' :
	{
	    _activeSearch = showSearch();
	    
	    if ( _activeSearch )
	    {
		_activeSearch = nextSearchResult(true);

		_curRefresh=0;
		
		if ( _activeSearch == false )
		{
		    refresh();
		    _msg = Chain("No search results");
		    
		    
		    writePage();
		    
		    showMessage();
		}
	    }
	    break;
	}
	case KEY_RETURN:
	{
	    if ( _activeSearch )
	    {
		_activeSearch = nextSearchResult(false);

		if ( _activeSearch == false )
		{
		    refresh();
		    _msg = Chain("No more search results");		    	          
		    writePage();		    
		    showMessage();
		}
	    }		
	    break;
	}
	case 'n' : 
	case ' ' :
	case KEY_NPAGE :
	{
	    nextRow();
	    int pdi=_pdi;
	    int pci=_pci;
	    for ( int i=0; i < PAGESIZE  - 1; i++ )
		nextRow();
	    _pdi=pdi;
	    _pci=pci;
	    break;
	}
	case 'p' :
	case KEY_PPAGE :
	{
	    prevRow();	    
	    int pdi=_pdi;
	    int pci=_pci;

	    for ( int i=0; i < PAGESIZE - 1; i++ )
		prevRow();
	    _pdi=pdi;
	    _pci=pci;
	    
	    break;
	}
	case KEY_RESIZE:
	{
	    _rows = LINES;
	    _cols = COLS;
	    _curRow=_rows-_footerSize+ 1;
	    break;  
	}
	case KEY_CNTRL_T:
	{
	    _terminate=true;
	    break;
	}	
	case 'h':
	{
	    showHelp(HELP_PAGE);
	    break;
	}	
	default:
	{
	    break;
	}	
	} 

	curs_set(0);
	
	writePage();
		
	wtimeout(stdscr, MSEC_SLEEPDELAY); 
	ch = wgetch(stdscr);
	
    } 
    while( ch != KEY_TAB && ! _terminate );             // continue until user hits q 

    _mode=Chain(MODE_EDIT);
    curs_set(1);
    
    // refresh(); 
    // clear();
    // endwin();

#else 
    cout << "Curses not enabled" << endl; 
#endif 
    return; 
}

bool Pager::nextRow()
{
    
    if ( _rowIdx < _rowHeight.Size()  - 1 )
    {
	if ( _rowSelected >= _rows + _vStart - _rowHeight[_rowIdx] - HEADERSIZE - _footerSize)
	    _vStart += _rowHeight[_rowIdx];
	else
	    _curRefresh=1;
	
	_rowSelected += _rowHeight[_rowIdx];

	_pdi = _rowSelected - _rowHeight[_rowIdx];
	_pci = _rowSelected - _rowHeight[_rowIdx] - _vStart + HEADERSIZE;
    
	_ndi = _rowSelected;
	_nci = _rowSelected - _vStart + HEADERSIZE;

	_rowIdx++;

	return true;
	
    }
    else
    {
	// if end of outData not reached, scroll forward linewise and remember offset
	if ( _vStart < _outData.Size() - _rows + ( HEADERSIZE + _footerSize ) )
	{
	    _vStart += 1;
	    // _offset++;
	    return true;
	}
    }

    return false;
}

bool Pager::prevRow()
{

    if ( _rowIdx > 0 )
    {
	_rowIdx--;
	
	if (_vStart >= _rowSelected - _rowHeight[_rowIdx] && _vStart > 0)
	{
	    _rowSelected -= _rowHeight[_rowIdx];
	    _vStart = _rowSelected;


	    // _debug = Chain("page mode up, ") + Chain(" RowSelectd=") + Chain(_rowSelected) + Chain(" vstart=") + Chain(_vStart);
	}
	else
	{
	    _curRefresh=1;
	    _rowSelected -= _rowHeight[_rowIdx];
	    // _debug = Chain("cursor mode up, ") + Chain(" RowSelectd=") + Chain(_rowSelected) + Chain(" vstart=") + Chain(_vStart);

	    _pdi = _rowSelected + _rowHeight[_rowIdx];
	    _pci = _rowSelected-_vStart + _rowHeight[_rowIdx] + 1;
	    
	    _ndi = _rowSelected;
	    _nci = _rowSelected - _vStart + HEADERSIZE;
	}
	return true;
    }
    return false;
}

void Pager::firstRow()
{    
    _rowIdx = 0;    
    _vStart = 0;
    _rowSelected = 0;
}

void Pager::lastRow()
{    
    _rowIdx = _rowHeight.Size() - 1;    
    _vStart = _outData.Size() - _rows +  HEADERSIZE + _footerSize;
    _rowSelected = _outData.Size() - _rowHeight[_rowIdx];
}

void Pager::writeHeader() 
{ 
#ifndef LFCNOCURSES 

    attron(A_BOLD); 
    wcolor_set(stdscr, COL_HEADER, NULL);

    // clear line
    for(int i = 0; i < _cols; i++) 
    {
	mvprintw(0, i, "%s", (char*)" "); 
    } 

    move(0,0); 

    int col=0;
    Chain *pAttr = _schema.First();
    int colPos=0;
    while ( pAttr ) 
    {
	Chain attr;
	int len;
        getAttrSpec(colPos, *pAttr, attr, len); 

	if (  col + attr.length() < _cols )
	    mvprintw(0, col, "%s", (char*)attr); 

	colPos++;
        col += len;
	pAttr = _schema.Next();
    } 

    attroff(A_BOLD); 
#else 
    throw Exception(EXLOC, "Curses not supported"); 
#endif 

} 

void Pager::writePage()
{

    if ( _curRefresh == 1 )
    {
	refreshRow(_pdi, _pci);
	refreshRow(_ndi, _nci);	
	_curRefresh=0;
	return;
    }

    writeHeader(); 

    
    wcolor_set(stdscr, COL_CONTENT, NULL);

    for(int i = 0; (i< _rows - 1 -_footerSize) && (i + _vStart < _outData.Size()); i++) 
    { 
	for(int j = 0; j < _cols; j++) 
	{
	    mvprintw(i + HEADERSIZE, j, "%s", (char*)" "); 
	} 
	
	int col=0;
	Chain *pAttr = _schema.First();
	int colPos=0;
	while ( pAttr ) 
	{
	    Chain attr;
	    int len;
	    getAttrSpec(colPos, *pAttr, attr, len); 
	    
	    Chain val = _outData[i+_vStart][colPos];
	    if ( val != Chain()  && col + val.length() < _cols )
		mvprintw(i + HEADERSIZE, col, "%s", (char*)val); 
	    col += len;
	    colPos++;
	    pAttr = _schema.Next();
	} 	    
    }

    for(int i = _outData.Size(); (i< _rows - 1 -_footerSize); i++) 
    { 
	for(int j = 0; j < _cols; j++) 
	{
	    mvprintw(i + HEADERSIZE, j, "%s", (char*)" "); 
	} 
    }

    
    if ( _rowSelected-_vStart+1 >= 0 && _mode == Chain(MODE_PAGE))
	refreshRow(_rowSelected, _rowSelected-_vStart+1);


    // writeDebug(); 

   
    writeFooter();

    refresh();
        
}


void Pager::writeFooter()
{
#ifndef LFCNOCURSES
    
    wcolor_set(stdscr, COL_FOOTER, NULL);

    attron(A_REVERSE);
    attron(A_BOLD);
    for(int i = 0; i < _cols; i++)
    {
        mvprintw(_rows-_footerSize, i, "%s", (char*)" ");
    }
    
    mvprintw(_rows-_footerSize, 0, " %s", (char*)_mode);

    if ( _mode == Chain(MODE_PAGE) )
    {
	if ( _status != Chain(STATUS_NONE) )
	{
	    if ( _status == Chain(STATUS_OK) )	 
		wcolor_set(stdscr, COL_OK, NULL);
	    else
	    {
		wcolor_set(stdscr, COL_ERROR, NULL);
	    }    
	    mvprintw(_rows-_footerSize, 7, " %s", (char*)_status);
	}
    }
    
    wcolor_set(stdscr, COL_FOOTER, NULL);

    if ( _mode == Chain(MODE_PAGE) )
	mvprintw(_rows-_footerSize, 30, "( h for Help)");
    else
	mvprintw(_rows-_footerSize, 30, "( Esc-h for Help)");

    // _debug=Chain("CurCol=") + Chain(_curCol);
    
    // mvprintw(_rows-_footerSize, 0, "Debug = %s Cntrl =  %d          ",(char*)_debug, _cntrl);

    if ( _rowHeight.Size() > 0 )
	mvprintw(_rows-_footerSize,_cols-23,"Row %04d ( %04d total)", _rowIdx+1, _rowHeight.Size());
    
    attroff(A_BOLD);
    attroff(A_REVERSE);

    wcolor_set(stdscr, COL_CONTENT, NULL);

    for ( int i=1; i<_footerSize; i++)
    {
	for(int j = 0; j < _cols; j++)
	{
	    mvprintw(_rows-_footerSize + i, j, "%s", (char*)" ");
	}
    }
    
    for ( int i=0; i<_footerSize; i++)
    {
	mvprintw(_rows-_footerSize+i+1, 0, "%s", _cmdInput[i+ _curOffset]);
    }
    wmove(stdscr, _curRow, _curCol);


#endif
}

void Pager::writeDebug() 
{ 
#ifndef LFCNOCURSES 

    wcolor_set(stdscr, COL_FOOTER, NULL);
    
    attron(A_REVERSE);
    move(_rows-1,0); 
    for(int i = 0; i < _cols; i++) 
    { 
        addch(' '); 
    } 

    mvprintw(_rows-1,0, "%s", (char*)_debug); 

    attroff(A_REVERSE); 
#endif 
} 


void Pager::refreshRow(int dataRow, int pageRow)
{
    if ( _outData.Size() == 0)
	return;
    
    int colno=0;
    
    // int rowColor = COL_CONTENT;

    wcolor_set(stdscr, COL_CONTENT, NULL);

    if ( _rowSelected == dataRow )
    {
	wcolor_set(stdscr, COL_SELECTED, NULL);
	wattron(stdscr, A_REVERSE);
	// rowColor = COL_SELECTED;
    }
    
    for(int i = 0; i < _cols; i++) 
    { 
        mvwprintw(stdscr, pageRow, i, "%s", (char*)" ");
    } 

    Chain *pAttr = _schema.First();
    int colPos=0;
    Chain *pVal = _outData[dataRow].First();
    while ( pAttr && pVal )
    {		       
	Chain attr;
	int len;
	getAttrSpec(colPos, *pAttr, attr, len);
		    
	if ( 1 ) // perhaps later we also make right align : lenInfo[0] == 'l' )
	{
	    if ( *pVal && colno + pVal->length() < _cols )
		mvwprintw(stdscr, pageRow, colno, "%s", (char*)*pVal);
	}
	else
	{
	    int offset = len - pVal->length();
	    if ( *pVal )
		mvwprintw(stdscr, pageRow, colno+offset, "%s", (char*)*pVal);	    
	}

	colPos++;
	colno = colno + len;
	
	// wcolor_set(stdscr, rowColor, NULL);
	
	pVal = _outData[dataRow].Next();
	
	pAttr = _schema.Next();
    }
    
    
    if ( _rowSelected == dataRow )
    {		
	// wcolor_set(stdscr, COL_CONTENT, NULL);
	wattroff(stdscr, A_REVERSE); 		
    }
}

void Pager::getAttrSpec(int colPos, const Chain& s, Chain& attr, int& len)
{    
    Tokenizer t1(s, ":");
    t1.nextToken(attr);
    Chain attrLen;
    t1.nextToken(attrLen);
    if ( attrLen == Chain("auto") )
	len = _outLen[colPos];
    else
	len = attrLen.asInteger();
}

void Pager::formatData(const ListT< ListT<Chain> > & outData)
{

    _outData.Empty();
    _rowHeight.Empty();
    
    ListT<Chain> *pRow = outData.First();
    
    int rowNo=0;
    while ( pRow )
    {
	int lineNo=0;
	
	bool moreLines=true;
	
	while ( moreLines )
	{
	    ListT<Chain> outRow;
	    
	    moreLines=false;

	    int colNum=0;
	    Chain *pCol = pRow->First();
	    while ( pCol )
	    {
		Tokenizer tok(*pCol, "\n");

		int i=0;
		Chain s;
		bool hasToken=false;
		while (  i <= lineNo )
		{
		    hasToken=tok.nextToken(s);		    		    
		    i++;
		}
		if ( hasToken )
		{
		    // cout << "Got token " << s << " for lineno = " << lineNo << endl;
		    outRow.Insert(s);

		    if ( _outLen[colNum] < s.length() + 1 )
			_outLen[colNum] = s.length() + 1;
		    
		    moreLines=true;
		}
		else
		{
		    // cout << "Got empty for lineno = " << lineNo << endl;
		    Chain empty;
		    outRow.Insert(empty);
		}
		colNum++;
		pCol = pRow->Next();
	    }
	    
	    if ( moreLines )
	    {
		_outData.Insert(outRow);
		lineNo++;
		rowNo++;
	    }
	}

	// cout << "Insert line height" << lineNo << endl;
	_rowHeight.Insert(lineNo);
	
	pRow = outData.Next();
    }
}

bool Pager::isCmdInput(int c)
{
    if ( isalnum (c)
	 || c == '_'
	 || c == '|'
	 || c == '-'
	 || c == '\''
	 || c == '/'
	 || c == '.'
	 || c == ','
	 || c == ';'
	 || c == '*'
	 || c == '['
	 || c == ']'
	 || c == ':'
	 || c == '@'
	 || c == '='
	 || c == '!'
	 || c == '?'
	 || c == '*'
	 || c == '+'
	 || c == '#'
	 || c == '('
	 || c == ')'
	 || c == '{'
	 || c == '}'
	 || c == '['
	 || c == ']'
	 || c == '%'
	 || c == '&'
	 || c == '$'
	 || c == '>'
	 || c == '<'
	 || c == ' ')
	return true;
    return false;
}



void Pager::addInput(int c)
{
    if ( isCmdInput(c) )
    {
	int bufRow = _curRow - ( _rows - _footerSize) - 1 + _curOffset; 
	
	// shift tail
	int i = _curCol;
	while ( _cmdInput[bufRow][i] != 0 )
	    i++;

	if ( i == MAXCMDCOL - 1 )
	    return;
	
	while ( i > _curCol )
	{
	    _cmdInput[bufRow][i] = _cmdInput[bufRow][i-1];
	    i--;
	}
	
	if ( _cmdInput[bufRow][_curCol] == 0 )
	    _cmdInput[bufRow][_curCol+1] = 0;
	
	_cmdInput[bufRow][_curCol] = c;
	
	_curCol++;
    }
}

Chain Pager::getCommand()
{
    Chain cmd;
    
    for ( int i=0; i< MAXCMDROW; i++)
    {
	Chain chunk(_cmdInput[i]);
	if ( i ==  0 )
	    cmd = chunk;
	else
	    cmd = cmd + Chain("\n") + chunk;
    }
    return cmd.cutTrailing(Chain(" \n"));
}

void Pager::setCommand(const Chain& cmd)
{
    initCmd();
    
    int i=0;
    int row=0;
    while ( row < MAXCMDROW && i < cmd.length() )
    {
	int col=0;
	bool eol=false;
	while (  i < cmd.length() && ! eol ) 
	{	    
	    if ( cmd[i] != '\n' )
	    {
		_cmdInput[row][col]=cmd[i];
		col++;
		i++;
	    }
	    else
		eol=true;
	}
	_cmdInput[row][col]=0;
	i++;
	row++;
    }
}

void Pager::showHelp(int helpMode)
{

    curs_set(0);
    
    int startx, starty;

    int xoff,yoff;
    getbegyx(stdscr, yoff, xoff);
    
    int x,y;
    getmaxyx(stdscr, y, x);

    startx = xoff + (x / 7);
    starty = yoff + (y / 7);

    
    WINDOW* helpwin = newwin( 15, 50, starty, startx);

    // wclear(_helpwin);

    keypad(helpwin, TRUE);
  
    wattron(helpwin, A_BOLD);

    if ( helpMode == HELP_PAGE )
	mvwprintw(helpwin, 1, 2, "%s", (char*)"Page Mode Commands");
    else
	mvwprintw(helpwin, 1, 2, "%s", (char*)"Edit Mode Commands");
    
    wattroff(helpwin, A_BOLD);

    box(helpwin, 0, 0);

    int i=3;
    if ( helpMode == HELP_PAGE )
    {
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Up/Down - Next / Previous line");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"s       - Skip to first line");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"e       - Skip to last line");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"n       - Next page");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"p       - Previous page");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"f       - Open search");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Return  - Next search result");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"h       - Show help");       
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Tab     - Switch to edit mode");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-t - Terminate");
    }
    else
    {
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cursor Key - Navigate");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-a    - Start of line");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-e    - End of line");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-d    - Clear");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-p    - Previous command");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-n    - Next command");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-x    - Execute command");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Esc-h      - Show help");       
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Esc-+/-    - Increase / Decrease edit area");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-l    - Switch to page mode");
	mvwprintw(helpwin, i++, 2, "%s",  (char*)"Cntrl-t    - Terminate");
    }
    
    int c = wgetch(helpwin);

    wrefresh(helpwin);
    
    delwin(helpwin);
    refresh();
    wclear(stdscr);

    curs_set(1);
    
}


void Pager::showMessage()
{
    
    curs_set(0);
    
    int startx, starty;

    int xoff,yoff;
    getbegyx(stdscr, yoff, xoff);
    
    int x,y;
    getmaxyx(stdscr, y, x);

    startx = xoff + (x / 7);
    starty = yoff + (y / 4);

    Tokenizer tok(_msg, Chain("\n"));
    Chain token;
    int cols = 10;
    int rows = 3;
    while (tok.nextToken(token) == true)
    {	    
	if ( token.length() + 4 > cols )
	    cols = token.length() + 4;
	rows++;
    }
    
    WINDOW* msgwin = newwin( rows, cols , starty, 5);

    keypad(msgwin, TRUE);
  
    wattron(msgwin, A_BOLD);
    mvwprintw(msgwin, 1, 2, "%s : %s", (char*)"Result", (char*)_status);
    wattroff(msgwin, A_BOLD);
    box(msgwin, 0, 0);

    tok.reset();
    int row=2;
    while (tok.nextToken(token) == true)
    {	    
	mvwprintw(msgwin, row, 2, "%s", (char*)token);
	row++;
    }
        
    refresh();

    int c = wgetch(msgwin);

    wclear(msgwin);
    delwin(msgwin);
    wrefresh(stdscr);
    
    curs_set(1);
    
}

bool Pager::showSearch()
{

    curs_set(1);
    
    int startx, starty;

    
    int xoff,yoff;
    getbegyx(stdscr, yoff, xoff);
    
    int x,y;
    getmaxyx(stdscr, y, x);

    startx = xoff + (x / 7);
    starty = yoff + (y / 4);

    
    WINDOW* searchwin = newwin( 4, MAXSEARCHLEN + 3, starty, 2);

    keypad(searchwin, TRUE);
  
    int curCol = 0;
    wmove(searchwin, 0, 0);
    refresh();
    
    // wmove(searchwin, 1, curcol + 1);

    int c = 0; // wgetch(msgwin);

    char searchBuffer[MAXSEARCHLEN];
    for ( int i=0; i< MAXSEARCHLEN; i++ )
	searchBuffer[i] = 0;

    do 
    {	
        switch (c) 
        { 	
	case KEY_LEFT:
	{
	    if ( curCol > 0 )
		curCol--;
	    break;
	}
	case KEY_RIGHT :
	{
	    if ( curCol < MAXSEARCHLEN )
		curCol++;
	    break;
	}
	case KEY_DEL:
	case KEY_BACKSPACE:
	{
	    if ( curCol > 0 )
	    {
		curCol--;
			       
		int i=curCol;
		while ( searchBuffer[i] > 0 )
		{
		    searchBuffer[i] = searchBuffer[i+1];
		    i++;
		}
		searchBuffer[i] = 0;	
	    }
	    break;
	}
	default:
	{
	    if ( isCmdInput(c) && curCol < MAXSEARCHLEN )
	    {		
		// mvprintw(, col, "%s", (char*)_cntrl);

		// shift tail
	        int i = curCol;
		while ( searchBuffer[i] != 0 )
		    i++;
		
		while ( i > curCol )
		{
		    searchBuffer[i] = searchBuffer[i-1];
		    i--;
		}

		if ( searchBuffer[curCol] == 0 )
		    searchBuffer[curCol+1] = 0;
		
		searchBuffer[curCol] = c;
				    
		curCol++;
	    }
	    break;
	}	
	} 

	// wclear(searchwin);
	wattron(searchwin, A_BOLD);
	mvwprintw(searchwin, 1, 2, "%s", (char*)"Enter search string");
	wattroff(searchwin, A_BOLD);

	box(searchwin, 0, 0);
	
	// clear searchbuffer
	for(int i = 1; i < MAXSEARCHLEN+2; i++) 
	{ 
	    mvwprintw(searchwin, 2, i, "%s", (char*)" ");
	} 

	
	mvwprintw(searchwin, 2, 2, "%s", (char*)searchBuffer);
	wmove(searchwin, 2, curCol + 2);
	
	// mvwprintw(searchwin, 2, 2, "Wmove ret = %d", r);

	wrefresh(searchwin);

	wtimeout(searchwin, -1); 
	c = wgetch(searchwin);
	
    } 
    while( c != KEY_RETURN && c != KEY_ESC );
    
    wclear(searchwin);
    delwin(searchwin);
    wrefresh(stdscr);
    
    curs_set(0);

    if ( c == KEY_ESC )
	return false;

    _searchPattern = Chain(searchBuffer);
    if ( _searchPattern != Chain(""))
	return true;
    return false;
}

bool Pager::nextSearchResult(bool isFirst)
{

    if ( _outData.Size() == 0 )
	return false;
    
    int rowIdx = _rowIdx;
    int rowSelected = _rowSelected;
    int vStart = _vStart;
    int pdi=_pdi;
    int pci=_pci;    
    int ndi = _ndi;
    int nci = _nci;
    
    bool moreRow=true;
    bool found=false;

    if ( isFirst  )
    {
	_matcher.setExpr(_searchPattern.toLower());
	_matcher.prepare();
    }
    else
    {
	moreRow = nextRow();
    }
    
    while ( ! found && moreRow )
    {
	found = matchRow(_rowSelected);

	// search for all line in row, also multiline rows
	
	int i=0;
       
	while ( ! found && i<_rowHeight[_rowIdx] )
	{    
	    found = matchRow(_rowSelected + i);	
	    i++;
	}	
	
	if ( ! found ) 
	    moreRow = nextRow();
    }

    if ( found )
    {
	_pdi=pdi;
	_pci=pci; 	
    }
    else
    {
	// restore to old position
	_rowIdx = rowIdx;    
	_rowSelected = rowSelected;
	_vStart = vStart;	
	_pdi = pdi;
	_pci = pci;    
	_ndi = ndi;
	_nci = nci;
    }
    return found;
}

bool Pager::matchRow(int row)
{
    Chain *pVal = _outData[row].First();
    while ( pVal )
    {	
	if ( _matcher.match(pVal->toLower()) )
	    return true;
	pVal = _outData[row].Next();	
    }
    return false;
}

int Pager::addHist(const Chain& cmd)
{
    _cmdHist.Insert(cmd);
        
    if ( _cmdHist.Size() > MAXHIST )
	_cmdHist.RemoveFirst();

    return _cmdHist.Size() - 1;
}

void Pager::initCmd()
{
    for ( int i=0; i< MAXCMDROW; i++)
	for ( int j=0; j< MAXCMDCOL; j++)
	    _cmdInput[i][j] = 0;
}
