///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoTrigger.cc  
// --------------                                                     
// Cego trigger class implementation 
//
// Design and Implementation by Bjoern Lemke               
//               
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoTrigger
//
// Description: Trigger procedure container class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// cego includes
#include "CegoTrigger.h"
#include "CegoTypeConverter.h"
#include "CegoProcException.h"
#include "CegoDatabaseFormater.h"

CegoTrigger::CegoTrigger()
{
}

CegoTrigger::CegoTrigger(const Chain& triggerName,
			 bool isBefore,
			 bool isOnInsert,
			 bool isOnUpdate,
			 bool isOnDelete,	       
			 const Chain& tableName, CegoProcBlock *pBlock)
{
    _triggerName = triggerName;
    _isBefore = isBefore;
    _isOnInsert = isOnInsert;
    _isOnUpdate = isOnUpdate;
    _isOnDelete = isOnDelete;    
    _tableName = tableName;
    _pBlock = pBlock;
}
 
CegoTrigger::~CegoTrigger()
{
    if ( _pBlock )
	delete _pBlock;
}

const Chain& CegoTrigger::getName() const
{
    return _triggerName;
}

bool CegoTrigger::isBefore() const
{
    return _isBefore;
}

bool CegoTrigger::isOnInsert() const
{
    return _isOnInsert;
}

bool CegoTrigger::isOnUpdate() const
{
    return _isOnUpdate;
}

bool CegoTrigger::isOnDelete() const
{
    return _isOnDelete;
}

const Chain& CegoTrigger::getTableName() const
{
    return _tableName;
}

CegoProcBlock* CegoTrigger::getBlock()
{
    return _pBlock;
}

void CegoTrigger::execute(ListT<CegoField>* pTriggerValueList)
{   
    _pBlock->setTriggerValueList(pTriggerValueList);
    
    CegoException excep = _pBlock->execute();
    
    if ( excep != NONE_EXCEP && excep != RETURN_EXCEP )
    {
	Chain msg = Chain("Trigger exception : ") + _pBlock->getExceptionMsg();
	throw Exception(EXLOC, msg);
    }
}

CegoTrigger& CegoTrigger::operator = ( const CegoTrigger& t)
{
    _triggerName = t._triggerName;
    _isBefore = t._isBefore;
    _isOnInsert = t._isOnInsert;
    _isOnUpdate = t._isOnUpdate;
    _isOnDelete = t._isOnDelete;    
    _tableName = t._tableName;
    _pBlock = t._pBlock;
    return (*this);
}

bool CegoTrigger::operator == ( const CegoTrigger& t) const
{
    if ( _triggerName == t._triggerName )
	return true;
    return false;
}

Chain CegoTrigger::getTriggerText(unsigned defTabSetId) const
{
    return _pBlock->toChain(defTabSetId);
}

Chain CegoTrigger::toChain(unsigned defTabSetId) const
{
    Chain s;
    s = Chain("trigger ") + _triggerName + Chain(" ");

    if ( _isBefore )
	s += Chain("before ");
    else
	s += Chain("after ");

    if ( _isOnInsert )
	s += Chain("insert ");
    if ( _isOnUpdate )
    {
	if ( _isOnInsert )
	    s += Chain("or ");
	s += Chain("update ");
    }
    if ( _isOnDelete )
    {
	if ( _isOnInsert || _isOnUpdate )
	    s += Chain("or ");		
	s += Chain("delete");
    }
    s += Chain("on ") + _tableName;

    s += Chain("\n");
    s += Chain("begin\n");

    Chain indent = Chain(DEFAULTINDENT);

    s += _pBlock->toChain(defTabSetId, indent);
    
    s += Chain("end");
    return s;
}

Chain CegoTrigger::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatTrigger(_triggerName,
				_isBefore,
				_isOnInsert,
				_isOnUpdate,
				_isOnDelete,
				_tableName,				
				_pBlock);
}
