///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoCheckObject.cc 
// ------------------
// Cego check object entry implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoCheckObject
//
// Description: Derived class from CegoDecodable to use as a container class for check objects
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoCheckObject.h"
#include "CegoXMLdef.h"

// LFC INCLUDES
#include <string.h>
#include <stdlib.h>

CegoCheckObject::CegoCheckObject()
{
    _pPredDesc = 0;
}

CegoCheckObject::CegoCheckObject(const CegoCheckObject& co) : CegoDecodableObject(co)
{
    _tabName = co._tabName;
    _pPredDesc = co._pPredDesc->clone();
}

CegoCheckObject::CegoCheckObject(ObjectType type, const Chain& objName, unsigned tabSetId) : CegoDecodableObject(type, objName, tabSetId)
{
    _pPredDesc = 0;
}

CegoCheckObject::CegoCheckObject(unsigned tabSetId, const Chain& checkName, const Chain& tabName, CegoPredicate *pPredDesc) : CegoDecodableObject(CegoObject::CHECK, checkName, tabSetId)
{
    _pPredDesc = pPredDesc;
    _tabName = tabName;
}

CegoCheckObject::~CegoCheckObject()
{
    if ( _pPredDesc )
    {
	delete _pPredDesc;
    }
}

void CegoCheckObject::setTabName(const Chain& tabName)
{
    _tabName = tabName;
}

const Chain& CegoCheckObject::getTabName() const
{
    return _tabName;
}
  
void  CegoCheckObject::setPredDesc(CegoPredicate *pPredDesc)
{
    _pPredDesc = pPredDesc;
}

CegoPredicate* CegoCheckObject::getPredDesc()
{
    return _pPredDesc;
}

unsigned CegoCheckObject::getEntrySize() const
{
    unsigned entrySize = CegoObject::getBaseSize();

    entrySize += 1; // object tabname size
    entrySize += _tabName.length();
    entrySize += _pPredDesc->getEncodingLength(0);

    return entrySize;
}

void CegoCheckObject::encode(char *buf) const
{
    char* bufPtr = buf;

    unsigned entrySize = getEntrySize();

    CegoObject::encodeBase(bufPtr, entrySize);
    bufPtr += CegoObject::getBaseSize();

    char c;

    c = (char)_tabName.length();
    memcpy (bufPtr, &c , 1);
    bufPtr++;

    memcpy(bufPtr, (char*)_tabName, _tabName.length());    
    bufPtr=bufPtr + _tabName.length();

    _pPredDesc->encode(bufPtr, 0);
}

void CegoCheckObject::decode(char *buf)
{
    char* bufPtr = buf;

    unsigned size;

    CegoObject::decodeBase(bufPtr, size);
    bufPtr += CegoObject::getBaseSize();

    bufPtr++;
    _tabName =  Chain(bufPtr);    
    bufPtr=bufPtr + _tabName.length();
    
    // TODO
    // No support for check objects which have to access table manager
    _pPredDesc = new CegoPredicate(bufPtr, 0, 0, getTabSetId() );
}

CegoCheckObject& CegoCheckObject::operator = ( const CegoCheckObject& co)
{
    CegoObject::operator=(co);

    _tabName = co._tabName;
    _pPredDesc = co._pPredDesc->clone();
    return (*this);
}

bool CegoCheckObject::operator == ( const CegoCheckObject& co)
{
    return CegoObject::operator==(co);
}

Chain CegoCheckObject::toChain(unsigned defTabSetId) const
{
    Chain s;
    s = CegoObject::toChain();

    s += Chain("TabName: ") + _tabName + Chain("\n");
    s += Chain("PredDesc: ") + _pPredDesc->toChain(defTabSetId);
    
    return s;
}

Chain CegoCheckObject::getInfo() const
{
    Chain s;
    s += Chain("Name=")+ getName() + Chain("\n");
    s += Chain("Check="); 
    s += _pPredDesc->toChain(0);
    s += Chain("\n");
    return s;    
}
