///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoLogThreadPool.cc  
// --------------------
// Cego log threadpool class implementation
//                                                         
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoLogThreadPool
//
// Description: Thread pool implementation for log message handling
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/NetHandler.h>
#include <lfcbase/Net.h>
#include <lfcbase/Sleeper.h>
#include <lfcbase/ThreadLock.h>
#include <lfcxml/XMLSuite.h>

// CEGO INCLUDES
#include "CegoDefs.h"
#include "CegoXMLdef.h"
#include "CegoLogThreadPool.h"
#include "CegoLogRecord.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

static ThreadLock queueLock("LOGQUEUE"); 
extern bool __lockStatOn;

CegoLogThreadPool::CegoLogThreadPool() : Thread()
{
    queueLock.init(LCKMNG_LOCKWAITDELAY);
}

CegoLogThreadPool::CegoLogThreadPool(CegoDatabaseManager *pDBMng) : Thread()
{
    queueLock.init(LCKMNG_LOCKWAITDELAY, __lockStatOn);

    _pDBMng = pDBMng;
    _poolLimit = 0;
    _samplePos = 0;    
    _terminated=false;

    _modId = _pDBMng->getModId("CegoLogThreadPool");
}

CegoLogThreadPool::CegoLogThreadPool(unsigned poolLimit, CegoDatabaseManager *pDBMng) : Thread()
{
    queueLock.init(LCKMNG_LOCKWAITDELAY, __lockStatOn);

    _pDBMng = pDBMng;
    _poolLimit = poolLimit;
    _samplePos = 0;    

    pDBMng->getDBHost(_logHostName);
    pDBMng->getLogPort(_logPortNo);

    _threadId = (unsigned long long*)malloc(_poolLimit * sizeof(unsigned long long));

    _threadLoad = (unsigned long long*)malloc(_poolLimit * sizeof(unsigned long long));
    for ( unsigned i = 0; i < THRMNG_NUMLOADSAMPLE; i++ )
    {
	_threadIdle[i] = (unsigned long long*)malloc(_poolLimit * sizeof(unsigned long long));
    }

    _numRequest = (unsigned long long*)malloc(_poolLimit * sizeof(unsigned long long));
    _threadState = (ThreadState*)malloc(_poolLimit * sizeof(ThreadState));
    _threadList = (CegoLogThread**)malloc(_poolLimit * sizeof(CegoLogThread*));

    _terminated=false;

    unsigned i=0;
    while ( i < _poolLimit )
    {	
	_threadState[i] = READY;
	_threadList[i]= new CegoLogThread(this, pDBMng);
	_threadId[i]=i;
	_numRequest[i]=0;

	_threadLoad[i]=0;
	for ( unsigned j = 0; j < THRMNG_NUMLOADSAMPLE; j++ )
	{
	    _threadIdle[j][i]=0;
	}

	_threadList[i]->start(&_threadId[i]);       
	i++;
    }

    _modId = _pDBMng->getModId("CegoLogThreadPool");
}

CegoLogThreadPool::~CegoLogThreadPool()  
{
    _joined=false;
    _terminated=true;

    unsigned count=0;
    while ( _joined == false && count < POOL_TERMWAIT )
    {
	Sleeper s;
	s.secSleep(1);
	count++;
    }
    
    if ( _joined )
    {
	_pDBMng->log(_modId, Logger::NOTICE, Chain("All log threads terminated"));
	join(getTid());
    }
    else
    {
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Canceling hanging log sessions ..."));
	cancel();
    }

    if ( _poolLimit > 0 )
    {    
	unsigned i=0;
	while ( i < _poolLimit )
	{
	    delete _threadList[i];
	    i++;
	}
	delete _threadId;

	delete _threadLoad;
	for ( unsigned i = 0; i < THRMNG_NUMLOADSAMPLE; i++ )
	{
	    delete _threadIdle[i];
	}

	delete _numRequest;
	delete _threadState;
    }
}

unsigned CegoLogThreadPool::getPoolLimit() const
{
    return _poolLimit;
}

void CegoLogThreadPool::getThreadInfo(unsigned i, unsigned long long& numRequest, unsigned long long& threadLoad, CegoLogThreadPool::ThreadState& state, Chain& action)
{
    state = _threadState[i];
    numRequest = _numRequest[i];
    threadLoad = _threadLoad[i];
    action = _threadList[i]->lastAction();
}

void CegoLogThreadPool::setState(unsigned i, CegoLogThreadPool::ThreadState state)
{
    _threadState[i] = state;
}

void CegoLogThreadPool::incNumRequest(unsigned i)
{
    _numRequest[i]++;
}

void CegoLogThreadPool::addThreadIdle(unsigned i, unsigned long long usec)
{
    _threadIdle[_samplePos][i] += usec;
}

void CegoLogThreadPool::setTid(unsigned i, unsigned long long tid)
{
    _threadId[i] = tid;
}

void* CegoLogThreadPool::job(void* arg)
{
    // special mode for batch import 
    if ( _poolLimit == 0 )
    {
	while ( ! _terminated )
	{
	    try 
	    {
		shiftRedoLogs();
		Sleeper s;
		s.secSleep(1);
	    }
	    catch ( Exception e)
	    {
		Chain msg;
		e.pop(msg);
		_pDBMng->log(_modId, Logger::LOGERR, Chain("Log thread failed : ") + msg); 
		// _terminated=true;
	    }
	}
	_joined=true;
    }
    else
    {
	try
	{
	    NanoTimer tim;
	    
	    Net net ( NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN, NETMNG_MAXSENDLEN );

	    if ( _logHostName != Chain() )
		net.serve(_logHostName, _logPortNo);
	    else
		net.serve6(_logPortNo);
	    
	    unsigned long long usedTime[THRMNG_NUMLOADSAMPLE];
	    for ( unsigned i = 0; i < THRMNG_NUMLOADSAMPLE; i++ )
	    {
		usedTime[i] = 0;
	    }

	    unsigned selectTimeout = _pDBMng->getSelectTimeout();
	    unsigned queueDelay = _pDBMng->getQueueDelay();
	    
	    while ( ! _terminated ) 
	    {
		usedTime[_samplePos] = 0;
		
		tim.reset();
		tim.start();
	    
		// take the mutex to avoid cpu load during wait position
		
		bool noReq=false;
		lockQueue();
		if ( _requestQueue.Size() == 0  )  noReq=true;
		unlockQueue();
		
		if ( noReq )
		    lockQueue();
		
		NetHandler* pHandle = 0;
		
		try
		{		    
		    pHandle = net.nextRequest( selectTimeout );		    
		}
		catch ( Exception e )
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Error on incoming connection : ") +  msg);
		}
		
		if ( noReq )
		{
		    unlockQueue();

		    Sleeper s;
		    s.microSleep(queueDelay);
		}
		
		if ( pHandle )
		{		    
		    _pDBMng->log(_modId, Logger::NOTICE, Chain("Connection request from <") + pHandle->getSource() + Chain(">"));
		    if ( pHandle->getMsgSize() > 0 )
		    {	
			lockQueue();
			
			if ( _requestQueue.Size() < NETMNG_MAXQUEUELEN )
			{
			    _requestQueue.Insert(pHandle);
			}
			else
			{
			    delete pHandle;
			    _pDBMng->log(_modId, Logger::NOTICE, Chain("Rejected incoming request since connection queue is full ( ") +  Chain(NETMNG_MAXQUEUELEN) + Chain(" max )"));
			}
		    	
			unlockQueue();
			
		    }
		    else
		    {
			delete pHandle;
			_pDBMng->log(_modId, Logger::NOTICE, Chain("Rejected empty request message"));
		    }		    
		}
		
		try
		{
		    // next job of the main thread is to archive redologs	    
		    shiftRedoLogs();
		}
		catch ( Exception e)
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Log thread failed : ") + msg); 
		}

		tim.stop();
		usedTime[_samplePos] += tim.getSum();
		
		tim.reset();
		tim.start();
		
		// calculate load
		unsigned i=0;
		while ( i < _poolLimit )
		{
		    unsigned long long ut=0;
		    unsigned long long ti=0;
		    for ( unsigned j=0; j< THRMNG_NUMLOADSAMPLE; j++ )
		    {
			ut+=usedTime[j];
			ti+=_threadIdle[j][i];
		    }
		    
		    unsigned long long l = 0;
		    if ( ut > 0 && ut > ti )
			l = (unsigned long long)100 - ( (unsigned long long)100 * ti ) / ut;

		    _threadLoad[i] = l > 0 ? l : 0; 
		    
		    i++;
		}
		
		_samplePos = ( _samplePos + 1 ) % THRMNG_NUMLOADSAMPLE;
		
		for ( unsigned i = 0; i < _poolLimit; i++ )
		{
		    _threadIdle[_samplePos][i]=0;
		}
	    }

	    unsigned i=0;
	    while ( i < _poolLimit )
	    {
#ifdef CGDEBUG	
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Waiting for log tread tid ") 
			     + Chain(_threadList[i]->getTid()) + Chain(" to terminate"));
#endif
		join(_threadList[i]->getTid());
		i++;
	    }
	    _joined=true;
	}
	catch ( Exception e)
	{
	    Chain msg;
	    e.pop(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, Chain("Log thread start failed : ") + msg); 
	    _terminated=true;
	}
    }
    return 0;
}

bool CegoLogThreadPool::isTerminated()
{
    return _terminated;
}

NetHandler* CegoLogThreadPool::nextRequest()
{
    lockQueue();
    NetHandler **pRequest = _requestQueue.First();
    if ( pRequest )
    {
	NetHandler *pN = *pRequest;
	_requestQueue.Remove(*pRequest);
	unlockQueue();
	return pN;
    }
    else
    {
	unlockQueue();
	return 0;
    }    
}

void CegoLogThreadPool::lockQueue()
{
    queueLock.writeLock();
}

void CegoLogThreadPool::unlockQueue()
{
    queueLock.unlock();
}

void CegoLogThreadPool::getLockStat(Chain& lockName, unsigned& lockCount, unsigned long long &numRdLock, unsigned long long &numWrLock, unsigned long long &sumRdDelay, unsigned long long &sumWrDelay)
{
    lockName = queueLock.getId();
    lockCount = queueLock.numLockTry();

    numRdLock = queueLock.numReadLock();
    numWrLock = queueLock.numWriteLock();
    sumRdDelay = 0;
    sumWrDelay = 0;
    
    if ( queueLock.numReadLock() > 0 )
	sumRdDelay = queueLock.sumReadDelay() / LCKMNG_DELRES;
    if ( queueLock.numWriteLock() > 0 )
	sumWrDelay = queueLock.sumWriteDelay() / LCKMNG_DELRES;
}

// wrapper method to trigger shiftRedoLogs method
// this is needed to rescue active online logfile after database crash
void CegoLogThreadPool::shiftActive(CegoDatabaseManager* pDBMng, const Chain& tableSet)
{

    // unsigned long modId = pDBMng->getModId("CegoLogThreadPool");
	
    ListT<Chain> lfList;
    ListT<unsigned> sizeList;
    ListT<Chain> statusList;
    
    pDBMng->getLogFileInfo(tableSet, lfList, sizeList, statusList);	
    
    Chain *pLogName = lfList.First();
    unsigned *pSize= sizeList.First();
    Chain *pStatus = statusList.First();
    
    while ( pLogName && pSize && pStatus )
    {
	
	Element *pLE = new Element(XML_LOGFILE_ELEMENT);
	pLE->setAttribute(XML_NAME_ATTR, *pLogName);
	pLE->setAttribute(XML_SIZE_ATTR, Chain(*pSize));
	
	pLE->setAttribute(XML_STATUS_ATTR, *pStatus);

	if ( *pStatus == Chain(XML_ACTIVE_VALUE) )
	{	    
	    unsigned tabSetId = pDBMng->getTabSetId(tableSet);

	    ListT<Chain> archPathList;
	    ListT<Chain> archIdList;
	    
	    pDBMng->getArchLogInfo(tabSetId, archIdList, archPathList);
    	
	    Chain *pArchPath = archPathList.First();
	    
	    while ( pArchPath )
	    {
		
		// pDBMng->log(modId, Logger::NOTICE, Chain("Archiving logfile ") + *pLogName + Chain(" to ") + *pArchPath);
						
		copyLog(pDBMng, tableSet, *pLogName, *pArchPath);
		pArchPath = archPathList.Next();
	    }    
	}
	
	pLogName = lfList.Next();
	pSize = sizeList.Next();
	pStatus = statusList.Next();
    }
}

void CegoLogThreadPool::shiftRedoLogs()
{    
    
    ListT<Chain> tsList;

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    _pDBMng->getActiveTableSet(dbHost, tsList, true, true); 

    Chain *pActiveTS = tsList.First();

    while ( pActiveTS )
    {
	
	unsigned tabSetId = _pDBMng->getTabSetId(*pActiveTS);
	
	ListT<Chain> archPathList;
	ListT<Chain> archIdList;
	ListT<Chain> occupiedLogList;
	
	_pDBMng->getArchLogInfo(tabSetId, archIdList, archPathList);
	_pDBMng->getOccupiedLogList(tabSetId, occupiedLogList);
    
	Chain *pOccupiedLog = occupiedLogList.First();
    
	while ( pOccupiedLog )
	{
	    Chain *pArchPath = archPathList.First();
	    
	    while ( pArchPath )
	    {		    
		copyLog(_pDBMng, *pActiveTS, *pOccupiedLog, *pArchPath);
		pArchPath = archPathList.Next();
	    }
	    
	    _pDBMng->setLogFileStatus(tabSetId, *pOccupiedLog, XML_FREE_VALUE);			
	    _pDBMng->doc2Xml();
	
	    pOccupiedLog = occupiedLogList.Next();
	}
	
	pActiveTS = tsList.Next();
    }
}

void CegoLogThreadPool::copyLog(CegoDatabaseManager* pDBMng, const Chain& tableSet, const Chain& logFileName, const Chain& archLogPath)
{
    File logFile(logFileName);
    
    logFile.open(File::READ);

    unsigned offset;
    logFile.readByte((char*)&offset, sizeof(unsigned));

    unsigned pos = sizeof(unsigned);
    unsigned long long lsn = 0;

    while ( pos < offset && lsn == 0 )
    {
	unsigned len;
	logFile.readByte((char*)&len, sizeof(unsigned));	
	
	// cout << "Read len : " << len << endl;
	
	pos += len + sizeof(unsigned);
	
	char* buf = new char[len];
	
	logFile.readByte(buf, len);
	CegoLogRecord lr;
	
	lr.decode(buf);
	
	// if ( lr.getAction() != CegoLogRecord::LOGREC_BUPAGE )
	lsn = lr.getLSN();
	
	delete[] buf;
    }
    
    logFile.close();

    // in case of no valid lsn entries, we skip this file 
    if ( lsn > 0 )
    {
    	
	Chain archLogFile = pDBMng->getArchiveLogName(tableSet, lsn);
	
	Chain archFileName = archLogPath + Chain("/") + archLogFile;

	unsigned long modId = pDBMng->getModId("CegoLogThreadPool");
	pDBMng->log(modId, Logger::NOTICE, Chain("Shift ") + logFileName + Chain(" to ") + archFileName);
	
	// we first write to a temp file to indicate, that the file is currently written
	// if finished, we move the file to the original name, so the file can be further processed ( e.g. for tape archiving )
	Chain tmpName = archLogPath + Chain("/.") + archLogFile;
	
	File archFile(tmpName);
	
	// _pDBMng->log(_modId, Logger::NOTICE, Chain("Archiving logfile ") + logFile.getFileName() + Chain(" to ") + archFileName); 
	
	archFile = logFile;
	
	archFile.open(File::APPEND);
	archFile.trunc(offset);
	archFile.close();
	
	archFile.rename(archFileName);	
    }
    return;
}
