///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoMediatorThread.cc  
// ---------------------                                                     
// Cego mediator thread class implementation
//                                                         
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoMediatorThread
// 
// Description: Mediator thread implementation for basic cego node operations
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// base includes
#include <lfcbase/Exception.h>
#include <lfcbase/Net.h>
#include <lfcbase/Sleeper.h>

// cego includes
#include "CegoMediatorThread.h"
#include "CegoDefs.h"
#include "CegoAdminHandler.h"

CegoMediatorThread::CegoMediatorThread(CegoDatabaseManager *pDBMng) : Thread()
{
    _pDBMng = pDBMng;
    _terminated = false;
    _modId = _pDBMng->getModId("CegoMediatorThread");
}

CegoMediatorThread::~CegoMediatorThread()  
{
    _terminated=true;
    _joined=false;
    unsigned count=0;
    while ( _joined == false && count < 10 )
    {
	Sleeper s;
	s.milliSleep(100);
	count++;
    }
    
    if ( _joined )
    {
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Mediator thread terminated"));
	join(getTid());
    }
    else
    {
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Canceling hanging mediator thread ..."));
	cancel();
    }
}

void CegoMediatorThread::getDbSpec(const Chain& dbSpecFileName, const Chain& medHost, unsigned medPort, const Chain& user, const Chain& passwd)
{
    Net n ( NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN, NETMNG_MAXSENDLEN );
    
    _pDBMng->log(_modId, Logger::NOTICE, Chain("Connecting to ") + medHost + Chain(":") + Chain(medPort) + Chain(" ..."));

    NetHandler *pN = n.connect(medHost, medPort);

    CegoAdminHandler *pAH = new CegoAdminHandler(_pDBMng, pN);

    CegoAdminHandler::ResultType res;
    res  = pAH->requestSession(user, passwd, false);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAH->getMsg(msg);	    
	_pDBMng->log(_modId, Logger::NOTICE,  msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {	
	Chain msg;
	pAH->getMsg(msg);

	delete pAH;
	delete pN;

	throw Exception(EXLOC, msg);
    }	    
    
    Chain dbSpec;
    res = pAH->reqGetDbSpec(dbSpec);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAH->getMsg(msg);
	
	_pDBMng->log(_modId, Logger::NOTICE,  msg);
		
	File dbSpecFile(dbSpecFileName);
	
	dbSpecFile.open(File::WRITE);
	dbSpecFile.writeChain(dbSpec);
	dbSpecFile.close();
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAH->getMsg(msg);

	pAH->closeSession();
	delete pAH;
	delete pN;

	throw Exception(EXLOC, msg);  
    } 

    pAH->closeSession();
    
    delete pAH;
    delete pN;
}

void* CegoMediatorThread::job(void* arg)
{
    while ( _terminated == false )
    {
	try
	{
	    _pDBMng->cleanSession(NETMNG_DBHANDLE_TTL);

	    unsigned i=0;
	    Sleeper s;
	    // NETMNG_DBHANDLE_CHK is given in seconds
	    while( i < NETMNG_DBHANDLE_CHK * 1000 && _terminated == false )
	    {
		s.milliSleep(100);
		i+=100;
	    }   
	}
	catch ( Exception e)
	{
	    e.print();
	}	
    }
    _joined = true;
    return 0;
}
