package de.lemkeit.cegojdbc;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.sql.Clob;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CegoClob implements Clob 
{

	final static Logger _logger = LoggerFactory.getLogger(CegoClob.class);

	final private int READBUFLEN = 100;
	private InputStream _is;
	private int _len;
	private char[] _clobBuf;
	
	
	public CegoClob()
	{
		_clobBuf = null;
		_is = null;
		_len = 0;
	}
	
	public CegoClob(InputStream is) throws IOException
	{
		ByteArrayOutputStream buffer = new ByteArrayOutputStream();

		int nRead;
		byte[] data = new byte[Constant.BLOBCHUNKSIZE];

		while ((nRead = is.read(data, 0, data.length)) != -1) 
		{
			buffer.write(data, 0, nRead);
		}
		// write terminating byte
		buffer.write(0);

		buffer.flush();

		_clobBuf = new String(buffer.toByteArray(), Constant.CHARSET).toCharArray();
		_len = _clobBuf.length;
		_is = null;
	}
	
	public CegoClob(InputStream is, int len)
	{
		_clobBuf = null;
		_is = is;
		_len = len;
	}
	
	public CegoClob(char[] clobBuf, int len)
	{
		if ( clobBuf[len-1] == 0)
		{
			_clobBuf = clobBuf;
		}
		else
		{
			// force zero termination
			_clobBuf = new char[len + 1];
		    System.arraycopy(clobBuf, 0, _clobBuf, 0, len);
			_clobBuf[len]=0;
		}
		_len = len;
	}

	public void allocate(long clobSize) throws SQLException
	{
		if ( clobSize > Constant.MAXBLOBSIZE )
			throw new SQLException("Clob size exceeded");
		
		_clobBuf = new char[ (int) clobSize];
		_len = (int)clobSize;
	}
	
	public void free() throws SQLException 
	{
		_clobBuf = null;
		_is = null;
		_len = 0;
	}

	public int setBytes(long pos, byte[] bytes, int offset, int len) throws SQLException, UnsupportedEncodingException 
	{
	     char[] buf = new String(bytes, (int)offset, len, Constant.CHARSET).toCharArray();
    
	     System.arraycopy(buf, 0, _clobBuf, (int) pos, len);
	     
	     /* the manual way
         for(int count=0;count<len;count++)
         {
            _clobBuf[(int)(count+pos)] = buf[ count];
         }
         */
         
         return len;
   }

	public InputStream getAsciiStream() throws SQLException 
	{
		if ( _is != null)
			return _is;
		if ( _clobBuf != null )
		{
			try 
			{
				byte[] bytes = new String(_clobBuf).getBytes(Constant.CHARSET);
				return new ByteArrayInputStream(bytes, 0, bytes.length);
			}
			catch (UnsupportedEncodingException e) 
			{
				throw new SQLException("Cannot get ascii stream :" + e.getMessage());
			}
		}
		throw new SQLException("No stream available");
	}

	public Reader getCharacterStream() throws SQLException 
	{		
		if ( _is != null)
			return new InputStreamReader(_is);

		StringReader sr = new StringReader(new String(_clobBuf));
		return sr;
	}

	public Reader getCharacterStream(long pos, long len) throws SQLException 
	{
		if ( _is != null)
		{
			String targetString = "";

			Reader reader = new InputStreamReader(_is);
			int intValueOfChar;
		    
		    try 
		    {
				while ((intValueOfChar = reader.read()) != -1) 
				{
				    targetString += (char) intValueOfChar;
				}
			} 
		    catch (IOException e) 
		    {
		    		throw new SQLException("Reader exception : " + e.getMessage());
			}
			
		    StringReader sr = new StringReader(targetString.substring((int)pos, (int)(pos + len)));
		    return sr;

		}
		else
		{
			StringReader sr = new StringReader(new String(_clobBuf, (int)pos, (int)len));
			return sr;
		}
	}

	public String getSubString(long pos, int len) throws SQLException 
	{
		 return new String(_clobBuf, (int)pos, len);
	}

	public long length() throws SQLException 
	{
		return _len;
	}

	public long position(String searchstr, long start) throws SQLException 
	{
		return new String(_clobBuf, (int)start, (int)(_len - start)).indexOf(searchstr);
	}

	public long position(Clob searchstr, long start) throws SQLException 
	{
		String s="";
		Reader reader = searchstr.getCharacterStream();
		int charsRead = -1;
		int offset = 0;
		char[] chars = new char[READBUFLEN];
		do
		{
		    try 
		    {
				charsRead = reader.read(chars, 0, chars.length);
			} 
		    catch (IOException e) 
		    {
		    		throw new SQLException("Clob read error : " + e.getMessage());
			}
		    if ( charsRead > 0 ) 
		    {
		    		s += new String(chars, offset, charsRead);
		    		offset+=charsRead;
		    }
		}
		while ( charsRead > 0 );
				
		return new String(_clobBuf, (int)start, (int)(_len - start)).indexOf(s);
	}
	
	public OutputStream setAsciiStream(long pos) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public Writer setCharacterStream(long arg0) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}

	public int setString(long pos, String s) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}

	public int setString(long arg0, String arg1, int arg2, int arg3)
			throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}

	public void truncate(long arg0) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}

}
