package de.lemkeit.cegojdbc;

import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CegoConnection implements Connection 
{

	final static Logger _logger = LoggerFactory.getLogger(CegoConnection.class);

	CegoNet _netDriver;
	boolean _isReadOnly = false;
	
	boolean _storeLocal = false;

	Map<String, Class<?>> _typeMap;
	
	public CegoConnection(CegoNet netDriver)
	{
		_netDriver = netDriver;
		_typeMap = new HashMap<String, Class<?>>();
	}
	
	public Statement createStatement() throws SQLException 
	{
		_logger.debug("Creating statement ...");
		Statement stmt = new CegoStatement(_netDriver, this, _storeLocal);	
		return stmt;
	}

	
	public PreparedStatement prepareStatement(String query) throws SQLException 
	{
		_logger.debug("Preparing statement ...");	
		CegoPreparedStatement stmt = new CegoPreparedStatement(_netDriver, this, _storeLocal);
		stmt.prepareQuery(query);	
		
		return stmt;
	}

	
	public CallableStatement prepareCall(String call) throws SQLException 
	{
		_logger.debug("Preparing call ...");		
		CegoCallableStatement stmt = new CegoCallableStatement(_netDriver, this, _storeLocal);
		stmt.prepareCall(call);	
		return stmt;
	}

	
	public String nativeSQL(String sqlString) throws SQLException 
	{
		return sqlString;
	}
	
	
	public void setAutoCommit(boolean autoCommit) throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		
		if ( autoCommit )
			s.execute("set autocommit on");
		else
			s.execute("set autocommit off");
		
		s.close();
	}
	
	
	public boolean getAutoCommit() throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		
		ResultSet rs = s.executeQuery("jdbc commitmode");

		s.close();
		
		if ( rs.next() )
		{
			String commitMode = rs.getString("COMMIT_MODE");

			rs.close();
			if ( commitMode.toUpperCase().equals(Constant.XML_TRUE_VALUE))
				return true;
			return false;
			
		}
		return false;
	}
	
	
	public void commit() throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		s.execute("commit");
		s.close();
	}
	
	
	public void rollback() throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		s.execute("rollback");
		s.close();
	}
	
	public void close() throws SQLException 
	{	
		try
		{
			_logger.debug("Closing session ...");			
			_netDriver.closeSession();
		}
		catch ( Exception e )
		{
			throw new SQLException(e);
		}

		/*
		long netTime = _netDriver.getElapsedTime();
		System.out.println(" NetDriver Time => " + netTime / 1000 + "." + netTime % 1000 + " sec");
		*/
	}

	
	public boolean isClosed() throws SQLException 
	{
		if ( _netDriver.isOpenSession() )
			return false;
		return true;
	}

	
	public DatabaseMetaData getMetaData() throws SQLException 
	{		
		return new CegoDatabaseMetaData(_netDriver, this);
	}

	
	public void setReadOnly(boolean isReadOnly) throws SQLException 
	{
		_isReadOnly = isReadOnly;
	}

	
	public boolean isReadOnly() throws SQLException 
	{
		return _isReadOnly;	
	}

	
	public void setCatalog(String catalog) throws SQLException 
	{
		// nothing to do
	}

	
	public String getCatalog() throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		ResultSet rs = s.executeQuery("jdbc catalog");
		s.close();
		
		if ( rs.next() )
		{
			String catalog = rs.getString("TABLE_CAT");

			rs.close();
			
			return catalog;
			
		}
		return null;
	}

	
	public void setTransactionIsolation(int level) throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);

		boolean levelOk = true;
		if ( level == Connection.TRANSACTION_NONE )
			s.execute("set isolation read_uncommitted");
		else if ( level == Connection.TRANSACTION_READ_UNCOMMITTED )
			s.execute("set isolation read_uncommitted");
		else if ( level == Connection.TRANSACTION_READ_COMMITTED )
			s.execute("set isolation read_committed");
		else
			levelOk = false;
		
		s.close();			
		
		if ( levelOk == false)
			throw new SQLException("Transaction level not supported");
	
	}

	
	public int getTransactionIsolation() throws SQLException 
	{
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		ResultSet rs = s.executeQuery("jdbc isolevel");

		int iso = Connection.TRANSACTION_NONE;
		
		if ( rs.next() )
		{
			String isolation = rs.getString("ISOLATION");
			
			if ( isolation.toUpperCase().equals(Constant.XML_READ_COMMITTED_VALUE))
				iso =  Connection.TRANSACTION_READ_COMMITTED;
			if ( isolation.toUpperCase().equals(Constant.XML_READ_UNCOMMITTED_VALUE))
				iso = Connection.TRANSACTION_READ_UNCOMMITTED;
			
		}
		rs.close();
		s.close();
		
		return iso;
		
	}

	
	public SQLWarning getWarnings() throws SQLException 
	{
		return null;
	}

	
	public void clearWarnings() throws SQLException 
	{
		return;
	}

	
	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");

		
		// if ( resultSetType == ResultSet.TYPE_SCROLL_INSENSITIVE )
		// 	_storeLocal = true;
		
		return createStatement();
	}

	
	public PreparedStatement prepareStatement(String query, int resultSetType, int resultSetConcurrency) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");

		return prepareStatement(query);
	}

	
	public CallableStatement prepareCall(String call, int resultSetType, int resultSetConcurrency) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");

		return prepareCall(call);
	}

	
	public void setTypeMap(Map<String, Class<?>> map) throws SQLException 
	{
		_typeMap = map;
		// throw new SQLFeatureNotSupportedException("Method not implemented");		
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public Map getTypeMap() throws SQLException 
	{
		// actually this map is empty
		return _typeMap;
	}

	
	public void setHoldability(int arg0) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException();		
	}

	
	public int getHoldability() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException();		
	}

	
	public Savepoint setSavepoint() throws SQLException 
	{
		return new CegoSavepoint();
	}

	
	public Savepoint setSavepoint(String name) throws SQLException 
	{
		return new CegoSavepoint(name);
	}

	
	public void rollback(Savepoint name) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}

	
	public void releaseSavepoint(Savepoint name) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}

	
	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");
		
		return createStatement();
	}

	
	public PreparedStatement prepareStatement(String query, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");

		return prepareStatement(query);		
	}

	
	public CallableStatement prepareCall(String query, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException 
	{
		if ( resultSetType != ResultSet.TYPE_FORWARD_ONLY && resultSetType != ResultSet.TYPE_SCROLL_INSENSITIVE )
			throw new SQLException("Just forward only and scroll insensitive result sets are supported");
		if ( resultSetConcurrency != ResultSet.CONCUR_READ_ONLY )
			throw new SQLException("Just read only result sets are supported");
		
		return prepareCall(query);		
	}

	
	public PreparedStatement prepareStatement(String query, int autoGeneratedKeys) throws SQLException 
	{
		return prepareStatement(query);
	}

	
	public PreparedStatement prepareStatement(String query, int[] columnIndexes) throws SQLException 
	{
		return prepareStatement(query);		
	}

	
	public PreparedStatement prepareStatement(String query, String[] columnNames) throws SQLException 
	{
		return prepareStatement(query);		
	}

	
	public Array createArrayOf(String arg0, Object[] arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}
	
	public Blob createBlob() throws SQLException 
	{
		return new CegoBlob();		
	}

	public Clob createClob() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public NClob createNClob() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public SQLXML createSQLXML() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public Struct createStruct(String arg0, Object[] arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public Properties getClientInfo() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public String getClientInfo(String name) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");		
	}

	
	public boolean isValid(int timeout) throws SQLException 
	{
		if ( isClosed())
			return false;
		return true;
		
		
		/* since some connection pools use the isValid method before each request,
		 * we avoid the following check 
		 
		CegoStatement s = new CegoStatement(_netDriver, this, true);
		
		ResultSet rs = s.executeQuery("jdbc prodname");
		
		String prodName = null;
		
		if ( rs.next() )
		{
			prodName = rs.getString("PROD_NAME");
		}

		rs.close();
		s.close();

		if ( prodName != null )
			return true;
		return false;
		*/
		
	}

	public void setClientInfo(Properties prop)
	{	
	}

	
	public void setClientInfo(String name, String value) 
	{
	}
	
	
	public boolean isWrapperFor(Class<?> iface) throws SQLException 
	{
		return iface.isInstance(this);	
	}

	
	public <T> T unwrap(Class<T> iface) throws SQLException 
	{
		try {
			return iface.cast(this);
		} catch (ClassCastException cce) {
			throw new SQLException("Unable to wrap");
		}		
	}
	
	CegoNet getHandler()
	{
		return _netDriver;
	}

	@Override
	public void setSchema(String schema) throws SQLException 
	{
		// TODO Auto-generated method stub	
	}

	@Override
	public String getSchema() throws SQLException 
	{
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void abort(Executor executor) throws SQLException 
	{
		// TODO Auto-generated method stub
		
	}

	@Override
	public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException 
	{
		// TODO Auto-generated method stub

	}

	@Override
	public int getNetworkTimeout() throws SQLException 
	{
		// TODO Auto-generated method stub
		return 0;
	}

}
