package de.lemkeit.cegojdbc;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;

import org.jdom2.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CegoDataRow {

	private static final Logger _logger = LoggerFactory.getLogger(CegoDataRow.class);
	
	private Element _e = null;
	private Hashtable<String, String> _ht = null;
	private String[] _rowData = null;
	
	private ByteBuffer[] _fastData = null;
	private int[] _fastType = null;
	private boolean[] _fastConverted = null;
	
	
	public CegoDataRow( Element e )
	{
		_e = e;
	}

	public CegoDataRow( Hashtable<String, String> ht )
	{
		_ht = ht;
	}
		
	public CegoDataRow( String[] rowData )
	{
		_rowData = rowData;
	}
	
	public CegoDataRow(int cols)
	{
		_fastData = new ByteBuffer[cols];
		_fastType = new int[cols];
		_fastConverted = new boolean[cols];
		for ( int i=0; i< cols ; i++)
			_fastConverted[i] = false;
	}

	public void setValue(int pos, ByteBuffer val, int type)
	{
		_fastData[pos] = val;
		_fastType[pos] = type;
	}
	
	public String getValue(String name)
	{		
		if ( _e != null )
		{
			return _e.getAttributeValue(name);
		}
		else if ( _ht != null )
		{
			return _ht.get(name);
		}
		return null;
	}
	
	public boolean hasRowArray()
	{
		if ( _rowData != null || _fastData != null)
			return true;
		return false;
	}
	
	public String getValueAsString(int i) throws Exception
	{
		// serial mode
		if ( _rowData != null )
		{
			return _rowData[i-1];
		}
		else // fast serial mode
		{
			if ( _fastData[i - 1] != null)
				_fastData[i - 1].rewind();	
			
			if ( _fastType[i - 1] == Constant.CEGO_INT_TYPE_ID )
				return Integer.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getInt()).toString();
			if ( _fastType[i - 1] == Constant.CEGO_FLOAT_TYPE_ID )
				return Float.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getFloat()).toString();
			if ( _fastType[i - 1] == Constant.CEGO_VARCHAR_TYPE_ID )
			{
				int l = _fastData[i - 1].array().length;
				// we have to strip zero byte
				return new String(_fastData[i - 1].array(), 0, l - 1);
			}
			if ( _fastType[i - 1] == Constant.CEGO_DATETIME_TYPE_ID )
			{
				Long l = Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong());
				// System.out.println("Date long value is " + l.longValue());
				Date d = new Date(l.longValue() * 1000);
				return d.toString();
			}			
			if ( _fastType[i - 1] == Constant.CEGO_LONG_TYPE_ID )
				return Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong()).toString();
			
			if ( _fastType[i - 1] == Constant.CEGO_BOOL_TYPE_ID )
			{
				Byte b = _fastData[i - 1].get();
				if ( b.intValue() > 0 ) 
						return "true";
				return "false";
			}
			
			if ( _fastType[i - 1] == Constant.CEGO_BIGINT_TYPE_ID )
			{
				int l = _fastData[i - 1].array().length;
				// we have to strip zero byte
				return new String(_fastData[i - 1].array(), 0, l - 1);
			}

			if ( _fastType[i - 1] == Constant.CEGO_DOUBLE_TYPE_ID )
				return Double.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getDouble()).toString();

			if ( _fastType[i - 1] == Constant.CEGO_DECIMAL_TYPE_ID )
			{
				int l = _fastData[i - 1].array().length;
				// we have to strip zero byte
				return new String(_fastData[i - 1].array(), 0, l - 1);
			}

			if ( _fastType[i - 1] == Constant.CEGO_SMALLINT_TYPE_ID )
				return Short.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getShort()).toString();
			
			if ( _fastType[i - 1] == Constant.CEGO_TINYINT_TYPE_ID )
				return Short.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).get()).toString();

			if ( _fastType[i - 1] == Constant.CEGO_BLOB_TYPE_ID )
				return "[" + Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong()).toString() + "]";

			if ( _fastType[i - 1] == Constant.CEGO_CLOB_TYPE_ID )
				return "[" + Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong()).toString() + "]";

			if ( _fastType[i - 1] == Constant.CEGO_NULL_TYPE_ID )
				return null;

			throw new Exception("Type still not supported");
		}
	}
	
	public Integer getValueAsInteger(int i) throws Exception
	{
		// serial mode
		if ( _rowData != null )
		{
			if ( _rowData[i-1] == null)
				return null;			
			return Integer.valueOf(_rowData[i-1]);
		}
		else // fast serial mode
		{

			_logger.debug("Fast Type id = " + _fastType[i - 1]);

			if ( _fastData[i - 1] != null)
				_fastData[i - 1].rewind();	
			
				
			if ( _fastType[i - 1] == Constant.CEGO_INT_TYPE_ID )
			{				
				_logger.debug("Retrieving value row from integer type");
				
				return Integer.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getInt());
			}		
			else if ( _fastType[i - 1] == Constant.CEGO_VARCHAR_TYPE_ID )
			{
				String val = getValueAsString(i);
				// System.out.println("String Value = <" + val + ">");
				if ( val == null)
					return null;
				if ( val.equals(new String()))
					return null;
				return Integer.valueOf(val);
			}
			if ( _fastType[i - 1] == Constant.CEGO_LONG_TYPE_ID )
			{				
				// cast from long to integer
				int intVal = Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong()).intValue();
				return Integer.valueOf(intVal);
			}	
			else if ( _fastType[i - 1] == Constant.CEGO_NULL_TYPE_ID )
			{
				return null;
			}
			throw new Exception("Cannot get int value from non-int type ( " + _fastType[i - 1] + " )");
		}
	}
	
	public Long getValueAsLong(int i) throws Exception
	{
		// serial mode
		if ( _rowData != null )
		{
			if ( _rowData[i-1] == null)
				return Long.valueOf(_rowData[i-1]);
			return null;
		}
		else // fast serial mode
		{
			
			if ( _fastData[i - 1] != null)
				_fastData[i - 1].rewind();	
			
			
			if ( _fastType[i - 1] == Constant.CEGO_LONG_TYPE_ID )
			{				
				return Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong());
			}	
			else if ( _fastType[i - 1] == Constant.CEGO_VARCHAR_TYPE_ID ||
					_fastType[i - 1] == Constant.CEGO_BIGINT_TYPE_ID )
			{
				String val = getValueAsString(i);
				if ( val == null)
					return null;
				if ( val.equals(""))
					return null;
				return Long.valueOf(val);
			}
			else if ( _fastType[i - 1] == Constant.CEGO_INT_TYPE_ID )
			{
				// cast from integer to long
				return Long.valueOf(Integer.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getInt()).longValue());
			}
			else if ( _fastType[i - 1] == Constant.CEGO_NULL_TYPE_ID )
			{
				return null;
			}
			
			throw new Exception("Cannot get long value from non-long type ( " + _fastType[i - 1] + " )");
		}
	}
	
	public java.sql.Date getValueAsDate(int i) throws Exception
	{
		// serial mode
		if ( _rowData != null )
		{
			if ( _rowData[i-1] == null)
				return null;
			
			String dateString = _rowData[i-1];
			
			SimpleDateFormat format = new SimpleDateFormat(CegoNet.getDateTimeFormat());
			java.util.Date d = null;
			try 
			{
				d = format.parse( dateString );
			} 
			catch (ParseException e) 
			{	
				throw new SQLException(e.getMessage());
			}

			return new java.sql.Date(d.getTime());
			
		}
		else // fast serial mode
		{
			if ( _fastData[i - 1] != null)
				_fastData[i - 1].rewind();	
			
			
			if ( _fastType[i - 1] == Constant.CEGO_DATETIME_TYPE_ID )
			{
				Long l = Long.valueOf(_fastData[i - 1].order(ByteOrder.LITTLE_ENDIAN).getLong());
				return new java.sql.Date(l.longValue() * 1000);
			}		
			else
			{
				throw new Exception("Cannot date value from non-date type");
			}
		}
	}
	
	public Element asElement()
	{
		return _e;
	}
	
	public String toString()
	{
		return _e.toString();
	}
}
