package de.lemkeit.cegojdbc;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.UnknownHostException;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.SQLException;
import java.util.List;
import java.util.StringTokenizer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CegoNet 
{
	private static final long serialVersionUID = 1L;
	
	final Logger _logger = LoggerFactory.getLogger(CegoNet.class);
	
	public static String _dtFormat;
	private int _protocol;
	private String _charset;
	
	private Socket _s = null;
	// private DataInput _lineInput;
	// private BufferedWriter _lineOutput;
	
	// private InputStreamReader _input;
	// private PrintStream _lineOutput;
	private OutputStream _byteOutput;
	private BufferedInputStream _byteInput;

	private String _host;
	private int _port;
	private String _tableSet;
	private String _user;
	private String _pwd;
	private String _dbProdName;
	private String _dbProdVersion;
	
	// char[] _charbuf = new char[Constant.MSGBUFLEN + Constant.SIZEBUFLEN];

	int _buflen = Constant.INITMSGBUFLEN + Constant.SIZEBUFLEN;
	byte[] _buf = new byte[_buflen]; // Constant.INITMSGBUFLEN + Constant.SIZEBUFLEN];

	private boolean _isOpenSession = false;
	private int _reqCount = 0;
	
	
	private List<CegoField> _currentSchema = null;
	
	// private long _elapsedTime = 0;
	
	public CegoNet(String host, int port, int protocol, String charset)
	{
		_host = host;
		_port = port;
		_protocol = protocol;
		_charset = charset;
		_reqCount = 0;
	}
	
	public static String getDateTimeFormat()
	{
		return _dtFormat;
	}
	
	public int getProtocol()
	{
		return _protocol;
	}
	
	public String getCharSet()
	{
		return _charset;
	}
	
	public int getReqCount()
	{
		return _reqCount;
	}
	
	private void connect() throws Exception 
	{

		try 
		{	
			_s = new Socket(_host, _port);
			_s.setTcpNoDelay(true);
			_byteInput = new BufferedInputStream(_s.getInputStream());			
			_byteOutput = _s.getOutputStream();			
		} 
		catch (UnknownHostException e) 
		{
			throw new Exception("Unknown host " + e.getMessage());
		} 
		catch (IOException e) 
		{			
			throw new Exception("IO Exception " + e.getMessage());
		}	
	}
	
	public String getHost()
	{
		return _host;
	}
	
	public int getPort()
	{
		return _port;
	}
	
	public String getTableSet()
	{
		return _tableSet;
	}
	
	public String getUser()
	{
		return _user;
	}
	
	public String getDBProdName()
	{
		return _dbProdName;
	}

	public String getDBProdVersion()
	{
		return _dbProdVersion;
	}

	
	public CegoNet fork() throws SQLException
	{
		try
		{
			CegoNet cegoNet = new CegoNet(_host, _port, _protocol, _charset);
			cegoNet.requestSession(_tableSet, _user, _pwd);
			return cegoNet;
		}
		catch ( Exception e)
		{
			throw new SQLException(e.getMessage());
		}
	}
	
	public void disconnect() throws SQLException
	{
		try
		{
			_s.close();
			_s = null;			
		}
		catch ( IOException e)
		{
			throw new SQLException(e.getMessage());
		}
	}
	
	public synchronized void requestSession(String tableSet, String user, String passwd) throws Exception
	{		
		_logger.debug("Connecting to host ...");
		connect();

		_logger.debug("Requesting session ...");

		AESCrypt aesCrypt = new AESCrypt(Constant.CEGOAESKEY, Constant.CEGOAESKEYLEN);
		String cryptpwd = aesCrypt.encrypt(passwd); 
		
		_tableSet = tableSet;
		_user = user;
		_pwd = passwd;

		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);

		dg.setSessionRequest(tableSet, user, cryptpwd);
				
		_logger.debug("Sending net message ..." + dg.asString());

		sendDoc(dg);
		
		_logger.debug("Message sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Message received");

		if ( response.getName().equals(Constant.DG_ERROR))
		{
			throw new Exception(response.getMsg());
		}

		_dbProdName = response.getDbProdName();
		_dbProdVersion = response.getDbProdVersion();
		
		// check db prod version
		StringTokenizer t = new StringTokenizer(Constant.CEGO_COMPATIBLE_VERSIONS, ",");
		
		boolean isCompatible = false;
		while ( t.hasMoreTokens() && isCompatible == false )
		{
			String checkVersion = t.nextToken();
			if ( _dbProdVersion.startsWith(checkVersion) == true )
				isCompatible = true;
		}
		if ( isCompatible == false )
		{			
			throw new Exception("Database version " + _dbProdVersion + " not compatible");
		}
		_dtFormat = response.getDateFormat(); 
					
		_isOpenSession = true;
		return;
			
	}

	public boolean isOpenSession()
	{
		return _isOpenSession;
	}
	
	@SuppressWarnings("unused")
	public synchronized void closeSession() throws SQLException
	{

		if ( _isOpenSession == false )
			throw new SQLException("No open session to close");
	
		_logger.debug("close session...");

		// P();

		try 
		{
				
			CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);
			
			if ( _protocol == Constant.NETMSG_SERIAL ||
					_protocol == Constant.NETMSG_FASTSERIAL )
			{				
				dg.addChain(Constant.SER_SESSION_CLOSE);

				sendDoc(dg);
				CegoNetMsg response = receiveDoc();
				
				if ( response.getName().equals(Constant.SER_OK))
				{
					String msg = response.nextChain();
					String affected = response.nextChain();
				}
				
			}
			else
			{
				dg.setType(Constant.XML_SESSIONCLOSE_REQUEST);
				sendDoc(dg);
				CegoNetMsg response = receiveDoc(); 
			}
			
			disconnect();
			
			_isOpenSession = false;
						
		} 
		catch (Exception e) 
		{
			throw new SQLException(e);
		}

		return;
	}

	public CegoNetMsg reqQueryOp(String query) throws Exception
	{

		if ( ! query.trim().endsWith(new String(";")) )
		{
			query = query.concat(new String(";"));
		}

		_logger.debug("Requesting query <" + query + ">");

		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);

		dg.setQueryRequest(query);
		
		sendDoc(dg);

		_logger.debug("Document sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Received response <" + response.getName() + ">");

		if ( response.getName().equals(Constant.DG_ERROR))	
		{
			throw new Exception(response.getMsg());				
		}
		return response;
		
	}

	public CegoNetMsg reqData() throws Exception
	{
		_logger.debug("Requesting more data ... ");
		sendChar(Constant.QUERY_MOREDATA);
		CegoNetMsg dataDoc = receiveDoc();
		return dataDoc;
	}
	
	@SuppressWarnings("unused")
	public void abort() throws Exception
	{
		sendChar(Constant.QUERY_ABORT);
		CegoNetMsg dataDoc = receiveDoc();	
	}
	
	public void reset() throws Exception
	{
		sendChar(Constant.QUERY_RESET);			
	} 
	

	private void sendDoc(CegoNetMsg dg) throws Exception
	{

		_logger.debug("Sending document " + dg.getName());

		// byte[] msg = dg.asString().getBytes(_charset);

		byte[] msg = dg.getBytes();

		String msgSize = Integer.valueOf(msg.length).toString();

		for ( int i = msgSize.length() ; i < Constant.SIZEBUFLEN; i++ )
		{
			msgSize += "@";
		}
		
		// String netMsg = msgSize + dg.asString();
		// _byteOutput.write(netMsg.getBytes(_charset));
		
		_byteOutput.write(msgSize.getBytes(_charset));
		
		_byteOutput.write(msg);
		
		_byteOutput.flush();
		
		// _lineOutput.write(netMsg.getBytes(_charset));
		// _lineOutput.print(msgSize);
		// _lineOutput.write(msg);
		 
	} 
	
	
	private void sendChar(char c) throws Exception
	{
		_byteOutput.write(c);
		_byteOutput.flush();
	}
		
	private void sendAck() throws Exception
	{
		char c = 1;
		_byteOutput.write(c);
		_byteOutput.flush();
	}

	/* not used
	private void sendNack() throws Exception
	{
		char c = 0;
		_byteOutput.write(c);
		_byteOutput.flush();
	}
	*/

	private CegoNetMsg receiveDoc() throws Exception
	{

		CegoNetMsg response = null;
		
		// lock area for thread synchronisation
		
		_logger.debug("Reading input ...");

		// long startTime = System.currentTimeMillis();
		// int recvByte = _lineInput.read(_charbuf, 0, Constant.MSGBUFLEN );
		int recvByte = _byteInput.read(_buf, 0, _buflen );

		/*
		long stopTime = System.currentTimeMillis();
		_elapsedTime += stopTime - startTime;
		*/
		
		// System.out.println("Received " + recvByte + " bytes");
		
		_logger.debug("Received " + recvByte + " bytes");


		int i=0;
		while ( _buf[i] != '@' && i < Constant.SIZEBUFLEN ) 
		{
			i++;
		}

		// int msgSize = new Integer ( new String(_charbuf, 0, i)).intValue();
		int msgSize = Integer.valueOf ( new String(_buf, 0, i)).intValue();

		// _logger.info("Detected msgsize is " + msgSize + " bytes");

		if ( msgSize > _buflen - Constant.SIZEBUFLEN )
		{
			// reallocation of message buffer required
			byte[] tmpbuf = new byte[msgSize + Constant.SIZEBUFLEN];
			System.arraycopy(_buf, 0, tmpbuf, 0, _buflen);
			_buflen = tmpbuf.length;
			_buf = tmpbuf;			
		}
		
		while ( recvByte < msgSize + Constant.SIZEBUFLEN )
		{
			while ( _byteInput.available() == 0 )
			{
				// _logger.debug("Waiting for input ...");
				Thread.sleep(100);
			}

			// int r = _lineInput.read(_charbuf, recvByte, Constant.MSGBUFLEN - recvByte );
			int r = _byteInput.read(_buf, recvByte, _buflen - recvByte );

			_logger.debug("Received " + r + " bytes");

			recvByte += r;					
		}

		_logger.debug("Message completed, received " + recvByte + " bytes at all");
						
	
		response = new CegoNetMsg( _buf, msgSize, _protocol, _charset, _currentSchema);	
		
		_currentSchema = response.getSchema();
		
		return response;

	}
	
	public String putBlob(Blob b) throws Exception
	{
		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);
		
		dg.setPutBlobRequest(_tableSet, b.length());
		
		sendDoc(dg);

		_logger.debug("Document sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Received response <" + response.getName() + ">");

		if ( response.getName().equals(Constant.DG_ERROR))	
		{
			throw new Exception(response.getMsg());				
		}
		else
		{
			String pstr = response.getPageId();

			InputStream is = b.getBinaryStream();

			byte[] byteBuf = new byte[Constant.CHUNKBUFSIZE];

			int len;

			_logger.debug("Sending blob data ...");

			while ( ( len = is.read(byteBuf, 0, Constant.CHUNKBUFSIZE) ) > 0 )
			{
				String msgSize = Integer.valueOf(len).toString();				

				for ( int i = msgSize.length() ; i < Constant.SIZEBUFLEN; i++ )
				{
					msgSize += "@";
				}

				_logger.debug("Sending blob chunk of size " + len);
				
				_byteOutput.write(msgSize.getBytes(_charset));
				_byteOutput.write(byteBuf, 0, len);
				_byteOutput.flush();
				
				int ack = _byteInput.read();
				// _logger.debug("ACK = " + ack);
				if ( ack != 1 )
				{
					throw new Exception("Blob write error");
				}
			}

			_logger.debug("Blob send completed");

			return "BLOB[" + pstr + "]";

		}
	}

	public String putClob(Clob c) throws Exception
	{
		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);
		
		dg.setPutClobRequest(_tableSet, c.length());
		
		sendDoc(dg);

		_logger.debug("Document sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Received response <" + response.getName() + ">");

		if ( response.getName().equals(Constant.DG_ERROR))	
		{
			throw new Exception(response.getMsg());				
		}
		else
		{
			String pstr = response.getPageId();

			InputStream is = c.getAsciiStream();

			byte[] byteBuf = new byte[Constant.CHUNKBUFSIZE];

			int len;

			_logger.debug("Sending clob data ...");

			while ( ( len = is.read(byteBuf, 0, Constant.CHUNKBUFSIZE) ) > 0 )
			{

				String msgSize = Integer.valueOf(len).toString();									
				
				for ( int i = msgSize.length() ; i < Constant.SIZEBUFLEN; i++ )
				{
					msgSize += "@";
				}

				_logger.debug("Sending blob chunk of size " + len);
				
				_byteOutput.write(msgSize.getBytes(_charset));
				_byteOutput.write(byteBuf, 0, len);
				_byteOutput.flush();
				
				int ack = _byteInput.read();
				// _logger.debug("ACK = " + ack);
				if ( ack != 1 )
				{
					throw new Exception("Clob write error");
				}
			}

			_logger.debug("Clob send completed");

			return "CLOB[" + pstr + "]";

		}
	}

	public CegoBlob getBlob(long pageId) throws Exception
	{
		CegoBlob b = null;

		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);

		dg.setGetBlobRequest(_tableSet, pageId);
				
		sendDoc(dg);

		_logger.debug("Document sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Received response <" + response.getName() + ">");

		if ( response.getName().equals(Constant.DG_ERROR))	
		{
			throw new Exception(response.getMsg());				
		}
		else
		{
			int blobSize = Integer.valueOf( response.getBlobSize() ).intValue();

			_logger.debug("Expected blobsize = " + blobSize);

			b = new CegoBlob();
			b.allocate(blobSize);

			int storedSize = 0;
			int pos = 0;

			while ( storedSize < blobSize )
			{

				_logger.debug("Receiving msg ...");

				sendAck();

				byte[] sizeBuf = new byte[Constant.SIZEBUFLEN];

				int recvByte = _byteInput.read(sizeBuf, 0, Constant.SIZEBUFLEN  );

				_logger.debug("Expecting " + recvByte + " bytes");

				int i=0;
				while ( sizeBuf[i] != '@' && i < Constant.SIZEBUFLEN ) 
				{
					i++;
				}

				int msgSize = Integer.valueOf ( new String(sizeBuf, 0, i) ).intValue();

				_logger.debug("MsgSize=" + msgSize + " bytes");

				byte[] byteBuf = new byte[msgSize];

				recvByte = 0;
				while ( recvByte < msgSize )
				{
					_logger.debug("Read data ..");

					int r = _byteInput.read(byteBuf, recvByte, msgSize - recvByte );
					recvByte += r;					
					_logger.debug("RecvByte=" + recvByte);

				}

				b.setBytes(pos, byteBuf, 0, msgSize);
				pos += msgSize;
				storedSize += msgSize;
			}
		}

		return b;
	}
	
	public CegoClob getClob(long pageId) throws Exception
	{
		CegoClob c = null;

		CegoNetMsg dg = new CegoNetMsg(_protocol, _charset);

		dg.setGetClobRequest(_tableSet, pageId);
				
		sendDoc(dg);

		_logger.debug("Document sent");

		CegoNetMsg response = receiveDoc();

		_logger.debug("Received response <" + response.getName() + ">");

		if ( response.getName().equals(Constant.DG_ERROR))	
		{
			throw new Exception(response.getMsg());				
		}
		else
		{
			int clobSize = Integer.valueOf( response.getClobSize() ).intValue();

			_logger.debug("Expected clobsize = " + clobSize);

			c = new CegoClob();
			c.allocate(clobSize);

			int storedSize = 0;
			int pos = 0;

			while ( storedSize < clobSize )
			{

				_logger.debug("Receiving msg ...");

				sendAck();

				byte[] sizeBuf = new byte[Constant.SIZEBUFLEN];

				int recvByte = _byteInput.read(sizeBuf, 0, Constant.SIZEBUFLEN  );

				_logger.debug("Expecting " + recvByte + " bytes");

				int i=0;
				while ( sizeBuf[i] != '@' && i < Constant.SIZEBUFLEN ) 
				{
					i++;
				}

				int msgSize = Integer.valueOf ( new String(sizeBuf, 0, i) ).intValue();

				_logger.debug("MsgSize=" + msgSize + " bytes");

				byte[] byteBuf = new byte[msgSize];

				recvByte = 0;
				while ( recvByte < msgSize )
				{
					_logger.debug("Read data ..");

					int r = _byteInput.read(byteBuf, recvByte, msgSize - recvByte );
					recvByte += r;					
					_logger.debug("RecvByte=" + recvByte);

				}

				c.setBytes(pos, byteBuf, 0, msgSize);
				pos += msgSize;
				storedSize += msgSize;
			}
		}

		return c;
	}

	public static long getSerialversionuid() {
		return serialVersionUID;
	}

}
