package de.lemkeit.cegojdbc;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.jdom2.DocType;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.output.XMLOutputter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CegoNetMsg 
{

	private static final Logger _logger = LoggerFactory.getLogger(CegoNetMsg.class);
	
	Document _doc = null;
	
	SAXBuilder _builder = null;
	CegoXMLGram _xml = null; 
	
	String _serial;
	
	String _charset;
	
	byte[] _fastSerial;
	int _fastPos;
	int _fastSize;
	
	CegoSerialTokenizer _serTok;
	
	String _dgName;
	
	String _serFormat;
	List<CegoField> _serSchema = null;
	String _serMsg;
	String _serTid;
	String _serDbProdName;
	String _serDbProdVersion;
	String _serDateTimeFormat;
	
	
	String _serPageId;
	String _serBlobSize;
	String _serClobSize;
	String _serAffected;
	
	
	List<CegoDataRow> _rowList;
	List<CegoDataRow> _outParamList;
	
	int _protocol;
	
	public CegoNetMsg(int protocol, String charset)
	{
		_protocol = protocol;
		_charset = charset;
		
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			_builder = new SAXBuilder();
			_doc = new Document();
			Element rootElement = new Element(Constant.XML_FRAME_ELEMENT);
			_doc.setRootElement(rootElement);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			_xml = new CegoXMLGram();
		}
		else if ( _protocol == Constant.NETMSG_SERIAL )
		{
			_serial = new String();
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			_fastSerial = new byte[Constant.FASTBUFFER];
			_fastSize = Constant.FASTBUFFER;
			_fastPos = 0;
		}
	}

	public CegoNetMsg(byte[] buf, int size, int protocol, String charset, List<CegoField> schema) throws Exception
	{
		_protocol = protocol;
		_charset = charset;
		_serSchema = schema;
		
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			String dgstring = new String(buf, Constant.SIZEBUFLEN, size, charset);
			
			try {
		
				_builder = new SAXBuilder();
				
				_logger.debug("Building doc from input\n" + dgstring);
				_doc = _builder.build( new ByteArrayInputStream( dgstring.getBytes(Constant.ENCODING)));
				_logger.debug("Building doc done");
				
				_dgName = _doc.getDocType().getElementName();
				
			} catch (JDOMException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			String dgstring = new String(buf, Constant.SIZEBUFLEN, size, charset);

			try
			{
				_xml = new CegoXMLGram();
				_logger.debug("Building doc from input\n" + dgstring);
				_xml.setXML(dgstring);
				_xml.parse();
				_logger.debug("Building doc done");				

				_dgName = _xml.getName();

			} 
			catch ( java.lang.Exception e)
			{
				e.printStackTrace();
			}
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL 
				|| _protocol == Constant.NETMSG_SERIAL )
		{

			if ( _protocol == Constant.NETMSG_SERIAL )
			{
						
				String dgstring = new String(buf, Constant.SIZEBUFLEN, size, charset);
				_serial = dgstring;
				_serTok = new CegoSerialTokenizer(_serial);
			}
			else
			{
				_fastSerial = buf;
				_fastPos = Constant.SIZEBUFLEN;
				_fastSize = size + Constant.SIZEBUFLEN;
			}
			
			String dgName = nextChain();

			if ( dgName.equals(Constant.SER_SDATA))
			{
				_dgName = Constant.DG_DATA;
				_serFormat = nextChain();
				readSchema();

				_logger.debug("Schema read");				

				_rowList = new LinkedList<CegoDataRow>();

				while ( hasMoreData() )
				{
					_rowList.add( readRow() );
				}

				_logger.debug("Rowlist prepared");				

			}
			else if ( dgName.equals(Constant.SER_FDATA))
			{
				_dgName = Constant.DG_DATA;
				_rowList = new LinkedList<CegoDataRow>();

				while ( hasMoreData() ) 
				{
					_rowList.add( readRow() );
				}				
			}
			else if ( dgName.equals(Constant.SER_OK))
			{
				_dgName = Constant.DG_OK;
				_serMsg = nextChain();
				_serAffected = nextChain();
			}
			else if ( dgName.equals(Constant.SER_ERROR))
			{
				_dgName = Constant.DG_ERROR;
				_serMsg = nextChain();
			}
			else if ( dgName.equals(Constant.SER_INFO))
			{
				_dgName = Constant.DG_INFO;
			}
			else if ( dgName.equals(Constant.SER_FIN))
			{
				_dgName = Constant.DG_OK;
			}
			else if ( dgName.equals(Constant.SER_SACK))
			{
				_dgName = Constant.DG_SACK;
				_serMsg = nextChain();
				_serTid = nextChain();
				_serDbProdName = nextChain();
				_serDbProdVersion = nextChain();	
				_serDateTimeFormat = nextChain();
			}
			else if ( dgName.equals(Constant.SER_PROCRES))
			{
				_dgName = Constant.DG_OK;

				_outParamList = new LinkedList<CegoDataRow>();

				while ( hasMoreData() ) 
				{
					_outParamList.add( readOutParam() );
				}
			}
			else if ( dgName.equals(Constant.SER_BLOBINFO))
			{
				_dgName = Constant.DG_BLOBINFO;
				_serPageId = nextChain();

			}
			else if ( dgName.equals(Constant.SER_BLOBSIZE))
			{
				_dgName = Constant.DG_BLOBSIZE;
				_serBlobSize = nextChain();
			}
			else if ( dgName.equals(Constant.SER_CLOBINFO))
			{
				_dgName = Constant.DG_CLOBINFO;
				_serPageId = nextChain();

			}
			else if ( dgName.equals(Constant.SER_CLOBSIZE))
			{
				_dgName = Constant.DG_CLOBSIZE;
				_serClobSize = nextChain();
			}
		}
	}
		
	public void setType(String dgName)
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			_doc.setDocType(new DocType(dgName));
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			_xml.setName(dgName);
		}
	}

	
	public String getName()
	{
		return _dgName;
	}

	
	public String getMsg()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_MSG_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_MSG_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serMsg;
		}
		return null;
	}

	public String getAffected()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_AFFECTED_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_AFFECTED_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serAffected;
		}
		return null;
	}
	
	public String getTid()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_TID_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_TID_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serTid;
		}
		return null;
	}

	public String getPageId()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_PAGEID_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_PAGEID_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serPageId;
		}
		return null;
	}

	public String getBlobSize()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_SIZE_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_SIZE_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serBlobSize;
		}
		return null;
	}

	public String getClobSize()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_SIZE_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_SIZE_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serClobSize;
		}
		return null;
	}

	
	public String getDbProdName()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_DBPRODNAME_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_DBPRODNAME_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serDbProdName;
		}
		return null;
	}
	
	public String getDbProdVersion()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_DBPRODVERSION_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_DBPRODVERSION_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serDbProdVersion;
		}
		return null;
	}

	public String getDateFormat()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(Constant.XML_DATETIMEFORMAT_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(Constant.XML_DATETIMEFORMAT_ATTR);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serDateTimeFormat;
		}
		return null;
	}

	
	public String asString()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			XMLOutputter outp = new XMLOutputter();
			return outp.outputString(_doc);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.toXML();
		}
		else if ( _protocol == Constant.NETMSG_SERIAL )
		{
			return _serial;
		}

		return null;
	}

	
	public byte[] getBytes() throws UnsupportedEncodingException
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			XMLOutputter outp = new XMLOutputter();
			return outp.outputString(_doc).getBytes(_charset);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.toXML().getBytes(_charset);
		}
		else if ( _protocol == Constant.NETMSG_SERIAL )
		{
			return _serial.getBytes(_charset);
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{			
			return Arrays.copyOfRange(_fastSerial, 0, _fastPos);
		}
	
		return null;
	}

	

	private void setAttribute(String name, String value)
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			_doc.getRootElement().setAttribute(name, value);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			_xml.setAttribute(name, value);
		}		
	}
	
	public String getAttribute(String name)
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			return _doc.getRootElement().getAttributeValue(name);
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getAttribute(name);
		}
		return null;
	}

	public void setSessionRequest(String tableSet, String user, String cryptpwd)
	{
		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_SESSION);
			addChain(tableSet);
			addChain(user);
			addChain(cryptpwd);
		}
		else
		{
			setType(Constant.XML_DBSESSION_REQUEST);
			setAttribute(Constant.XML_USER_ATTR, user);
			setAttribute(Constant.XML_PASSWD_ATTR, cryptpwd);
			setAttribute(Constant.XML_TABLESET_ATTR, tableSet);
		}
	}

	public void setQueryRequest(String query)
	{

		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_QUERY);
			addChain(query);
		}
		else
		{
			setType(Constant.XML_QUERY_REQUEST);
			setAttribute(Constant.XML_CMD_ATTR, query);
		}
	}

	public void setPutBlobRequest(String tableSet, long blobSize)
	{

		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_PUTBLOB);
			addChain(tableSet);
			addChain(Long.valueOf(blobSize).toString());
		}
		else
		{
			setType(Constant.XML_PUTBLOB_REQUEST);
			setAttribute(Constant.XML_TABLESET_ATTR, tableSet);
			setAttribute(Constant.XML_SIZE_ATTR, Long.valueOf(blobSize).toString());
		}
	}
	
	public void setPutClobRequest(String tableSet, long clobSize)
	{

		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_PUTCLOB);
			addChain(tableSet);
			addChain(Long.valueOf(clobSize).toString());
		}
		else
		{
			setType(Constant.XML_PUTCLOB_REQUEST);
			setAttribute(Constant.XML_TABLESET_ATTR, tableSet);
			setAttribute(Constant.XML_SIZE_ATTR, Long.valueOf(clobSize).toString());
		}
	}
	
	public void setGetBlobRequest(String tableSet, long pageId)
	{

		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_GETBLOB);
			addChain(tableSet);
			addChain(Long.valueOf(pageId).toString());

		}
		else
		{
			setType(Constant.XML_GETBLOB_REQUEST);
			setAttribute(Constant.XML_TABLESET_ATTR, tableSet);
			setAttribute(Constant.XML_PAGEID_ATTR, Long.valueOf(pageId).toString());
		}
	}

	public void setGetClobRequest(String tableSet, long pageId)
	{

		if ( _protocol == Constant.NETMSG_SERIAL || _protocol == Constant.NETMSG_FASTSERIAL )
		{
			addChain(Constant.SER_GETCLOB);
			addChain(tableSet);
			addChain(Long.valueOf(pageId).toString());

		}
		else
		{
			setType(Constant.XML_GETCLOB_REQUEST);
			setAttribute(Constant.XML_TABLESET_ATTR, tableSet);
			setAttribute(Constant.XML_PAGEID_ATTR, Long.valueOf(pageId).toString());
		}
	}

	public List<CegoField> getSchema()
	{
		if ( _protocol == Constant.NETMSG_JDOM )
		{
			List<CegoField> schema = new ArrayList<CegoField>();

			Element rootElement = _doc.getRootElement();
			List<Element> childList = rootElement.getChildren(Constant.XML_SCHEMA_ELEMENT);
			for ( int i=0; i< childList.size() ; i++)
			{
				Element colElement = (Element) childList.get(i);
				schema.add(new CegoField(colElement.getAttributeValue(Constant.XML_COLNAME_ATTR),
						0, // colElement.getAttributeValue(Constant.XML_COLTYPE_ATTR),
						Integer.valueOf(colElement.getAttributeValue(Constant.XML_COLSIZE_ATTR)).intValue(),
						Integer.valueOf(colElement.getAttributeValue(Constant.XML_JAVATYPE_ATTR) ).intValue()));
			}			

			return schema;
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getSchema();
		}
		else if ( _protocol == Constant.NETMSG_SERIAL )
		{
			return _serSchema;
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _serSchema;
		}

		return null;
	}

	public List<CegoDataRow> getRowList()
	{
		if ( _protocol == Constant.NETMSG_JDOM  )
		{
			List<CegoDataRow> rowList = new LinkedList<CegoDataRow>();
			
			Iterator<Element> it = _doc.getRootElement().getChildren(Constant.XML_ROW_ELEMENT).iterator();
			while ( it.hasNext() )
			{
				Element e = (Element)it.next();
				rowList.add( new CegoDataRow(e));
			}
			if ( rowList.size() > 0 )
				return rowList;
			return null;
			
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			if ( _xml.getRowList().size() > 0 )
				return _xml.getRowList();
			return null;
		}
		else if ( _protocol == Constant.NETMSG_SERIAL 
				|| _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _rowList;
			
		}
		
		return null;
	}

	public List<CegoDataRow> getOutParamList()
	{
		if ( _protocol == Constant.NETMSG_JDOM  )
		{
			List<CegoDataRow> rowList = new LinkedList<CegoDataRow>();
			
			Iterator<Element> it = _doc.getRootElement().getChildren( Constant.XML_OUTPARAM_ELEMENT).iterator();
			while ( it.hasNext() )
			{
				Element e = (Element)it.next();
				rowList.add( new CegoDataRow(e));
			}
			return rowList;
			
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			return _xml.getOutParamList();
		}
		else if ( _protocol == Constant.NETMSG_SERIAL 
				|| _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _outParamList;
			
		}
		return null;
	}

	public void addRows(List<CegoDataRow> rowList) 
	{
		
		if ( _protocol == Constant.NETMSG_JDOM  )
		{
			Iterator<CegoDataRow> it = rowList.iterator();
			while ( it.hasNext() )
			{
				Element e = (Element) it.next().asElement().clone();
				_logger.debug("Adding row ..... " + e.toString());
				
				_doc.getRootElement().addContent(e);
			}
		}
		else if ( _protocol == Constant.NETMSG_XMLNATIVE )
		{
			Iterator<CegoDataRow> it = rowList.iterator();
			while ( it.hasNext() )
			{	
				// Element e = (Element) it.next().asElement().clone();
				// _logger.debug("Adding row ..... " + e.toString());
				_xml.getRowList().add(it.next());
			}
		}
		else if ( _protocol == Constant.NETMSG_SERIAL 
				|| _protocol == Constant.NETMSG_FASTSERIAL )
		{
			if ( _rowList == null)
			{
				_rowList = new LinkedList<CegoDataRow>();
			}
			Iterator<CegoDataRow> it = rowList.iterator();
			while ( it.hasNext() )
			{	
				// Element e = (Element) it.next().asElement().clone();
				 _logger.debug("Adding row ..... " );
				_rowList.add(it.next());
			}
		}
		

	}
	
	//////////////////////////////
	// Dedicated serial methods //
    //////////////////////////////
	
	private boolean hasMoreData()
	{
		if ( _protocol == Constant.NETMSG_SERIAL )
		{
			return _serTok.hasMoreTokens();
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			return _fastPos < _fastSize; 
		}
		return false;
	}
	
	public void addChain(String value)
	{
		if ( _protocol == Constant.NETMSG_SERIAL )
		{
			if ( _serial != null )
				_serial = _serial + new String(Constant.SER_SEP);
			
			if ( value == null )
				_serial = _serial + new String(Constant.SER_NULL);
			else
			{
				_serial = _serial 
					+ Integer.valueOf(value.getBytes().length).toString() 
					+ new String(Constant.SER_SEP)
					+ value;
			}
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			int l = value.getBytes().length;
			
			ByteBuffer b = ByteBuffer.allocate(4);
			b.order(ByteOrder.LITTLE_ENDIAN).putInt(l);
			
			// TODO : check if _fastSerial size exceeds
			
			if ( _fastPos + l + 4 > _fastSize )
			{
				_fastSerial = Arrays.copyOf(_fastSerial, _fastPos + l + 4 );
				_fastSize = _fastPos + l + 4;
			}
									
			System.arraycopy(b.array(), 0, _fastSerial, _fastPos, 4);
			_fastPos+=4;
			System.arraycopy(value.getBytes(), 0, _fastSerial, _fastPos, l);
			_fastPos+=l;
		}	
	}

	public String nextChain()
	{
		if ( _protocol == Constant.NETMSG_SERIAL )
		{
			// System.out.println("Calling next Token");
			return _serTok.nextToken();
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{			
			ByteBuffer b = ByteBuffer.allocate(4);

			System.arraycopy(_fastSerial, _fastPos, b.array(), 0, 4);
			_fastPos += 4;

			int l = b.order(ByteOrder.LITTLE_ENDIAN).getInt();
			String s = null;
			
			if ( l > 0 )
			{
				s = new String(_fastSerial, _fastPos, l);
				_fastPos += l;
			}	
			
			return s;
		}
		return null;
	}

	@SuppressWarnings("unused")
	public void readSchema() throws Exception
	{				
		_serSchema = new ArrayList<CegoField>();
		
		int schemaSize;
		if ( _protocol == Constant.NETMSG_SERIAL )
		{
			schemaSize = Integer.valueOf ( _serTok.nextToken() ).intValue();
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			
			ByteBuffer b = ByteBuffer.allocate(4);

			System.arraycopy(_fastSerial, _fastPos, b.array(), 0, 4);
			_fastPos += 4;			
			schemaSize = b.order(ByteOrder.LITTLE_ENDIAN).getInt();
			
		}
		else
		{
			throw new Exception("Method not supported for protocol type");
		}
		
		for ( int i=0; i< schemaSize ; i++)
		{		
			String colTable = nextChain();			
			String colName = nextChain();
			String colNull = nextChain();
			String colDef = nextChain();
			String colTypeId = nextChain();
			
			int colSize = Integer.valueOf(nextChain()).intValue();
					
			_serSchema.add(new CegoField(colName, Integer.valueOf(colTypeId).intValue(), colSize, 0));
		}
	}


	CegoDataRow readRow() throws Exception
	{

		// List<CegoField> serSchema = new ArrayList<CegoField>();

		int cols = 0;
		if ( _protocol == Constant.NETMSG_SERIAL )
		{
			cols = Integer.valueOf ( _serTok.nextToken() ).intValue();

			// Hashtable<String, String> ht = new Hashtable<String, String>();
			
			_logger.debug("Reading " + cols + " cols" );
			// System.out.println("Reading " + cols + " cols");
			
			String[] rowData = new String[cols];
			
			for ( int i=0; i< cols ; i++)
			{

				String val = _serTok.nextToken();
				
				if ( val.equals(Constant.SER_NULL) )
				{
					rowData[i] = null;
				}
				else
				{
					rowData[i] = val;
				}
			}
			
			return new CegoDataRow( rowData );

		
		}
		else if ( _protocol == Constant.NETMSG_FASTSERIAL )
		{
			ByteBuffer cb = ByteBuffer.allocate(4);
			System.arraycopy(_fastSerial, _fastPos, cb.array(), 0, 4);
			_fastPos += 4;
						
			cols = cb.order(ByteOrder.LITTLE_ENDIAN).getInt();

			CegoDataRow dataRow = new CegoDataRow(cols);

			for ( int i=0; i< cols ; i++)
			{

				ByteBuffer fb = ByteBuffer.allocate(4);
				System.arraycopy(_fastSerial, _fastPos, fb.array(), 0, 4);
				_fastPos += 4;
				int l = fb.order(ByteOrder.LITTLE_ENDIAN).getInt();
				if ( l > 0 )
				{
					ByteBuffer vb = ByteBuffer.allocate(l);
					System.arraycopy(_fastSerial, _fastPos, vb.array(), 0, l);
					_fastPos += l;				
					dataRow.setValue(i, vb, _serSchema.get(i).getColTypeId());
				}
				else
				{
					dataRow.setValue(i, null, Constant.CEGO_NULL_TYPE_ID);
				}
			}
			return dataRow;
		}
		else
		{
			throw new Exception("Method not supported for protocol type");
		}
	}
	
	CegoDataRow readOutParam()
	{

		Hashtable<String, String> ht = new Hashtable<String, String>();
		
		String paramName = nextChain();
		String paramType = nextChain();
		String paramValue = nextChain();

		// System.out.println("Reading Param " + paramName + " " + paramValue);
		ht.put(Constant.XML_NAME_ATTR, paramName);
		ht.put(Constant.XML_TYPE_ATTR, paramType);
		ht.put(Constant.XML_VALUE_ATTR, paramValue);
			
		return new CegoDataRow( ht );
	}	

}
