package de.lemkeit.cegojdbc;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
// import java.sql.PreparedStatement;
import java.sql.NClob;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.StringTokenizer;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CegoPreparedStatement extends CegoStatement implements java.sql.PreparedStatement 
{

	private static final Logger _logger = LoggerFactory.getLogger(CegoPreparedStatement.class);
	
	private Vector<String> _frags = new Vector<String>();
	private String[] _paramArray = null;
	private Vector<Blob> _blobVector = new Vector<Blob>();
	private Vector<Clob> _clobVector = new Vector<Clob>();
	
	private int _numParam = 0;
	
	private CegoNet _netDriver;
	
	public CegoPreparedStatement(CegoNet netDriver, CegoConnection con, boolean storeLocal) 
	{	
		super(netDriver, con, storeLocal);
		_netDriver = netDriver;
		
	}
	
	public void prepareQuery(String query) throws SQLException 
	{
	
		query += new String(" ");
		_frags = new Vector<String>();
		
		boolean isSingleQuote = false;
		int lastEnded = 0;
		int i;
		
		for ( i=0; i<query.length();i++)
		{
			switch ( query.charAt(i) )
			{
			case '\'':
				isSingleQuote = ! isSingleQuote;
				break;
				
			case '?':
				if ( ! isSingleQuote );
				{
					_frags.add(query.substring(lastEnded, i));
					_numParam++;
					lastEnded = i+1;
				}
				break;
				
			default:
			
				break;
			}
		}
		_frags.add(query.substring(lastEnded, i-1));
		_paramArray = new String[_numParam];	
	}

	
	public boolean execute() throws SQLException 
	{
		handleLobs();
				
		String paramizedQuery = new String();

		int i;
		for ( i = 0; i < _numParam; i++)
		{
			paramizedQuery += _frags.get(i) + _paramArray[i];
		}
		paramizedQuery += _frags.get(i);
				
		return super.execute(paramizedQuery);
	
	}
	
	
	public ResultSet executeQuery() throws SQLException 
	{

		handleLobs();

		String paramizedQuery = new String();
		
		int i;
		for ( i = 0; i < _numParam; i++)
		{
			paramizedQuery += _frags.get(i) + _paramArray[i];
		}
		paramizedQuery += _frags.get(i);
		
		_logger.debug("Paramized query is " + paramizedQuery);
		
		// System.out.println("Query=" + paramizedQuery);
		
		ResultSet rs = super.executeQuery(paramizedQuery);
		return rs;
		
	}

	
	public int executeUpdate() throws SQLException 
	{
		
		handleLobs();
		
		String paramizedQuery = new String();

		int i;
		for ( i = 0; i < _numParam; i++)
		{
			paramizedQuery += _frags.get(i) + _paramArray[i];
		}
		paramizedQuery += _frags.get(i);
		
		
		// System.out.println("Update=" + paramizedQuery);
		
		return super.executeUpdate(paramizedQuery);
		
	}

	public void setParam(int pos, String paramName) throws SQLException 
	{
		
		if ( pos >= 1 && pos <= _numParam )
		{
			// System.out.println("Setting param=" + paramName);
			
			_paramArray[pos-1] = paramName;
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
		
	}
	
	public void setNull(int pos, int type) throws SQLException 
	{
		
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = null;
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
		
	}

	
	public void setBoolean(int pos, boolean val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			if ( val )
				_paramArray[pos-1] = "true";
			else
				_paramArray[pos-1] = "false";
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}		
	}
	
	public void setByte(int arg0, byte arg1) throws SQLException
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setShort(int arg0, short arg1) throws SQLException
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setInt(int pos, int val) throws SQLException 
	{

		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = Integer.valueOf(val).toString();
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
	}
	
	public void setLong(int pos, long val) throws SQLException 
	{
		
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = new String("(long)") +  Long.valueOf(val).toString();
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
	}

	public void setFloat(int pos, float val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = Float.valueOf(val).toString();
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
	}
	
	public void setDouble(int pos, double val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = Double.valueOf(val).toString();
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
	}
	
	public void setBigDecimal(int pos, BigDecimal val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = val.toString();
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}		
	}
	
	public void setString(int pos, String val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			String escValue = val.replaceAll("'", Constant.QESC);
			_paramArray[pos-1] = new String("'") + escValue + new String("'");
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
	}
	
	public void setBytes(int pos, byte[] val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			throw new SQLException("Method not implemented");					
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}		
	}
	
	public void setDate(int pos, Date val) throws SQLException 
	{
		setDate(pos, val, Calendar.getInstance());
	}
	
	public void setTime(int pos, Time val) throws SQLException 
	{
		setTime(pos, val, Calendar.getInstance());
	}
	
	public void setTimestamp(int pos, Timestamp val) throws SQLException 
	{
		setTimestamp(pos, val, Calendar.getInstance());
	}
	
	public void setAsciiStream(int pos, InputStream stream, int len) throws SQLException 
	{
		setClob(pos, new CegoClob(stream, len));
	}
	
	public void setUnicodeStream(int pos, InputStream stream, int len) throws SQLException
	{
		setClob(pos, new CegoClob(stream, len));
	}
	
	public void setBinaryStream(int pos, InputStream stream, int len) throws SQLException 
	{
		setBlob(pos, new CegoBlob(stream, len));
	}
	
	public void clearParameters() throws SQLException 
	{
		for ( int i = 0; i < _numParam; i++)
		{
			_paramArray[i] = null;
		}
	}
	
	public void setObject(int pos, Object x, int targetSqlType, int scaleOrLength) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}
	
	public void setObject(int pos, Object x, int targetSqlType) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}
	
	public void setObject(int pos, Object x) throws SQLException 
	{
		
		if ( pos >= 1 && pos <= _numParam )
		{
			if ( x instanceof CegoDecimal )
				_paramArray[pos-1] = new String("(decimal)") +  x.toString();
			else if ( x instanceof CegoBigInt )
				_paramArray[pos-1] = new String("(bigint)") +  x.toString();
			else	
				throw new SQLException("Method not implemented");				
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}
		
	}
	
	public void addBatch() throws SQLException 
	{
		String paramizedQuery = new String();

		int i;
		for ( i = 0; i < _numParam; i++)
		{
			paramizedQuery += _frags.get(i) + _paramArray[i];
		}
		paramizedQuery += _frags.get(i);

		super.addBatch(paramizedQuery);	
	}
	
	public void setCharacterStream(int arg0, Reader arg1, int arg2) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}
	
	public void setRef(int arg0, Ref arg1) throws SQLException 
	{
		throw new SQLException("Method not implemented");		
	}
	
	public void setBlob(int pos, Blob b) throws SQLException 
	{
		if ( _blobVector.size() < pos)
		{
			_blobVector.setSize(pos);
		}		
		_blobVector.set(pos-1, b);	
	}
	
	public void setClob(int pos, Clob c) throws SQLException 
	{
		if ( _clobVector.size() < pos)
		{
			_clobVector.setSize(pos);
		}		
		_clobVector.set(pos-1, c);	
	}
	
	public void setArray(int arg0, Array arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}

	public ResultSetMetaData getMetaData() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setDate(int pos, Date val, Calendar cal) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			Time t = new Time(val.getTime());
			_paramArray[pos-1] = new String("scandate('%Y-%m-%d %H:%M:%S', '") 
				+ val.toString() + " " + t.toString()
				+ new String("')");
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}	
	}
	
	public void setTime(int pos, Time val, Calendar cal) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{		
			
			Date d = new Date(val.getTime());
			_paramArray[pos-1] = new String("scandate('%Y-%m-%d %H:%M:%S', '")
				+ d.toString() + " " + val.toString() 
				+ new String("')");
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}		
	}

	public void setTimestamp(int pos, Timestamp val, Calendar cal) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			
			StringTokenizer tok = new StringTokenizer(val.toString(), ".");
			
			String d = tok.nextToken();
			
			_paramArray[pos-1] = new String("scandate('%Y-%m-%d %H:%M:%S', '")
				+ d 
				+ new String("')");
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}			
	}

	public void setNull(int pos, int sqlType, String typeName) throws SQLException 
	{
 		setNull(pos, sqlType);
	}
	
	public void setURL(int pos, URL val) throws SQLException 
	{
		if ( pos >= 1 && pos <= _numParam )
		{
			_paramArray[pos-1] = new String("'") + val.toString() + new String("'");
		}
		else
		{
			throw new SQLException("Parameter index exceeded");
		}		
	}
	
	public ParameterMetaData getParameterMetaData() throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
			
	public void setAsciiStream(int pos, InputStream stream) throws SQLException
	{
		try
		{
			setClob(pos, new CegoClob(stream));
		}
		catch ( IOException e)
		{
			throw new SQLException(e.getMessage());
		}
	}

	public void setAsciiStream(int pos, InputStream stream, long len) throws SQLException
	{
		setClob(pos, new CegoClob(stream, (int)len));
	}
	
	public void setBinaryStream(int pos, InputStream stream) throws SQLException 
	{
		try
		{
			setBlob(pos, new CegoBlob(stream));
		}
		catch ( IOException e)
		{
			throw new SQLException(e.getMessage());
		}
	}
	
	public void setBinaryStream(int pos, InputStream stream, long len) throws SQLException
	{		
		setBlob(pos, new CegoBlob(stream, (int)len));
	}
	
	public void setBlob(int pos, InputStream stream) throws SQLException 
	{
		try
		{
			setBlob(pos, new CegoBlob(stream));
		}
		catch ( IOException e)
		{
			throw new SQLException(e.getMessage());
		}
	}

	public void setBlob(int pos, InputStream stream, long len) throws SQLException
	{
		setBlob(pos, new CegoBlob(stream, (int)len));
	}
	
	public void setCharacterStream(int arg0, Reader arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setClob(int pos, Reader reader) throws SQLException 
	{
		try
		{
			
			char[] charBuffer = new char[8 * 1024];
		    StringBuilder builder = new StringBuilder();
		    int numCharsRead;
		    while ((numCharsRead = reader.read(charBuffer, 0, charBuffer.length)) != -1) {
		        builder.append(charBuffer, 0, numCharsRead);
		    }
		    
		    InputStream targetStream = new ByteArrayInputStream( builder.toString().getBytes(StandardCharsets.UTF_8));		 
		    reader.close();
			
			setClob(pos, new CegoClob(targetStream));
		}
		catch ( IOException e)
		{
			throw new SQLException(e.getMessage());
		}
	}
	
	public void setClob(int arg0, Reader arg1, long arg2) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNCharacterStream(int arg0, Reader arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNClob(int arg0, NClob arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNClob(int arg0, Reader arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNClob(int arg0, Reader arg1, long arg2) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setNString(int arg0, String arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setRowId(int arg0, RowId arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	public void setSQLXML(int arg0, SQLXML arg1) throws SQLException 
	{
		throw new SQLFeatureNotSupportedException("Method not implemented");
	}
	
	private void handleLobs() throws SQLException
	{
		
		for ( int i=0; i < _blobVector.size(); i++)
		{
			Blob b = _blobVector.get(i);
			if ( b != null )
			{

				CegoNet nd = null;
				
				try 
				{

					nd = _netDriver.fork();
					_logger.debug("Sending blob for vector pos " + i + " of size " + b.length());
					_paramArray[i] = nd.putBlob(b);
					
				}
				catch ( Exception e )
				{
					e.printStackTrace();
					throw new SQLException("Cannot send blob : " + e.getMessage());
				}
				finally
				{
					if ( nd != null )
					{
						try
						{
							nd.closeSession();
						}
						catch ( Exception e)
						{
							throw new SQLException(e);
						}
					}
				}
			}
			else
			{
				_logger.debug("Empty blob vector for pos " + i);
			}
		}
		
		for ( int i=0; i < _clobVector.size(); i++)
		{
			Clob c = _clobVector.get(i);
			if ( c != null )
			{

				CegoNet nd = null;
				
				try 
				{

					nd = _netDriver.fork();
					_logger.debug("Sending clob for vector pos " + i + " of size " + c.length());
					_paramArray[i] = nd.putClob(c);
					
				}
				catch ( Exception e )
				{
					e.printStackTrace();
					throw new SQLException("Cannot send clob : " + e.getMessage());
				}
				finally
				{
					if ( nd != null )
					{
						try
						{
							nd.closeSession();
						}
						catch ( Exception e)
						{
							throw new SQLException(e);
						}
					}
				}
			}
			else
			{
				_logger.debug("Empty clob vector for pos " + i);
			}
		}
		
		// initialize lob vectors for next call
		
		if ( _blobVector.size() > 0)
			_blobVector = new Vector<Blob>();
		if ( _clobVector.size() > 0)
			_clobVector = new Vector<Clob>();


	}
}
