package de.lemkeit.cegojdbc;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.util.LinkedList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CegoStatement implements java.sql.Statement
{
		
	private static final Logger _logger = LoggerFactory.getLogger(CegoStatement.class);
	
	private CegoConnection _con;
	private CegoNet _netDriver;
	private int _maxRow = 0;
	private int _maxFieldSize = 0;
	private int _queryTimeout = Constant.QUERY_TIMEOUT;
	private int _fetchDirection = ResultSet.FETCH_FORWARD;
	private int _fetchSize = Constant.FETCHSIZE;
	private int _currency = ResultSet.CONCUR_READ_ONLY;
	
	@SuppressWarnings("unused")
	private boolean _escapeProcessing = false;
	@SuppressWarnings("unused")
	private String _cursorName;
	
	private List<SQLWarning> _warnList = new LinkedList<SQLWarning>();
	private List<String> _batchList = new LinkedList<String>();
	
	private ResultSet _resultSet;
	private int _updateCount;
	
	private boolean _storeLocal;
	
	private boolean _isClosed;
	private boolean _isPoolable;
	private boolean _forkedDriver;
	
	
	CegoNetMsg _dg = null;
	
	// with storeLocal = true, all tuples fetched into a local buffer in the execution method 
	// otherwise, the tuples are just fetched via next method in the ResultSet
	public CegoStatement(CegoNet netDriver, CegoConnection con, boolean storeLocal)
	{
		_con = con;
		_netDriver = netDriver;
		_storeLocal = storeLocal;
		_isClosed = false;
		_isPoolable = false;
		_forkedDriver = false;
	}
	
	public int getUpdateCount() throws SQLException 
	{	
		return _updateCount;
	}

	public ResultSet executeQuery(String query) throws SQLException 
	{

		_logger.debug("executeQuery <" + query + ">");

		// treat nested statements
		if ( _netDriver.getReqCount() > 0 )
		{
			_netDriver = _netDriver.fork();
			_forkedDriver=true;
		}
		
		try
		{
			_dg = _netDriver.reqQueryOp(query);
		

			ResultSet rs = null;

			List<CegoField> schema = _dg.getSchema();

			if ( _dg.getName().equals(Constant.DG_INFO))
			{
				rs = new CegoResultSet(schema);
			}
			else if ( _dg.getName().equals(Constant.DG_DATA))
			{
				if ( _storeLocal == true )
				{

					if ( _dg.getRowList() == null )
					{
						_logger.debug("Request more data" );

						_dg = _netDriver.reqData();
					}

					CegoNetMsg aggDoc = new CegoNetMsg(_netDriver.getProtocol(), _netDriver.getCharSet());

					_logger.debug("Received doctype " + _dg.getName() );

					while ( _dg.getName().equals(Constant.DG_DATA))
					{
						_logger.debug("Adding rowlist for " + _dg.getName() );

						aggDoc.addRows( _dg.getRowList() );

						_dg = _netDriver.reqData();

						_logger.debug("Received further doctype " + _dg.getName() );

					}

					if ( _dg.getName().equals(Constant.DG_OK))
					{
						rs = new CegoResultSet(_netDriver, aggDoc, schema);
					}
					else if ( _dg.getName().equals(Constant.DG_ERROR))
					{
						throw new SQLException(_dg.getAttribute(Constant.XML_MSG_ATTR));
					}
				}
				else
				{
					rs = new CegoResultSet(_netDriver, _dg.getSchema(), _dg.getRowList());
				}			
			}
			else if ( _dg.getName().equals(Constant.DG_OK))
			{
				rs = new CegoResultSet();
			}
					
			// we also provide result with getResultSet method 
			_resultSet = rs;
			return rs;
		
		}
		catch ( Exception e )
		{
			_logger.error(e.getMessage());
			throw new SQLException(e.getMessage());
		}
	}

	public int executeUpdate(String query) throws SQLException 
	{
		
		_logger.debug("executeUpdate <" + query + ">");

		// treat nested statements
		if ( _netDriver.getReqCount() > 0 )
		{
			_netDriver = _netDriver.fork();
			_forkedDriver=true;
		}
				
		try 
		{
			_dg = _netDriver.reqQueryOp(query);
		}
		catch ( Exception e )
		{
			throw new SQLException(e.getMessage());
		}
				
		String val = _dg.getAffected();
		if ( val == null )
			return 0;
		
		return Integer.valueOf(val).intValue();
	}

	public boolean execute(String query) throws SQLException 
	{	
		_logger.debug("execute <" + query + ">");

		CegoNetMsg dg = null;

		// treat nested statements
		if ( _netDriver.getReqCount() > 0 )
		{
			_netDriver = _netDriver.fork();
			_forkedDriver=true;
		}
					
		try
		{
			dg = _netDriver.reqQueryOp(query);

			if ( dg.getName().equals(Constant.DG_INFO))
			{
				_resultSet = new CegoResultSet(dg.getSchema());
				return true;
			}
			else if ( dg.getName().equals(Constant.DG_DATA))
			{

				
				if ( _storeLocal == true )
				{

					// receiving data doc
					CegoNetMsg dataDoc = _netDriver.reqData();
					_resultSet = new CegoResultSet(_netDriver, dataDoc, _dg.getSchema());

					// receiving ok doc
					dataDoc = _netDriver.reqData();
					return true;
				}
				else
				{
					_resultSet = new CegoResultSet(_netDriver, dg.getSchema(), dg.getRowList());
					return true;
				}	
			}
			else if ( dg.getName().equals(Constant.DG_OK))
			{

				String val = dg.getAffected();
				if ( val != null )
				{
					_updateCount = Integer.valueOf(val).intValue();
				}
				return false;
			}

			// if unknown, we return false and set updateCount to -1 
			_updateCount = -1;
			return false;

		}
		catch ( Exception e )
		{
			throw new SQLException(e);
		}
	}

	public List<CegoDataRow> getOutParamList() 
	{
		// System.out.println("Size=" + _doc.getRootElement().getChildren(Constant.XML_OUTPARAM_ELEMENT).size());
		return _dg.getOutParamList();
	}
	
	public void close() throws SQLException 
	{	
		// if not closed by user, we have to close it
		if ( _resultSet != null)
			_resultSet.close();

		if ( _forkedDriver )
		{
			_netDriver.closeSession();
		}
					
		_isClosed = true;
	}
	
	public int getMaxFieldSize() throws SQLException 
	{
		return _maxFieldSize;
	}
	
	public void setMaxFieldSize(int maxFieldSize) throws SQLException 
	{
		_maxFieldSize = maxFieldSize;
	}
	
	public int getMaxRows() throws SQLException 
	{	
		return _maxRow;
	}

	public void setMaxRows(int maxRow) throws SQLException 
	{	
		_maxRow = maxRow;	
	}
	
	public void setEscapeProcessing(boolean isEnabled) throws SQLException 
	{	
		_escapeProcessing = isEnabled;	
	}

	public int getQueryTimeout() throws SQLException 
	{	
		return _queryTimeout;
	}
	
	public void setQueryTimeout(int queryTimeout) throws SQLException 
	{	
		_queryTimeout = queryTimeout;
	}
	
	public void cancel() throws SQLException 
	{	
		// _netDriver.abort();
	}
	
	public SQLWarning getWarnings() throws SQLException 
	{	
		if ( _warnList.isEmpty() )
			return null;
		return _warnList.get(0);	
	}
	
	public void clearWarnings() throws SQLException 
	{	
		_warnList.clear();
	}
	
	public void setCursorName(String cursorName) throws SQLException 
	{	
		_cursorName = cursorName;	
	}
	
	public ResultSet getResultSet() throws SQLException 
	{	
		if ( _resultSet != null)
			return _resultSet;
		return new CegoResultSet();
	}
	
	public boolean getMoreResults() throws SQLException 
	{	
		// no multi results set supported for now
		return false;
	}
	
	public void setFetchDirection(int fetchDirection) throws SQLException {
		
		_fetchDirection = fetchDirection;
		
	}
	
	public int getFetchDirection() throws SQLException {

		return _fetchDirection;
	}
	
	public void setFetchSize(int fetchSize) throws SQLException {
	
		_fetchSize = fetchSize;
	}
	
	public int getFetchSize() throws SQLException {
		
		return _fetchSize;
	}
	
	public int getResultSetConcurrency() throws SQLException {

		return _currency;
	}
	
	public int getResultSetType() throws SQLException {
		
		return ResultSet.TYPE_FORWARD_ONLY;
	}
	
	public void addBatch(String cmd) throws SQLException {
		
		_batchList.add(cmd);
		
	}
	
	public void clearBatch() throws SQLException {

		_batchList.clear();
	}

	public int[] executeBatch() throws SQLException {
		
		int[] retArray = new int[_batchList.size()];
		
		for ( int i = 0 ; i < _batchList.size() ; i++ )
		{
			try {
			
				retArray[i] = executeUpdate(_batchList.get(i));
				
			}
			catch ( SQLException e)
			{
				 retArray[i] = EXECUTE_FAILED;
			}	
		}		
		return retArray;
	}
	
	public Connection getConnection() throws SQLException {
		
		return _con;
	}
	
	public boolean getMoreResults(int current) throws SQLException
	{
		// cego does not support multiple result sets
		return false;
	}
	
	public ResultSet getGeneratedKeys() throws SQLException 
	{
		// does not support generated keys
		return new CegoResultSet();
	}
	
	public int executeUpdate(String query, int autoGeneratedKeys) throws SQLException 
	{
		return executeUpdate(query);
	}
	
	public int executeUpdate(String query, int[] columnIndexes) throws SQLException 
	{
		return executeUpdate(query);
	}
	
	public int executeUpdate(String query, String[] columnNames) throws SQLException 
	{
		return executeUpdate(query);
	}
	
	public boolean execute(String query, int autoGeneratedKeys) throws SQLException 
	{
		return execute(query);
	}
	
	public boolean execute(String query, int[] columnIndexes) throws SQLException 
	{
		return execute(query);
	}

	public boolean execute(String query, String[] columnNames) throws SQLException 
	{
		return execute(query);
	}
	
	public int getResultSetHoldability() throws SQLException 
	{	
		return ResultSet.CLOSE_CURSORS_AT_COMMIT;
	}
	
	public boolean isClosed() throws SQLException 
	{
		return _isClosed;
	}
	
	public boolean isPoolable() throws SQLException 
	{
		return _isPoolable;
	}
	
	public void setPoolable(boolean isPoolable) throws SQLException 
	{
		if ( _isClosed )
			throw new SQLException("Statment is closed");
		_isPoolable = isPoolable;
	}
	
	public boolean isWrapperFor(Class<?> iface) throws SQLException 
	{	
		return iface.isInstance(this);
	}
	
	public <T> T unwrap(Class<T> iface) throws SQLException 
	{
		try {
			return iface.cast(this);
		} catch (ClassCastException cce) {
			throw new SQLException("Unable to wrap");
		}
	}

	public void closeOnCompletion() throws SQLException 
	{
	}

	public boolean isCloseOnCompletion() throws SQLException 
	{
		return false;
	}
}
