#ifndef _SCREEN_H_INCLUDED_ 
#define _SCREEN_H_INCLUDED_ 
/////////////////////////////////////////////////////////////////////////////// 
// 
// Screen.h 
// --------
// Next generation Curses based screen interface definition 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2022 Bjoern Lemke 
// 
// INTERFACE MODULE 
// 
// Class: Screen 
// 
// Description: Description: NG Cursor based screen utiliy class
//
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <ctype.h> 
#include <string.h> 

#ifndef LFCNOCURSES
#include <ncurses.h>
#endif

#include "Chain.h" 
#include "ListT.h" 
#include "Datetime.h" 
#include "SigHandler.h" 

#define SC_RED 1
#define SC_GREEN 2
#define SC_YELLOW 3
#define SC_BLUE 4

#define ATTRSEPTOKEN ":"
#define LISTSEPTOKEN ","
#define VALSEPTOKEN "="

#define VAL_MAX_LEN 100 
#define VAL_MAX_NUM 20 

#define MAX_ATTR_WIN 5


class Screen : public SigHandler { 

public: 

    class Panel {
    public:
	Panel(Panel* pParent) { _pParent = pParent; _pNext = this; };
	virtual ~Panel() {};

	void setStatus(const Chain& status);

	// Chain& getStatus() { return _status; };

	virtual void show(bool doRec) = 0;
	virtual void handleKey(unsigned c) = 0;
	
	virtual WINDOW* getWindow() = 0;

	virtual void enter() {};
	virtual void refresh() {};
	virtual bool doRefresh() { return false; };

	Panel* nextPanel() { return _pNext; };
	
    protected:

	Chain _status;
	Panel* _pParent;
	Panel* _pNext;
    };

    class Message : public Panel
    { 
    public: 
	Message(Panel* pParent);
	~Message();

	void setInfo(const Chain& title, const Chain& msg, unsigned height = 0);
    
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();
	
    private:

	unsigned _height;
	unsigned _width;
	unsigned _offset;
	unsigned _msgHeight;
	

	Chain _title;
	Chain _msg;
	
	WINDOW *_msgwin;
    };

    class Select : public Panel { 
    public: 
	Select(Panel* pParent);
	~Select();

        void setItems(const Chain& title, const ListT<Chain>& items);

        virtual Panel* onSelect(unsigned selected) { return this; };

	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
        void handleKey(unsigned c);
        WINDOW* getWindow();

	void setSelectedKey(const Chain& key);
	Chain getSelectedKey();
	Chain getSelectedValue();

    protected:
	
       Chain getItem(unsigned i) { return _items[i]; };
	
    private:

	Chain _title;
	unsigned _height;
	unsigned _width;
	unsigned _highlight;
	ListT<Chain> _items;
	WINDOW *_selectwin;
	Message* _pError;
    };
   

    class GridColor { 
    public: 
	GridColor() {};
	GridColor(const Chain& val) {
	    _val = val;
	    _code = 0;
	}
	GridColor(const Chain& val, unsigned code)  { 
	    _val = val; 
	    _code = code; 
	}; 
	~GridColor() {}; 
	const Chain& getValue() { return _val; }; 
	unsigned getCode() { return _code; }; 
	GridColor& operator = (const GridColor& tc) { 
	    _val = tc._val; 
	    _code = tc._code; 
	    return(*this); 
	};
	
	bool operator == ( const GridColor& tc) const {
	    return _val == tc._val;
	};
	
    private: 
	Chain _val; 
	unsigned _code; 
    };

    class Grid : public Panel { 
    public: 

	Grid(Panel *pParent);
	~Grid();
	
	void setSchema(const Chain& title, const ListT<Chain>& gridSchema, const ListT<GridColor>& colorMap, unsigned height);
	
	// void setData( ListT< ListT<Chain> > &gridData);
	virtual Panel* onSelect(unsigned rowNum) { return this; };
	virtual Panel* onKey(unsigned c, unsigned rowNum) { return this; };
	
	virtual ListT< ListT<Chain> > getData() {
	    ListT< ListT<Chain> > gridData;
	    return gridData;
	}

	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();

    private:

	void refreshRow(unsigned tabRow, unsigned rowno);

	unsigned _curStat;
	Chain _title;
	bool _isSelectable;
	unsigned _initHeight;
	unsigned _height;
	unsigned _numCol;
	unsigned _rowSelected;
	unsigned _vStart;

	ListT<Chain> _schema;
	ListT< ListT<Chain> > _table;

	ListT<GridColor> _colorMap;
	
	WINDOW *_gridwin;
	Message* _pError;

    };
   
    
    class Confirm : public Panel { 
    public: 
	Confirm(Panel* pParemt);
	~Confirm();
	virtual Panel* onConfirm() { return _pParent; };
	virtual Panel* onCancel() { return _pParent; };

	void setInfo(const Chain& title, const Chain& details);

	Panel* setError(const Chain& title, const Chain& msg);
	
	
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();
	
    private:

	Chain _title;
	Chain _details;
    
	unsigned _width;
	unsigned _height;
    
	unsigned _confVal;
	
	unsigned _curRow; 
	unsigned _curCol;

	Message* _pError;
	WINDOW* _confwin;
    };

    class Form : public Panel { 
    public: 
	Form(Panel *pParent);
	~Form();

	void setAttrList(const Chain& title, const ListT<Chain>& attrList, unsigned maxVisible);
	void setValueList(const ListT<Chain>& valList);

	void clear();
	void reset();
	
	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();
	
	virtual Panel* onConfirm(const ListT<Chain>& valList) { return _pParent; };
	virtual Panel* onCancel() { return _pParent; };
	
    private:
	
	Chain _title;
	ListT<Chain> _attrList;
	
	char _inputArray[VAL_MAX_NUM][VAL_MAX_LEN];
	
	unsigned _maxLenArray[VAL_MAX_NUM];
	unsigned _vposArray[VAL_MAX_NUM];
	
	unsigned _colOffset;
	unsigned _height;
	unsigned _width;
	unsigned _maxVisible;
	
	unsigned _curRow;
	unsigned _curCol;

	Message* _pError;
	WINDOW* _formwin;
	
	void getAttrTypeValue(const Chain& s, Chain& attr, Chain& type, unsigned& maxLen, Chain& value);
	
	void getListKey(const Chain& s, const Chain& value, Chain& key);
	void getListValue(const Chain& s, const Chain& key, Chain& value);
	
	void prevCursorPos(unsigned attrListSize, unsigned colOffset, unsigned curRow, unsigned curCol,
			   unsigned& prevCurRow, unsigned& prevCurCol);
	
	void nextCursorPos(unsigned attrListSize, unsigned colOffset, unsigned curRow, unsigned curCol,
			   unsigned& nextCurRow, unsigned& nextCurCol, unsigned navMode); 


	ListT<Chain> getValues();
	
	class FormSelect : public Select {
	    
	public:
	    
	    FormSelect(Panel* pParent) : Select(pParent) {		
	    }
	    
	    void setup(const Chain& title, const ListT<Chain>& items, char* pInput) {		
		_pInput = pInput;
		setItems(title, items);

		// setStatus(Chain("Setup select with ") + Chain(pInput));
		setSelectedKey(Chain(pInput));
	    }
		   
	    Panel* onSelect(unsigned selected);
	    
	private:
	    char* _pInput;
	};

	FormSelect* _pSelect;

    };
    
    
    class Attribute : public Panel { 
    public:

	enum Layout { HORIZONTAL, VERTICAL, OVERLAY };
	
	Attribute(Panel* pParent, Layout layout);
	~Attribute();

	void setAttrList(const ListT<Chain>& titleList, const ListT< ListT<Chain> >& attrList);
	    
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();
	
    private:

	WINDOW* createWindow(unsigned xpos, unsigned ypos, const Chain& title, ListT<Chain> *pAttrList);
	Layout _layout;
	ListT<Chain> _titleList;
	unsigned _visibleIdx;
	ListT< ListT<Chain> > _attrList;
	WINDOW* _attrwin[MAX_ATTR_WIN];
    };
    
    
    class Menu : public Panel
    { 
    public: 
	Menu();
	~Menu();
	
	void show(bool doRec);
	void handleKey(unsigned c);
	WINDOW* getWindow();
	void regItem(const Chain& name, Panel *pPanel);
	
    private:

	
	class MenuItem { 
	public: 
	    MenuItem() { }; 
	    MenuItem(const Chain& name, Panel *pPanel) 
		{ 
		    _name = name; 
		    _pPanel = pPanel; 
		}; 
	    ~MenuItem() {}; 
	    const Chain& getName() { return _name; }; 
	    Panel* getPanel() { return _pPanel; }; 
	    MenuItem& operator = (const MenuItem& mi) 
		{ 
		    _name = mi._name; 
		    _pPanel = mi._pPanel; 
		    return(*this); 
		}; 
	private: 
	    Chain _name; 
	    Panel *_pPanel; 
	}; 

	ListT<MenuItem> _menu;
	unsigned _menuSelected;
    };

  
    Screen(); 
    ~Screen(); 

    void setTimeout(unsigned timeout);
    void regShortCut(char c, unsigned code);
    void setRoot(Panel* pPanel);
    
    void showScreen();


    
private:
    

    class ShortCut { 
    public: 
        ShortCut() {}; 
        ShortCut(char c, unsigned code) 
        { 
           _c = c; 
           _code = code; 
        }; 
        ~ShortCut() {}; 
        char getShortCut() { return _c; }; 
        unsigned getCode() { return _code; }; 
        ShortCut& operator = (const ShortCut& sc) 
        { 
           _c = sc._c; 
           _code = sc._code; 
           return(*this); 
        }; 
    private: 
        char _c; 
        unsigned _code; 
    };

    ListT<ShortCut> _scList; 

    void sigCatch(int sig); 

    Panel *_pPanel;

    unsigned _timeout;
    unsigned _statusOffset;

}; 


#endif 
