///////////////////////////////////////////////////////////////////////////////
//
// ThreadLock.cc 
// -------------
// ThreadLock class implementation
//                                                
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//                                                         
// IMPLEMENTATION MODULE
//
// Class: ThreadLock
// 
// Description: All operations on POSIX thread locks
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <iostream>

#include "ThreadLock.h"
#include "Exception.h"
#include "NanoTimer.h"

#define DELAYFAC 1000000

ThreadLock::ThreadLock(const Chain& id)
{
    _id = id;
}

ThreadLock::~ThreadLock()
{
}

const Chain& ThreadLock::getId() const
{
    return _id;
}

void ThreadLock::setId(const Chain& id)
{
    _id = id;
}

void ThreadLock::init(unsigned lockDelay, bool doStat)
{
    _doStat = doStat;
    _readLockCount=0;
    _writeLockCount=0;
    _numLockTry=0;

    _readDelay = 0;
    _writeDelay = 0;

    _lockDelay = lockDelay;

    int ret;
    if ( ( ret = pthread_rwlock_init(&_rwlock, NULL)) != 0 )
    {
	Chain msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	throw Exception(EXLOC, msg);
    }
    if ( doStat )
    {
	if ( ( ret = pthread_rwlock_init(&_statlock, NULL)) != 0 )
	{
	    Chain msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	    throw Exception(EXLOC, msg);
	}
    }
}

void ThreadLock::readLock()
{   

    NanoTimer *pTimer = 0;

    if ( _doStat )
    {
	pTimer = new NanoTimer();
	pTimer->start();

	pthread_rwlock_wrlock(&_statlock);
	_numLockTry++;
	_readLockCount++;
	pthread_rwlock_unlock(&_statlock);	
    }

    int ret = pthread_rwlock_rdlock(&_rwlock);

    if ( _doStat )
    {
	pTimer->stop();
	pthread_rwlock_wrlock(&_statlock);
	_numLockTry--;
	_readDelay += pTimer->getSum();
	pthread_rwlock_unlock(&_statlock);
	delete pTimer;
    }
    
    if ( ret != 0 )
    {
	Chain msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	throw Exception(EXLOC, msg);
    }
}

void ThreadLock::writeLock()
{   
    NanoTimer *pTimer = 0;

    if ( _doStat )
    {
	pTimer = new NanoTimer();
	pTimer->start();

	pthread_rwlock_wrlock(&_statlock);
	_numLockTry++;
	_writeLockCount++;
	pthread_rwlock_unlock(&_statlock);	
    }

    int ret = pthread_rwlock_wrlock(&_rwlock);

    if ( _doStat )
    {
	pTimer->stop();
	pthread_rwlock_wrlock(&_statlock);
	_numLockTry--;
	_writeDelay += pTimer->getSum();
	pthread_rwlock_unlock(&_statlock);
	delete pTimer;	
    }

    if  ( ret != 0 )
    {
	Chain msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	throw Exception(EXLOC, msg);
    }
}

void ThreadLock::readLock(unsigned timeout)
{   
    NanoTimer *pTimer = 0;
    
    if ( _doStat )
    {
	pTimer = new NanoTimer();
	pTimer->start();

	pthread_rwlock_wrlock(&_statlock);
	_numLockTry++;
	_readLockCount++;
	pthread_rwlock_unlock(&_statlock);
    }

    struct timespec delay;

    delay.tv_sec = 0 ; 
    delay.tv_nsec = _lockDelay * DELAYFAC;
    
    bool isLocked = false;
    bool isError = false;
    unsigned locktime=0;
    Chain msg;

    while ( locktime < timeout && isLocked == false && isError == false)
    {
	int ret;
	ret = pthread_rwlock_tryrdlock(&_rwlock);
	if ( ret == 0 )
	{
	    isLocked = true;
	}
	else if ( ret == EBUSY )
	{
	   
	    if ( nanosleep(&delay, NULL) == -1 )
	    {
		msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
		isError=true;
	    }

	    locktime += _lockDelay;
	}
	else
	{
	    isError = true;
	    msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	}
    }

    if ( _doStat )
    {
	pTimer->stop();
	
	pthread_rwlock_wrlock(&_statlock);
	_numLockTry--;
	_readDelay += pTimer->getSum();
	pthread_rwlock_unlock(&_statlock);
	
	delete pTimer;
    }

    if ( isLocked )
	return;

    if ( isError == false )
	msg = Chain("Lock timeout exceeded for <") + _id + Chain("> after ") + Chain(locktime) + Chain(" msec");

    throw Exception(EXLOC, msg);
}

void ThreadLock::writeLock(unsigned timeout)
{  
    NanoTimer *pTimer = 0;
    
    if ( _doStat )
    {
	pTimer = new NanoTimer();
	pTimer->start();
	pthread_rwlock_wrlock(&_statlock);
	_numLockTry++;
	_writeLockCount++;
	pthread_rwlock_unlock(&_statlock);
    }

    struct timespec delay;

    delay.tv_sec = 0; 
    delay.tv_nsec = _lockDelay * DELAYFAC; 

    bool isLocked = false;
    bool isError = false;
    unsigned locktime=0;
    Chain msg;

    while ( locktime < timeout && isLocked == false && isError == false )
    {
	int ret;
	ret = pthread_rwlock_trywrlock(&_rwlock);

	if ( ret == 0 )
	{
	    isLocked = true;
	}
	else if ( ret == EBUSY )
	{
	    if ( nanosleep(&delay, NULL) == -1 )
	    {
		isError = true;
		msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	    }

	    locktime += _lockDelay;
	}
	else
	{
	    isError = true;
	    msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	}
    }

    if ( _doStat )
    {
	pTimer->stop();
	pthread_rwlock_wrlock(&_statlock);
	_numLockTry--;
	_writeDelay += pTimer->getSum();
	pthread_rwlock_unlock(&_statlock);
	delete pTimer;
    }

    if ( isLocked )
	return;

    if ( isError == false )
	msg = Chain("Lock timeout exceeded for <") + _id + Chain("> after ") + Chain(locktime) + Chain(" msec");

    throw Exception(EXLOC, msg);
}

void ThreadLock::unlock()
{   
    int ret;
    if ( ( ret = pthread_rwlock_unlock(&_rwlock)) != 0 )
    {
	Chain msg = Chain("ThreadLock system error : ") + Chain(strerror(ret));
	throw Exception(EXLOC, msg);
    }
}

unsigned long long ThreadLock::numReadLock()
{
    return _readLockCount;
}

unsigned long long ThreadLock::numWriteLock()
{
    return _writeLockCount;
}

unsigned ThreadLock::numLockTry()
{
    return _numLockTry;
}

unsigned long long ThreadLock::sumReadDelay()
{
    return _readDelay;
}

unsigned long long ThreadLock::sumWriteDelay()
{
    return _writeDelay;
}
