///////////////////////////////////////////////////////////////////////////////
//                                                         
// Bigdecimal.cc
// -------------
// BigDecimal implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: BigDecimal
// 
// Description: BigDecimal implementation
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include "Exception.h"
#include "BigInteger.h"
#include "BigDecimal.h"
#include "Tokenizer.h"

BigDecimal::BigDecimal()
{
    _val = Chain("00");
    _scale=1;
    _isPositive = true;
}

BigDecimal::BigDecimal(const Chain& val)
{
    
    if ( val.subChain(1,1) == Chain("+"))
    {
	_isPositive=true;
	_val = val.subChain(2, val.length());
    }
    else if ( val.subChain(1,1) == Chain("-"))
    {
	_isPositive=false;
	_val = val.subChain(2, val.length());
    }
    else
    {
	_isPositive=true;
	_val = val;
    }
    
    if ( ! _val.isDec() )
    {
	Chain msg = Chain("Invalid decimal format for <") + val + Chain(">");
	throw Exception(EXLOC, msg);
    }
	    
    if ( _val.subChain(1,1) == Chain(".") )
	_val = Chain("0") + _val;
    
    Tokenizer tok(_val, Chain("."));
    Chain left, right;
    tok.nextToken(left);
    tok.nextToken(right);    

    left = left.truncLeft("0");
    if ( left.length() < 2 )
    {
	left = Chain("0");	
    }
    
    if ( right.length() < 2 )
    {
	right = Chain("0");
    }
    
    _scale = right.length() - 1; //  > 0 ? right.length() - 1 : 1;
	       
    // cout << "Left = " << left << endl;
    // cout << "Right = " << right << endl;
    // cout << "Scale = " << _scale << endl;
    
    _val = left + right;

    // cout << "Val=" << _val << endl;
}

BigDecimal::BigDecimal(const Chain& val, unsigned scale)
{

    /* this is a private constructor, so no need for any format validation */
    
    if ( val.subChain(1,1) == Chain("+"))
    {
	_isPositive=true;
	_val = val.subChain(2, val.length());
    }
    else if ( val.subChain(1,1) == Chain("-"))
    {
	_isPositive=false;
	_val = val.subChain(2, val.length());
    }
    else
    {
	_isPositive=true;
	_val = val;
    }
    
    _scale = scale;
    
    if ( (unsigned)_val.length() - 1  <= _scale )
    {
	unsigned j = (unsigned)_val.length() - 1;
		
	for ( unsigned i = j; i <= _scale ; i++)
	{
	    _val = "0" + _val;
	}
    }
    // cout << "Val=" << _val << endl;
}

BigDecimal::~BigDecimal()
{
}

BigDecimal BigDecimal::abs() const
{
    return BigDecimal(_val, _scale);
}

bool BigDecimal::isPositive() const
{
    return _isPositive;
}

void BigDecimal::negate()
{
    _isPositive = false;
}

BigDecimal BigDecimal::add(const BigDecimal& d) const
{
    Chain v1 = _val;
    Chain v2 = d._val;
    unsigned maxScale;

    if ( _scale > d._scale )
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }
    else if ( _scale < d._scale )
    {
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }    
    else
    {
	maxScale = _scale;
    }

    BigInteger i1(v1);
    BigInteger i2(v2);

    if ( _isPositive == false )
	i1.negate();
    if ( d._isPositive == false )
	i2.negate();


    BigInteger i3 = i1 + i2;

    return BigDecimal(i3.toChain(), maxScale); 
    
}

BigDecimal BigDecimal::sub(const BigDecimal& d) const
{

    Chain v1 = _val;
    Chain v2 = d._val;
    unsigned maxScale;

    if ( _scale > d._scale )
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }
    else if ( _scale < d._scale )
    {
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }    
    else
    {
	maxScale = _scale;
    }
    BigInteger i1(v1);
    if ( _isPositive == false )
	i1.negate();
    BigInteger i2(v2);
    if ( d._isPositive == false )
	i2.negate();

    BigInteger i3 =  i1 - i2;

    return BigDecimal(i3.toChain(), maxScale); 

}

BigDecimal BigDecimal::mul(const BigDecimal& d) const
{

    Chain v1 = _val;
    Chain v2 = d._val;

    BigInteger i1(v1);
    if ( _isPositive == false )
	i1.negate();
    BigInteger i2(v2);
    if ( d._isPositive == false )
	i2.negate();
    
    BigInteger i3 =  i1 * i2;
    
    return BigDecimal(i3.toChain(), _scale + d._scale); 
    
}

BigDecimal BigDecimal::div(const BigDecimal& d) const
{
    Chain v1 = _val;
    Chain v2 = d._val;

    unsigned i=0;
    while ( i < (unsigned)v1.length() )
    {
        if ( v1[i] != '0' )
           break;
        i++;
    }

    if ( i == (unsigned)v1.length() - 1 )
    {
        return BigDecimal();
    }	 
    unsigned maxScale;
    unsigned corScale;
    
    if ( d._scale > _scale )
    {	
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }
    else
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }

    // for div calculation, we use absolute values
    
    BigInteger i1(v1);    
    BigInteger i2(v2);
    
    corScale=0;
    while ( i1 < i2 )
    {
	i1 = i1.mul(BigInteger(10));
	corScale++;
    }

    while ( corScale < maxScale )
    {
	i1 = i1.mul(BigInteger(10));
	// cout << i1 << endl;
	corScale++;
    }

    // cout << "maxscale = " << maxScale << endl;
    // cout << "i1=" << i1 << " i2=" << i2 << endl;
    BigInteger i3 =  i1 / i2;
    // cout << "i3=" << i3 << endl;

    if ( ( _isPositive == false && d._isPositive == true )
	 || ( _isPositive == true && d._isPositive == false ) )
	i3.negate();
    
    Chain v = i3.toChain();
    if ( (unsigned)(i3.toChain().length())-1 < maxScale )
    {
	unsigned c = 1 + maxScale - ( (unsigned)(i3.toChain().length()) -1 );
	while ( c )
	{
	    v = Chain("0") + v;
	    c--;
	}
    }
    
    // cout << "V = " << v << endl;
    // cout << "MaxScale = " << maxScale << endl;
    return BigDecimal(v, maxScale); 
}

BigDecimal BigDecimal::scaleTo(unsigned scale, RoundMode mode) const
{

    // cout << "Scale=" << _scale << endl;
    // cout << "Val=" << _val << endl;
    
    BigDecimal scaleTarget;
    
    if ( scale == _scale )
	return *this;
    else if ( _val.truncLeft("0") == Chain() )
    {
	scaleTarget = BigDecimal(Chain("0"), scale);
    }
    else if ( scale > _scale )
    {
	Chain val = _val;
        unsigned c = scale - _scale;
	while ( c )
	{
	    val = val + Chain("0");
	    c--;
	}
	scaleTarget = BigDecimal(val, scale);
    }
    else
    {
	unsigned deltaScale = _scale - scale;
	
	Chain roundVal = _val.subChain(1, _val.length() - deltaScale -1);	
	Chain roundCheck =  _val.subChain(_val.length() - deltaScale , _val.length() - deltaScale);

	// cout << "RoundCheck=" << roundCheck << endl;

	bool roundUp=true;

	if ( mode == DOWN
	     || ( roundCheck.asUnsigned() <= 5 && mode == HALFDOWN ) )
	{
	    roundUp=false;
	}
	
	if ( roundUp ) // roundCheck.asUnsigned() >= 5 )
	{
	    Chain delta = Chain("0");
	    for ( unsigned i=0; i<scale-1; i++ )
		delta += Chain("0");
	    delta += Chain("1");	    
	    scaleTarget = BigDecimal(roundVal, scale) + BigDecimal(delta, scale);
	}
	else
	{
	    scaleTarget = BigDecimal(roundVal, scale);
	}
    }

    if ( _isPositive == false )
	scaleTarget.negate();

    return scaleTarget;
}

unsigned BigDecimal::getScale() const
{
    return _scale;
}

unsigned long long BigDecimal::length() const
{
    return _val.length();
}

Chain BigDecimal::toChain() const
{
    // cout << "Val=" << _val << endl;
    // cout << "Scale=" << _scale << endl;
    
    Chain s;

    if ( _isPositive == false )
	s = Chain("-");
    
    s +=  _val.subChain(1, _val.length() - _scale - 1) 
	+ Chain(".")
	+ _val.subChain(_val.length() - _scale, _val.length()); 	

    return s;
}

BigDecimal& BigDecimal::operator = ( const BigDecimal& d)
{
    _val = d._val;
    _isPositive = d._isPositive;
    _scale = d._scale;
    return (*this);
}

BigDecimal& BigDecimal::operator += ( const BigDecimal& d)
{
    this->add(d);
    return (*this);
}

BigDecimal& BigDecimal::operator -= ( const BigDecimal& d)
{
    this->add(d);
    return (*this);
}

bool BigDecimal::operator == ( const BigDecimal& d) const
{
    if ( _val == d._val && _scale == d._scale && _isPositive == d._isPositive )
	return true;
    
    if ( _val.truncLeft("0") ==  d._val.truncLeft("0") )
	return true;
    
    if (  d._isPositive != _isPositive )
        return false;

    return compAbs(d) == 0;
}

bool BigDecimal::operator != ( const BigDecimal& d) const
{
    return ! ( d == *this );
}

bool BigDecimal::operator < ( const BigDecimal& d) const
{
    if (  d._isPositive == false && _isPositive == true )
        return false;

    if (  d._isPositive == true && _isPositive == false )
        return true;

    if (  d._isPositive == true && _isPositive  == true)
	return compAbs(d) == -1;

    return compAbs(d) == 1;
}

bool BigDecimal::operator > ( const BigDecimal& d) const
{
    if (  d._isPositive == false && _isPositive == true )
        return true;

    if (  d._isPositive == true && _isPositive == false )
        return false;
    
    if (  d._isPositive == true && _isPositive  == true)
	return compAbs(d) == 1;

    return compAbs(d) == -1;
}

bool BigDecimal::operator <= ( const BigDecimal& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive )
	return true;
    else
	return *this < d;	
}

bool BigDecimal::operator >= ( const BigDecimal& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive)
	return true;
    else
	return *this > d;
}

BigDecimal operator + ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.add(d2);
}

BigDecimal operator - ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.sub(d2);
}

BigDecimal operator * ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.mul(d2);
}

BigDecimal operator / ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.div(d2);
}

ostream& operator << (ostream& s, const  BigDecimal& d)
{
    cout << d.toChain();
    return s;
}

Chain BigDecimal::mulDec(const Chain& val, unsigned dim) const
{
    Chain res = val;
    while ( dim > 0 )
    {
	res += Chain("0");
	dim--;
    }
    return res;
}

int BigDecimal::compAbs(const BigDecimal& d) const
{
    if ( ( _val.length() - _scale  ) < ( d._val.length() - d._scale ) )
        return -1;
    else if ( ( _val.length() - _scale  ) > ( d._val.length() - d._scale ) )
        return 1;
    else
    {
        if ( _scale < d._scale )
        {
	    Chain dval = _val;
	    unsigned i=0;
	    while ( i < d._scale - _scale )
	    {
		dval+="0";
		i++;
	    }
            BigInteger v1( d._val);
            BigInteger v2( dval );
		    
            if ( v2 > v1 )
		return 1;
            else if ( v2 < v1 )
               return -1;
        }
        else
        {
	    Chain dval = d._val;
	    unsigned i=0;
	    while ( i < _scale - d._scale )
	    {
		dval+="0";
		i++;
	    }
            BigInteger v1( dval);
            BigInteger v2( _val);
	    
            if ( v2 > v1 )
		return 1;
            else if ( v2 < v1 )
		return -1;
        }
    }
    return 0;
}
