///////////////////////////////////////////////////////////////////////////////
//                                                         
// Logger.cc
// ---------
// Managing log messages
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: Logger
// 
// Description: Managing module notices and debug messages
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////


// LFC INCLUDES
#include "Datetime.h"
#include "Logger.h"

#ifndef DISABLE_SYSLOG
#include <syslog.h>
#endif

// DEFINES

Logger::Logger()
{
    _pF = 0;
#ifndef DISABLE_SYSLOG    
    _writeSysLog = false;
#endif
}

Logger::Logger(const Chain& logFile, const Chain& progname)
{
    if ( logFile != Chain("") )
    {
	_pF = new File(logFile);
	if (_pF)
	{
	    _pF->open(File::APPEND);
	}
    }
    
#ifndef DISABLE_SYSLOG
    if ( progname != Chain("") )
    {
	_writeSysLog = true;
	openlog ((char*)progname, LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
	setlogmask (LOG_UPTO (LOG_DEBUG));
    }
    else
    {
	_writeSysLog = false;
    }
#endif
    
    _logLevel = Logger::NOTICE;
    _modId = 0;
}

Logger::~Logger()
{
    if (_pF)
    {
	delete _pF;
    }
}

void Logger::logModule(unsigned long modId, const Chain& module, Logger::LogLevel level)
{
    if ( modId > 0 )
    {	
	ModEntry *pME = _modSet.Find(ModEntry(modId));
	if ( pME )
	{
	    pME->setModule(module);
	    pME->setLevel(level);
	}
	else
	{
	    _modSet.Insert(ModEntry(modId, module, level));
	}
    }
}
	
void Logger::log(unsigned long modId, Logger::LogLevel level, const Chain& msg)
{
    ModEntry *pME = _modSet.Find(ModEntry(modId));

    if ( pME  )
    {
	// cout << "Logging <" << msg << " on level " << level << " MsgLevel = " << pME->getLogLevel() << endl;
	if ( level <= pME->getLogLevel()   )
	{
	    Chain cat;
	    
	    switch (level)
	    {
	    case NOTICE:
		cat = Chain("NOTICE");
		break;
	    case LOGERR:
		cat = Chain("ERROR");
		break;
	    case LOGALERT:
		cat = Chain("ALERT");
		break;
	    case DEBUG:
		cat = Chain("DEBUG");
		break;
	    case NONE:
		cat = Chain("NONE");
		break;
	    }
	    
	    Datetime dt;
	    
#ifndef DISABLE_SYSLOG
	    
	    int priority = 0;
	    
	    switch (level)
	    {
	    case NOTICE:
		priority = LOG_NOTICE;
		break;
	    case LOGERR:
		priority = LOG_ERR;
		break;
	    case LOGALERT:
		priority = LOG_ALERT;
		break;
	    case DEBUG:
		priority = LOG_DEBUG;
		break;
	    default:
		// case NONE:
		priority = -1;
		break;
	    }
	    if ( priority >= 0 && _writeSysLog )
	    {
		syslog (priority, "%s", (char*)msg);
	    }
#endif
	    
	    if ( _pF )
	    {
		_pF->writeChain(dt.asChain() + Chain(" [") + pME->getModule() + Chain("] ") + cat + Chain(" : ") + msg + Chain("\n"));
	    }
	}
    }
}

Logger& Logger::operator << ( unsigned long modId)
{
    ModEntry *pME = _modSet.Find(ModEntry(modId));
    
    if ( pME )
    {
	_modId = modId;
    }
    else
    {
	_modId = 0;
    }
    return (*this);
}

Logger& Logger::operator << ( const Logger::LogLevel& level)
{
    ModEntry *pME = _modSet.Find(ModEntry(_modId));

    if ( pME )
    {
	_msgLevel = level;
	_logLevel = pME->getLogLevel();
	if ( level <= _logLevel   )
	{
	    Chain cat;
	    Datetime dt;
	    switch (level)
	    {
	    case NOTICE:
		cat = Chain("NOTICE");
		break;
	    case LOGERR:
		cat = Chain("ERROR");
		break;
	    case LOGALERT:
		cat = Chain("ALERT");
		break;
	    case DEBUG:
		cat = Chain("DEBUG");
		break;
	    case NONE:
		cat = Chain("NONE");
		break;
	    }

	    if ( _pF )
	    {
		_pF->writeChain(dt.asChain() + Chain(":") + pME->getModule() + Chain(":") + cat + Chain(":"));
	    }
	}
    }
    return (*this);
}
    
Logger& Logger::operator << ( const Chain& str)
{
    if ( _msgLevel <= _logLevel && _modId > 0 )
    {
#ifndef DISABLE_SYSLOG
	
	int priority;
	
	switch (_msgLevel)
	{
	case NOTICE:
	    priority = LOG_NOTICE;
	    break;
	case LOGERR:
	    priority = LOG_ERR;
	    break;
	case LOGALERT:
	    priority = LOG_ALERT;
	    break;
	case DEBUG:
	    priority = LOG_DEBUG;
	    break;
	default:
	    // case NONE:
	    priority = -1;
	    break;
	}
	if ( priority >= 0 && _writeSysLog )
	{
	    syslog (priority, "%s", (char*)str);
	}
#endif
	
	if ( _pF )
	{
	    _pF->writeChain(str);
	}
    }
    return (*this);   
}

Logger& Logger::operator << ( const char* str)
{
    Chain s(str);
    *this << s;
    return (*this);
}
