///////////////////////////////////////////////////////////////////////////////
//                                                         
// Net.cc
// ------
// Base network driver
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke 
//
// IMPLEMENTATION MODULE
//
// Class: Net
// 
// Description: All operations on network resources
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

#include <string.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#define TIMEVAL struct timeval
#include <errno.h>

#include "Exception.h"
#include "Net.h"

Chain getSourceInfo(struct sockaddr *s);

Net::Net(unsigned initMsgBufLen, unsigned sizeBufLen, unsigned maxSendLen)
{
    _initMsgBufLen = initMsgBufLen;
    _sizeBufLen = sizeBufLen;
    _maxSendLen = maxSendLen;   
    _csock=0;
}

Net::~Net()
{
    if ( _csock )
    {
	::close(_csock);
    }
}

NetHandler* Net::connect(const Chain& hostname, const Chain& service)
{
    struct addrinfo hints, *res0, *res;
    int err;
    int sock;
    
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;

    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res0) ) != 0 ) 
    {
	Chain msg = Chain("Cannot adr info for ") + hostname;
	throw Exception(EXLOC, msg);
    }

    for (res=res0; res!=NULL; res=res->ai_next) 
    {
	sock = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (sock < 0) 
	{
	    continue;
	}
	
	if ( ::connect(sock, res->ai_addr, res->ai_addrlen) != 0) 
	{
	    ::close(sock);
	    continue;
	}
	
	break;
    }

    freeaddrinfo(res0);

    if (res == NULL) 
    {
	Chain msg = Chain("Cannot connect to ") + hostname;
	throw Exception(EXLOC, msg);
    }
    

    NetHandler* pNH = new NetHandler(sock, _initMsgBufLen, _sizeBufLen, _maxSendLen);
    return pNH;
}

NetHandler* Net::connect(const Chain& hostname, const Chain& service, unsigned timeout)
{
    struct addrinfo hints, *res0, *res;
    int err;
    int sock;
    
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;

    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res0) ) != 0 ) 
    {
	Chain msg = Chain("Cannot get adr info for ") + hostname;
	throw Exception(EXLOC, msg);
    }

    for (res=res0; res!=NULL; res=res->ai_next) 
    {
	sock = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (sock < 0) 
	{
	    continue;
	}

	int opt;
	
	if ((opt = fcntl(sock, F_GETFL, NULL)) < 0)
	{
	    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);	
	}
	
	if (fcntl(sock, F_SETFL, opt | O_NONBLOCK) < 0)
	{
	    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);	
	}
	
	if ( ::connect(sock, res->ai_addr, res->ai_addrlen) != 0) 
	{

	    if (errno == EINPROGRESS) 
	    {
		fd_set wait_set;
		
		FD_ZERO(&wait_set);
		FD_SET(sock, &wait_set);
		
		// struct timeval tv;
		TIMEVAL tv;
		tv.tv_sec = timeout;
		tv.tv_usec = 0;
		
		if ( ::select(sock + 1, NULL, &wait_set, NULL, &tv) < 0 )
		{
		    Chain msg = Chain("select system error : ") + Chain(strerror(errno)); 
		    throw Exception(EXLOC, msg);
		}
		
		if (fcntl(sock, F_SETFL, opt) < 0)
		{
		    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
		    throw Exception(EXLOC, msg);	
		}
	    }
	    else
	    {
		::close(sock);
		continue;
	    }
	}
	   
	break;
    }

    freeaddrinfo(res0);

    if (res == NULL) 
    {
	Chain msg = Chain("Cannot connect to ") + hostname;
	throw Exception(EXLOC, msg);
    }
    
    NetHandler* pNH = new NetHandler(sock, _initMsgBufLen, _sizeBufLen, _maxSendLen);
    return pNH;
}


///////////////////////////////////////////////////////////
// Serves connections on given hostname and service      //
// Protocol and port name is figured out via getaddrinfo //
///////////////////////////////////////////////////////////

void Net::serve(const Chain& hostname, const Chain& service)
{
    struct addrinfo hints, *res;
    int err;

    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;

    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res) ) != 0 ) 
    {
	Chain msg = Chain("Cannot get adr info for ") + hostname + Chain("/") + service;
	throw Exception(EXLOC, msg);
    }

    try
    {
	if ( (_csock=socket(res->ai_family,res->ai_socktype,res->ai_protocol)) == 0)
	    throw Exception(EXLOC, "socket system error");
	
	int optVal = 1;
	if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
	{
	    Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	}
		
	if ( ::bind(_csock, res->ai_addr, res->ai_addrlen) < 0 )
	{
	    Chain msg = Chain("bind system error on service ") + service + Chain(" : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	}
	
	if ( ::listen(_csock,3) < 0 )
	{
	    Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);
	}
    }
    catch ( Exception e )
    {
	freeaddrinfo(res);
	throw e;
    }
}


//////////////////////////////////////
// Serves v4 connetions on any host //
//////////////////////////////////////

void Net::serve(unsigned port)
{
    struct sockaddr_in sa;

    if ( ( _csock = ::socket(AF_INET,SOCK_STREAM,0) ) == 0 )
    	throw Exception(EXLOC, "socket system error");

    int optVal = 1;

    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = INADDR_ANY;
    sa.sin_port = htons(port);

    if ( ::bind(_csock,(struct sockaddr *)&sa, sizeof(sa) ) < 0 )
    {
	Chain msg = Chain("bind system error on port ") + Chain(port) + Chain(" : ") + Chain(strerror(errno));
    	throw Exception(EXLOC, msg);
    }

    if ( ::listen(_csock,3) < 0 )
    {
	Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }
}

/////////////////////////////////////////////
// Serves v4 and v6 connetions on any host //
/////////////////////////////////////////////

void Net::serve6(unsigned port)
{
    struct sockaddr_in6 sa;

    if ( ( _csock = ::socket(AF_INET6,SOCK_STREAM,0) ) == 0 )
    	throw Exception(EXLOC, "socket system error");

    int optVal = 1;

    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }

    // we force double stack here
    int v6Only = 0;
    if ( setsockopt(_csock, IPPROTO_IPV6, IPV6_V6ONLY, (void *)&v6Only, sizeof(v6Only)) )
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    sa.sin6_family = AF_INET6;
    sa.sin6_addr = in6addr_any;
    sa.sin6_port = htons(port);

    if ( ::bind(_csock,(struct sockaddr *)&sa, sizeof(sa) ) < 0 )
    {
	Chain msg = Chain("bind system error on port ") + Chain(port) + Chain(" : ") + Chain(strerror(errno));
    	throw Exception(EXLOC, msg);
    }

    if ( ::listen(_csock,3) < 0 )
    {
	Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }
}

/* 
   with nextRequest, incoming requests are accepted on a well known port. Further
   incoming messages from the partner are handled outside the method ( in a dedicated thread ) 
*/

NetHandler* Net::nextRequest(unsigned timeout)
{   
    int s;
    // struct timeval t;
    TIMEVAL t;
    t.tv_sec = timeout / 1000000;
    t.tv_usec = timeout % 1000000;

    fd_set fdSet;
    
    FD_ZERO(&fdSet);
    FD_SET(_csock, &fdSet);
    
    if ( ( s = ::select(_csock + 1, &fdSet, NULL, NULL, &t) ) < 0 )
    {
	Chain msg = Chain("select system error : ") + Chain(strerror(errno)); 
	throw Exception(EXLOC, msg);
    }

    int nsocket = 0;
    
    if ( s > 0 )
    {
	if ( FD_ISSET(_csock, &fdSet) )
	{
	    FD_CLR(_csock, &fdSet);
	    
	    struct sockaddr_in sa;

	    socklen_t sa_size=sizeof(sa);
	    
	    // check for new connections ( non-blocking )
	    if ( ( nsocket = ::accept(_csock, (struct sockaddr *)&sa, &sa_size) ) < 0 )
	    {
		if ( errno != EWOULDBLOCK )
		{
		    Chain msg = Chain("accept system error: ") + Chain(strerror(errno));
		    throw Exception(EXLOC, msg);
		}
	    }
	    if ( nsocket > 0 )
	    {
		NetHandler* pNetHandle = new NetHandler(nsocket, _initMsgBufLen, _sizeBufLen, _maxSendLen);

		pNetHandle->setSource(getSourceInfo( (struct sockaddr *)&sa ));
		
		try 
		{
		    pNetHandle->readMsg();
		}
		catch ( Exception e)
		{	    
		    pNetHandle->disconnect();
		    delete pNetHandle;   
		    return 0;
		}
		return pNetHandle;
	    }				
	}
    }
    return 0;
}

Chain getSourceInfo(struct sockaddr *s)
{
    struct sockaddr_in *sin = (struct sockaddr_in *)s;

    /* ntoa just works for ipv4 
    cout << "ntoa result : " << endl;
    cout << inet_ntoa (sin->sin_addr);
    cout << endl;
    */
    
    char ip[INET6_ADDRSTRLEN];
    // uint16_t port;

    void *addr;
    // char *ipver;
    
    if (sin->sin_family == AF_INET)	
    {
	// IPv4
	// cout << "Detected v4 address" << endl;
	struct sockaddr_in *ipv4 = (struct sockaddr_in *)sin;
	addr = &(ipv4->sin_addr);
    }
    else	
    {
	// IPv6
	// cout << "Detected v6 address" << endl;
	struct sockaddr_in6 *ipv6 = (struct sockaddr_in6 *)sin;
	addr = &(ipv6->sin6_addr);
    }
    
    return Chain( inet_ntop(sin->sin_family, addr, ip, sizeof(ip)) );

}
