///////////////////////////////////////////////////////////////////////////////
//                                                         
// Element.cc
// ----------
// XML element class implementation
//                                               
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: Element
//
// Description: XML element container class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include "Element.h"
#include "XMLEscaper.h"

#include <string.h>
#include <stdlib.h>

Element::Element(unsigned refCount)
{
    _pParent = 0;
    _refCount = refCount;
    _pOutStream = 0;
}

Element::Element(const Chain& name, unsigned refCount)
{
    _name=name;
    _pParent=0;
    _refCount=refCount;
    _pOutStream = 0;
}

Element::~Element()
{    
    Element **pElem = _childList.First();
    while ( pElem )
    {
	if ( (*pElem)->getRef() == 1 )
	{
	    (*pElem)->clear();
	    delete (*pElem);
	}
	else
	{
	    (*pElem)->decRef();
	}
	pElem = _childList.Next();
    }
    
    char **pData = _dataList.First();
    while ( pData )
    {		
	free(*pData);
	pData = _dataList.Next();
    }	
    _dataList.Empty();
}

void Element::incRef()
{
    _refCount++;
}

void Element::decRef()
{
    _refCount--;
}

unsigned Element::getRef()
{
    return _refCount; 
}

void Element::clear()
{
    Element **pElem = _childList.First();
    while ( pElem )
    {
	if ( (*pElem)->getRef() == 1 )
	{
	    (*pElem)->clear();
	    delete (*pElem);
	}
	else
	{
	    (*pElem)->decRef();
	}
	pElem = _childList.Next();
    }
    _childList.Empty();

    char **pData = _dataList.First();
    while ( pData )
    {		
	free(*pData);
	pData = _dataList.Next();
    }	
    _dataList.Empty();
}

Element* Element::getParent()
{
    return _pParent;
}

void Element::setParent(Element *pParent)
{
    _pParent = pParent;
}

const Chain& Element::getName() const
{
    return _name;
}

void Element::setName(const Chain& name)
{
    _name = name;
}

void Element::setAttributeList(ListT<Attribute>& attrList)
{
    _attrList = attrList;
}

const ListT<Attribute>& Element::getAttributeList() const
{
    return _attrList;
}

void Element::setAttribute(const Chain& attr, const Chain& value)
{
    Attribute* pAttr = _attrList.Find( Attribute( attr ) ); 
    if ( pAttr )
    {
	pAttr->setValue(value);
    }
    else
    {
	_attrList.Insert ( Attribute ( attr, value ) );
    }
}

bool Element::hasAttribute(const Chain& attr) const
{
    if ( _attrList.Find( Attribute( attr ) ) )
	return true;
    return false;
}

Chain Element::getAttributeValue(const Chain& attr) const
{
    Attribute* pAttr = _attrList.Find( Attribute( attr ) ); 
    if ( pAttr )
    {
	return pAttr->getValue();
    }
    return Chain();
}

void Element::addContent(Element* pElement )
{
    pElement->setParent(this);
    pElement->incRef();
    _childList.Insert(pElement);
}

ListT<Element*> Element::getAllChildren() const
{
    return _childList;
}

bool Element::removeChild(Element *pElement)
{
    bool isRemoved =  _childList.Remove(pElement);
    if ( isRemoved )
    {
	if ( pElement->getRef() == 1 )
	{
	    pElement->clear();
	    delete pElement;
	}
	else
	{
	    pElement->decRef();
	}
    }
    return isRemoved;
}

ListT<Element*> Element::getChildren(const Chain& name) const
{
    ListT<Element*> elemList;

    Element **pElem = _childList.First();
    while ( pElem )
    {
	if ( (Chain)(*pElem)->getName() == (Chain)name)
	{
	    elemList.Insert(*pElem);
	}
	pElem = _childList.Next();
    }
    return elemList;
}

void Element::setOutStream(XMLOutStream *pOutStream)
{
    _pOutStream = pOutStream;
}

XMLOutStream* Element::getOutStream()
{
    return _pOutStream;
}

void Element::setText(const Chain& t)
{
    _text = t;
    XMLEscaper esc;    
    esc.descape(_text);
}

Chain Element::getXMLText() const
{
    Chain text = _text.cutTrailing(Chain(" \n\t"));
    XMLEscaper esc;    
    esc.escape(text);
    return text;
}

void Element::addData(char* data)
{
    _dataList.Insert(data);
}

ListT<char*>& Element::getDataList()
{
    return _dataList;
}

Element* Element::createClone()
{
    Element* pClone = new Element( _name );

    Attribute* pAttr = _attrList.First(); 
    while ( pAttr )
    {
	pClone->setAttribute(pAttr->getName(), pAttr->getValue());
	pAttr = _attrList.Next(); 
    }
    pClone->setText( _text );

    Element **pElem = _childList.First();
    while ( pElem )
    {
	Element* pE = (*pElem)->createClone();
	pClone->addContent(pE);
	pElem = _childList.Next();
    }

    char **pData = _dataList.First();
    while ( pData )
    {
	unsigned long l = strlen(*pData) + 1;
	char* pCopyData = (char*)malloc(l);
	strcpy(pCopyData, *pData);
	pCopyData[l]=0;       
	pClone->addData(pCopyData);

	pData = _dataList.Next();
    }       

    return pClone;
}

Chain Element::getText() const
{
    return _text.cutTrailing(Chain(" \n\t"));
}

Element& Element::operator=(const Element& elem)
{
    _name = elem._name;
    _attrList = elem._attrList;
    _text = elem._text;
    _pParent = elem._pParent;
    _refCount = elem._refCount;
    _pOutStream = elem._pOutStream;

    char **pData = elem._dataList.First();
    while ( pData )
    {
	unsigned long l = strlen(*pData) + 1;
	char* pCopyData = (char*)malloc(l);
	strcpy(pCopyData, *pData);
	pCopyData[l]=0;
	pData = elem._dataList.Next();
    }       
    return (*this);
}
    
bool Element::operator==(const Element& elem) const
{
    if ( _name == elem._name )
	return true;
    return false;
}
