///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoCheckObject.cc 
// ------------------
// Cego check object entry implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoCheckObject
//
// Description: Derived class from CegoDecodable to use as a container class for check objects
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoCheckObject.h"
#include "CegoXMLdef.h"

// LFC INCLUDES
#include <string.h>
#include <stdlib.h>

CegoCheckObject::CegoCheckObject()
{
    _pPredDesc = 0;
}

CegoCheckObject::CegoCheckObject(const CegoCheckObject& co) : CegoDecodableObject(co)
{
    _tabName = co._tabName;
    _pPredDesc = co._pPredDesc->clone();
}

CegoCheckObject::CegoCheckObject(ObjectType type, const Chain& objName, int tabSetId) : CegoDecodableObject(type, objName, tabSetId)
{
    _pPredDesc = 0;
}

CegoCheckObject::CegoCheckObject(int tabSetId, const Chain& checkName, const Chain& tabName, CegoPredicate *pPredDesc) : CegoDecodableObject(CegoObject::CHECK, checkName, tabSetId)
{
    _pPredDesc = pPredDesc;
    _tabName = tabName;
}

CegoCheckObject::~CegoCheckObject()
{
    if ( _pPredDesc )
    {
	delete _pPredDesc;
    }
}

void CegoCheckObject::setTabName(const Chain& tabName)
{
    _tabName = tabName;
}

const Chain& CegoCheckObject::getTabName() const
{
    return _tabName;
}
  
void  CegoCheckObject::setPredDesc(CegoPredicate *pPredDesc)
{
    _pPredDesc = pPredDesc;
}

CegoPredicate* CegoCheckObject::getPredDesc()
{
    return _pPredDesc;
}

int CegoCheckObject::getEntrySize() const
{
    int entrySize = CegoObject::getBaseSize();

    entrySize += 1; // object tabname size
    entrySize += _tabName.length();
    entrySize += _pPredDesc->getEncodingLength(0);

    return entrySize;
}

void CegoCheckObject::encode(char *buf) const
{
    char* bufPtr = buf;

    int entrySize = getEntrySize();

    CegoObject::encodeBase(bufPtr, entrySize);
    bufPtr += CegoObject::getBaseSize();

    char c;

    c = (char)_tabName.length();
    memcpy (bufPtr, &c , 1);
    bufPtr++;

    memcpy(bufPtr, (char*)_tabName, _tabName.length());    
    bufPtr=bufPtr + _tabName.length();

    _pPredDesc->encode(bufPtr, 0);
}

void CegoCheckObject::decode(char *buf)
{
    char* bufPtr = buf;

    int size;

    CegoObject::decodeBase(bufPtr, size);
    bufPtr += CegoObject::getBaseSize();

    bufPtr++;
    _tabName =  Chain(bufPtr);    
    bufPtr=bufPtr + _tabName.length();
    
    // TODO
    // No support for check objects which have to access table manager
    _pPredDesc = new CegoPredicate(bufPtr, 0, 0, getTabSetId() );
}

CegoCheckObject& CegoCheckObject::operator = ( const CegoCheckObject& co)
{
    CegoObject::operator=(co);

    _tabName = co._tabName;
    _pPredDesc = co._pPredDesc->clone();
    return (*this);
}

bool CegoCheckObject::operator == ( const CegoCheckObject& co)
{
    return CegoObject::operator==(co);
}

Chain CegoCheckObject::toChain(int defTabSetId) const
{
    Chain s;
    s = CegoObject::toChain();

    s += Chain("TabName: ") + _tabName + Chain("\n");
    s += Chain("PredDesc: ") + _pPredDesc->toChain(defTabSetId);
    
    return s;
}

Chain CegoCheckObject::getFormatted() const
{
    Chain s;

    int maxAttrLen = 12;
    int maxAttrLen1 = 12;
    int maxAttrLen2 = 12;
    
    s += "+" + fill("-", maxAttrLen + 1) + "+" + "\n";
    s += "| ObjectName : ";
    s += getName();
    s += fill(" ", maxAttrLen - 12 - getName().length()) + "|\n";
	
    s += "| ObjectType : check " +  fill(" ", maxAttrLen - 18) + "|\n";

    s += "| Table      : " + _tabName;
    s += fill(" ", maxAttrLen - 12 - _tabName.length()) + "|" + "\n";
	
    s += "+---------" + fill("-", maxAttrLen1-9)  + "----------" +  fill("-", maxAttrLen2-9) + "+\n";
    s += "| Check   " +  fill(" ", maxAttrLen1-9) + "          " +  fill(" ", maxAttrLen2-9) + "|\n";
    s += "+---------" + fill("-", maxAttrLen1-9) + "+---------" +  fill("-", maxAttrLen2-9) + "+\n";
	
    s += _pPredDesc->toChain(0);

    s +="+" + fill("-", maxAttrLen1) + "+" + fill("-", maxAttrLen2) + "+\n";

    return s;
}

Chain CegoCheckObject::fill(const Chain& s, int num) const
{    
    Chain fs = Chain();
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }
    return fs;
}
