///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBTreeObject.cc 
// ------------------
// Cego btree object class implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBTreeObject
// 
// Description: Binary tree index structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoBTreeObject.h"
#include "CegoTypeConverter.h"
#include "CegoXMLdef.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoBTreeObject::CegoBTreeObject()
{
}

CegoBTreeObject::CegoBTreeObject(const CegoBTreeObject& oe) : CegoContentObject(oe)
{
    _pageId = oe._pageId;
    _relevance = oe._relevance;
}

CegoBTreeObject::CegoBTreeObject(int tabSetId, CegoObject::ObjectType type, const Chain& objName ) : CegoContentObject(tabSetId, type, objName)
{
    _pageId = 0;
    _relevance = 0;
}

CegoBTreeObject::CegoBTreeObject(int tabSetId, CegoObject::ObjectType type,  const Chain& objName, const ListT<CegoField>& schema, const Chain& tabName) : CegoContentObject(tabSetId, type, objName, tabName, schema)
{
    _pageId = 0;
    _relevance=0;
}

CegoBTreeObject::~CegoBTreeObject()
{   
}

void CegoBTreeObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

void CegoBTreeObject::setDataPageId(PageIdType pageId)
{
    _pageId = pageId;
}

PageIdType CegoBTreeObject::getDataPageId() const
{
    return _pageId;
}

void CegoBTreeObject::increaseRelevance()
{
    _relevance++;
}

void CegoBTreeObject::decreaseRelevance()
{
    _relevance--;
}

int CegoBTreeObject::getRelevance() const
{
    return _relevance;
}

int CegoBTreeObject::getEntrySize() const
{
    int entrySize = CegoContentObject::getBaseContentSize();

    entrySize = entrySize + sizeof(PageIdType); // pageId
    entrySize = entrySize + sizeof(int); // relevance
        
    return entrySize;
}

void CegoBTreeObject::encode(char *buf) const
{
    char* bufPtr = buf;

    int entrySize = getEntrySize();

    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();

    memcpy(bufPtr, &_pageId, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(bufPtr, &_relevance, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

void CegoBTreeObject::decode(char *buf)
{
    char* bufPtr = buf;

    int size;

    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();
    
    memcpy(&_pageId, bufPtr, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(&_relevance, bufPtr, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

bool CegoBTreeObject::isValid()
{
    return _pageId != 0;
}

CegoBTreeObject& CegoBTreeObject::operator = ( const CegoBTreeObject& oe) 
{
    CegoContentObject::operator=(oe);
    _pageId = oe._pageId;
    _relevance = oe._relevance;
    return (*this);
}

bool CegoBTreeObject::operator == ( const CegoBTreeObject& oe)
{
    return CegoContentObject::operator==(oe);
}

Chain CegoBTreeObject::getId(CegoProcBlock *pBlock) const
{
    return getTabName() + Chain("@") + getTableSet() + Chain("@") + getName();
}

Chain CegoBTreeObject::toChain(int defTabSetId) const
{
    if  ( defTabSetId == getTabSetId() )
	return getTabName() + Chain(" ") + getName();
    else
	return getTabName() + Chain("@") + getTableSet() + Chain(" ") + getName();
}

Chain CegoBTreeObject::getFormatted() const
{
    Chain s;
	
    int maxAttrLen = 12;
    int maxDefLen = 10;

    CegoField* pF = _schema.First();
    while (pF)
    {
	if (maxAttrLen < pF->getAttrName().length())
	    maxAttrLen = pF->getAttrName().length();
	if (maxDefLen < pF->getValue().valAsChain().length())
	    maxDefLen = pF->getValue().valAsChain().length();
	
	pF = _schema.Next();
    }
    
    s += Chain("+") + fill("-", maxAttrLen) + fill(Chain("-"), maxDefLen + 30) + Chain("+\n");
    s += Chain("| ObjectName : ") + getName() + fill(" ", maxAttrLen + maxDefLen + 17 - getName().length()) + Chain("|\n");
	
    s += Chain("| ObjectType : ");
    switch ( getType() )
    {
    case CegoObject::BTREE:
	s += Chain("btree          "); 
	break;
    case CegoObject::UBTREE:
	s += Chain("unique btree   ");
	break;
    case CegoObject::PBTREE:
	s += Chain("primary btree   ");
	break;
    case CegoObject::SYSTEM:
    case CegoObject::TABLE:
    case CegoObject::VIEW:
    case CegoObject::RBSEG:
    case CegoObject::FKEY:
    case CegoObject::PROCEDURE:
    case CegoObject::CHECK:
    case CegoObject::JOIN:
    case CegoObject::PAVLTREE:
    case CegoObject::UAVLTREE:
    case CegoObject::AVLTREE:
    case CegoObject::TRIGGER:
    case CegoObject::ALIAS:
    case CegoObject::UNDEFINED:
	break;
    }

    s += fill(" ", maxAttrLen + maxDefLen + 1 ) + Chain("|\n");
    
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");
    s += Chain("| Attribute ") + fill(" ", maxAttrLen-10) + Chain("| Type          | Default ") + fill(" ", maxDefLen - 8) + Chain("| Nullable |\n");
    s += Chain("+-----------") + fill("-", maxAttrLen-10) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");
	
    pF = _schema.First();
    while (pF)
    {
	int num = maxAttrLen - pF->getAttrName().length() ;
	s += Chain("| ") + pF->getAttrName() + fill(" ", num) + Chain(" |");
	    	    
	switch (pF->getType())
	{
	case INT_TYPE:
	{
	    s += Chain("  int          |");
	    break;
	}
	case LONG_TYPE:
	{
	    s +=  Chain("  long         |");
	    break;
	}
	case BOOL_TYPE:
	{
	    s += Chain("  bool         |");
	    break;
	}
	case DATETIME_TYPE:
	{
	    s += Chain("  datetime     |");
	    break;
	}
	case VARCHAR_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  string[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case BIGINT_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  bigint[") + l + Chain("]") + fill(" ", 6 - l.length()) + Chain("|");
	    break;
	}
	case SMALLINT_TYPE:
	{
	    s += Chain("  smallint     |");
	    break;
	}
	case TINYINT_TYPE:
	{
	    s += Chain("  tinyint      |");
	    break;
	}	
	case DECIMAL_TYPE:
	{
	    Chain l(pF->getLength());
	    s += Chain("  decimal[") + l + Chain("]") + fill(" ", 5 - l.length()) + Chain("|");
	    break;
	}
	case FLOAT_TYPE:
	{
	    s += Chain(" float         |");
	    break;
	}
	case DOUBLE_TYPE:
	{
	    s += Chain(" double        |");
	    break;
	}
	case BLOB_TYPE:
	{
	    s += Chain(" blob          |");
	    break;
	}
	case CLOB_TYPE:
	{
	    s += Chain(" clob          |");
	    break;
	}
	case NULL_TYPE:
	case PAGEID_TYPE:
	    break;
	}
	
	Chain defVal = pF->getValue().valAsChain();
	
	s += Chain(" ") + defVal + fill(" ", maxDefLen - defVal.length()) + Chain(" |");
	
	if ( pF->isNullable() )
	    s += Chain(" y        |");
	else
	    s += Chain(" n        |");
	
	pF = _schema.Next();
	    
	s += "\n";
    }
    s += Chain("+") + fill("-", maxAttrLen + 1) + Chain("+---------------+") + fill("-", maxDefLen + 1) + Chain("+----------+\n");

    return s;
}

Chain CegoBTreeObject::fill(const Chain& s, int num) const
{    
    Chain fs = Chain();
    while (num > 0)
    {
	fs = fs + s;
	num--;
    }

    return fs;
}

CegoContentObject* CegoBTreeObject::clone(bool isAttrRef)
{
    CegoBTreeObject *pClone = new CegoBTreeObject(*this);
    return (pClone);
}
