///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdminThread.cc  
// ------------------                                                     
// Cego admin thread class implementation
//                                                         
// Design and Implementation by Bjoern Lemke
//               
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdminThread
// 
// Description: Handling all database admin requests
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Exception.h>
#include <lfcbase/Host.h>
#include <lfcbase/Datetime.h>
#include <lfcbase/AESCrypt.h>
#include <lfcbase/Sleeper.h>
#include <lfcbase/Net.h>
#include <lfcbase/ThreadLock.h>
#include <lfcbase/CommandExecuter.h>
#include <lfcbase/Directory.h>

// CEGO INCLUDES
#include "CegoXMLdef.h"
#include "CegoAdminThread.h"
#include "CegoAdminThreadPool.h"
#include "CegoXPorter.h"
#include "CegoRecoveryManager.h"
#include "CegoBufferPoolEntry.h"

// POSIX INCLUDES
#include <stdlib.h>
#include <string.h>

extern char __caseSensitiveFlag;
extern char __quoteEscapeFlag;
extern char __allowDuplicateNull;

extern char __currencySymbol;
extern char __decimalPoint;

extern Chain __dateTimeFormat;
extern ListT<Chain> __dateFormatList;
extern ThreadLock __dateFormatLock;

CegoAdminThread::CegoAdminThread() : Thread()
{
}

CegoAdminThread::CegoAdminThread(CegoAdminThreadPool *pPool, CegoDatabaseManager *pDBMng, CegoDbThreadPool *pDbPool, CegoLogThreadPool *pLogPool) : Thread()
{
    _pDBMng = pDBMng;
    _pPool = pPool;
    _pTabMng = new CegoDistManager(pDBMng);

    // not auth for admin, required for tableset recovery to access all objects
    _pTabMng->disableAuth();
    
    _pDbPool = pDbPool;
    _pLogPool = pLogPool;

    _modId = _pDBMng->getModId("CegoAdminThread");
}

CegoAdminThread::~CegoAdminThread()  
{
}

const Chain& CegoAdminThread::lastAction() const
{
    return _lastAction;
}

void* CegoAdminThread::job(void* arg)
{	
    _idx = *(unsigned long long*)arg;

    _pTim = new NanoTimer();
    
    _pPool->setTid(_idx, getTid());
    _pTabMng->setThreadId(getTid());

    int queueDelay = _pDBMng->getQueueDelay();
	
    while ( ! _pPool->isTerminated() )
    {	
	try
	{
	    _pTim->reset();
	    _pTim->start();

	    _pRequest = _pPool->nextRequest();
	    
	    if ( _pRequest ) 
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": serving service request"));
#endif
		_pPool->setState(_idx, CegoAdminThreadPool::CONNECTED);
	
		_pDBMng->increaseActiveAdmThread();			

		CegoAdminHandler* pAH = 0;
		try
		{
		    pAH = new CegoAdminHandler(_pDBMng, _pRequest);
		    serveSession(pAH);
		    delete pAH;
		}
		catch ( Exception e)
		{	
		    if ( pAH )
			delete pAH;

		    Chain msg;
		    Chain module;
		    int line;
		    
		    Chain exep;
		    while ( e.pop(module, line, msg) )
		    {
			exep += Chain("\n\t") + module + Chain("(") +  Chain(line) + Chain(") : ") + msg;
		    }

		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": Admin session aborted : ") + exep);

		}
		_pDBMng->decreaseActiveAdmThread();
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": service request finished"));
#endif

		_pPool->setState(_idx, CegoAdminThreadPool::READY);

		delete _pRequest;	
	    }
	    else
	    {		
		// _pDBMng->log(_modId, Logger::NOTICE, Chain("Thread ") + Chain(_idx) + Chain(": Waiting for request" ));
		Sleeper s;
		s.microSleep(queueDelay);
	    }

	    Chain tableSet;
	    Chain secondary;
	    Chain mediator;
	    Chain user;
	    Chain passwd;

	    // check for asynchronous recovery requests
	    bool doRecovery = _pDBMng->nextRecovery(tableSet);
	    if ( doRecovery )
	    {
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);
		
		try 
		{
		    _pDBMng->increaseActiveAdmThread();

		    _pDBMng->log(_modId, Logger::NOTICE, Chain("Thread ") + Chain(_idx) + Chain(": Tableset recovery started for tableset " ) + tableSet);
		    CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::REMOTE);
		    unsigned long long lsn = recoveryMng.recoverTableSet(tableSet, 0, 0);

		    _pDBMng->log(_modId, Logger::NOTICE, Chain("Thread ") + Chain(_idx) + Chain(": Tableset " ) + tableSet + Chain(" recovered to lsn ") + Chain(lsn));
		    
		    _pDBMng->decreaseActiveAdmThread();
		    _pPool->setState(_idx, CegoAdminThreadPool::READY);
		}
		catch ( Exception e)
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg);
		}
		_pPool->setState(_idx, CegoAdminThreadPool::READY);
	    }

	    // check for asynchronous copy requests
	    int copyId;
	    bool doCopy = _pDBMng->nextCopy(copyId, tableSet, secondary, mediator, user, passwd);
	    if ( doCopy )
	    {
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);
		_pDBMng->increaseActiveAdmThread();

		_pDBMng->setCopyStatus(copyId, Chain("Starting copy"));

		try 
		{
#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) 
				 + Chain(": Copying tableset ") + tableSet + Chain(" to secondary"));
#endif
    
		    if ( _pDBMng->getTableSetRunState(tableSet) == Chain(XML_OFFLINE_VALUE) )
		    {
			copyTableSet(copyId, tableSet, secondary, mediator, user, passwd, false);
		    }
		    else if ( _pDBMng->getTableSetRunState(tableSet) == Chain(XML_ONLINE_VALUE) )
		    {
			_pTabMng->beginBackup(tableSet, Chain("Tableset copy"));		    
			copyTableSet(copyId, tableSet, secondary, mediator, user, passwd, true);			
			_pTabMng->endBackup(tableSet, Chain("Tableset copy"), false);
		    }
		    else
		    {
			Chain msg = Chain("Invalid runstate ") + _pDBMng->getTableSetRunState(tableSet) + Chain(" for tableset copy");
			throw Exception(EXLOC, msg);
		    }

		    _pDBMng->setCopyStatus(copyId, "Copy finished");
		}
		catch ( Exception e)
		{
		    Chain msg;
		    e.pop(msg);
		    _pDBMng->setCopyStatus(copyId, msg);		
		    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg );
		}
		_pDBMng->decreaseActiveAdmThread();
		_pPool->setState(_idx, CegoAdminThreadPool::READY);		    
	    }
	    _pTim->stop();
	    _pPool->addThreadIdle(_idx, _pTim->getSum());
	}
	catch ( Exception e)
	{
	    Chain msg;
	    e.pop(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, Chain("Thread ") + Chain(_idx) + Chain(": ") + msg );
	}
    }

    delete _pTim;
    
    // _pDBMng->log(_modId, Logger::NOTICE, Chain("Thread ") + Chain(_idx) + Chain(": Terminated" ));
    return 0;
}

void CegoAdminThread::serveSession(CegoAdminHandler *pAH)
{   
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": serving session"));
#endif
    
    if ( pAH->acceptSession() )
    {
	bool isTraceOn;
	if ( _pDBMng->checkAdminUser(pAH->getUser(), pAH->getPassword(), isTraceOn) == false )
	{
	    Chain msg = Chain("Invalid user or password");
	    pAH->sendError(msg);
	    return;
	}
	else
	{
	    Chain msg = Chain("Access granted");
	    pAH->sendResponse(msg);
	}

	_user = pAH->getUser();
	_password = pAH->getPassword();

	bool isTerminated = false;
	
	while ( isTerminated == false && _pPool->isTerminated() == false)
	{	    
	    CegoAdminHandler::RequestType reqType;
	    reqType = pAH->acceptRequest();

	    _pTim->stop();
	    _pPool->addThreadIdle(_idx, _pTim->getSum());
	    _pTim->reset();
	    _pTim->start();
   
	    if ( reqType != CegoAdminHandler::REQTIMEOUT )
	    {
		if ( isTraceOn )
		{
		    _pDBMng->incUserQuery(pAH->getUser());
		}
		_pPool->incNumRequest(_idx);
		
		_pPool->setState(_idx, CegoAdminThreadPool::BUSY);
		
		_pTim->stop();
		_pPool->addThreadIdle(_idx, _pTim->getSum());

		isTerminated = serveRequest(pAH, reqType);

		_pTim->reset();
		_pTim->start();
		
		_pPool->setState(_idx, CegoAdminThreadPool::CONNECTED);
	    }
	    else
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": admin client request timeout occured, waitung ..."));
#endif
	    }
	}
    }
}

bool CegoAdminThread::serveRequest(CegoAdminHandler *pAH, CegoAdminHandler::RequestType reqType)
{
    bool isTerminated = false;

    try
    {	
	switch ( reqType ) 
	{		    
	case CegoAdminHandler::RETRIEVETABLESET:
	{
	    srvRetrieveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::LOCKINFO:
	{
	    srvLockInfo(pAH);
	    break;
	}
	case CegoAdminHandler::LOCKSTAT:
	{
	    srvLockStat(pAH);
	    break;
	}
	case CegoAdminHandler::PARAMETERINFO:
	{
	    srvParameterInfo(pAH);
	    break;
	}
	case CegoAdminHandler::POOLINFO:
	{
	    srvPoolInfo(pAH);
	    break;
	}
	case CegoAdminHandler::POOLENTRYLIST:
	{
	    srvPoolEntryList(pAH);
	    break;
	}
	case CegoAdminHandler::THREADINFO:
	{
	    srvThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DBTHREADINFO:
	{
	    srvDbThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DBTHREADHIST:
	{
	    srvDbThreadHistory(pAH);
	    break;
	}
	case CegoAdminHandler::QUERYHISTLAST:
	{
	    srvQueryHistoryLast(pAH);
	    break;
	}
	case CegoAdminHandler::QUERYHISTCOST:
	{
	    srvQueryHistoryCost(pAH);
	    break;
	}
	case CegoAdminHandler::SET_NUMQUERYLAST:
	{
	    srvNumQueryLast(pAH);
	    break;
	}
	case CegoAdminHandler::SET_NUMQUERYCOST:
	{
	    srvNumQueryCost(pAH);
	    break;
	}
	case CegoAdminHandler::ADMTHREADINFO:
	{
	    srvAdmThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::LOGTHREADINFO:
	{
	    srvLogThreadInfo(pAH);
	    break;
	}
	case CegoAdminHandler::DBTHREADABORT:
	{
	    srvDbThreadAbort(pAH);
	    break;
	}
	case CegoAdminHandler::DBSESSIONINFO:
	{
	    srvDbSessionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::COPYINFO:
	{
	    srvCopyInfo(pAH);
	    break;
	}
	case CegoAdminHandler::SHOWQUERYCACHE:
	{
	    srvShowQueryCache(pAH);
	    break;
	}
	case CegoAdminHandler::LISTQUERYCACHE:
	{
	    srvListQueryCache(pAH);
	    break;
	}
	case CegoAdminHandler::CLEANQUERYCACHE:
	{
	    srvCleanQueryCache(pAH);
	    break;
	}
	case CegoAdminHandler::SETQUERYCACHEPARAM:
	{
	    srvSetQueryCacheParam(pAH);
	    break;
	}
	case CegoAdminHandler::SHOWTABLECACHE:
	{
	    srvShowTableCache(pAH);
	    break;
	}
	case CegoAdminHandler::LISTTABLECACHE:
	{
	    srvListTableCache(pAH);
	    break;
	}
	case CegoAdminHandler::CLEANTABLECACHE:
	{
	    srvCleanTableCache(pAH);
	    break;
	}
	case CegoAdminHandler::SETTABLECACHEPARAM:
	{
	    srvSetTableCacheParam(pAH);
	    break;
	}
	case CegoAdminHandler::DEFINE_TABLESET:
	{
	    srvDefineTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_TABLESET:
	{
	    srvRemoveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::RESET_TABLESET:
	{
	    srvResetTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::CREATE_TABLESET:
	{
	    srvCreateTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::DROP_TABLESET:
	{
	    srvDropTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::START_TABLESET:
	{
	    srvStartTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::STOP_TABLESET:
	{
	    srvStopTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::START_RECOVERY:
	{
	    srvStartRecovery(pAH);	      	
	    break;
	}
	case CegoAdminHandler::STOP_RECOVERY:
	{
	    srvStopRecovery(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DBSPEC:
	{		
	    srvGetDbSpec(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DBINFO:
	{		
	    srvGetDbInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_TSINFO:
	{		
	    srvGetTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_DETAILED_TSINFO:
	{		
	    srvGetDetailedTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_ARCHLOG_INFO:
	{		
	    srvGetArchLogInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_TRANSACTION_INFO:
	{		
	    srvGetTransactionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::GET_BUSTAT:
	{		
	    srvGetBUStatInfo(pAH);
	    break;
	}
	case CegoAdminHandler::PROP_TSINFO:
	{		
	    srvPropTSInfo(pAH);
	    break;
	}
	case CegoAdminHandler::COPY_TABLESET:
	{
	    srvCopyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::COPY_FILE:
	{
	    srvCopyFile(pAH);
	    break;
	}
	case CegoAdminHandler::GET_NODELIST:
	{		
	    srvGetNodeList(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TABLESETNODE:
	{
	    srvSetTableSetNode(pAH);
	    break;
	}
	case CegoAdminHandler::SET_LSN:
	{
	    srvSetLSN(pAH);
	    break;
	}
	case CegoAdminHandler::SET_RUNSTATE:
	{
	    srvSetRunState(pAH);
	    break;
	}
	case CegoAdminHandler::SET_SYNCSTATE:
	{
	    srvSetSyncState(pAH);
	    break;
	}
	case CegoAdminHandler::EXPORT_TABLESET:
	{
	    srvExportTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::IMPORT_TABLESET:
	{
	    srvImportTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::EXPORT_TABLE:
	{
	    srvExportTable(pAH);
	    break;
	}
	case CegoAdminHandler::IMPORT_TABLE:
	{
	    srvImportTable(pAH);
	    break;
	}
	case CegoAdminHandler::ADD_DATAFILE:
	{
	    srvAddDatafile(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_USER:
	{
	    srvUserInfo(pAH);
	    break;
	}	    
	case CegoAdminHandler::ADD_USER:
	{
	    srvAddUser(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_USER:
	{
	    srvRemoveUser(pAH);
	    break;
	}
	case CegoAdminHandler::CHANGE_PWD:
	{
	    srvChangePwd(pAH);
	    break;
	}
	case CegoAdminHandler::LIST_ROLE:
	{
	    srvListRole(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_ROLE:
	{
	    srvShowRole(pAH);
	    break;
	}
	case CegoAdminHandler::CREATE_ROLE:
	{
	    srvCreateRole(pAH);
	    break;
	}
	case CegoAdminHandler::DROP_ROLE:
	{
	    srvDropRole(pAH);
	    break;
	}
	case CegoAdminHandler::ASSIGN_ROLE:
	{
	    srvAssignRole(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_ROLE:
	{
	    srvRemoveRole(pAH);
	    break;
	}
	case CegoAdminHandler::SET_PERM:
	{
	    srvSetPerm(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_PERM:
	{
	    srvRemovePerm(pAH);
	    break;
	}
	case CegoAdminHandler::USER_TRACE:
	{
	    srvUserTrace(pAH);
	    break;
	}
	case CegoAdminHandler::SYNC_TABLESET:
	{
	    srvSyncTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::INIT_LOGS:
	{
	    srvInitLogs(pAH);
	    break;
	}
	case CegoAdminHandler::ADD_ARCHLOG:
	{
	    srvAddArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::REMOVE_ARCHLOG:
	{
	    srvRemoveArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::BEGIN_BACKUP:
	{
	    srvBeginBackup(pAH);
	    break;
	}
	case CegoAdminHandler::EXECUTE_BACKUP:
	{
	    srvExecuteBackup(pAH);
	    break;
	}
	case CegoAdminHandler::END_BACKUP:
	{
	    srvEndBackup(pAH);
	    break;
	}
	case CegoAdminHandler::LIST_BACKUP:
	{
	    srvListBackup(pAH);
	    break;
	}
	case CegoAdminHandler::RESTORE:
	{
	    srvRestore(pAH);
	    break;
	}
	case CegoAdminHandler::RECOVER:
	{
	    srvRecover(pAH);
	    break;
	}
	case CegoAdminHandler::SECSWITCH:
	{
	    srvSecSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::VERIFY_TABLESET:
	{
	    srvVerifyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::CORRECT_TABLESET:
	{
	    srvCorrectTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::DUMP_OBJECT:
	{
	    srvDumpObject(pAH);
	    break;
	}
	case CegoAdminHandler::SET_LOGMNG:
	{
	    srvSetLogMng(pAH);
	    break;
	}
	case CegoAdminHandler::SET_BACKUPMNG:
	{
	    srvSetBackupMng(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_LOGMNG:
	{
	    srvShowLogMng(pAH);
	    break;
	}
	case CegoAdminHandler::SHOW_BACKUPMNG:
	{
	    srvShowBackupMng(pAH);
	    break;
	}
	case CegoAdminHandler::SET_CHECKPOINT:
	{
	    srvSetCheckpoint(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSINITFILE:
	{
	    srvSetTSInitFile(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSSYSSIZE:
	{
	    srvSetTSSysSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSTMPSIZE:
	{
	    srvSetTSTmpSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSAPPSIZE:
	{
	    srvSetTSAppSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSROOTPATH:
	{
	    srvSetTSRootPath(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSLOGNUM:
	{
	    srvSetTSLogNum(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSLOGSIZE:
	{
	    srvSetTSLogSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSSORTAREASIZE:
	{
	    srvSetTSSortAreaSize(pAH);
	    break;
	}
	case CegoAdminHandler::SET_TSLOGUSER:
	{
	    srvSetTSLogUser(pAH);
	    break;
	}
	case CegoAdminHandler::ENABLE_TABLECACHE:
	{
	    srvEnableTableCache(pAH);
	    break;
	}
	case CegoAdminHandler::DISABLE_TABLECACHE:
	{
	    srvDisableTableCache(pAH);
	    break;
	}
	case CegoAdminHandler::ENABLE_QUERYCACHE:
	{
	    srvEnableQueryCache(pAH);
	    break;
	}
	case CegoAdminHandler::DISABLE_QUERYCACHE:
	{
	    srvDisableQueryCache(pAH);
	    break;
	}	
	// mediator requests
	case CegoAdminHandler::MED_NOTIFY:
	{
	    medNotify(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DEFINE_TABLESET:
	{
	    medDefineTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_REMOVE_TABLESET:
	{
	    medRemoveTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RESET_TABLESET:
	{
	    medResetTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CREATE_TABLESET:
	{
	    medCreateTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DROP_TABLESET:
	{
	    medDropTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_START_TABLESET:
	{
	    medStartTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_STOP_TABLESET:
	{
	    medStopTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_COPY_TABLESET:
	{
	    medCopyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SWITCH_TABLESET:
	{
	    medSwitchTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ADD_ARCHLOG:
	{
	    medAddArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_REMOVE_ARCHLOG:
	{
	    medRemoveArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ADD_DATAFILE:
	{
	    medAddDatafile(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_TABLESETLIST:
	{
	    medGetTableSetList(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_DETAILEDTABLESETINFO:
	{
	    medGetDetailedTableSetInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_OBJECTINFO:
	{
	    medGetObjectInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_ARCHLOGINFO:
	{
	    medGetArchLogInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_TRANSACTIONINFO:
	{
	    medGetTransactionInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_GET_BUSTAT:
	{
	    medGetBUStatInfo(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RESET_BUSTAT:
	{
	    medResetBUStat(pAH);
	    break;
	}
	case CegoAdminHandler::MED_BEGIN_BACKUP:
	{
	    medBeginBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_EXECUTE_BACKUP:
	{
	    medExecuteBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_END_BACKUP:
	{
	    medEndBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_LIST_BACKUP:
	{
	    medListBackup(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RECOVER:
	{
	    medRecover(pAH);
	    break;
	}
	case CegoAdminHandler::MED_RESTORE:
	{
	    medRestore(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SECSWITCH:
	{
	    medSecSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::MED_SECRELOCATE:
	{
	    medSecRelocate(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CHECK_TABLESET:
	{
	    medCheckTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_VERIFY_TABLESET:
	{
	    medVerifyTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_CORRECT_TABLESET:
	{
	    medCorrectTableSet(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ENABLE_ARCHLOG:
	{
	    medEnableArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DISABLE_ARCHLOG:
	{
	    medDisableArchLog(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ENABLE_APPENDMODE:
	{
	    medEnableAppendMode(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DISABLE_APPENDMODE:
	{
	    medDisableAppendMode(pAH);
	    break;
	}
	case CegoAdminHandler::MED_ENABLE_AUTOCORRECT:
	{
	    medEnableAutoCorrect(pAH);
	    break;
	}
	case CegoAdminHandler::MED_DISABLE_AUTOCORRECT:
	{
	    medDisableAutoCorrect(pAH);
	    break;
	}
	case CegoAdminHandler::SEC_MEDSWITCH:
	{
	    secMedSwitch(pAH);
	    break;
	}
	case CegoAdminHandler::SEC_MEDRELOCATE:
	{
	    secMedRelocate(pAH);
	    break;
	}
	case CegoAdminHandler::SESSION_CLOSE:
	{
	    isTerminated=true;
	    
#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain(" Thread ") + Chain(_idx) + Chain(": terminating session"));
#endif
	    pAH->sendResponse(Chain("Session Closed"));	    
	    break;
	}
	case CegoAdminHandler::GET_OBJECT_INFO:
	case CegoAdminHandler::REGISTER_HOST:
	case CegoAdminHandler::REQTIMEOUT:
	case CegoAdminHandler::UNKNOWN:
	{
	    pAH->sendError(Chain("Invalid request"));
	}    
	}
    }
    catch ( Exception e )
    {       
	Chain msg;
	Chain module;
	int line;
			
	Chain exep;
	while ( e.pop(module, line, msg) )
	{
	    exep += Chain("\n\t") + module + Chain("(") +  Chain(line) + Chain(") : ") + msg;
	}
	
	_pDBMng->log(_modId, Logger::LOGERR, Chain("Admin Thread ") + Chain(_idx) + Chain(" : ") + exep);
	
	pAH->sendError(msg);
    }    
    return isTerminated;
}

void CegoAdminThread::copyTableSet(int copyId, const Chain& tableSet, const Chain& secondary, const Chain& mediator, const Chain& user, const Chain& passwd, bool copyOnline)
{
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Copying tableset ") +  tableSet + Chain("..."));
#endif

    int adminPort;
    _pDBMng->getAdminPort(adminPort);

#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Thread ") + Chain(_idx) + Chain(": Connecting to secondary ") +  secondary + Chain("..."));
#endif

    Net n (  NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN, NETMNG_MAXSENDLEN );
    
    NetHandler *pNS = n.connect(secondary, adminPort);
    CegoAdminHandler *pAHS = new CegoAdminHandler(_pDBMng, pNS);

    pAHS->requestSession(user, passwd, false);

    NetHandler *pNM = n.connect(mediator, adminPort);

    CegoAdminHandler *pAHM = new CegoAdminHandler(_pDBMng, pNM);
    pAHM->requestSession(user, passwd, false);

    CegoAdminHandler::ResultType res;	
    
    res = pAHS->reqSetSyncState(tableSet, XML_ON_COPY_VALUE);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);      
    }    
    
    res = pAHM->reqSetSyncState(tableSet, XML_ON_COPY_VALUE);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);      
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);	
    }    

    // request system files

    Chain dbSysFileName = _pDBMng->getSysFileName(tableSet);
    Chain dbTempFileName = _pDBMng->getTmpFileName(tableSet);
    Chain tsTicket = _pDBMng->getTSTicket(tableSet);
    
    res = pAHS->reqCopyFile(dbSysFileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    
    
    res = pAHS->reqCopyFile(dbTempFileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    

    if ( copyOnline )
    {
	res = pAHS->reqCopyFile(tsTicket);
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}
    }
    else
    {
	res = pAHS->reqSetLSN(tableSet, _pDBMng->getCommittedLSN(tableSet));
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}
    }

    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> sizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, sizeList);
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, sizeList);
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, sizeList);
    
    Chain *pFileName = dfList.First();
    
    while ( pFileName  ) 
    {	
	_pDBMng->log(_modId, Logger::NOTICE, Chain("Copying datafile ") + *pFileName + Chain(" ..."));

	if ( _pDBMng )
	{
	    _pDBMng->setCopyStatus(copyId, Chain("Copying ") + *pFileName );		
	}
	
	CegoAdminHandler::ResultType res;
	res = pAHS->reqCopyFile( *pFileName, copyId);
	
	if ( res == CegoAdminHandler::ADM_OK )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::NOTICE, msg);
	}
	else if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    Chain msg;
	    pAHS->getMsg(msg);
	    _pDBMng->log(_modId, Logger::LOGERR, msg);
	}    
	   
	pFileName = dfList.Next();	    
    }

    res = pAHS->reqInitLogs(tableSet);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }

    res = pAHS->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHS->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    

    res = pAHM->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
    
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::NOTICE, msg);
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	pAHM->getMsg(msg);
	_pDBMng->log(_modId, Logger::LOGERR, msg);
    }    

    _pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);

    pAHS->closeSession();
    pAHM->closeSession();
    
    delete pAHS;
    delete pAHM;

    delete pNS;
    delete pNM;    		
}

void CegoAdminThread::srvRetrieveTableSet(CegoAdminHandler *pAH)
{
    Chain masterHost;
    pAH->getHostName( masterHost );
    
    CegoAdminHandler* pMasterAH = 0;

    pMasterAH = getSession(Chain("master"), masterHost, pAH->getUser(), pAH->getPassword());

    Chain dbSpec;
    CegoAdminHandler::ResultType res = pMasterAH->reqGetDbSpec(dbSpec);
    
    Chain msg;
    pMasterAH->getMsg(msg);

    closeSession(pMasterAH);

    if ( pAH->syncWithInfo(Chain("primary"), masterHost, msg, 0) == false )
	return;
        
    if ( res != CegoAdminHandler::ADM_OK )
	throw Exception(EXLOC, Chain("Admin action failed"));
    
    XMLSuite xml(dbSpec);
    Document *pDoc = new Document;
    xml.setDocument(pDoc);
    
    try
    {
	xml.parse();
    }
    catch ( Exception e )
    {
	Chain xmlmsg;
	e.pop(xmlmsg);
	Chain msg = Chain("XML parse error at line ") + Chain( xml.getLineNo()) + Chain(" : ") + xmlmsg ;
	_pDBMng->log(_modId, Logger::LOGERR, msg);
	throw Exception(EXLOC, Chain("Cannot retrieve tableset information"), e);
    }

    Element *pDBElement = pDoc->getRootElement();

    ListT<Element*> tsList = pDBElement->getChildren(XML_TABLESET_ELEMENT);

    ListT<Chain> localList;

    _pDBMng->getTableSetList(localList);

    Element **pTS = tsList.First();
    while ( pTS ) 
    {
	Chain tableSet = (*pTS)->getAttributeValue(XML_NAME_ATTR);

	if ( localList.Find( tableSet ) == 0 )
	{	    
	    _pDBMng->setTableSetInfo(tableSet, (*pTS)->createClone() );       	 
	}   
	pTS = tsList.Next();
    }
   
    pAH->sendResponse(Chain("Tableset info retrieved"));
}

void CegoAdminThread::srvLockInfo(CegoAdminHandler *pAH)
{
    try
    {
	Chain lockCat;
	pAH->getLockCat( lockCat );
	
	Element *pLockInfo = new Element(XML_LOCKINFO_ELEMENT);
	Element *pLockEntry;
	
	Chain lockName;
	int lockCount;
	unsigned long long numRdLock;
	unsigned long long numWrLock;
	unsigned long long sumRdDelay;
	unsigned long long sumWrDelay;
	
	if ( lockCat == Chain(XML_LOCK_THREADPOOL_VALUE) )
	{
	    _pPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));    
	    pLockInfo->addContent(pLockEntry);
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDbPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pLogPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDBMng->getDBMLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);    
	    
	    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	    _pDBMng->getXMLLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	    pLockInfo->addContent(pLockEntry);
	}  
	else if ( lockCat == Chain(XML_LOCK_LOGMNG_VALUE) )
	{
	    ListT<Chain> tsList;
	    Chain dbHost;
	    _pDBMng->getDBHost(dbHost);
	    _pDBMng->getActiveTableSet(dbHost, tsList, true, false);
	    
	    Chain *pTabSet = tsList.First();
	    while ( pTabSet )
	    {
		int tabSetId = _pDBMng->getTabSetId(*pTabSet);
		
		pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
		_pDBMng->getLMLockStat(tabSetId, lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
		pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
		pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		pLockInfo->addContent(pLockEntry);    
			
		pTabSet = tsList.Next();
	    }
	}
	else if ( lockCat == Chain(XML_LOCK_QUERYCACHE_VALUE) )
	{
	    ListT<Chain> tsList;
	    Chain dbHost;
	    _pDBMng->getDBHost(dbHost);
	    _pDBMng->getActiveTableSet(dbHost, tsList, true, false);
	    
	    Chain *pTabSet = tsList.First();
	    while ( pTabSet )
	    {
		int tabSetId = _pDBMng->getTabSetId(*pTabSet);
		
		if ( _pDBMng->getQCLockStat(tabSetId, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay) )
		{
		    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
		    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), *pTabSet);
		    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		    pLockInfo->addContent(pLockEntry);    
		}
		pTabSet = tsList.Next();
	    }
	}
	else if ( lockCat == Chain(XML_LOCK_TABLECACHE_VALUE) )
	{
	    ListT<Chain> tsList;
	    Chain dbHost;
	    _pDBMng->getDBHost(dbHost);
	    _pDBMng->getActiveTableSet(dbHost, tsList, true, false);
	    
	    Chain *pTabSet = tsList.First();
	    while ( pTabSet )
	    {
		int tabSetId = _pDBMng->getTabSetId(*pTabSet);

		if ( _pDBMng->getTCLockStat(tabSetId, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay) )
		{		
		    pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);		    
		    pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), *pTabSet);
		    pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		    pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		    pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		    pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		    pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		    pLockInfo->addContent(pLockEntry);    
		}
		pTabSet = tsList.Next();
	    }
	}
	else 
	{
	    int low = 0;
	    int high = 0;
	    
	    if ( lockCat == Chain(XML_LOCK_REC_VALUE) )
	    {
		low = 0;
		high = _pDBMng->getNumRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_RBREC_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema();
		high = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_SYSREC_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_SYSPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_DATAPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema() 
		    + _pDBMng->getNumDataPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_IDXPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_RBPAGE_VALUE) )
	    {
		low = _pDBMng->getNumRecordSema() 
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema();		
	    }
	    else if ( lockCat == Chain(XML_LOCK_DATAFILE_VALUE) )
	    {		
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema() 
		    + _pDBMng->getNumDataFileSema();
	    }
	    else if ( lockCat == Chain(XML_LOCK_BUFFERPOOL_VALUE) )
	    {		
		low = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema() 
		    + _pDBMng->getNumDataPageSema()
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema()
		    + _pDBMng->getNumDataFileSema();
		high = _pDBMng->getNumRecordSema()
		    + _pDBMng->getNumRBRecordSema()
		    + _pDBMng->getNumSysRecordSema()
		    + _pDBMng->getNumSysPageSema()
		    + _pDBMng->getNumDataPageSema() 
		    + _pDBMng->getNumIndexPageSema()
		    + _pDBMng->getNumRBPageSema() 
		    + _pDBMng->getNumDataFileSema() 
		    + _pDBMng->getNumBufferPoolSema();
	    }
	    else
	    {		
		throw Exception(EXLOC, "Unknown lock cat");
	    }
	    
	    int semId;	    
	    for (  semId = low; semId < high; semId++ )
	    {
		pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
		_pTabMng->getLHLockStat(semId, lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
		pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
		pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
		pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
		pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
		pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
		pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
		pLockInfo->addContent(pLockEntry);    
	    }
	}
	
	_lastAction = Chain("LockInfo");
	pAH->sendResponse(Chain("LockInfo"), 
		      pLockInfo);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	pAH->sendError(msg);	
    }
}

void CegoAdminThread::srvLockStat(CegoAdminHandler *pAH)
{
    try
    {	
	Element *pLockStat = new Element(XML_LOCKSTAT_ELEMENT);
	Element *pLockEntry;
	
	Chain lockGroup;
	Chain lockName;
	int numLock;
	int lockCount;
	unsigned long long numRdLock;
	unsigned long long numWrLock;
	unsigned long long sumRdDelay;
	unsigned long long sumWrDelay;
	
	_pPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));    
	pLockStat->addContent(pLockEntry);
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDbPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDbPool->getAggThreadLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));	
	pLockStat->addContent(pLockEntry);    

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pLogPool->getLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getDBMLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getXMLLockStat(lockName, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockName);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(1));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getAggQCLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getAggTCLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);
	
	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	_pDBMng->getAggLMLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);
	
	lockGroup = Chain(RECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(RBRECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(SYSRECLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(SYSPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(DATAPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(IDXPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(RBPAGELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(DATAFILELOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    

	lockGroup = Chain(POOLLOCKGROUP);
	_pTabMng->getLHAggLockStat(lockGroup, numLock, lockCount, numRdLock, numWrLock, sumRdDelay, sumWrDelay);

	pLockEntry = new Element(XML_LOCKENTRY_ELEMENT);
	pLockEntry->setAttribute(Chain(XML_LOCKID_ATTR), lockGroup);
	pLockEntry->setAttribute(Chain(XML_NUMLOCK_ATTR), Chain(numLock));
	pLockEntry->setAttribute(Chain(XML_LOCKCOUNT_ATTR), Chain(lockCount));
	pLockEntry->setAttribute(Chain(XML_RDLOCKHIT_ATTR), Chain(numRdLock));
	pLockEntry->setAttribute(Chain(XML_RDLOCKDELAY_ATTR), Chain(sumRdDelay));
	pLockEntry->setAttribute(Chain(XML_WRLOCKHIT_ATTR), Chain(numWrLock));
	pLockEntry->setAttribute(Chain(XML_WRLOCKDELAY_ATTR), Chain(sumWrDelay));
	pLockStat->addContent(pLockEntry);    
	
	_lastAction = Chain("LockStat");
	pAH->sendResponse(Chain("LockStat"), 
			  pLockStat);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	pAH->sendError(msg);	
    }
}

void CegoAdminThread::srvParameterInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("ParameterInfo");
        
    // get several parameters here
    Chain dbName = _pDBMng->getDbName();
    int pageSize = _pDBMng->getPageSize();
    int queueDelay = _pDBMng->getQueueDelay();
    
    CegoDbHandler::ProtocolType protType = _pDBMng->getProtType();
    
    int numRBRecordSema = _pDBMng->getNumRBRecordSema();
    int numRecordSema = _pDBMng->getNumRecordSema();
    int numSysRecordSema = _pDBMng->getNumSysRecordSema();
    int numSysPageSema = _pDBMng->getNumSysPageSema();
    int numDataPageSema = _pDBMng->getNumDataPageSema();
    int numIndexPageSema = _pDBMng->getNumIndexPageSema();
    int numRBPageSema = _pDBMng->getNumRBPageSema();
    int numDataFileSema = _pDBMng->getNumDataFileSema();
    int numBufferPoolSema = _pDBMng->getNumBufferPoolSema();
    int maxFixTries = _pDBMng->getMaxFixTries();
    int maxPageDelete = _pDBMng->getMaxPageDelete();

    int pageLockTimeout = _pDBMng->getPageLockTimeout();
    int recLockTimeout = _pDBMng->getRecLockTimeout();
    int fileLockTimeout = _pDBMng->getFileLockTimeout();
    int poolLockTimeout = _pDBMng->getPoolLockTimeout();
    int numLockTries = _pDBMng->getNumLockTries();

    int numQueryLast = _pDBMng->getNumQueryLast();
    int numQueryCost = _pDBMng->getNumQueryCost();

    int maxSendLen = _pDBMng->getMaxSendLen();
    
    Chain dateTimeFormat = __dateTimeFormat; // _pDBMng->getDateTimeFormat();
    Chain archRestoreProg = _pDBMng->getArchRestoreProg();    
    int archRestoreTimeout = _pDBMng->getArchRestoreTimeout();    
    Chain backupProg = _pDBMng->getBackupProg();
    
    Element *pParameterList = new Element(XML_PARAMETERLIST_ELEMENT);

    Element *pParameterInfo;

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_DBNAME_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, dbName);
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_PROTOCOL_ATTR);
    if ( protType == CegoDbHandler::XML )	
	pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain("xml"));
    else if ( protType == CegoDbHandler::SERIAL )
	pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain("serial"));
    else if ( protType == CegoDbHandler::FASTSERIAL )
	pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain("fastserial"));
    
    pParameterList->addContent(pParameterInfo);
    
    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_PAGESIZE_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(pageSize));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_QUEUEDELAY_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(queueDelay));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMRECSEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numRecordSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMRBRECSEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numRBRecordSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMSYSRECSEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numSysRecordSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMSYSPAGESEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numSysPageSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMDATAPAGESEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numDataPageSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMIDXPAGESEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numIndexPageSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMRBPAGESEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numRBPageSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMDATAFILESEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numDataFileSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMBUFFERPOOLSEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numBufferPoolSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMBUFFERPOOLSEMA_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numDataFileSema));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_MAXFIXTRIES_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(maxFixTries));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_MAXPAGEDELETE_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(maxPageDelete));
    pParameterList->addContent(pParameterInfo);    

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_PAGELOCKTIMEOUT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(pageLockTimeout));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_RECLOCKTIMEOUT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(recLockTimeout));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_POOLLOCKTIMEOUT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(poolLockTimeout));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_FILELOCKTIMEOUT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(fileLockTimeout));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMLOCKTRIES_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numLockTries));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_MAXSENDLEN_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(maxSendLen));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_DATETIMEFORMAT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(dateTimeFormat));
    pParameterList->addContent(pParameterInfo);

    Chain *pScanFormat = __dateFormatList.First();
    while ( pScanFormat )
    {
       pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
       pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_SCANDATETIMEFORMAT_ATTR);
       pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, *pScanFormat);
       pParameterList->addContent(pParameterInfo);
       pScanFormat = __dateFormatList.Next();
    }
    
    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_LOGMNGPROG_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(archRestoreProg));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_LOGMNGTIMEOUT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(archRestoreTimeout));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_BACKUPMNGPROG_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(backupProg));
    pParameterList->addContent(pParameterInfo);

    Chain csMode = Chain(XML_ID_VALUE);
    if ( __caseSensitiveFlag == 1 )
        csMode = Chain(XML_STR_VALUE);
    else if ( __caseSensitiveFlag == 2 )
        csMode = Chain(XML_NONE_VALUE);
    
    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_CSMODE_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, csMode);
    pParameterList->addContent(pParameterInfo);

    Chain qescMode = Chain(XML_OFF_VALUE);
    if ( __quoteEscapeFlag )
    {
	qescMode = Chain(XML_ON_VALUE);
    }

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_QESCMODE_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, qescMode);
    pParameterList->addContent(pParameterInfo);

    Chain dupNull = Chain(XML_OFF_VALUE);
    if ( __allowDuplicateNull )
	dupNull = Chain(XML_ON_VALUE);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_DUPLICATENULL_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, dupNull);
    pParameterList->addContent(pParameterInfo);
    
    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_CURRENCYSYMBOL_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(__currencySymbol));
    pParameterList->addContent(pParameterInfo);
    
    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_DECIMALPOINT_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(__decimalPoint));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMQUERYLAST_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numQueryLast));
    pParameterList->addContent(pParameterInfo);

    pParameterInfo = new Element(XML_PARAMETERINFO_ELEMENT);
    pParameterInfo->setAttribute(XML_PARAMNAME_ATTR, XML_NUMQUERYCOST_ATTR);
    pParameterInfo->setAttribute(XML_PARAMVALUE_ATTR, Chain(numQueryCost));
    pParameterList->addContent(pParameterInfo);
	    
    pAH->sendResponse(Chain("ParameterInfo"), 
		      pParameterList);    
}

void CegoAdminThread::srvPoolInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("PoolInfo");
    
    int pageSize;
    unsigned long long numTotal;
    unsigned long long numUsed;
    unsigned long long numFree;
    unsigned long long numDirty;
    unsigned long long numFixes;
    unsigned long long numPersistent;
    unsigned long long numNoSync;
    unsigned long long numDiskRead;
    unsigned long long numDiskWrite;
    double hitRate;
    double spreadRate;
    unsigned long long readDelay;
    unsigned long long writeDelay;
    unsigned long long curFixCount;
    unsigned long long maxFixCount;
    unsigned long long avgFixTry;
    
    unsigned long long statStart;
    unsigned long long uptime;
    
    _pDBMng->poolInfo(pageSize, numTotal, numUsed, numFree, numDirty, numFixes, numPersistent, numNoSync, numDiskRead, numDiskWrite, hitRate, spreadRate, readDelay, writeDelay, curFixCount, maxFixCount, avgFixTry, statStart, uptime);
    
    Element *pPoolInfo = new Element(XML_POOLINFO_ELEMENT);
    
    // format hitrate
    Chain hrFormated = Chain(hitRate, "%3.2f") + Chain("%");
    Chain srFormated = Chain(spreadRate, "%3.3f");

    pPoolInfo->setAttribute(XML_PAGESIZE_ATTR, Chain(pageSize));
    pPoolInfo->setAttribute(XML_NUMTOTAL_ATTR, Chain(numTotal));
    pPoolInfo->setAttribute(XML_NUMUSED_ATTR, Chain(numUsed));
    pPoolInfo->setAttribute(XML_NUMFREE_ATTR, Chain(numFree));
    pPoolInfo->setAttribute(XML_NUMDIRTY_ATTR, Chain(numDirty));
    pPoolInfo->setAttribute(XML_NUMFIXES_ATTR, Chain(numFixes));
    pPoolInfo->setAttribute(XML_NUMPERSISTENT_ATTR, Chain(numPersistent));
    pPoolInfo->setAttribute(XML_NUMNOSYNC_ATTR, Chain(numNoSync));
    pPoolInfo->setAttribute(XML_NUMDISKREAD_ATTR, Chain(numDiskRead));
    pPoolInfo->setAttribute(XML_NUMDISKWRITE_ATTR, Chain(numDiskWrite));
    pPoolInfo->setAttribute(XML_BPHITRATE_ATTR, hrFormated);
    pPoolInfo->setAttribute(XML_SPREADRATE_ATTR, srFormated);
    pPoolInfo->setAttribute(XML_READDELAY_ATTR, Chain(readDelay));
    pPoolInfo->setAttribute(XML_WRITEDELAY_ATTR, Chain(writeDelay));
    pPoolInfo->setAttribute(XML_CURFIXCOUNT_ATTR, Chain(curFixCount));
    pPoolInfo->setAttribute(XML_MAXFIXCOUNT_ATTR, Chain(maxFixCount));
    pPoolInfo->setAttribute(XML_AVGFIXTRY_ATTR, Chain(avgFixTry));

    Datetime ts(statStart);
    pPoolInfo->setAttribute(XML_STATSTART_ATTR, ts.asChain(BUPMNG_STATDTFORMAT));

    pPoolInfo->setAttribute(XML_UPTIME_ATTR, Chain(uptime));
        
    pAH->sendResponse(Chain("PoolInfo"), 
		      pPoolInfo);    
}

void CegoAdminThread::srvPoolEntryList(CegoAdminHandler *pAH)
{
    _lastAction = Chain("PoolEntryList");
     
    ListT<CegoBufferPoolEntry> entryList;
    _pDBMng->getPoolEntryList(entryList);
    
    CegoBufferPoolEntry *pEntry = entryList.First();

    while ( pEntry )
    {
	Element *pPoolEntryList = new Element(XML_POOLENTRYLIST_ELEMENT);

	int numEntry=0;
	while ( pEntry && numEntry < 20)
	{
	    Element *pEntryElement = new Element(XML_POOLENTRY_ELEMENT);

	    pEntryElement->setAttribute(XML_SEGMENT_ATTR, Chain(pEntry->getSegment()));
	    pEntryElement->setAttribute(XML_POS_ATTR, Chain(pEntry->getPos()));
	    pEntryElement->setAttribute(XML_OCCSTATE_ATTR, Chain(pEntry->getOccState()));
	    
	    if ( pEntry->isDirty() )
		pEntryElement->setAttribute(XML_ISDIRTY_ATTR, Chain("y"));
	    else
		pEntryElement->setAttribute(XML_ISDIRTY_ATTR, Chain("n"));
	    
	    pEntryElement->setAttribute(XML_NUMFIXES_ATTR, Chain(pEntry->getNumFixes()));
	    
	    pEntryElement->setAttribute(XML_TSID_ATTR, Chain(pEntry->getTabSetId()));
	    // pEntryElement->setAttribute(XML_FILEID_ATTR, Chain(pEntry->getFileId()));
	    pEntryElement->setAttribute(XML_PAGEID_ATTR, Chain(pEntry->getPageId()));
	    pEntryElement->setAttribute(XML_FIXSTAT_ATTR, Chain(pEntry->getFixStat()));
	    pEntryElement->setAttribute(XML_NUMUSAGE_ATTR, Chain(pEntry->getNumUsage()));
	    
	    pPoolEntryList->addContent(pEntryElement);
	    pEntry = entryList.Next();
	    numEntry++;
	}
	if ( pAH->syncWithInfo(Chain("local"), Chain("local"), Chain("PoolEntryList"), pPoolEntryList) == false )
	    return;
    }
    pAH->sendResponse(Chain("PoolEntryList"));    
}

void CegoAdminThread::srvThreadInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("ThreadInfo");
		    
    int numDbThread;
    int numAdmThread;
    int numLogThread;
    int actDbThread;
    int actAdmThread;
    int actLogThread;
       
    _pDBMng->getThreadInfo(numDbThread, numAdmThread, numLogThread, actDbThread, actAdmThread, actLogThread);
    
    Element *pDbInfo = new Element(XML_THREADINFO_ELEMENT);	
    
    pDbInfo->setAttribute(XML_NUMDBTHREAD_ATTR, Chain(numDbThread));
    pDbInfo->setAttribute(XML_NUMADMTHREAD_ATTR, Chain(numAdmThread));
    pDbInfo->setAttribute(XML_NUMLOGTHREAD_ATTR, Chain(numLogThread));
    
    pDbInfo->setAttribute(XML_ACTIVEDBTHREAD_ATTR, Chain(actDbThread));
    pDbInfo->setAttribute(XML_ACTIVEADMTHREAD_ATTR, Chain(actAdmThread));
    pDbInfo->setAttribute(XML_ACTIVELOGTHREAD_ATTR, Chain(actLogThread));
    
    pAH->sendResponse(Chain("ThreadInfo"), 
		      pDbInfo);    
}

void CegoAdminThread::srvDbThreadInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("DBThreadInfo");
        
    int poolLimit = _pDbPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoDbThreadPool::ThreadState state;
	unsigned long long numRequest;
	unsigned long long numQueryRequest;
	unsigned long long threadLoad;
	unsigned long long allocatedSortArea;
	Chain lastAction;
	_pDbPool->getThreadInfo(i, numRequest, numQueryRequest, threadLoad, allocatedSortArea, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));	
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_NUMQUERYREQUEST_ATTR, Chain(numQueryRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));
	pN->setAttribute(XML_ALLOCATEDSORT_ATTR, Chain(allocatedSortArea));
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	
	if ( state == CegoDbThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoDbThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoDbThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pThreadInfo->addContent(pN);
    }
    
    pAH->sendResponse(Chain("DBThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvDbThreadHistory(CegoAdminHandler *pAH)
{
    _lastAction = Chain("DBThreadHistory");

    int threadId;
    pAH->getThreadId(threadId);

    ListT<CegoDbThread::QueryEntry> queryHistory;
    _pDbPool->getThreadHistory(threadId, queryHistory);

    Element *pQueryHist = new Element(XML_QUERYHIST_ELEMENT);

    CegoDbThread::QueryEntry *pQE = queryHistory.First();
    while ( pQE )
    {
	Element *pElem = new Element(XML_QUERY_ELEMENT);

	Datetime ts(pQE->getTimestamp());

	pElem->setAttribute(XML_THRIDX_ATTR, Chain(pQE->getIndex()));
	pElem->setAttribute(XML_TIMESTAMP_ATTR, ts.asChain());
	pElem->setAttribute(XML_QUERY_ATTR, pQE->getQuery());
	pElem->setAttribute(XML_COST_ATTR, Chain(pQE->getCost()));

	pQueryHist->addContent(pElem);
	
	pQE = queryHistory.Next();	
    }    
    pAH->sendResponse(Chain("DBThreadHist"), 
		      pQueryHist);
}

void CegoAdminThread::srvQueryHistoryLast(CegoAdminHandler *pAH)
{
    _lastAction = Chain("QueryHistoryLast");
    
    int numLast;
    pAH->getNumLast(numLast);
    
    ListT<CegoDbThread::QueryEntry> queryList;
    _pDbPool->getQueryHistoryLast(numLast, queryList);
    
    Element *pQueryHist = new Element(XML_QUERYHIST_ELEMENT);
    
    CegoDbThread::QueryEntry *pQE = queryList.First();
    while ( pQE )
    {
	Element *pElem = new Element(XML_QUERY_ELEMENT);
	
	Datetime ts(pQE->getTimestamp());
	
	pElem->setAttribute(XML_THRIDX_ATTR, Chain(pQE->getIndex()));
	pElem->setAttribute(XML_TIMESTAMP_ATTR, ts.asChain());
	pElem->setAttribute(XML_QUERY_ATTR, pQE->getQuery());
	pElem->setAttribute(XML_COST_ATTR, Chain(pQE->getCost()));
	
	pQueryHist->addContent(pElem);
	
	pQE = queryList.Next();	
    }    
    pAH->sendResponse(Chain("QueryHistLast"), 
		      pQueryHist);
}

void CegoAdminThread::srvQueryHistoryCost(CegoAdminHandler *pAH)
{
    _lastAction = Chain("QueryHistoryCost");

    int numCost;
    pAH->getNumCost(numCost);

    ListT<CegoDbThread::QueryEntry> queryList;
    _pDbPool->getQueryHistoryCost(numCost, queryList);

    Element *pQueryHist = new Element(XML_QUERYHIST_ELEMENT);

    CegoDbThread::QueryEntry *pQE = queryList.First();
    while ( pQE )
    {
	Element *pElem = new Element(XML_QUERY_ELEMENT);

	Datetime ts(pQE->getTimestamp());

	pElem->setAttribute(XML_THRIDX_ATTR, Chain(pQE->getIndex()));
	pElem->setAttribute(XML_TIMESTAMP_ATTR, ts.asChain());
	pElem->setAttribute(XML_QUERY_ATTR, pQE->getQuery());
	pElem->setAttribute(XML_COST_ATTR, Chain(pQE->getCost()));
	
	pQueryHist->addContent(pElem);
	
	pQE = queryList.Next();	
    }    
    pAH->sendResponse(Chain("QueryHistCost"), 
		      pQueryHist);
}

void CegoAdminThread::srvNumQueryLast(CegoAdminHandler *pAH)
{
    _lastAction = Chain("NumQueryLast");
    
    int numLast;
    pAH->getNumLast(numLast);

    _pDBMng->setNumQueryLast(numLast);
    pAH->sendResponse(Chain("Num query last set"));
}

void CegoAdminThread::srvNumQueryCost(CegoAdminHandler *pAH)
{
    _lastAction = Chain("NumQueryCost");
    
    int numCost;
    pAH->getNumCost(numCost);
				    
    _pDBMng->setNumQueryCost(numCost);
    
    pAH->sendResponse(Chain("Num query cost set"));
}

void CegoAdminThread::srvAdmThreadInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("AdminThreadInfo");    
    
    int poolLimit = _pPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoAdminThreadPool::ThreadState state;
	unsigned long long numRequest;
	unsigned long long threadLoad;
	Chain lastAction;
	_pPool->getThreadInfo(i, numRequest, threadLoad, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));
	
	if ( state == CegoAdminThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoAdminThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoAdminThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	pThreadInfo->addContent(pN);
    }

    pAH->sendResponse(Chain("AdminThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvLogThreadInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("LogThreadInfo");    

    int poolLimit = _pLogPool->getPoolLimit();
    
    Element *pThreadInfo = new Element(XML_THREADINFO_ELEMENT);
    
    for ( int i = 0 ; i < poolLimit ; i++ )
    {
	CegoLogThreadPool::ThreadState state;
	unsigned long long numRequest;
	unsigned long long threadLoad;
	Chain lastAction;
	_pLogPool->getThreadInfo(i, numRequest, threadLoad, state, lastAction);
	
	Element *pN = new Element(XML_THREAD_ELEMENT);
	pN->setAttribute(XML_THID_ATTR, Chain(i));
	pN->setAttribute(XML_NUMREQUEST_ATTR, Chain(numRequest));
	pN->setAttribute(XML_THREADLOAD_ATTR, Chain(threadLoad));

	if ( state == CegoLogThreadPool::READY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_READY_VALUE);
	else if ( state == CegoLogThreadPool::CONNECTED )
	    pN->setAttribute(XML_STATUS_ATTR, XML_CONNECTED_VALUE);
	else if ( state == CegoLogThreadPool::BUSY )
	    pN->setAttribute(XML_STATUS_ATTR, XML_BUSY_VALUE);
	
	pN->setAttribute(XML_LASTACTION_ATTR, lastAction);
	pThreadInfo->addContent(pN);
    }
    
    pAH->sendResponse(Chain("LogThreadinfo"), 
		      pThreadInfo);
}

void CegoAdminThread::srvDbThreadAbort(CegoAdminHandler *pAH)
{
    int threadId;
    pAH->getThreadId(threadId);

    _lastAction = Chain("ThreadAbort");
    _pDbPool->abortThread(threadId);
    pAH->sendResponse(Chain("Thread aborted"));		
}	

void CegoAdminThread::srvDbSessionInfo(CegoAdminHandler *pAH)
{
    _lastAction = Chain("DbSessionInfo");    

    Element *pSessionInfo = _pDBMng->getSessionInfo(NETMNG_DBHANDLE_TTL);
    
    pAH->sendResponse(Chain("DbSessionInfo"), 
		      pSessionInfo);
}

void CegoAdminThread::srvCopyInfo(CegoAdminHandler *pAH)
{        
    Element *pCopyInfo = _pDBMng->getCopyInfo();
    
    pAH->sendResponse(Chain("CopyInfo"), 
		      pCopyInfo);

    _lastAction = Chain("CopyInfo");    
}

void CegoAdminThread::srvShowQueryCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);   

    Element *pCacheInfo = 0;

    CegoQueryCache* pCache = _pDBMng->getQueryCache(tableSet);

    if ( pCache )
    {
	pCacheInfo = pCache->getCacheInfo();
	pAH->sendResponse(Chain("QueryCache Info"), pCacheInfo); 
    }
    else
    {
	pAH->sendResponse(Chain("QueryCache not available")); 
    }

    _lastAction = Chain("ShowQueryCache");    
}

void CegoAdminThread::srvListQueryCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);   

    Element *pCacheList = 0;

    CegoQueryCache* pCache = _pDBMng->getQueryCache(tableSet);

    if ( pCache )
    {
	pCacheList = pCache->getCacheList();
	pAH->sendResponse(Chain("QueryCache List"), pCacheList); 
    }
    else
    {
	pAH->sendResponse(Chain("QueryCache not available")); 
    }

    _lastAction = Chain("ListQueryCache");    
}

void CegoAdminThread::srvCleanQueryCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);   

    CegoQueryCache* pCache = _pDBMng->getQueryCache(tableSet);

    if ( pCache )
    {
	pCache->clean();
    }

    pAH->sendResponse(Chain("Query Cache cleaned"));		

    _lastAction = Chain("CleanQueryCache");    
}

void CegoAdminThread::srvSetQueryCacheParam(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    int maxEntry;
    pAH->getQueryCacheMaxEntry(maxEntry);

    int maxSize;
    pAH->getQueryCacheMaxSize(maxSize);

    int threshold;
    pAH->getQueryCacheThreshold(threshold);

    int hashRange;
    pAH->getQueryCacheHashRange(hashRange);

    if ( maxEntry > 0 )
	_pDBMng->setMaxQueryCacheEntry(tableSet, maxEntry);
    if ( maxSize > 0 )
	_pDBMng->setMaxQueryCacheSize(tableSet, maxSize);
    if ( hashRange > 0 )
	_pDBMng->setQueryCacheHashRange(tableSet, hashRange);
    if ( threshold != -1 )
	_pDBMng->setQueryCacheThreshold(tableSet, threshold);
    
    pAH->sendResponse(Chain("QueryCache params set"));		

    _lastAction = Chain("SetQueryCacheParam");       
}

void CegoAdminThread::srvShowTableCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);   

    Element *pCacheInfo = 0;

    CegoTableCache* pCache = _pDBMng->getTableCache(tableSet);

    if ( pCache )
    {
	pCacheInfo = pCache->getCacheInfo();
	pAH->sendResponse(Chain("TableCache Info"), pCacheInfo); 
    }
    else
    {
	pAH->sendResponse(Chain("TableCache not available")); 
    }

    _lastAction = Chain("ShowTableCache");    
}

void CegoAdminThread::srvListTableCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Element *pCacheList = 0;

    CegoTableCache* pCache = _pDBMng->getTableCache(tableSet);

    if ( pCache )
    {
	pCacheList = pCache->getCacheList();
	pAH->sendResponse(Chain("TableCache List"), pCacheList); 
    }
    else
    {
	pAH->sendResponse(Chain("TableCache not enabled")); 
    }
    
    _lastAction = Chain("ListTableCache");    
}

void CegoAdminThread::srvCleanTableCache(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    CegoTableCache* pCache = _pDBMng->getTableCache(tableSet);

    if ( pCache )
    {
	pCache->clean();
    }

    pAH->sendResponse(Chain("TableCache cleaned"));		
    _lastAction = Chain("CleanTableCache"); 
}

void CegoAdminThread::srvSetTableCacheParam(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain cacheFilter;
    pAH->getTableCacheFilter(cacheFilter);
    
    int maxEntry;
    pAH->getTableCacheMaxEntry(maxEntry);

    int maxSize;
    pAH->getTableCacheMaxSize(maxSize);

    int hashRange;
    pAH->getTableCacheHashRange(hashRange);

    if ( cacheFilter != Chain() )
	_pDBMng->setTableCacheFilter(tableSet, cacheFilter);
    if ( maxEntry > 0  )
	_pDBMng->setTableCacheMaxEntry(tableSet, maxEntry);
    if ( maxSize > 0 ) 
	_pDBMng->setTableCacheMaxSize(tableSet, maxSize);
    if ( hashRange > 0 )
	_pDBMng->setTableCacheHashRange(tableSet, hashRange);
    
    pAH->sendResponse(Chain("TableCache params set"));		

    _lastAction = Chain("SetTableCacheParam");
}

void CegoAdminThread::srvDefineTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    Chain tsRoot;

    Chain primary;
    Chain secondary;
    Chain mediator;

    int sysFid;
    int tmpFid;
    
    int sysSize;
    int tmpSize;
    int appSize;

    int logFileSize;
    int logFileNum;

    unsigned long long sortAreaSize;
    int branchId;
    
    pAH->getTableSet(tableSet);
    pAH->getTableSetRoot(tsRoot);

    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getMediator(mediator);

    pAH->getSysFileId(sysFid);
    pAH->getTmpFileId(tmpFid);

    pAH->getSysSize(sysSize);
    pAH->getTmpSize(tmpSize);
    pAH->getAppSize(appSize);

    pAH->getLogFileSize(logFileSize);
    pAH->getLogFileNum(logFileNum);
    pAH->getBranchId(branchId);

    pAH->getSortAreaSize(sortAreaSize);

    _lastAction = Chain("Define TableSet") + tableSet;
    
    _pDBMng->addTableSetDef(tableSet, tsRoot, primary, secondary, mediator, sysFid, tmpFid, sysSize, tmpSize, appSize,
			    logFileSize, logFileNum, sortAreaSize, branchId); 
    
    pAH->sendResponse(Chain("Table set defined"));
}

void CegoAdminThread::srvRemoveTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _pDBMng->rmTableSetDef(tableSet);
    _lastAction = Chain("Remove TableSet ") + tableSet;    
    pAH->sendResponse(Chain("Table set removed"));
}

void CegoAdminThread::srvResetTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _pTabMng->resetTableSet(tableSet);
    _lastAction = Chain("Reset TableSet ") + tableSet;    
    pAH->sendResponse(Chain("Table set reset"));
}

void CegoAdminThread::srvCreateTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Create TableSet ") + tableSet;
    _pTabMng->createTableSet(tableSet);
    pAH->sendResponse(Chain("Table set created"));
}

void CegoAdminThread::srvDropTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Drop TableSet ") + tableSet;    
    _pTabMng->dropTableSet(tableSet);
    pAH->sendResponse(Chain("Table set dropped"));
}

void CegoAdminThread::srvStartTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;   
    pAH->getTableSet(tableSet);
    bool cleanIt = pAH->getCleanup();
    bool forceload = pAH->getForceload();
    bool cpDump = pAH->getCPDump();
    
    Chain secondary = _pDBMng->getSecondary(tableSet);

    _lastAction = Chain("Start TableSet ") + tableSet;    

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, true);
    try
    {
	_pTabMng->startDistTableSet(tableSet, secondary, cleanIt, true, true, cpDump);    
	if ( forceload )
	{
	    // int tabSetId = _pDBMng->getTabSetId(tableSet);
	    _pDbPool->loadObjects(tabSetId);
	}
	pAH->sendResponse(Chain("Tableset started"));	
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot start tableset"), e);
    }
    _pDBMng->unlockPool(tabSetId);
}

void CegoAdminThread::srvStopTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Stop TableSet ") + tableSet;    

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, true);
    try
    {
	_pDbPool->unloadObjects(tabSetId);	
	_pTabMng->stopTableSet(tableSet, true);
	pAH->sendResponse(Chain("Tableset stopped"));
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot stop tableset"), e);
    }
    _pDBMng->unlockPool(tabSetId);	
}

void CegoAdminThread::srvVerifyTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Verify TableSet ") + tableSet;

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, false);
    try
    {
	_pTabMng->setActiveUser(tableSet, _user, _password);
	
	ListT<Chain> tableList; 
	_pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);
	
	Chain dbHost;
	_pDBMng->getDBHost(dbHost);
	
	Chain *pTable = tableList.First();
	while ( pTable )
	{
	    Element* pVerificationInfo = _pTabMng->verifyTable(tableSet, *pTable);
	    if ( pAH->syncWithInfo(Chain("primary"), dbHost, Chain("Verification of table ") + *pTable, pVerificationInfo) == false)
		return;
	    
	    pTable = tableList.Next();		
	}
	
	ListT<Chain> viewList; 
	_pTabMng->getDistObjectList(tableSet, CegoObject::VIEW, viewList);
	
	Chain *pView = tableList.First();
	while ( pView )
	{
	    Element* pVerificationInfo = _pTabMng->verifyView(tableSet, *pView);
	    if ( pAH->syncWithInfo(Chain("primary"), dbHost, Chain("Verification of view ") + *pView, pVerificationInfo) == false)
		return;
	    
	    pView = viewList.Next();		
	}
	
	ListT<Chain> procList; 
	_pTabMng->getDistObjectList(tableSet, CegoObject::PROCEDURE, procList);
	
	Chain *pProc = procList.First();
	while ( pProc )
	{
	    Element* pVerificationInfo = _pTabMng->verifyProcedure(tableSet, *pProc);
	    if ( pAH->syncWithInfo(Chain("primary"), dbHost, Chain("Verification of procedure ") + *pProc, pVerificationInfo) == false)
		return;
	    
	    pProc = procList.Next();		
	}
	pAH->sendResponse(Chain("Tableset verified"));
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot verify tableset"), e);
    }
    _pDBMng->unlockPool(tabSetId);   
}
    
void CegoAdminThread::srvCorrectTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);   
    _lastAction = Chain("Correct TableSet ") + tableSet;

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, false);
    try
    {
	_pTabMng->setActiveUser(tableSet, _user, _password);
	
	ListT<Chain> tableList; 
	_pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);
	
	Chain dbHost;
	_pDBMng->getDBHost(dbHost);
	
	Chain *pTable = tableList.First();
	while ( pTable )
	{
	    Element* pVerificationInfo = _pTabMng->correctTable(tableSet, *pTable);
	    if ( pAH->syncWithInfo(Chain("primary"), dbHost, Chain("Correction of table ") + *pTable,
				   pVerificationInfo) == false)
		return;
	    
	    pTable = tableList.Next();		
	}
	pAH->sendResponse(Chain("Tableset corrected"));
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot correct tableset"), e);
    }
    _pDBMng->unlockPool(tabSetId);
}

void CegoAdminThread::srvDumpObject(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    Chain objName;
    CegoObject::ObjectType objType;
    pAH->getTableSet(tableSet);
    pAH->getObjName(objName);
    pAH->getObjType(objType);
    
    _lastAction = Chain("Dump object ") + objName;    

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    Chain objDump;
    _pTabMng->dumpObject(tabSetId, objName, objType, objDump);

    cout << "Objectdump=" << objDump << endl;

    pAH->sendData(objDump);
}

void CegoAdminThread::srvSetLogMng(CegoAdminHandler *pAH)
{   
    Chain progName;
    int timeout;
    pAH->getLogManager(progName);
    pAH->getLogManagerTimeout(timeout);
    
    _lastAction = Chain("set log manager");    

    _pDBMng->setArchRestoreProg(progName);
    _pDBMng->setArchRestoreTimeout(timeout);

    pAH->sendResponse(Chain("Log manager set"));
}

void CegoAdminThread::srvShowLogMng(CegoAdminHandler *pAH)
{   
    _lastAction = Chain("show log manager");    

    Chain progName = _pDBMng->getArchRestoreProg();
    int timeout = _pDBMng->getArchRestoreTimeout();
    
    Element *pLMI = new Element(XML_LOGMNGINFO_ELEMENT);
    pLMI->setAttribute(XML_LOGMNGPROG_ATTR, progName);
    pLMI->setAttribute(XML_LOGMNGTIMEOUT_ATTR, Chain(timeout));
    
    pAH->sendResponse(Chain("Log manager Info"), pLMI);
}

void CegoAdminThread::srvSetBackupMng(CegoAdminHandler *pAH)
{   
    Chain progName;
    pAH->getBackupManager(progName);
    
    _lastAction = Chain("set backup manager");    

    _pDBMng->setBackupProg(progName);

    pAH->sendResponse(Chain("Backup manager set"));
}

void CegoAdminThread::srvShowBackupMng(CegoAdminHandler *pAH)
{   
    _lastAction = Chain("show backup manager");    

    Chain progName = _pDBMng->getBackupProg();
    
    Element *pLMI = new Element(XML_BACKUPMNGINFO_ELEMENT);
    pLMI->setAttribute(XML_BACKUPMNGPROG_ATTR, progName);
    
    pAH->sendResponse(Chain("Backup manager Info"), pLMI);
}

void CegoAdminThread::srvSetCheckpoint(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int tmout;
    pAH->getTimeoutValue(tmout);
    
    _lastAction = Chain("Set checkpoint for ") + tableSet;    

    _pDBMng->setCheckpointInterval(tableSet, tmout);
    
    pAH->sendResponse(Chain("Checkpoint interval set"));    
}

void CegoAdminThread::srvSetTSInitFile(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain initFile;
    pAH->getTSInitFile(initFile);
    
    _lastAction = Chain("Set initfile for ") + tableSet;    

    _pDBMng->setTSInitFile(tableSet, initFile);
    
    pAH->sendResponse(Chain("Initfile set"));
}

void CegoAdminThread::srvSetTSSysSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int sysSize;
    pAH->getSysSize(sysSize);
    
    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }

    _lastAction = Chain("Set system size for ") + tableSet;
    
    _pDBMng->setTSSysSize(tableSet, sysSize);
    
    pAH->sendResponse(Chain("System size set"));    
}

void CegoAdminThread::srvSetTSTmpSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int tmpSize;
    pAH->getTmpSize(tmpSize);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set temp size for ") + tableSet;    

    _pDBMng->setTSTmpSize(tableSet, tmpSize);
    
    pAH->sendResponse(Chain("Temp size set"));
}

void CegoAdminThread::srvSetTSAppSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int appSize;
    pAH->getAppSize(appSize);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set app size for ") + tableSet;    

    _pDBMng->setTSAppSize(tableSet, appSize);
    
    pAH->sendResponse(Chain("App size set"));
}

void CegoAdminThread::srvSetTSRootPath(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain tsRoot;
    pAH->getTSRootPath(tsRoot);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
        
    _lastAction = Chain("Set tsroot for ") + tableSet;

    _pDBMng->resetTSRoot(tableSet, tsRoot);

    pAH->sendResponse(Chain("TSRoot set"));    
}

void CegoAdminThread::srvSetTSLogNum(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int logNum;
    pAH->getTSLogNum(logNum);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set logNum for ") + tableSet;    

    _pDBMng->setTSLogNum(tableSet, logNum);
    
    pAH->sendResponse(Chain("Log number set"));
}

void CegoAdminThread::srvSetTSLogSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    int logSize;
    pAH->getTSLogSize(logSize);

    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be in state DEFINED for reconfiguration");
    }
    
    _lastAction = Chain("Set logSize for ") + tableSet;    

    _pDBMng->setTSLogSize(tableSet, logSize);
    
    pAH->sendResponse(Chain("Log size set"));
}

void CegoAdminThread::srvSetTSSortAreaSize(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    unsigned long long sortAreaSize;
    pAH->getTSSortAreaSize(sortAreaSize);
    
    _lastAction = Chain("Set sort area size for ") + tableSet;    

    _pDBMng->setTSSortAreaSize(tableSet, sortAreaSize);
    
    pAH->sendResponse(Chain("Sort area size set"));
}

void CegoAdminThread::srvSetTSLogUser(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain logUser;
    pAH->getTSLogUser(logUser);

    _lastAction = Chain("Set loguser for ") + tableSet;

    _pDBMng->setTSLogUser(tableSet, logUser);

    pAH->sendResponse(Chain("Loguser set"));    
}

void CegoAdminThread::srvStartRecovery(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Start Recovery for ") + tableSet;    
    _pDBMng->startRecovery(tableSet);
   
    pAH->sendResponse(Chain("Starting recovery ..."));
}

void CegoAdminThread::srvStopRecovery(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);

    _lastAction = Chain("Stop Recovery for ") + tableSet;    
	
    if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
    {
	_pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
	
	while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
	{
	    Sleeper s;
	    s.secSleep(LOGMNG_RECOVERY_DELAY);
	}
    }

    pAH->sendResponse(Chain("Stopped recovery"));
}      	

void CegoAdminThread::srvGetDbSpec(CegoAdminHandler *pAH)
{
    pAH->sendResponse(Chain("Copy of DbSpec complete"), 
		      _pDBMng->getDbSpec());       	
}

void CegoAdminThread::srvGetDbInfo(CegoAdminHandler *pAH)
{
    pAH->sendResponse(Chain(CEGO_PRODUCT) + Chain(" ") + Chain(CEGO_VERSION));
}

void CegoAdminThread::srvCopyTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    Chain secondary;
    pAH->getSecondary(secondary);
    Chain mediator;
    pAH->getMediator(mediator);

    _pDBMng->startCopy(tableSet, secondary, mediator, pAH->getUser(), pAH->getPassword(), Chain("Copy requested"));

    pAH->sendResponse(Chain("Starting asynchronous copy"));
}
       	
void CegoAdminThread::srvCopyFile(CegoAdminHandler *pAH)
{
    Chain fileName;
    pAH->getFileName(fileName);
    int fileSize;
    pAH->getFileSize(fileSize);

    File copyFile(fileName);
    
    copyFile.open(File::WRITE);
    
    NetHandler *pN = pAH->getNetHandle();
    
    Chain msg = Chain("File <") + fileName + Chain("> (") + Chain(fileSize) + Chain(" bytes) expected to receive"); 
    pAH->sendResponse(msg);
    
    int recvLen = 0;
    while ( recvLen < fileSize )
    {	
	pN->readMsg();
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Received ") + Chain(pN->getMsgSize()) + Chain(" bytes"));
#endif	

	copyFile.writeByte(pN->getMsg(), pN->getMsgSize());
	recvLen += pN->getMsgSize();
	
	pN->sendAck();	
    }
    copyFile.close();       	
}

void CegoAdminThread::srvGetNodeList(CegoAdminHandler *pAH)
{
    Element* pNodeInfo = _pDBMng->getNodeList();     
    _lastAction = Chain("GetNodeList");
    pAH->sendResponse(Chain("Node Info"), 
		      pNodeInfo);
}

void CegoAdminThread::srvGetTSInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTSInfo = _pDBMng->getTableSetInfo(tableSet); 
    
    _lastAction = Chain("GetTSInfo");
    pAH->sendResponse(Chain("TS Info"), 
		      pTSInfo);
}

void CegoAdminThread::srvGetDetailedTSInfo(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTSInfo = getDetailedTSInfo(tableSet); 
    
    _lastAction = Chain("GetDetailedTSInfo");
    pAH->sendResponse(Chain("Detailed TS Info"), 
		      pTSInfo);
}

void CegoAdminThread::srvGetArchLogInfo(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pArchInfo = getArchLogInfo(tableSet); 
    
    _lastAction = Chain("GetArchLogInfo");
    pAH->sendResponse(Chain("Arch Log Info"), 
		      pArchInfo);
}

void CegoAdminThread::srvGetTransactionInfo(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pTAInfo = getTransactionInfo(tableSet); 
    
    _lastAction = Chain("GetTransactionInfo");
    pAH->sendResponse(Chain("Transaction Info"), 
		      pTAInfo);
}

void CegoAdminThread::srvGetBUStatInfo(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Element* pBUSInfo = getBUStatInfo(tableSet); 
    
    _lastAction = Chain("GetBUStatInfo");
    pAH->sendResponse(Chain("BUStat Info"), 
		      pBUSInfo);
}

void CegoAdminThread::srvPropTSInfo(CegoAdminHandler *pAH)
{    
    Element *pTSInfo = pAH->getTSInfo();

    if ( pTSInfo )
    {
	Chain tableSet = pTSInfo->getAttributeValue(XML_NAME_ATTR);
	_pDBMng->setTableSetInfo(tableSet, pTSInfo->createClone()); 
    }
    pAH->sendResponse(Chain("Propagate TS Info"));    
}

void CegoAdminThread::srvSetTableSetNode(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain primary;
    Chain secondary;
    Chain mediator;
    
    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getMediator(mediator);
    
    _pDBMng->setPrimary(tableSet, primary);
    _pDBMng->setSecondary(tableSet, secondary);
    _pDBMng->setMediator(tableSet, mediator);
    
    pAH->sendResponse(Chain("Tableset nodes set"));
}

void CegoAdminThread::srvSetLSN(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    unsigned long long lsn;
    pAH->getLSN(lsn);
    
    _pDBMng->setTableSetRunState(tableSet, XML_OFFLINE_VALUE);
    _pDBMng->setCommittedLSN(tableSet, lsn);
    
    pAH->sendResponse(Chain("Tableset LSN set"));
}

void CegoAdminThread::srvSetRunState(CegoAdminHandler *pAH)
{    
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain runState;
    pAH->getRunState(runState);
    
    _pDBMng->setTableSetRunState(tableSet, runState);
    
    pAH->sendResponse(Chain("Tableset runstate set"));
}

void CegoAdminThread::srvSetSyncState(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    pAH->getTableSet(tableSet);
    
    Chain syncState;
    pAH->getSyncState(syncState);
    
    _pDBMng->setTableSetSyncState(tableSet, syncState);
    
    pAH->sendResponse(Chain("Tableset syncstate set"));
}

void CegoAdminThread::srvExportTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain expFile;
    bool isStructure;
    Chain expMode;

    pAH->getTableSet(tableSet);
    pAH->getFileName(expFile);
    pAH->getIsStructure(isStructure);
    pAH->getMode(expMode);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    // to avoid lock timeouts during tableset export, we avoid pool locking
    // seems to be not needful
    
    try
    {    
	Chain status = _pDBMng->getTableSetRunState(tableSet);
	if ( status != Chain(XML_ONLINE_VALUE) )
	{
	    throw Exception(EXLOC, "Tableset must be online for export");
	}
	
	_lastAction = Chain("Export tableset ") + tableSet;    
        
	_pTabMng->setActiveUser(tableSet, _user, _password);
	CegoXPorter exp(_pTabMng, pAH);
	
	if ( expMode == Chain(XML_XML_VALUE) )
	{
	    exp.xmlExportTableSet(tableSet, isStructure, expFile);
	}
	else if ( expMode == Chain(XML_BIN_VALUE) )
	{
	    exp.binExportTableSet(tableSet, isStructure, expFile, false);
	}
	else if ( expMode == Chain(XML_PLAIN_VALUE) )
	{
	    exp.binExportTableSet(tableSet, isStructure, expFile, true);
	}
	pAH->sendResponse(Chain("Tableset " + tableSet + " exported"));

    }
    catch ( Exception e )
    {
	throw Exception(EXLOC, Chain("Cannot export tableset"), e);
    }
}

void CegoAdminThread::srvImportTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain impFile;
    bool isStructure;
    Chain impMode;

    pAH->getTableSet(tableSet);
    pAH->getFileName(impFile);
    pAH->getIsStructure(isStructure);
    pAH->getMode(impMode);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    // during import we stop any log activies
    _pDBMng->stopLog(tabSetId);

    // and we lock bufffer pool for corresponding tableset
    _pDBMng->lockPool(tabSetId, false);

    try
    {    
	Chain status = _pDBMng->getTableSetRunState(tableSet);
	if ( status != Chain(XML_ONLINE_VALUE) )
	{
	    throw Exception(EXLOC, "Tableset must be online for import");
	}
	
	_lastAction = Chain("Import tableset ") + tableSet;    
	
	_pTabMng->setActiveUser(tableSet, _user, _password);    
	CegoXPorter imp(_pTabMng, pAH);
	
	if ( impMode == Chain(XML_XML_VALUE) )
	{
	    imp.xmlImportTableSet(tableSet, isStructure, impFile);
	}
	else if ( impMode == Chain(XML_BIN_VALUE) )
	{
	    imp.binImportTableSet(tableSet, isStructure, impFile, false);
	}
	else if ( impMode == Chain(XML_PLAIN_VALUE) )
	{
	    imp.binImportTableSet(tableSet, isStructure, impFile, true);
	}
	
	pAH->sendResponse(Chain("Tableset " + tableSet + " imported"));
    }    
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	_pDBMng->startLog(tabSetId);
	throw Exception(EXLOC, Chain("Cannot import tableset"), e);
    }

    _pDBMng->unlockPool(tabSetId);
    _pDBMng->startLog(tabSetId);  	

    // the commit import actions, we write checkpoint here
    _pDBMng->writeCheckPoint(tableSet, true, false, _pTabMng->getLockHandle());
}

void CegoAdminThread::srvExportTable(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain tableName;
    Chain expFile;
    Chain expMode;

    pAH->getTableSet(tableSet);
    pAH->getTableName(tableName);
    pAH->getFileName(expFile);
    pAH->getMode(expMode);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);

    // same as for whole tableset, no locking here

    try
    {    
	Chain status = _pDBMng->getTableSetRunState(tableSet);
	if ( status != Chain(XML_ONLINE_VALUE) )
	{
	    throw Exception(EXLOC, "Tableset must be online for export");
	}
	
	_pTabMng->setActiveUser(tableSet, _user, _password);
	
	unsigned long long numExported = 0;
	
	CegoXPorter exp(_pTabMng, pAH);
	if ( expMode == Chain(XML_XML_VALUE) )
	{
	    numExported = exp.xmlExportTable(tableSet, tableName, expFile);
	}
	else if ( expMode == Chain(XML_BIN_VALUE) )
	{
	    numExported = exp.binExportTable(tableSet, tableName, expFile, false);
	}
	else if ( expMode == Chain(XML_PLAIN_VALUE) )
	{
	    numExported = exp.binExportTable(tableSet, tableName, expFile, true);
	}
	
	pAH->sendResponse(Chain("Table " + tableName + " exported with " + Chain(numExported) + " rows"));
    }
    catch ( Exception e )
    {
	throw Exception(EXLOC, Chain("Cannot export table"), e);
    }
}
    
void CegoAdminThread::srvImportTable(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain tableName;
    Chain impFile;
    Chain impMode;

    pAH->getTableSet(tableSet);
    pAH->getTableName(tableName);
    pAH->getFileName(impFile);
    pAH->getMode(impMode);

    int tabSetId = _pDBMng->getTabSetId(tableSet);
   
    // during import we stop any log activies
    _pDBMng->stopLog(tabSetId);

    _pDBMng->lockPool(tabSetId, false);

    try
    {    
	Chain status = _pDBMng->getTableSetRunState(tableSet);
	if ( status != Chain(XML_ONLINE_VALUE) )
	{
	    throw Exception(EXLOC, "Tableset must be online for import");
	}

	_pTabMng->setActiveUser(tableSet, _user, _password);
	
	CegoXPorter imp(_pTabMng, pAH);
	
	unsigned long long numImported = 0;
	
	if ( impMode == Chain(XML_XML_VALUE) )
	{
	    numImported = imp.xmlImportTable(tableSet, tableName, impFile);
	}
	else if ( impMode == Chain(XML_BIN_VALUE) )
	{
	    numImported = imp.binImportTable(tableSet, tableName, impFile, false);
	}
	else if ( impMode == Chain(XML_PLAIN_VALUE) )
	{
	    numImported = imp.binImportTable(tableSet, tableName, impFile, true);
	}
	
	if ( numImported > 0 )
	    pAH->sendResponse(Chain("Table " + tableName + " imported with " + Chain(numImported) + " rows"));
	else
	    pAH->sendResponse(Chain("No rows available for table " + tableName));
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot import tableset"), e);
    }

    _pDBMng->unlockPool(tabSetId);
    _pDBMng->startLog(tabSetId);  	

    // the commit import actions, we write checkpoint here
    _pDBMng->writeCheckPoint(tableSet, true, false, _pTabMng->getLockHandle());
}
    
void CegoAdminThread::srvAddDatafile(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain fileType;
    Chain dataFile;
    int fileSize;
    int fileId;
    
    pAH->getTableSet(tableSet);
    pAH->getFileType(fileType);
    pAH->getFileId(fileId);
    pAH->getFileName(dataFile);
    pAH->getFileSize(fileSize);

    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, false);
    try
    {    
	_pTabMng->addDataFile(tableSet, fileType, fileId,  dataFile, fileSize);
	_pDBMng->writeCheckPoint(tableSet, true, false, _pTabMng->getLockHandle());
	
	try
	{
	    _pDBMng->addDataFile(tableSet, fileType, fileId, dataFile, fileSize); 	    
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);		
	}
	pAH->sendResponse(Chain("Data file " + dataFile + " added"));   
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot add datafile"), e);
    }
    _pDBMng->unlockPool(tabSetId);
}
    
void CegoAdminThread::srvUserInfo(CegoAdminHandler *pAH)
{
    Element* pUserInfo = _pDBMng->getUserList(); 
    
    _lastAction = Chain("showUser");
    pAH->sendResponse(Chain("User Info"), pUserInfo); 
}

void CegoAdminThread::srvAddUser(CegoAdminHandler *pAH)
{
    Chain user;
    Chain password;
    pAH->getUser(user);
    pAH->getPassword(password);

    _pDBMng->addUser(user, password);
    
    pAH->sendResponse(Chain("User " + user + " added"));
}	

void CegoAdminThread::srvRemoveUser(CegoAdminHandler *pAH)
{
    Chain user;
    pAH->getUser(user);    
    _pDBMng->removeUser(user);
    pAH->sendResponse(Chain("User " + user + " removed"));
}
	
void CegoAdminThread::srvChangePwd(CegoAdminHandler *pAH)
{
    Chain user;
    Chain password;

    pAH->getUser(user);
    pAH->getPassword(password);
    _pDBMng->changePassword(user, password);
    pAH->sendResponse(Chain("Password changed for user " + user));		    
}

void CegoAdminThread::srvListRole(CegoAdminHandler *pAH)
{
    Element* pRoleList = _pDBMng->getRoleList();     
    _lastAction = Chain("listRole");
    pAH->sendResponse(Chain("Role List"), pRoleList);
}

void CegoAdminThread::srvShowRole(CegoAdminHandler *pAH)
{
    Chain role;

    pAH->getRole(role);
    Element* pPermInfo = _pDBMng->getPermInfo(role);
    pAH->sendResponse(Chain("Role info"), pPermInfo);
}

void CegoAdminThread::srvCreateRole(CegoAdminHandler *pAH)
{    
    Chain role;

    pAH->getRole(role);
    _pDBMng->createRole(role);
    pAH->sendResponse(Chain("Role " ) + role + Chain(" created"));
}

void CegoAdminThread::srvDropRole(CegoAdminHandler *pAH)
{    
    Chain role;    

    pAH->getRole(role);
    _pDBMng->dropRole(role);
    pAH->sendResponse(Chain("Role ") + role + Chain(" dropped"));
}

void CegoAdminThread::srvAssignRole(CegoAdminHandler *pAH)
{
    Chain user;
    Chain role;    

    pAH->getUser(user);
    pAH->getRole(role);
    _pDBMng->assignUserRole(user, role);
    pAH->sendResponse(Chain("Role set for user " + user));		    
}

void CegoAdminThread::srvRemoveRole(CegoAdminHandler *pAH)
{
    Chain user;
    Chain role;

    pAH->getUser(user);
    pAH->getRole(role);   
    _pDBMng->removeUserRole(user, role);
    pAH->sendResponse(Chain("Role ") + role + Chain(" removed from user ") + user);
}

void CegoAdminThread::srvSetPerm(CegoAdminHandler *pAH)
{
    Chain role;
    Chain tableSet;
    Chain filter;
    Chain perm;
    Chain permid;    

    pAH->getRole(role);
    pAH->getTableSet(tableSet);
    pAH->getFilter(filter);
    pAH->getPerm(perm);
    pAH->getPermId(permid);
    _pDBMng->setPerm(role, permid, tableSet, filter, perm);
    pAH->sendResponse(Chain("Permission ") + permid + Chain(" set"));
}

void CegoAdminThread::srvRemovePerm(CegoAdminHandler *pAH)
{
    Chain role;
    Chain permid;

    pAH->getRole(role);
    pAH->getPermId(permid);
    _pDBMng->removePerm(role, permid);
    pAH->sendResponse(Chain("Permission ") + permid + Chain(" removed"));
}

void CegoAdminThread::srvUserTrace(CegoAdminHandler *pAH)
{
    Chain user;
    bool isOn;

    pAH->getUser(user);
    pAH->getTrace(isOn);
    _pDBMng->setUserTrace(user, isOn);
    pAH->sendResponse(Chain("User trace set for ") + user);		    
}

void CegoAdminThread::srvSyncTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    Chain escCmd;
    int timeout;

    pAH->getTableSet(tableSet);
    pAH->getEscapeCmd(escCmd);
    pAH->getTimeoutValue(timeout);
    pAH->getBUMsg(bumsg);    
    int tabSetId = _pDBMng->getTabSetId(tableSet);
    _pDBMng->lockPool(tabSetId, false);
    try
    {    
	_pTabMng->syncDistTableSet(tableSet, bumsg, escCmd, timeout);
	
	Chain msg;
	if ( escCmd == Chain() )
	{
	    msg = Chain("Tableset ") + tableSet + Chain(" in sync");
	}
	else
	{
	    msg = Chain("Tableset ") + tableSet + Chain(" in sync with escape command");
	}
	pAH->sendResponse(msg);
    }
    catch ( Exception e )
    {
	_pDBMng->unlockPool(tabSetId);
	throw Exception(EXLOC, Chain("Cannot synchronize tableset"), e);
    }
    _pDBMng->unlockPool(tabSetId);
}

void CegoAdminThread::srvInitLogs(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);    
    _pDBMng->initLogFiles(tableSet, true);
    pAH->sendResponse(Chain("Logs for tableset " + tableSet + " initialized"));
}

void CegoAdminThread::srvAddArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain archId;
    Chain archPath;

    pAH->getTableSet(tableSet);
    pAH->getArchId(archId);
    pAH->getArchPath(archPath);
    _pDBMng->addArchLog(tableSet, archId, archPath);
    pAH->sendResponse(Chain("Archlog " + archPath + " added to " + tableSet));    
}

void CegoAdminThread::srvRemoveArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain archId;

    pAH->getTableSet(tableSet);
    pAH->getArchId(archId);
    _pDBMng->removeArchLog(tableSet, archId);
    pAH->sendResponse(Chain("Archlog id " + archId + " removed from " + tableSet));
}

void CegoAdminThread::srvBeginBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain msg;

    pAH->getTableSet(tableSet);
    pAH->getBUMsg(msg);
    _pTabMng->beginBackup(tableSet, msg);
    pAH->sendResponse(Chain("Backup started"));
}

void CegoAdminThread::srvExecuteBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain msg;
    Chain result;
    
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(msg);
    result = executeBackup(pAH, tableSet, msg);  
    pAH->sendResponse(Chain("Backup execution status : ") + result);
}

void CegoAdminThread::srvEndBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain msg;
    bool keepTicket;

    pAH->getTableSet(tableSet);
    pAH->getBUMsg(msg);
    pAH->getKeepTicket(keepTicket);
    _pTabMng->endBackup(tableSet, msg, keepTicket);    
    pAH->sendResponse(Chain("Backup ended"));
}

void CegoAdminThread::srvListBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain pit;
    Element *pBackupInfo;
    
    pAH->getTableSet(tableSet);
    pAH->getPit(pit);
    pBackupInfo = getBackupInfo(tableSet, pit);
    _lastAction = Chain("listBackup");
    pAH->sendResponse(Chain("Backup list"), pBackupInfo);
}

void CegoAdminThread::srvRestore(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain backupId;
    
    pAH->getTableSet(tableSet);
    pAH->getBackupId(backupId);
    Chain result = executeRestore(pAH, tableSet, backupId);
    pAH->sendResponse(Chain("Restore execution status : ") + result);
}

void CegoAdminThread::srvRecover(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain pit;
    long long pitTS = 0;
    
    pAH->getTableSet(tableSet);
    pAH->getPit(pit);
    if ( pit != Chain() )
    {
	__dateFormatLock.readLock(DBM_LOCKTIMEOUT);
	try
	{
	    Datetime pitTime;
	    pitTime = Datetime(pit, __dateFormatList);
	    pitTS = pitTime.asLong();
	}
	catch ( Exception e )
	{
	    __dateFormatLock.unlock();
	    throw Exception(EXLOC, Chain("Cannot get pit information"), e);   
	}
	__dateFormatLock.unlock();
    }

    Chain tsTicketName = _pDBMng->getTSTicket(tableSet);
    if ( tsTicketName == Chain() )
    {
	Chain msg = Chain("Ticket file not defined");	
	throw Exception(EXLOC, msg);
    }
    
    if ( File::exists(tsTicketName) == false )
    {
	Chain msg = Chain("Backup ticket ") + tsTicketName + Chain(" not found");	
	throw Exception(EXLOC, msg);
    }

    CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::LOCAL);
    unsigned long long lsn = recoveryMng.recoverTableSet(tableSet, pitTS, pAH);
    pAH->sendResponse(Chain("Tableset recovered to lsn ") + Chain(lsn));
}

void CegoAdminThread::srvSecSwitch(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain secondary;
    
    pAH->getTableSet(tableSet);
    pAH->getSecondary(secondary);
    _pTabMng->logTo(tableSet, secondary);
    _pDBMng->setSecondary(tableSet, secondary);
    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" log to  ") + secondary);
}

void CegoAdminThread::srvEnableTableCache(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);
    _pDBMng->setTableCacheMode(tableSet, true);
    _pDBMng->allocateTableCache(tableSet);
    pAH->sendResponse(Chain("Table cache enabled"));
}

void CegoAdminThread::srvDisableTableCache(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);
    _pDBMng->releaseTableCache(tableSet);
    _pDBMng->setTableCacheMode(tableSet, false);    
    pAH->sendResponse(Chain("Table cache disabled"));
}

void CegoAdminThread::srvEnableQueryCache(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);
    _pDBMng->setQueryCacheMode(tableSet, true);
    _pDBMng->allocateQueryCache(tableSet);
    pAH->sendResponse(Chain("Query cache enabled"));
}

void CegoAdminThread::srvDisableQueryCache(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);
    _pDBMng->releaseQueryCache(tableSet);    
    _pDBMng->setQueryCacheMode(tableSet, false);
    pAH->sendResponse(Chain("Query cache disabled"));
}

//////////////////////////////
///// mediator services //////
//////////////////////////////

void CegoAdminThread::medNotify(CegoAdminHandler *pAH)
{    
    Chain hostName;
    Chain hostStatus;
    ListT<Chain> tsList;
    ListT<Chain> syncList;
    ListT<Chain> runList;

    pAH->getHostName(hostName);
    pAH->getHostStatus(hostStatus);
    pAH->getTableSyncStateList(tsList, runList, syncList);
    
#ifdef CGDEBUG
    _pDBMng->log(_modId, Logger::DEBUG, Chain("Setting host status <") + hostStatus + Chain("> for host <") 
		+ hostName + Chain(">"));
#endif	

    _pDBMng->setHostStatus(hostName, hostStatus);

    Chain *pTS = tsList.First();
    Chain *pSync = syncList.First();
    Chain *pRun = runList.First();
    
    while ( pTS && pSync && pRun)
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Setting tableset status for <") + *pTS + Chain("> to ") + *pSync );
#endif
	_pDBMng->setTableSetRunState(*pTS, *pRun);
	_pDBMng->setTableSetSyncState(*pTS, *pSync);
	pTS = tsList.Next();
	pSync = syncList.Next();
	pRun = runList.Next();
    }
    pAH->sendResponse(Chain("Notify ok"));   
}

void CegoAdminThread::medDefineTableSet(CegoAdminHandler *pAH)
{   
    Chain tableSet;
    Chain tsRoot;
    Chain primary;
    Chain secondary;
    int sysSize;
    int tmpSize;
    int appSize;
    int logFileSize;
    int logFileNum;
    unsigned long long sortAreaSize;
    int branchId;
    
    pAH->getTableSet(tableSet);
    pAH->getTableSetRoot(tsRoot);
    pAH->getPrimary(primary);
    pAH->getSecondary(secondary);
    pAH->getSysSize(sysSize);
    pAH->getTmpSize(tmpSize);
    pAH->getAppSize(appSize);
    pAH->getLogFileSize(logFileSize);
    pAH->getLogFileNum(logFileNum);
    pAH->getSortAreaSize(sortAreaSize);
    pAH->getBranchId(branchId);
    
    int sysFid = _pDBMng->nextTSID();
    int tmpFid = _pDBMng->nextFID();

    Chain resultMsg;

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    
    if ( primary == Chain() )
    {
	primary = dbHost;
    }

    if ( secondary == Chain() )
    {
	secondary = dbHost;
    }

    if ( dbHost != primary )
    {
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }

    if ( dbHost != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }
    
    CegoAdminHandler::ResultType res;
    
    if ( pPrimaryAH )
    {
	res = pPrimaryAH->reqDefineTableSet(tableSet, tsRoot, primary, secondary, dbHost, sysFid, tmpFid, sysSize, tmpSize, appSize, logFileSize, logFileNum, sortAreaSize);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK )
	    throw Exception(EXLOC, msg);
    }
    
    if ( pSecondaryAH )
    {
	res = pSecondaryAH->reqDefineTableSet(tableSet, tsRoot, primary, secondary, dbHost, sysFid, tmpFid, sysSize, tmpSize, appSize, logFileSize, logFileNum, sortAreaSize);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
		
	if ( res != CegoAdminHandler::ADM_OK )
	    throw Exception(EXLOC, msg);
    }
    
    _pDBMng->addTableSetDef(tableSet, tsRoot, primary, secondary, dbHost, sysFid, tmpFid, sysSize, tmpSize, appSize, logFileSize, logFileNum, sortAreaSize, branchId); 
    
    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" defined"));
}

void CegoAdminThread::medRemoveTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    pAH->getTableSet(tableSet);        
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status defined to be removed");	
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);	
    }
    
    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;

    if ( dbHost != primary )
    {
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }
    
    CegoAdminHandler::ResultType res;
        
    if ( pPrimaryAH )
    {
	res = pPrimaryAH->reqRemoveTableSet(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( dbHost != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }

    if ( pSecondaryAH )
    {
	res = pSecondaryAH->reqRemoveTableSet(tableSet);
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);	

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }
    
    _pDBMng->rmTableSetDef(tableSet); 

    Chain msg = Chain("Tableset ") + tableSet + Chain(" removed");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medResetTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    pAH->getTableSet(tableSet);
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);	
    }
    
    if ( dbHost != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;
	res = pPrimaryAH->reqResetTableSet(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	
	pAH->syncWithInfo(Chain("primary"), primary, msg);
    }
    else
    {	   
	_pTabMng->resetTableSet(tableSet); 	
    }

    Chain msg = Chain("Tableset ") + tableSet + Chain(" reset");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medCreateTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain resultMsg;

    pAH->getTableSet(tableSet);
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( status != Chain(XML_DEFINED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status defined to be created");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;	
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    
    try 
    {	
	_pDBMng->getStatusByHost(primary, primaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	_pTabMng->createTableSet(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
 
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());	
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqCreateTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{
	    res = pPrimaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	    Chain msg;
	    pPrimaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pPrimaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("primary"), primary, msg);
	    }
	}
    }

    if ( primary == secondary )
    {
	_pDBMng->setTableSetSyncState(tableSet,XML_SYNCHED_VALUE);
    }
    else
    {
	CegoAdminHandler* pSecondaryAH = 0;
 
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pSecondaryAH->reqSetRunState(tableSet, XML_OFFLINE_VALUE);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}

	res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);

	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
	
	closeSession(pSecondaryAH);
	
	_pDBMng->setTableSetSyncState(tableSet,XML_NOT_SYNCHED_VALUE);
    }

    _pDBMng->setTableSetRunState(tableSet,XML_OFFLINE_VALUE);    

    Chain msg = Chain("Tableset ") + tableSet + Chain(" created");

    pAH->sendResponse(msg);
}

void CegoAdminThread::medCopyTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain resultMsg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primary == secondary )
    {
	Chain msg = Chain("Cannot copy tablset, primary and secondary are identical (") + primary + Chain("=") + secondary + Chain(")");
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;

    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;
    
    if ( mediator != primary )
    { 
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
    }
    
    if ( mediator != secondary )
    {
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
    }

    if ( pSecondaryAH == 0 )
    {
	// local reset

	int tabSetId = _pDBMng->getTabSetId(tableSet);
	    
	if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
	{

	    _pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
	    
	    while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Waiting for recovery end ..."));
#endif
		Sleeper s;
		s.secSleep(LOGMNG_RECOVERY_DELAY);
	    }
	}
    }
    else
    {
#ifdef CGDEBUG
	_pDBMng->log(_modId, Logger::DEBUG, Chain("Stopping recovery on secondary ..."));
#endif

	CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	       
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);    
	}	
    }
    
    if ( pPrimaryAH == 0 )
    {
	_pDBMng->startCopy(tableSet, secondary, mediator, pAH->getUser(), pAH->getPassword(), Chain("Copy requested"));
	pAH->syncWithInfo(Chain("primary"), primary, Chain("Copy started"));
    }
    else
    {
	CegoAdminHandler::ResultType res = pPrimaryAH->reqCopyTableSet(tableSet, secondary, mediator);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	if ( pSecondaryAH == 0 )
	{
	    _pDBMng->startRecovery(tableSet);	    
	}
	else
	{	    
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStartRecovery(tableSet);    

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    } 
    
    pAH->sendResponse(Chain("Tableset copy started"));
}

void CegoAdminThread::medDropTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain resultMsg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_OFFLINE_VALUE) && status != Chain(XML_CHECKPOINT_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline to be dropped");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");	
	throw Exception(EXLOC, msg);	
    }

    if ( mediator == primary )
    {	
	_pTabMng->dropTableSet(tableSet);   
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	CegoAdminHandler::ResultType res = pPrimaryAH->reqDropTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	    
	pAH->syncWithInfo(Chain("primary"), primary, msg);
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, Chain("Admin action failed"));
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqDropTableSet(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}
    }

    _pDBMng->setTableSetRunState(tableSet,XML_DEFINED_VALUE);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" dropped");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medStartTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);
    bool cleanIt = pAH->getCleanup();
    bool forceload = pAH->getForceload();
    bool cpDump = pAH->getCPDump();
    bool noInit = pAH->getNoInit();

    _lastAction = Chain("Start tableset ") + tableSet;

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_OFFLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline to be started");       	
	throw Exception(EXLOC, msg);
    }
        
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");	
	throw Exception(EXLOC, msg);

    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");	
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	
	_pDBMng->lockPool(tabSetId, true);

	try 
	{
	    _pTabMng->startDistTableSet(tableSet, secondary, cleanIt, true, true, cpDump);
	    
	    if ( forceload )
	    {
		_pDbPool->loadObjects(tabSetId);
	    }

	    if ( noInit == false )
	    {
		Chain batchFileName = _pDBMng->getTSInitFile(tableSet);

		if ( batchFileName != Chain() )
		{
		    File batchFile(batchFileName);
		    
		    if ( batchFile.exists() ) 
		    {
			_pDBMng->log(_modId, Logger::NOTICE, Chain("Processing init file ") + batchFileName);
			
			CegoAction* pAction = new CegoAction(_pTabMng);
			
			pAction->setLogToFile(true);
			
			try
			{
			    pAction->setTableSet(tableSet);
			    
			    Chain errorMsg;
			    if ( pAction->processBatchFile(batchFileName, false, false, errorMsg) == false )
				throw Exception(EXLOC, errorMsg);	    
			}
			catch ( Exception e )
			{
			    delete pAction;
			    throw Exception(EXLOC, Chain("Execution of init file ") + batchFileName + Chain(" failed"), e);
			}
			
			delete pAction;
		    }
		    else
		    {
			throw Exception(EXLOC, Chain("Init file <") + batchFileName + Chain("> does not exist"));
		    }
		}
	    }
	}
	catch ( Exception e )
	{
	    _pDBMng->unlockPool(tabSetId);
	    
	    Chain msg;
	    e.pop(msg);	    
	    throw Exception(EXLOC, msg);	    
	}
	
	_pDBMng->unlockPool(tabSetId);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqStartTableSet(tableSet, cleanIt, forceload);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( primary != secondary )
    {
	CegoAdminHandler* pSecondaryAH = 0;
	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	CegoAdminHandler::ResultType res = pSecondaryAH->reqStartRecovery(tableSet);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
		
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}	
    }
    
    _pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" started");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medStopTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Stop tableset ") + tableSet;

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain resultMsg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status active to be stopped");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try
    {	
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);	
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	
	_pDBMng->lockPool(tabSetId, true);

	try 
	{
	    _pDbPool->unloadObjects(tabSetId);
		
	    // set tableset runstate first to offline to give other pending thread a chance to abort
	    _pDBMng->setTableSetRunState(tableSet, XML_OFFLINE_VALUE);
	    
	    if ( _pDbPool->isBusy() )
	    {
		// since there might be some dead locks hanging around, we wait double time of DB_LOCKDELAY
		Sleeper::secSleep(TS_STOPDELAY);
	    }
	    
	    _pTabMng->stopDistTableSet(tableSet, true);
	}
	catch ( Exception e )
	{
	    _pDBMng->unlockPool(tabSetId);
	    
	    Chain msg;
	    e.pop(msg);	
	    throw Exception(EXLOC, msg);
	}
	
	_pDBMng->unlockPool(tabSetId);
    }
    else
    {      	
	CegoAdminHandler* pPrimaryAH = 0;
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqStopTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}
    }

    _pDBMng->setTableSetRunState(tableSet,XML_OFFLINE_VALUE);

    Chain msg = Chain("Tableset ") + tableSet + Chain(" stopped");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medSwitchTableSet(CegoAdminHandler *pAH)
{    
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain runStatus;
    Chain syncStatus;
    Chain resultMsg;

    bool isOnline = false;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    runStatus = _pDBMng->getTableSetRunState(tableSet);
    syncStatus = _pDBMng->getTableSetSyncState(tableSet);

    Chain newPrimary = secondary;
    Chain newSecondary = mediator;
    Chain newMediator = mediator;

    if ( syncStatus != Chain(XML_SYNCHED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" is not in sync");	
	throw Exception(EXLOC, msg);	
    }

    if ( runStatus == Chain(XML_ONLINE_VALUE) )
    {
	isOnline = true;
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;	
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);	
    }

    if ( primary == secondary )
    {
	Chain msg = Chain("Primary and secondary identical, switch not useful");
	throw Exception(EXLOC, msg);	
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    // first step : make sure primary host is down
    
    if ( mediator == primary )
    {	
	if ( isOnline )
	{
	    try 
	    {
#ifdef CGDEBUG
		_pDBMng->log(_modId, Logger::DEBUG, Chain("Stopping tableset ") + tableSet);
#endif
		int tabSetId = _pDBMng->getTabSetId(tableSet);
		_pDbPool->unloadObjects(tabSetId);

		_pTabMng->stopTableSet(tableSet, true);
	    }
	    catch ( Exception e )
	    {
		Chain msg;
		e.pop(msg);
		throw Exception(EXLOC, msg);	
	    }
	}
    }
    else
    {
	try
	{
	    CegoAdminHandler* pPrimaryAH = 0;
	    CegoAdminHandler::ResultType res;
	    
	    pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	    if ( isOnline )
	    {
		res = pPrimaryAH->reqStopTableSet(tableSet);

		Chain msg;
		pPrimaryAH->getMsg(msg);
		
		if ( res != CegoAdminHandler::ADM_OK)
		{
		    closeSession(pPrimaryAH);
		    throw Exception(EXLOC, msg);
		}
		else
		{
		    pAH->syncWithInfo(Chain("primary"), primary, msg);    		    
		}
	    }
	    
	    res = pPrimaryAH->reqSetTableSetNode(tableSet, newPrimary, newSecondary, newMediator);
	    Chain msg;
	    pPrimaryAH->getMsg(msg);
	    closeSession(pPrimaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	}
	catch ( Exception e ) 
	{
	    // ignore, primary no more available
	}
    }
	
    // second step : activate secondary

    if ( mediator == secondary )
    {
	if ( isOnline )
	{   	    
	    int tabSetId = _pDBMng->getTabSetId(tableSet);
	    
	    if ( _pDBMng->getRecoveryMode(tabSetId) == CegoDatabaseManager::ON )
	    {		
		_pDBMng->setRecoveryMode(tabSetId, CegoDatabaseManager::REQOFF);
		
		while ( _pDBMng->getRecoveryMode(tabSetId) != CegoDatabaseManager::OFF )
		{
#ifdef CGDEBUG
		    _pDBMng->log(_modId, Logger::DEBUG, Chain("Waiting for recovery end ... "));
#endif
		    Sleeper s;
		    s.secSleep(LOGMNG_RECOVERY_DELAY);
		}
	    }

#ifdef CGDEBUG
	    _pDBMng->log(_modId, Logger::DEBUG, Chain("Starting tableset ") + tableSet);
#endif

	    // TODO : check forceload, autocorrect, temp reset and cpDump flags
	    _pTabMng->startDistTableSet(tableSet, newSecondary, false, false, false, false);
	    
	    // load objects not really required, for performance reasons we do not
	    // _pDbPool->loadObjects(tabSetId);   
	}
    }
    else
    {
	CegoAdminHandler* pSecondaryAH = 0;	
	CegoAdminHandler::ResultType res;

	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	if ( isOnline )
	{
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	 
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}

	res = pSecondaryAH->reqSetTableSetNode(tableSet, newPrimary, newSecondary, newMediator);

	Chain msg;
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);    
	}

	if ( isOnline )
	{
	    res = pSecondaryAH->reqStartTableSet(tableSet, false, false);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}

	if ( newPrimary != newSecondary )
	{    
	    res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}
	else
	{
	    res = pSecondaryAH->reqSetSyncState(tableSet, XML_SYNCHED_VALUE);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		closeSession(pSecondaryAH);
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}   
	closeSession(pSecondaryAH);
    }

    if ( newPrimary != newSecondary )
	_pDBMng->setTableSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
    else
	_pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);

    _pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);

    _pDBMng->setPrimary(tableSet, newPrimary);
    _pDBMng->setSecondary(tableSet, newSecondary);
    _pDBMng->setMediator(tableSet, newMediator);
    
    Chain msg = Chain("Tableset ") + tableSet + Chain(" switched");
    pAH->sendResponse(msg);
}

void CegoAdminThread::medAddArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain archId;
    pAH->getArchId(archId);
    
    Chain archPath;
    pAH->getArchPath(archPath);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain resultMsg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + mediator + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;
 
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }
    
    if ( mediator != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqAddArchLog(tableSet, archId, archPath);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), secondary, msg);
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqAddArchLog(tableSet, archId, archPath);

	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("primary"), secondary, msg);
	    }
	}	
    }
    
    _pDBMng->addArchLog(tableSet, archId, archPath);
   
    pAH->sendResponse("Archlog added");
}

void CegoAdminThread::medRemoveArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain archId;
    pAH->getArchId(archId);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;

    Chain resultMsg;

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }
       
    if ( mediator != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqRemoveArchLog(tableSet, archId);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);	    
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqRemoveArchLog(tableSet, archId);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    }

    _pDBMng->removeArchLog(tableSet, archId);
    
    ListT<Chain> archIdList;
    ListT<Chain> archPathList;
    _pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);
    
    if ( archIdList.Size() == 0 )
    {	    
	_pDBMng->disableArchLog(tableSet);
    }
    
    pAH->sendResponse(Chain("Archlog removed"));
}

void CegoAdminThread::medAddDatafile(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain fileName;
    pAH->getFileName(fileName);
    
    int fileSize;
    pAH->getFileSize(fileSize);

    Chain fileType;
    pAH->getFileType(fileType);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);	
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);		
    }

    int fileId = _pDBMng->nextFID();
    
    if ( mediator == primary )
    {
	Chain runState = _pDBMng->getTableSetRunState(tableSet);
	
	if ( runState != Chain(XML_DEFINED_VALUE) )
	{	
	    _pTabMng->addDataFile(tableSet, fileType, fileId, fileName, fileSize);
	    
	    if ( runState == Chain(XML_ONLINE_VALUE) )
	    {
		_pDBMng->writeCheckPoint(tableSet, true, false, _pTabMng->getLockHandle());
	    }
	}
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqAddDataFile(tableSet, fileType, fileId,  fileName, fileSize);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( primary != secondary && secondary != mediator )
    {	
	CegoAdminHandler* pSecondaryAH = 0;
	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pSecondaryAH->reqAddDataFile(tableSet, fileType, fileId, fileName, fileSize);
	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }	
    
    try
    {
	_pDBMng->addDataFile(tableSet, fileType, fileId, fileName, fileSize); 	    
    }
    catch ( Exception e )
    {
	Chain msg = e.getBaseMsg();
	throw Exception(EXLOC, msg);		
    }
	
    pAH->sendResponse(Chain("Datafile added"));
}

void CegoAdminThread::medGetTableSetList(CegoAdminHandler *pAH)
{    
    Element* pTabSetInfo = _pDBMng->getTableSetList(); 

    bool getUsage;
    pAH->getUsage(getUsage);
    if ( getUsage )
    {	
	ListT<Element*> tabSetList = pTabSetInfo->getChildren(XML_TABLESET_ELEMENT);
	Element **pTabSet = tabSetList.First();
	while ( pTabSet )
	{
	    Chain tableSet = (*pTabSet)->getAttributeValue(XML_NAME_ATTR);
	    addTableSetUsage(*pTabSet);
	    
	    pTabSet = tabSetList.Next();
	}
    }
    pAH->sendResponse(Chain("Tableset list"), pTabSetInfo);		      
}

void CegoAdminThread::medGetDetailedTableSetInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain primary;
    Chain mediator;
    
    try 
    {
	primary = _pDBMng->getPrimary(tableSet);
	mediator = _pDBMng->getMediator(tableSet);
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);
	throw Exception(EXLOC, msg);
    }

    Element *pTSInfo;

    if ( primary == mediator )
    {    
	pTSInfo = getDetailedTSInfo(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetDetailedTSInfo(tableSet);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    Chain msg;
	    pPrimaryAH->getMsg(msg);   
	    throw Exception(EXLOC, msg);
	}

	pTSInfo = pPrimaryAH->getTSInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);	
    }
    
    pAH->sendResponse(Chain("Tableset information"), pTSInfo);
}

void CegoAdminThread::medGetArchLogInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain primary;
    Chain mediator;

    pAH->getTableSet(tableSet);
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    Element *pArchInfo;

    if ( primary == mediator )
    {    
	pArchInfo = getArchLogInfo(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetArchLogInfo(tableSet);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    Chain msg;
	    pPrimaryAH->getMsg(msg);   
	    throw Exception(EXLOC, msg);
	}

	pArchInfo = pPrimaryAH->getArchInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);	
    }
    
    pAH->sendResponse(Chain("Archive log information"), pArchInfo);
}


void CegoAdminThread::medGetTransactionInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Chain primary;
    Chain mediator;
    
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    Element *pTAInfo;

    if ( primary == mediator )
    {    
	pTAInfo = getTransactionInfo(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetTransactionInfo(tableSet);

	pTAInfo = pPrimaryAH->getTAInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("Transaction info"), pTAInfo);   
}

void CegoAdminThread::medGetBUStatInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Chain primary;
    Chain mediator;
    
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    Element *pBUSInfo = 0;

    if ( primary == mediator )
    {    
	pBUSInfo = getBUStatInfo(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetBUStatInfo(tableSet);

	// TODO
	// pBUSInfo = pPrimaryAH->getBUStatInfo(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("BUStat info"), pBUSInfo);   
}

void CegoAdminThread::medResetBUStat(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Chain primary;
    Chain mediator;
    
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    if ( primary == mediator )
    {    
	_pTabMng->resetBUStat(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = CegoAdminHandler::ADM_OK; // pPrimaryAH->reqGetBUStatInfo(tableSet);

	// TODO
	// pBUSInfo = pPrimaryAH->getBUStatInfo(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("BUStat reset done"));   
}

void CegoAdminThread::medGetObjectInfo(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain resultMsg;

    Chain primary;
    Chain mediator;
    
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    Element *pObjectInfo;

    if ( primary == mediator )
    {    
	pObjectInfo = getObjectInfo(tableSet);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetObjectInfo(tableSet);

	pObjectInfo = pPrimaryAH->getObjectInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("Object info"), pObjectInfo);
}

void CegoAdminThread::medBeginBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(bumsg);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online to begin backup");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {    
	_pTabMng->beginBackup(tableSet, bumsg);
    }
    else
    {	
	CegoAdminHandler* pPrimaryAH = 0;
	
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqBeginBackup(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	_pDBMng->setTableSetRunState(tableSet,XML_BACKUP_VALUE);
    }

    pAH->sendResponse(Chain("Backup mode started"));    
}

void CegoAdminThread::medExecuteBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(bumsg);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status backup to execute backup");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    Chain result;
    if ( mediator == primary )
    {
	result = executeBackup(pAH, tableSet, bumsg);
    }
    else
    {	
	CegoAdminHandler* pPrimaryAH = 0;
	
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqExecuteBackup(tableSet);

	pPrimaryAH->getMsg(result);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, result);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, result);
	}       
    }

    pAH->sendResponse(Chain("Backup execution status : ") + result);    
}


void CegoAdminThread::medEndBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    pAH->getTableSet(tableSet);
    pAH->getBUMsg(bumsg);

    bool keepTicket;
    pAH->getKeepTicket(keepTicket);
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( status != Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status backup to end backup");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
   
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {	
	_pTabMng->endBackup(tableSet, bumsg, keepTicket);	
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
	
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqEndBackup(tableSet, keepTicket);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	_pDBMng->setTableSetRunState(tableSet, XML_ONLINE_VALUE);	
    }

    pAH->sendResponse(Chain("Backup mode finished"));
}

void CegoAdminThread::medListBackup(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain bumsg;
    Chain pit;
    
    pAH->getTableSet(tableSet);
    pAH->getPit(pit);

    // verify pit formt
    if ( pit != Chain() )
    {
	__dateFormatLock.readLock(DBM_LOCKTIMEOUT);
	try
	{
	    Datetime pitTime;
	    pitTime = Datetime(pit, __dateFormatList);	    
	}
	catch ( Exception e )
	{
	    __dateFormatLock.unlock();
	    throw Exception(EXLOC, Chain("Cannot get pit information"), e);   
	}
	__dateFormatLock.unlock();
    }
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
   
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    Element *pBackupInfo = 0;

    if ( primary == mediator )
    {
	pBackupInfo = getBackupInfo(tableSet, pit);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqListBackup(tableSet, pit);

	// TODO
	// pBackupInfo = pPrimaryAH->getBackupInfo();

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    pAH->sendResponse(Chain("Backup info"), pBackupInfo);    
}

void CegoAdminThread::medRestore(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain backupId;
    
    pAH->getTableSet(tableSet);
    pAH->getBackupId(backupId);
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain primaryStatus;
    Chain secondaryStatus;
    Chain logMngProg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( status != Chain(XML_OFFLINE_VALUE) && status != Chain(XML_DEFINED_VALUE)  )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline or defined to be restored");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    Chain result;
    if ( mediator == primary )
    {
	result = executeRestore(pAH, tableSet, backupId);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
		
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqRestore(tableSet, backupId);

	pPrimaryAH->getMsg(result);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, result);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, result);
	}	
    }

    pAH->sendResponse(Chain("Restore execution status : ") + result);
}

void CegoAdminThread::medRecover(CegoAdminHandler *pAH)
{
    Chain tableSet;
    Chain pit;
    unsigned long long pitTS = 0;
    
    pAH->getTableSet(tableSet);
    pAH->getPit(pit);

    if ( pit != Chain() )
    {	
	__dateFormatLock.readLock(DBM_LOCKTIMEOUT);
	try
	{
	    Datetime pitTime;
	    pitTime = Datetime(pit, __dateFormatList);
	    pitTS = pitTime.asLong();
	}
	catch ( Exception e )
	{
	    __dateFormatLock.unlock();
	    throw Exception(EXLOC, Chain("Cannot get pit information"), e);   
	}
	__dateFormatLock.unlock();
    }
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
    Chain primaryStatus;
    Chain secondaryStatus;
    Chain logMngProg;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( status != Chain(XML_OFFLINE_VALUE) && status != Chain(XML_DEFINED_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status offline or defined to be recovered");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	// check for external logmanager 
	Chain archRestoreProg = _pDBMng->getArchRestoreProg();

	if ( archRestoreProg == Chain())
	{
	    Chain msg = Chain("No log manager defined");
	    throw Exception(EXLOC, msg);		
	}
	
	if ( archRestoreProg != Chain("NONE") )
	{
	    if ( File::exists(archRestoreProg) == false )
	    {
		Chain msg = Chain("External logmanager <") + archRestoreProg + Chain("> does not exist");
		throw Exception(EXLOC, msg);		
	    }
	}

	// check for existing backup ticket, must exists to trigger recovery
	Chain tsTicketName = _pDBMng->getTSTicket(tableSet);	

	if ( tsTicketName == Chain())
	{
	    Chain msg = Chain("No ticket file defined");
	    throw Exception(EXLOC, msg);		
	}

	if ( File::exists(tsTicketName) == false )
	{
	    Chain msg = Chain("Backup ticket ") + tsTicketName + Chain(" not found");	
	    throw Exception(EXLOC, msg);
	}

	unsigned long long lsn;

	CegoRecoveryManager recoveryMng(_pTabMng, CegoRecoveryManager::LOCAL);
	lsn = recoveryMng.recoverTableSet(tableSet, pitTS, pAH);
	
	pAH->syncWithInfo(Chain("mediator"), mediator, Chain("Tableset recovered to lsn ") + Chain(lsn) + Chain("\n"));
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;
		
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqRecover(tableSet, pit);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}	
    }

    pAH->sendResponse(Chain("Recover completed"));   
}

void CegoAdminThread::medSecSwitch(CegoAdminHandler *pAH)
{
    Chain tableSet;
    
    pAH->getTableSet(tableSet);
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	try
	{
	    _pTabMng->logTo(tableSet, mediator);
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);  
	}
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqSecSwitch(tableSet, mediator);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    pAH->sendResponse(Chain("Secondary switch"));
}

void CegoAdminThread::medSecRelocate(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain newSecondary;
    pAH->getSecondary(newSecondary);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    bool isArchiveMode;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    isArchiveMode = _pDBMng->isArchiveMode(tableSet);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( isArchiveMode == false )
    {
	Chain msg = Chain("No archive mode enabled");
	throw Exception(EXLOC, msg);
    }
    if ( mediator == primary )
    {       
	try
	{
	    _pTabMng->logTo(tableSet, newSecondary);
	}
	catch ( Exception e )
	{
	    Chain msg = e.getBaseMsg();
	    throw Exception(EXLOC, msg);	   
	}
    }
    else // ( mediator != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res;

	res = pPrimaryAH->reqSecSwitch(tableSet, newSecondary);

	Chain msg;
	pPrimaryAH->getMsg(msg);
		    
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	res = pPrimaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	pPrimaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, newSecondary, mediator);

	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( mediator != secondary && primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;
	    
	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	   
	    CegoAdminHandler::ResultType res = pSecondaryAH->reqStopRecovery(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    closeSession(pSecondaryAH);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	}	
    }

    if ( mediator != newSecondary )
    {
	CegoAdminHandler* pSecondaryAH = 0;
       
	pSecondaryAH = getSession(Chain("secondary"), newSecondary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;
	Chain msg;

	res = pSecondaryAH->reqPropTSInfo(_pDBMng->getTableSetInfo(tableSet));
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}

	res = pSecondaryAH->reqSecSwitch(tableSet, newSecondary);
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetRunState(tableSet, _pDBMng->getTableSetRunState(tableSet));
	pSecondaryAH->getMsg(msg);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
	pSecondaryAH->getMsg(msg);
	    
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pSecondaryAH);
	    throw Exception(EXLOC, msg);
	}
	
	res = pSecondaryAH->reqSetTableSetNode(tableSet, primary, newSecondary, mediator);

	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
    }

    if ( primary != newSecondary )
    {
	_pDBMng->setTableSetSyncState(tableSet, XML_NOT_SYNCHED_VALUE);
    }
    else
    {
	_pDBMng->setTableSetSyncState(tableSet, XML_SYNCHED_VALUE);
    }

    _pDBMng->setSecondary(tableSet, newSecondary);
    
    pAH->sendResponse(Chain("Secondary relocated"));
}

void CegoAdminThread::secMedSwitch(CegoAdminHandler *pAH)
{
    Chain tableSet;
    
    pAH->getTableSet(tableSet);
    
    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;
 
    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( dbHost != secondary )
    {
	Chain msg = Chain("Invalid secondary host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator != secondary )
    {
	CegoAdminHandler* pSecondaryAH = 0;
	CegoAdminHandler::ResultType res;

	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());

	res = pSecondaryAH->reqSetTableSetNode(tableSet, primary, secondary, secondary);	
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }

    if ( mediator != primary )
    {
	CegoAdminHandler* pPrimaryAH = 0;
	CegoAdminHandler::ResultType res;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, secondary, secondary);	
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    _pDBMng->setMediator(tableSet, secondary);    

    pAH->sendResponse("Secondary switched");
}

void CegoAdminThread::secMedRelocate(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain newMediator;
    pAH->getMediator(newMediator);
    
    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    if ( dbHost != secondary )
    {
	Chain msg = Chain("Invalid secondary host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
        
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);
    
    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);       
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( primary != newMediator )
    {
	CegoAdminHandler* pPrimaryAH = 0;
       
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;

	res = pPrimaryAH->reqSetTableSetNode(tableSet, primary, secondary, newMediator);
	Chain msg;
	pPrimaryAH->getMsg(msg);
     	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pPrimaryAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }

    if ( secondary != newMediator )
    {
	CegoAdminHandler* pMediatorAH = 0;
       
	pMediatorAH = getSession(Chain("mediator"), newMediator, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res;

	res = pMediatorAH->reqPropTSInfo(_pDBMng->getTableSetInfo(tableSet));

	Chain msg;
	pMediatorAH->getMsg(msg);
		
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    closeSession(pMediatorAH);
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("mediator"), newMediator, msg);
	}

	res = pMediatorAH->reqSetTableSetNode(tableSet, primary, secondary, newMediator);
	pMediatorAH->getMsg(msg);
	closeSession(pMediatorAH);

	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("mediator"), newMediator, msg);
	}
    }

    _pDBMng->setMediator(tableSet, newMediator);
    
    pAH->sendResponse("Mediator relocated");
}

void CegoAdminThread::medCheckTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    Chain dbHost;
    _pDBMng->getDBHost(dbHost);
    Chain primary = _pDBMng->getPrimary(tableSet);
    Chain secondary = _pDBMng->getSecondary(tableSet);
    Chain mediator = _pDBMng->getMediator(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    try
    {
	_pDBMng->getStatusByHost(primary, primaryStatus);
	_pDBMng->getStatusByHost(secondary, secondaryStatus);	
    }
    catch ( Exception e )
    {
	Chain msg;
	e.pop(msg);	
	throw Exception(EXLOC, msg);	
    }

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }

    CegoAdminHandler* pPrimaryAH = 0;
    CegoAdminHandler* pSecondaryAH = 0;

    Element* pTSInfo = _pDBMng->getTableSetInfo(tableSet); 
	
    if ( mediator != primary )
    {
	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pPrimaryAH->reqGetTSInfo(tableSet);
	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}
    }
    
    if ( mediator != secondary )
    {	
	pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	
	CegoAdminHandler::ResultType res = pSecondaryAH->reqGetTSInfo(tableSet);
	Chain msg;
	pSecondaryAH->getMsg(msg);
	closeSession(pSecondaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	}
    }

    Element *pCheck = checkTS(Chain("mediator"), mediator, pTSInfo, pPrimaryAH, pSecondaryAH);
    
    if ( pPrimaryAH )
	closeSession(pPrimaryAH);
    if ( pSecondaryAH )
	closeSession(pSecondaryAH);

    pCheck->setAttribute(XML_MEDIATOR_ATTR, mediator);
    pCheck->setAttribute(XML_PRIMARY_ATTR, primary);
    pCheck->setAttribute(XML_SECONDARY_ATTR, secondary);
    
    pAH->sendResponse(Chain("Check completed"), pCheck);
}

void CegoAdminThread::medVerifyTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Verify tableset ") + tableSet;

    Chain resultMsg;

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online to be verified");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);		
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	_pDBMng->lockPool(tabSetId, true);

	try 
	{
	    _pTabMng->setActiveUser(tableSet, _user, _password);
	    
	    ListT<Chain> tableList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);
	    
	    Chain *pTable = tableList.First();
	    while ( pTable)
	    {		
		Element *pVerificationInfo = _pTabMng->verifyTable(tableSet, *pTable);
		if ( pAH->syncWithInfo(Chain("primary"), primary, Chain("Verification of ") + *pTable, pVerificationInfo) == false )
		{
		    return;
		}

		pTable = tableList.Next();
	    }

	    ListT<Chain> viewList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::VIEW, viewList);
	    
	    Chain *pView = viewList.First();
	    while ( pView )
	    {
		Element* pVerificationInfo = _pTabMng->verifyView(tableSet, *pView);
		if ( pAH->syncWithInfo(Chain("primary"), primary, Chain("Verification of ") + *pView, pVerificationInfo) == false )
		    return;
		pView = viewList.Next();
	    }
	    
	    ListT<Chain> procList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::PROCEDURE, procList);
	    
	    Chain *pProc = procList.First();
	    while ( pProc )
	    {
		Element* pVerificationInfo = _pTabMng->verifyProcedure(tableSet, *pProc);
		if ( pAH->syncWithInfo(Chain("primary"), primary, Chain("Verification of ") + *pProc, pVerificationInfo) == false )
		    return;

		pProc = procList.Next();
	    }
	    
	    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" verified"));
	}
	catch ( Exception e )
	{
	    _pDBMng->unlockPool(tabSetId);
	    
	    Chain msg;
	    e.pop(msg);
	    throw Exception(EXLOC,msg);
	}
	_pDBMng->unlockPool(tabSetId);	
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqVerifyTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;

	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    
	    CegoAdminHandler::ResultType res = CegoAdminHandler::ADM_ERROR; // TODO  = pSecondaryAH->reqVerifyTS(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }
	    closeSession(pSecondaryAH);
	}	
    }
}

void CegoAdminThread::medCorrectTableSet(CegoAdminHandler *pAH)
{
    Chain tableSet;
    pAH->getTableSet(tableSet);

    _lastAction = Chain("Correct tableset ") + tableSet;

    Chain resultMsg;

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);
    
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Tableset ") + tableSet + Chain(" must be in status online to be corrected");
	throw Exception(EXLOC, msg);		
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;

    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	int tabSetId = _pDBMng->getTabSetId(tableSet);
	_pDBMng->lockPool(tabSetId, true);

	try 
	{
	    _pTabMng->setActiveUser(tableSet, _user, _password);

	    ListT<Chain> tableList; 
	    _pTabMng->getDistObjectList(tableSet, CegoObject::TABLE, tableList);
   
	    Chain *pTable = tableList.First();
	    while ( pTable)
	    {		
		Element *pCorrectionInfo = _pTabMng->correctTable(tableSet, *pTable);
		pAH->syncWithInfo(Chain("primary"), primary, Chain("Correction of ") + *pTable, pCorrectionInfo);
		pTable = tableList.Next();	    
	    }
    
	    pAH->sendResponse(Chain("Tableset ") + tableSet + Chain(" corrected"));
	}
	catch ( Exception e )
	{
	    _pDBMng->unlockPool(tabSetId);
	    
	    Chain msg;
	    e.pop(msg);
	    throw Exception(EXLOC, msg); 
	}

	_pDBMng->unlockPool(tabSetId);
    }
    else
    {
	CegoAdminHandler* pPrimaryAH = 0;

	pPrimaryAH = getSession(Chain("primary"), primary, pAH->getUser(), pAH->getPassword());

	CegoAdminHandler::ResultType res = pPrimaryAH->reqVerifyTableSet(tableSet);

	Chain msg;
	pPrimaryAH->getMsg(msg);
	closeSession(pPrimaryAH);
	
	if ( res != CegoAdminHandler::ADM_OK)
	{	    
	    throw Exception(EXLOC, msg);
	}
	else
	{
	    pAH->syncWithInfo(Chain("primary"), primary, msg);
	}

	if ( primary != secondary )
	{
	    CegoAdminHandler* pSecondaryAH = 0;

	    pSecondaryAH = getSession(Chain("secondary"), secondary, pAH->getUser(), pAH->getPassword());
	    
	    CegoAdminHandler::ResultType res = CegoAdminHandler::ADM_ERROR; // TODO  = pSecondaryAH->reqVerifyTS(tableSet);
	    Chain msg;
	    pSecondaryAH->getMsg(msg);
	    
	    if ( res != CegoAdminHandler::ADM_OK)
	    {
		throw Exception(EXLOC, msg);
	    }
	    else
	    {
		pAH->syncWithInfo(Chain("secondary"), secondary, msg);
	    }

	    closeSession(pSecondaryAH);
	}	
    }
}

void CegoAdminThread::medEnableArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	ListT<Chain> archIdList;
	ListT<Chain> archPathList;
	_pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);

	if ( archIdList.Size() == 0 )
	{	    
	    Chain msg = Chain("No valid archive locations available");
	    throw Exception(EXLOC, msg);	    
	}
	_pDBMng->enableArchLog(tableSet);

	// switch to new backup branch
	_pDBMng->nextTableSetBackupBranch(tableSet);

    }
    else
    {
	Chain msg = Chain("Archive mode just supported in single node mode");
	throw Exception(EXLOC, msg);	
    }
    
    pAH->sendResponse(Chain("Arch mode enabled"));
}

void CegoAdminThread::medDisableArchLog(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;

    _pDBMng->getDBHost(dbHost);    
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status == Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset is already in backup mode");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	_pDBMng->disableArchLog(tableSet);
    }
    else
    {
	Chain msg = Chain("Archive mode just supported in single node mode");
	throw Exception(EXLOC, msg);
    }

    pAH->sendResponse(Chain("Arch mode disabled"));
}

void CegoAdminThread::medEnableAppendMode(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain mediator;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	_pDBMng->setAppendMode(tableSet, true);
    }
    else
    {
	Chain msg = Chain("Append mode just supported in single node mode");
	throw Exception(EXLOC, msg);	
    }
    
    pAH->sendResponse(Chain("Append mode enabled"));
}

void CegoAdminThread::medDisableAppendMode(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain mediator;
 
    _pDBMng->getDBHost(dbHost);   
    primary = _pDBMng->getPrimary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    if ( mediator == primary )
    {
	_pDBMng->setAppendMode(tableSet, false);
    }
    else
    {
	Chain msg = Chain("Append mode just supported in single node mode");
	throw Exception(EXLOC, msg);
    }
    pAH->sendResponse(Chain("Append mode disabled"));    
}

void CegoAdminThread::medEnableAutoCorrect(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    _pDBMng->getDBHost(dbHost);
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	_pDBMng->setAutoCorrect(tableSet, true);
    }
    else
    {
	Chain msg = Chain("Auto Correct just supported in single node mode");
	throw Exception(EXLOC, msg);	
    }
    
    pAH->sendResponse(Chain("Auto correct enabled"));
}

void CegoAdminThread::medDisableAutoCorrect(CegoAdminHandler *pAH)
{
    Chain tableSet;

    pAH->getTableSet(tableSet);

    Chain dbHost;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain status;

    Chain resultMsg;
 
    _pDBMng->getDBHost(dbHost);   
    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    status = _pDBMng->getTableSetRunState(tableSet);

    if ( status == Chain(XML_BACKUP_VALUE) )
    {
	Chain msg = Chain("Tableset is already in backup mode");
	throw Exception(EXLOC, msg);
    }

    if ( dbHost != mediator )
    {
	Chain msg = Chain("Invalid mediator host ") + dbHost + Chain(" for tableset ") + tableSet;
	throw Exception(EXLOC, msg);
    }

    Chain primaryStatus;
    Chain secondaryStatus;
    
    _pDBMng->getStatusByHost(primary, primaryStatus);
    _pDBMng->getStatusByHost(secondary, secondaryStatus);

    if ( primaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Primary host ") + primary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    if ( secondaryStatus != Chain(XML_ONLINE_VALUE) )
    {
	Chain msg = Chain("Secondary host ") + secondary + Chain(" not online");
	throw Exception(EXLOC, msg);
    }
    
    if ( mediator == primary )
    {
	_pDBMng->setAutoCorrect(tableSet, false);
    }
    else
    {
	Chain msg = Chain("Auto Correct just supported in single node mode");
	throw Exception(EXLOC, msg);
    }
    pAH->sendResponse(Chain("Auto correct disabled"));    
}

CegoAdminHandler* CegoAdminThread::getSession(const Chain& role, const Chain& hostname, const Chain& user, const Chain& pwd)
{    
    Net n (  NETMNG_MSG_BUFLEN, NETMNG_SIZEBUFLEN, NETMNG_MAXSENDLEN );
    
    NetHandler *pN;

    int adminPort;
    _pDBMng->getAdminPort(adminPort);
     
    try 
    {
	pN = n.connect(hostname, adminPort);	
    }
    catch ( Exception e )
    {
	throw Exception(EXLOC, Chain("Cannot get session to ") + hostname, e);	
    }

    CegoAdminHandler* pAH = new CegoAdminHandler( _pDBMng, pN);
    
    CegoAdminHandler::ResultType res = pAH->requestSession(user, pwd, false);
    if ( res != CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	pAH->getMsg(msg);
	closeSession(pAH);

	throw Exception(EXLOC, msg);	
    }
    
    return pAH;
}

void CegoAdminThread::closeSession(CegoAdminHandler* pAH)
{
    pAH->closeSession();
    NetHandler* pN = pAH->getNetHandle();
    delete pAH;
    delete pN;
}

Element* CegoAdminThread::checkTS(const Chain& hostType, const Chain& hostName, Element *pInfo, CegoAdminHandler *pPAH,  CegoAdminHandler *pSAH)
{    
    Chain primaryRS;
    Chain secondaryRS;

    Chain primarySS;
    Chain secondarySS;

    Chain primaryMed;
    Chain secondaryMed;

    Chain primaryPrim;
    Chain secondaryPrim;

    Chain primarySec;
    Chain secondarySec;
   
    if ( pPAH )
    {
	Element *pPrimaryCheck = pPAH->getTableSetCheckInfo();
	
	if  ( pPrimaryCheck )
	{
	    primaryRS = pPrimaryCheck->getAttributeValue(XML_RUNSTATE_ATTR);
	    primarySS = pPrimaryCheck->getAttributeValue(XML_SYNCSTATE_ATTR);
	    primaryPrim = pPrimaryCheck->getAttributeValue(XML_PRIMARY_ATTR);
	    primarySec = pPrimaryCheck->getAttributeValue(XML_SECONDARY_ATTR);
	    primaryMed = pPrimaryCheck->getAttributeValue(XML_MEDIATOR_ATTR);
	}
    }
    
    if ( pSAH )
    {
	Element *pSecondaryCheck = pSAH->getTableSetCheckInfo();
	
	if ( pSecondaryCheck ) 
	{
	    secondaryRS = pSecondaryCheck->getAttributeValue(XML_RUNSTATE_ATTR);
	    secondarySS = pSecondaryCheck->getAttributeValue(XML_SYNCSTATE_ATTR);
	    secondaryPrim = pSecondaryCheck->getAttributeValue(XML_PRIMARY_ATTR);
	    secondarySec = pSecondaryCheck->getAttributeValue(XML_SECONDARY_ATTR);
	    secondaryMed = pSecondaryCheck->getAttributeValue(XML_MEDIATOR_ATTR);
	}
    }
    
    Element *pCheck =  new Element(XML_TABLESETCHECK_ELEMENT);
    Element *pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);

    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_RUNSTATE_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_RUNSTATE_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryRS);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryRS);
    pCheck->addContent(pAttrCheck);
    
    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_SYNCSTATE_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_SYNCSTATE_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primarySS);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondarySS);
    pCheck->addContent(pAttrCheck);

    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_PRIMARY_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_PRIMARY_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryPrim);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryPrim);
    pCheck->addContent(pAttrCheck);

    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_SECONDARY_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_SECONDARY_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primarySec);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondarySec);
    pCheck->addContent(pAttrCheck);

    pAttrCheck = new Element(XML_ATTRCHECK_ELEMENT);
    pAttrCheck->setAttribute(XML_CHECKATTR_ATTR, XML_MEDIATOR_ATTR);
    pAttrCheck->setAttribute(XML_MVAL_ATTR, pInfo->getAttributeValue(XML_MEDIATOR_ATTR));
    pAttrCheck->setAttribute(XML_PVAL_ATTR, primaryMed);
    pAttrCheck->setAttribute(XML_SVAL_ATTR, secondaryMed);
    pCheck->addContent(pAttrCheck);

    return pCheck;
}

Element* CegoAdminThread::getDetailedTSInfo(const Chain& tableSet)
{
    Chain tsRoot;
    Chain primary;
    Chain secondary;
    Chain mediator;
    Chain runState;
    Chain syncState;
    Chain tsTicket;
    Chain archMode;
    Chain autoCorrect;
    Chain appendMode;
    Chain tableCache;
    Chain queryCache;
    Chain cpDump;
    Chain logUser;

    Chain tableCacheFilter;
    int tableCacheMaxEntry;
    int tableCacheMaxSize;
    int tableCacheHashRange;
    
    int queryCacheMaxEntry;
    int queryCacheMaxSize;
    int queryCacheThreshold;
    int queryCacheHashRange;
    
    Chain tsInitFile;
    int checkpoint;

    primary = _pDBMng->getPrimary(tableSet);
    secondary = _pDBMng->getSecondary(tableSet);
    mediator = _pDBMng->getMediator(tableSet);
    runState = _pDBMng->getTableSetRunState(tableSet);
    syncState = _pDBMng->getTableSetSyncState(tableSet);
    tsRoot = _pDBMng->getTSRoot(tableSet);
    tsTicket = _pDBMng->getTSTicket(tableSet);
    tsInitFile = _pDBMng->getTSInitFile(tableSet);
    checkpoint = _pDBMng->getCheckpointInterval(tableSet);
    logUser = _pDBMng->getTSLogUser(tableSet);
    
    if ( _pDBMng->isArchiveMode(tableSet) )
	archMode = Chain(XML_ON_VALUE);
    else
	archMode = Chain(XML_OFF_VALUE);

    if ( _pDBMng->getAutoCorrect(tableSet) )
	autoCorrect = Chain(XML_ON_VALUE);
    else
	autoCorrect = Chain(XML_OFF_VALUE);

    if ( _pDBMng->getAppendMode(tableSet) )
	appendMode = Chain(XML_ON_VALUE);
    else
	appendMode = Chain(XML_OFF_VALUE);
    
    if ( _pDBMng->getTableCacheMode(tableSet) )
        tableCache = Chain(XML_ON_VALUE);
    else
	tableCache = Chain(XML_OFF_VALUE);

    tableCacheFilter = _pDBMng->getTableCacheFilter(tableSet);
    tableCacheMaxSize = _pDBMng->getTableCacheMaxSize(tableSet);

    tableCacheMaxEntry = _pDBMng->getTableCacheMaxEntry(tableSet);
    tableCacheHashRange = _pDBMng->getTableCacheHashRange(tableSet);

    if ( _pDBMng->getQueryCacheMode(tableSet) )
        queryCache = Chain(XML_ON_VALUE);
    else
	queryCache = Chain(XML_OFF_VALUE);

    queryCacheMaxEntry = _pDBMng->getMaxQueryCacheEntry(tableSet);
    queryCacheMaxSize = _pDBMng->getMaxQueryCacheSize(tableSet);
    queryCacheThreshold = _pDBMng->getQueryCacheThreshold(tableSet);
    queryCacheHashRange = _pDBMng->getQueryCacheHashRange(tableSet);
    
    Element *pTSInfo = new Element(XML_TABLESET_ELEMENT);
    
    pTSInfo->setAttribute(XML_NAME_ATTR, tableSet);
    pTSInfo->setAttribute(XML_RUNSTATE_ATTR, runState);
    pTSInfo->setAttribute(XML_SYNCSTATE_ATTR, syncState);
    pTSInfo->setAttribute(XML_PRIMARY_ATTR, primary);		
    pTSInfo->setAttribute(XML_SECONDARY_ATTR, secondary);	
    pTSInfo->setAttribute(XML_MEDIATOR_ATTR, mediator);		
    pTSInfo->setAttribute(XML_TSROOT_ATTR, tsRoot);
    pTSInfo->setAttribute(XML_TSTICKET_ATTR, tsTicket);
    pTSInfo->setAttribute(XML_TSINITFILE_ATTR, tsInitFile);
    pTSInfo->setAttribute(XML_CHECKPOINT_ATTR, Chain(checkpoint));
    pTSInfo->setAttribute(XML_LOGUSER_ATTR, logUser);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);
    unsigned long long clsn = _pDBMng->getCommittedLSN(tableSet);
    unsigned long long wlsn = _pDBMng->getCurrentLSN(tabSetId);
    
    unsigned long long tid = _pDBMng->getTID(tableSet);
    unsigned long long sortAreaSize = _pDBMng->getTSSortAreaSize(tabSetId);
    Chain branchId = _pDBMng->getTableSetBackupBranch(tableSet);

    if ( _pDBMng->checkPointDumpEnabled(tabSetId) )
        cpDump = Chain(XML_ON_VALUE);
    else
	cpDump = Chain(XML_OFF_VALUE);

    Chain sysFileName = _pDBMng->getSysFileName(tableSet);
    Chain tempFileName = _pDBMng->getTmpFileName(tableSet);
    
    pTSInfo->setAttribute(XML_SYSFILE_ATTR, sysFileName);
    pTSInfo->setAttribute(XML_TEMPFILE_ATTR, tempFileName);
    
    pTSInfo->setAttribute(XML_SORTAREASIZE_ATTR, Chain(sortAreaSize));
    pTSInfo->setAttribute(XML_BRANCHID_ATTR, branchId);    
    pTSInfo->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
    pTSInfo->setAttribute(XML_TID_ATTR, Chain(tid));
    pTSInfo->setAttribute(XML_LSN_ATTR, Chain(clsn));
    pTSInfo->setAttribute(XML_WRITTENLSN_ATTR, Chain(wlsn));
    pTSInfo->setAttribute(XML_ARCHMODE_ATTR, archMode);
    pTSInfo->setAttribute(XML_AUTOCORRECT_ATTR, autoCorrect);
    pTSInfo->setAttribute(XML_APPENDMODE_ATTR, appendMode);
    
    pTSInfo->setAttribute(XML_CPDUMP_ATTR, cpDump);
    
    pTSInfo->setAttribute(XML_TABLECACHE_ATTR, tableCache);
    pTSInfo->setAttribute(XML_QUERYCACHE_ATTR, queryCache);

    pTSInfo->setAttribute(XML_QUERYCACHEMAXENTRY_ATTR, Chain(queryCacheMaxEntry));
    pTSInfo->setAttribute(XML_QUERYCACHEMAXSIZE_ATTR, Chain(queryCacheMaxSize));
    pTSInfo->setAttribute(XML_QUERYCACHETHRESHOLD_ATTR, Chain(queryCacheThreshold));
    pTSInfo->setAttribute(XML_QUERYCACHEHASHRANGE_ATTR, Chain(queryCacheHashRange));
    
    pTSInfo->setAttribute(XML_TABLECACHEFILTER_ATTR, Chain(tableCacheFilter));
    pTSInfo->setAttribute(XML_TABLECACHEMAXSIZE_ATTR, Chain(tableCacheMaxSize));
    pTSInfo->setAttribute(XML_TABLECACHEMAXENTRY_ATTR, Chain(tableCacheMaxEntry));
    pTSInfo->setAttribute(XML_TABLECACHEHASHRANGE_ATTR, Chain(tableCacheHashRange));

    int totalSysPages=0;
    int usedSysPages=0;
    int totalTempPages=0;
    int usedTempPages=0;
    
    Chain status =_pDBMng->getTableSetRunState(tableSet);
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	int sysfid = tabSetId;
	int tmpfid = _pDBMng->getTmpFid(tableSet);
	
	totalSysPages = _pDBMng->getNumPages(sysfid);
	usedSysPages = _pDBMng->getNumUsedPages(sysfid, _pTabMng->getLockHandle());
	totalTempPages = _pDBMng->getNumPages(tmpfid);
	usedTempPages = _pDBMng->getNumUsedPages(tmpfid, _pTabMng->getLockHandle());	    
    }
    else
    {
	totalSysPages = _pDBMng->getSysSize(tableSet);
	totalTempPages = _pDBMng->getTmpSize(tableSet);
    }
    
    pTSInfo->setAttribute(XML_SYSTOTAL_ATTR, Chain(totalSysPages));
    pTSInfo->setAttribute(XML_SYSUSED_ATTR, Chain(usedSysPages));
    
    pTSInfo->setAttribute(XML_TEMPTOTAL_ATTR, Chain(totalTempPages));
    pTSInfo->setAttribute(XML_TEMPUSED_ATTR, Chain(usedTempPages));
    
    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> dfSizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, dfSizeList);
    
    Chain *pFileName = dfList.First();
    int *pFid = fidList.First();
    int *pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_APPFILE_VALUE));
		
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}
	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();
	pDFSize = dfSizeList.Next();
    }
    
    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_SYSFILE_VALUE));
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}
	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();
	pDFSize = dfSizeList.Next();
    }
    
    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    pDFSize = dfSizeList.First();
    
    while ( pFileName && pFid )
    {	
	Element *pF = new Element(XML_DATAFILE_ELEMENT);
	pF->setAttribute(XML_NAME_ATTR, *pFileName);
	pF->setAttribute(XML_TYPE_ATTR, Chain(XML_TEMPFILE_VALUE));
	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}
	else
	{
	    totalPages = *pDFSize;
	}
	
	pF->setAttribute(XML_NUMTOTAL_ATTR, Chain(totalPages));
	pF->setAttribute(XML_NUMUSED_ATTR, Chain(usedPages));
	
	pTSInfo->addContent(pF);
	
	pFileName = dfList.Next();
	pFid = fidList.Next();	
	pDFSize = dfSizeList.Next();
    }
    
    ListT<Chain> lfList;
    ListT<int> sizeList;
    ListT<Chain> statusList;
    
    _pDBMng->getLogFileInfo(tableSet, lfList, sizeList, statusList);	
    
    Chain *pLogName = lfList.First();
    int *pSize= sizeList.First();
    Chain *pStatus = statusList.First();
    
    while ( pLogName && pSize && pStatus )
    {
	Element *pLE = new Element(XML_LOGFILE_ELEMENT);
	pLE->setAttribute(XML_NAME_ATTR, *pLogName);
	pLE->setAttribute(XML_SIZE_ATTR, Chain(*pSize));
	
	pLE->setAttribute(XML_STATUS_ATTR, *pStatus);

	if ( *pStatus == Chain(XML_ACTIVE_VALUE) )
	{
	    pLE->setAttribute(XML_OFFSET_ATTR, Chain(_pDBMng->getLogOffset(tabSetId)));
	}
	else
	{
	    pLE->setAttribute(XML_OFFSET_ATTR, Chain(0));
	}
	
	pTSInfo->addContent(pLE);
	
	pLogName = lfList.Next();
	pSize = sizeList.Next();
	pStatus = statusList.Next();
    }
          
    return pTSInfo; 
}


Element* CegoAdminThread::getArchLogInfo(const Chain& tableSet)
{
    Element *pArchInfo = new Element(XML_ARCHINFO_ELEMENT);

    ListT<Chain> archIdList;
    ListT<Chain> archPathList;

    _pDBMng->getArchLogInfo(tableSet, archIdList, archPathList);
    
    Chain branch = _pDBMng->getTableSetBackupBranch(tableSet);
    
    Chain *pArchId = archIdList.First();
    Chain *pArchPath = archPathList.First();
    
    while ( pArchId && pArchPath)
    {	
	Element *pNAL = new Element(XML_ARCHIVELOG_ELEMENT);
	pNAL->setAttribute(XML_ARCHID_ATTR, *pArchId);
	pNAL->setAttribute(XML_ARCHPATH_ATTR, *pArchPath);

	// retrive num files for archive for archive destination
	
	Directory archDir(*pArchPath);
	ListT<Chain> fileList = archDir.list();

	int numFile=0;
	for ( Chain *pF = fileList.First(); pF != 0; pF = fileList.Next() )
	{
	    Tokenizer tok(*pF, Chain(LOGMNG_ARCHLOGSEP));
	    Chain ts,br;
	    if ( tok.nextToken(ts) )
		if ( ts == tableSet )
		    if ( tok.nextToken(br) )
			if ( br == branch )
			    numFile++;
	}
	
	pNAL->setAttribute(XML_NUMFILE_ATTR, Chain(numFile));

	// retrive num files for archive for log manager

	Chain numArchived = executeLogInfo(tableSet, *pArchPath);
	pNAL->setAttribute(XML_NUMARCHIVED_ATTR, numArchived);	
	
	pArchInfo->addContent(pNAL);

	pArchId = archIdList.Next();	
	pArchPath = archPathList.Next();
    }

    return pArchInfo;
}

void CegoAdminThread::addTableSetUsage(Element *pTSE)
{
    Chain tableSet = pTSE->getAttributeValue(XML_NAME_ATTR);
    
    int tabSetId = _pDBMng->getTabSetId(tableSet);
    
    int totalSysPages=0;
    int usedSysPages=0;
    int totalTempPages=0;
    int usedTempPages=0;
    int totalAppPages=0;
    int usedAppPages=0;
   
    Chain status =_pDBMng->getTableSetRunState(tableSet);
    
    if ( status == Chain(XML_ONLINE_VALUE) )
    {
	int sysfid = tabSetId;
	int tmpfid = _pDBMng->getTmpFid(tableSet);
	
	totalSysPages = _pDBMng->getNumPages(sysfid);
	usedSysPages = _pDBMng->getNumUsedPages(sysfid, _pTabMng->getLockHandle());
	totalTempPages = _pDBMng->getNumPages(tmpfid);
	usedTempPages = _pDBMng->getNumUsedPages(tmpfid, _pTabMng->getLockHandle());	    
    }
    
    ListT<Chain> dfList;
    ListT<int> fidList;
    ListT<int> dfSizeList;
    
    _pDBMng->getDataFileInfo(tableSet, Chain(XML_APPFILE_VALUE), dfList, fidList, dfSizeList);
    
    Chain *pFileName = dfList.First();
    int *pFid = fidList.First();
    
    while ( pFileName && pFid )
    {	
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalAppPages+=totalPages;
	usedAppPages+=usedPages;

	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_SYSFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    
    while ( pFileName && pFid )
    {		
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalSysPages+=totalPages;
	usedSysPages+=usedPages;
		
	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    dfList.Empty();
    fidList.Empty();
    dfSizeList.Empty();

    _pDBMng->getDataFileInfo(tableSet, Chain(XML_TEMPFILE_VALUE), dfList, fidList, dfSizeList);
    pFileName = dfList.First();
    pFid = fidList.First();
    
    while ( pFileName && pFid )
    {		
	int totalPages = 0;
	int usedPages = 0;
	
	if ( status == Chain(XML_ONLINE_VALUE) )
	{
	    totalPages = _pDBMng->getNumPages(*pFid);
	    usedPages = _pDBMng->getNumUsedPages(*pFid, _pTabMng->getLockHandle());
	}

	totalTempPages+=totalPages;
	usedTempPages+=usedPages;
	       
	pFileName = dfList.Next();
	pFid = fidList.Next();
    }

    int percentSysUsage = 0;
    if ( totalSysPages > 0 )
	percentSysUsage = usedSysPages  / ( totalSysPages  / 100 );

    int percentTempUsage = 0;
    if ( totalTempPages > 0 )
	percentTempUsage = usedTempPages  / ( totalTempPages  / 100 );

    int percentAppUsage = 0;
    if ( totalAppPages > 0 )
	percentAppUsage = usedAppPages  / ( totalAppPages  / 100 );

    pTSE->setAttribute(XML_SYSUSAGE_ATTR, Chain(percentSysUsage) + Chain("%"));  
    pTSE->setAttribute(XML_TEMPUSAGE_ATTR, Chain(percentTempUsage) + Chain("%"));  
    pTSE->setAttribute(XML_APPUSAGE_ATTR, Chain(percentAppUsage) + Chain("%"));
}

Element* CegoAdminThread::getTransactionInfo(const Chain& tableSet)
{
    Element *pTA = new Element(XML_TAINFO_ELEMENT);

    CegoTransactionManager *pTM = _pTabMng->getTransactionManager();

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    ListT<Chain> rboList;    

    _pTabMng->getObjectList(tabSetId, CegoObject::RBSEG, rboList);
    
    Chain *pRBO = rboList.First();
    while ( pRBO )
    {
	Element *pTE = new Element(XML_TAENTRY_ELEMENT);
	
	int numOp;
	pTM->getTransactionInfo(tabSetId, *pRBO, numOp);
	pTE->setAttribute(XML_RBO_ATTR, *pRBO);
	pTE->setAttribute(XML_NUMOP_ATTR, Chain(numOp));
	
	pTA->addContent(pTE);
	pRBO = rboList.Next();
    }   
    return pTA;
}

Element* CegoAdminThread::getBUStatInfo(const Chain& tableSet)
{
    Chain status = _pDBMng->getTableSetRunState(tableSet);
    if ( status != Chain(XML_ONLINE_VALUE) )
    {
	throw Exception(EXLOC, "Tableset must be online to list backup statistics");
    }

    Element *pBUS = new Element(XML_BUSINFO_ELEMENT);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    CegoTableObject boe;    
    _pTabMng->getObject(tabSetId, Chain(SYSTAB_BUSTAT_ID), CegoObject::SYSTEM, boe);
    ListT<CegoField> fvl = boe.getSchema();

    CegoObjectCursor* pC = _pTabMng->getObjectCursor(tabSetId, SYSTAB_BUSTAT_ID, SYSTAB_BUSTAT_ID, CegoObject::SYSTEM);
    
    CegoDataPointer dp;    
    int len;
    char *pc;
    
    pc = (char*)pC->getFirst(len, dp);		
    
    while ( pc && len > 0 )
    {
	unsigned long long tid;
	unsigned long long tastep;	    
	CegoTupleState ts;
       
	int toff = CegoQueryHelper::decodeTupleHeader(tid, tastep, ts, pc);

	char* tp = pc + toff;
	int tlen = len - toff;
	
	CegoQueryHelper::decodeFVL(fvl, tp, tlen);			    

	Element *pBUE = new Element(XML_BUENTRY_ELEMENT);
		
	CegoField *pF = fvl.First();
	while ( pF )	    
	{
	    if ( pF->getAttrName() == Chain(SYSTAB_TS_ATTR) )		
		pBUE->setAttribute(XML_TIMESTAMP_ATTR, pF->getValue().valAsChain());
	    else if ( pF->getAttrName() == Chain(SYSTAB_BUINFO_ATTR) )
		pBUE->setAttribute(XML_BUINFO_ATTR, pF->getValue().valAsChain());
	    else if ( pF->getAttrName() == Chain(SYSTAB_BUMSG_ATTR) )
	    {
		if ( pF->getValue().isNull() )
		    pBUE->setAttribute(XML_BUMSG_ATTR, Chain("none"));
		else			 
		    pBUE->setAttribute(XML_BUMSG_ATTR, pF->getValue().valAsChain());   
	    }
	    pF = fvl.Next();
	}
	
	pBUS->addContent(pBUE);

	pc = (char*)pC->getNext(len, dp);
    }

    pC->abort();
    delete pC;
   
    return pBUS;
}

Element* CegoAdminThread::getBackupInfo(const Chain& tableSet, const Chain& pit)
{
    Element *pBI = new Element(XML_BACKUPLIST_ELEMENT);

    ListT<Chain> backupList = executeListBackup(tableSet, pit);
        
    Chain *pBackup = backupList.First();

    while ( pBackup )
    {
	Element *pBE = new Element(XML_BACKUP_ELEMENT);

	Tokenizer t(*pBackup, Chain(","));

	Chain backupID;
	Chain backupTS;

	if ( t.nextToken(backupID) == false )
	{
	    throw Exception(EXLOC, Chain("Invalid backup list from backup manager"));
	}

	if ( t.nextToken(backupTS) == false )
	{
	    throw Exception(EXLOC, Chain("Invalid backup list from backup manager"));
	}
	
	pBE->setAttribute(XML_BACKUPID_ATTR, backupID);
	pBE->setAttribute(XML_BACKUPTS_ATTR, backupTS);
	pBI->addContent(pBE);

	pBackup = backupList.Next();
    }

    return pBI;
}

Element* CegoAdminThread::getObjectInfo(const Chain& tableSet)
{
    Element *pOI = new Element(XML_OBJLIST_ELEMENT);

    int tabSetId = _pDBMng->getTabSetId(tableSet);

    ListT<Chain> tableList;
    _pTabMng->getObjectList(tabSetId, CegoObject::TABLE, tableList);

    Chain *pTableName = tableList.First();

    while ( pTableName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_TABOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pTableName);	
	pOI->addContent(pOE);

	pTableName = tableList.Next();
    }

    ListT<Chain> idxList;
    _pTabMng->getObjectList(tabSetId, CegoObject::AVLTREE, idxList);

    Chain *pIndexName = idxList.First();

    while ( pIndexName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_IDXOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pIndexName);	
	pOI->addContent(pOE);

	pIndexName = idxList.Next();
    }

    ListT<Chain> keyList;
    _pTabMng->getObjectList(tabSetId, CegoObject::FKEY, keyList);

    Chain *pKeyName = keyList.First();

    while ( pKeyName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_FKEYOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pKeyName);	
	pOI->addContent(pOE);

	pKeyName = keyList.Next();
    }

    ListT<Chain> checkList;
    _pTabMng->getObjectList(tabSetId, CegoObject::CHECK, checkList);

    Chain *pCheckName = checkList.First();

    while ( pCheckName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_CHECKOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pCheckName);	
	pOI->addContent(pOE);

	pCheckName = checkList.Next();
    }

    ListT<Chain> viewList;
    _pTabMng->getObjectList(tabSetId, CegoObject::VIEW, viewList);

    Chain *pViewName = viewList.First();

    while ( pViewName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_VIEWOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pViewName);	
	pOI->addContent(pOE);

	pViewName = viewList.Next();
    }

    ListT<Chain> procList;
    _pTabMng->getObjectList(tabSetId, CegoObject::PROCEDURE, procList);

    Chain *pProcName = procList.First();

    while ( pProcName )
    {
	Element *pOE = new Element(XML_OBJ_ELEMENT);
	
	pOE->setAttribute(XML_OBJTYPE_ATTR, XML_PROCOBJ_VALUE);
	pOE->setAttribute(XML_OBJNAME_ATTR, *pProcName);	
	pOI->addContent(pOE);

	pProcName = procList.Next();
    }
    
    return pOI;
}


Chain CegoAdminThread::executeBackup(CegoAdminHandler* pAH, const Chain& tableSet, const Chain& msg)
{
    ListT<Chain> fileList;

    Chain tsTicket = _pDBMng->getTSTicket(tableSet);
    fileList.Insert(tsTicket);
    
    Chain sysFileName = _pDBMng->getSysFileName(tableSet);
    fileList.Insert(sysFileName);

    Chain tempFileName = _pDBMng->getTmpFileName(tableSet);
    fileList.Insert(tempFileName);
    
    ListT<Chain> dataFileList;
    _pDBMng->getDataFileInfo(tableSet, dataFileList);
    
    Chain *pDataFileName = dataFileList.First();
    while ( pDataFileName )
    {
	fileList.Insert(*pDataFileName);
	pDataFileName = dataFileList.Next();
    }

    Chain backupProg = _pDBMng->getBackupProg();
    
    if ( backupProg == Chain() )
    {
	throw Exception(EXLOC, Chain("No backup manager defined"));	
    }

    if ( File::exists(backupProg) == false )
    {
	throw Exception(EXLOC, Chain("Backup manager program <") + backupProg + Chain("> does not exist"));
    }

    Chain branch = _pDBMng->getTableSetBackupBranch(tableSet);
    
    Chain backupCommand = backupProg + Chain(" -t ") + tableSet
	+ Chain(" -b \"") + branch + Chain("\"")
	+ Chain(" -a \"");
    
    Chain *pFileName = fileList.First();
    while ( pFileName )
    {
	backupCommand += *pFileName;
	pFileName = fileList.Next();
	if ( pFileName )
	    backupCommand += Chain(" ");
    }
    backupCommand += Chain("\"");
    
    BackupStream buStream(pAH);

    buStream.open(backupCommand);
    
    return Chain("OK");
}

ListT<Chain> CegoAdminThread::executeListBackup(const Chain& tableSet, const Chain& pit)
{
    Chain backupProg = _pDBMng->getBackupProg();

    if ( backupProg == Chain() )
    {
	throw Exception(EXLOC, Chain("No backup manager defined"));	
    }
    
    if ( File::exists(backupProg) == false )
    {
	throw Exception(EXLOC, Chain("Backup manager program <") + backupProg + Chain("> does not exist"));	
    }

    Chain branch = _pDBMng->getTableSetBackupBranch(tableSet);
    Chain backupCommand = backupProg + Chain(" -t ") + tableSet;
    if ( pit != Chain() )
	backupCommand += Chain(" -p \"") + pit + Chain("\"");
    backupCommand += Chain(" -b \"") + branch + Chain("\"")
	+ Chain(" -l ");
    
    Chain backupInfo = CommandExecuter::executeStream(backupCommand);
    Tokenizer t(backupInfo, Chain("\n"));
    Chain backupId;
    ListT<Chain> backupList;
    while ( t.nextToken(backupId) )
    {
	backupList.Insert(backupId);
    }
    return backupList;
}

Chain CegoAdminThread::executeRestore(CegoAdminHandler* pAH, const Chain& tableSet, const Chain& backupId)
{    
    Chain backupProg = _pDBMng->getBackupProg();

    if ( backupProg == Chain() )
    {
	throw Exception(EXLOC, Chain("No backup manager defined"));	
    }
	
    if ( File::exists(backupProg) == false )
    {
	throw Exception(EXLOC, Chain("Backup manager program <") + backupProg + Chain("> does not exist"));	
    }

    Chain branch = _pDBMng->getTableSetBackupBranch(tableSet);
    
    Chain restoreCommand = backupProg + Chain(" -t ") + tableSet
	+ Chain(" -b \"") + branch + Chain("\"")
	+ Chain(" -r ") + backupId;

    BackupStream buStream(pAH);

    buStream.open(restoreCommand);

    return Chain("OK");
}

Chain CegoAdminThread::executeLogInfo(const Chain& tableSet, const Chain& archPath)
{    

    Chain logmngProg = _pDBMng->getArchRestoreProg();
    Chain branch = _pDBMng->getTableSetBackupBranch(tableSet);
    
    if ( logmngProg == Chain() )
    {
	throw Exception(EXLOC, Chain("No log manager defined"));	
    }
	
    if ( File::exists(logmngProg) == false )
    {
	throw Exception(EXLOC, Chain("Log manager program <") + logmngProg + Chain("> does not exist"));	
    }

    LogInfoStream infoStream;
    
    try
    {
	Chain infoCommand = logmngProg + Chain(" -m list -t ") + tableSet
	    + Chain(" -p \"") + archPath + Chain("\"")
	    + Chain(" -b \"") + branch + Chain("\"")
	    + Chain(" -o numarchived") ;
		
	infoStream.open(infoCommand);	
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	
	Chain m = Chain("Log Manager Error ") + msg;
	throw Exception(EXLOC, m);
    }

    return infoStream.getResult().cutTrailing(" \n");
}
