///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoBTreeObject.cc 
// ------------------
// Cego btree object class implementation
//     
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoBTreeObject
// 
// Description: Binary tree index structure
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoBTreeObject.h"
#include "CegoTypeConverter.h"
#include "CegoXMLdef.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoBTreeObject::CegoBTreeObject()
{
}

CegoBTreeObject::CegoBTreeObject(const CegoBTreeObject& oe) : CegoContentObject(oe)
{
    _pageId = oe._pageId;
    _relevance = oe._relevance;
}

CegoBTreeObject::CegoBTreeObject(int tabSetId, CegoObject::ObjectType type, const Chain& objName ) : CegoContentObject(tabSetId, type, objName)
{
    _pageId = 0;
    _relevance = 0;
}

CegoBTreeObject::CegoBTreeObject(int tabSetId, CegoObject::ObjectType type,  const Chain& objName, const ListT<CegoField>& schema, const Chain& tabName) : CegoContentObject(tabSetId, type, objName, tabName, schema)
{
    _pageId = 0;
    _relevance=0;
}

CegoBTreeObject::~CegoBTreeObject()
{   
}

void CegoBTreeObject::setSchema(const ListT<CegoField>& schema)
{
    _schema = schema;
}

void CegoBTreeObject::setDataPageId(PageIdType pageId)
{
    _pageId = pageId;
}

PageIdType CegoBTreeObject::getDataPageId() const
{
    return _pageId;
}

void CegoBTreeObject::increaseRelevance()
{
    _relevance++;
}

void CegoBTreeObject::decreaseRelevance()
{
    _relevance--;
}

int CegoBTreeObject::getRelevance() const
{
    return _relevance;
}

int CegoBTreeObject::getEntrySize() const
{
    int entrySize = CegoContentObject::getBaseContentSize();

    entrySize = entrySize + sizeof(PageIdType); // pageId
    entrySize = entrySize + sizeof(int); // relevance
        
    return entrySize;
}

void CegoBTreeObject::encode(char *buf) const
{
    char* bufPtr = buf;

    int entrySize = getEntrySize();

    CegoContentObject::encodeBaseContent(bufPtr, entrySize);
    bufPtr += CegoContentObject::getBaseContentSize();

    memcpy(bufPtr, &_pageId, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(bufPtr, &_relevance, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

void CegoBTreeObject::decode(char *buf)
{
    char* bufPtr = buf;

    int size;

    CegoContentObject::decodeBaseContent(bufPtr, size);
    bufPtr += CegoContentObject::getBaseContentSize();
    
    memcpy(&_pageId, bufPtr, sizeof(PageIdType));
    bufPtr = bufPtr + sizeof(PageIdType);

    memcpy(&_relevance, bufPtr, sizeof(int));
    bufPtr = bufPtr + sizeof(int);
}

bool CegoBTreeObject::isValid()
{
    return _pageId != 0;
}

CegoBTreeObject& CegoBTreeObject::operator = ( const CegoBTreeObject& oe) 
{
    CegoContentObject::operator=(oe);
    _pageId = oe._pageId;
    _relevance = oe._relevance;
    return (*this);
}

bool CegoBTreeObject::operator == ( const CegoBTreeObject& oe)
{
    return CegoContentObject::operator==(oe);
}

Chain CegoBTreeObject::getId(CegoProcBlock *pBlock) const
{
    return getTabName() + Chain("@") + getTableSet() + Chain("@") + getName();
}

Chain CegoBTreeObject::toChain(int defTabSetId) const
{
    if  ( defTabSetId == getTabSetId() )
	return getTabName() + Chain(" ") + getName();
    else
	return getTabName() + Chain("@") + getTableSet() + Chain(" ") + getName();
}

Chain CegoBTreeObject::getInfo() const
{
    Chain s;

    CegoField *pF = _schema.First();
    while (pF)
    {	
	s += Chain("Attr=") + pF->getAttrName()
	    + Chain(" ") + Chain(CEGO_TYPE_MAP[pF->getType()]) + Chain("[") + Chain(pF->getLength()) + Chain("]\n");
	pF = _schema.Next();	    
    }
    return s;
}

CegoContentObject* CegoBTreeObject::clone(bool isAttrRef)
{
    CegoBTreeObject *pClone = new CegoBTreeObject(*this);
    return (pClone);
}
