///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoJDBCInfo.cc
// ---------------
// Cego JDBC information handler
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoJDBCInfo
// 
// Description: JDBC request handling
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include "CegoJDBCInfo.h"
#include "CegoField.h"
#include "CegoOutput.h"
#include "CegoTypeConverter.h"

CegoJDBCInfo::CegoJDBCInfo(CegoDistManager* pTabMng, CegoDbHandler* pDbHandle)
{
    _pTabMng = pTabMng;
    _pDbHandle = pDbHandle;
}

CegoJDBCInfo::~CegoJDBCInfo()
{
}

void CegoJDBCInfo::checkAccess() const
{
    if ( _pTabMng->getDBMng()->verifyJDBC(_pTabMng->getUser()) == false )
    {
	throw Exception(EXLOC, Chain("User not authorized for JDBC information"));	
    }
}

void CegoJDBCInfo::handleRequest(const ListT<Chain>& argList) const
{
    Chain *pReq = argList.First();

    if ( pReq )
    {
	if ( *pReq == Chain("catalog") )
	{
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    ListT<CegoFieldValue> fa;
	    fa.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName() ) );
	    o.rowOut(fa);
	    
	    o.tailOut();
	}
	else if ( *pReq == Chain("schema") )
	{	    
	    checkAccess();

	    ListT<Chain> tsList;
	    _pTabMng->getDBMng()->getTableSetList(tsList);
	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    Chain *pS = tsList.First();
	    while ( pS )
	    {
		ListT<CegoFieldValue> fa;
		fa.Insert( CegoFieldValue(VARCHAR_TYPE, *pS) );
		o.rowOut(fa);
		pS = tsList.Next();
	    }
	    o.tailOut();
	}
	else if ( *pReq == Chain("tabletype") )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "TABLE_TYPE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    ListT<CegoFieldValue> t1;
	    t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("TABLE")));
	    o.rowOut(t1);
	    
	    ListT<CegoFieldValue> t2;
	    t2.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("VIEW")));
	    o.rowOut(t2);	    	    
	    o.tailOut();
	}
	else if ( *pReq == Chain("tab") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();

	    if ( pTS )
	    {		
		ListT<CegoField> schema;
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN)); 
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_TYPE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "REMARKS", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SELF_REFERENCING_COL_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "REF_GENERATION", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();

		ListT<Chain> tabList;
		_pTabMng->getDistObjectList(*pTS, CegoObject::TABLE, tabList);

		Chain *pS = tabList.First();
		while ( pS )
		{
		    ListT<CegoFieldValue> f1;
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("TABLE")));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("Cego table")));
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    
		    o.rowOut(f1);
		    pS = tabList.Next();
		}
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }	
	}
	else if ( *pReq == Chain("vie") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();

	    if ( pTS )
	    {		
		ListT<CegoField> schema;
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN)); 
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_TYPE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "REMARKS", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SELF_REFERENCING_COL_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "REF_GENERATION", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();

		ListT<Chain> viewList;
		_pTabMng->getDistObjectList(*pTS, CegoObject::VIEW, viewList);
				
		Chain *pS = viewList.First();
		while ( pS )
		{
		    ListT<CegoFieldValue> f1;
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("VIEW")));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("Cego view")));
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    
		    o.rowOut(f1);
		    pS = viewList.Next();
		}
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }		
	}
	else if ( *pReq == Chain("proc") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();

	    if ( pTS )
	    {
		ListT<Chain> objList;
		_pTabMng->getDistObjectList(*pTS, CegoObject::PROCEDURE, objList);
				
		ListT<CegoField> schema;
		schema.Insert( CegoField("JDBC", "JDBC", "PROCEDURE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PROCEDURE_SCHEMA", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PROCEDURE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "RESERVED1", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "RESERVED2", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "RESERVED3", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "REMARKS", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PROCEDURE_TYPE", SMALLINT_TYPE, sizeof(short)));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();
		
		Chain *pS = objList.First();
		while ( pS )
		{
		    ListT<CegoFieldValue> f1;
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pS));
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue());
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, "Cego stored procedure"));
		    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0));
		    o.rowOut(f1);
		    pS = objList.Next();
		}
		
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("pkey") )
	{
	    checkAccess();
	    
	    Chain *pTS = argList.Next();
	    Chain *pTable = argList.Next();
	    
	    if ( pTS  && pTable)
	    {		
		ListT<CegoTableObject> idxList;
		ListT<CegoBTreeObject> btreeList;
		ListT<CegoKeyObject> keyList;
		ListT<CegoCheckObject> checkList;
		ListT<CegoTriggerObject> triggerList;
		ListT<CegoAliasObject> aliasList;
		int numInvalid;

		_pTabMng->getDistObjectByTableList(*pTS, *pTable, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);
		
		ListT<CegoField> schema;
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "COLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "KEQ_SEQ", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PK_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();
		
		CegoTableObject *pIdx = idxList.First();
		while ( pIdx )
		{		    
		    if ( pIdx->getType() == CegoObject::PAVLTREE )
		    {
			CegoTableObject oe;
			_pTabMng->getDistObject(*pTS, pIdx->getName(), CegoObject::AVLTREE, oe);
			
			int i=1;
			CegoField *pF = oe.getSchema().First();
			while ( pF )
			{
			    ListT<CegoFieldValue> f1;
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, oe.getTabName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, i));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pIdx->getName()));
			    
			    o.rowOut(f1);
			    
			    i++;
			    pF = oe.getSchema().Next();			    
			}
		    }
		    pIdx = idxList.Next();
		    
		    CegoBTreeObject *pBTree = btreeList.First();
		    while ( pBTree )
		    {			
			if ( pBTree->getType() == CegoObject::PBTREE )
			{
			    CegoBTreeObject oe;
			    _pTabMng->getDistObject(*pTS, pBTree->getName(), CegoObject::BTREE, oe);
			    
			    int i=1;
			    CegoField *pF = oe.getSchema().First();
			    while ( pF )
			    {
				ListT<CegoFieldValue> f1;
				f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
				f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
				f1.Insert( CegoFieldValue(VARCHAR_TYPE, oe.getTabName()));
				f1.Insert( CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
				f1.Insert( CegoFieldValue(SMALLINT_TYPE, i));
				f1.Insert( CegoFieldValue(VARCHAR_TYPE, pBTree->getName()));
				
				o.rowOut(f1);
				
				i++;
				pF = oe.getSchema().Next();				
			    }
			}
			pBTree = btreeList.Next();			
		    }
		}
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("ikey") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();
	    Chain *pTable = argList.Next();

	    if ( pTS  && pTable)
	    {
		ListT<CegoTableObject> idxList;
		ListT<CegoBTreeObject> btreeList;
		ListT<CegoKeyObject> keyList;
		ListT<CegoCheckObject> checkList;
		ListT<CegoTriggerObject> triggerList;
		ListT<CegoAliasObject> aliasList;
		int numInvalid;

		_pTabMng->getDistObjectByTableList(*pTS, *pTable, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);

		ListT<CegoField> schema;
 		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKCOLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKCOLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "KEQ_SEQ", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "UPADATE_RULE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "DELETE_RULE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FK_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PK_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "DEFERRABILITY", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();
		
		CegoKeyObject *pKey = keyList.First();
		while ( pKey )
		{		    
		    if ( pKey->getType() == CegoObject::FKEY )
		    {
			CegoKeyObject ko;
			_pTabMng->getDistObject(*pTS, pKey->getName(), CegoObject::FKEY, ko);
			
			int i=1;
			CegoField *pKF = ko.getKeySchema().First();
			CegoField *pRF = ko.getRefSchema().First();
			while ( pKF && pRF)
			{
			    ListT<CegoFieldValue> f1;
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getTabName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pKF->getAttrName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getRefTable()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pRF->getAttrName()));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, i));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // update rule
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // delete rule
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getName()));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // deferrability
			    
			    o.rowOut(f1);
			    
			    i++;
			    pKF = ko.getKeySchema().Next();
			    pRF = ko.getRefSchema().Next();			    
			}
		    }
		    pKey = keyList.Next();
		}
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("ekey") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();
	    Chain *pTable = argList.Next();

	    if ( pTS  && pTable)
	    {
		ListT<CegoTableObject> idxList;
		ListT<CegoBTreeObject> btreeList;
		ListT<CegoKeyObject> keyList;
		ListT<CegoCheckObject> checkList;
		ListT<CegoTriggerObject> triggerList;
		ListT<CegoAliasObject> aliasList;
		int numInvalid;

		_pTabMng->getDistObjectByTableList(*pTS, *pTable, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);

		ListT<CegoField> schema;

 		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKTABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PKCOLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKTABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FKCOLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "KEQ_SEQ", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "UPADATE_RULE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "DELETE_RULE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "FK_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "PK_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "DEFERRABILITY", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();

		CegoKeyObject *pKey = keyList.First();
		while ( pKey )
		{		    
		    if ( pKey->getType() == CegoObject::FKEY )
		    {
			CegoKeyObject ko;
			_pTabMng->getDistObject(*pTS, pKey->getName(), CegoObject::FKEY, ko);
			
			int i=1;
			CegoField *pKF = ko.getKeySchema().First();
			CegoField *pRF = ko.getRefSchema().First();
			while ( pKF && pRF)
			{
			    ListT<CegoFieldValue> f1;
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getTabName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pKF->getAttrName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getRefTable()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pRF->getAttrName()));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, i));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // update rule
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // delete rule
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getName()));
			    f1.Insert( CegoFieldValue(VARCHAR_TYPE, ko.getName()));
			    f1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // deferrability
			    
			    o.rowOut(f1);
			    
			    i++;
			    pKF = ko.getKeySchema().Next();
			    pRF = ko.getRefSchema().Next();
			}
		    }
		    pKey = keyList.Next();
		}
		
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("idx") || *pReq == Chain("uidx") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();
	    Chain *pTable = argList.Next();

	    if ( pTS  && pTable)
	    {
		ListT<CegoTableObject> idxList;
		ListT<CegoBTreeObject> btreeList;
		ListT<CegoKeyObject> keyList;
		ListT<CegoCheckObject> checkList;
		ListT<CegoTriggerObject> triggerList;
		ListT<CegoAliasObject> aliasList;
		int numInvalid;

		_pTabMng->getDistObjectByTableList(*pTS, *pTable, idxList, btreeList, keyList, checkList, triggerList, aliasList, numInvalid);

		ListT<CegoField> schema;

 		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "NON_UNIQUE", BOOL_TYPE, sizeof(char)));
		schema.Insert( CegoField("JDBC", "JDBC", "INDEX_QUALIFIER", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "INDEX_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE", SMALLINT_TYPE, sizeof(short)));
		schema.Insert( CegoField("JDBC", "JDBC", "ORDINAL_POSITION", SMALLINT_TYPE, sizeof(short)));
		schema.Insert( CegoField("JDBC", "JDBC", "COLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "ASC_OR_DESC", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "CARDINALITY", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "PAGES", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "FILTER_CONDITION", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();

		CegoTableObject *pIdx = idxList.First();
		while ( pIdx )
		{	    
		    CegoTableObject io;
		    _pTabMng->getDistObject(*pTS, pIdx->getName(), CegoObject::AVLTREE, io);
		    
		    int i=1;
		    CegoField *pIF = io.getSchema().First();
		    while ( pIF)
		    {
			ListT<CegoFieldValue> f1;
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, io.getTabName()));
			
			if ( pIdx->getType() == CegoObject::AVLTREE )
			    f1.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // index is non-unique
			else
			    f1.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // pindex and uindex are unique
			
			f1.Insert( CegoFieldValue()); // index catalog
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, io.getName())); // index name
			f1.Insert( CegoFieldValue(SMALLINT_TYPE, 3)); // tableIndexOther
			f1.Insert( CegoFieldValue(SMALLINT_TYPE, i)); // column pos
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, pIF->getAttrName())); // column name
			f1.Insert( CegoFieldValue()); // sort sequence
			f1.Insert( CegoFieldValue(INT_TYPE, 0)); // cardinality - not supported
			f1.Insert( CegoFieldValue(INT_TYPE, 0)); // index pages - not supported
			f1.Insert( CegoFieldValue()); // filter condition
					      					      
			o.rowOut(f1);
			
			i++;
			pIF = io.getSchema().Next();
		    }
		    pIdx = idxList.Next();
		}

		CegoBTreeObject *pBTree = btreeList.First();
		while ( pBTree )
		{		   
		    CegoBTreeObject bto;
		    _pTabMng->getDistObject(*pTS, pBTree->getName(), CegoObject::BTREE, bto);
		    
		    int i=1;
		    CegoField *pIF = bto.getSchema().First();
		    while ( pIF)
		    {
			ListT<CegoFieldValue> f1;
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, bto.getTabName()));
			
			if ( pBTree->getType() == CegoObject::BTREE )
			    f1.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // index is non-unique
			else
			    f1.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // pindex and uindex are unique
			
			f1.Insert( CegoFieldValue()); // index catalog
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, bto.getName())); // index name
			f1.Insert( CegoFieldValue(SMALLINT_TYPE, 3)); // tableIndexOther
			f1.Insert( CegoFieldValue(SMALLINT_TYPE, i)); // column pos			
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, pIF->getAttrName())); // column name
			f1.Insert( CegoFieldValue()); // sort sequence
			f1.Insert( CegoFieldValue(INT_TYPE, 0)); // cardinality - not supported
			f1.Insert( CegoFieldValue(INT_TYPE, 0)); // index pages - not supported
			f1.Insert( CegoFieldValue()); // filter condition
					      					      
			o.rowOut(f1);
			
			i++;
			pIF = bto.getSchema().Next();
		    }
		    pBTree = btreeList.Next();
		}		
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("colinfo") )
	{
	    checkAccess();

	    Chain *pTS = argList.Next();
	    Chain *pTable = argList.Next();

	    if ( pTS && pTable )
	    {
		CegoTableObject oe;

		if ( _pTabMng->distObjectExists(*pTS, *pTable, CegoObject::TABLE) )
		    _pTabMng->getDistObject(*pTS, *pTable, CegoObject::TABLE, oe);
		else if ( _pTabMng->distObjectExists( *pTS, *pTable, CegoObject::VIEW) )
		    _pTabMng->getDistObject(*pTS, *pTable, CegoObject::VIEW, oe);		    
		else
		{
		    Chain msg = "Unknown object " + *pTable;
		    throw Exception (EXLOC, msg);
		} 
		
		ListT<CegoField> schema;
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_CAT", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_SCHEM", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "TABLE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "COLUMN_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "DATA_TYPE", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "TYPE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "COLUMN_SIZE", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "BUFFER_LENGTH", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "DECIMAL_DIGITS", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "NUM_PREC_RADIX", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "NULLABLE", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "REMARKS", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "COLUMN_DEF", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SQL_DATA_TYPE", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "SQL_DATETIME_SUB", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "CHAR_OCTET_LENGTH",INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "ORDINAL_POSITION", INT_TYPE, sizeof(int)));
		schema.Insert( CegoField("JDBC", "JDBC", "IS_NULLABLE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SCOPE_CATLOG", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SCOPE_SCHEMA", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SCOPE_TABLE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SCOPE_TABLE", VARCHAR_TYPE, MAX_OBJNAME_LEN));
		schema.Insert( CegoField("JDBC", "JDBC", "SOURCE_DATA_TYPE", SMALLINT_TYPE, sizeof(short)));
		
		CegoOutput o(schema);
		if ( _pDbHandle )
		    o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
		
		o.headOut();
		
		int i=1;
		CegoField *pF = oe.getSchema().First();
		while ( pF )
		{
		    ListT<CegoFieldValue> f1;
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, _pTabMng->getDBMng()->getDBName()));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTS));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, *pTable));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, pF->getAttrName()));
		    f1.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[pF->getType()]));
		    f1.Insert( CegoFieldValue(VARCHAR_TYPE, CEGO_TYPE_MAP[pF->getType()]));
		    f1.Insert( CegoFieldValue(INT_TYPE, pF->getLength()));
		    f1.Insert( CegoFieldValue()); // BUFFER_LENGTH not used
		    f1.Insert( CegoFieldValue(INT_TYPE, 0 ));
		    f1.Insert( CegoFieldValue(INT_TYPE, 10 ));
		    f1.Insert( CegoFieldValue(INT_TYPE, 0 ));
		    f1.Insert( CegoFieldValue()); // REMARKS
		    f1.Insert( CegoFieldValue()); // COLUMN_DEF not used
		    f1.Insert( CegoFieldValue()); // SQL_DATA_TYPE
		    f1.Insert( CegoFieldValue()); // SQL_DATETIME_SUB
		    f1.Insert( CegoFieldValue(INT_TYPE, 0 ));
		    f1.Insert( CegoFieldValue(INT_TYPE, i ));

		    if ( pF->isNullable() )
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, "YES"));
		    else
			f1.Insert( CegoFieldValue(VARCHAR_TYPE, "NO"));
		    
		    f1.Insert( CegoFieldValue()); // SCOPE_CATLOG
		    f1.Insert( CegoFieldValue()); // SCOPE_SCHEMA
		    f1.Insert( CegoFieldValue()); // SCOPE_TABLE
		    f1.Insert( CegoFieldValue()); // SOURCE_DATA_TYPE

		    o.rowOut(f1);
		    i++;
		    pF = oe.getSchema().Next();
		}		
		o.tailOut();
	    }
	    else
	    {
		throw Exception(EXLOC, "Missing args for JDBC info");
	    }
	}
	else if ( *pReq == Chain("prodname") )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "PROD_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    ListT<CegoFieldValue> t1;
	    t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain(CEGO_PRODUCT)));
	    o.rowOut(t1);
	    
	    o.tailOut();
	    
	}
	else if ( *pReq == Chain("prodversion") )
	{	   
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "PROD_VERSION", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    ListT<CegoFieldValue> t1;
	    t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain(CEGO_VERSION)));
	    o.rowOut(t1);	    
	    o.tailOut();
	}
	else if ( *pReq == Chain("commitmode") )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "COMMIT_MODE", BOOL_TYPE, sizeof(char)));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();

	    ListT<CegoFieldValue> t1;
	    if ( _pTabMng->getAutoCommit() )
		t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("true")));
	    else
		t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("false")));
	    o.rowOut(t1);	    
	    o.tailOut();	    
	}
	else if ( *pReq == Chain("isolevel") )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "ISOLATION", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    ListT<CegoFieldValue> t1;
	    if ( _pTabMng->getIsolationLevel() == CegoTableManager::READ_COMMITTED)
		t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("read_committed")));
	    else if ( _pTabMng->getIsolationLevel() == CegoTableManager::READ_UNCOMMITTED)
		t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("read_uncommitted")));

	    o.rowOut(t1);	    
	    o.tailOut();	    
	}
	else if ( *pReq == Chain("typeinfo") )
	{	    
	    ListT<CegoField> schema;
	    schema.Insert( CegoField("JDBC", "JDBC", "TYPE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    schema.Insert( CegoField("JDBC", "JDBC", "DATA_TYPE", INT_TYPE, sizeof(int)));
	    schema.Insert( CegoField("JDBC", "JDBC", "PRECISION", INT_TYPE, sizeof(int)));
	    schema.Insert( CegoField("JDBC", "JDBC", "LITERAL_PREFIX", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    schema.Insert( CegoField("JDBC", "JDBC", "LITERAL_SUFFIX", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    schema.Insert( CegoField("JDBC", "JDBC", "CREATE_PARAMS", VARCHAR_TYPE, MAX_OBJNAME_LEN));	    
	    schema.Insert( CegoField("JDBC", "JDBC", "NULLABLE", SMALLINT_TYPE, sizeof(short)));
	    schema.Insert( CegoField("JDBC", "JDBC", "CASE_SENSITIVE", BOOL_TYPE, sizeof(char)));
	    schema.Insert( CegoField("JDBC", "JDBC", "SEARCHABLE", SMALLINT_TYPE, sizeof(short)));
	    schema.Insert( CegoField("JDBC", "JDBC", "UNSIGNED_ATTRIBUTE", BOOL_TYPE, sizeof(char)));
	    schema.Insert( CegoField("JDBC", "JDBC", "FIXED_PREC_SCALE", BOOL_TYPE, sizeof(char)));
	    schema.Insert( CegoField("JDBC", "JDBC", "AUTO_INCREMENT", BOOL_TYPE, sizeof(char)));
	    schema.Insert( CegoField("JDBC", "JDBC", "LOCAL_TYPE_NAME", VARCHAR_TYPE, MAX_OBJNAME_LEN));
	    schema.Insert( CegoField("JDBC", "JDBC", "MINIMUM_SCALE", SMALLINT_TYPE, sizeof(short)));
	    schema.Insert( CegoField("JDBC", "JDBC", "MAXIMUM_SCALE", SMALLINT_TYPE, sizeof(short)));
	    schema.Insert( CegoField("JDBC", "JDBC", "SQL_DATA_TYPE", INT_TYPE, sizeof(int)));
	    schema.Insert( CegoField("JDBC", "JDBC", "SQL_DATETIME_SUB", INT_TYPE, sizeof(int)));
	    schema.Insert( CegoField("JDBC", "JDBC", "NUM_PREC_RADIX", INT_TYPE, sizeof(int)));
	    
	    CegoOutput o(schema);
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    
	    o.headOut();
	    
	    // STRING type
	    ListT<CegoFieldValue> t1;
	    t1.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("string"))); // type name
	    t1.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[VARCHAR_TYPE])); // SQL type
	    t1.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t1.Insert( CegoFieldValue()); // literal prefix
	    t1.Insert( CegoFieldValue()); // literal suffix
	    t1.Insert( CegoFieldValue()); // create params	
	    t1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // unsigned
	    t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t1.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t1.Insert( CegoFieldValue()); // local type name
	    t1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t1.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t1.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t1.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t1.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t1);
	    
	    // INT type	    
	    ListT<CegoFieldValue> t2;
	    t2.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("int"))); // type name
	    t2.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[INT_TYPE])); // SQL type
	    t2.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t2.Insert( CegoFieldValue()); // literal prefix
	    t2.Insert( CegoFieldValue()); // literal suffix
	    t2.Insert( CegoFieldValue()); // create params	
	    t2.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t2.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t2.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t2.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t2.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t2.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t2.Insert( CegoFieldValue()); // local type name
	    t2.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t2.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t2.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t2.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t2.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t2);
	    
	    // LONG type	    
	    ListT<CegoFieldValue> t3;
	    t3.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("long"))); // type name
	    t3.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[LONG_TYPE])); // SQL type
	    t3.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t3.Insert( CegoFieldValue()); // literal prefix
	    t3.Insert( CegoFieldValue()); // literal suffix
	    t3.Insert( CegoFieldValue()); // create params	
	    t3.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t3.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t3.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t3.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t3.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t3.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t3.Insert( CegoFieldValue()); // local type name
	    t3.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t3.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t3.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t3.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t3.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t3);

	    // BOOL type	    
	    ListT<CegoFieldValue> t4;
	    t4.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("bool"))); // type name
	    t4.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[BOOL_TYPE])); // SQL type
	    t4.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t4.Insert( CegoFieldValue()); // literal prefix
	    t4.Insert( CegoFieldValue()); // literal suffix
	    t4.Insert( CegoFieldValue()); // create params	
	    t4.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t4.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t4.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t4.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // unsigned
	    t4.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t4.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t4.Insert( CegoFieldValue()); // local type name
	    t4.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t4.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t4.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t4.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t4.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t4);

	    // DATETIME type	    
	    ListT<CegoFieldValue> t5;
	    t5.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("datetime"))); // type name
	    t5.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[DATETIME_TYPE])); // SQL type
	    t5.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t5.Insert( CegoFieldValue()); // literal prefix
	    t5.Insert( CegoFieldValue()); // literal suffix
	    t5.Insert( CegoFieldValue()); // create params	
	    t5.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t5.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t5.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t5.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // unsigned
	    t5.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t5.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t5.Insert( CegoFieldValue()); // local type name
	    t5.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t5.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t5.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t5.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t5.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t5);

	    // BIGINT type	    
	    ListT<CegoFieldValue> t6;
	    t6.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("bigint"))); // type name
	    t6.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[BIGINT_TYPE])); // SQL type
	    t6.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t6.Insert( CegoFieldValue()); // literal prefix
	    t6.Insert( CegoFieldValue()); // literal suffix
	    t6.Insert( CegoFieldValue()); // create params	
	    t6.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t6.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t6.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t6.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t6.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t6.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t6.Insert( CegoFieldValue()); // local type name
	    t6.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t6.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t6.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t6.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t6.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t6);

	    // FLOAT type	    
	    ListT<CegoFieldValue> t7;
	    t7.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("datetime"))); // type name
	    t7.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[FLOAT_TYPE])); // SQL type
	    t7.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t7.Insert( CegoFieldValue()); // literal prefix
	    t7.Insert( CegoFieldValue()); // literal suffix
	    t7.Insert( CegoFieldValue()); // create params	
	    t7.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t7.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t7.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t7.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t7.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t7.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t7.Insert( CegoFieldValue()); // local type name
	    t7.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t7.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t7.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t7.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t7.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t7);

	    // DOUBLE type	    
	    ListT<CegoFieldValue> t8;
	    t8.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("double"))); // type name
	    t8.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[DOUBLE_TYPE])); // SQL type
	    t8.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t8.Insert( CegoFieldValue()); // literal prefix
	    t8.Insert( CegoFieldValue()); // literal suffix
	    t8.Insert( CegoFieldValue()); // create params	
	    t8.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t8.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t8.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t8.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t8.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t8.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t8.Insert( CegoFieldValue()); // local type name
	    t8.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t8.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t8.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t8.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t8.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t8);

	    // DECIMAL type	    
	    ListT<CegoFieldValue> t9;
	    t9.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("decimal"))); // type name
	    t9.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[DECIMAL_TYPE])); // SQL type
	    t9.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t9.Insert( CegoFieldValue()); // literal prefix
	    t9.Insert( CegoFieldValue()); // literal suffix
	    t9.Insert( CegoFieldValue()); // create params	
	    t9.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t9.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t9.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t9.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // unsigned
	    t9.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t9.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t9.Insert( CegoFieldValue()); // local type name
	    t9.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t9.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t9.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t9.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t9.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t9);

	    // SMALLINT type	    
	    ListT<CegoFieldValue> t10;
	    t10.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("smallint"))); // type name
	    t10.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[SMALLINT_TYPE])); // SQL type
	    t10.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t10.Insert( CegoFieldValue()); // literal prefix
	    t10.Insert( CegoFieldValue()); // literal suffix
	    t10.Insert( CegoFieldValue()); // create params	
	    t10.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t10.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t10.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t10.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // unsigned
	    t10.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t10.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t10.Insert( CegoFieldValue()); // local type name
	    t10.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t10.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t10.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t10.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t10.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    o.rowOut(t10);

	    // TINYINT type	    
	    ListT<CegoFieldValue> t11;
	    t11.Insert( CegoFieldValue(VARCHAR_TYPE, Chain("tinyint"))); // type name
	    t11.Insert( CegoFieldValue(INT_TYPE, CEGO_JAVATYPE_MAP[TINYINT_TYPE])); // SQL type
	    t11.Insert( CegoFieldValue(INT_TYPE, 0)); // precision
	    t11.Insert( CegoFieldValue()); // literal prefix
	    t11.Insert( CegoFieldValue()); // literal suffix
	    t11.Insert( CegoFieldValue()); // create params	
	    t11.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // nullable
	    t11.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // case sensitive
	    t11.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // searchable
	    t11.Insert( CegoFieldValue(BOOL_TYPE, Chain("true"))); // unsigned
	    t11.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // fixed prec scale
	    t11.Insert( CegoFieldValue(BOOL_TYPE, Chain("false"))); // autoincrement
	    t11.Insert( CegoFieldValue()); // local type name
	    t11.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // minumum scale
	    t11.Insert( CegoFieldValue(SMALLINT_TYPE, 0)); // maximum scale
	    t11.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL data type
	    t11.Insert( CegoFieldValue(INT_TYPE, 0)); // SQL datetime sub
	    t11.Insert( CegoFieldValue(INT_TYPE, 0)); // num prec radix
	    
	    o.rowOut(t11);
	    o.tailOut();	    
	}
	else
	{	    
	    CegoOutput o;
	    if ( _pDbHandle )
		o.setDbHandle(_pDbHandle, NETMNG_MAXTUPLECOUNT, NETMNG_MAXBYTECOUNT);
	    o.chainOut("Unknown JDBC command");	    
	}
    }
}
