///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdmAction.cc  
// ----------------                                                        
// Cego semantic parser adm actions
//
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdmAction
// 
// Description: Semantic actions for cgadm
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/Tokenizer.h>
#include <lfcbase/Host.h>
#include <lfcbase/Datetime.h>

#include "CegoXMLdef.h"
#include "CegoAdmAction.h"
#include "CegoOutput.h"

#include <stdlib.h>
#include <string.h>

#define DEFTMPSIZE 100
#define DEFSYSSIZE 100
#define DEFAPPSIZE 3000
#define DEFLOGNUM 3
#define DEFLOGSIZE 1000000
#define DEFSORTAREASIZE 10000000
#define DEFTSROOT ""
#define DEFPRIMARY Chain()
#define DEFSECONDARY Chain()
#define DEFTIMEOUT 10
#define DEFMAXCACHEENTRY 200
#define DEFMAXCACHESIZE 500000
#define DEFCACHEHASHRANGE 30
#define MAXHOSTLEN 25

extern char __quoteEscapeFlag;
extern char __caseSensitiveFlag;

CegoAdmAction::CegoAdmAction(CegoAdminHandler *pAH, CegoModule *pModule, CegoAdmAction::ResultMode resultMode, bool rawMode) :  CegoAdm(), SigHandler() 
{
    init();
    
    install(SIGINT);
#ifndef HAVE_MINGW
    install(SIGPIPE);
#endif

    _isAborted = false;
	
    _pC = 0;
    _pModule = pModule;
    _pAH = pAH;
    _pit = Chain();
    _rawMode = rawMode;
    _resultMode = resultMode;
    _keepTicket = false;
    _doCleanup = false;
    _doForceload = false;
    _doCPDump = false;
    _doNoInit = false;
    
    _xportMode = XML_XML_VALUE;
    _timeout = DEFTIMEOUT;
    setDefault();

    // default query cache param values
    _maxEntry = 0;
    _maxSize = 0;
    _threshold = -1;
    _hashRange = 0;
    _cacheFilter = Chain();

    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("ROLE"), VARCHAR_TYPE, 10)); 
    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("HOSTNAME"), VARCHAR_TYPE, MAXHOSTLEN)); 
    _admSchema.Insert(CegoField(Chain("ADM"), Chain("ADM"), Chain("MSG"), VARCHAR_TYPE, 50));

    _stringBuf = (char*)malloc(MAXSTRINGLEN);
    _stringBufLen = MAXSTRINGLEN;
    
    if ( _stringBuf == 0 ) 
    {
	Chain msg("Malloc system error"); 
	throw Exception(EXLOC, msg);
    }

    _modId = _pModule->getModId("CegoAdmAction");
}

CegoAdmAction::~CegoAdmAction()  
{
}

void CegoAdmAction::statementAction()
{
}

void CegoAdmAction::rawModeOn()
{
    _rawMode = true;
}

void CegoAdmAction::rawModeOff()
{
    _rawMode = false;
}

bool CegoAdmAction::getRawMode()
{
    return _rawMode;
}

void CegoAdmAction::setCommandChain(char *pC)
{
    _pC = pC;
    
    // init some vars
    _buMsg = Chain("NONE");
    _syncCmd = Chain();
    _timeout = DEFTIMEOUT;

    _maxEntry = 0;
    _maxSize = 0;
    _threshold = -1;
    _hashRange = 0;
    _cacheFilter = Chain();
}

char CegoAdmAction::nextChar() 
{   
    if ( _pC == 0 )
	return 0;

    if ( (int)(*_pC) < 0 )
    {
	Chain msg = Chain("Invalid character <") + Chain((int)(*_pC)) + Chain("> detected"); 
	throw Exception(EXLOC, msg);
    }

    // cout << "Character read : " << *_pC << endl;

    if ( *_pC == '\'' )
    {
	_pC++;
	readChain();
	return 0;
    }
 
    if ( *_pC == 0 )
    {
        return 0; 
    }
    
    char c = *_pC; 
    _pC++;

    if ( __caseSensitiveFlag == 0 )
	return c;
    else
	return (char)tolower((int)c);
} 

void CegoAdmAction::readChain()
{
    setReserved(STRINGVAL); 
    
    int i=0;    

    if ( __quoteEscapeFlag )
    {
	bool moreChar = true; 
	while ( moreChar )
	{	    
	    if ( *_pC == '\'' ) 
	    {
		_pC++;
		if ( *_pC == '\'' )
		{
		    _stringBuf[i] = '\'';
		    i++;
		}
		else
		{
		    moreChar = false;
		}
	    }
	    else
	    {
		_stringBuf[i] = *_pC; 		
		i++;
	    }
	  
	    if ( i == _stringBufLen ) 
	    { 
		reallocateStringBuf();
	    }
	    
	    if ( moreChar )
	    {
		_pC++;
		
		if ( *_pC == 0 )
		{
		    Chain msg("Unterminated string"); 
		    throw Exception(EXLOC, msg);
		}
	    }
	}
    }
    else
    {
	while ( *_pC != '\'' ) 
	{ 
	    // cout << "Storing: " << *_pC << endl; 
	    
	    if ( *_pC == '\\' ) 
	    { 
		_pC++; 
		
		if ( *_pC == 'n' ) 
		{ 
		    _stringBuf[i] = '\n'; 
		} 
		else if ( *_pC == 'r' ) 
		{ 
		    _stringBuf[i] = '\r'; 
		} 
		else if ( *_pC == 't' ) 
		{ 
		    _stringBuf[i] = '\t'; 
		} 
		else if ( *_pC == '\'' ) 
		{ 
		    _stringBuf[i] = '\''; 
		}
		else if ( *_pC == '\\' ) 
		{ 
		    _stringBuf[i] = '\\'; 
		} 
		else 
		{ 
		    Chain msg = Chain("Invalid escape character <") + Chain(*_pC) + Chain(">"); 
		    throw Exception(EXLOC, msg);
		} 
	    } 
	    else
	    { 
		_stringBuf[i] = *_pC; 
	    } 
	    
	    _stringBuf[i] = *_pC; 
	    i++; 
	    
	    if ( i == _stringBufLen ) 
	    {
		reallocateStringBuf();
	    }
	    
	    _pC++;
	    
	    if ( *_pC == 0 )
	    {
		Chain msg("Unterminated string"); 
		throw Exception(EXLOC, msg);
	    }
	}
	_pC++; 
    }

    _stringBuf[i] = 0; 

    return; 
}

void CegoAdmAction::backChar() 
{
    if ( _pC != 0 )
	_pC--;
}

void CegoAdmAction::reallocateStringBuf()
{
    // reallocation required
    _stringBufLen += MAXSTRINGLEN;
    char *newBuf = (char*)malloc( _stringBufLen );
    if ( newBuf == 0 ) 
    {
	Chain msg("Malloc system error"); 
	throw Exception(EXLOC, msg);
    }

    // we just have to copy the previous len
    memcpy(newBuf, _stringBuf, _stringBufLen - MAXSTRINGLEN);
    free(_stringBuf);
    _stringBuf=newBuf;
}

void CegoAdmAction::printTokenList() 
{   
    cout << "TokenList is " << endl;
	
    ListT<Chain> tlist =  getTokenList();
    Chain* pS = tlist.First();
    while (pS)
    {
	cout << *pS << endl;
	pS = tlist.Next();
    }
}

void CegoAdmAction::cleanUp() 
{
    _isAborted = false;
    _tsPerm = Chain();
    _filterPerm = Chain();
    _rightPerm = Chain();
}

void CegoAdmAction::showQueryCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqShowQueryCache(tableSet);
    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    if ( _pAH->getCacheInfo(oe, info, format) )
    {
	CegoOutput output(oe.getSchema(), format);
	output.setRawMode(_rawMode);
	output.tabOut(info);
    }
    Chain msg;
    _pAH->getMsg(msg);
    
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listQueryCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqListQueryCache(tableSet);
    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    if ( _pAH->getCacheList(oe, info, format) )
    {	
	CegoOutput output(oe.getSchema(), format);
	output.setRawMode(_rawMode);
	output.tabOut(info);
    }
    Chain msg;
    _pAH->getMsg(msg);
    
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::cleanQueryCacheAction()
{    
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqCleanQueryCache(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setQueryCacheParamAction()
{
    Chain tableSet;
    
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqSetQueryCacheParam(tableSet, _maxEntry, _maxSize, _threshold, _hashRange);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;    
}

void CegoAdmAction::setQueryCacheMaxEntry()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_maxEntry = pS->asInteger();
    }
}

void CegoAdmAction::setQueryCacheMaxSize()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_maxSize = pS->asInteger();
    }
}

void CegoAdmAction::setQueryCacheHashRange()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_hashRange = pS->asInteger();
    }
}

void CegoAdmAction::setQueryCacheThreshold()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_threshold = pS->asInteger();
    }
}

void CegoAdmAction::showTableCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqShowTableCache(tableSet);
    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    if ( _pAH->getCacheInfo(oe, info, format) )
    {
	CegoOutput output(oe.getSchema(), format);
	output.setRawMode(_rawMode);
	output.tabOut(info);
    }
    Chain msg;
    _pAH->getMsg(msg);
    
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listTableCacheAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqListTableCache(tableSet);
    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    if ( _pAH->getCacheList(oe, info, format) )
    {	
	CegoOutput output(oe.getSchema(), format);
	output.setRawMode(_rawMode);
	output.tabOut(info);
    }
    Chain msg;
    _pAH->getMsg(msg);
    
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::cleanTableCacheAction()
{
    Chain tableSet;
    
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqCleanTableCache(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTableCacheParamAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;

    res = _pAH->reqSetTableCacheParam(tableSet, _cacheFilter, _maxEntry, _maxSize, _hashRange);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;    
}

void CegoAdmAction::setTableCacheMaxEntry()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_maxEntry = pS->asInteger();
    }
}

void CegoAdmAction::setTableCacheMaxSize()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_maxSize = pS->asInteger();
    }
}

void CegoAdmAction::setTableCacheHashRange()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_hashRange = pS->asInteger();
    }
}

void CegoAdmAction::setTableCacheFilter()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_cacheFilter = Chain(_stringBuf);
    }
}

void CegoAdmAction::setDefault()
{
    _primary = DEFPRIMARY;
    _secondary = DEFSECONDARY;
    _tsRoot = DEFTSROOT;
    _logSize = DEFLOGSIZE;
    _logNum = DEFLOGNUM;
    _sortAreaSize = DEFSORTAREASIZE;
    _branchId = 0;
    _sysSize = DEFSYSSIZE;
    _appSize = DEFAPPSIZE;
    _tmpSize = DEFTMPSIZE;
}

/////////////////////////////////
// cego semantic action methods //
/////////////////////////////////

void CegoAdmAction::lockInfoAction() 
{
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    if ( pS )
    {
	Chain lockCat;
	if ( *pS  == Chain("rec") )
	{
	    lockCat  = Chain(XML_LOCK_REC_VALUE);
	}
	else if ( *pS  == Chain("rbrec") )
	{
	    lockCat = Chain(XML_LOCK_RBREC_VALUE);
	}
	else if ( *pS  == Chain("sysrec") )
	{
	    lockCat = Chain(XML_LOCK_SYSREC_VALUE);
	}
	else if ( *pS  == Chain("tp") )
	{
	    lockCat = Chain(XML_LOCK_THREADPOOL_VALUE);
	}
	else if ( *pS == Chain("sp") )
	{
	    lockCat = Chain(XML_LOCK_SYSPAGE_VALUE);
	}
	else if ( *pS == Chain("dp") )
	{
	    lockCat = Chain(XML_LOCK_DATAPAGE_VALUE);
	}
	else if ( *pS == Chain("ix") )
	{
	    lockCat = Chain(XML_LOCK_IDXPAGE_VALUE);
	}
	else if ( *pS == Chain("rb") )
	{
	    lockCat = Chain(XML_LOCK_RBPAGE_VALUE);
	}
	else if ( *pS == Chain("df") )
	{
	    lockCat = Chain(XML_LOCK_DATAFILE_VALUE);
	}
	else if ( *pS == Chain("bp") )
	{
	    lockCat = Chain(XML_LOCK_BUFFERPOOL_VALUE);
	}
	else if ( *pS == Chain("lm") )
	{
	    lockCat = Chain(XML_LOCK_LOGMNG_VALUE);
	}
	else if ( *pS == Chain("qc") )
	{
	    lockCat = Chain(XML_LOCK_QUERYCACHE_VALUE);
	}
	else if ( *pS == Chain("tc") )
	{
	    lockCat = Chain(XML_LOCK_TABLECACHE_VALUE);
	}
	else
	{
	    throw Exception(EXLOC, "Invalid lock category");
	}

	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqLockInfo(lockCat);

	handleMedResult(res);
	
	Chain msg;
	_pAH->getMsg(msg);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getLockInfo(oe, info);
	
	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);
	if ( _rawMode == false )
	    cout << msg << endl;
    }
}

void CegoAdmAction::lockStatAction() 
{
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqLockStat();    
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getLockStat(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::poolInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqPoolInfo();
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getPoolInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();
    ListT<CegoFieldValue> *pFVL = info.First();
    while ( pFVL )
    {
	output.rowOut(*pFVL);	
	pFVL = info.Next();
    }
    output.tailOut();

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::parameterInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqParameterInfo();
    handleMedResult(res); 

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getParameterInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();
    ListT<CegoFieldValue> *pFVL = info.First();
    while ( pFVL )
    {
	output.rowOut(*pFVL);	
	pFVL = info.Next();
    }

    output.tailOut();

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::poolEntryListAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqPoolEntryList();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	Chain msg;
	_pAH->getMsg(msg);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;
	
	_pAH->getPoolEntryList(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);
		
	res = _pAH->nextInfo();	
    }
    
    if ( _rawMode == false )
    {
	Chain msg;
	_pAH->getMsg(msg);	
	cout << msg << endl;
    }
}

void CegoAdmAction::dbSessionInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbSessionInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getDbSessionInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listTableSetAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetTableSetList(true);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getTableSetList(oe, info, true);
    
    CegoOutput output(oe.getSchema(), Chain("llllrrr"));
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::threadInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqThreadInfo();
    handleMedResult(res);	

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;

    _pAH->getThreadInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dbThreadInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbThreadInfo();	
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dbThreadQueryHistAction() 
{
    int threadId = 0;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	threadId = pS->asInteger();
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqDbThreadHistory(threadId);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadHistory(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dbQueryHistLastAction()
{
    int numLast = 0;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	numLast = pS->asInteger();
    }
    
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqQueryHistoryLast(numLast);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadHistory(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dbQueryHistCostAction()
{
    int numLast = 0;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	numLast = pS->asInteger();
    }
    
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqQueryHistoryCost(numLast);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getDbThreadHistory(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::admThreadInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqAdmThreadInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getAdmThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::logThreadInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqLogThreadInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getLogThreadInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::copyInfoAction() 
{       
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqCopyInfo();
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain format;
    _pAH->getCopyInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;	
}

void CegoAdmAction::buFileInfoAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getBuFileInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::buStatInfoAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medGetBUStatInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getBUStatInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::buStatResetAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();

    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->medResetBUStat(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::abortDbThreadAction() 
{
    int threadId = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	threadId = pS->asInteger();
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->reqDbThreadAbort(threadId);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::retrieveTableSetAction()
{
    Chain masterHost(_stringBuf);
    CegoAdminHandler::ResultType res;
    res = _pAH->reqRetrieveTableSet(masterHost);	
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
} 

void CegoAdmAction::defineTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medDefineTableSet(tableSet,
				  _tsRoot,
				  _primary,
				  _secondary,
				  _sysSize,
				  _tmpSize,
				  _appSize,
				  _logSize,
				  _logNum, 
				  _sortAreaSize,
				  _branchId);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
    
    setDefault();
}

void CegoAdmAction::startTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medStartTableSet(tableSet, _doCleanup, _doForceload, _doCPDump, _doNoInit);
    _doCleanup=false;
    _doForceload=false;
    _doCPDump=false;
    _doNoInit = false;

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::stopTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;    
    res = _pAH->medStopTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::copyTableSetAction() 
{
    CegoAdminHandler::ResultType res;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medCopyTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::switchTableSetAction() 
{
    CegoAdminHandler::ResultType res;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medSwitchTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dropTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medDropTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::removeTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medRemoveTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::resetTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medResetTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::createTableSetAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    res = _pAH->medCreateTableSet(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::syncTableSetAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSyncTableSet(tableSet, _buMsg, _syncCmd, _timeout);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::beginBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medBeginBackup(tableSet, _buMsg);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::executeBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medExecuteBackup(tableSet, _buMsg);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::endBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medEndBackup(tableSet, _buMsg, _keepTicket);
    _keepTicket = false;
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listBackupAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medListBackup(tableSet, _pit);
    handleMedResult(res);

    // reset pit to 0
    _pit = Chain();

    Chain msg;
    _pAH->getMsg(msg);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getBackupInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::keepTicketAction()
{
    _keepTicket = true;
}

void CegoAdmAction::syncCmdAction()
{
    _syncCmd = Chain(_stringBuf);
}

void CegoAdmAction::timeoutAction()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_timeout = pS->asInteger();
    }
}

void CegoAdmAction::msgAction()
{
    _buMsg = Chain(_stringBuf);
}

void CegoAdmAction::cleanupAction()
{
    _doCleanup = true;
}

void CegoAdmAction::forceloadAction()
{
    _doForceload = true;
}

void CegoAdmAction::cpDumpAction()
{
    _doCPDump = true;
}

void CegoAdmAction::noInitAction()
{
    _doNoInit = true;
}

void CegoAdmAction::xmlModeAction()
{
    _xportMode = XML_XML_VALUE;
}

void CegoAdmAction::binModeAction()
{
    _xportMode = XML_BIN_VALUE;
}

void CegoAdmAction::plainModeAction()
{
    _xportMode = XML_PLAIN_VALUE;
}

void CegoAdmAction::restoreTableSetAction()
{
    Chain tableSet;
    Chain backupId(_stringBuf);

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    
    CegoAdminHandler::ResultType res;
    res = _pAH->medRestore(tableSet, backupId);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::recoverTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->medRecover(tableSet, _pit);
    handleMedResult(res);

    // reset pit to 0
    _pit = Chain();

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setLogMngAction()
{
    Chain progName(_stringBuf);

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetLogMng(progName, _timeout);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;

    _timeout = DEFTIMEOUT;
}

void CegoAdmAction::setBackupMngAction()
{
    Chain progName(_stringBuf);

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetBackupMng(progName);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::showLogMngAction()
{
    CegoAdminHandler::ResultType res;
    res = _pAH->reqShowLogMng();

    if ( res == CegoAdminHandler::ADM_OK )
    {

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getLogMngInfo(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);

	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::showBackupMngAction()
{
    CegoAdminHandler::ResultType res;
    res = _pAH->reqShowBackupMng();

    if ( res == CegoAdminHandler::ADM_OK )
    {

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getBackupMngInfo(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);

	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::setTSInitFileAction()
{
    Chain initFileName(_stringBuf);

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSInitFile(tableSet, initFileName);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSRootPathAction()
{
    Chain tsRoot(_stringBuf);

    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSRootPath(tableSet, tsRoot);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSLogNumAction()
{
    int logNum = 0;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	logNum = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSLogNum(tableSet, logNum);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSLogSizeAction()
{
    int logSize = 0;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	logSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSLogSize(tableSet, logSize);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSSortAreaSizeAction()
{
    unsigned long long sortAreaSize = 0;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	sortAreaSize = pS->asUnsignedLongLong();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSSortAreaSize(tableSet, sortAreaSize);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSLogUserAction()
{
    Chain logUser;
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	logUser = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSLogUser(tableSet, logUser);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSSysSizeAction()
{
    Chain tableSet;
    int sysSize = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	sysSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSSysSize(tableSet, sysSize);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSTmpSizeAction()
{
    Chain tableSet;
    int tmpSize = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tmpSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSTmpSize(tableSet, tmpSize);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTSAppSizeAction()
{
    Chain tableSet;
    int appSize = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	appSize = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTSAppSize(tableSet, appSize);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setCheckpointAction()
{
    Chain tableSet;
    int interval = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	interval = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetCheckpoint(tableSet, interval);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setNumQueryLastAction()
{    
    int numLast = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	numLast = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqNumQueryLast(numLast);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setNumQueryCostAction()
{   
    int numCost = 0;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	numCost = pS->asInteger();
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqNumQueryCost(numCost);
    handleMedResult(res);
    
    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::switchSecondaryAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    
    CegoAdminHandler::ResultType res;
    res = _pAH->medSecSwitch(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::switchMediatorAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    
    CegoAdminHandler::ResultType res;
    res = _pAH->secMedSwitch(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::relocateSecondaryAction()
{
    Chain secondary(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medSecRelocate(tableSet, secondary);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::relocateMediatorAction()
{
    Chain mediator(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    
    CegoAdminHandler::ResultType res;
    res = _pAH->secMedRelocate(tableSet, mediator);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::checkTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    
    CegoAdminHandler::ResultType res;
    res = _pAH->medCheckTableSet(tableSet);
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetCheck(oe, info);
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
}

void CegoAdmAction::verifyTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medVerifyTableSet(tableSet);
    
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetVerificationInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;
	_pAH->getTableSetVerificationInfo(oe, info);
	
	ListT<CegoFieldValue>* pFVL = info.First();
	while ( pFVL )
	{
	    output.rowOut(*pFVL);
	    pFVL = info.Next();
	}	
	res = _pAH->nextInfo();
    }
    output.tailOut();

    Chain msg;
    _pAH->getMsg(msg);

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	    cout << msg << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::correctTableSetAction()
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medCorrectTableSet(tableSet);
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    _pAH->getTableSetCorrectionInfo(oe, info);

    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO )
    {
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;
	_pAH->getTableSetCorrectionInfo(oe, info);
	
	ListT<CegoFieldValue>* pFVL = info.First();
	while ( pFVL )
	{
	    output.rowOut(*pFVL);
	    pFVL = info.Next();
	}	
	res = _pAH->nextInfo();
    }
    output.tailOut();

    Chain msg;
    _pAH->getMsg(msg);

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	    cout << msg << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::getDbSpecAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain dbSpec;
    res = _pAH->reqGetDbSpec(dbSpec);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	Chain msg;
	_pAH->getMsg(msg);
	cout << msg << endl;
	cout << "DBSPEC=" << dbSpec << endl;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::listNodeAction() 
{
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqGetNodeList();

    if ( res == CegoAdminHandler::ADM_OK )
    {
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > info;

	_pAH->getNodeInfo(oe, info);

	CegoOutput output(oe.getSchema());
	output.setRawMode(_rawMode);
	output.tabOut(info);

	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::getDbInfoAction() 
{
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqGetDbInfo();

    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::tableSetInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetDetailedTableSetInfo(tableSet);
    
    handleMedResult(res);    

    Chain msg;
    _pAH->getMsg(msg);
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getDetailedTableSetInfo(oe, info);

    CegoOutput output(oe.getSchema(), Chain("rl"));
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dataFileInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;   
    res = _pAH->medGetDetailedTableSetInfo(tableSet);

    handleMedResult(res);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    Chain msg;
    Chain format;
    _pAH->getMsg(msg);
    _pAH->getDataFileInfo(oe, info, format);
    
    CegoOutput output(oe.getSchema(), format);
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::archLogInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetArchLogInfo(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getArchLogInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::logInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetDetailedTableSetInfo(tableSet);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getLogInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::objInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetObjectInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
        
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getObjectInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::transactionInfoAction() 
{
    Chain tableSet;
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->medGetTransactionInfo(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getTransactionInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTableSetNodeAction() 
{
    Chain tableSet;
    Chain primary;
    Chain secondary;
    Chain mediator;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	mediator = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 secondary = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 primary = *pS;
    }
    pS = getTokenList().Next();
    if ( pS )
    {
	 tableSet = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetTableSetNode(tableSet, primary, secondary, mediator);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::copyFileAction() 
{
    CegoAdminHandler::ResultType res;   
    Chain fileName(_stringBuf);
    res = _pAH->reqCopyFile(fileName);
    if ( res == CegoAdminHandler::ADM_OK )
    {
	if ( _rawMode == false )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg << endl;
	}
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::exportTableAction()
{
    Chain exportFile(_stringBuf);

    Chain tableSet;
    Chain tableName;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableName = *pS;
    }

    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqExportTable(tableSet, tableName, exportFile, _xportMode);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;

    // reset to default
    _xportMode = XML_XML_VALUE;
}

void CegoAdmAction::exportTableSetAction()
{
    Chain exportFile(_stringBuf);
    Chain tableSet;
    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    res = _pAH->reqExportTableSet(tableSet, false, exportFile, _xportMode);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;

    // reset to default
    _xportMode = XML_XML_VALUE;
}

void CegoAdmAction::exportTableSetStructureAction()
{
    Chain exportFile(_stringBuf);
    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqExportTableSet(tableSet, true, exportFile, _xportMode);
    handleMedResult(res);

    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::importTableAction()
{
    Chain importFile(_stringBuf);

    Chain tableSet;
    Chain tableName;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableName = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTable(tableSet, tableName, importFile, _xportMode);	
    handleMedResult(res);

    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::importTableSetAction()
{
    Chain importFile(_stringBuf);
    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTableSet(tableSet, false, importFile, _xportMode);
    handleMedResult(res);

    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::importTableSetStructureAction()
{
    Chain importFile(_stringBuf);

    Chain tableSet;

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqImportTableSet(tableSet, true, importFile, _xportMode);
    handleMedResult(res);

    _xportMode = XML_XML_VALUE;

    Chain msg;
    _pAH->getMsg(msg);

    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::addDataFileAction()
{
    int numPages = 0;
    Chain tableSet;
    Chain dataFile(_stringBuf);
    Chain fileType;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	numPages = pS->asInteger();
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	if ( *pS == Chain("app" ) )
	{
	    fileType = Chain(XML_APPFILE_VALUE);
	}
	else if ( *pS == Chain("temp" ) )
	{
	    fileType = Chain(XML_TEMPFILE_VALUE);
	}
	else if ( *pS == Chain("sys" ) )
	{
	    fileType = Chain(XML_SYSFILE_VALUE);
	}
	else
	{
	    throw Exception(EXLOC, Chain("Invalid file type"));
	}
    }

    CegoAdminHandler::ResultType res;    
    res = _pAH->medAddDataFile(tableSet, fileType, dataFile, numPages);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listUserAction()
{
    CegoAdminHandler::ResultType res;
    res = _pAH->reqShowUser();

    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getUserInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);    
}

void CegoAdmAction::addUserAction()
{
    Chain user;
    Chain password(_stringBuf);

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqAddUser(user, password);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::removeUserAction()
{
    Chain tableSet;
    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->reqRemoveUser(user);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::changePwdAction()
{
    Chain user;
    Chain password(_stringBuf);

    Chain* pS = getTokenList().First();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqChangePwd(user, password);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::assignRoleAction()
{
    Chain user;
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqAssignRole(user, role);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::removeRoleAction()
{
    Chain user;
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqRemoveRole(user, role);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::listRoleAction()
{
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqListRole();
    
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getRoleList(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
}

void CegoAdmAction::showRoleAction()
{
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }

    CegoAdminHandler::ResultType res;   
    res = _pAH->reqShowRole(role);
    
    handleMedResult(res);

    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;
    
    _pAH->getRoleInfo(oe, info);
    
    CegoOutput output(oe.getSchema());
    output.setRawMode(_rawMode);
    output.tabOut(info);
}

void CegoAdmAction::createRoleAction()
{
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqCreateRole(role);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::dropRoleAction()
{
    Chain role;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    res = _pAH->reqDropRole(role);

    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setPermAction()
{
    Chain role;
    Chain permid;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }

    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	permid = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqSetPermission(role, permid, _tsPerm, _filterPerm, _rightPerm);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::setTableSetPerm()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_tsPerm = *pS;
    }
}

void CegoAdmAction::setFilterPerm()
{
    _filterPerm = Chain(_stringBuf);
}

void CegoAdmAction::setRightPerm()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_rightPerm = *pS;
    }
}

void CegoAdmAction::removePermAction()
{
    Chain role;
    Chain permid;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	role = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	permid = *pS;
    }

    CegoAdminHandler::ResultType res;
    res = _pAH->reqRemovePermission(role, permid);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::traceOnAction()
{
    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqUserTrace(user, true);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::traceOffAction()
{
    Chain user;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	user = *pS;
    }

    CegoAdminHandler::ResultType res;   
    res = _pAH->reqUserTrace(user, false);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::addArchLogAction()
{
    Chain tableSet;
    Chain archId;
    Chain archPath(_stringBuf);

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	archId = *pS;
    }
       
    CegoAdminHandler::ResultType res;    
    res = _pAH->medAddArchLog(tableSet, archId, archPath);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::removeArchLogAction()
{
    Chain tableSet;
    Chain archId;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	archId = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medRemoveArchLog(tableSet, archId);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::enableArchLogAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medEnableArchLog(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::disableArchLogAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;
    
    res = _pAH->medDisableArchLog(tableSet);
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::enableAppendModeAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medEnableAppendMode(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::disableAppendModeAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medDisableAppendMode(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::enableAutoCorrectAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medEnableAutoCorrect(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::disableAutoCorrectAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->medDisableAutoCorrect(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::enableTableCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqEnableTableCache(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::disableTableCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqDisableTableCache(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::enableQueryCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqEnableQueryCache(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::disableQueryCacheAction()
{
    Chain tableSet;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqDisableQueryCache(tableSet);
    
    handleMedResult(res);

    Chain msg;
    _pAH->getMsg(msg);
    if ( _rawMode == false )
	cout << msg << endl;
}

void CegoAdmAction::storeSysSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_sysSize = pS->asInteger();
    }
}

void CegoAdmAction::storeTmpSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_tmpSize = pS->asInteger();
    }
}

void CegoAdmAction::storeAppSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_appSize = pS->asInteger();
    }
}

void CegoAdmAction::storeLogSizeAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_logSize = pS->asInteger();
    }
}

void CegoAdmAction::storeLogNumAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_logNum = pS->asInteger();
    }
}

void CegoAdmAction::storeSortAreaAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_sortAreaSize = pS->asUnsignedLongLong();
    }
}

void CegoAdmAction::storePrimaryAttr()
{
    _primary = Chain(_stringBuf);   
}

void CegoAdmAction::storeSecondaryAttr()
{
    _secondary = Chain(_stringBuf);   
}

void CegoAdmAction::storeTSRootAttr()
{
    _tsRoot = Chain(_stringBuf);
}

void CegoAdmAction::storeBranchAttr()
{
    Chain* pS = getTokenList().First();
    if ( pS )
    {
	_branchId = pS->asInteger();
    }
}

void CegoAdmAction::storePit()
{
    _pit = Chain(_stringBuf);
}

void CegoAdmAction::dumpTable()
{
    Chain tableSet;
    Chain table;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	table = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqDumpObject(tableSet, CegoObject::TABLE, table);
	
    if ( res == CegoAdminHandler::ADM_INFO )
    {
	Chain tableDump;

	_pAH->nextInfo();
	_pAH->getMsg(tableDump);

	cout << tableDump;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }    
}

void CegoAdmAction::dumpUIndex()
{
    Chain tableSet;
    Chain table;

    Chain* pS = getTokenList().First();
    if ( pS )
    {
	tableSet = *pS;
    }
    pS = getTokenList().Next();
    pS = getTokenList().Next();
    if ( pS )
    {
	table = *pS;
    }
        
    CegoAdminHandler::ResultType res;   
    res = _pAH->reqDumpObject(tableSet, CegoObject::UAVLTREE, table);
	
    if ( res == CegoAdminHandler::ADM_INFO )
    {
	Chain tableDump;

	_pAH->nextInfo();
	_pAH->getMsg(tableDump);

	cout << tableDump;
    }
    else if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }    
}

void CegoAdmAction::handleMedResult(CegoAdminHandler::ResultType res)
{
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);    
	throw Exception(EXLOC, msg);
    }
    
    CegoOutput output(_admSchema, Chain("llm"));
    output.setRawMode(_rawMode);    

    if ( _resultMode == VERBOSE )
	output.headOut();

    while ( res == CegoAdminHandler::ADM_INFO && _isAborted == false )
    {	
	if ( _resultMode == VERBOSE )
	{
	    Chain hostRole;
	    Chain hostName;
	    Chain msg;
	    
	    _pAH->getHostRole(hostRole);
	    _pAH->getHostName(hostName);
	    _pAH->getMsg(msg);

	    Chain normHost;
	    if ( hostName.length() > MAXHOSTLEN-3 )
		normHost = hostName.subChain(1, MAXHOSTLEN-3) + Chain(".. ");
	    else
		normHost = hostName;

	    ListT<CegoFieldValue> fvl;
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, hostRole));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, normHost));
	    fvl.Insert(CegoFieldValue(VARCHAR_TYPE, msg));
	    
	    output.rowOut(fvl);
	}
	else if ( _resultMode == NORMAL )
	{
	    Chain msg;
	    _pAH->getMsg(msg);
	    cout << msg;
	    cout.flush();
	}

	res = _pAH->nextInfo();	
    }

    if ( _isAborted && res == CegoAdminHandler::ADM_INFO )
	res = _pAH->abort();
    	
    if ( _resultMode == VERBOSE )
	output.tailOut();
    
    
    if ( res == CegoAdminHandler::ADM_ERROR )
    {
	Chain msg;
	_pAH->getMsg(msg);
	throw Exception(EXLOC, msg);
    }
}

void CegoAdmAction::sigCatch(int sig) 
{    
    try
    {	
	install(SIGINT);
#ifndef HAVE_MINGW
	install(SIGPIPE);
#endif

	if ( _abortEnabled )
	{
	    cout << endl << "Aborting admin command .." << endl;
	    _isAborted = true;
	}
	else
	{
	    cout << "Use quit to quit" << endl;
	}
    }
    catch ( Exception e)
    {
	Chain msg;
	e.pop(msg);
	cout << "Error : " << msg << endl;
    }   
}
