///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoDistDbHandler.cc
// --------------------
// Cego global db handler class implementation
//                                                         
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoDistDbHandler
//
// Description: Database client interface to the access the database backend via network 
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Datetime.h>
#include <lfcbase/NetHandler.h>
#include <lfcxml/Element.h>

// CEGO INCLUDES
#include "CegoDistDbHandler.h"
#include "CegoTypeConverter.h"
#include "CegoXMLdef.h"

CegoDistDbHandler::CegoDistDbHandler(NetHandler *pN, CegoDbHandler::ProtocolType pt, CegoModule *pModule) : CegoDbHandler(pN, pt, pModule)
{
    _modId = pModule->getModId("CegoDistDbHandler");
}

CegoDistDbHandler::~CegoDistDbHandler()
{
}

/*
void CegoDistDbHandler::getDeleteArg(Chain& tableSet, Chain& tableName, CegoPredDesc*& pPred,
				     CegoDistManager *pGTM)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);	
	    tableName = _xml.getDocument()->getRootElement()->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> pel = _xml.getDocument()->getRootElement()->getChildren(XML_PRED_ELEMENT);
	    
	    Element **pPE = pel.First();
	    if ( pPE )
	    {   
		pPred = new CegoPredDesc(*pPE, pGTM);
	    }    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getUpdateArg(Chain& tableSet, 
				     Chain& tableName,  
				     ListT<CegoField>& fl,
				     ListT<CegoExpr*>& exprList,
				     CegoPredDesc*& pPred,
				     CegoDistManager *pGTM)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);	
	    tableName = _xml.getDocument()->getRootElement()->getAttributeValue(XML_TABLENAME_ATTR);
	    	    
	    ListT<Element*> pel = _xml.getDocument()->getRootElement()->getChildren(XML_PRED_ELEMENT);
	    
	    Element **pPE = pel.First();
	    if ( pPE )
	    {   
		pPred = new CegoPredDesc(*pPE, pGTM);
	    }	    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}
*/

void CegoDistDbHandler::getCreateTableArg(Chain& tableSet, Chain& tableName, ListT<CegoField>& fl, ListT<CegoField>& idxList)
{    
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> colList = pRoot->getChildren(XML_COL_ELEMENT);
	    
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
		Chain colSize = (*pCol)->getAttributeValue(XML_COLSIZE_ATTR);
		
		CegoTypeConverter tc;
		CegoField f(CegoField(tableName, tableName, colName, tc.getTypeId(colType), colSize.asInteger()));	    
		fl.Insert(f);
		
		pCol = colList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getCreateViewArg(Chain& tableSet, Chain& viewName, ListT<CegoField>& fl, Chain& viewText)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    viewName = pRoot->getAttributeValue(XML_VIEWNAME_ATTR);
	    viewText = pRoot->getAttributeValue(XML_VIEWTEXT_ATTR);
	    
	    ListT<Element*> colList = pRoot->getChildren(XML_COL_ELEMENT);
	    
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
		Chain colSize = (*pCol)->getAttributeValue(XML_COLSIZE_ATTR);
		
		CegoTypeConverter tc;
		CegoField f(CegoField(viewName, viewName, colName, tc.getTypeId(colType), colSize.asInteger()));	    
		fl.Insert(f);
		pCol = colList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

void CegoDistDbHandler::getCreateProcedureArg(Chain& tableSet, Chain& procName, Chain& procText)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    procName = pRoot->getAttributeValue(XML_PROCNAME_ATTR);
	    procText = pRoot->getAttributeValue(XML_PROCTEXT_ATTR);
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

/* deprecated, since XML encoding for PredDesc is no more supported

void CegoDistDbHandler::getCreateCheckArg(Chain& tableSet, Chain& checkName, Chain& tableName, CegoPredDesc*& pPred,
				     CegoDistManager *pGTM)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);	
	    checkName = _xml.getDocument()->getRootElement()->getAttributeValue(XML_NAME_ATTR);
	    tableName = _xml.getDocument()->getRootElement()->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> pel = _xml.getDocument()->getRootElement()->getChildren(XML_PRED_ELEMENT);
	    
	    Element **pPE = pel.First();
	    if ( pPE )
	    {   
		pPred = new CegoPredDesc(*pPE, pGTM);
	    }    
	}    
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

void CegoDistDbHandler::getAlterTableArg(Chain& tableSet, Chain& tableName, ListT<CegoAlterDesc>& alterList)
{
    if ( _protType == CegoDbHandler::XML )
    {
	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> alterElementList = pRoot->getChildren(XML_ALTER_ELEMENT);
	    
	    Element **pAlterElement = alterElementList.First();
	    while ( pAlterElement ) 
	    {
		CegoAlterDesc ad(*pAlterElement);
		alterList.Insert(ad);
		pAlterElement = alterElementList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

*/

void CegoDistDbHandler::getDropTableArg(Chain& tableSet, Chain& tableName, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    tableName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId( pRoot->getAttributeValue(XML_OBJTYPE_ATTR) );
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getCreateIndexArg(Chain& tableSet, Chain& indexName, Chain& tableName, ListT<CegoField>& idxList, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {   
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    indexName = pRoot->getAttributeValue(XML_INDEXNAME_ATTR);
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> colList = pRoot->getChildren(XML_COL_ELEMENT);
	    
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		
		CegoField f(CegoField(tableName, colName));
		idxList.Insert(f);
		
		pCol = colList.Next();
	    }
	    
	    if ( pRoot->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_INDEX_VALUE) )
		type = CegoObject::AVLTREE;
	    else if ( pRoot->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_PINDEX_VALUE) )
		type = CegoObject::PAVLTREE;
	    else if ( pRoot->getAttributeValue(XML_INDEXTYPE_ATTR) == Chain(XML_UINDEX_VALUE) )
		type = CegoObject::UAVLTREE;	    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getObjectInfoArg(int& tabSetId, Chain& tableName, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tabSetId = pRoot->getAttributeValue(XML_TSID_ATTR).asInteger();
	    tableName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId( pRoot->getAttributeValue(XML_OBJTYPE_ATTR) );
	}    
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getGetTableArg(int& tabSetId, Chain& tableName, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tabSetId = pRoot->getAttributeValue(XML_TSID_ATTR).asInteger();
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId(pRoot->getAttributeValue(XML_TABLETYPE_ATTR));	    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getRenameArg(Chain& tableSet, Chain& objName, CegoObject::ObjectType& type, Chain& newObjName)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId(pRoot->getAttributeValue(XML_TABLETYPE_ATTR));
	    
	    newObjName = pRoot->getAttributeValue(XML_NEWOBJNAME_ATTR);
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getReorgArg(Chain& tableSet, Chain& objName, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId(pRoot->getAttributeValue(XML_TABLETYPE_ATTR));
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getGetObjectListArg(int& tabSetId, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tabSetId = pRoot->getAttributeValue(XML_TSID_ATTR).asInteger();
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId(pRoot->getAttributeValue(XML_TABLETYPE_ATTR));	    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getGetObjectByTableListArg(Chain& tableSet, Chain& tableName)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	} 
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}
    
void CegoDistDbHandler::getSyncArg(Chain& tableSet, Chain& escCmd, int& timeout)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    escCmd = pRoot->getAttributeValue(XML_ESCCMD_ATTR);
	    timeout = pRoot->getAttributeValue(XML_TIMEOUT_ATTR).asInteger();
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getPageCountArg(Chain& tableSet, Chain& objName, CegoObject::ObjectType& type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);
	    objName = pRoot->getAttributeValue(XML_OBJNAME_ATTR);
	    
	    CegoTypeConverter tc;
	    type = tc.getObjectTypeId(pRoot->getAttributeValue(XML_OBJTYPE_ATTR));
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqInsertOp(const Chain& tableSet, const Chain& tableName, const ListT<CegoField>& fl)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	_xml.getDocument()->clear();
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	
	CegoField *pF = fl.First();
	while ( pF ) 
	{
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	    pColElement->setAttribute(XML_COLVAL_ATTR, pF->getValue().valAsChain());
	    
	    CegoTypeConverter tc;     
	    pColElement->setAttribute(XML_COLTYPE_ATTR, tc.getTypeString(pF->getType()));
	    
	    pRoot->addContent(pColElement);
	    pF = fl.Next();
	}
	
	_xml.getDocument()->setRootElement(pRoot);
	_xml.getDocument()->setDocType(XML_INSERT_REQUEST);
	
	Chain request;
	_xml.getXMLChain(request);
	
	_pN->setMsg(request, request.length());
	
	_pN->writeMsg();
	
	_pN->readMsg();
	
	_xml.getDocument()->clear();
	_xml.setChain( _pN->getMsg() );
	_xml.parse();
	
	Chain docType = _xml.getDocument()->getDocType();
	
	if ( docType == Chain(XML_OK_DOC) )
	{
	    return DB_OK;
	}
	else if ( docType == Chain(XML_ERROR_DOC) )
	{
	    return DB_ERROR;
	}
	else
	{
	    throw Exception(EXLOC, Chain("Invalid document type"));
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

/*
CegoDistDbHandler::ResultType CegoDistDbHandler::reqUpdateOp(const Chain& tableSet, 
							     const Chain& tableName,
							     const ListT<CegoField>& updSchema, 
							     const ListT<CegoExpr*>& exprList,
							     CegoPredDesc* pPred)
{   
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
        
	CegoField *pF = updSchema.First();
	while ( pF )
	{
	    pRoot->addContent(pF->toElement());
	    pF = updSchema.Next();
	}
	CegoExpr **pExpr = exprList.First();
	while ( pExpr )
	{
	    pRoot->addContent((*pExpr)->toElement());
	    pExpr = exprList.Next();
	}
	
	pRoot->addContent(pPred->toElement());
	
	return sendXMLReq(XML_UPDATE_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqDeleteOp(const Chain& tableSet, const Chain& tableName, CegoPredDesc* pPred)
{    
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	
	pRoot->addContent(pPred->toElement());
	
	return sendXMLReq(XML_DELETE_REQUEST, pRoot);    
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}
*/

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateTableOp(const Chain& tableSet, const Chain& tableName, CegoObject::ObjectType type,
								  const ListT<CegoField>& fl, const ListT<CegoField>& idxList)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	
	CegoField *pF = fl.First();
	while ( pF ) 
	{
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	    
	    CegoTypeConverter tc;
	    pColElement->setAttribute(XML_COLTYPE_ATTR, tc.getTypeString(pF->getType()));
	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));
	    
	    pRoot->addContent(pColElement);
	    pF = fl.Next();
	}
	return sendXMLReq(XML_CREATETABLE_REQUEST, pRoot);    
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

/*
CegoDistDbHandler::ResultType CegoDistDbHandler::reqAlterTableOp(const Chain& tableSet, const Chain& tableName, const ListT<CegoAlterDesc>& alterList)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	
	CegoAlterDesc *pAD = alterList.First();
	while ( pAD ) 
	{
	    pRoot->addContent(pAD->toElement());
	    pAD = alterList.Next();
	}
	
	return sendXMLReq(XML_ALTERTABLE_REQUEST, pRoot);    	
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}
*/

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateIndexOp(const Chain& tableSet, 
								  const Chain& indexName, 
								  const Chain& tableName, 
								  CegoObject::ObjectType type, 
								  const ListT<CegoField>& idxList)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_INDEXNAME_ATTR, indexName);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	if ( type == CegoObject::AVLTREE )
	    pRoot->setAttribute(XML_INDEXTYPE_ATTR, XML_INDEX_VALUE);
	else if ( type == CegoObject::PAVLTREE )
	    pRoot->setAttribute(XML_INDEXTYPE_ATTR, XML_PINDEX_VALUE);
	else if ( type == CegoObject::UAVLTREE )
	    pRoot->setAttribute(XML_INDEXTYPE_ATTR, XML_UINDEX_VALUE);
		
	CegoField *pF = idxList.First();
	while ( pF ) 
	{
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	    
	    pRoot->addContent(pColElement);
	    pF = idxList.Next();
	}
	
	return sendXMLReq(XML_CREATEINDEX_REQUEST, pRoot);    
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateFKeyOp(int tabSetId, const Chain& fkey, 
								 const Chain& tableName, 
								 const ListT<CegoField>& keyList, 
								 const Chain& refTable, 
								 const ListT<CegoField>& refList)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
	pRoot->setAttribute(XML_FKEY_ATTR, fkey);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	pRoot->setAttribute(XML_REFTABLENAME_ATTR, refTable);
		
	CegoField *pK = keyList.First();
	while ( pK ) 
	{
	    Element *pColElement = new Element(XML_KEY_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pK->getAttrName());
	    
	    pRoot->addContent(pColElement);
	    pK = keyList.Next();
	}
	
	CegoField *pR = refList.First();
	while ( pR ) 
	{
	    Element *pColElement = new Element(XML_REF_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pR->getAttrName());
	    
	    pRoot->addContent(pColElement);
	    pR = refList.Next();
	}
	
	return sendXMLReq(XML_CREATE_FKEY_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqTableDataOp(int tabSetId, const Chain& tableName, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_TABLETYPE_ATTR,  tc.getObjectTypeString(type));
	
	return sendXMLReq(XML_GETTABLE_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqGetObjectListOp(int tabSetId, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
	
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_TABLETYPE_ATTR, tc.getObjectTypeString(type));
	
	return sendXMLReq(XML_GETOBJLIST_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqGetObjectByTableListOp(const Chain& tableSet, const Chain& tableName)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	
	return sendXMLReq(XML_GETOBJLISTBYTABLE_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqRenameOp(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type, const Chain& newObjName)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
	
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_TYPE_ATTR, tc.getObjectTypeString(type));    
	pRoot->setAttribute(XML_NEWOBJNAME_ATTR, newObjName);
       	
	return sendXMLReq(XML_OBJRENAME_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqSyncOp(const Chain& tableSet, const Chain& escCmd, int timeout)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_ESCCMD_ATTR, escCmd);
	pRoot->setAttribute(XML_TIMEOUT_ATTR, Chain(timeout));
	
	return sendXMLReq(XML_SYNC_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqGetPageCount(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
	
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_OBJTYPE_ATTR, tc.getObjectTypeString(type));    
    
	return sendXMLReq(XML_GETPAGECOUNT_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqReorgObjectOp(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
	
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_TYPE_ATTR, tc.getObjectTypeString(type));    
	
	return sendXMLReq(XML_REORG_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

/*
CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateCheckOp(const Chain& tableSet, const Chain& checkName, const Chain& tableName, CegoPredDesc* pPred)
{    
    if ( _protType == CegoDbHandler::XML )
    {
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_NAME_ATTR, checkName);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);    
	pRoot->addContent(pPred->toElement());
	
	return sendXMLReq(XML_CREATECHECK_REQUEST, pRoot);	
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}
*/

CegoDistDbHandler::ResultType CegoDistDbHandler::reqObjectInfoOp(int tabSetId, const Chain& objName, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);

	pRoot->setAttribute(XML_TSID_ATTR, Chain(tabSetId));
	pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
        
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_OBJTYPE_ATTR, tc.getObjectTypeString(type));
	
	return sendXMLReq(XML_OBJECTINFO_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqDropObjectOp(const Chain& tableSet, const Chain& objName, CegoObject::ObjectType type)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_OBJNAME_ATTR, objName);
        
	CegoTypeConverter tc;
	pRoot->setAttribute(XML_OBJTYPE_ATTR, tc.getObjectTypeString(type));
	
	return sendXMLReq(XML_DROP_OBJECT_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateViewOp(const Chain& tableSet, const Chain& viewName, const ListT<CegoField>& fl, const Chain& viewText)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_VIEWNAME_ATTR, viewName);
	pRoot->setAttribute(XML_VIEWTEXT_ATTR, viewText);
	
	CegoField *pF = fl.First();
	while ( pF ) 
	{
	    Element *pColElement = new Element(XML_COL_ELEMENT);
	    pColElement->setAttribute(XML_COLNAME_ATTR, pF->getAttrName());
	    
	    CegoTypeConverter tc;
	    pColElement->setAttribute(XML_COLTYPE_ATTR, tc.getTypeString(pF->getType()));
	    pColElement->setAttribute(XML_COLSIZE_ATTR, Chain(pF->getLength()));
	    
	    pRoot->addContent(pColElement);
	    pF = fl.Next();
	}
	
	return sendXMLReq(XML_CREATEVIEW_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateProcOp(const Chain& tableSet, const Chain& procName, const Chain& procText)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_PROCNAME_ATTR, procName);
	pRoot->setAttribute(XML_PROCTEXT_ATTR, procText);
	return sendXMLReq(XML_CREATEPROCEDURE_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCreateTriggerOp(const Chain& tableSet, const Chain& triggerName, const Chain& tableName, const Chain& triggerText)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	pRoot->setAttribute(XML_TRIGGERNAME_ATTR, triggerName);
	pRoot->setAttribute(XML_TABLENAME_ATTR, tableName);
	pRoot->setAttribute(XML_TRIGGERTEXT_ATTR, triggerText);
	return sendXMLReq(XML_CREATETRIGGER_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqStartTransactionOp(const Chain& tableSet)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	return sendXMLReq(XML_STARTTRANSACTION_REQUEST, pRoot);       
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqCommitTransactionOp(const Chain& tableSet)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	return sendXMLReq(XML_COMMITTRANSACTION_REQUEST, pRoot);
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqRollbackTransactionOp(const Chain& tableSet)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	return sendXMLReq(XML_ROLLBACKTRANSACTION_REQUEST, pRoot);        
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

CegoDistDbHandler::ResultType CegoDistDbHandler::reqGetTidOp(const Chain& tableSet)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	pRoot->setAttribute(XML_TABLESET_ATTR, tableSet);
	return sendXMLReq(XML_GETTID_REQUEST, pRoot);        
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getArgValue(const Chain& arg, Chain& value)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    value = pRoot->getAttributeValue(arg);	
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }	
}

void CegoDistDbHandler::getInsertArg(Chain& tableSet, Chain& tableName, ListT<CegoField>& fl)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    
	    tableSet = pRoot->getAttributeValue(XML_TABLESET_ATTR);	
	    tableName = pRoot->getAttributeValue(XML_TABLENAME_ATTR);
	    
	    ListT<Element*> colList = pRoot->getChildren(XML_COL_ELEMENT);
	    
	    Element **pCol = colList.First();
	    while ( pCol ) 
	    {
		Chain colName = (*pCol)->getAttributeValue(XML_COLNAME_ATTR);
		Chain colType = (*pCol)->getAttributeValue(XML_COLTYPE_ATTR);
		Chain colVal = (*pCol)->getAttributeValue(XML_COLVAL_ATTR);
		
		CegoField f(CegoField(tableName, colName));
		
		CegoTypeConverter tc;
		CegoDataType type;
		type = tc.getTypeId(colType);
		
		CegoFieldValue fv(type, colVal);
		
		f.setValue(fv);
		fl.Insert(f);
		
		pCol = colList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::sendObjList(const ListT<Chain>& objList)
{
#ifdef DEBUG
    _pModule->log(_modId, Logger::DEBUG, Chain("Sending object info"));
#endif

    if ( _protType == CegoDbHandler::XML )
    {	
	_xml.getDocument()->clear();
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	Chain *pObj = objList.First();
	while ( pObj ) 
	{
	    Element *pObjElement = new Element(XML_OBJ_ELEMENT);
	    pObjElement->setAttribute(XML_NAME_ATTR, *pObj);
	    
	    pRoot->addContent(pObjElement);
	    pObj = objList.Next();
	}
	
	_xml.getDocument()->setRootElement(pRoot);    
	_xml.getDocument()->setDocType(XML_INFO_DOC);
	
	Chain response;
	_xml.getXMLChain(response);
	
#ifdef DEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
	_pModule->log(_modId, Logger::DEBUG, response);
	_pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif
	
	_pN->setMsg((char*)response, response.length());
	_pN->writeMsg();
	
	_xml.getDocument()->clear();	
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::sendObjByTableList(const ListT<CegoTableObject>& idxList,
					   const ListT<CegoKeyObject>& keyList,
					   const ListT<CegoCheckObject>& checkList)
{
    throw Exception(EXLOC, Chain("No more supported"));
}

void CegoDistDbHandler::sendTID(int tid)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	_xml.getDocument()->clear();
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_TID_ATTR, Chain(tid));   
	
	_xml.getDocument()->setRootElement(pRoot);    
	_xml.getDocument()->setDocType(XML_INFO_DOC);
	
	Chain response;
	_xml.getXMLChain(response);
	
#ifdef DEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
	_pModule->log(_modId, Logger::DEBUG, response);
	_pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif
		
	_pN->setMsg((char*)response, response.length());
	_pN->writeMsg();
	
	_xml.getDocument()->clear();
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::sendPageCount(int pageCount)
{
    if ( _protType == CegoDbHandler::XML )
    {	
	_xml.getDocument()->clear();
	
	Element* pRoot = new Element(XML_FRAME_ELEMENT);
	
	pRoot->setAttribute(XML_PAGECOUNT_ATTR, Chain(pageCount));   
	
	_xml.getDocument()->setRootElement(pRoot);    
	_xml.getDocument()->setDocType(XML_INFO_DOC);
	
	Chain response;
	_xml.getXMLChain(response);
	
#ifdef DEBUG
	_pModule->log(_modId, Logger::DEBUG, Chain("--- XML ---"));
	_pModule->log(_modId, Logger::DEBUG, response);
	_pModule->log(_modId, Logger::DEBUG, Chain("--- --- ---"));
#endif    
	
	_pN->setMsg((char*)response, response.length());
	_pN->writeMsg();
	
	_xml.getDocument()->clear();
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getObjectList(ListT<Chain>& objList)
{   
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{	    
	    ListT<Element*> objElementList = pRoot->getChildren(XML_OBJ_ELEMENT);
	    
	    Element **pObjEntry = objElementList.First();
	    while ( pObjEntry ) 
	    {
		objList.Insert((*pObjEntry)->getAttributeValue(XML_NAME_ATTR));
		pObjEntry = objElementList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}

void CegoDistDbHandler::getObjectByTableList(ListT<CegoTableObject>& idxList, 
					     ListT<CegoKeyObject>& keyList,
					     ListT<CegoCheckObject>& checkList)
{
    /*
    if ( _protType == CegoDbHandler::XML )
    {	
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    ListT<Element*> objElementList = pRoot->getChildren(XML_OBJ_ELEMENT);
	    
	    Element **pObjEntry = objElementList.First();
	    while ( pObjEntry ) 
	    {
		
		CegoTypeConverter tc;
		CegoObject::ObjectType type = tc.getObjectTypeId((*pObjEntry)->getAttributeValue(XML_OBJTYPE_ATTR));
		
		switch ( type )
		{
		case CegoObject::AVLTREE:
		case CegoObject::PAVLTREE:
		case CegoObject::UAVLTREE:
		{
		    CegoTableObject io;
		    io.putElement(*pObjEntry);
		    idxList.Insert(io);
		    break;
		}
		case CegoObject::FKEY:
		{
		    CegoKeyObject ko;
		    ko.putElement(*pObjEntry);
		    keyList.Insert(ko);
		    break;
		}
		case CegoObject::CHECK:
		{
		    CegoCheckObject co;
		    co.putElement(*pObjEntry);
		    checkList.Insert(co);
		    break;
		}
		case CegoObject::SYSTEM:
		case CegoObject::TABLE:
		case CegoObject::VIEW:
		case CegoObject::RBSEG:
		case CegoObject::PROCEDURE:
		case CegoObject::JOIN:
		case CegoObject::PBTREE:
		case CegoObject::UBTREE:
		case CegoObject::BTREE:
		case CegoObject::TRIGGER:
		case CegoObject::ALIAS:
		case CegoObject::UNDEFINED:
		{
		    break;
		}
		}
		
		pObjEntry = objElementList.Next();
	    }
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
    */
    throw Exception(EXLOC, Chain("No more supported"));
}

void CegoDistDbHandler::getPageCount(int& pageCount)
{
    if ( _protType == CegoDbHandler::XML )
    {
	Element *pRoot = _xml.getDocument()->getRootElement();
	if ( pRoot )
	{
	    pageCount = pRoot->getAttributeValue(XML_PAGECOUNT_ATTR).asInteger();	    
	}
    }
    else
    {
	throw Exception(EXLOC, Chain("Serial protocol still not supported"));
    }
}
